/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_PIXEL_DESCRIPTION_TABLE_H_
#define SPH_PIXEL_DESCRIPTION_TABLE_H_

#include "sph_pixel_descriptor.h"
#include "sph_spectrum.h"
#include "sph_error.h"
#include "sph_ifs_lenslet_model.h"
#include "sph_spectral_region.h"
#include "sph_master_frame.h"
#include <cpl.h>
extern sph_error_code SPH_PDT_GENERAL;
extern sph_error_code SPH_PDT_SPECTRA_NONE_FOUND;
extern sph_error_code SPH_PDT_REGION_LENSLET_MISMATCH;

typedef struct _sph_pixel_description_table_
{
    int                     nx;  // number pixel x
    int                     ny;  // number pixel y
    int                     nregions; /* The number of distinct regions with spectra */
    int                     maxlength; // maximum spectral length
    int                     idlongest; // id of spectrum with max length
    double                  offx;   // the offset in x (pixels)
    double                  offy; // the offset in y (pixels)
    sph_spectral_region**	regions; // the spectral regions.
    sph_pixel_descriptor*   arr; // the pixel descrption array
} sph_pixel_description_table;

sph_pixel_description_table*
sph_pixel_description_table_new(int nx, int ny, double offx, double offy);

sph_pixel_description_table*
sph_pixel_description_table_new_from_image( cpl_image* image,
                                        double offx,
                                        double offy,
                                        double threshold);

sph_pixel_description_table*
sph_pixel_description_table_new_from_labels( cpl_image* image,
                                        double offx,
                                        double offy);

sph_pixel_description_table*
sph_pixel_description_table_new_shift( sph_pixel_description_table* pdt,
		sph_ifs_lenslet_model* model,
                  double offx,
                  double offy);

cpl_image*
sph_pixel_description_table_get_specid_image(sph_pixel_description_table* self);

sph_error_code
sph_pixel_description_table_setup_regions( sph_pixel_description_table* self );

sph_spectral_region*
sph_pixel_description_table_get_region( sph_pixel_description_table* self, int specid );

sph_pixel_description_table*
sph_pixel_description_table_load(const char* fitsfilename);

/*
sph_pixel_description_table*
sph_pixel_description_table_load_from_table(const char* fitsfilename);
*/

int
sph_pixel_description_table_save(sph_pixel_description_table* self,
                                 const char* fitsfilename, sph_ifs_lenslet_model* model);

void sph_pixel_description_table_delete(sph_pixel_description_table* self);

sph_pixel_descriptor*
sph_pixel_description_table_get_descriptor( sph_pixel_description_table* self,
                                            int xx,
                                            int yy
                                          ) CPL_ATTR_NONNULL;

const sph_pixel_descriptor*
sph_pixel_description_table_get_descriptor_const
    (const sph_pixel_description_table* self,
     int xx,
     int yy) CPL_ATTR_NONNULL;

sph_pixel_descriptor*
sph_pixel_description_table_get_array(sph_pixel_description_table* self)
    CPL_ATTR_NONNULL CPL_ATTR_PURE;

const sph_pixel_descriptor*
sph_pixel_description_table_get_array_const
(const sph_pixel_description_table* self) CPL_ATTR_NONNULL CPL_ATTR_PURE;

/*
sph_spectrum*
sph_pixel_description_table_extract_spectrum(sph_pixel_description_table* self,
                                            int lensid,
                                            cpl_image* otherimage,
                                            cpl_vector* wavs
                                          );
*/

sph_spectrum*
sph_pixel_description_table_extract_spectrum_mframe(sph_pixel_description_table*,
                                                    int,
                                                    const sph_master_frame*,
                                                    const cpl_vector*)
    CPL_ATTR_ALLOC;

sph_error_code
sph_pixel_description_table_correct_non_linear(sph_pixel_description_table* self, 
		double square_term, double lin_term, double offset);

sph_error_code
sph_pixel_description_table_set_spectra_wavelengths(
		sph_pixel_description_table* self,
		int specid,
		const cpl_vector* iwavs,
		short int no_spline_fit);

sph_error_code
sph_pixel_description_table_set_wavelengths(
		sph_pixel_description_table* self,
		int specid,
		const cpl_vector* distp,
		const cpl_vector* idwavs );

cpl_vector*
sph_pixel_description_table_get_wavs( sph_pixel_description_table* self, int specid);

/*
sph_error_code
sph_pixel_description_table_associate_lenslets( sph_pixel_description_table* self,
		sph_ifs_lenslet_model* insmodel );
*/

cpl_mask*
sph_pixel_description_table_get_mask(sph_pixel_description_table* self);

cpl_image*
sph_pixel_description_table_get_wavimage(sph_pixel_description_table* self) ;

cpl_image*
sph_pixel_description_table_get_dispimage(sph_pixel_description_table* self) ;

sph_error_code
sph_pixel_description_table_broaden(sph_pixel_description_table* self);

sph_pixel_description_table*
sph_pixel_description_table_new_from_model(
		sph_ifs_lenslet_model* model,
		double offx,
		double offy);

sph_error_code
sph_pixel_description_table_collapse_line_total(
		const sph_pixel_description_table* self,
		const sph_master_frame* mframe,
		int yy, int minx, int maxx,
		double* value,
		int* bpix,
		double* rms,
		double* weight,
		double* delta_lambda );

cpl_error_code
sph_pixel_description_table_save_dfs(sph_pixel_description_table* self,
                                     const char* fitsfilename,
                                     sph_ifs_lenslet_model* model,
                                     cpl_frameset* allframes,
                                     cpl_frame* template_frame,
                                     cpl_parameterlist* params,
                                     const char* tag,
                                     const char* recipe,
                                     const char* pipename,
                                     const cpl_propertylist* plist );

sph_error_code
sph_pixel_description_table_set_wavs_lensid(
		sph_pixel_description_table* self,
		int lid,
		cpl_vector* wavs );

int
sph_pixel_description_table_get_maxlength( sph_pixel_description_table* self );

int
sph_pixel_description_table_find_maxlength( sph_pixel_description_table* self );

sph_pixel_description_table*
sph_pixel_description_table_new_from_images(
		cpl_image* image_id, cpl_image* image_sid, cpl_image* image_wav,
		cpl_image* image_dw, cpl_image* image_ddw, cpl_image* image_ill,
		double offx,
		double offy );

#endif /*SPH_PIXEL_DESCRIPTION_TABLE_H_*/
