/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */


#include "sph_test.h"
#include "sph_error.h"
#include "sph_master_frame.h"
#include "sph_double_image.h"
#include "sph_cube.h"
#include "sph_spectrum.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_ifs_keywords.h"
#include "sph_zpl_keywords.h"
#include "sph_ird_keywords.h"
#include "sph_ifs_tags.h"
#include "sph_zpl_tags.h"
#include "sph_ird_tags.h"
#include "sph_pixel_description_table.h"
#include "sph_ldt.h"
#include "sph_fits.h"
#include "sph_keyword_manager.h"
#include "sph_framecombination.h"
#include "sph_shared_irdifs.h"

#include <cpl.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <strings.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <limits.h>
#include <assert.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_test
 *
 * @par Descirption:
 * This module provides functionality that are useful for testing purposes.
 * The functions defined here are not supposed ot be used in any other context
 * apart from testing of recipes or unit testing of functions.
 *
 * @par Synopsis:
 * No specific structure associated with this module.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static int (*sph_test_fpost)(void);


/*----------------------------------------------------------------------------*/
/**
 @brief    Get a process specific id
 @return   The process specific id
 */
/*----------------------------------------------------------------------------*/
static unsigned sph_id(void) {
    const pid_t pid = getpid();

    return (unsigned)pid;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Any function needed by the test
 @param    inparam     Input parameter
 @return   returns a value
 */
/*----------------------------------------------------------------------------*/

static
char** testfiles = NULL;
static
int ntestfiles = 0;

const char* frame_group_to_string(cpl_frame_group group) {
    switch (group) {
    case CPL_FRAME_GROUP_RAW:
        return CPL_FRAME_GROUP_RAW_ID;
        break;
    case CPL_FRAME_GROUP_NONE:
        return "NONE";
        break;
    case CPL_FRAME_GROUP_CALIB:
        return CPL_FRAME_GROUP_CALIB_ID;
        break;
    case CPL_FRAME_GROUP_PRODUCT:
        return CPL_FRAME_GROUP_PRODUCT_ID;
        break;
    default:
        return "???";
        break;
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the DIT list that is placed in the CALIB database to
 ensure within recipes that the DIT of frames are ok.
 @deprecated Note that this code is not supposed to be used anymore -- but rather
 that the functionality found sph_utils.c is to be used.
 @return   returns errorcode.
 */
/*----------------------------------------------------------------------------*/
int sph_test_create_dit_list(void) {
    int ii;
    int numrows = 10;
    float exptime = 1.3;
    cpl_table* dit_table = cpl_table_new(numrows);

    if (cpl_table_new_column(dit_table, "NREADOUTS", CPL_TYPE_INT)
            != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: Error adding column: %s", cpl_func,
                __LINE__, cpl_error_get_where());
        return (int) cpl_error_get_code();

    }
    if (cpl_table_new_column(dit_table, "DIT", CPL_TYPE_FLOAT)
            != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, "%s():%d: Error adding column: %s", cpl_func,
                __LINE__, cpl_error_get_where());
        return (int) cpl_error_get_code();
    }

    for (ii = 0; ii < numrows; ii++) {
        cpl_table_set(dit_table, "NREADOUTS", ii, ii + 1);
        cpl_table_set(dit_table, "DIT", ii, (ii + 1) * exptime);
    }
    cpl_table_dump(dit_table, 0, 10, NULL);
    cpl_table_save(dit_table, NULL, NULL, "dit_table.fits", CPL_IO_DEFAULT);
    return CPL_ERROR_NONE;
}

cpl_error_code sph_test_retag_frames_by_group(cpl_frameset* inoutframes,
        cpl_frame_group group, const char* newtag) {
    int ff = 0;
    cpl_frameset* dumlist = NULL;
    cpl_frame* aframe = NULL;

    dumlist = cpl_frameset_new();
    for (ff = 0; ff < cpl_frameset_get_size(inoutframes); ++ff) {
        aframe = cpl_frame_duplicate(
                cpl_frameset_get_position(inoutframes, ff));
        if (cpl_frame_get_group(aframe) == group) {
            cpl_frame_set_tag(aframe, newtag);
        }
        cpl_frameset_insert(dumlist, aframe);
    }
    for (ff = 0; ff < cpl_frameset_get_size(inoutframes); ++ff) {
        aframe = cpl_frameset_get_position(inoutframes, ff);
        cpl_frameset_erase_frame(inoutframes, aframe);
    }cpl_ensure_code( cpl_frameset_get_size(inoutframes) == 0,
            CPL_ERROR_ILLEGAL_OUTPUT);
    for (ff = 0; ff < cpl_frameset_get_size(dumlist); ++ff) {
        aframe = cpl_frame_duplicate(cpl_frameset_get_position(dumlist, ff));
        cpl_frameset_insert(inoutframes, aframe);
    }
    cpl_frameset_delete(dumlist);
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw dark frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ifs_raw_dark_frameset(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    int xx = 0;
    int yy = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    gsl_rng* pRNG = NULL;
    unsigned long ulSeed = 0;

    pl = cpl_propertylist_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_IFS_KEYWORD_VALUE_DPR_TYPE_DARK);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    for (ii = 0; ii < nframes; ++ii) {
        image = cpl_image_new(SPH_IFS_MASTER_DARK_RON_URX,
                              SPH_IFS_MASTER_DARK_RON_URY, CPL_TYPE_DOUBLE);
        for (xx = 0; xx < SPH_IFS_MASTER_DARK_RON_URX; ++xx) {
            for (yy = 0; yy < SPH_IFS_MASTER_DARK_RON_URY; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1,
                        9.0 + gsl_ran_gaussian(pRNG, 1.0));
            }
        }
        cpl_image_set(image, 1000, 1000, 50000.0);
        cpl_image_set(image, 999, 999, 0.0);
        sprintf(czFilename, "temp_%u_%d.fits", sph_id(), ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_IFS_TAG_DARK_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
        cpl_image_delete(image);
    }
    gsl_rng_free(pRNG);
    cpl_propertylist_delete(pl);
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw dark frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ifs_raw_dark_frameset_big(cpl_frameset* frames,
        int nframes) {
    cpl_propertylist* pl = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    gsl_rng* pRNG = NULL;
    unsigned long ulSeed = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    pl = cpl_propertylist_new();

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_IFS_KEYWORD_VALUE_DPR_TYPE_DARK);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    for (int ii = 0; ii < nframes; ++ii) {
        const cpl_type_bpp bpp = CPL_TYPE_FLOAT;
        cpl_image* image = cpl_image_new(2048, 2048, CPL_TYPE_DOUBLE);
        cpl_frame* frame = NULL;
        char czFilename[256];

        for (int xx = 0; xx < 2048; ++xx) {
            for (int yy = 0; yy < 2048; ++yy) {
                cpl_image_set(image, xx + 1, yy + 1,
                        ii * 10.0 + gsl_ran_gaussian(pRNG, 1.0));
            }
        }
        sprintf(czFilename, "temp_%u_%d.fits", sph_id(), ii);
        rerr = cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_IFS_TAG_DARK_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
        cpl_image_delete(image);
    }
    cpl_propertylist_delete(pl);
    gsl_rng_free(pRNG);
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw dark frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ird_raw_dark_frameset(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;

    pl = cpl_propertylist_new();

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_IRD_KEYWORD_VALUE_DPR_TYPE_DARK);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    for (ii = 0; ii < nframes; ++ii) {
        image = cpl_image_new(256, 124, CPL_TYPE_DOUBLE);
        cpl_image_fill_noise_uniform(image, (double) ii * 10.0,
                (double) (ii + 1) * 10.0);
        sprintf(czFilename, "temp_%u_%d.fits", sph_id(), ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        cpl_image_delete(image);
        image = NULL;
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_IRD_TAG_DARK_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }

    cpl_propertylist_delete(pl);
    pl = NULL;
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw dark frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_zpl_raw_dark_frameset(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;

    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    // cpl_boolean            cam1_is_set = 0;
    cpl_boolean cam2_is_set = 0;

    pl = cpl_propertylist_new();

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_ZPL_KEYWORD_VALUE_DPR_TYPE_DARK);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);
    //add DIT ID to the header, this is only for the test reasons. There will be no DIT ID in
    //the fits file. DIT ID will be determined basing on the cpl dit table
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_DIT_ID, 1);

    //add CAMERA ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    //add FILTER ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FILTER_ID,
            SPH_ZPL_KEYWORD_VALUE_FILTER1_ID);

    for (ii = 0; ii < nframes; ++ii) {
        if (ii >= (int) (nframes / 2) && !cam2_is_set) {
            //printf("create darkframeset: append KEYWORD_ZPL_CAMERA2_ID, ii =  %d\n", ii);
            //cpl_propertylist_update_int(pl, KEYWORD_ZPL_CAMERA_NAME,KEYWORD_ZPL_CAMERA2_ID );
            //cpl_propertylist_update_int(pl, KEYWORD_ZPL_FILTER_NAME,KEYWORD_ZPL_FILTER2_ID );
            cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_DIT_ID, 2);
            cpl_propertylist_update_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.6);

            cam2_is_set = 1;
        }
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        cpl_image_fill_noise_uniform(image, (double) ii * 10.0,
                (double) (ii + 1) * 10.0);
        sprintf(czFilename, "temp_%d.fits", ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_ZPL_TAG_DARK_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw dark frameset with 2 different set of exposure time 1.3 & 1.6

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_zpl_raw_dark_frameset_et(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;

    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    cpl_boolean exp2_is_set = 0;

    pl = cpl_propertylist_new();

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_ZPL_KEYWORD_VALUE_DPR_TYPE_DARK);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);
    //add CAMERA ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
    //add FILTER ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FILTER_ID,
            SPH_ZPL_KEYWORD_VALUE_FILTER1_ID);

    for (ii = 0; ii < nframes; ++ii) {
        if (ii >= (int) (nframes / 2) && !exp2_is_set) {
            cpl_propertylist_update_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.6);
            exp2_is_set = 1;
        }
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        cpl_image_fill_noise_uniform(image, (double) ii * 10.0,
                (double) (ii + 1) * 10.0);
        sprintf(czFilename, "temp_%d.fits", ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_ZPL_TAG_DARK_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Create and return the standard parameterlist for IFS spectra postions

 @return   the standard parameterlis or NULL if an error occcured.

 This function creates the standard parameterlist for the recipe
 and returns it.

 The return value is NULL when an error occured.
 */
/*----------------------------------------------------------------------------*/
cpl_parameterlist* sph_test_create_paramlist_ifs_spectra_positions(void) {
    cpl_parameterlist* result = NULL;
    cpl_parameter* p = NULL;
    result = cpl_parameterlist_new();

    p = cpl_parameter_new_value("ifs.spectra_positions.outfilename",
            CPL_TYPE_STRING, "No description", "ifs.spectra_positions",
            "spectra_positions");

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_enum("ifs.spectra_positions.coll_alg", CPL_TYPE_INT,
            "No description", "ifs.spectra_positions", 1, 3, 0, 1, 2);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_enum("ifs.spectra_positions.method", CPL_TYPE_INT,
            "No description", "ifs.spectra_positions", 1, 2, 0, 1);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_value("ifs.spectra_positions.threshold",
            CPL_TYPE_DOUBLE, "No description", "ifs.spectra_positions", 100.0);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_value("ifs.spectra_positions.dither_tolerance",
            CPL_TYPE_DOUBLE, "No description", "ifs.spectra_positions", 0.01);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_value("ifs.spectra_positions.subtract_dark",
            CPL_TYPE_BOOL, "No description", "ifs.spectra_positions", 0);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_range("ifs.spectra_positions.clean_mean.reject_high",
            CPL_TYPE_INT, "No description", "ifs.spectra_positions", 0, 0, 20);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);
    p = cpl_parameter_new_range("ifs.spectra_positions.clean_mean.reject_low",
            CPL_TYPE_INT, "No description", "ifs.spectra_positions", 0, 0, 20);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);

    return result;

}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create and return the standard parameterlist for IFS master dark

 @return   the standard parameterlis or NULL if an error occcured.

 This function creates the standard parameterlist for the recipe
 and returns it.

 The return value is NULL when an error occured.
 */
/*----------------------------------------------------------------------------*/
cpl_parameterlist* sph_test_create_paramlist_ifs_master_dark(void) {
    cpl_parameterlist* result = NULL;
    cpl_parameter* p = NULL;
    result = cpl_parameterlist_new();

    p = cpl_parameter_new_enum("ifs.master_dark.coll_alg", CPL_TYPE_INT,
            "The collapse algorithm", "ifs.master_dark",
            SPH_COLL_ALG_CLEAN_MEAN, 3, SPH_COLL_ALG_CLEAN_MEAN,
            SPH_COLL_ALG_MEAN, SPH_COLL_ALG_MEDIAN);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ifs.master_dark.outfilename", CPL_TYPE_STRING,
            "The name of the master dark output file", "ifs.master_dark",
            "master_dark.fits");

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ifs.master_dark.make_badpix", CPL_TYPE_BOOL,
            "Flag to set if static badpixel file should be created.",
            "ifs.master_dark", 0);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ifs.master_dark.badpixfilename",
            CPL_TYPE_STRING, "The name of the static badpixel output file",
            "ifs.master_dark", "static_badpixel_mask.fits");

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ifs.master_dark.clean_mean.reject_low",
            CPL_TYPE_INT, "The number of pixels to reject at the lower end",
            "ifs.master_dark", 0);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ifs.master_dark.clean_mean.reject_high",
            CPL_TYPE_INT, "The number of pixels to reject at the higher end",
            "ifs.master_dark", 0);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);

    return result;

}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create and return the standard parameterlist for IFS master dark

 @return   the standard parameterlis or NULL if an error occcured.

 This function creates the standard parameterlist for the recipe
 and returns it.

 The return value is NULL when an error occured.
 */
/*----------------------------------------------------------------------------*/
cpl_parameterlist* sph_test_create_paramlist_ird_master_dark(void) {
    cpl_parameterlist* result = NULL;
    cpl_parameter* p = NULL;
    result = cpl_parameterlist_new();

    p = cpl_parameter_new_enum("ird.master_dark.coll_alg", CPL_TYPE_INT,
            "The collapse algorithm", "ird.master_dark",
            SPH_COLL_ALG_CLEAN_MEAN, 3, SPH_COLL_ALG_CLEAN_MEAN,
            SPH_COLL_ALG_MEAN, SPH_COLL_ALG_MEDIAN);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ird.master_dark.outfilename", CPL_TYPE_STRING,
            "The name of the master dark output file", "ird.master_dark",
            "master_dark.fits");

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ird.master_dark.make_badpix", CPL_TYPE_BOOL,
            "Flag to set if static badpixel file should be created.",
            "ird.master_dark", 0);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ird.master_dark.badpixfilename",
            CPL_TYPE_STRING, "The name of the static badpixel output file",
            "ird.master_dark", "static_badpixel_mask.fits");

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);

    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ird.master_dark.clean_mean.reject_low",
            CPL_TYPE_INT, "The number of pixels to reject at the lower end",
            "ird.master_dark", 0);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);

    p = cpl_parameter_new_value("ird.master_dark.clean_mean.reject_high",
            CPL_TYPE_INT, "The number of pixels to reject at the higher end",
            "ird.master_dark", 0);

    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(result, p);

    return result;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create and return the standard parameterlist for the recipe

 @return   the standard parameterlis or NULL if an error occcured.

 This function creates the standard parameterlist for the recipe
 and returns it.

 The return value is NULL when an error occured.
 */
/*----------------------------------------------------------------------------*/
cpl_parameterlist* sph_test_create_paramlist_zpl_master_dark(void) {
    cpl_parameterlist* result = NULL;
    cpl_parameter* p = NULL;
    result = cpl_parameterlist_new();

    if (result == NULL) {

        sph_error_raise(SPH_ERROR_INFO_MESSAGE, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create the "
                        "cpl_parameterlist for the zpl master dark");

        return NULL;
    } else {
        /* Code to set up parameters GENERATED DO NOT EDIT */

        p = cpl_parameter_new_value("zpl.master_dark.outfilename",
                CPL_TYPE_STRING, "No description", "zpl.master_dark",
                "masterdark.cube.fits");

        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(result, p);
        p = cpl_parameter_new_enum("zpl.master_dark.coll_alg", CPL_TYPE_INT,
                "No description", "zpl.master_dark", 1, 3, 0, 1, 2);

        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(result, p);
        p = cpl_parameter_new_value("zpl.master_dark.make_badpix",
                CPL_TYPE_BOOL, "No description", "zpl.master_dark", 0);

        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(result, p);
        p = cpl_parameter_new_value("zpl.master_dark.badpixfilename",
                CPL_TYPE_STRING, "No description", "zpl.master_dark",
                "static_badpixels.fits");

        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(result, p);
        p = cpl_parameter_new_range(
                "zpl.master_dark.coll_alg.clean_mean.reject_high", CPL_TYPE_INT,
                "No description", "zpl.master_dark", 0, 0, 20);

        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(result, p);
        p = cpl_parameter_new_range(
                "zpl.master_dark.coll_alg.clean_mean.reject_low", CPL_TYPE_INT,
                "No description", "zpl.master_dark", 0, 0, 20);

        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(result, p);
        /* End of code to set up parameters BUT ITS STILL NOT SAFE TO EDIT! FILE IS GENERATED !*/
    }

    return result;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw master dark frame and add to framelist.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 Creates a mock master dark frame and adds it to the framelist. The
 mock master dark has a value of 0.1 counts / s for all pixels.

 @return   returns error code.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ifs_master_dark_frame(cpl_frameset* frames,
        int numberDITs) {
    int ii = 0;
    cpl_frame* frame = NULL;
    sph_master_frame* mframe = NULL;
    sph_cube* darkcube = NULL;
    const char* czFilename = "temp_master_dark.fits";
    cpl_error_code rerr = CPL_ERROR_NONE;
    cpl_parameterlist* params = NULL;

    params = sph_test_create_paramlist_ifs_master_dark();

    darkcube = sph_cube_new(czFilename);

    for (ii = 0; ii < numberDITs; ++ii) {
        mframe = sph_master_frame_new(256, 256);
        cpl_image_delete(mframe->image);
        mframe->image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        cpl_image_set(mframe->badpixelmap, 100, 100, 1.0);
        cpl_image_set(mframe->badpixelmap, 99, 99, 1.0);
        cpl_image_fill_noise_uniform(mframe->image, (ii + 1) * 9.0 - 0.0001,
                (ii + 1) * 9.0 + 0.0001);
        sph_cube_append_master(darkcube, mframe, ii * 1.0 + 1.0);
        sph_master_frame_delete(mframe);
    }

    sph_cube_finalise_file(darkcube->filename);
    frame = sph_utils_create_calib_frame(czFilename, CPL_FRAME_TYPE_IMAGE,
            CPL_FRAME_LEVEL_FINAL, frames, params,
            SPH_IFS_KEYWORD_VALUE_PRO_CATG_MASTER_DARK,
            SPH_RECIPE_NAME_IFS_MASTER_DARK, SPH_PIPELINE_NAME_IFS, NULL);

    cpl_frame_set_tag(frame, SPH_IFS_TAG_DARK_CALIB);

    cpl_parameterlist_delete(params);
    params = NULL;
    sph_cube_delete(darkcube);
    darkcube = NULL;
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw master dark frame and add to framelist.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 Creates a mock raw master dark frame and adds it to the framelist. The
 mock master dark has a value of 0.1 counts / s for all pixels.

 @return   returns pointer to new frame.
 */
/*----------------------------------------------------------------------------*/
sph_pixel_description_table*
sph_test_create_ifs_master_pdt_frame(cpl_frameset* frames) {
    cpl_frame* frame = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_ifs_lenslet_model* model = NULL;
    char czFilename[256];
    cpl_parameterlist* params = NULL;

    params = sph_test_create_paramlist_ifs_spectra_positions();

    model = sph_test_create_small_lenslet_model();
    sprintf(czFilename, "temp_spectra_positions.fits");

    pdt = sph_pixel_description_table_new_from_model(model, 0.0, 0.0);
    sph_pixel_description_table_save(pdt, czFilename, model);

    if (frames) {
        frame = cpl_frame_duplicate(
                sph_utils_create_calib_frame(czFilename, CPL_FRAME_TYPE_TABLE,
                        CPL_FRAME_LEVEL_FINAL, frames, params,
                        SPH_IFS_KEYWORD_VALUE_PRO_CATG_PIXEL_DESCRIPTION_TAB,
                        SPH_RECIPE_NAME_IFS_SPECTRA_POSITIONS,
                        SPH_PIPELINE_NAME_IFS, NULL));

        cpl_frame_set_tag(frame, SPH_IFS_TAG_SPEC_POS_CALIB);
        cpl_frameset_insert(frames, frame);
        sph_ifs_lenslet_model_delete(model);
        model = NULL;
        cpl_parameterlist_delete(params);
        params = NULL;
        sph_pixel_description_table_delete(pdt);
        return NULL;
    }
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    cpl_parameterlist_delete(params);
    params = NULL;
    return pdt;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw master dark frame and add to framelist.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 Creates a mock raw master dark frame and adds it to the framelist. The
 mock master dark has a value of 0.1 counts / s for all pixels.

 @return   returns pointer to new frame.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ird_master_dark_frame(cpl_frameset* frames,
        int numberDITs) {
    int ii = 0;
    cpl_frame* frame = NULL;
    sph_master_frame* mframe = NULL;
    sph_cube* darkcube = NULL;
    char czFilename[256];
    cpl_error_code rerr = CPL_ERROR_NONE;
    cpl_parameterlist* params = NULL;

    params = sph_test_create_paramlist_ifs_master_dark();

    sprintf(czFilename, "temp_master_dark.fits");

    darkcube = sph_cube_new(czFilename);

    for (ii = 0; ii < numberDITs; ++ii) {
        mframe = sph_master_frame_new(256, 124);
        cpl_image_delete(mframe->image);
        mframe->image = cpl_image_new(256, 124, CPL_TYPE_DOUBLE);
        cpl_image_fill_noise_uniform(mframe->image, (ii + 1) * 0.01 - 0.01,
                (ii + 1) * 0.01 + 0.01);
        sph_cube_append_master(darkcube, mframe, ii * 1.0 + 1.0);
    }

    sph_cube_finalise_file(darkcube->filename);
    frame = sph_utils_create_calib_frame(czFilename, CPL_FRAME_TYPE_IMAGE,
            CPL_FRAME_LEVEL_FINAL, frames, params,
            SPH_IRD_KEYWORD_VALUE_PRO_CATG_MASTER_DARK,
            SPH_RECIPE_NAME_IFS_MASTER_DARK, SPH_PIPELINE_NAME_IFS, NULL);
    cpl_frame_set_tag(frame, SPH_IRD_TAG_DARK_CALIB);
    cpl_frameset_insert(frames, frame);

    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a small lenslet model for 256x256 detector

 @param none

 @return the new lenslet model

 Description: Create a new lenslet model of a 256x256 detector

 */
/*----------------------------------------------------------------------------*/
sph_ifs_lenslet_model* sph_test_create_small_lenslet_model(void) {
    sph_ifs_lenslet_model* model = NULL;

    model = sph_ifs_lenslet_model_new();

    if (!model)
        return NULL;

    model->detsize_pixels = 256;
    model->detsize_microns = model->detsize_pixels * model->pixsize_microns;
    model->lenslets_per_side = 8;
    // Lines below prob. not needed.
    //model->stretch_x = 2.0 * SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS / ( CPL_MATH_SQRT3 * model->lensize_microns );
    //model->stretch_y = 1.047 * SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS / ( 3.0 * model->lensize_microns ) ;

    return model;
}
cpl_vector* sph_test_cpl_vector_linear(int np, double min, double max) {
    cpl_vector* result = cpl_vector_new(np);
    int ii = 0;
    double d = 0.0;

    if (!result)
        return NULL;
    if (np == 0)
        return NULL;
    d = (max - min) / np;
    for (ii = 0; ii < np; ++ii) {
        cpl_vector_set(result, ii, min + ii * d);
    }
    return result;
}
cpl_vector* sph_test_cpl_vector_gauss(int np, double min, double max,
        double mean, double sigma, double peak) {
    cpl_vector* result = cpl_vector_new(np);
    int ii = 0;
    double d = 0.0;
    double x = 0.0;
    double val = 0.0;

    if (!result)
        return NULL;
    if (np == 0)
        return NULL;
    d = (max - min) / np;
    for (ii = 0; ii < np; ++ii) {
        x = min + ii * d - mean;
        val = peak * exp(-x * x / sigma * sigma);
        cpl_vector_set(result, ii, val);
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw dff frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ifs_raw_dff_frameset_broad_band(
        cpl_frameset* frames, int nframes, int lampOnId) {
    int ii = 0;
    int xx = 0;
    int yy = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    gsl_rng* pRNG = NULL;
    unsigned long ulSeed = 0;
    int lamps_on_off[6] = { 0, 0, 0, 0, 0, 0 };

    pl = cpl_propertylist_new();

    cpl_ensure( lampOnId <= 6 && lampOnId > 0, CPL_ERROR_ILLEGAL_INPUT,
            CPL_ERROR_ILLEGAL_INPUT);
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);
    lamps_on_off[lampOnId - 1] = CPL_TRUE;
    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_IFS_KEYWORD_VALUE_DPR_TYPE_DFF);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME2, 1.3);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_CALIB_LAMBDA, 1.2);
    if (lamps_on_off[0])
        cpl_propertylist_append_bool(pl, SPH_IFS_KEYWORD_LAMP1_SWITCH,
                lamps_on_off[0]);
    if (lamps_on_off[1])
        cpl_propertylist_append_bool(pl, SPH_IFS_KEYWORD_LAMP2_SWITCH,
                lamps_on_off[1]);
    if (lamps_on_off[2])
        cpl_propertylist_append_bool(pl, SPH_IFS_KEYWORD_LAMP3_SWITCH,
                lamps_on_off[2]);
    if (lamps_on_off[3])
        cpl_propertylist_append_bool(pl, SPH_IFS_KEYWORD_LAMP4_SWITCH,
                lamps_on_off[3]);
    if (lamps_on_off[4])
        cpl_propertylist_append_bool(pl, SPH_IFS_KEYWORD_LAMP5_SWITCH,
                lamps_on_off[4]);
    if (lamps_on_off[5])
        cpl_propertylist_append_bool(pl, SPH_IFS_KEYWORD_LAMP6_SWITCH,
                lamps_on_off[5]);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    for (ii = 0; ii < nframes; ++ii) {
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        for (xx = 0; xx < 256; ++xx) {
            for (yy = 0; yy < 256; ++yy) {
                cpl_image_set(
                        image,
                        xx + 1,
                        yy + 1,
                        9.0 * ((ii + 1) * (xx + yy))
                                + gsl_ran_gaussian(pRNG, 1.0));
            }
        }

        sprintf(czFilename, "temp_lamp_%u_%d_%d.fits", sph_id(), lampOnId, ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_IFS_TAG_DFF_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
        cpl_image_delete(image);
        image = NULL;
    }
    cpl_propertylist_delete(pl);
    gsl_rng_free(pRNG);
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw iff frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_zpl_raw_iff_frameset(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    int xx = 0;
    int yy = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    gsl_rng* pRNG = NULL;
    unsigned long ulSeed = 0;

    cpl_boolean cam2_is_set = 0;

    pl = cpl_propertylist_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_ZPL_KEYWORD_VALUE_DPR_TYPE_IFF);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    //add DIT ID to the header, this is only for the test reasons. There will be no DIT ID in
    //the fits file. DIT ID will be determined basing on the cpl dit table
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_DIT_ID, 1);

    // add CAMERA ID and FILTER ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FILTER_ID,
            SPH_ZPL_KEYWORD_VALUE_FILTER1_ID);
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    for (ii = 0; ii < nframes; ++ii) {
        if (ii >= (int) (nframes / 2) && !cam2_is_set) {

            cpl_propertylist_update_int(pl, SPH_ZPL_KEYWORD_DIT_ID, 2);
            cpl_propertylist_update_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 2.6);
            cam2_is_set = 1;
        }
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        for (xx = 0; xx < 256; ++xx) {
            for (yy = 0; yy < 256; ++yy) {
                cpl_image_set(
                        image,
                        xx + 1,
                        yy + 1,
                        9.0 * (ii + 1) * (xx + yy + 10)
                                + gsl_ran_gaussian(pRNG, 1.0) + 2.96);
            }
        }

        sprintf(czFilename, "temp_%d.fits", ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_ZPL_TAG_IFF_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw pff (polarization flat field) frameset.
 Two sort of images will be created filled with constant counts.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_zpl_raw_pff_const_frameset(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    int xx = 0;
    int yy = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    gsl_rng* pRNG = NULL;
    unsigned long ulSeed = 0;

    float coeff = 10.0;

    pl = cpl_propertylist_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_ZPL_KEYWORD_VALUE_DPR_TYPE_PFF);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    //add DIT ID to the header, this is only for the test reasons. There will be no DIT ID in
    //the fits file. DIT ID will be determined basing on the cpl dit table
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_DIT_ID, 1);

    // add CAMERA ID and FILTER ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FILTER_ID,
            SPH_ZPL_KEYWORD_VALUE_FILTER1_ID);
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    for (ii = 0; ii < nframes; ++ii) {
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        for (xx = 0; xx < 256; ++xx) {
            for (yy = 0; yy < 256; ++yy) {
                //cpl_image_set( image, xx + 1, yy + 1, 9.0 * (ii+1) * (xx + yy + 10 ) + gsl_ran_gaussian(pRNG, 1.0 ) + 2.96 );
                //cpl_image_set( image, xx + 1, yy + 1, coeff * (xx + yy + 10 ) + gsl_ran_gaussian(pRNG, 1.0 ) + 2.96 );
                cpl_image_set(image, xx + 1, yy + 1, coeff * 10);
            }
        }

        sprintf(czFilename, "temp_%d.fits", ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_ZPL_TAG_PFF_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw meff (modem efficiency) frameset.
 Two sort of images will be created filled with constant counts.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_zpl_raw_meff_const_frameset(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    int xx = 0;
    int yy = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;
    gsl_rng* pRNG = NULL;
    unsigned long ulSeed = 0;

    float coeff = 10.0;

    pl = cpl_propertylist_new();

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (ulSeed == 0) {
        ulSeed = (unsigned long) time(NULL) + (unsigned long) clock();
    }
    gsl_rng_set(pRNG, ulSeed);

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_ZPL_KEYWORD_VALUE_DPR_TYPE_MEFF);
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    //add DIT ID to the header, this is only for the test reasons. There will be no DIT ID in
    //the fits file. DIT ID will be determined basing on the cpl dit table
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_DIT_ID, 1);

    // add CAMERA ID and FILTER ID to the header
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FILTER_ID,
            SPH_ZPL_KEYWORD_VALUE_FILTER1_ID);
    cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
            SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    for (ii = 0; ii < nframes; ++ii) {
        image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
        for (xx = 0; xx < 256; ++xx) {
            for (yy = 0; yy < 256; ++yy) {
                //cpl_image_set( image, xx + 1, yy + 1, 9.0 * (ii+1) * (xx + yy + 10 ) + gsl_ran_gaussian(pRNG, 1.0 ) + 2.96 );
                //cpl_image_set( image, xx + 1, yy + 1, coeff * (xx + yy + 10 ) + gsl_ran_gaussian(pRNG, 1.0 ) + 2.96 );
                cpl_image_set(image, xx + 1, yy + 1, coeff * 10);
            }
        }

        sprintf(czFilename, "temp_%d.fits", ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_ZPL_TAG_MODEM_EFF_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a mock raw iff frameset.

 @param    inframes         the framelist to which the new frames will be
 appended
 @param    nframes          number of frames to create

 @return   returns pointer to new frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_create_ird_raw_iff_frameset_ci(cpl_frameset* frames,
        int nframes) {
    int ii = 0;
    cpl_image* image = NULL;
    cpl_frame* frame = NULL;
    char czFilename[256];
    cpl_propertylist* pl = NULL;
    cpl_type_bpp bpp = CPL_TYPE_FLOAT;
    cpl_error_code rerr = CPL_ERROR_NONE;

    pl = cpl_propertylist_new();

    cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_DPR_TYPE,
            SPH_IRD_KEYWORD_VALUE_DPR_TYPE_IRDFLAT);
    cpl_propertylist_append_string(pl, SPH_IRD_KEYWORD_IRD_MODE,
            SPH_IRD_KEYWORD_VALUE_DPR_MODE_CI);
    cpl_propertylist_append_int(pl, SPH_IRD_KEYWORD_MODE_ID,
            SPH_IRD_KEYWORD_VALUE_DPR_MODE_CI_ID);
    cpl_propertylist_append_int(pl, SPH_IRD_KEYWORD_COMMON_FILTER_ID, 1);
    cpl_propertylist_append_int(pl, SPH_IRD_KEYWORD_DUAL_FILTER_ID_LEFT, 0);
    cpl_propertylist_append_string(pl, SPH_IRD_KEYWORD_DUAL_FILTER_NAME_LEFT,
            "GENERIC DUAL FILTER");
    cpl_propertylist_append_int(pl, SPH_IRD_KEYWORD_DUAL_FILTER_ID_RIGHT, 0);
    cpl_propertylist_append_string(pl, SPH_IRD_KEYWORD_DUAL_FILTER_NAME_RIGHT,
            "GENERIC DUAL FILTER");
    cpl_propertylist_append_float(pl, SPH_COMMON_KEYWORD_EXPTIME, 1.3);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NREADS, 1);

    for (ii = 0; ii < nframes; ++ii) {
        image = cpl_image_new(256, 124, CPL_TYPE_DOUBLE);
        cpl_image_fill_noise_uniform(image, (double) ii * 10.0,
                (double) (ii + 1) * 10.0);
        sprintf(czFilename, "temp_%d.fits", ii);
        cpl_image_save(image, czFilename, bpp, pl, CPL_IO_DEFAULT);
        frame = cpl_frame_new();
        cpl_frame_set_filename(frame, czFilename);
        cpl_frame_set_tag(frame, SPH_IRD_TAG_FLAT_RAW);
        cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
        cpl_frameset_insert(frames, frame);
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the results file.

 @param czFilename      output filename

 This creates the results file in the format of the SPHERE testing framework.
 The file is read and interpreted by the runtests.pl perl script.
 */
/*----------------------------------------------------------------------------*/

int sph_test_create_results_file(const char* czFilename) {
    assert( czFilename != NULL);
    return 0;
}
void sph_test_testfiles_add_file(const char* filename) {
    if (testfiles == NULL) {
        testfiles = cpl_calloc(2048 * 32, sizeof(char*));
    }

    testfiles[ntestfiles] = cpl_calloc(256, sizeof(char));
    (void)strcpy(testfiles[ntestfiles], filename);
    ntestfiles++;
}

void sph_test_cleanup_testfiles(void) {
    int ii = 0;
    if (ntestfiles < 1)
        return;
    if (testfiles == NULL)
        return;
    for (ii = 0; ii < ntestfiles; ++ii) {
        unlink(testfiles[ii]);
        cpl_free(testfiles[ii]);
        testfiles[ii] = NULL;
    }
    cpl_free(testfiles);
    testfiles = NULL;
}
cpl_frame* sph_test_cubify_image(cpl_image* inimage, const char* tag,
        cpl_frame_group group) {
    cpl_frame* result = NULL;
    sph_cube* acube = NULL;
    sph_master_frame* mframe = NULL;

    result = sph_filemanager_create_temp_frame("test_frame.fits", tag, group);

    mframe = sph_master_frame_new_from_cpl_image(inimage);
    if (!mframe) {
        cpl_frame_delete(result);
        return NULL;
    }
    acube = sph_cube_new(cpl_frame_get_filename(result));
    if (!acube) {
        sph_master_frame_delete(mframe);
        cpl_frame_delete(result);
        return NULL;
    }
    sph_cube_append_master(acube, mframe, 1.0);
    sph_master_frame_delete(mframe);
    sph_cube_finalise_file(cpl_frame_get_filename(result));
    sph_cube_delete(acube);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a frame for an image
 * @param self the image to frame
 * @param bpp the BPP to use for saving
 * @param tag the tag
 * @param group the cpl_frame_group of the frame
 *
 * @return the new cpl_frame
 *
 * This function saves the image and creates a temporary frame
 *  for it with the functiuon sph_test_frame_create_temp_frame.
 * The filename will be of the format: "test_image_<tag>_<group>.fits"
 */
/*----------------------------------------------------------------------------*/
cpl_frame* sph_test_frame_image(cpl_image* inimage, cpl_type_bpp bpp,
        const char* tag, cpl_frame_group group) {
    cpl_frame* result = NULL;
    char basef[256];

    if (group == CPL_FRAME_GROUP_RAW) {
        sprintf(basef, "%s_RAW.fits", tag);
    }
    if (group == CPL_FRAME_GROUP_CALIB) {
        sprintf(basef, "%s_CALIB.fits", tag);
    }
    if (group == CPL_FRAME_GROUP_PRODUCT) {
        sprintf(basef, "%s_PRODUCT.fits", tag);
    }
    if (group == CPL_FRAME_GROUP_NONE) {
        sprintf(basef, "%s_OTHER.fits", tag);
    }
    result = sph_filemanager_create_temp_frame(basef, tag, group);
    cpl_image_save(inimage, cpl_frame_get_filename(result), bpp, NULL,
            CPL_IO_DEFAULT);
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get a int keyword from file
 * @param filename
 * @param extension
 * @param keywordname
 *
 * @return returns the keyword value of the given keyword or 0 on error
 *
 * Returns the keyword value of the given keyword or 0 on error (and cpl_error
 * is set).
 */
/*----------------------------------------------------------------------------*/
int sph_test_get_key_int(const char* filename, unsigned int ext,
        const char* name) {
    int value = 0;
    cpl_propertylist* plist = NULL;

    plist = sph_keyword_manager_load_properties(filename, ext);
    if (cpl_propertylist_has(plist, name)) {
        value = cpl_propertylist_get_int(plist, name);
    }
    cpl_propertylist_delete(plist);
    plist = NULL;
    return value;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get a double keyword from file
 * @param filename
 * @param extension
 * @param keywordname
 *
 * @return returns the keyword value of the given keyword or 0 on error
 *
 * Returns the keyword value of the given keyword or 0 on error (and cpl_error
 * is set).
 */
/*----------------------------------------------------------------------------*/
double sph_test_get_key_double(const char* filename, unsigned int ext,
        const char* name) {
    double value = 0;
    cpl_propertylist* plist = NULL;

    plist = sph_keyword_manager_load_properties(filename, ext);
    if (cpl_propertylist_has(plist, name)) {
        value = cpl_propertylist_get_double(plist, name);
    }
    cpl_propertylist_delete(plist);
    plist = NULL;
    return value;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get a string keyword from file
 * @param filename
 * @param extension
 * @param keywordname
 *
 * @return returns the keyword value of the given keyword or 0 on error
 *
 * Returns the keyword value of the given keyword or 0 on error (and cpl_error
 * is set).
 */
/*----------------------------------------------------------------------------*/
const char*
sph_test_get_key_string(const char* filename, unsigned int ext,
        const char* name) {
    const char* value = NULL;
    cpl_propertylist* plist = NULL;

    plist = sph_keyword_manager_load_properties(filename, ext);
    if (cpl_propertylist_has(plist, name)) {
        value = cpl_propertylist_get_string(plist, name);
    }
    cpl_propertylist_delete(plist);
    plist = NULL;
    return value;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add a int keyword to the frame
 * @param frame
 * @param extension
 * @param keywordname
 * @param value
 *
 * Adds an integer type keyword to the given frame.
 * The keywordname is @em not passed through the keyword manager.
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_frame_add_key_int(cpl_frame* frame, unsigned int ext,
        const char* name, int value) {
    cpl_property* prop = cpl_property_new(name, CPL_TYPE_INT);

    if (!prop || !name || !frame) {
        return;
    }
    cpl_property_set_int(prop, value);

    sph_fits_update_property(cpl_frame_get_filename(frame), prop, ext);

    cpl_property_delete(prop);
    prop = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add a string keyword to the frame
 * @param frame
 * @param extension
 * @param keywordname
 * @param value
 *
 * Adds an string type keyword to the given frame.
 * The keywordname is @em not passed through the keyword manager.
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_frame_add_key_string(cpl_frame* frame, unsigned int ext,
        const char* name, const char* value) {
    cpl_property* prop = cpl_property_new(name, CPL_TYPE_STRING);

    if (!prop || !name || !frame) {
        return;
    }
    cpl_property_set_string(prop, value);

    sph_fits_update_property(cpl_frame_get_filename(frame), prop, ext);

    cpl_property_delete(prop);
    prop = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add a double keyword to the frame
 * @param frame
 * @param extension
 * @param keywordname
 * @param value
 *
 * Adds an double type keyword to the given frame.
 * The keywordname is @em not passed through the keyword manager.
 *
 */
/*----------------------------------------------------------------------------*/
void sph_test_frame_add_key_double(cpl_frame* frame, unsigned int ext,
        const char* name, double value) {
    cpl_property* prop = cpl_property_new(name, CPL_TYPE_DOUBLE);

    if (!prop || !name || !frame) {
        return;
    }
    cpl_property_set_double(prop, value);

    sph_fits_update_property(cpl_frame_get_filename(frame), prop, ext);

    cpl_property_delete(prop);
    prop = NULL;
}

void sph_test_frame_add_datestamp(cpl_frame* aframe) {
    struct tm * timestruct;
    time_t timet = -1;
    char date[256];
    struct timeval timev;
    int msec = 0;

    gettimeofday(&timev, NULL);
    timet = timev.tv_sec;
    timestruct = gmtime(&timet);
    msec = (int) timev.tv_usec / 100;
    sprintf(date, "%d-%d-%dT%d:%d:%d.%4d", timestruct->tm_year,
            timestruct->tm_mon, timestruct->tm_mday, timestruct->tm_hour,
            timestruct->tm_min, timestruct->tm_sec, msec);
    sph_test_frame_add_key_string(aframe, 0, "DATE", date);
    return;
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Initialise all sphere and CPL systems for testing.
 @return   error code

 Does some standard debug initialisation of error reporting systems for
 the Sphere API. If there is an error
 returns SPH_ERROR_GENERAL. In particular the routine does the following:
 <ol>
     <li> call sph_error_new <li>
 </ol>
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_nop_code(void) {

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    End all sphere and CPL systems.

 @return   error code

 Closes the Sphere API and CPL system.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_end_test(void)
{
  sph_filemanager_delete(1);
  sph_keyword_manager_delete();
  sph_error_delete();
  sph_polygon_free_all();

  return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Check the test results.


 This returns EXIT_SUCCESS if the last CUnit test suite has passed,
 otherwise EXIT_FAILURE.
 */
/*----------------------------------------------------------------------------*/

int sph_test_end(void) {
    int fpost_stat;

    cpl_test_error(CPL_ERROR_NONE);
    fpost_stat = (*sph_test_fpost)();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_zero(fpost_stat);

    return cpl_test_end(0);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Initialize the unit testing
 @param    filename The name of the source file
 @return   zero (indicating success)
 */
/*----------------------------------------------------------------------------*/
int sph_test_init_macro(const char * filename)
{

    cpl_test_init_macro(filename, PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    sph_test_fpost = NULL;

    sph_error_new();
    sph_error_set_reportlevel( 0 );
    sph_keyword_manager_new();

   return 0;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Call the pre-testing initializer, store the post-testing deallocator
 @param    suite  A string describing the test
 @param    fpre   The pre-testing initializer, of type int (0 for OK)
 @param    fpost  The post-testing deallocator, of type int (0 for OK)
 @return   non-NULL (indicating success)
 */
/*----------------------------------------------------------------------------*/
const void * sph_add_suite(const char* suite, 
                           int (*fpre)(void),
                           int (*fpost)(void))
{
    cpl_msg_info(cpl_func, "Suite: %s", suite);

    cpl_test_zero((*fpre)());

    sph_test_fpost = fpost;

    return (const void*)suite;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Call the testing function
 @param    null  NULL, not used
 @param    name  A string describing the test function
 @param    ftest The function with the test, of type void
 @return   non-NULL (indicating success)
 */
/*----------------------------------------------------------------------------*/
const void * sph_test_do(const void* null, const char* name,
                         void (*ftest)(void))
{
    cpl_msg_info(cpl_func, "Test: %s", name);

    (*ftest)();

    return (const void*)name;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Do nothing with an int
 @param    self The integer
 */
/*----------------------------------------------------------------------------*/
void sph_test_nop_int(int self)
{
    cpl_msg_debug(cpl_func, "%d", self);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Warn about missing testing
 @param    self The test to warn about
 */
/*----------------------------------------------------------------------------*/
void sph_test_later(const char * self)
{
    cpl_msg_warning(cpl_func, "%s", self);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Fail with a message regarding missing testing
 @param    self The test to warn about
 */
/*----------------------------------------------------------------------------*/
void sph_test_fail(const char * self)
{
    cpl_msg_warning(cpl_func, "%s", self);
    cpl_test_assert(self == NULL);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Do nothing with a string
 @param    self The integer
 */
/*----------------------------------------------------------------------------*/
void sph_test_nop_char(const char* self)
{
    cpl_msg_debug(cpl_func, "%s", self);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Finish testing
 */
/*----------------------------------------------------------------------------*/
int sph_test_get_error(void)
{
    return cpl_test_end(0);
}

/*----------------------------------------------------------------------------*/
/**
  @brief Test if two numerical expressions differ by a given absolute tolerance
  @param first     The first value in the comparison, side-effects are allowed
  @param second    The second value in the comparison, side-effects are allowed
  @param tolerance A tolerance
  @see   cpl_test_abs()

 */
/*----------------------------------------------------------------------------*/
void sph_test_noneq_abs(double first, double second, double tol)
{
    cpl_test_assert(fabs(tol) < fabs(first - second));
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the memory and the files of a frameset
 @param    self The frameset to delete
 */
/*----------------------------------------------------------------------------*/
void sph_test_frameset_delete(cpl_frameset* self)
{
    const cpl_frame* iframe = self ? cpl_frameset_get_first_const(self) : NULL;

    while (iframe != NULL) {
        const char * filename = cpl_frame_get_filename(iframe);
        const char * tag      = cpl_frame_get_tag(iframe);

        cpl_msg_debug(cpl_func, "Deleting(%s): %s", tag, filename);

        cpl_test_nonnull(filename);
        cpl_test_zero(unlink(filename));

        iframe = cpl_frameset_get_next_const(self);
    }
    cpl_test_null(iframe);
    cpl_frameset_delete(self);
}


/**@}*/
