/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_test_image_tools.h"
#include "sph_error.h"
#include "sph_time.h"
#include "sph_common_keywords.h"
#include <math.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new flat image
 * @param nx        size of image
 * @param ny        size of image
 * @param value        value to set image to
 *
 * @return new cpl_image of type CPL_TYPE_INT
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_image_tools_create_flat_image_int(int nx, int ny, int value) {
    cpl_image* result = cpl_image_new(nx, ny, CPL_TYPE_INT);
    int xx = 0;
    int yy = 0;
    if (!result) {
        return NULL;
    }

    for (xx = 0; xx < nx; ++xx) {
          for (yy = 0; yy < ny; ++yy) {
              cpl_image_set(result, xx + 1, yy + 1, value);
          }
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a striped image
 * @param nx        size of image
 * @param ny        size of image
 * @param nstripes  number of vertical stripes
 * @param type      the cpl_type
 * @param minvalue        value to set stripes to
 * @param maxvalue        value to set stripes to
 *
 * The resulting image will have nstripes vertical stripes of size
 * sx = nx / nstripes.
 * Value will be minvalue and maxvalue alternating.
 * @return new cpl_image of type given
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_image_tools_create_striped_image(int nx, int ny, int nstripes,
        cpl_type type, double minvalue, double maxvalue) {
    cpl_image* result = cpl_image_new(nx, ny, type);
    int xx = 0;
    int yy = 0;
    int sc = 1;
    int sdx = 0;
    int sx = 0;
    double value = minvalue;
    if (!result) {
        return NULL;
    }
    if (nstripes < 2 || nstripes % 2 != 0) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Must have more than 1 stripes and an even number of stripes.");
        return NULL;
    }
    sdx = (int) (nx / nstripes);
    for (xx = 0; xx < nx; ++xx) {
        if (sx > sdx) {
            sx = 0;
            sc = sc + 1;
            if (value == minvalue)
                value = maxvalue;
            else
                value = minvalue;
        }
        for (yy = 0; yy < ny; ++yy) {
            cpl_image_set(result, xx + 1, yy + 1, value);
        }
        sx++;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new cube from the image
 * @param the image that will be the first plane of the cube
 * @param filename      the filename
 * @param BPP           the bits per plane to use
 * @param pl            optional propertylist
 * @param mode          cpl writing mode
 * @return
 *
 *
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_image_tools_save_image_as_new_cube(cpl_image* im,
        const char* filename, cpl_type_bpp BPP, cpl_propertylist* pl,
        unsigned int mode) {
    cpl_imagelist* imlist = NULL;

    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);

    imlist = cpl_imagelist_new();
    cpl_imagelist_set(imlist, cpl_image_duplicate(im), 0);
    cpl_imagelist_save(imlist, filename, BPP, pl, mode);
    cpl_imagelist_delete(imlist);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create an image with a grid of points
 * @param nx        size of image
 * @param ny        size of image
 * @param npx  number of points in x
 * @param npy  number of points in y
 * @param type      the cpl_type
 * @param maxvalue        value to set points to
 * @param psize       the size of the points
 *
 * The resulting image will have npx points in x and npy points iny
 *
 * @return new cpl_image of type given
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_image_tools_create_grid_image(int nx, int ny, int npx, int npy,
        cpl_type type, double maxvalue, double psize) {
    cpl_image* result = cpl_image_new(nx, ny, type);
    int xx = 0;
    int yy = 0;
    int sdx = 0;
    int sdy = 0;
    if (!result) {
        return NULL;
    }
    sdx = (double) nx / (double) npx;
    sdy = (double) ny / (double) npy;
    for (xx = 1; xx <= npx; ++xx) {
        for (yy = 1; yy <= npy; ++yy) {
            sph_test_image_tools_add_gauss(result, xx * sdx, yy * sdy, psize,
                    maxvalue);
        }
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new flat image
 *
 * @param nx        size of image
 * @param ny        size of image
 * @param value        value to set image to
 *
 * @return new cpl_image of type CPL_TYPE_DOUBLE
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_test_image_tools_create_flat_image_double(int nx, int ny, double value) {
    cpl_image* result = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    if (!result) {
        return NULL;
    }

    for (int yy = 0; yy < ny; ++yy) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set(result, xx + 1, yy + 1, value);
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add badpixels to an image
 * @param self   The image to add badpixels to
 * @param nbads  The number of badpixels
 * @param pRNG   Pointer to gsl_rng
 *
 * @return 0 when all ok, otherwise -1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_image_tools_add_badpixles(cpl_image* self,
                                                  int        nbads,
                                                  gsl_rng*   pRNG)
{
    const cpl_size nx = cpl_image_get_size_x(self);
    const cpl_size ny = cpl_image_get_size_y(self);
    cpl_boolean remprng = CPL_FALSE;
    int ii;

    if (!pRNG) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        remprng = CPL_TRUE;
    }

    for (ii = 0; ii < nbads; ii++) {
        cpl_size x = gsl_ran_flat(pRNG, 0.0, (double)nx);
        cpl_size y = gsl_ran_flat(pRNG, 0.0, (double)ny);
        int bpix;
        do {
            x = gsl_ran_flat(pRNG, 0.0, (double)nx);
            y = gsl_ran_flat(pRNG, 0.0, (double)ny);
            bpix = cpl_image_is_rejected(self, x + 1, y + 1);
        } while (bpix);

        if (cpl_image_reject(self, x + 1, y + 1)) break;
    }

    if (remprng) {
        gsl_rng_free(pRNG);

    }

    return ii < nbads ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add dead (zero) pixels to an image
 * @param self   The image to add badpixels to
 * @param nbads  The number of badpixels
 * @param pRNG   Pointer to gsl_rng
 *
 * @return 0 when all ok, otherwise -1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_image_tools_add_deadpixels(cpl_image* self,
                                                  int        nbads,
                                                  gsl_rng*   pRNG)
{
    const cpl_size nx = cpl_image_get_size_x(self);
    const cpl_size ny = cpl_image_get_size_y(self);
    cpl_boolean remprng = CPL_FALSE;
    int ii;

    if (!pRNG) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        remprng = CPL_TRUE;
    }

    for (ii = 0; ii < nbads; ii++) {
        const cpl_size x   = gsl_ran_flat(pRNG, 0.0, (double)nx);
        const cpl_size y   = gsl_ran_flat(pRNG, 0.0, (double)ny);
        if (cpl_image_set(self, x + 1, y + 1, 0.0)) break;
    }

    if (remprng) {
        gsl_rng_free(pRNG);
    }

    return ii < nbads ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add hot pixels to an image all with the given value
 * @param self   The image to add badpixels to
 * @param nbads  The number of badpixels
 * @param hotval The hotvalue
 * @param pRNG   Pointer to gsl_rng
 *
 * @return 0 when all ok, otherwise -1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_image_tools_add_hotpixels(cpl_image* self,
                                                  int        nbads,
                                                  double     hotval,
                                                  gsl_rng*   pRNG)
{
    const cpl_size nx = cpl_image_get_size_x(self);
    const cpl_size ny = cpl_image_get_size_y(self);
    cpl_boolean remprng = CPL_FALSE;
    int ii;

    if (!pRNG) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        remprng = CPL_TRUE;
    }

    for (ii = 0; ii < nbads; ii++) {
        const cpl_size x   = gsl_ran_flat(pRNG, 0.0, (double)nx);
        const cpl_size y   = gsl_ran_flat(pRNG, 0.0, (double)ny);
        if (cpl_image_set(self, x + 1, y + 1, hotval)) break;
    }

    if (remprng) {
        gsl_rng_free(pRNG);
    }

    return ii < nbads ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add crazy pixels to an image with random values zero up to val
 * @param self   The image to add badpixels to
 * @param nbads  The number of badpixels
 * @param hotval The hotvalue
 * @param pRNG   Pointer to gsl_rng
 *
 * @return 0 when all ok, otherwise -1
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_test_image_tools_add_crazypixels(cpl_image* self,
                                                    int        nbads,
                                                    double     hotval,
                                                    gsl_rng*   pRNG)
{
    const cpl_size nx = cpl_image_get_size_x(self);
    const cpl_size ny = cpl_image_get_size_y(self);
    cpl_boolean remprng = CPL_FALSE;
    int ii;

    if (!pRNG) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        remprng = CPL_TRUE;
    }

    for (ii = 0; ii < nbads; ii++) {
        const double   val = gsl_ran_flat(pRNG, 0.0, hotval);
        const cpl_size x   = gsl_ran_flat(pRNG, 0.0, (double)nx);
        const cpl_size y   = gsl_ran_flat(pRNG, 0.0, (double)ny);
        if (cpl_image_set(self, x + 1, y + 1, val)) break;
    }

    if (remprng) {
        gsl_rng_free(pRNG);
    }

    return ii < nbads ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add gaussian noise to an image
 * @param inimage the image to add noise to
 * @param noise_sigma    the sigma of the noise
 *
 * @return 0 when all ok, otherwise -1
 *
 */
/*----------------------------------------------------------------------------*/
int sph_test_image_tools_add_noise(cpl_image* inimage, double noise_sigma,
        gsl_rng* pRNG) {
    double val = 0.0;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    int remprng = 0;
    if (!pRNG) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        remprng = 1;
    }
    if (inimage && pRNG) {
        for (xx = 0; xx < cpl_image_get_size_x(inimage); ++xx) {
            for (yy = 0; yy < cpl_image_get_size_y(inimage); ++yy) {
                val = cpl_image_get(inimage, xx + 1, yy + 1, &bpix);
                if (!bpix) {
                    val = val + gsl_ran_gaussian(pRNG, noise_sigma);
                    cpl_image_set(inimage, xx + 1, yy + 1, val);
                }
            }
        }
    } else {
        return -1;
    }
    if (remprng) {
        gsl_rng_free(pRNG);

    }
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Replace pixel values with random poisson samples
 * @param inimage the image to "noisify"
 * @param pRnG        pointer to the gsl random number generator
 *
 * @return 0 when all ok, otherwise -1
 *
 * This replaces the pixel values in the image with poisson values, where
 * the poisson mu value is the original pixel value.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_test_image_tools_apply_poisson_noise(cpl_image* inimage, gsl_rng* pRNG) {
    double val = 0.0;
    int oval = 0;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    int intern = 0;
    if (pRNG == NULL) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        intern = 1;
    }
    if (inimage && pRNG) {
        for (xx = 0; xx < cpl_image_get_size_x(inimage); ++xx) {
            for (yy = 0; yy < cpl_image_get_size_y(inimage); ++yy) {
                val = cpl_image_get(inimage, xx + 1, yy + 1, &bpix);
                if (!bpix) {
                    oval = gsl_ran_poisson(pRNG, val);
                    cpl_image_set(inimage, xx + 1, yy + 1, oval);
                }
            }
        }
    } else {
        return -1;
    }
    if (intern) {
        gsl_rng_free(pRNG);

    }
    return 0;
}
int sph_test_image_tools_add_in_window(cpl_image* inimage, int minx, int miny,
        int maxx, int maxy, double val) {
    int xx = 0;
    int yy = 0;
    int bpix = 0;

    if (minx < 1)
        minx = 1;
    if (miny < 1)
        miny = 1;
    if (maxx > cpl_image_get_size_x(inimage))
        maxx = cpl_image_get_size_x(inimage);
    if (maxy > cpl_image_get_size_y(inimage))
        maxy = cpl_image_get_size_y(inimage);
    for (xx = minx; xx <= maxx; ++xx) {
        for (yy = miny; yy <= maxy; ++yy) {
            cpl_image_set(inimage, xx, yy,
                    cpl_image_get(inimage, xx, yy, &bpix) + val);
        }
    }

    return 0;
}
int sph_test_image_tools_add_speckle(cpl_image* inimage, double xpos,
        double ypos, double size, double value) {
    return sph_test_image_tools_add_gauss(inimage, xpos, ypos, size, value);
}

int sph_test_image_tools_add_gauss(cpl_image* inimage, double dxpos,
        double dypos, double psize, double value) {
    int size0 = (int) psize * 20;
    int size;
    int xpos = (int) dxpos;
    int ypos = (int) dypos;
    int ypos0;
    int xpos0;
    int nx = cpl_image_get_size_x(inimage);
    int ny = cpl_image_get_size_y(inimage);
    double ddx;
    double ddy;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_image* smallim = NULL;
    cpl_image* extractim = NULL;

    cpl_ensure_code(inimage, CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NONE);
    if (size0 < 2) {
        SPH_ERR("Too small gaussian to add to image.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    size = size0;
    xpos0 = xpos - size * 0.5;
    ypos0 = ypos - size * 0.5;
    SPH_RAISE_CPL_RESET;
    if ((int) xpos0 < 0)
        xpos0 = 0;
    if ((int) ypos0 < 0)
        ypos0 = 0;

    if ((int) xpos0 + size >= nx) {
        size = nx - (int) xpos0;
    }
    if ((int) ypos0 + size >= ny)
        size = ny - (int) ypos0;
    ddx = dxpos - (int) (xpos0);
    ddy = dypos - (int) (ypos0);
    smallim = cpl_image_new(size, size, CPL_TYPE_DOUBLE);
    SPH_RAISE_CPL_RESET
    //SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"ddx=%f,ddy=%f",ddx,ddy);
    rerr = cpl_image_fill_gaussian(smallim, ddx, ddy, value, psize, psize);
    SPH_RAISE_CPL_RESET
    extractim = cpl_image_extract(inimage, (int) (xpos0) + 1, (int) (ypos0) + 1,
            (int) (xpos0) + size, (int) (ypos0) + size);
    if (extractim == NULL) {
        SPH_RAISE_CPL;SPH_ERROR_RAISE_ERR(
                SPH_ERROR_GENERAL,
                "Could not extract image size %d from %d,%d.", size, (int)xpos0, (int)ypos0);
        return SPH_ERROR_GENERAL;
    }
    rerr = cpl_image_add(extractim, smallim);
    SPH_RAISE_CPL_RESET
    rerr = cpl_image_copy(inimage, extractim, (int) (xpos0) + 1,
            (int) (ypos0) + 1);
    cpl_image_delete(smallim);
    smallim = NULL;
    cpl_image_delete(extractim);
    extractim = NULL;
    SPH_RAISE_CPL_RESET;
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add a gradient to an image
 * @param im  the input image (must be of type CPL_TYPE_FLOAT or CPL_TYPE_DOUBLE)
 * @param min   the minumum value (value added to first pixel column)
 * @param max   the maximum value (value added to last pixel column)
 * @return error code
 *
 * Adds a linear gradient from min to max inclusive to the image.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_image_tools_add_gradient_x(cpl_image* im, double min,
        double max) {
    int nx = 0;
    int ny = 0;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    double grad = 0.0;
    double val = 0.0;

    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(
            cpl_image_get_type(im) == CPL_TYPE_DOUBLE || cpl_image_get_type(im) == CPL_TYPE_FLOAT,
            CPL_ERROR_ILLEGAL_INPUT);
    nx = cpl_image_get_size_x(im);
    cpl_ensure_code(nx > 1, CPL_ERROR_ILLEGAL_INPUT);
    ny = cpl_image_get_size_y(im);
    for (xx = 0; xx < nx; ++xx) {
        grad = (max - min) / ((double) nx - 1.0) * (double) xx + min;
        for (yy = 0; yy < ny; ++yy) {
            val = cpl_image_get(im, xx + 1, yy + 1, &bpix);
            cpl_image_set(im, xx + 1, yy + 1, val + grad);
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Add a gradient to an image
 * @param im  the input image (must be of type CPL_TYPE_FLOAT or CPL_TYPE_DOUBLE)
 * @param min   the minumum value (value added to bottom pixel row)
 * @param max   the maximum value (value added to top pixel row)
 * @return error code
 *
 * Adds a linear gradient from min to max inclusive to the image.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_image_tools_add_gradient_y(cpl_image* im, double min,
        double max) {
    int nx = 0;
    int ny = 0;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    double grad = 0.0;
    double val = 0.0;

    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(
            cpl_image_get_type(im) == CPL_TYPE_DOUBLE || cpl_image_get_type(im) == CPL_TYPE_FLOAT,
            CPL_ERROR_ILLEGAL_INPUT);
    nx = cpl_image_get_size_x(im);
    ny = cpl_image_get_size_y(im);
    cpl_ensure_code(ny > 1, CPL_ERROR_ILLEGAL_INPUT);
    for (yy = 0; yy < ny; ++yy) {
        grad = (max - min) / ((double) ny - 1.0) * (double) yy + min;
        for (xx = 0; xx < nx; ++xx) {
            val = cpl_image_get(im, xx + 1, yy + 1, &bpix);
            cpl_image_set(im, xx + 1, yy + 1, val + grad);
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Multiply a gradient to an image
 * @param im  the input image (must be of type CPL_TYPE_FLOAT or CPL_TYPE_DOUBLE)
 * @param min   the minumum value (value multiplied at first pixel column)
 * @param max   the maximum value (value multiplied at last pixel column)
 * @return error code
 *
 * Multiply a linear gradient from min to max inclusive to the image.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_image_tools_multiply_gradient_x(cpl_image* im,
        double min, double max) {
    int nx = 0;
    int ny = 0;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    double grad = 0.0;
    double val = 0.0;

    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(
            cpl_image_get_type(im) == CPL_TYPE_DOUBLE || cpl_image_get_type(im) == CPL_TYPE_FLOAT,
            CPL_ERROR_ILLEGAL_INPUT);
    nx = cpl_image_get_size_x(im);
    cpl_ensure_code(nx > 1, CPL_ERROR_ILLEGAL_INPUT);
    ny = cpl_image_get_size_y(im);
    for (xx = 0; xx < nx; ++xx) {
        grad = (max - min) / ((double) nx - 1.0) * (double) xx + min;
        for (yy = 0; yy < ny; ++yy) {
            val = cpl_image_get(im, xx + 1, yy + 1, &bpix);
            cpl_image_set(im, xx + 1, yy + 1, val * grad);
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Multiply a gradient to an image
 * @param im  the input image (must be of type CPL_TYPE_FLOAT or CPL_TYPE_DOUBLE)
 * @param min   the minumum value (value multiplied at bottom pixel row)
 * @param max   the maximum value (value multiplied at top pixel row)
 * @return error code
 *
 * Multiply a linear gradient from min to max inclusive to the image.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_test_image_tools_multiply_gradient_y(cpl_image* im,
        double min, double max) {
    int nx = 0;
    int ny = 0;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    double grad = 0.0;
    double val = 0.0;
    cpl_ensure_code(im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(
            cpl_image_get_type(im) == CPL_TYPE_DOUBLE || cpl_image_get_type(im) == CPL_TYPE_FLOAT,
            CPL_ERROR_ILLEGAL_INPUT);
    nx = cpl_image_get_size_x(im);
    ny = cpl_image_get_size_y(im);
    cpl_ensure_code(ny > 1, CPL_ERROR_ILLEGAL_INPUT);
    for (yy = 0; yy < ny; ++yy) {
        grad = (max - min) / ((double) ny - 1.0) * (double) yy + min;
        for (xx = 0; xx < nx; ++xx) {
            val = cpl_image_get(im, xx + 1, yy + 1, &bpix);
            cpl_image_set(im, xx + 1, yy + 1, val * grad);
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Validate two images according to pixel values
 * @param image A
 * @param image B
 * @param maxdiff maximal allowed difference
 *
 * @return 1 if match, 0 otherwise.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_test_image_tools_validate_images_are_same(cpl_image* imA,
        cpl_image* imB, double maxdiff) {
    int result = 0;
    int xx = 0;
    int yy = 0;
    double val1 = 0.0;
    double val2 = 0.0;
    int rej1 = 0;
    int rej2 = 0;

    if (cpl_image_get_size_x(imA) != cpl_image_get_size_x(imB)) {
        result = 0;
    } else if (cpl_image_get_size_x(imA) != cpl_image_get_size_x(imB)) {
        result = 0;
    } else {
        result = 1;
        for (xx = 0; xx < cpl_image_get_size_x(imA); ++xx) {
            for (yy = 0; yy < cpl_image_get_size_y(imA); ++yy) {
                val1 = cpl_image_get(imA, xx + 1, yy + 1, &rej1);
                val2 = cpl_image_get(imB, xx + 1, yy + 1, &rej2);
                if (rej1 != rej2)
                    result = 0;
                if (fabs(val1 - val2) > maxdiff)
                    result = 0;
            }
        }
    }
    return result;
}
