/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ifs_tags.h"
#include "sph_test_ifs_instrument_model.h"
#include "sph_ifs_science_dr.h"
#include "sph_common_keywords.h"
#include "sph_test_image_tools.h"
#include "sph_fits.h"
#include "sph_frame_validator.h"
#include "sph_ifs_science.c"
#include "sph_time.h"
#include "sph_create_flat.c"
#include "sph_create_super_flat.c"
#include "sph_ifs_science_dr_run.c"
#include "sph_ifs_instrument_flat_run.c"
#include "sph_ifs_subtract_dark_scaled.c"
#include "sph_ifs_test_fixtures_science_raw.h"
#include "sph_ifs_test_fixtures_calib_products.h"

#include <math.h>

static
void add_ifs_science_keywords_to_frames(double mjd, char datestr[256],
		cpl_frame* mframe, cpl_frameset* rawframes) {
	while (mframe) {
		sph_test_frame_add_datestamp(mframe);
		mjd += 20.0;
		sph_test_frame_add_key_double(mframe, 0, SPH_COMMON_KEYWORD_MJD_OBS,
				mjd);
		sph_time_mjd_to_iso8601string(mjd + 100.0, datestr);
		sph_test_frame_add_key_string(mframe, 0,
		SPH_COMMON_KEYWORD_DET_FRAM_UTC, datestr);
		sph_test_frame_add_key_double(mframe, 0, SPH_COMMON_KEYWORD_EPOCH,
				2000.0);
		sph_test_frame_add_key_double(mframe, 0, SPH_COMMON_KEYWORD_DEC, 0.0);
		sph_test_frame_add_key_double(mframe, 0, SPH_COMMON_KEYWORD_RA, 0.0);
		sph_test_frame_add_key_string(mframe, 0,
		SPH_COMMON_KEYWORD_DROT2_MODE,
		SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB);
		mframe = cpl_frameset_get_next(rawframes);
	}
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

static sph_ifs_science_dr* cutest_setup_recipe(void) {
    sph_test_ngc_ir_simulator* ngc = NULL;
    sph_ifs_lenslet_model* lensmodel = NULL;
    sph_test_ifs_instrument_model* insmodel = NULL;
    cpl_frame* mframe = NULL;
    cpl_frameset* rawframes = NULL;
    double mjd = 0.0;
    char datestr[256];
    static sph_ifs_science_dr* srecipe = NULL;

    if (srecipe == NULL) {
        sph_time_start_timer(1);
        ngc = sph_test_ngc_ir_simulator_new(1024, 1024);
        cpl_test_nonnull( ngc );

        lensmodel = sph_ifs_lenslet_model_new();
        lensmodel->lenslets_per_side = 30;
        lensmodel->detsize_pixels = ngc->det_size_x;
        insmodel = sph_test_ifs_instrument_model_new_from_lenslet_model(
                lensmodel);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Time %f", sph_time_stop_timer(1));
        cpl_test_nonnull( insmodel );

        rawframes = sph_ifs_test_fixtures_science_raw_constant_flux(ngc,
                insmodel);

        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Time %f", sph_time_stop_timer(1));
        cpl_test_nonnull( rawframes );

        mframe = cpl_frameset_get_first(rawframes);
        sph_time_mjd_from_string(&mjd, "2000-01-01T00:00:00.00");

		add_ifs_science_keywords_to_frames(mjd, datestr, mframe, rawframes);

        mframe = sph_ifs_test_fixtures_calib_products_wave_calib(insmodel);

        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Time %f", sph_time_stop_timer(1));
        cpl_test_nonnull( mframe );
        cpl_frameset_insert(rawframes, mframe);

        srecipe = sph_ifs_science_dr_new(rawframes,
                sph_ifs_science_dr_create_paramlist());
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Time %f", sph_time_stop_timer(1));
    }
    cpl_test_nonnull( srecipe );
    sph_test_ngc_ir_simulator_delete(ngc);
    ngc = NULL;
    sph_test_ifs_instrument_model_delete(insmodel);
    insmodel = NULL;
    return srecipe;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_science_dr_run(void) {
    sph_ifs_science_dr* recipe = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_ldt* ldt = NULL;
    cpl_frameset* prodframes = NULL;
    int ll = 0;
    cpl_vector* vals = NULL;
    double rms = 0;
    FILE* fp = NULL;
    int pp = 0;
    recipe = cutest_setup_recipe();
    cpl_test_nonnull( recipe );
    rerr = sph_ifs_science_dr_run(recipe);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    prodframes = sph_utils_extract_frames(recipe->inframes,
            SPH_IFS_TAG_SCIENCE_DR_CALIB);
    cpl_test_eq(cpl_frameset_get_size(prodframes), 1);
    ldt = sph_ldt_load(
            cpl_frame_get_filename(cpl_frameset_get_first(prodframes)));

    cpl_test_assert( ldt);
    if (ldt) {
        fp = fopen("ldt_table.txt", "w");
        vals = cpl_vector_new(ldt->nlens);
        for (ll = 0; ll < ldt->nlens; ++ll) {
            cpl_vector_set(vals, ll,
                    sph_lenslet_descriptor_get_median(ldt->arr[ll], &rms));
            fprintf(fp, "%3d  %3d,%3d     %4.2f\n", ll, ldt->arr[ll]->u,
                    ldt->arr[ll]->v, cpl_vector_get(vals, ll));
            if (ldt->arr[ll]->u < 2 && ldt->arr[ll]->u > -2
                    && ldt->arr[ll]->v < 2 && ldt->arr[ll]->v > -2) {
                for (pp = 1; pp < cpl_vector_get_size(ldt->reflambdas); ++pp) {
                    cpl_test_abs(ldt->arr[ll]->values[pp],
                                 cpl_vector_get(vals, ll), 0.1);
                }
            }
        }
        fprintf(fp, "\n\n------------------------------------\n\n");
        fprintf(fp, "Mean median: %f\n", cpl_vector_get_mean(vals));
        fclose(fp);
        cpl_test_abs( cpl_vector_get_mean(vals), 1000.0, 0.00000001);
        cpl_vector_delete(vals);
        vals = NULL;
        sph_ldt_delete(ldt);
        ldt = NULL;
    }

    cpl_frameset_delete(prodframes);
    prodframes = sph_utils_extract_frames(recipe->inframes,
            SPH_IFS_TAG_SPEC_DECONV_CALIB);
    cpl_test_eq(cpl_frameset_get_size(prodframes), 1);
    cpl_frameset_delete(prodframes);
    prodframes = sph_utils_extract_frames(recipe->inframes,
            SPH_IFS_TAG_SIMPLE_ADI_CALIB);
    cpl_test_null( prodframes);

    cpl_parameterlist_delete(recipe->inparams);
    cpl_frameset_delete(recipe->inframes);
    cpl_frameset_delete(prodframes);
    sph_ifs_science_dr_delete(recipe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_ifs_science_dr_run module",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (cpl_msg_get_level() <= CPL_MSG_DEBUG) {
        if (NULL
            == sph_test_do(pSuite, "cutest_ifs_science_dr_run",
                           cutest_ifs_science_dr_run)) {
            return sph_test_get_error();
        }
    }

    return sph_test_end();
}

/**@}*/
