/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_common_keywords.h"
#include "sph_utils.h"
#include "sph_ifs_subtract_dark_scaled.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ifs_proc_apply_master_dark_no_dit_info(void) {
    sph_master_frame* mdark = NULL;
    sph_master_frame* rawframe = NULL;
    int nx = 256;
    int ny = 256;
    int bpix = 0;

    mdark = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(mdark->image, -100.0, 100.0);
    cpl_image_set(mdark->badpixelmap, 100, 100, 1);
    cpl_image_set(mdark->image, 100, 100, 100000000);

    rawframe = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(rawframe->image, -100.0, 100.0);
    cpl_image_set(rawframe->image, 100, 100, 100000000);

    cpl_test_eq(sph_ifs_subtract_dark_scaled(rawframe, mdark),
            CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(rawframe->badpixelmap,100,100,&bpix), 1);
    cpl_test_eq(sph_master_frame_get_nbads(rawframe), 1);
    cpl_test_abs( cpl_image_get_absflux(rawframe->image), 0.0,
            0.0001);

    sph_master_frame_delete(rawframe);
    rawframe = NULL;
    sph_master_frame_delete(mdark);
    mdark = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ifs_proc_apply_master_dark_dit_short_match(void) {
    sph_master_frame* mdark = NULL;
    sph_master_frame* rawframe = NULL;
    int nx = 256;
    int ny = 256;
    int bpix = 0;

    mdark = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(mdark->image, -100.0, 100.0);
    cpl_image_set(mdark->badpixelmap, 100, 100, 1);
    cpl_image_set(mdark->image, 100, 100, 100000000);
    cpl_propertylist_update_double(mdark->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            1.7);

    rawframe = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(rawframe->image, -100.0, 100.0);
    cpl_image_set(rawframe->image, 100, 100, 100000000);
    cpl_propertylist_update_double(rawframe->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            1.7);

    cpl_test_eq(sph_ifs_subtract_dark_scaled(rawframe, mdark),
            CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(rawframe->badpixelmap,100,100,&bpix), 1);
    cpl_test_eq(sph_master_frame_get_nbads(rawframe), 1);
    cpl_test_abs( cpl_image_get_absflux(rawframe->image), 0.0,
            0.0001);

    sph_master_frame_delete(rawframe);
    rawframe = NULL;
    sph_master_frame_delete(mdark);
    mdark = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ifs_proc_apply_master_dark_dit_short_no_matchA(void) {
    sph_master_frame* mdark = NULL;
    sph_master_frame* rawframe = NULL;
    cpl_error_code code;
    const int nx = 256;
    const int ny = 256;

    mdark = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(mdark->image, -100.0, 100.0);
    cpl_image_set(mdark->badpixelmap, 100, 100, 1);
    cpl_image_set(mdark->image, 100, 100, 100000000);
    cpl_propertylist_update_double(mdark->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            1.7);

    rawframe = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(rawframe->image, -100.0, 100.0);
    cpl_image_set(rawframe->image, 100, 100, 100000000);
    cpl_propertylist_update_double(rawframe->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            2.7);

    code = sph_ifs_subtract_dark_scaled(rawframe, mdark);
    /* FIXME: Changed expected error from CPL_ERROR_ILLEGAL_INPUT */
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    sph_master_frame_delete(rawframe);
    rawframe = NULL;
    sph_master_frame_delete(mdark);
    mdark = NULL;

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ifs_proc_apply_master_dark_dit_short_no_matchB(void) {
    sph_master_frame* mdark = NULL;
    sph_master_frame* rawframe = NULL;
    cpl_error_code code;
    const int nx = 256;
    const int ny = 256;

    mdark = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(mdark->image, -100.0, 100.0);
    cpl_image_set(mdark->badpixelmap, 100, 100, 1);
    cpl_image_set(mdark->image, 100, 100, 100000000);
    cpl_propertylist_update_double(mdark->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            2.7);

    rawframe = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(rawframe->image, -100.0, 100.0);
    cpl_image_set(rawframe->image, 100, 100, 100000000);
    cpl_propertylist_update_double(rawframe->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            1.7);

    code = sph_ifs_subtract_dark_scaled(rawframe, mdark);
    /* FIXME: Changed expected error from CPL_ERROR_ILLEGAL_INPUT */
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    sph_master_frame_delete(rawframe);
    rawframe = NULL;
    sph_master_frame_delete(mdark);
    mdark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ifs_proc_apply_master_dark_dit_long_no_match(void) {
    sph_master_frame* mdark = NULL;
    sph_master_frame* rawframe = NULL;
    int nx = 256;
    int ny = 256;
    int bpix = 0;
    cpl_image* im = NULL;

    mdark = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(mdark->image, -100.0, 100.0);
    cpl_image_set(mdark->badpixelmap, 100, 100, 1);
    cpl_image_set(mdark->image, 100, 100, 100000000);
    cpl_propertylist_update_double(mdark->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            5.7);

    rawframe = sph_master_frame_new(nx, ny);
    sph_test_image_tools_add_gradient_x(rawframe->image, -100.0, 100.0);
    cpl_image_multiply_scalar(rawframe->image, 2.0);
    cpl_image_set(rawframe->image, 100, 100, 100000000);
    cpl_propertylist_update_double(rawframe->properties, SPH_COMMON_KEYWORD_SEQ1DIT,
            11.4);

    cpl_test_eq(sph_ifs_subtract_dark_scaled(rawframe, mdark),
            CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(rawframe->badpixelmap,100,100,&bpix), 1);
    cpl_test_eq(sph_master_frame_get_nbads(rawframe), 1);
    im = sph_master_frame_extract_image(rawframe, 1);
    cpl_test_abs( cpl_image_get_absflux(im), 0.0, 0.01);
    cpl_image_delete(im);
    im = NULL;
    sph_master_frame_delete(rawframe);
    rawframe = NULL;
    sph_master_frame_delete(mdark);
    mdark = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testuite IFS apply dark", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing apply dark w/o DIT info",
                    cutest_sph_ifs_proc_apply_master_dark_no_dit_info)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing apply dark short dit (match)",
                    cutest_sph_ifs_proc_apply_master_dark_dit_short_match)) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing apply dark short dit (match)",
                    cutest_sph_ifs_proc_apply_master_dark_dit_short_no_matchA)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "Testing apply dark short dit (match)",
                    cutest_sph_ifs_proc_apply_master_dark_dit_short_no_matchB)) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing apply dark long dit (no match)",
                    cutest_sph_ifs_proc_apply_master_dark_dit_long_no_match)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
