/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_ird_andromeda.h"
#include "sph_ird_instrument_model.h"
#include "sph_test_ird_fixtures.h"
#include "sph_fctable.h"
#include "sph_ird_tags.h"

static
void cutest_sph_ird_andromeda_test_it(void);
static
int cutest_sph_ird_andromeda_init_testsuite(void);
static
int cutest_sph_ird_andromeda_clean_testsuite(void);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("A cutest_sph_ird_andromeda testsuite",
            cutest_sph_ird_andromeda_init_testsuite,
            cutest_sph_ird_andromeda_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing andromeda",
                    cutest_sph_ird_andromeda_test_it)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_andromeda_test_it(void) {
    sph_ird_andromeda* androm = NULL;
    cpl_frameset* rawframes = NULL;
    sph_ird_instrument_model* model = NULL;
    int ff = 0;
    int nf = 2;
    int nx = 256;
    int ny = 128;
    cpl_frame* psf_frame = NULL;
    cpl_image* check_image = NULL;
    cpl_image* psf_im = NULL;
    cpl_image* flat_im = NULL;
    cpl_frame* model_frame = NULL;
    cpl_propertylist* pl = NULL;
    sph_fctable* fctab = NULL;
    int pp = 0;
    cpl_apertures* aps = NULL;

    flat_im = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

    androm = cpl_calloc(1, sizeof(sph_ird_andromeda));
    model = sph_test_ird_fixtures_instrument_model(nx, ny);

    pl = sph_ird_instrument_model_get_as_propertylist(model);

    cpl_image_add_scalar(flat_im, 1.0);

    model_frame = sph_test_frame_image(flat_im, CPL_TYPE_DOUBLE,
            SPH_IRD_TAG_INSTRUMENT_MODEL, CPL_FRAME_GROUP_CALIB);

    cpl_image_save(flat_im, cpl_frame_get_filename(model_frame),
            CPL_TYPE_DOUBLE, pl, CPL_IO_CREATE);

    psf_im = cpl_image_new(nx / 2, ny, CPL_TYPE_DOUBLE);

    sph_test_image_tools_add_gauss(psf_im, nx / 4.0, ny / 2.0, 4.0, 1000.0);

    rawframes = cpl_frameset_new();
    androm->fcasciis = cpl_frameset_new();

    for (ff = 0; ff < nf; ++ff) {
        cpl_frameset_insert(
                rawframes,
                sph_test_ird_fixtures_create_science_frame_stable_fc_rot(10,
                        model, 0.0, 0.0, NULL, NULL, NULL, ff * 100.0, 10.0,
                        1.0));
        cpl_frameset_insert(
                androm->fcasciis,
                sph_test_ird_fixtures_create_corresponding_fc_frame(
                        cpl_frameset_get_position(rawframes,
                                cpl_frameset_get_size(rawframes) - 1), 1));
        fctab = sph_fctable_new();
        for (pp = 0; pp < 10; ++pp) {
            sph_fctable_add_row_double(fctab, pp * 0.01, nx / 4 + 0.5,
                    ny / 2 + 0.5, nx / 4 + 0.5, ny / 2 + 0.5,
                    ff * 100.0 + pp * 10.0);
        }
        sph_fctable_save_ascii(
                fctab,
                cpl_frame_get_filename(
                        cpl_frameset_get_position(androm->fcasciis,
                                cpl_frameset_get_size(androm->fcasciis) - 1)));
        sph_fctable_delete(fctab);
    }
    androm->flat_frame = model_frame;
    androm->inframes = rawframes;
    androm->rawframes = rawframes;
    androm->current_raw_frameset = rawframes;
    androm->andromeda_outfilename = "androm.fits";
    androm->leftlist_filename = "androm_left.fits";
    androm->rightlist_filename = "androm_right.fits";
    androm->only_prep = 0;
    androm->rhomin = 5.0;
    androm->rhomax = 50.0;
    androm->min_ang_dist = 5.0;
    androm->psf_frame = sph_test_frame_image(psf_im, CPL_TYPE_DOUBLE,
            SPH_IRD_TAG_PSF_REFERENCE_CALIB, CPL_FRAME_GROUP_CALIB);
    androm->psf_size = 8;
    androm->window_maxx = nx / 2;
    androm->window_maxy = ny;
    androm->window_minx = 1;
    androm->window_miny = 1;
    androm->inparams = sph_ird_andromeda_create_paramlist();
    sph_ird_andromeda_run(androm);

    androm->current_raw_frameset = NULL;
    androm->rawframes = NULL;
    androm->inframes = NULL;
    cpl_frame_delete(psf_frame);
    psf_frame = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_frameset_delete(androm->fcasciis);
    androm->fcasciis = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_frame_delete(model_frame);
    model_frame = NULL;
    cpl_image_delete(flat_im);
    flat_im = NULL;
    cpl_image_delete(psf_im);
    psf_im = NULL;

    check_image = cpl_image_load(androm->andromeda_outfilename, CPL_TYPE_DOUBLE,
            0, 0);
    cpl_test_nonnull( check_image );
    cpl_image_threshold(check_image, 0.0, DBL_MAX, 0.0, DBL_MAX);
    aps = cpl_apertures_extract_sigma(check_image, 5.0);
    cpl_test_eq(cpl_apertures_get_size(aps), 1);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Centroid: %f, %f", cpl_apertures_get_centroid_x(aps,1), cpl_apertures_get_centroid_y(aps,1));
    cpl_test_abs( cpl_apertures_get_centroid_x(aps,1),
            nx / 4.0 + 30.0, 2.0);
    cpl_test_abs( cpl_apertures_get_centroid_y(aps,1), nx / 4.0, 2.0);
    sph_ird_instrument_model_delete(model);
    model = NULL;
    cpl_frame_delete(androm->psf_frame);
    androm->psf_frame = NULL;
    cpl_parameterlist_delete(androm->inparams);
    androm->inparams = NULL;
    cpl_free(androm);
    androm = NULL;
    cpl_image_delete(check_image);
    check_image = NULL;
    cpl_apertures_delete(aps);
    aps = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_ird_andromeda_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_sph_ird_andromeda_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/**@}*/
