/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "cutest_zpl_master_bias.h"
#include "sph_zpl_master_bias.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_master_bias_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    sph_test_nop_code();
    return CPL_ERROR_NONE;
}

static int cutest_clean_master_bias_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

void cutest_zpl_master_bias_run_1(void) {
    sph_zpl_master_bias* zpl_master_bias = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 3;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    inparams = sph_zpl_master_bias_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_plus(&zexpcube, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_BIAS_PREPROC_CAM1);
    rerr = cpl_frameset_insert(inframes, zexpcube);
    sph_test_simulator_create_zplpreproc_cube_minus(&zexpcube1, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_BIAS_PREPROC_CAM2);
    rerr = cpl_frameset_insert(inframes, zexpcube1);
    // cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    zpl_master_bias = sph_zpl_master_bias_new(inframes, inparams);
    cpl_test_nonnull( zpl_master_bias );

    rerr = sph_zpl_master_bias_run(zpl_master_bias);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    if (rerr
            == CPL_ERROR_NONE && zpl_master_bias->master_bias_quadimage_cam1 != NULL) {cpl_test_abs( cpl_image_get_absflux(zpl_master_bias->master_bias_quadimage_cam1->zero_image->iframe->rmsmap)/(nx*ny), sqrt(2./(nframes-1.5)), 1e-5 );
}
    if (rerr
            == CPL_ERROR_NONE && zpl_master_bias->master_bias_quadimage_cam2 != NULL) {cpl_test_abs( cpl_image_get_absflux(zpl_master_bias->master_bias_quadimage_cam2->zero_image->iframe->rmsmap)/(nx*ny), sqrt(2./(nframes-1.5)), 1e-5 );
}

    rerr = sph_zpl_master_bias_delete(zpl_master_bias);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}
/*
 void cutest_zpl_master_bias_run_2(void) {
 sph_zpl_master_bias*        zpl_master_bias     = NULL;
 cpl_parameterlist*          inparams            = NULL;
 cpl_frameset*               inframes            = NULL;
 cpl_frame*                    zexpcube_minus        = NULL;
 cpl_frame*                    zexpcube_plus        = NULL;
 sph_error_code              rerr                = CPL_ERROR_NONE;
 int                            nframes                = 3;
 //int                         nx                    = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
 //int                         ny                    = 100; // -//-


 inparams = sph_zpl_master_bias_create_paramlist();
 inframes = cpl_frameset_new();

 // plus frame cube
 sph_test_simulator_create_zplpreproc_cube_plus( &zexpcube_plus, nframes );
 cpl_frame_set_tag( zexpcube_plus, SPH_ZPL_TAG_BIAS_PREPROC_RAW);
 rerr = cpl_frameset_insert( inframes, zexpcube_plus );

 //minus frame cube
 sph_test_simulator_create_zplpreproc_cube_minus( &zexpcube_minus, nframes );
 cpl_frame_set_tag( zexpcube_minus, SPH_ZPL_TAG_BIAS_PREPROC_RAW);
 rerr = cpl_frameset_insert( inframes, zexpcube_minus );

 zpl_master_bias = sph_zpl_master_bias_new( inframes, inparams );
 cpl_test_nonnull( zpl_master_bias  );

 rerr = sph_zpl_master_bias_run( zpl_master_bias );
 cpl_test_eq_error(rerr, CPL_ERROR_NONE );
 //printf("rms: %4.4f\n", cpl_image_get_absflux(zpl_master_bias->master_bias_quadimage->zero_image->iframe->rmsmap)/(nx*ny));
 //cpl_test_abs( cpl_image_get_absflux(zpl_master_bias->master_bias_quadimage->zero_image->iframe->rmsmap)/(nx*ny),
 //                      sqrt( ((169.-110.)*(169.-110.)+(170.-110.)*(170.-110.)+(171.-110.)*(171.-110.)+
 //                              (51.-110)*(51.-110.)+(50.-110)*(50.-110.)+(49.-110.)*(49.-110.))/6.), 1e-5 );

 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ), (169.+170.+171.+51.+50.+49.)/6, 1e-20 );
 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ), (29.+30.+31.+151.+150.+149.)/6, 1e-20 );

 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ),  (29.+30.+31.+151.+150.+149.)/6, 1e-20 );
 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ),(169.+170.+171.+51.+50.+49.)/6, 1e-20 );

 rerr = sph_zpl_master_bias_delete( zpl_master_bias );
 cpl_test_error(CPL_ERROR_NONE );

 //Destroy here input parameters and inframes explicitly
 cpl_frameset_delete( inframes );
 cpl_parameterlist_delete(inparams);

 return;
 }

 void cutest_zpl_master_bias_run_3(void) {
 sph_zpl_master_bias*        zpl_master_bias     = NULL;
 cpl_parameterlist*          inparams            = NULL;
 cpl_frameset*               inframes            = NULL;
 cpl_frame*                  zexpcube_minus      = NULL;
 cpl_frame*                  zexpcube_plus       = NULL;
 sph_error_code              rerr                = CPL_ERROR_NONE;
 int                         nz	                = 3;
 int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
 int                         ny                  = 100; // -//-
 int                     zero_odd    = 169;
 int                     zero_even   = 29;
 int                     pi_odd      = 29;
 int                     pi_even     = 169;
 double 					ovsc[8]		= {5.0, 1.0, 6.0, 2.0, 7.0, 3.0, 8.0, 4.0};
 //double 					ovsc[8]		= {1.0, 1.0, 1.0, 2.0, 0.0, 3.0, 0.0, 4.0};

 int                     zero_odd_1   = 49;
 int                     zero_even_1  = 149;
 int                     pi_odd_1     = 149;
 int                     pi_even_1    = 49;
 double 					ovsc_1[8]	 = {15.0, 1.0, 16.0, 2.0, 17.0, 3.0, 18.0, 4.0};
 //double 					ovsc_1[8]		= {0.0, 1.0, 0.0, 2.0, 0.0, 3.0, 0.0, 4.0};
 double 					adu1_zero = 0;
 double					adu2_zero = 0;


 inparams = sph_zpl_master_bias_create_paramlist();
 inframes = cpl_frameset_new();

 // zpl frame cube with overscan table
 zexpcube_plus = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
 "test_master_bias_zplexp_ovsc_cube",  zero_odd,  zero_even,  pi_odd, pi_even, ovsc,
 1, 1);
 cpl_frame_set_tag( zexpcube_plus, SPH_ZPL_TAG_BIAS_PREPROC_RAW);
 rerr = cpl_frameset_insert( inframes, zexpcube_plus );

 // zpl frame cube with overscan table
 zexpcube_minus = sph_test_simulator_create_zpl_exposure_overscans_cube( nx, ny, nz,
 "test_master_bias_zplexp_ovsc_cube_1",  zero_odd_1,  zero_even_1,  pi_odd_1,
 pi_even_1, ovsc_1, 1, 0 );
 cpl_frame_set_tag( zexpcube_minus, SPH_ZPL_TAG_BIAS_PREPROC_RAW);
 rerr = cpl_frameset_insert( inframes, zexpcube_minus );

 zpl_master_bias = sph_zpl_master_bias_new( inframes, inparams );
 cpl_test_nonnull( zpl_master_bias  );

 rerr = sph_zpl_master_bias_run( zpl_master_bias );
 cpl_test_eq_error(rerr, CPL_ERROR_NONE );

 adu1_zero = (169.-ovsc[0] +170.-(ovsc[0]+1)+171.-(ovsc[0]+2)+49.-ovsc_1[0]+50.-ovsc_1[0]+51.-ovsc_1[0])/6.;
 adu2_zero = (169.-ovsc[2] +170.-(ovsc[2]+1)+171.-(ovsc[2]+2)+49.-ovsc_1[2]+50.-ovsc_1[2]+51.-ovsc_1[2])/6.;

 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ), 0.5*(adu1_zero+adu2_zero), 1e-10);

 adu1_zero = (29.-ovsc[0] +30.-(ovsc[0]+1)+31.-(ovsc[0]+2)+149.-ovsc_1[0]+150.-ovsc_1[0]+151.-ovsc_1[0])/6.;
 adu2_zero = (29.-ovsc[2] +30.-(ovsc[2]+1)+31.-(ovsc[2]+2)+149.-ovsc_1[2]+150.-ovsc_1[2]+151.-ovsc_1[2])/6.;

 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ), 0.5*(adu1_zero+adu2_zero), 1e-10 );

 adu1_zero = (29.-ovsc[4] +30.-(ovsc[4]+1)+31.-(ovsc[4]+2)+149.-ovsc_1[4]+150.-ovsc_1[4]+151.-ovsc_1[4])/6.;
 adu2_zero = (29.-ovsc[6] +30.-(ovsc[6]+1)+31.-(ovsc[6]+2)+149.-ovsc_1[6]+150.-ovsc_1[6]+151.-ovsc_1[6])/6.;

 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ), 0.5*(adu1_zero+adu2_zero) , 1e-10 );

 adu1_zero = (169.-ovsc[4] +170.-(ovsc[4]+1)+171.-(ovsc[4]+2)+49.-ovsc_1[4]+50.-ovsc_1[4]+51.-ovsc_1[4])/6.;
 adu2_zero = (169.-ovsc[6] +170.-(ovsc[6]+1)+171.-(ovsc[6]+2)+49.-ovsc_1[6]+50.-ovsc_1[6]+51.-ovsc_1[6])/6.;

 cpl_test_abs( cpl_propertylist_get_double( zpl_master_bias->master_bias_quadimage->qclist,
 SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ), 0.5*(adu1_zero+adu2_zero), 1e-10 );



 rerr = sph_zpl_master_bias_delete( zpl_master_bias );
 cpl_test_error(CPL_ERROR_NONE );

 //Destroy here input parameters and inframes explicitly
 cpl_frameset_delete( inframes );
 cpl_parameterlist_delete(inparams);

 return;
 }

 */

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("master_bias_test", cutest_init_master_bias_testsuite,
            cutest_clean_master_bias_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "sph_bias_master_bias_run_1",
                    cutest_zpl_master_bias_run_1)) {
        return sph_test_get_error();
    }
    /*
     if ( NULL == sph_test_do(pSuite, "sph_bias_master_bias_run_2",
     cutest_zpl_master_bias_run_2 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_bias_master_bias_run_3: subtract overscan regions first",
     cutest_zpl_master_bias_run_3 ) )
     {
     return sph_test_get_error();
     }
     */

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
