/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cutest_zpl_science_p1.h"
#include "sph_zpl_science_p1.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_fctable.h"
#include "sph_zpl_science_p1_run.c"

#include <cpl.h>

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_zpl_science_p1"
#define SPH_CFG  SPH_BASE ".cfg"
#ifndef SPH_STD_ZPL_POL_HIGH_CATG
#define SPH_STD_ZPL_POL_HIGH_CATG "sph_zpl_polhigh_star_table.fits"
#endif

/*-----------------------------------------------------------------------------
 Prototypes
 -----------------------------------------------------------------------------*/
void cutest_zpl_science_p1_run_0(int argc, char* argv[]);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

#ifdef SPH_TEST_SCIENCE_P1
static
void cutest_zpl_science_create_head_file_hwpz_plus_0(const char* infile) {
    FILE* fp;
    char* outfilename = NULL;

    outfilename = sph_filemanager_filename_new_ext(infile, "head");

    fp = fopen(outfilename, "w");
    if (fp != NULL) {
        fprintf(fp, "ESO DRS ZPL HWPZ SETTING = 0.0\n");
        fclose(fp);
    }
    cpl_free(outfilename);
    outfilename = NULL;

}

static
void cutest_zpl_science_create_head_file_hwpz_minus_45(const char* infile) {
    FILE* fp;
    char* outfilename = NULL;

    outfilename = sph_filemanager_filename_new_ext(infile, "head");
    fp = fopen(outfilename, "w");
    if (fp != NULL) {
        fprintf(fp, "ESO DRS ZPL HWPZ SETTING = 45.0\n");
        fclose(fp);
    }
    cpl_free(outfilename);
    outfilename = NULL;

}

static sph_keyword_manager* cutest_zpl_science_create_cfg_file(void) {
    FILE* fp;
    sph_keyword_manager* km = NULL;

    fp = fopen(SPH_CFG, "w");
    fprintf(fp,
            "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.NAME = \"ESO DRS ZPL HWPZ SETTING\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.TYPE = \"DOUBLE\"\n");
    fprintf(fp, "SPH.KEYWORDS.ESO_DRS_ZPL_HWPZ_SETTING.VALUE = \"0.0\"\n");
    fclose(fp);

    km = sph_keyword_manager_new();
    cpl_test_nonnull(km);

    return km;
}
#endif

static int cutest_init_science_p1_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    sph_test_nop_code();
    return CPL_ERROR_NONE;
}


static int cutest_clean_science_p1_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_science_p1 with preproc frames but without keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_science_p1_run_1(void) {
    sph_zpl_science_p1* zpl_science_p1 = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    cpl_propertylist* pl = NULL;
    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure* zplexp_plane = NULL;

    int nframes = 5;
    //double dx[] = {1.0, 1.0, 1.0, 1.0, 1.0 };
    //double dy[] = {1.0, -1.0, 2.0, -2.0, 3.0 };
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };
    double mean[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double fullmean = 0.0;

    inparams = sph_zpl_science_p1_create_paramlist();
    inframes = cpl_frameset_new();

    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);
    //rerr = cpl_frameset_insert( inframes, fcframe );

    //simulate cube of rotated pre-processed (ZPL EXP) science frames
    sph_test_simulator_create_zplpreproc_science_cube(&zexpcube, fcframe, nx,
            ny);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_SCIENCE_P1_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    for (int ii = 0; ii < nframes; ii++) {
        zplexp_plane = sph_zpl_exposure_load(zexpcube, ii);
        mean[ii] = cpl_image_get_mean(zplexp_plane->image_zero_odd)
                + cpl_image_get_mean(zplexp_plane->image_zero_even)
                + cpl_image_get_mean(zplexp_plane->image_pi_odd)
                + cpl_image_get_mean(zplexp_plane->image_pi_even);
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean = fullmean + mean[ii];
        sph_zpl_exposure_delete(zplexp_plane);
    }

    fullmean = fullmean / nframes;
    //printf("fullmean %f = ", fullmean);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    //load fctable
    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fcframe));
    cpl_frame_delete(fcframe); fcframe = NULL;
    fname2 = sph_filemanager_filename_new_ext(cpl_frame_get_filename(zexpcube),
            "txt");
    fname = sph_filemanager_new_filename_from_base(fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename(frame_fc, fname);
    cpl_frame_set_tag(frame_fc, SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc, CPL_FRAME_GROUP_CALIB);

    //sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
    cpl_free(fname2);
    fname2 = NULL;
    cpl_free(fname);
    fname = NULL;
    sph_fctable_delete(fctable);
    fctable = NULL;


    rerr = cpl_frameset_insert(inframes, frame_fc);

    zpl_science_p1 = sph_zpl_science_p1_new(inframes, inparams);
    cpl_test_nonnull( zpl_science_p1 );

    rerr = sph_zpl_science_p1_run(zpl_science_p1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_science_p1->science_p1_plus_q_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            fullmean, 1e-3);
    //cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 280.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    //destroy filemanager and keywordmanager
    sph_filemanager_delete(0);
    sph_keyword_manager_delete();

    rerr = sph_zpl_science_p1_delete(zpl_science_p1);
    cpl_test_error(CPL_ERROR_NONE);

    //destroy inframes and inparams of the test
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);


    return;
}

#ifdef SPH_TEST_SCIENCE_P1
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_science_p1 with preproc frames but without keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_science_p1_run_1_1(void) {
    sph_zpl_science_p1* zpl_science_p1 = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    cpl_propertylist* pl = NULL;
    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure* zplexp_plane = NULL;
    cpl_table* fctab = NULL;
    cpl_propertylist* pl_fctab = NULL;

    int nframes = 5;
    //double dx[] = {1.0, 1.0, 1.0, 1.0, 1.0 };
    //double dy[] = {1.0, -1.0, 2.0, -2.0, 3.0 };
    double dx[] = {0.0, 0.0, 0.0, 0.0, 0.0};
    double dy[] = {0.0, 0.0, 0.0, 0.0, 0.0};
    double angle[] = {0, 15, 30, 45, 60};
    double mean[] = {0.0, 0.0, 0.0, 0.0, 0.0};

    inparams = sph_zpl_science_p1_create_paramlist();
    inframes = cpl_frameset_new();

    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);
    //rerr = cpl_frameset_insert( inframes, fcframe );

    //simulate cube of rotated pre-processed (ZPL EXP) science frames
    sph_test_simulator_create_zplpreproc_science_cube(&zexpcube, fcframe, nx, ny );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_SCIENCE_P1_PREPROC_CAM1 );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    double fullmean = 0.0;
    for ( int ii = 0; ii < nframes; ii++) {
        zplexp_plane = sph_zpl_exposure_load( zexpcube , ii);
        mean[ii] = cpl_image_get_mean( zplexp_plane->image_zero_odd )+
        cpl_image_get_mean( zplexp_plane->image_zero_even )+
        cpl_image_get_mean( zplexp_plane->image_pi_odd) +
        cpl_image_get_mean( zplexp_plane->image_pi_even );
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean = fullmean + mean[ii];
        sph_zpl_exposure_delete( zplexp_plane );
    }

    fullmean = fullmean/nframes;
    //printf("fullmean %f = ", fullmean);

    rerr = cpl_frameset_insert( inframes, zexpcube );

    //load fctable
    fctable = sph_fctable_load_ascii( cpl_frame_get_filename( fcframe ));
    fname2 = sph_filemanager_filename_new_ext( cpl_frame_get_filename(zexpcube), "txt");
//    fname2 = sph_filemanager_filename_new_ext( cpl_frame_get_filename(zexpcube), "fits");
    fname = sph_filemanager_new_filename_from_base( fname2, "fctable");
//    sph_fctable_save_ascii(fctable, fname);
    fctab = sph_fctable_get_as_cpl_table(fctable);
    pl_fctab = cpl_propertylist_new();
    cpl_propertylist_append_string( pl_fctab, SPH_COMMON_KEYWORD_PRO_FIRST_RAW_FRAME, cpl_frame_get_filename(zexpcube) );
    cpl_table_save( fctab, pl_fctab, pl_fctab, fname, CPL_IO_CREATE);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename( frame_fc, fname );
    cpl_frame_set_tag(frame_fc,SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc,CPL_FRAME_GROUP_CALIB);

    //sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
    cpl_free(fname2); fname2 = NULL;
    cpl_free(fname); fname = NULL;
    sph_fctable_delete(fctable); fctable = NULL;
    cpl_frame_delete( fcframe );

    rerr = cpl_frameset_insert( inframes, frame_fc );

    zpl_science_p1 = sph_zpl_science_p1_new( inframes, inparams);
    cpl_test_nonnull( zpl_science_p1 );

    rerr = sph_zpl_science_p1_run( zpl_science_p1 );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_science_p1->science_p1_outfilename_q_cam1, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), fullmean, 1e-3);
    //cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 280.0, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    rerr = sph_zpl_science_p1_delete( zpl_science_p1 );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_science_p1 with preproc frames but with .head files hwpz keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_science_p1_run_2(void) {
    sph_zpl_science_p1* zpl_science_p1 = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    cpl_propertylist* pl = NULL;
    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure* zplexp_plane = NULL;
    sph_keyword_manager* keyman = NULL;

    int nframes = 5;
    //double dx[] = {1.0, 1.0, 1.0, 1.0, 1.0 };
    //double dy[] = {1.0, -1.0, 2.0, -2.0, 3.0 };
    double dx[] = {0.0, 0.0, 0.0, 0.0, 0.0};
    double dy[] = {0.0, 0.0, 0.0, 0.0, 0.0};
    double angle[] = {0, 15, 30, 45, 60};
    double mean[] = {0.0, 0.0, 0.0, 0.0, 0.0};

    //create config
    keyman = cutest_zpl_science_create_cfg_file();
    cpl_test_nonnull( keyman );

    inparams = sph_zpl_science_p1_create_paramlist();
    inframes = cpl_frameset_new();

    //FIRST CUBE
    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);
    //rerr = cpl_frameset_insert( inframes, fcframe );

    //simulate cube of rotated pre-processed (ZPL EXP) science frames
    sph_test_simulator_create_zplpreproc_science_cube(&zexpcube, fcframe, nx, ny );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_SCIENCE_P1_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );

    double fullmean = 0.0;
    for ( int ii = 0; ii < nframes; ii++) {
        zplexp_plane = sph_zpl_exposure_load( zexpcube , ii);
        mean[ii] = cpl_image_get_mean( zplexp_plane->image_zero_odd )+
        cpl_image_get_mean( zplexp_plane->image_zero_even )+
        cpl_image_get_mean( zplexp_plane->image_pi_odd) +
        cpl_image_get_mean( zplexp_plane->image_pi_even );
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean = fullmean + mean[ii];
        sph_zpl_exposure_delete( zplexp_plane );
    }

    //create .head file for first zplexp cube
    cutest_zpl_science_create_head_file_hwpz_plus_0( cpl_frame_get_filename(zexpcube) );

    rerr = cpl_frameset_insert( inframes, zexpcube );

    //load fctable
    fctable = sph_fctable_load_ascii( cpl_frame_get_filename( fcframe ));
    fname2 = sph_filemanager_filename_new_ext( cpl_frame_get_filename(zexpcube), "txt");
    fname = sph_filemanager_new_filename_from_base( fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename( frame_fc, fname );
    cpl_frame_set_tag(frame_fc,SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc,CPL_FRAME_GROUP_CALIB);

    //sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
    cpl_free(fname2); fname2 = NULL;
    cpl_free(fname); fname = NULL;
    sph_fctable_delete(fctable); fctable = NULL;
    cpl_frame_delete( fcframe );

    rerr = cpl_frameset_insert( inframes, frame_fc );

    //SECOND CUBE
    angle[0] = 5.0; angle[1] = 25.0; angle[2] = 40.0; angle[3] = 60.0; angle[4] = 75.0;
    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);

    //simulate cube of rotated pre-processed (ZPL EXP) science frames
    sph_test_simulator_create_zplpreproc_science_cube(&zexpcube1, fcframe, nx, ny );
    cpl_frame_set_tag( zexpcube1, SPH_ZPL_TAG_SCIENCE_P1_PREPROC_RAW );
    cpl_frame_set_group( zexpcube1, CPL_FRAME_GROUP_RAW );

    //double fullmean = 0.0;
    for ( int ii = 0; ii < nframes; ii++) {
        zplexp_plane = sph_zpl_exposure_load( zexpcube1 , ii);
        mean[ii] = cpl_image_get_mean( zplexp_plane->image_zero_odd )+
        cpl_image_get_mean( zplexp_plane->image_zero_even )+
        cpl_image_get_mean( zplexp_plane->image_pi_odd) +
        cpl_image_get_mean( zplexp_plane->image_pi_even );
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean = fullmean + mean[ii];
        sph_zpl_exposure_delete( zplexp_plane );
    }

    fullmean = fullmean/(2*nframes);

    //create .head file for second zplexp cube1
    cutest_zpl_science_create_head_file_hwpz_minus_45(cpl_frame_get_filename(zexpcube1));

    rerr = cpl_frameset_insert( inframes, zexpcube1 );

    //load fctable
    fctable = sph_fctable_load_ascii( cpl_frame_get_filename( fcframe ));
    fname2 = sph_filemanager_filename_new_ext( cpl_frame_get_filename(zexpcube1), "txt");
    fname = sph_filemanager_new_filename_from_base( fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename( frame_fc, fname );
    cpl_frame_set_tag(frame_fc,SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc,CPL_FRAME_GROUP_CALIB);

    //sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
    cpl_free(fname2); fname2 = NULL;
    cpl_free(fname); fname = NULL;
    sph_fctable_delete(fctable); fctable = NULL;
    cpl_frame_delete( fcframe );

    rerr = cpl_frameset_insert( inframes, frame_fc );

    zpl_science_p1 = sph_zpl_science_p1_new( inframes, inparams);
    cpl_test_nonnull( zpl_science_p1 );

    rerr = sph_zpl_science_p1_run( zpl_science_p1 );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_science_p1->science_p1_outfilename, 0);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), fullmean, 1e-3);
    //cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 280.0, 1e-10);
    cpl_propertylist_delete( pl ); pl = NULL;

    rerr = sph_zpl_science_p1_delete( zpl_science_p1 );
    cpl_test_error(CPL_ERROR_NONE );

    sph_keyword_manager_delete();

    return;
}

#endif

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_science_p1 with preproc frames but without hwpz keywords
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_science_p1_run_3(void) {
    sph_zpl_science_p1* zpl_science_p1 = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    cpl_propertylist* pl = NULL;
    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure* zplexp_plane = NULL;

    double ovsc[8] = { 15.0, 1.0, 25.0, 2.0, 35.0, 3.0, 45.0, 4.0 };

    int nframes = 5;
    //double dx[] = {1.0, 1.0, 1.0, 1.0, 1.0 };
    //double dy[] = {1.0, -1.0, 2.0, -2.0, 3.0 };
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };
    double mean[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double fullmean = 0.0;

    inparams = sph_zpl_science_p1_create_paramlist();
    inframes = cpl_frameset_new();

    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);
    //rerr = cpl_frameset_insert( inframes, fcframe );

    //simulate cube of rotated pre-processed (ZPL EXP) science frames
    sph_test_simulator_create_zplpreproc_science_cube_overscan(&zexpcube,
            fcframe, nx, ny, ovsc);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_SCIENCE_P1_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    for (int ii = 0; ii < nframes; ii++) {
        zplexp_plane = sph_zpl_exposure_load(zexpcube, ii);
        mean[ii] = cpl_image_get_mean(zplexp_plane->image_zero_odd)
                + cpl_image_get_mean(zplexp_plane->image_zero_even)
                + cpl_image_get_mean(zplexp_plane->image_pi_odd)
                + cpl_image_get_mean(zplexp_plane->image_pi_even)
                - (ovsc[0] + ovsc[2] + ovsc[4] + ovsc[6]);
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean = fullmean + mean[ii];
        sph_zpl_exposure_delete(zplexp_plane);
    }

    fullmean = fullmean / nframes;
    //printf("fullmean %f = ", fullmean);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    //load fctable
    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fcframe));
    cpl_frame_delete(fcframe);

    fname2 = sph_filemanager_filename_new_ext(cpl_frame_get_filename(zexpcube),
            "txt");
    fname = sph_filemanager_new_filename_from_base(fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename(frame_fc, fname);
    cpl_frame_set_tag(frame_fc, SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc, CPL_FRAME_GROUP_CALIB);

    //sph_fctable_save_ascii(fctable, cpl_frame_get_filename(frame));
    cpl_free(fname2);
    fname2 = NULL;
    cpl_free(fname);
    fname = NULL;
    sph_fctable_delete(fctable);
    fctable = NULL;

    rerr = cpl_frameset_insert(inframes, frame_fc);
    //destroy filemanager and keywordmanager

    zpl_science_p1 = sph_zpl_science_p1_new(inframes, inparams);
    //cpl_test_nonnull( zpl_science_p1 );

    rerr = sph_zpl_science_p1_run(zpl_science_p1);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_science_p1->science_p1_plus_q_outfilename_cam2, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            fullmean, 1e-3);
    //cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), 280.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    sph_filemanager_delete(0);
    sph_keyword_manager_delete();

    rerr = sph_zpl_science_p1_delete(zpl_science_p1);
    cpl_test_error(CPL_ERROR_NONE);

    //destroy inframes and inparams of the test
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_science_p1 interaction with high polarimetry catalog
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_science_p1_run_0(int argc, char* argv[]) {
    cpl_test_lt(0, argc);
    char *base = cpl_strdup(argv[0]);
    char *end = strstr(base, "ZIMPOL");
    cpl_test_noneq_ptr(NULL, end);
    end[6] = 0;  // Truncate after the 'L' in "ZIMPOL"

    char *catfname = cpl_sprintf("%s/data/%s", base, SPH_STD_ZPL_POL_HIGH_CATG);
    cpl_frame *cat = sph_test_zpl_simulator_frame_template(catfname);

    if (!access(catfname, F_OK)) {
        double p = 42, theta = 3.14159;
        cpl_boolean result;

        // doesn't even get past filter check (and so shouldn't touch p or theta)
        result = try_extract_expected_from_catg(cat, "BogusFilter", 203, -71, &p, &theta);
        cpl_test_eq(CPL_FALSE, result);
        cpl_test_abs(42, p, 1e-6);
        cpl_test_abs(3.14159, theta, 1e-6);

        // filter V, spot on, with good p & theta values
        result = try_extract_expected_from_catg(cat, "V", 137.76, -44.8, &p, &theta);
        cpl_test_eq(CPL_TRUE, result);
        cpl_test_abs(2.6, p, 1e-6);
        cpl_test_abs(47.7, theta, 1e-6);

        // filter R (good) but not close enough
        result = try_extract_expected_from_catg(cat, "N_R", 203, -71, &p, &theta);
        cpl_test_eq(CPL_FALSE, result);

        // close enough with theta==NAN for filter I
        result = try_extract_expected_from_catg(cat, "N_I", 298.67, -15.99, &p, &theta);
        cpl_test_eq(CPL_TRUE, result);
        cpl_test_abs(2.24, p, 1e-6);
        cpl_test_eq(isnan(theta), 1);
    }

    cpl_frame_delete(cat);
    cpl_free(catfname);
    cpl_free(base);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_zpl_science_p1 recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[]) {
    const void* pSuite = NULL;


    (void)sph_test_init();
    (void)sph_keyword_manager_set_cfg(SPH_CFG);

    pSuite = sph_add_suite("science_p1_test", cutest_init_science_p1_testsuite,
            cutest_clean_science_p1_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }

    cutest_zpl_science_p1_run_0(argc, argv);

#ifdef SPH_TEST_SCIENCE_P1
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_science_p1_run_1: 2 cube-rawframes/preproc without keywords +Q/-Q and +U/-U",
                    cutest_zpl_science_p1_run_1)) {
        return sph_test_get_error();
    }

     if ( NULL == sph_test_do(pSuite, "sph_science_p1_run_1_1: 2 cube-rawframes/preproc without keywords +Q/-Q and +U/-U",
     cutest_zpl_science_p1_run_1_1 ) )
     {
     return sph_test_get_error();
     }

     if ( NULL == sph_test_do(pSuite, "sph_science_p1_run_2: 2 cube-rawframes/preproc with "
     ".head files containing keywords for +Q[hwp2=0.0] and -Q[hwp2=45.0] ",
     cutest_zpl_science_p1_run_2 ) )
     {
     return sph_test_get_error();
     }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_science_p1_run_3: 1 cube-rawframes/preproc with overscan tables",
                    cutest_zpl_science_p1_run_3)) {
        return sph_test_get_error();
    }
#endif

    return sph_test_end();
}

/**@}*/
