/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_point_pattern.h"
#include "sph_andromeda_support.h"
#include "sph_andromeda_core.h"
#include <math.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_sph_andromeda_core"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Other Helper Functions
 */
/*----------------------------------------------------------------------------*/
static cpl_imagelist* cutest_sph_andromeda_core_helper_create_imlist(int nangs,
        int nx, int ny, cpl_vector** angles, cpl_imagelist** varinlist) {
    cpl_image* imageA = NULL;
    cpl_image* imageB = NULL;
    cpl_imagelist* imlist = NULL;
    int ii = 0;
    double ang = 0.0;
    double dx = 0.0;
    double dy = 0.0;

    if (angles) {
        *angles = cpl_vector_new(nangs);
        cpl_test_nonnull( *angles );
    }
    if (varinlist) {
        *varinlist = cpl_imagelist_new();
        cpl_test_nonnull( *varinlist );
    }
    imlist = cpl_imagelist_new();
    cpl_test_nonnull( imlist );
    imageB = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(imageB, 128.5, 128.5, 1.0, 1.0, 1.0);

    for (ii = 0; ii < nangs; ++ii) {
        ang = ii * 15.7;
        dx = 10.0 * cos(CPL_MATH_RAD_DEG * ang);
        dy = 10.0 * sin(CPL_MATH_RAD_DEG * ang);

        imageA = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);

        if (angles)
            cpl_vector_set(*angles, ii, ang);

        cpl_image_fill_gaussian(imageA, 128.5 + dx, 128.5 + dy, 1.0, 1.0, 1.0);
        cpl_image_add(imageA, imageB); //sph_test_image_tools_add_gauss(imageA,128.0,128.0,1.0,1.0);
        cpl_imagelist_set(imlist, imageA, ii);
        if (varinlist) {
            imageA = sph_test_image_tools_create_flat_image_double(nx, ny, 0.1);
            cpl_imagelist_set(*varinlist, imageA, ii);
        }
    }
    cpl_image_delete(imageB);
    return imlist;
}
static sph_psfcube*
cutest_sph_andromeda_core_helper_setup_psfcube(void) {
    sph_psfcube* result = NULL;
    cpl_image* psfin = NULL;

    psfin = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(psfin, 16.0, 16.0, 1.0, 1.0, 1.0);

    result = sph_andromeda_support_calc_psf_shift_subpix(psfin, 10);
    cpl_image_delete(psfin);
    return result;
}

static sph_andromeda_core*
cutest_sph_andromeda_core_helper_setupcore(void) {
    cpl_vector* angles = NULL;
    cpl_image* psfin = NULL;
    cpl_bivector* indices = NULL;
    sph_psfcube* psfcube = NULL;
    sph_andromeda_core* core = NULL;
    cpl_imagelist* imlist = NULL;

    imlist = cutest_sph_andromeda_core_helper_create_imlist(2, 256, 256, NULL,
            NULL);
    cpl_test_nonnull( imlist );
    angles = cpl_vector_new(2);
    cpl_test_nonnull( angles );
    cpl_vector_set(angles, 0, 0.0);
    cpl_vector_set(angles, 1, 90.0);
    indices = cpl_bivector_wrap_vectors(cpl_vector_new(1), cpl_vector_new(1));
    cpl_test_nonnull( indices );
    cpl_vector_set(cpl_bivector_get_x(indices), 0, 0);
    cpl_vector_set(cpl_bivector_get_y(indices), 0, 1);

    psfin = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(psfin, 16.0, 16.0, 1.0, 1.0, 1.0);

    psfcube = sph_andromeda_support_calc_psf_shift_subpix(psfin, 10);
    cpl_test_nonnull( psfcube );

    core = sph_andromeda_support_init_core(imlist, angles, indices, psfcube);

    cpl_image_delete(psfin);

    return core;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_andromeda_core_get_flux_test1(void) {
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* diffims = NULL;
    cpl_imagelist* varlist = NULL;
    cpl_imagelist* varimlist = NULL;
    cpl_imagelist* weightimlist = NULL;
    cpl_vector* angles = NULL;
    cpl_bivector* indices = NULL;
    sph_psfcube* psfcube = NULL;
    cpl_image* result = NULL;
    cpl_vector* gamma = NULL;
    cpl_error_code code;

    imlist = cutest_sph_andromeda_core_helper_create_imlist(10, 256, 256,
            &angles, &varlist);
    cpl_test_nonnull( imlist );
    cpl_test_nonnull( varlist );
    cpl_test_nonnull( angles );
    indices = sph_andromeda_support_create_angpairs(angles, 0.1, NULL);
    cpl_test_nonnull( indices );
    psfcube = cutest_sph_andromeda_core_helper_setup_psfcube();

    diffims = sph_andromeda_support_angular_difference(imlist, varlist,
            cpl_bivector_get_x(indices), cpl_bivector_get_y(indices), 0.1, 50.0,
            0, &gamma, &varimlist, &weightimlist);

    result = sph_andromeda_core_get_flux(diffims, cpl_bivector_get_x(indices),
            cpl_bivector_get_y(indices), angles, psfcube, 0, weightimlist, 0.1,
            50.0, gamma, 0, NULL, NULL, NULL, NULL);

    code = cpl_image_save(result, SPH_BASE "_flux_test.fits", CPL_TYPE_DOUBLE,
                          NULL, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_zero(unlink(SPH_BASE "_flux_test.fits"));
    cpl_vector_delete(angles);
    cpl_vector_delete(gamma);
    cpl_bivector_delete(indices);
    cpl_imagelist_delete(imlist);
    cpl_imagelist_delete(diffims);
    cpl_imagelist_delete(varimlist);
    cpl_imagelist_delete(weightimlist);
    cpl_imagelist_delete(varlist);
    sph_psfcube_delete(psfcube);
    cpl_image_delete(result);
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    sph_andromeda_core* core;

    (void)sph_test_init();

    pSuite = sph_add_suite("Testing cutest_sph_andromeda_core",
                           cutest_init_testsuite, cutest_clean_testsuite);
    cpl_test_nonnull(pSuite);

    cpl_test_nonnull(sph_test_do(pSuite,
                                 "cutest_sph_andromeda_core_get_flux_test1",
                                 cutest_sph_andromeda_core_get_flux_test1));


    core = cutest_sph_andromeda_core_helper_setupcore();
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull(core);

    sph_andromeda_core_delete(core);

    return sph_test_end();
}

/**@}*/
