/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <float.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_create_flat.h"
#include "sph_test_image_tools.h"
#include "sph_master_frame.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_smart_imagelist.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat(void) {
    cpl_parameterlist* comb_params = NULL;
    cpl_frameset* rawframes = NULL;
    sph_master_frame* result = NULL;
    cpl_image* image = NULL;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    int xx = 0;
    int yy = 0;
    int bpix = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    rawframes = cpl_frameset_new();
    comb_params = cpl_parameterlist_new();
    sph_filemanager_new();
    for (ii = 0; ii < 3; ++ii) {
        image = sph_test_image_tools_create_flat_image_double(256, 256,
                (ii + 1) * 1000.0);
        sph_test_image_tools_add_noise(image, 2.0, pRNG);
        cpl_frameset_insert(
                rawframes,
                sph_test_frame_image(image, CPL_BPP_32_SIGNED, "NONE",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(image);
        image = NULL;
    }
    result = sph_create_flat(rawframes, 0, NULL, NULL, NULL, -100.0, 100.0,
            1000.0, NULL, NULL, 0, NULL);
    cpl_test_nonnull( result );
    for (ii = 0; ii < 100; ++ii) {
        xx = gsl_ran_flat(pRNG, 1.0, cpl_image_get_size_x(result->image));
        yy = gsl_ran_flat(pRNG, 1.0, cpl_image_get_size_y(result->image));
        cpl_test_abs(cpl_image_get(result->image,xx,yy,&bpix), 1.0,
                0.01);
        cpl_test_zero(bpix);
    }cpl_test_abs(sph_master_frame_get_mean(result,NULL), 1.0,
            0.0001);
    sph_master_frame_delete(result);
    result = NULL;
    gsl_rng_free(pRNG);
    pRNG = NULL;
    cpl_parameterlist_delete(comb_params);
    comb_params = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat_with_dark(void) {
    cpl_parameterlist* comb_params = NULL;
    cpl_frameset* rawframes = NULL;
    sph_master_frame* result = NULL;
    cpl_image* image = NULL;
    cpl_image* dark = NULL;
    sph_master_frame* dark_mframe = NULL;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    int xx = 0;
    int yy = 0;
    int bpix = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    rawframes = cpl_frameset_new();
    comb_params = cpl_parameterlist_new();
    dark = sph_test_image_tools_create_striped_image(256, 256, 6,
            CPL_TYPE_DOUBLE, 1000.0, 10000.0);
    for (ii = 0; ii < 3; ++ii) {
        image = sph_test_image_tools_create_flat_image_double(256, 256,
                (ii + 1) * 1000.0);
        cpl_image_add(image, dark);
        sph_test_image_tools_add_noise(image, 2.0, pRNG);
        cpl_frameset_insert(
                rawframes,
                sph_test_frame_image(image, CPL_BPP_32_SIGNED, "NONE",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(image);
        image = NULL;
    }
    dark_mframe = sph_master_frame_new_from_cpl_image(dark);
    cpl_image_set(dark_mframe->badpixelmap, 10, 10, 1.0);
    result = sph_create_flat(rawframes, 0, NULL, dark_mframe, NULL, -100.0,
            100.0, 1000.0, NULL, NULL, 0, NULL);
    cpl_test_nonnull( result );
    for (ii = 0; ii < 100; ++ii) {
        xx = gsl_ran_flat(pRNG, 1.0, cpl_image_get_size_x(result->image));
        yy = gsl_ran_flat(pRNG, 1.0, cpl_image_get_size_y(result->image));
        cpl_test_abs(cpl_image_get(result->image,xx,yy,&bpix), 1.0,
                0.01);
        cpl_test_zero(bpix);
    }cpl_test_abs(sph_master_frame_get_mean(result,NULL), 1.0,
            0.0001);
    cpl_test_noneq(cpl_image_get(result->badpixelmap,10,10,&bpix), 0);
    cpl_test_zero(cpl_image_get(result->badpixelmap,11,11,&bpix));
    sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(dark_mframe);
    dark_mframe = NULL;
    gsl_rng_free(pRNG);
    pRNG = NULL;
    cpl_parameterlist_delete(comb_params);
    comb_params = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_image_delete(dark);
    dark = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat_with_dark_robust(void) {
    cpl_parameterlist* comb_params = NULL;
    cpl_frameset* rawframes = NULL;
    sph_master_frame* result = NULL;
    cpl_image* image = NULL;
    cpl_image* dark = NULL;
    sph_master_frame* dark_mframe = NULL;
    int ii = 0;
    gsl_rng* pRNG = NULL;
    int xx = 0;
    int yy = 0;
    int bpix = 0;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    rawframes = cpl_frameset_new();
    comb_params = cpl_parameterlist_new();
    dark = sph_test_image_tools_create_striped_image(256, 256, 6,
            CPL_TYPE_DOUBLE, 1000.0, 10000.0);
    for (ii = 0; ii < 5; ++ii) {
        image = sph_test_image_tools_create_flat_image_double(256, 256,
                (ii + 1) * 1000.0);
        cpl_image_add(image, dark);
        sph_test_image_tools_add_noise(image, 2.0, pRNG);
        cpl_frameset_insert(
                rawframes,
                sph_test_frame_image(image, CPL_BPP_32_SIGNED, "NONE",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(image);
        image = NULL;
    }
    dark_mframe = sph_master_frame_new_from_cpl_image(dark);
    cpl_image_set(dark_mframe->badpixelmap, 10, 10, 1.0);
    result = sph_create_flat(rawframes, 1, NULL, dark_mframe, NULL, -100.0,
            100.0, 1000.0, NULL, NULL, 0, NULL);
    cpl_test_nonnull( result );
    sph_master_frame_save(result, "flat.fits", NULL);
    for (ii = 0; ii < 100; ++ii) {
        xx = gsl_ran_flat(pRNG, 1.0, cpl_image_get_size_x(result->image));
        yy = gsl_ran_flat(pRNG, 1.0, cpl_image_get_size_y(result->image));
        cpl_test_abs(cpl_image_get(result->image,xx,yy,&bpix), 1.0,
                0.01);
        cpl_test_zero(bpix);
    }cpl_test_abs(sph_master_frame_get_mean(result,NULL), 1.0, 0.001);
    cpl_test_noneq(cpl_image_get(result->badpixelmap,10,10,&bpix), 0);
    cpl_test_zero(cpl_image_get(result->badpixelmap,11,11,&bpix));
    sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(dark_mframe);
    dark_mframe = NULL;
    gsl_rng_free(pRNG);
    pRNG = NULL;
    cpl_parameterlist_delete(comb_params);
    comb_params = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_image_delete(dark);
    dark = NULL;
    cpl_error_reset();
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_create_flat_with_dark_lampflux_check(void) {
    cpl_parameterlist* comb_params = NULL;
    cpl_frameset* rawframes = NULL;
    sph_master_frame* result = NULL;
    cpl_image* image = NULL;
    cpl_image* dark = NULL;
    sph_master_frame* dark_mframe = NULL;
    int ii = 0;
    cpl_vector* lampflux = NULL;
    cpl_vector* lampcounts = NULL;
    cpl_vector* lampfluxstdev = NULL;
    gsl_rng* pRNG = NULL;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    rawframes = cpl_frameset_new();
    comb_params = cpl_parameterlist_new();
    dark = sph_test_image_tools_create_striped_image(256, 256, 6,
            CPL_TYPE_DOUBLE, 1000.0, 10000.0);
    for (ii = 0; ii < 3; ++ii) {
        image = sph_test_image_tools_create_flat_image_double(256, 256,
                (ii + 1) * 1000.0);
        cpl_image_add(image, dark);
        sph_test_image_tools_add_noise(image, 2.0, pRNG);
        cpl_frameset_insert(
                rawframes,
                sph_test_frame_image(image, CPL_BPP_32_SIGNED, "NONE",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(image);
        image = NULL;
    }
    dark_mframe = sph_master_frame_new_from_cpl_image(dark);
    cpl_image_set(dark_mframe->badpixelmap, 10, 10, 1.0);
    result = sph_create_flat(rawframes, 0, NULL, dark_mframe, NULL, -100.0,
            100.0, 1000.0, &lampflux, &lampcounts, 0, &lampfluxstdev);
    cpl_test_nonnull( lampflux );
    cpl_test_nonnull( lampcounts );
    cpl_test_nonnull( lampfluxstdev );
    cpl_test_nonnull( result );
    cpl_test_eq(cpl_vector_get_size(lampflux), 3);
    cpl_test_eq(cpl_vector_get_size(lampcounts), 3);
    for (ii = 0; ii < cpl_vector_get_size(lampflux); ++ii) {
        cpl_test_abs( cpl_vector_get(lampflux,ii), (ii + 1) * 1000.0,
                1.1);
        cpl_test_abs( cpl_vector_get(lampcounts,ii), (ii + 1) * 1000.0,
                1.1);
        cpl_test_abs( cpl_vector_get(lampfluxstdev,ii), 2.0, 1.1);
    }
    sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(dark_mframe);
    dark_mframe = NULL;
    cpl_parameterlist_delete(comb_params);
    comb_params = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_image_delete(dark);
    dark = NULL;
    gsl_rng_free(pRNG);
    pRNG = NULL;
    cpl_vector_delete(lampflux);
    lampflux = NULL;
    cpl_vector_delete(lampcounts);
    lampcounts = NULL;
    cpl_vector_delete(lampfluxstdev);
    lampfluxstdev = NULL;
    return;
}

static
void cutest_test_create_flat_pure_frames_5(void) {
    cpl_parameterlist* comb_params = NULL;
    cpl_frameset* rawframes = NULL;
    sph_master_frame* result = NULL;
    cpl_image* image = NULL;
    int ii = 0;
    int nframes = 5;

    rawframes = cpl_frameset_new();
    comb_params = cpl_parameterlist_new();
    sph_filemanager_new();
    for (ii = 0; ii < nframes; ++ii) {
        image = sph_test_image_tools_create_flat_image_double(256, 256,
                (2500.0 + ii * 10.0));
        cpl_frameset_insert(
                rawframes,
                sph_test_frame_image(image, CPL_BPP_32_SIGNED, "NONE",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(image);
        image = NULL;
    }
    result = sph_create_flat(rawframes, 0, NULL, NULL, NULL, -100.0, 100.0,
            1000.0, NULL, NULL, 0, NULL);
    cpl_test_nonnull( result );
    cpl_test_abs(sph_master_frame_get_mean(result,NULL), 1.0, 0.0001);
    cpl_test_abs( cpl_image_get_mean(result->rmsmap), 0.0, 0.000001);
    //small rms value
    cpl_test_abs( cpl_image_get_mean(result->badpixelmap), 0.0,
            0.000001);
    //no badpixel, mean badpixel must be 1.0

    sph_master_frame_save(result, "test_creat_flat_5.fits.tmp", NULL);

    sph_master_frame_delete(result);
    result = NULL;
    cpl_parameterlist_delete(comb_params);
    comb_params = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    return;
}

static
void cutest_test_create_flat_pure_frames_3_error(void) {
    cpl_parameterlist* comb_params = NULL;
    cpl_frameset* rawframes = NULL;
    sph_master_frame* result = NULL;
    cpl_image* image = NULL;
    int ii = 0;
    int nframes = 3;

    rawframes = cpl_frameset_new();
    comb_params = cpl_parameterlist_new();
    sph_filemanager_new();
    for (ii = 0; ii < nframes; ++ii) {
        image = sph_test_image_tools_create_flat_image_double(256, 256,
                (2500.0 + ii * 10.0));
        cpl_frameset_insert(
                rawframes,
                sph_test_frame_image(image, CPL_BPP_32_SIGNED, "NONE",
                        CPL_FRAME_GROUP_RAW));
        cpl_image_delete(image);
        image = NULL;
    }
    result = sph_create_flat(rawframes, 0, NULL, NULL, NULL, -100.0, 100.0,
            1000.0, NULL, NULL, 0, NULL);
    cpl_test_nonnull( result );
    //ASSERT with wrong huge rms = 1.0E10
    //printf("rmsmap: %f10 \n", cpl_image_get_mean(result->rmsmap));
    //cpl_test_abs( cpl_image_get_mean(result->rmsmap), 1E10, 0.0001); //huge rms!!!
    cpl_test_abs( cpl_image_get_mean(result->rmsmap), 0.0, 0.0001);
    //rms is not 0.0 within tolerance
    cpl_test_abs( cpl_image_get_mean(result->badpixelmap), 0.0,
            0.000001);
    //mean badpixel must be 1.0

    sph_master_frame_save(result, "test_creat_flat_3.fits.tmp", NULL);

    sph_master_frame_delete(result);
    result = NULL;
    cpl_parameterlist_delete(comb_params);
    comb_params = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Creation of flat", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "simple test case", cutest_test_create_flat)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "test case with dark",
                    cutest_test_create_flat_with_dark)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "test case with dark robust",
                    cutest_test_create_flat_with_dark_robust)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "test case with dark, lamp flux output",
                    cutest_test_create_flat_with_dark_lampflux_check)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "simple test case with pure flat 5 images: produces correct result",
                    cutest_test_create_flat_pure_frames_5)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "simple test case with pure flat 3 images: produces error (inverse verification to make it run) ",
                    cutest_test_create_flat_pure_frames_3_error)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
