/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/


#include "sph_double_image.h"
#include "sph_common_keywords.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include <cpl.h>

#include <math.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_master_frame Unit test for the sph_master_frame module
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

static
int cutest_init_sph_double_image_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;

}

static
int cutest_clean_sph_double_image_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_double_image_new_empty function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_new_empty(void) {
    sph_double_image* di = NULL;

    di = sph_double_image_new_empty();
    cpl_test_nonnull( di );
    sph_double_image_delete(di);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_double_image_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_new(void) {
    sph_double_image* di = NULL;
    const int nx = 100;
    const int ny = 200;

    di = sph_double_image_new(nx, ny);
    cpl_test_nonnull( di );

    cpl_test_eq(cpl_image_get_size_x( di->iframe->image ), nx);
    cpl_test_eq(cpl_image_get_size_y( di->pframe->badpixelmap ), ny);

    sph_double_image_delete(di);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_quality_check
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_quality_check(void) {
    sph_double_image* di = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 200;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;

    //double image with no-noise
    di = sph_test_zpl_simulator_create_double_image(nx, ny, iframe_value,
            pframe_value);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            21.5, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            50.9, 1e-20);
    sph_double_image_save(di, "test_sph_double_image.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_double_image_save(di, "test_sph_double_image_noise.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);

    //load saved double image with noise and test qclist
    di = sph_double_image_load("test_sph_double_image_noise.fits.tmp", 0);
    cpl_test_nonnull( di );
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            iframe_value, 3*sigma_iframe/sqrt(nx*ny));
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_IFRAME ),
            sigma_iframe, 3*sigma_iframe/sqrt(2.0*nx*ny));
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            pframe_value, 3*sigma_pframe/sqrt(nx*ny));
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_PFRAME ),
            sigma_pframe, 3*sigma_pframe/sqrt(2.0*nx*ny));
    sph_double_image_delete(di);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_quality_check_keysuffix
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_quality_check_keysuffix(void) {
    sph_double_image* di = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 200;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    char   qkeyname[1024];

    //double image with no-noise
    di = sph_test_zpl_simulator_create_double_image(nx, ny, iframe_value,
            pframe_value);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check_keysuffix(di, "QPLUS");
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, "QPLUS" );
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, qkeyname ),
            21.5, 1e-20);
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, "QPLUS" );
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, qkeyname ),
            50.9, 1e-20);
    sph_double_image_save(di, "test_sph_double_image.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check_keysuffix(di, "QMINUS");
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_double_image_save(di, "test_sph_double_image_noise.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);

    //load saved double image with noise and test qclist
    di = sph_double_image_load("test_sph_double_image_noise.fits.tmp", 0);
    cpl_test_nonnull( di );

    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, "QMINUS" );
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, qkeyname ),
            iframe_value, 3*sigma_iframe/sqrt(nx*ny));
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_IFRAME, "QMINUS" );
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, qkeyname ),
            sigma_iframe, 3*sigma_iframe/sqrt(2.0*nx*ny));
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, "QMINUS" );
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, qkeyname ),
            pframe_value, 3*sigma_pframe/sqrt(nx*ny));
    strcpy(qkeyname, "");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_PFRAME, "QMINUS" );
    cpl_test_abs(
            cpl_propertylist_get_double ( di->qclist, qkeyname ),
            sigma_pframe, 3*sigma_pframe/sqrt(2.0*nx*ny));
    sph_double_image_delete(di);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_quality_check
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_duplicate(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_dupl = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 200;
    double iframe_value = 21.5;
    double pframe_value = 50.9;

    //double image with no-noise
    di = sph_test_zpl_simulator_create_double_image(nx, ny, iframe_value,
            pframe_value);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            21.5, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            50.9, 1e-20);
    sph_double_image_save(di, "test_sph_double_image.fits.tmp", NULL,
            CPL_IO_CREATE);

    di_dupl = sph_double_image_duplicate(di);
    sph_double_image_delete(di);

    cpl_test_abs(
            cpl_propertylist_get_double( di_dupl->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            21.5, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( di_dupl->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            50.9, 1e-20);
    cpl_test_eq(cpl_image_get_size_x( di_dupl->iframe->image ), nx);
    cpl_test_eq(cpl_image_get_size_y( di_dupl->iframe->image ), ny);

    sph_double_image_delete(di_dupl);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_expand2y
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_expand2y(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_copy = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 200;
    int ny = 100;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    int nxnew, nynew, pis_rejected;
    double val_expand, val_orig;

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    sph_double_image_save(di, "test_sph_double_image_noise.fits.tmp", NULL,
            CPL_IO_CREATE);
    di_copy = sph_double_image_duplicate(di);

    rerr = sph_double_image_expand2y(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    nxnew = cpl_image_get_size_x(di->iframe->image);
    nynew = cpl_image_get_size_y(di->iframe->image);
    cpl_test_eq(nxnew, nx);
    cpl_test_eq(nynew, ny*2);
    for (int xx = 0; xx < nxnew; ++xx) {
        for (int yy = 0; yy < nynew; ++yy) {
            val_expand = cpl_image_get(di->iframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            cpl_test_abs(val_expand, val_orig, 1e-20);
            val_expand = cpl_image_get(di->pframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
        }
    }

    sph_double_image_save(di, "test_sph_double_image_noise_expand2y.fits.tmp",
            NULL, CPL_IO_CREATE);
    sph_double_image_delete(di);
    sph_double_image_delete(di_copy);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_topbottom_lines
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_topbottom_lines(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_copy = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 200;
    int ny = 100;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    int nxnew, nynew, pis_rejected;
    double val_expand, val_orig;
    cpl_image* im = NULL;
    cpl_image* im1 = NULL;
    cpl_image* im2 = NULL;
    int llx, lly, urx, ury;

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    sph_double_image_save(di, "test_sph_double_image_noise.fits.tmp", NULL,
            CPL_IO_CREATE);
    di_copy = sph_double_image_duplicate(di);

    rerr = sph_double_image_expand2y(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    nxnew = cpl_image_get_size_x(di->iframe->image);
    nynew = cpl_image_get_size_y(di->iframe->image);
    cpl_test_eq(nxnew, nx);
    cpl_test_eq(nynew, ny*2);
    for (int xx = 0; xx < nxnew; ++xx) {
        for (int yy = 0; yy < nynew; ++yy) {
            val_expand = cpl_image_get(di->iframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            cpl_test_abs(val_expand, val_orig, 1e-20);
            val_expand = cpl_image_get(di->pframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
        }
    }

    sph_double_image_save(di, "test_sph_double_image_noise_expand2y.fits.tmp",
            NULL, CPL_IO_CREATE);

    //add top and bottom lines to the generated double image
    sph_double_image_add_topbottom_lines(di);

    //iframe
    llx = 1;
    lly = 1;
    urx = cpl_image_get_size_x(di->iframe->image);
    ury = 1;
    im = cpl_image_extract(di->iframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = 2;
    urx = cpl_image_get_size_x(di->iframe->image);
    ury = 2;
    im1 = cpl_image_extract(di->iframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = 3;
    urx = cpl_image_get_size_x(di->iframe->image);
    ury = 3;
    im2 = cpl_image_extract(di->iframe->image, llx, lly, urx, ury);

    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im1 ), 1e-10);
    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im2 ), 1e-10);

    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im1);
    im1 = NULL;
    cpl_image_delete(im2);
    im2 = NULL;

    llx = 1;
    lly = cpl_image_get_size_y(di->iframe->image);
    urx = cpl_image_get_size_x(di->iframe->image);
    ury = cpl_image_get_size_y(di->iframe->image);
    im = cpl_image_extract(di->iframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = cpl_image_get_size_y(di->iframe->image) - 1;
    urx = cpl_image_get_size_x(di->iframe->image);
    ury = cpl_image_get_size_y(di->iframe->image) - 1;
    im1 = cpl_image_extract(di->iframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = cpl_image_get_size_y(di->iframe->image) - 2;
    urx = cpl_image_get_size_x(di->iframe->image);
    ury = cpl_image_get_size_y(di->iframe->image) - 2;
    im2 = cpl_image_extract(di->iframe->image, llx, lly, urx, ury);

    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im1 ), 1e-10);
    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im2 ), 1e-10);
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im1);
    im1 = NULL;
    cpl_image_delete(im2);
    im2 = NULL;

    //pframe
    llx = 1;
    lly = 1;
    urx = cpl_image_get_size_x(di->pframe->image);
    ury = 1;
    im = cpl_image_extract(di->pframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = 2;
    urx = cpl_image_get_size_x(di->pframe->image);
    ury = 2;
    im1 = cpl_image_extract(di->pframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = 3;
    urx = cpl_image_get_size_x(di->pframe->image);
    ury = 3;
    im2 = cpl_image_extract(di->pframe->image, llx, lly, urx, ury);

    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im1 ), 1e-10);
    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im2 ), 1e-10);

    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im1);
    im1 = NULL;
    cpl_image_delete(im2);
    im2 = NULL;

    llx = 1;
    lly = cpl_image_get_size_y(di->pframe->image);
    urx = cpl_image_get_size_x(di->pframe->image);
    ury = cpl_image_get_size_y(di->pframe->image);
    im = cpl_image_extract(di->pframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = cpl_image_get_size_y(di->pframe->image) - 1;
    urx = cpl_image_get_size_x(di->pframe->image);
    ury = cpl_image_get_size_y(di->pframe->image) - 1;
    im1 = cpl_image_extract(di->pframe->image, llx, lly, urx, ury);

    llx = 1;
    lly = cpl_image_get_size_y(di->pframe->image) - 2;
    urx = cpl_image_get_size_x(di->pframe->image);
    ury = cpl_image_get_size_y(di->pframe->image) - 2;
    im2 = cpl_image_extract(di->pframe->image, llx, lly, urx, ury);

    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im1 ), 1e-10);
    cpl_test_abs( cpl_image_get_absflux( im ),
            cpl_image_get_absflux( im2 ), 1e-10);
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im1);
    im1 = NULL;
    cpl_image_delete(im2);
    im2 = NULL;

    sph_double_image_save(di,
            "test_sph_double_image_noise_expand2y_addtopbottom.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);
    sph_double_image_delete(di_copy);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_cutedge_columns
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_cutedge_columns(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_copy = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 200;
    int ny = 100;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    int nxnew, nynew, pis_rejected;
    double val_expand, val_orig;
    cpl_image* im = NULL;
    int llx, lly, urx, ury;

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    sph_double_image_save(di, "test_sph_double_image_noise.fits.tmp", NULL,
            CPL_IO_CREATE);
    di_copy = sph_double_image_duplicate(di);

    rerr = sph_double_image_expand2y(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    nxnew = cpl_image_get_size_x(di->iframe->image);
    nynew = cpl_image_get_size_y(di->iframe->image);
    cpl_test_eq(nxnew, nx);
    cpl_test_eq(nynew, ny*2);
    for (int xx = 0; xx < nxnew; ++xx) {
        for (int yy = 0; yy < nynew; ++yy) {
            val_expand = cpl_image_get(di->iframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            cpl_test_abs(val_expand, val_orig, 1e-20);
            val_expand = cpl_image_get(di->pframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
        }
    }

    sph_double_image_save(di, "test_sph_double_image_noise_expand2y.fits.tmp",
            NULL, CPL_IO_CREATE);

    sph_double_image_delete(di_copy);
    di_copy = NULL;
    di_copy = sph_double_image_duplicate(di);

    sph_double_image_cutedge_columns(di);
    llx = 2;
    lly = 1;
    urx = cpl_image_get_size_x(di_copy->iframe->image) - 1;
    ury = cpl_image_get_size_y(di_copy->iframe->image);
    im = cpl_image_extract(di_copy->iframe->image, llx, lly, urx, ury);

    cpl_test_abs( cpl_image_get_absflux( di->iframe->image ),
            cpl_image_get_absflux( im ), 1e-10);
    cpl_image_delete(im);
    im = NULL;

    im = cpl_image_extract(di_copy->pframe->image, llx, lly, urx, ury);

    sph_double_image_save(di,
            "test_sph_double_image_noise_expand2y_cutedge.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);
    sph_double_image_delete(di_copy);
    cpl_image_delete(im);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_quality_check
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_ext_check(void) {
    sph_double_image* di = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 100;
    int ny = 200;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    char str[1024];
    const char* iframe = "IFRAME";
    const char* pframe = "PFRAME";

    //double image with no-noise
    di = sph_test_zpl_simulator_create_double_image(nx, ny, iframe_value,
            pframe_value);
    cpl_test_nonnull( di );
    rerr = sph_double_image_quality_check(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME ),
            21.5, 1e-20);
    cpl_test_abs(
            cpl_propertylist_get_double( di->qclist, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME ),
            50.9, 1e-20);
    sph_double_image_save(di, "test_sph_double_image.fits.tmp", NULL,
            CPL_IO_CREATE);
    sph_double_image_delete(di);

    //check extnames of the double image extensions saved in double image
    sprintf(str, "%s.%s", iframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_IMAGE_EXTNAME);
    cpl_test_eq( 0,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", iframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_BADPIXMAP_EXTNAME);
    cpl_test_eq( 1,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", iframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_NCOMBMAP_EXTNAME);
    cpl_test_eq( 2,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", iframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_RMSMAP_EXTNAME);
    cpl_test_eq( 3,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", pframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_IMAGE_EXTNAME);
    cpl_test_eq( 4,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", pframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_BADPIXMAP_EXTNAME);
    cpl_test_eq( 5,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", pframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_NCOMBMAP_EXTNAME);
    cpl_test_eq( 6,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));
    sprintf(str, "%s.%s", pframe,
            SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_RMSMAP_EXTNAME);
    cpl_test_eq( 7,
            cpl_fits_find_extension("test_sph_double_image.fits.tmp", str));

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_create_squared
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_create_squared(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_copy = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 20;
    int  ny = 9;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    int nxnew, nynew;

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    sph_double_image_save(di, "test_sph_double_image_noise_sq.fits.tmp", NULL,
            CPL_IO_CREATE);
    di_copy = sph_double_image_duplicate(di);

    //rerr = sph_double_image_expand2y(di);
    rerr = sph_double_image_create_squared(di);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    nxnew = cpl_image_get_size_x(di->iframe->image);
    nynew = cpl_image_get_size_y(di->iframe->image);
    cpl_test_eq(nxnew, nx);
    cpl_test_eq(nynew, ny*2 + 2);
    /*
    for (int xx = 0; xx < nxnew; ++xx) {
        for (int yy = 0; yy < nynew + 2; ++yy) {
            val_expand = cpl_image_get(di->iframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            cpl_test_abs(val_expand, val_orig, 1e-20);
            val_expand = cpl_image_get(di->pframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
        }
    }
    */
    sph_double_image_save(di, "test_sph_double_image_noise_squared.fits.tmp",
            NULL, CPL_IO_CREATE);
    sph_double_image_delete(di);
    sph_double_image_delete(di_copy);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_interpolate_y
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_interpolate_y_zpl_imaging(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_copy = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 20;
    int ny = 10;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    int nxnew, nynew;

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    sph_double_image_save(di, "test_sph_double_image_noise_yint.fits.tmp", NULL,
            CPL_IO_CREATE);
    di_copy = sph_double_image_duplicate(di);

    //rerr = sph_double_image_expand2y(di);
    rerr = sph_double_image_interpolate_y_zpl_imaging(di, CPL_FALSE);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    nxnew = cpl_image_get_size_x(di->iframe->image);
    nynew = cpl_image_get_size_y(di->iframe->image);
    cpl_test_eq(nxnew, nx);
    cpl_test_eq(nynew, ny*2 );

    //printf("orig=%f, interp=%f \n", cpl_image_get_flux( di_copy->iframe->image),
    //        cpl_image_get_flux( di->iframe->image), 1e-10 );

    cpl_test_abs( cpl_image_get_flux( di_copy->iframe->image ),
                            cpl_image_get_flux( di->iframe->image) +
                            cpl_image_get_flux_window( di->iframe->image, 1, nynew, nxnew, nynew)/2.0,
                            1e-10 );

    //printf("orig win=%f\n", cpl_image_get_flux_window( di_copy->iframe->image, 1, 1, nx, 2 ) );
    //printf("interp win=%f\n", cpl_image_get_flux_window( di->iframe->image, 1, 1, nxnew, 3 ) );

/*
    for (int yy = 0; yy < nynew; yy=yy + 2) {
        for (int xx = 0; xx < nxnew; ++xx) {
            val_expand = cpl_image_get(di->iframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            cpl_test_abs(val_expand*2.0, val_orig, 1e-20);
            val_expand = cpl_image_get(di->pframe->image, xx + 1, yy + 1,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            cpl_test_abs(val_expand*2.0, val_orig, 1e-20);

        }
    }

    for (int yy = 0; yy < nynew - 2; yy=yy + 2) {
        for (int xx = 0; xx < nxnew; ++xx) {
            val_expand = cpl_image_get(di->iframe->image, xx + 1, yy + 2,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            val_orig_b = cpl_image_get(di_copy->iframe->image, xx + 1, yy / 2 + 2,
                    &pis_rejected);
            cpl_test_abs(val_expand, val_orig/4. + val_orig_b/4., 1e-20);
            val_expand = cpl_image_get(di->pframe->image, xx + 1, yy + 2,
                    &pis_rejected);
            val_orig = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 1,
                    &pis_rejected);
            val_orig_b = cpl_image_get(di_copy->pframe->image, xx + 1, yy / 2 + 2,
                    &pis_rejected);
            cpl_test_abs(val_expand, val_orig/4. + val_orig_b/4., 1e-20);

        }
    }
*/
    sph_double_image_save(di, "test_sph_double_image_noise_yinterpolated.fits.tmp",
            NULL, CPL_IO_CREATE);
    sph_double_image_delete(di);
    sph_double_image_delete(di_copy);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_double_image_interpolate_y
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_double_image_interpolate_y_zpl_polarimetry(void) {
    sph_double_image* di = NULL;
    sph_double_image* di_copy = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;
    int nx = 20;
    int ny = 10;
    double iframe_value = 21.5;
    double pframe_value = 50.9;
    double sigma_iframe = 1.0;
    double sigma_pframe = 5.0;
    int nxnew, nynew;

    //double image with noise
    di = sph_test_zpl_simulator_create_double_image_noise_uniform(nx, ny,
            iframe_value, pframe_value, sigma_iframe, sigma_pframe);
    cpl_test_nonnull( di );
    sph_double_image_save(di, "test_sph_double_image_noise_yint_p.fits.tmp", NULL,
            CPL_IO_CREATE);
    di_copy = sph_double_image_duplicate(di);

    rerr = sph_double_image_interpolate_y_zpl_polarimetry(di, CPL_FALSE);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    nxnew = cpl_image_get_size_x(di->iframe->image);
    nynew = cpl_image_get_size_y(di->iframe->image);
    cpl_test_eq(nxnew, nx);
    cpl_test_eq(nynew, ny*2 + 2);

    //printf("orig=%f, interp=%f \n", cpl_image_get_flux( di_copy->iframe->image),
    //        cpl_image_get_flux( di->iframe->image), 1e-10 );

    cpl_test_abs( cpl_image_get_flux( di_copy->iframe->image ),
                            cpl_image_get_flux( di->iframe->image),
                            1e-10 );

    cpl_test_abs( cpl_image_get_flux( di_copy->pframe->image ),
                            cpl_image_get_flux( di->pframe->image),
                            1e-10 );


    //printf("orig win=%f\n", cpl_image_get_flux_window( di_copy->iframe->image, 1, 1, nx, 2 ) );
    //printf("interp win=%f\n", cpl_image_get_flux_window( di->iframe->image, 1, 1, nxnew, 3 ) );

    sph_double_image_save(di, "test_sph_double_image_noise_yinterpolated_p.fits.tmp",
            NULL, CPL_IO_CREATE);
    sph_double_image_delete(di);

    di = sph_double_image_duplicate(di_copy);

    rerr = sph_double_image_interpolate_y_zpl_polarimetry(di, CPL_TRUE);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    cpl_test_abs( cpl_image_get_flux( di_copy->iframe->image ),
                            cpl_image_get_flux( di->iframe->image) +
                            cpl_image_get_flux_window( di_copy->iframe->image, 1, 1, nx, 1)/4.0 +
                            cpl_image_get_flux_window( di_copy->iframe->image, 1, ny, nx, ny)/4.0,
                            1e-10 );
    sph_double_image_delete(di);
    sph_double_image_delete(di_copy);

    return;
}


/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("cutest_sph_double_image",
            cutest_init_sph_double_image_testsuite,
            cutest_clean_sph_double_image_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite, "sph_double_image_new_empty",
                    cutest_sph_double_image_new_empty)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "sph_double_image_new",
                    cutest_sph_double_image_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_double_image_quality_check",
                    cutest_sph_double_image_quality_check)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_double_image_quality_check_keysuffix",
                    cutest_sph_double_image_quality_check_keysuffix)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_double_image_duplicate",
                    cutest_sph_double_image_duplicate)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_double_image_expand2y",
                    cutest_sph_double_image_expand2y)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_double_image_add_topbottom_lines",
                    cutest_sph_double_image_topbottom_lines)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_double_image_cutedge_columns",
                    cutest_sph_double_image_cutedge_columns)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_sph_double_image_ext_check",
                    cutest_sph_double_image_ext_check)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_sph_double_image_create_squared",
                    cutest_sph_double_image_create_squared ))  {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_sph_double_image_interpolate_y_zpl_imaging",
                    cutest_sph_double_image_interpolate_y_zpl_imaging ))  {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(pSuite, "cutest_sph_double_image_interpolate_y_zpl_polarimetry",
                    cutest_sph_double_image_interpolate_y_zpl_polarimetry ))  {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
