/* $Id: $
 *
 * This file is part of the SPHERE Pipesph_fft
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_fft.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"

#include <cpl.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <float.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/
#define SPH_BASE "cutest_sph_fft"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_sph_fft_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_sph_fft_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_new(void) {
    sph_fft* sphfft = NULL;

    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );
    sph_fft_delete(sphfft);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_rotate_prepare(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im, 128, 128, 10.0, 50.0, 50.0);
    cpl_test_nonnull( im );
    sph_fft_rotate_prepare(sphfft, im, 90.0, 0.0, 0.0);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_abs( sphfft->angle, 1.57, 0.1);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_forward_inverse_radix2(void) {
    sph_fft* sphfft = NULL;
    cpl_image* imim = NULL;
    cpl_image* imreal = NULL;
    cpl_image* imin = NULL;
    double val = 0.0;
    cpl_error_code code;
    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    imin = sph_test_image_tools_create_striped_image(256, 256, 256,
            CPL_TYPE_DOUBLE, 0.0, 1.0);
    cpl_test_nonnull( imin );

    code = sph_fft_prepare(sphfft, imin);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_fft_crop_complex_array(sphfft, 256, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull( sphfft->complexdata );
    code = sph_fft_forward(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );
    code = cpl_image_save(imreal, SPH_BASE "_fft_forward_inverse.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(imim, SPH_BASE "_fft_forward_inverse.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    code = sph_fft_inverse(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );
    code = cpl_image_save(imreal, SPH_BASE "_fft_forward_inverse.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_image_save(imim, SPH_BASE "_fft_forward_inverse.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_image_subtract(imin, imreal);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    val = cpl_image_get_absflux(imin);
    cpl_test_lt(val, 1e-10);

    cpl_test_zero(unlink(SPH_BASE "_fft_forward_inverse.fits"));

    cpl_image_delete(imin);
    imin = NULL;
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_forward_inverse_mixedradix(void) {
    sph_fft* sphfft = NULL;
    cpl_image* imim = NULL;
    cpl_image* imreal = NULL;
    cpl_image* imin = NULL;
    double val = 0.0;
    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    imin = sph_test_image_tools_create_striped_image(256, 256, 256,
            CPL_TYPE_DOUBLE, 0.0, 1.0);
    cpl_test_nonnull( imin );

    sph_fft_prepare(sphfft, imin);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_nonnull( sphfft->wavetab );
    cpl_test_nonnull( sphfft->workspace );
    sph_fft_forward(sphfft);

    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );

    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    sph_fft_inverse(sphfft);
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );

    cpl_image_subtract(imin, imreal);
    val = cpl_image_get_absflux(imin);
    cpl_test_lt(val, 1e-10);

    cpl_image_delete(imin);
    imin = NULL;
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_pad_real_space(void) {
    sph_fft* sphfft = NULL;
    cpl_image* imim = NULL;
    cpl_image* imreal = NULL;
    cpl_image* imin = NULL;
    double val = 0.0;
    cpl_error_code code;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);
    cpl_test_nonnull( sphfft );

    imin = sph_test_image_tools_create_striped_image(256, 256, 256,
            CPL_TYPE_DOUBLE, 0.0, 1.0);
    cpl_test_nonnull( imin );

    code = sph_fft_prepare(sphfft, imin);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_nonnull( sphfft->wavetab );
    cpl_test_nonnull( sphfft->workspace );
    code = sph_fft_pad_complex_array_real_space(sphfft, 312);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_image_delete(imin);
    imin = NULL;
    imin = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imin );
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );
    code = cpl_image_save(imreal, SPH_BASE "_pad_real_space.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(imim, SPH_BASE "_pad_real_space.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    code = sph_fft_forward(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_fft_inverse(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );
    code = cpl_image_save(imreal, SPH_BASE "_pad_real_space.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(imim, SPH_BASE "_pad_real_space.fits",
                          CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = cpl_image_subtract(imin, imreal);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    val = cpl_image_get_absflux(imin);
    cpl_test_lt(val, 1e-10);

    cpl_test_zero(unlink(SPH_BASE "_pad_real_space.fits"));

    cpl_image_delete(imin);
    imin = NULL;
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_pad_freq_space(void) {
    sph_fft* sphfft = NULL;
    cpl_image* imim = NULL;
    cpl_image* imreal = NULL;
    cpl_image* imin = NULL;
    double val = 0.0;
    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    imin = sph_test_image_tools_create_striped_image(256, 256, 256,
            CPL_TYPE_DOUBLE, 0.0, 1.0);
    cpl_test_nonnull( imin );
    sph_fft_prepare(sphfft, imin);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_nonnull( sphfft->wavetab );
    cpl_test_nonnull( sphfft->workspace );
    sph_fft_forward(sphfft);

    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );

    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;
    sph_fft_pad_complex_array_freq_space(sphfft, 512);
    sph_fft_crop_complex_array(sphfft, 256, 1);
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );

    sph_fft_inverse(sphfft);
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );

    cpl_image_subtract(imin, imreal);
    val = cpl_image_get_absflux(imin);
    cpl_test_lt(val, 1e-10);

    cpl_image_delete(imin);
    imin = NULL;
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_set_complex(void) {
    sph_fft* sphfft = NULL;
    cpl_image* imim = NULL;
    cpl_image* imreal = NULL;
    cpl_image* imin = NULL;
    double val = 0.0;
    int xx = 0;
    int yy = 0;
    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    imin = sph_test_image_tools_create_flat_image_double(256, 256, 1.0);
    for (xx = 0; xx < 256; ++xx) {
        for (yy = 0; yy < 256; ++yy) {
            if (xx % 2 && yy % 2)
                cpl_image_set(imin, xx + 1, yy + 1, 0.0);
        }
    }cpl_test_nonnull( imin );
    sph_fft_prepare(sphfft, imin);
    cpl_test_nonnull( sphfft->complexdata );
    sph_fft_forward(sphfft);
    for (xx = -256; xx < 256; ++xx) {
        for (yy = -256; yy < 256; ++yy) {
            if (xx * xx + yy * yy > 20000)
                sph_fft_set_complex(sphfft, xx, yy, 0.0, 0.0);
        }
    }
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );

    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;
    sph_fft_inverse(sphfft);
    imreal = sph_fft_complex2image_real(sphfft);
    cpl_test_nonnull( imreal );
    imim = sph_fft_complex2image_imag(sphfft);
    cpl_test_nonnull( imim );
    cpl_image_delete(imim);
    imim = NULL;
    imim = cpl_image_extract(imreal, 129, 129, 384, 384);
    cpl_image_subtract_scalar(imim, cpl_image_get_mean(imim));

    val = cpl_image_get_max_window(imim, 15, 15, 240, 240);
    cpl_test_lt(val, 0.05);
    val = cpl_image_get_min_window(imim, 15, 15, 240, 240);
    cpl_test_lt(-0.05, val);
    val = cpl_image_get_mean_window(imim, 15, 15, 240, 240);
    cpl_test_lt(val, 0.05);

    cpl_image_delete(imin);
    imin = NULL;
    cpl_image_delete(imreal);
    imreal = NULL;
    cpl_image_delete(imim);
    imim = NULL;

    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_shift(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    cpl_error_code code;
    double val = 0.0;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im, 128.0, 128.0, 10.0, 2.5, 2.5);
    im2 = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im2, 130.0, 129.0, 10.0, 2.5, 2.5);

    cpl_test_nonnull( im );
    cpl_test_nonnull( im2 );
    sph_fft_prepare(sphfft, im);
    sph_fft_forward(sphfft);
    sph_fft_shift(sphfft, 2.0, 1.0);
    sph_fft_inverse(sphfft);
    tim = sph_fft_complex2image_real(sphfft);

    cpl_test_nonnull( tim );
    code = cpl_image_save(im, SPH_BASE "_shift.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_shift.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(tim, SPH_BASE "_shift.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_subtract(im2, tim);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_shift.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    val = cpl_image_get_absflux(im2);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_mean(im2);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_max(im2);
    cpl_test_lt(val, 1e-10);

    cpl_test_zero(unlink(SPH_BASE "_shift.fits"));

    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im2);
    im2 = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_get_scale_vector(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    int xx = 0;
    int yy = 0;
    double val = 0.0;
    double sigma = 0.0;
    cpl_error_code code;

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);
    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( im );

    for (xx = 0; xx < 11; ++xx) {
        for (yy = 0; yy < 11; ++yy) {
            sph_test_image_tools_add_gauss(im, xx * 25.6, yy * 25.6, 2.5, 1.0);
        }
    }
    code = cpl_image_save(im, SPH_BASE "_input_image.fits", CPL_TYPE_DOUBLE,
                          NULL, CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    val = sph_fft_get_scale_vector(sphfft, im, &sigma);
    cpl_msg_info(cpl_func, "Found: %f +/-%f\n", val, sigma);
    cpl_test_abs( val, 25.6, sigma);

    cpl_test_zero(unlink(SPH_BASE "_input_image.fits"));

    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im2);
    im2 = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_pad_real_space_test2(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    cpl_error_code code;
    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im, 128.0, 128.0, 10.0, 2.5, 2.5);
    im2 = cpl_image_new(512, 512, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im2, 256.0, 256.0, 10.0, 2.5, 2.5);

    cpl_test_nonnull( im );
    cpl_test_nonnull( im2 );
    code = sph_fft_prepare(sphfft, im);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_fft_pad_complex_array_real_space(sphfft, 512);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

//    sph_fft_crop_complex_array(sphfft,256,0);
    code = sph_fft_forward(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = sph_fft_inverse(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    tim = sph_fft_complex2image_real(sphfft);

    cpl_test_nonnull( tim );
    code = cpl_image_save(im, SPH_BASE "_pad.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_pad.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(tim, SPH_BASE "_pad.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_subtract(im2, tim);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_pad.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_zero(unlink(SPH_BASE "_pad.fits"));

    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im2);
    im2 = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_scale(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    cpl_image* tim2 = NULL;
    double val = 0.0;
    cpl_error_code code;
    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    code = cpl_image_fill_gaussian(im, 128.5, 128.5, 10.0, 5.5, 5.5);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    im2 = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    code = cpl_image_fill_gaussian(im2, 128.5, 128.5, 10.0, 5.5 * 256 / 312,
                                   5.5 * 256 / 312);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_nonnull( im );
    cpl_test_nonnull( im2 );

    tim = sph_fft_scale(sphfft, im2, 312.0 / 256.0);
    cpl_test_nonnull( tim );
    code = cpl_image_save(im, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(tim, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_subtract(im, tim);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    tim2 = sph_fft_complex2image_imag(sphfft); //5.0/2.5);
    code = cpl_image_save(tim2, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    val = cpl_image_get_absflux(im);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_mean(im);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_max(im);
    cpl_test_lt(val, 1e-10);

    cpl_test_zero(unlink(SPH_BASE "_scale.fits"));

    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(tim2);
    tim2 = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im2);
    im2 = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_scale_fftw_many(void) {
    sph_fft* sphfft = NULL;
    cpl_image* imref = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* diffim = NULL;
    gsl_rng* pRNG = NULL;
    int nrepeats = 10;
    int nx = 256;
    int ny = 256;
    double centx = 128.5;
    double centy = 128.5;
    int ii = 0;
    double scale = 0.0;
    double norm = 0.0;
    double max = 0.0;
    double refflux = 0.0;
    pRNG = gsl_rng_alloc(gsl_rng_taus);

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );
    cpl_test_error(CPL_ERROR_NONE);

    imref = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    sph_test_image_tools_add_gauss(imref, centx, centy, 5.5, 1000.0);
    sph_test_image_tools_add_gauss(imref, centx + 20.0, centy, 5.0, 1000.0);
    refflux = cpl_image_get_flux(imref);
    max = cpl_image_get_max(imref);
    for (ii = 0; ii < nrepeats; ++ii) {
        scale = gsl_ran_flat(pRNG, 1.0, 2.0);
        im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
        sph_test_image_tools_add_gauss(im, centx, centy, 5.5 / scale, 1000.0);
        sph_test_image_tools_add_gauss(im, centx + 20.0 / scale, centy,
                5.0 / scale, 1000.0);
        norm = refflux / cpl_image_get_flux(im);
        cpl_image_multiply_scalar(im, norm);
        im2 = sph_fft_scale(sphfft, im, scale);
        diffim = cpl_image_subtract_create(im2, imref);
        cpl_image_power(diffim, 2.0);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Operation: %d, scaling: "
                "%f, max difference on %f: %f, mean diff: %f", ii, scale, max, sqrt(cpl_image_get_max_window(diffim,50,50,206,206)), sqrt(cpl_image_get_mean_window(diffim,50,50,206,206)));
        cpl_test_abs(
                sqrt(cpl_image_get_max_window(diffim,50,50,206,206))/max, 0.0,
                0.001);
        cpl_test_abs(
                sqrt(cpl_image_get_mean_window(diffim,50,50,206,206))/max, 0.0,
                0.001);
        cpl_image_delete(im);
        im = NULL;
        cpl_image_delete(im2);
        im2 = NULL;
        cpl_image_delete(diffim);
        diffim = NULL;
    }
    cpl_image_delete(imref);
    imref = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    gsl_rng_free(pRNG);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_scale_fftw(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    double val = 0.0;
    cpl_error_code code;
    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    code = cpl_image_fill_gaussian(im, 128.5, 128.5, 10.0, 5.5, 5.5);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    im2 = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    code = cpl_image_fill_gaussian(im2, 128.5, 128.5, 10.0, 5.5 * 256 / 312,
            5.5 * 256 / 312);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_nonnull( im );
    cpl_test_nonnull( im2 );

    tim = sph_fft_scale(sphfft, im2, 312.0 / 256.0);
    cpl_test_nonnull( tim );
    code = cpl_image_save(im, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im2, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(tim, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_subtract(im, tim);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(im, SPH_BASE "_scale.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    val = cpl_image_get_absflux(im);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_mean(im);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_max(im);
    cpl_test_lt(val, 1e-10);

    cpl_test_zero(unlink(SPH_BASE "_scale.fits"));

    cpl_image_delete(tim);
    tim = NULL;
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(im2);
    im2 = NULL;
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_rotate_and_shift(void) {
    sph_fft* sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* imin = NULL;
    cpl_image* imin2 = NULL;
    cpl_image* tim = NULL;
    double val = 0.0;
    cpl_size xx = 0;
    cpl_size yy = 0;
    cpl_error_code code;

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( im );

    im2 = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( im2 );
    imin = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( imin );
    imin2 = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( imin2 );

    code = cpl_image_fill_gaussian(im, 128.0, 128.0, 10.0, 5.0, 50.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_fill_gaussian(im2, 128.0, 128.0, 10.0, 50.0, 5.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_fill_gaussian(imin, 148.0, 118.0, 10.0, 5.0, 50.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_fill_gaussian(imin2, 148.0, 118.0, 10.0, 50.0, 5.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_add(im, im2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_add(imin, imin2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_image_delete(im2);
    im2 = NULL;
    cpl_image_delete(imin2);
    imin2 = NULL;

    code = sph_fft_rotate_prepare(sphfft, imin, 0.0, -20.0, 10.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_abs( sphfft->angle, 0 * 1.57 / 2.0, 0.1);

    code = sph_fft_rotate(sphfft);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    tim = sph_fft_rotate_complex2image(sphfft);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( tim );
    code = cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_DEFAULT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(imin, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_subtract(tim, im);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_multiply(tim, tim);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    // For validity, check window excluding boundary since
    // shifting cuts regions

    val = cpl_image_get_absflux_window(tim, 20, 20, 235, 235);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_mean_window(tim, 20, 20, 235, 235);
    cpl_test_lt(val, 1e-10);
    val = cpl_image_get_max_window(tim, 20, 20, 235, 235);
    cpl_test_lt(val, 1e-10);
    cpl_image_get_maxpos(tim, &xx, &yy);

    cpl_test_zero(unlink(SPH_BASE "_rot.fits"));

    cpl_image_delete(tim);
    cpl_image_delete(im);
    cpl_image_delete(imin);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_rotate(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    double val = 0.0;
    cpl_size xx = 0;
    cpl_size yy = 0;
    double xcent = 0;
    double ycent = 0;

    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    im2 = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im, 16.5, 16.5, 10.0, 1.0, 10.0);
    cpl_image_fill_gaussian(im2, 16.5, 16.5, 10.0, 10.0, 1.0);
    cpl_image_add(im, im2);
    cpl_image_delete(im2);
    im2 = NULL;
    cpl_test_nonnull( im );
    xcent = cpl_image_get_centroid_x(im);
    ycent = cpl_image_get_centroid_y(im);
    cpl_msg_info(cpl_func, "Centroid: %g %g", xcent, ycent);
    sph_fft_rotate_prepare(sphfft, im, 90.0, 0.0, 0.0);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_abs( sphfft->angle, 2 * 1.57 / 2.0, 0.1);
    sph_fft_rotate(sphfft);
    tim = sph_fft_rotate_complex2image(sphfft);
    cpl_test_nonnull( tim );
    xcent = cpl_image_get_centroid_x(tim);
    ycent = cpl_image_get_centroid_y(tim);
    cpl_msg_info(cpl_func, "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim, im);
    cpl_image_multiply(tim, tim);
    cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);

    // For validity, check window excluding boundary since current
    // algorithm creates spurious signals on boundary pixels on
    // top boundary. Not totally sure why.

    val = cpl_image_get_absflux_window(tim, 10, 10, 22, 22);
    cpl_test_lt(val, 1e-4);
    val = cpl_image_get_mean_window(tim, 5, 5, 26, 26);
    cpl_test_lt(val, 1e-6);
    val = cpl_image_get_max_window(tim, 10, 10, 22, 22);
    cpl_test_lt(val, 1e-5);
    cpl_image_get_maxpos(tim, &xx, &yy);

    cpl_test_zero(unlink(SPH_BASE "_rot.fits"));

    cpl_image_delete(tim);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_smoothe_image(void) {
    cpl_image* im  = cpl_image_new(34, 32, CPL_TYPE_DOUBLE);
    cpl_image* im1;
    double xcent0, ycent0;
    double xcent, ycent;
    double aflux;
    cpl_error_code code;


    code = cpl_image_fill_gaussian(im, 17.5, 16.5, 10.0, 1.0, 1.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    im1 = cpl_image_duplicate(im);

    xcent0 = cpl_image_get_centroid_x(im);
    ycent0 = cpl_image_get_centroid_y(im);
    aflux = cpl_image_get_absflux(im);

    code = sph_fft_smoothe_image(im, NULL, 8.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    xcent = cpl_image_get_centroid_x(im);
    ycent = cpl_image_get_centroid_y(im);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Centres: %f,%f vs %f,%f",
                         xcent0, ycent0, xcent, ycent);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Fluxes: %f vs %f", aflux,
                         cpl_image_get_absflux(im));

    cpl_test_abs( xcent0, xcent, 500.0 * DBL_EPSILON);
    cpl_test_abs( ycent0, ycent, 500.0 * DBL_EPSILON);

    cpl_test_image_abs(im, im1, 2.0);

    cpl_image_delete(im);
    cpl_image_delete(im1);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_rotate_image(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    double val = 0.0;
    cpl_size xx = 0;
    cpl_size yy = 0;
    double xcent = 0;
    double ycent = 0;

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    im2 = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    cpl_image_fill_gaussian(im, 16.5, 16.5, 10.0, 1.0, 10.0);
    cpl_image_fill_gaussian(im2, 16.5, 16.5, 10.0, 10.0, 1.0);
    cpl_image_add(im, im2);
    cpl_image_delete(im2);
    im2 = NULL;
    cpl_test_nonnull( im );
    xcent = cpl_image_get_centroid_x(im);
    ycent = cpl_image_get_centroid_y(im);
    cpl_msg_info(cpl_func, "Centre of image: %f, %f\n", xcent, ycent);
    tim = sph_fft_rotate_image(sphfft, im, 90.0);
    cpl_test_nonnull( tim );
    xcent = cpl_image_get_centroid_x(tim);
    ycent = cpl_image_get_centroid_y(tim);
    cpl_msg_info(cpl_func, "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim, im);
    cpl_image_multiply(tim, tim);
    cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);

    // For validity, check window excluding boundary since current
    // algorithm creates spurious signals on boundary pixels on
    // top boundary. Not totally sure why.

    val = cpl_image_get_absflux_window(tim, 10, 10, 22, 22);
    cpl_test_lt(val, 1e-4);
    val = cpl_image_get_mean_window(tim, 5, 5, 26, 26);
    cpl_test_lt(val, 1e-6);
    val = cpl_image_get_max_window(tim, 10, 10, 22, 22);
    cpl_test_lt(val, 1e-5);
    cpl_image_get_maxpos(tim, &xx, &yy);

    cpl_test_zero(unlink(SPH_BASE "_rot.fits"));

    cpl_image_delete(tim);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_rotate_image_shift(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* tim = NULL;
    cpl_size xx = 0;
    cpl_size yy = 0;
    double xcent = 0;
    double ycent = 0;

    sphfft = sph_fft_new(SPH_FFT_FFTW3_DP);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    im2 = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    //cpl_image_fill_gaussian( im, 16.5, 16.5, 10.0, 1.0, 10.0 );
    cpl_image_fill_gaussian(im2, 16.5, 16.5, 1000.0, 2.0, 2.0);
    cpl_image_add(im, im2);
    cpl_image_delete(im2);
    im2 = NULL;
    cpl_test_nonnull( im );
    xcent = cpl_image_get_centroid_x(im);
    ycent = cpl_image_get_centroid_y(im);
    cpl_msg_info(cpl_func, "Centre of image: %f, %f\n", xcent, ycent);
    sph_fft_set_shift(sphfft, 7.0, 0.0);
    tim = sph_fft_rotate_image(sphfft, im, 90.0);
    cpl_test_nonnull( tim );
    xcent = cpl_image_get_centroid_x(tim);
    ycent = cpl_image_get_centroid_y(tim);
    cpl_msg_info(cpl_func, "Centre of rotated image: %f, %f\n", xcent, ycent);
    cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_image_save(im, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_image_subtract(tim, im);
    cpl_image_multiply(tim, tim);
    cpl_image_save(tim, SPH_BASE "_rot.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);

    // For validity, check window excluding boundary since current
    // algorithm creates spurious signals on boundary pixels on
    // top boundary. Not totally sure why.

    cpl_image_get_maxpos(tim, &xx, &yy);

    cpl_test_zero(unlink(SPH_BASE "_rot.fits"));

    cpl_image_delete(tim);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_skewx(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    double xcent = 0;
    double ycent = 0;
    int xx = 0;
    int yy = 0;

    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    im = cpl_image_new(32, 32, CPL_TYPE_DOUBLE);
    for (xx = 0; xx < 16; ++xx) {
        for (yy = 0; yy < 16; ++yy) {
            cpl_image_set(im, xx + 9, yy + 9, 1.0);
        }
    }cpl_test_nonnull( im );
    xcent = cpl_image_get_centroid_x(im);
    ycent = cpl_image_get_centroid_y(im);
    sph_fft_skewx(sphfft, im, 45.0);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_abs( sphfft->angle, 1.57 / 2.0, 0.1);
    tim = sph_fft_rotate_complex2image(sphfft);
    cpl_test_nonnull( tim );
    xcent = cpl_image_get_centroid_x(tim);
    ycent = cpl_image_get_centroid_y(tim);
    cpl_msg_info(cpl_func, "Centroid: %g %g", xcent, ycent);
    cpl_image_save(tim, SPH_BASE "_skewx.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);

    cpl_test_zero(unlink(SPH_BASE "_skewx.fits"));

    cpl_image_delete(tim);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_skewy(void) {
    sph_fft* sphfft = NULL;
    cpl_image* im = NULL;
    cpl_image* tim = NULL;
    double xcent = 0;
    double ycent = 0;
    cpl_error_code code;

    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( sphfft );

    im = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_test_nonnull( im );
    code = cpl_image_fill_gaussian(im, 128.5, 128.5, 10.0, 5.0, 5.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    xcent = cpl_image_get_centroid_x(im);
    ycent = cpl_image_get_centroid_y(im);
    cpl_msg_info(cpl_func, "Centroid: %g %g", xcent, ycent);
    sph_fft_skewy(sphfft, im, 45.0);
    cpl_test_nonnull( sphfft->complexdata );
    cpl_test_abs( sphfft->angle, 1.57 / 2.0, 0.1);
    tim = sph_fft_rotate_complex2image(sphfft);
    cpl_test_nonnull( tim );
    xcent = cpl_image_get_centroid_x(tim);
    ycent = cpl_image_get_centroid_y(tim);
    cpl_msg_info(cpl_func, "Centroid: %g %g", xcent, ycent);
    cpl_image_save(tim, SPH_BASE "_skewy.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);

    cpl_test_zero(unlink(SPH_BASE "_skewy.fits"));

    cpl_image_delete(tim);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    sphfft = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_fft_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_fft_delete(void) {
    sph_fft* sphfft = NULL;

    sphfft = sph_fft_new(SPH_FFT_GSL_RADIX2);

    cpl_test_nonnull( sphfft );

    sph_fft_delete(sphfft);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    const void* pSuite;


    (void)sph_test_init();


    pSuite = sph_add_suite("sph_fft_test", cutest_init_sph_fft_testsuite,
            cutest_clean_sph_fft_testsuite);
    cpl_test_nonnull(pSuite);


    if (NULL == sph_test_do(pSuite, "sph_fft_new", cutest_sph_fft_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_fft_smoothe_image",
                    cutest_sph_fft_smoothe_image)) {
        return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_get_scale_vector",
                             cutest_sph_fft_get_scale_vector ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_forward_inverse_radix2",
                             cutest_sph_fft_forward_inverse_radix2 ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_forward_inverse_mixedradix",
                             cutest_sph_fft_forward_inverse_mixedradix ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_shift",
                             cutest_sph_fft_shift ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_pad_real_space",
                             cutest_sph_fft_pad_real_space ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_pad_real_space_test2",
                             cutest_sph_fft_pad_real_space_test2 ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_pad_freq_space",
                             cutest_sph_fft_pad_freq_space ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_set_complex",
                             cutest_sph_fft_set_complex ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_scale",
                             cutest_sph_fft_scale ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_scale_fftw_many",
                             cutest_sph_fft_scale_fftw_many ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_scale_fftw",
                             cutest_sph_fft_scale_fftw ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_rotate_prepare",
                             cutest_sph_fft_rotate_prepare ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_rotate_and_shift",
                             cutest_sph_fft_rotate_and_shift ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_rotate",
                             cutest_sph_fft_rotate ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_rotate_image",
                             cutest_sph_fft_rotate_image ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_rotate_image_shift",
                             cutest_sph_fft_rotate_image_shift ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_skewx",
                             cutest_sph_fft_skewx ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_skewy",
                             cutest_sph_fft_skewy ) )
    {
       return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "sph_fft_delete",
                             cutest_sph_fft_delete ) )
    {
       return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
