/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_fitting.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_vector.h>
static
int cutest_test_fitting_fit_gauss2D__(double peakval, double fwhm,
        double guess_offset_x, double guess_offset_y, double guess_fwhm,
        double sigma, double bgvalue);
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_poly1d(void) {
    cpl_vector* xvals = cpl_vector_new(10);
    cpl_vector* yvals = cpl_vector_new(10);
    int ii;
    double xv;
    cpl_polynomial* poly = NULL;
    cpl_size pows = 0;
    for (ii = 0; ii < 10; ++ii) {
        xv = ii * 1.0;
        cpl_vector_set(xvals, ii, xv);
        cpl_vector_set(yvals, ii, 0.7 + 0.2 * (xv) + 0.05 * (xv) * (xv));
    }

    poly = sph_fitting_fit_poly1d(xvals, yvals, NULL, 0, 2, 0, 0.0, NULL);
    pows = 0;
    cpl_test_nonnull( poly );
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.7, 0.01);
    pows = 1;
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.2, 0.01);
    pows = 2;
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.05, 0.01);
    cpl_vector_delete(xvals);
    cpl_vector_delete(yvals);
    cpl_polynomial_delete(poly);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_poly1d_drop(void) {
    cpl_vector* xvals = cpl_vector_new(10);
    cpl_vector* yvals = cpl_vector_new(10);
    int ii;
    double xv;
    cpl_polynomial* poly = NULL;
    cpl_size pows = 0;
    for (ii = 0; ii < 10; ++ii) {
        xv = ii * 1.0;
        cpl_vector_set(xvals, ii, 10.0 - xv);
        cpl_vector_set(yvals, ii, 0.7 + 0.2 * (xv) + 0.05 * (xv) * (xv));
    }

    poly = sph_fitting_fit_poly1d(xvals, yvals, NULL, 0, 2, 0, 0.0, NULL);
    pows = 0;
    cpl_test_nonnull( poly );
    //cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.7, 0.01 );
    pows = 1;
    //cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.2, 0.01 );
    pows = 2;
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.05, 0.01);
    cpl_vector_delete(xvals);
    cpl_vector_delete(yvals);
    cpl_polynomial_delete(poly);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_estimate_error(void) {
    cpl_vector* xvals = cpl_vector_new(10);
    cpl_vector* yvals = cpl_vector_new(10);
    cpl_vector* errs = cpl_vector_new(10);
    cpl_vector* errv = NULL;
    cpl_vector* means = NULL;
    int ii;
    double xv;
    cpl_polynomial* poly = NULL;
    cpl_size pows = 0;
    double red = 0.0;
    for (ii = 0; ii < 10; ++ii) {
        xv = ii * 1.0;
        cpl_vector_set(xvals, ii, xv);
        cpl_vector_set(yvals, ii, 0.7 + 2.1 * xv + 0.05 * xv * xv);
        cpl_vector_set(errs, ii, 0.1 * sqrt(0.7 + 2.1 * xv + 0.05 * xv * xv));
    }

    poly = sph_fitting_fit_poly1d(xvals, yvals, errs, 0, 2, 0, 122.0, &red);
    pows = 0;
    cpl_test_nonnull( poly );
    cpl_test_abs(red, 0.0, 2.0);
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.7, 0.1);
    pows = 1;
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 2.1, 0.1);
    pows = 2;
    cpl_test_abs( cpl_polynomial_get_coeff(poly,&pows), 0.05, 0.1);
    errv = sph_fitting_estimate_error(poly, xvals, yvals, errs, &means);
    cpl_test_nonnull( errv );
    cpl_test_nonnull( means );
    cpl_test_abs(cpl_vector_get(means,0), 0.7, 0.1);
    cpl_test_abs(cpl_vector_get(means,1), 2.1, 0.1);
    cpl_test_abs(cpl_vector_get(means,2), 0.05, 0.01);
    cpl_test_abs(cpl_vector_get(errv,0), 0.1, 0.1);
    cpl_test_abs(cpl_vector_get(errv,1), 0.1, 0.1);
    cpl_test_abs(cpl_vector_get(errv,2), 0.1, 0.1);
    //cpl_vector_dump(means,stdout);
    //cpl_vector_dump(errv,stdout);
    cpl_vector_delete(xvals);
    cpl_vector_delete(yvals);
    cpl_vector_delete(errs);
    cpl_vector_delete(errv);
    cpl_vector_delete(means);
    cpl_polynomial_delete(poly);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_robust(void) {
    cpl_vector* xvals = cpl_vector_new(10);
    cpl_vector* yvals = cpl_vector_new(10);
    double lin = 0.0;
    double offset = 0.0;
    int ii;
    double xv;
    for (ii = 0; ii < 10; ++ii) {
        xv = ii * 1.0;
        cpl_vector_set(xvals, ii, xv);
        cpl_vector_set(yvals, ii, 0.7 + 2.2 * xv);
    }
    cpl_vector_set(yvals, 5, 1000.0);
    lin = sph_fitting_fit_robust(xvals, yvals, 1.0, 5.0, &offset);
    cpl_test_abs( offset, 0.7, 0.01);
    cpl_test_abs( lin, 2.2, 0.01);
    cpl_vector_delete(xvals);
    cpl_vector_delete(yvals);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_poly2d(void) {
    cpl_vector* xvals = cpl_vector_new(100); // 10x10 array
    cpl_vector* yvals = cpl_vector_new(100);
    cpl_vector* zvals = cpl_vector_new(100);
    cpl_vector* vv = NULL;
    int xx, yy;
    double xv, yv;
    double diff, tdiff;
    cpl_polynomial* poly = cpl_polynomial_new(2);
    cpl_polynomial* polynew = NULL;
    cpl_size pows[2] = { 0, 0 };

    cpl_polynomial_set_coeff(poly, pows, 0.7);
    pows[0] = pows[1] = 1;
    cpl_polynomial_set_coeff(poly, pows, 0.2);
    pows[0] = pows[1] = 2;
    cpl_polynomial_set_coeff(poly, pows, 0.001);
    pows[0] = 2;
    pows[1] = 0;
    cpl_polynomial_set_coeff(poly, pows, 0.01);
    pows[0] = 0;
    pows[1] = 2;
    cpl_polynomial_set_coeff(poly, pows, 0.02);

    for (xx = 0; xx < 10; ++xx) {
        for (yy = 0; yy < 10; ++yy) {
            xv = xx * 1.0;
            yv = yy * 1.0;
            cpl_vector_set(xvals, yy * 10 + xx, xv);
            cpl_vector_set(yvals, yy * 10 + xx, yv);
            vv = cpl_vector_new(2);
            cpl_vector_set(vv, 0, xv);
            cpl_vector_set(vv, 1, yv);
            cpl_vector_set(zvals, yy * 10 + xx, cpl_polynomial_eval(poly, vv));
            cpl_vector_delete(vv);
        }
    }

    polynew = sph_fitting_fit_poly2d(xvals, yvals, zvals, NULL, 2, 2);
    tdiff = 0;
    for (xx = 0; xx < 10; ++xx) {
        for (yy = 0; yy < 10; ++yy) {
            xv = xx * 1.0;
            yv = yy * 1.0;
            vv = cpl_vector_new(2);
            cpl_vector_set(vv, 0, xv);
            cpl_vector_set(vv, 1, yv);
            diff = cpl_polynomial_eval(polynew, vv)
                    - cpl_vector_get(zvals, yy * 10 + xx);
            diff = diff
                    / (0.5
                            * (cpl_polynomial_eval(polynew, vv)
                                    + cpl_vector_get(zvals, yy * 10 + xx)));
            tdiff = tdiff + diff * diff;
            cpl_vector_delete(vv);
        }
    }
    tdiff = sqrt(tdiff) / 100;
    cpl_test( tdiff<0.02);

    cpl_vector_delete(xvals);
    cpl_vector_delete(yvals);
    cpl_vector_delete(zvals);
    cpl_polynomial_delete(polynew);
    cpl_polynomial_delete(poly);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_robust_imlist(void) {
    cpl_imagelist* imlistA = cpl_imagelist_new();
    cpl_image* im = NULL;
    cpl_vector* xvals = cpl_vector_new(5);
    cpl_image* lin = NULL;
    cpl_image* offset = NULL;
    int bpix = 0;
    gsl_rng* pRNG = gsl_rng_alloc(gsl_rng_taus);
    cpl_error_code code;


    for (int ii = 0; ii < 5; ++ii) {
        const double xv = (ii + 1) * 1000.0;
        code = cpl_vector_set(xvals, ii, xv);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        im = sph_test_image_tools_create_flat_image_double(256, 256, xv);
        cpl_test_zero(sph_test_image_tools_add_noise(im, 0.1, pRNG));
        code = cpl_imagelist_set(imlistA, im, ii);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
    }

    lin = sph_fitting_fit_robust_imlist(imlistA, xvals, 0.1, 10.0, &offset,
                                        NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( lin );
    cpl_test_nonnull( offset );
    for (int xx = 0; xx < cpl_image_get_size_x(lin); ++xx) {
        for (int yy = 0; yy < cpl_image_get_size_y(lin); ++yy) {
            cpl_test_abs(cpl_image_get(lin, xx + 1, yy + 1, &bpix), 1.0, 0.01);
        }
    }

    code = cpl_image_save(lin, "linim.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_CREATE);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_save(offset, "linim.fits", CPL_TYPE_DOUBLE, NULL,
                          CPL_IO_EXTEND);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_zero(unlink("linim.fits"));

    cpl_imagelist_delete(imlistA);
    cpl_image_delete(lin);
    cpl_image_delete(offset);
    cpl_vector_delete(xvals);
    gsl_rng_free(pRNG);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_lm_dero(void) {
    const int N = 10, p = 5;
    double r[N], y[N], sigma_y[N];
    double init[] = { 8.0, 9.0, 14, 14, 90 };
    int status;
    cpl_vector* fit;
    cpl_vector* fiterr;
    double gran = 1e-4;

    /* This is the data to be fitted */
    const double abcde[] = { 9, 10, 15.0, 16.0, 100.0 };

    const gsl_rng_type* type;
    gsl_rng* ran;
    gsl_rng_env_setup();

    type = gsl_rng_default;
    ran = gsl_rng_alloc(type);

    for (int i = 0; i < N; i++) {
        r[i] = i * 2 * CPL_MATH_PI / (N - 1);
        y[i] = abcde[0] * cos(r[i]) + abcde[1] * sin(r[i])
                + abcde[2] * cos(2 * r[i]) + abcde[3] * sin(2 * r[i])
                + abcde[4];
        //+ gsl_ran_gaussian (ran, .1);
        sigma_y[i] = 1.0;
        //printf ("data: %u %g %g %g\n", i, r[i], y[i], sigma_y[i]);
    };

    fit = cpl_vector_new(p + 1);
    fiterr = cpl_vector_new(p + 1);
    status = sph_fitting_lm_dero(N, r, y, sigma_y, init, fit, fiterr);

    cpl_test_zero(status);
    for (int i = 0; i < p; i++) {
        cpl_test_abs(abcde[i], cpl_vector_get( fit, i), gran);
    }

    gsl_rng_free(ran);
    cpl_vector_delete(fit);
    cpl_vector_delete(fiterr);
}

static
void cutest_test_cpl_bug(void) {
    cpl_image* im = NULL;
    cpl_image* imext = NULL;
    int xx = 0;
    int yy = 0;
    cpl_mask* mask = NULL;
    cpl_mask* nmask = NULL;
    im = cpl_image_new(200, 200, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(im, 2.0);
    for (xx = 0; xx < 200; ++xx) {
        for (yy = 0; yy < 200; ++yy) {
            cpl_image_reject(im, xx + 1, yy + 1);
        }
    }
    cpl_image_accept(im, 50, 50);
    imext = cpl_image_extract(im, 30, 30, 70, 70);
    cpl_test_lt(1.0, cpl_image_get_absflux(im));
    cpl_test_lt(1.0, cpl_image_get_absflux(imext));
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(imext);
    imext = NULL;

    mask = cpl_mask_new(200, 200);
    cpl_test_zero(cpl_mask_count(mask));
    cpl_mask_not(mask);
    cpl_test_noneq(cpl_mask_count(mask), 0);
    cpl_mask_set(mask, 50, 50, 0);
    nmask = cpl_mask_extract(mask, 30, 30, 70, 70);
    cpl_mask_not(nmask);
    cpl_test_eq(cpl_mask_count(nmask), 1);
    cpl_mask_not(mask);
    cpl_test_eq(cpl_mask_count(mask), 1);
    cpl_mask_delete(mask);
    mask = NULL;
    cpl_mask_delete(nmask);
    nmask = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_gauss2D(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    double xpos = 256.0;
    double ypos = 256.0;
    double xw = 6.0;
    double yw = 6.0;
    cpl_image* im = NULL;
    cpl_image* errim = NULL;
    double bgvalue = 100.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;
    int nx = 512;
    int ny = 512;

    im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    sph_test_image_tools_add_gauss(im, xpos + 4.0, ypos + 4.0, 4.0, 100000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    errim = cpl_image_duplicate(im);
    cpl_image_power(errim, 0.5);

    rerr = sph_fitting_fit_gauss2D(im, errim, &xpos, &ypos, &xw, &yw, 3, 3);
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(errim);
    errim = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_abs( xpos, 260.0, 0.15);
    cpl_test_abs( ypos, 260.0, 0.15);
    cpl_test_abs( xw, 4.0, 0.1);
    cpl_test_abs( yw, 4.0, 0.1);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Measurement of gauss: %f, %f, %f, %f", xpos, ypos, xw, yw);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_gauss2D_explore(void) {
    int ii = 0;
    int result = 0;

    for (ii = 0; ii < 50; ++ii) {
        result = cutest_test_fitting_fit_gauss2D__(100000.0, 2.93124 + 0.1 * ii,
                0.0, 0.0, 4.0, 0.00000001, 0.0);
        cpl_test_zero(result);
        //printf("Width: %6.3f, result: %d\n", 1.63124 + 0.1 * ii, result);
    }
    for (ii = 0; ii < 50; ++ii) {
        result = cutest_test_fitting_fit_gauss2D__(100.0, 2.93124 + 0.1 * ii,
                0.0, 0.0, 4.0, 0.1, 0.1);
        cpl_test_zero(result);
        //printf("Width: %6.3f, result: %d\n", 1.63124 + 0.1 * ii, result);
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_test_fitting_fit_gauss2D__(double peakval, double fwhm,
        double guess_offset_x, double guess_offset_y, double guess_fwhm,
        double sigma, double bgvalue) {
    sph_error_code rerr = CPL_ERROR_NONE;
    double xpos0 = 128.789;
    double ypos0 = 128.124;
    double xpos = xpos0 + guess_offset_x;
    double ypos = ypos0 + guess_offset_y;
    double xw = guess_fwhm;
    double yw = guess_fwhm;
    cpl_image* im = NULL;
    cpl_image* imerr = NULL;
    double vpeak = 0.0;
    int nx = 256;
    int ny = 256;

    im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    cpl_image_fill_gaussian(im, xpos0, ypos0, 1000.0, fwhm, fwhm);

    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakval / vpeak);
    cpl_image_add_scalar(im, bgvalue);
    sph_test_image_tools_add_noise(im, sigma, NULL);

    imerr = sph_test_image_tools_create_flat_image_double(nx, ny, sigma);
    cpl_error_reset();
    rerr = sph_fitting_fit_gauss2D(im, imerr, &xpos, &ypos, &xw, &yw, 3, 3);
    cpl_error_reset();
    cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(imerr);
    imerr = NULL;

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    //cpl_test_abs( xpos, 260.0, 0.01);
    //cpl_test_abs( ypos, 260.0, 0.01);
    //cpl_test_abs( xw, 4.0, 0.1);
    //cpl_test_abs( yw, 4.0, 0.1);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Measurement of gauss: %f, %f, %f, %f", xpos, ypos, xw, yw);
    if (fabs(xpos - xpos0) > 0.05)
        return 1;
    if (fabs(ypos - ypos0) > 0.05)
        return 1;
    if (fabs(xw - fwhm) > 0.1)
        return 1;
    if (fabs(yw - fwhm) > 0.1)
        return 1;
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_fitting_fit_gauss2D_no_errimage(void) {
    cpl_error_code code = CPL_ERROR_NONE;
    double xpos = 256.0;
    double ypos = 256.0;
    double xw = 1.0;
    double yw = 1.0;
    cpl_image* im = NULL;
    const double bgvalue = 10.0;
    const double peakvalue = 35000.0;
    double vpeak = 0.0;
    cpl_size nfail;
    const int nx = 512;
    const int ny = 512;
    const int maxtry = 3;

    im = sph_test_image_tools_create_flat_image_double(nx, ny, 0.0);
    sph_test_image_tools_add_gauss(im, xpos + 4.0, ypos + 4.0, 4.0, 100000.0);
    vpeak = cpl_image_get_max(im);
    code = cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_image_add_scalar(im, bgvalue);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for (int itry = 0; itry < maxtry; itry++) {
        int retval;

        retval = sph_test_image_tools_apply_poisson_noise(im, NULL);

        cpl_test_zero(retval);

        code = sph_fitting_fit_gauss2D(im, NULL, &xpos, &ypos, &xw, &yw, 3, 3);

        if (code == CPL_ERROR_NONE) break;

        if (itry + 1 < maxtry)
            cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    }

    nfail = cpl_test_get_failed();
    cpl_test_abs( xpos, 260.0, 0.35);
    cpl_test_abs( ypos, 260.0, 0.35);
    cpl_test_abs( xw, 4.0, 0.3);
    cpl_test_abs( yw, 4.0, 0.3);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                         "Measurement of gauss: %f, %f, %f, %f", xpos, ypos, xw, yw);
    if (cpl_test_get_failed() > nfail) {
        cpl_msg_warning(cpl_func, "The above four tests occasionally fail due "
                        "to the randomness of the test data. Please rerun...");
    }

    cpl_image_delete(im);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    (void)sph_test_init();

    pSuite = sph_add_suite("Testing sph_fitting", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing fitting 2D gauss explore",
                    cutest_test_fitting_fit_gauss2D_explore)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting 1D poly",
                    cutest_test_fitting_fit_poly1d)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting 1D poly drop",
                    cutest_test_fitting_fit_poly1d_drop)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Estimate error",
                    cutest_test_fitting_estimate_error)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting 2D poly",
                    cutest_test_fitting_fit_poly2d)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting LM dero",
                    cutest_test_fitting_lm_dero)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting robust",
                    cutest_test_fitting_fit_robust)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting robust imlist",
                    cutest_test_fitting_fit_robust_imlist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting gauss 2d",
                    cutest_test_fitting_fit_gauss2D)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fitting gauss 2d no errimage",
                    cutest_test_fitting_fit_gauss2D_no_errimage)) {
        return sph_test_get_error();
    }

    if ( NULL == sph_test_do(pSuite, "Testing cpl bug",
                             cutest_test_cpl_bug ) )
        {
            return sph_test_get_error();
        }

    return sph_test_end();
}

/**@}*/
