/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ifs_lenslet_model.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test.h"
#include "sph_image_grid.h"
#include "sph_utils.h"

#include <cpl.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <float.h>

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_new();
static
void cutest_ifs_lenslet_model_save();
static
void cutest_ifs_lenslet_model_load();
static
void cutest_ifs_lenslet_model_get_centre(void);
static
void cutest_ifs_lenslet_model_new_from_propertylist();
static
void cutest_ifs_lenslet_model_new_from_propertylist_distmodel();
static
void cutest_ifs_lenslet_model_get_as_propertylist();
static
void cutest_ifs_lenslet_model_predict_spec_loc_pp();
static
void cutest_ifs_lenslet_model_predict_spec_loc_pp_model();
#ifdef SPH_TEST_LENSLET_MODEL
static
void cutest_ifs_lenslet_model_predict_spec_loc();
static
void cutest_ifs_lenslet_model_predict_spec_loc_offset();
static
void cutest_ifs_lenslet_model_predict_spec_loc_no_duplicates();
static
void cutest_ifs_lenslet_model_predict_spec_loc_fractional_offset();
#endif
static
void cutest_ifs_lenslet_model_get_poly();
static
void cutest_ifs_lenslet_model_get_coords();
static
void cutest_ifs_lenslet_model_create_pdt_images();
static
void cutest_ifs_lenslet_model_get_index();
static
void
cutest_ifs_lenslet_model_get_lambda_vector(void);
static
void cutest_ifs_lenslet_model_delete();
static
void cutest_ifs_lenslet_model_set_lenslet_distortion();
static
void cutest_ifs_lenslet_model_set_get_hex_w_lenslet_distortion();
static
int cutest_init_ifs_lenslet_model_testsuite();

static
int cutest_clean_ifs_lenslet_model_testsuite();

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_ifs_lenslet_model    unit tests for sph_ifs_lenslet_model
 * and associated functions.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_new(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( model );
    cpl_test_abs( model->detsize_microns,
            SPH_IFS_LENSLET_MODEL_DET_SIZE_MICRONS, 0.0);
    cpl_test_eq(model->detsize_pixels, SPH_IFS_LENSLET_MODEL_DET_SIZE_PIX);
    cpl_test_abs( model->lensize_microns,
            SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_MICRONS, 0.0);
    cpl_test_eq( model->lenslets_per_side,
            SPH_IFS_LENSLET_MODEL_LENSLETS_SIDE);
    cpl_test_abs( model->pixsize_microns,
            SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS, 0.0);
    cpl_test_abs( model->speclength_microns,
            SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS, 0.0);
    cpl_test_abs( model->specwidth_microns,
            SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS, 0.0);
    cpl_test_abs(
            model->speclength_pixels,
            SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS / SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS,
            0.000001);
    cpl_test_abs(
            model->specwidth_pixels,
            SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS / SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS,
            0.00001);
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_save(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );

    rerr = sph_ifs_lenslet_model_save(model, "lenslet_model_test.ini");

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_load function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_load(void) {
    sph_ifs_lenslet_model* model = NULL;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ifs_lenslet_model_load("lenslet_model_test.ini");

#ifdef SPH_NO_ASCII
    cpl_test_null( model );
    cpl_error_reset();
    return;
#endif
    cpl_test_nonnull( model );
    /*Verify */
    cpl_test_abs( model->detsize_microns,
            SPH_IFS_LENSLET_MODEL_DET_SIZE_MICRONS, 0.0);
    cpl_test_eq(model->detsize_pixels, SPH_IFS_LENSLET_MODEL_DET_SIZE_PIX);
    cpl_test_abs( model->lensize_microns,
            SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_MICRONS, 0.0);
    cpl_test_eq( model->lenslets_per_side,
            SPH_IFS_LENSLET_MODEL_LENSLETS_SIDE);
    cpl_test_abs( model->pixsize_microns,
            SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS, 0.0);
    cpl_test_abs( model->speclength_microns,
            SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS, 0.0);
    cpl_test_abs( model->specwidth_microns,
            SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS, 0.0);
    cpl_test_abs(
            model->speclength_pixels,
            SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS / SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS,
            0.000001);
    cpl_test_abs(
            model->specwidth_pixels,
            SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS / SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS,
            0.000001);
    SPH_RAISE_CPL;
    sph_ifs_lenslet_model_delete(model);
    SPH_RAISE_CPL;cpl_test_error(CPL_ERROR_NONE);
    SPH_RAISE_CPL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_new_from_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_get_centre(void) {
    sph_ifs_lenslet_model* model = NULL;
    double x = 0;
    double y = 0;

    model = sph_ifs_lenslet_model_new();
    cpl_test_eq(sph_ifs_lenslet_model_get_centre(model,0,0,&x, &y),
            CPL_ERROR_NONE);
    cpl_test_abs(x, 0.0, 0.000000001);
    cpl_test_abs(y, 0.0, 0.000000001);
    cpl_test_eq(sph_ifs_lenslet_model_get_centre(model,1,1,&x, &y),
            CPL_ERROR_NONE);
    cpl_test_abs(x, CPL_MATH_SQRT3 * model->lensize_microns * 1.5,
            0.000000001);
    cpl_test_abs(y, 1.5 * model->lensize_microns, 0.000000001);
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_new_from_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_new_from_propertylist(void) {
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* pl = NULL;

    /* Setup and run ...*/
    cpl_error_reset();
    pl = cpl_propertylist_new();
    cpl_test_nonnull( pl );
    cpl_propertylist_append_int(pl, SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME, 2048);
    cpl_propertylist_append_int(pl, SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME,
            145);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME,
            10.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME,
            1.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME,
            100.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME,
            4.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME,
            1.333);

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_ifs_lenslet_model_new_from_propertylist(pl);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( model );
    /*Verify */
    cpl_test_abs( model->detsize_microns, 2048.0, DBL_EPSILON);
    cpl_test_eq(model->detsize_pixels, 2048);
    cpl_test_abs( model->lensize_microns, 10.0, DBL_EPSILON);
    cpl_test_eq(model->lenslets_per_side, 145);
    cpl_test_abs( model->pixsize_microns, 1.0, DBL_EPSILON);
    cpl_test_abs( model->speclength_microns, 100.0, DBL_EPSILON);
    cpl_test_abs( model->specwidth_microns, 4.0, DBL_EPSILON);
    cpl_test_abs( model->rotangle, 1.333, DBL_EPSILON);
    cpl_propertylist_delete(pl);
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_new_from_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_get_lambda_vector(void) {
    sph_ifs_lenslet_model* model = sph_ifs_lenslet_model_new();
    const cpl_vector* lvect = sph_ifs_lenslet_model_get_lambda_vector(model);

    cpl_test_nonnull( model );
    cpl_test_nonnull( lvect );
    cpl_test_eq(cpl_vector_get_size(lvect), (int)model->speclength_pixels);
    cpl_test_abs(cpl_vector_get(lvect,0), model->minlambda, 0.001);
    cpl_test_abs(
            cpl_vector_get(lvect, cpl_vector_get_size(lvect) - 1 ),
            model->maxlambda, 0.001);

    sph_ifs_lenslet_model_delete(model);
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_new_from_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_new_from_propertylist_distmodel(void) {
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* pl = NULL;
    sph_distortion_model* dist = NULL;
    cpl_propertylist* pl2 = NULL;
    cpl_error_code code;
    /* Setup and run ...*/
    pl = cpl_propertylist_new();
    dist = sph_distortion_model_new(NULL, NULL);
    cpl_test_nonnull( pl );
    code = cpl_propertylist_append_int(pl, SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME, 2048);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_propertylist_append_int(pl, SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME,
            145);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME,
            10.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME,
            1.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME,
            100.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME,
            4.0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    code = cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME,
            1.333);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    pl2 = sph_distortion_model_get_proplist(dist);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( pl2 );
    cpl_msg_info(cpl_func, "Created %d distortion properties",
                 (int)cpl_propertylist_get_size(pl2));
    code = cpl_propertylist_append(pl, pl2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_propertylist_delete(pl2);
    pl2 = NULL;
    cpl_test(cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_DISTMAP_COEFFX " 0_0"));
    cpl_test(cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_DISTMAP_COEFFY " 0_0"));
    model = sph_ifs_lenslet_model_new_from_propertylist(pl);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( model );
    /*Verify */
    cpl_test_abs( model->detsize_microns, 2048.0, DBL_EPSILON);
    cpl_test_eq(model->detsize_pixels, 2048);
    cpl_test_abs( model->lensize_microns, 10.0, DBL_EPSILON);
    cpl_test_eq(model->lenslets_per_side, 145);
    cpl_test_abs( model->pixsize_microns, 1.0, DBL_EPSILON);
    cpl_test_abs( model->speclength_microns, 100.0, DBL_EPSILON);
    cpl_test_abs( model->specwidth_microns, 4.0, DBL_EPSILON);
    cpl_test_abs( model->rotangle, 1.333, DBL_EPSILON);
    cpl_test_nonnull( model->detector_distortion );
    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    sph_distortion_model_delete(dist);
    dist = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_get_as_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_get_as_propertylist(void) {
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl2 = NULL;

    /* Setup and run ...*/
    cpl_error_reset();
    pl = cpl_propertylist_new();
    cpl_test_nonnull( pl );
    cpl_propertylist_append_int(pl, SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME, 2048);
    cpl_propertylist_append_int(pl, SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME,
            145);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME,
            10.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME,
            1.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME,
            100.0);
    cpl_propertylist_append_double(pl, SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME,
            4.0);

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_ifs_lenslet_model_new_from_propertylist(pl);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( model );
    pl2 = sph_ifs_lenslet_model_get_as_propertylist(model);
    cpl_test_nonnull( pl2 );

    /*Verify */
    cpl_test_abs( model->detsize_microns, 2048, 0.0);
    cpl_test_eq(model->detsize_pixels, 2048);
    cpl_test_abs( model->lensize_microns, 10.0, 0.0);
    cpl_test_eq(model->lenslets_per_side, 145);
    cpl_test_abs( model->pixsize_microns, 1.0, 0.0);
    cpl_test_abs( model->speclength_microns, 100.0, 0.0);
    cpl_test_abs( model->specwidth_microns, 4.0, 0.0);

    cpl_test(
            cpl_propertylist_has(pl2, SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME));
    cpl_test_eq(
            cpl_propertylist_get_int( pl2, SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME),
            2048);
    cpl_test_eq(
            cpl_propertylist_get_int( pl2, SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME),
            145);
    cpl_test_abs(
            cpl_propertylist_get_double( pl2, SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME),
            10.0, 0.0);
    cpl_test_abs(
            cpl_propertylist_get_double( pl2, SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME),
            1.0, 0.0);
    cpl_test_abs(
            cpl_propertylist_get_double( pl2, SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME),
            100, 0.0);
    cpl_test_abs(
            cpl_propertylist_get_double( pl2, SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME),
            4.0, 0.0);

    cpl_propertylist_delete(pl);
    cpl_propertylist_delete(pl2);
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_predict_spec_loc
 function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_predict_spec_loc_pp(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* timage = NULL;
    cpl_image* diffim = NULL;
    sph_point_pattern* pp0 = NULL;
    sph_point_pattern* pp = NULL;
    cpl_image* timage_shifted = NULL;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    double offx = 1.0;
    double offy = 2.0;

    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();
    model->lenslets_per_side = 7;
    model->detsize_pixels = 128;

    cpl_test_nonnull( model );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    pp0 = sph_ifs_lenslet_model_predict_spec_loc_pp(model, 0.0, 0.0);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( pp0 );
    timage = sph_point_pattern_create_image(pp0, 128, 128, 2.0);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( timage );
    cpl_test_error(CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
    pp = sph_ifs_lenslet_model_predict_spec_loc_pp(model, offx, offy);
    cpl_test_nonnull( pp );
    timage_shifted = sph_point_pattern_create_image(pp, 128, 128, 2.0);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( timage_shifted );
    cpl_test_error(CPL_ERROR_NONE);
    diffim = cpl_image_new(model->detsize_pixels, model->detsize_pixels,
            CPL_TYPE_DOUBLE);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( diffim );
    cpl_image_save(timage,
            "sph_lenslet_model_predict_spec_loc_pp_shift_residual.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(timage_shifted,
            "sph_lenslet_model_predict_spec_loc_pp_shift_residual.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);

    for (xx = 0; xx < model->detsize_pixels; ++xx) {
        for (yy = 0; yy < model->detsize_pixels; ++yy) {
            if (xx + offx >= 0 && yy + offy >= 0
                    && yy + offy < model->detsize_pixels
                    && xx + offx < model->detsize_pixels) {
                cpl_image_set(
                        diffim,
                        xx + 1,
                        yy + 1,
                        cpl_image_get(timage, xx + 1, yy + 1, &bpix)
                                - cpl_image_get(timage_shifted, xx + 1 + offx,
                                        yy + 1 + offy, &bpix));
            }
        }
    }cpl_test_abs(cpl_image_get_absflux_window(diffim,20,20,108,108),
            0.0, 0.00001);

    cpl_image_save(diffim,
            "sph_lenslet_model_predict_spec_loc_pp_shift_residual.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);

    sph_point_pattern_delete(pp);
    pp = NULL;
    sph_point_pattern_delete(pp0);
    pp0 = NULL;
    cpl_image_delete(timage);
    timage = NULL;
    cpl_image_delete(timage_shifted);
    timage_shifted = NULL;
    cpl_image_delete(diffim);
    diffim = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_predict_spec_loc
 function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_predict_spec_loc_pp_model(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* timage = NULL;
    cpl_image* diffim = NULL;
    sph_point_pattern* pp0 = NULL;
    sph_point_pattern* pp = NULL;
    cpl_image* timage_shifted = NULL;
    int xx = 0;
    int yy = 0;
    int bpix = 0;
    double offx = 1.0;
    double offy = 2.0;

    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();
    model->lenslets_per_side = 7;
    model->detsize_pixels = 128;

    cpl_test_nonnull( model );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    pp0 = sph_ifs_lenslet_model_predict_spec_loc_pp(model, 0.0, 0.0);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( pp0 );
    timage = sph_point_pattern_create_image(pp0, 128, 128, 2.0);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( timage );
    cpl_test_error(CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
    model->zero_offsetx = model->zero_offsetx + offx;
    model->zero_offsety = model->zero_offsety + offy;
    pp = sph_ifs_lenslet_model_predict_spec_loc_pp(model, 0.0, 0.0);
    cpl_test_nonnull( pp );
    timage_shifted = sph_point_pattern_create_image(pp, 128, 128, 2.0);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( timage_shifted );
    cpl_test_error(CPL_ERROR_NONE);
    diffim = cpl_image_new(model->detsize_pixels, model->detsize_pixels,
            CPL_TYPE_DOUBLE);
    SPH_RAISE_CPL_RESET;cpl_test_nonnull( diffim );
    cpl_image_save(timage, "sph_lenslet_model_predict_spec_loc_pp_mod.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(timage_shifted,
            "sph_lenslet_model_predict_spec_loc_pp_mod.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_EXTEND);

    for (xx = 0; xx < model->detsize_pixels; ++xx) {
        for (yy = 0; yy < model->detsize_pixels; ++yy) {
            if (xx + offx >= 0 && yy + offy >= 0
                    && yy + offy < model->detsize_pixels
                    && xx + offx < model->detsize_pixels) {
                cpl_image_set(
                        diffim,
                        xx + 1,
                        yy + 1,
                        cpl_image_get(timage, xx + 1, yy + 1, &bpix)
                                - cpl_image_get(timage_shifted, xx + 1 + offx,
                                        yy + 1 + offy, &bpix));
            }
        }
    }cpl_test_abs(cpl_image_get_absflux_window(diffim,20,20,108,108),
            0.0, 0.00001);

    cpl_image_save(diffim, "sph_lenslet_model_predict_spec_loc_pp_mod.fits",
            CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);

    sph_point_pattern_delete(pp);
    pp = NULL;
    sph_point_pattern_delete(pp0);
    pp0 = NULL;
    cpl_image_delete(timage);
    timage = NULL;
    cpl_image_delete(timage_shifted);
    timage_shifted = NULL;
    cpl_image_delete(diffim);
    diffim = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_get_poly function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_get_poly(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* timage = NULL;
    sph_image_grid* tgrid = NULL;
    int uu = 0;
    int vv = 0;
    sph_polygon* pol = NULL;
    /* Setup and run ...*/
    int nrings = 0;

    cpl_error_reset();
    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    timage = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    tgrid = sph_image_grid_new_from_image(timage, -2500.5, 2500.5, -2500.5,
            2500.5);
    sph_image_grid_rotate_and_shift(tgrid, 0.0, 0.0, 20.0, 0.0, 0.0);
    nrings = model->lenslets_per_side;
    cpl_image_delete(timage);
    timage = NULL;
    cpl_test_eq(nrings, SPH_IFS_LENSLET_MODEL_LENSLETS_SIDE);
    for (vv = -nrings; vv <= nrings; ++vv) {
        if (vv >= 0) {
            for (uu = -nrings; uu <= nrings - vv; ++uu) {
                pol = sph_ifs_lenslet_model_get_poly(model, uu, vv);
                cpl_test_nonnull( pol );
                sph_image_grid_map_polygon(tgrid, pol, 4.0);
                sph_polygon_delete(pol);
                pol = NULL;
            }
        } else {
            for (uu = 0 - (nrings + vv); uu <= nrings; ++uu) {
                pol = sph_ifs_lenslet_model_get_poly(model, uu, vv);
                cpl_test_nonnull( pol );
                sph_image_grid_map_polygon(tgrid, pol, 4.0);
                sph_polygon_delete(pol);
                pol = NULL;
            }
        }
    }

    /*Verify */
    timage = sph_image_grid_extract_image(tgrid);
    rerr += cpl_image_save(timage, "lensletimage.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_image_grid_delete(tgrid);
    tgrid = NULL;
    cpl_image_delete(timage);
    timage = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_get_coords function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_get_coords(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    int u = -100000;
    int v = -100000;
    double dx = 0.0;
    double dy = 0.0;
    int ii = 0;
    sph_polygon* poly = NULL;
    double x = 0.0;
    double y = 0.0;
    unsigned long ulSeed = 1;
    gsl_rng* pRNG = NULL;
    int inside = 0;
    sph_image_grid* markgrid = NULL;
    cpl_image* markimage = NULL;

    /* Setup and run ...*/
    cpl_error_reset();
    pRNG = gsl_rng_alloc(gsl_rng_taus);

    gsl_rng_set(pRNG, ulSeed);

    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );

    markimage = cpl_image_new(548, 548, CPL_TYPE_DOUBLE);

    markgrid = sph_image_grid_new_from_image(markimage,
            -model->lensize_microns * model->lenslets_per_side,
            model->lensize_microns * model->lenslets_per_side,
            -model->lensize_microns * model->lenslets_per_side,
            model->lensize_microns * model->lenslets_per_side);
    dx = model->lensize_microns * CPL_MATH_SQRT3;
    dy = model->lensize_microns * 3.0 / 2.0;

    rerr = sph_ifs_lenslet_model_get_coords(model, 0.0, 0.0, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(u);
    cpl_test_zero(v);

    rerr = sph_ifs_lenslet_model_get_coords(model, dx, 0.0, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(u, 1);
    cpl_test_zero(v);

    rerr = sph_ifs_lenslet_model_get_coords(model, dx / 2.0, dy, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(u);
    cpl_test_eq(v, 1);

    rerr = sph_ifs_lenslet_model_get_coords(model, dx - 0.0001, dy, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(u);
    cpl_test_eq(v, 1);

    rerr = sph_ifs_lenslet_model_get_coords(model, dx + 0.00001, dy, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(u, 1);
    cpl_test_eq(v, 1);
    rerr = sph_ifs_lenslet_model_get_coords(model, dx, dy * 2.0, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(u);
    cpl_test_eq(v, 2);
    rerr = sph_ifs_lenslet_model_get_coords(model, dx * 1.1, dy * 2.0, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(u);
    cpl_test_eq(v, 2);
    rerr = sph_ifs_lenslet_model_get_coords(model, dx * 1.3, dy * 2.0, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_zero(u);
    cpl_test_eq(v, 2);
    rerr = sph_ifs_lenslet_model_get_coords(model,
            dx * model->lenslets_per_side, 0.0, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(u, model->lenslets_per_side);
    cpl_test_zero(v);
    rerr = sph_ifs_lenslet_model_get_coords(model, 0.00001,
            dy * model->lenslets_per_side, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(u, -72);
    cpl_test_eq(v, model->lenslets_per_side);
    rerr = sph_ifs_lenslet_model_get_coords(model, 0.0,
            dy * (model->lenslets_per_side + 1), &u, &v);
    cpl_test_eq(rerr, SPH_IFS_LENSLET_MODEL_OUTSIDE_ARRAY);
    rerr = sph_ifs_lenslet_model_get_coords(model, -52.6 * dx, 87.5005 * dy, &u,
            &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(u, -96);
    cpl_test_eq(v, 87);
    poly = sph_ifs_lenslet_model_get_poly(model, u, v);
    cpl_test_nonnull( poly );
    inside = sph_polygon_test_inside(poly, -52.6 * dx, 87.5005 * dy);
    cpl_test( inside);
    rerr = sph_ifs_lenslet_model_get_coords(model, -556.78, -1201.48, &u, &v);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_eq(u, 1);
    cpl_test_eq(v, -5);
    sph_polygon_delete(poly);
    poly = NULL;
    poly = sph_ifs_lenslet_model_get_poly(model, u, v);
    cpl_test_nonnull( poly );
    inside = sph_polygon_test_inside(poly, -556.78, -1201.48);
    cpl_test( inside);
    sph_polygon_delete(poly);
    poly = NULL;
    for (ii = 0; ii < 100000; ++ii) {
        x = gsl_ran_flat(pRNG,
                -model->lensize_microns * (model->lenslets_per_side + 0.5),
                model->lensize_microns * (model->lenslets_per_side + 0.5));
        y = gsl_ran_flat(pRNG,
                -model->lensize_microns * (model->lenslets_per_side + 0.5),
                model->lensize_microns * (model->lenslets_per_side + 0.5));

        rerr = sph_ifs_lenslet_model_get_coords(model, x, y, &u, &v);
        cpl_test_eq_error(rerr, CPL_ERROR_NONE);
        poly = sph_ifs_lenslet_model_get_poly(model, u, v);
        cpl_test_nonnull( poly );
        inside = sph_polygon_test_inside(poly, x, y);
        cpl_test( inside);

        if (!inside) {
            sph_image_grid_mark_point(markgrid, x, y, 1.0);
        }

        sph_polygon_delete(poly);
        poly = NULL;
    }
    cpl_image_delete(markimage);
    markimage = NULL;
    markimage = sph_image_grid_extract_image(markgrid);
    cpl_image_save(markimage, "markimage.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_DEFAULT);
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_image_delete(markimage);
    markimage = NULL;
    sph_image_grid_delete(markgrid);
    markgrid = NULL;
    gsl_rng_free(pRNG);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ldt_predict_spec_loc
 function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_create_pdt_images(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* image_wav = NULL;
    cpl_image* image_id = NULL;
    cpl_image* specidim = NULL;
    cpl_image* image_ill = NULL;
    cpl_image* image_dw = NULL;

    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();
    model->lenslets_per_side = 64;

    cpl_test_nonnull( model );

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Started PDT image creations...");
    rerr = sph_ifs_lenslet_model_create_pdt_images(model, &image_wav, &image_id,
            &image_ill, &image_dw, 0.0, 0.0);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Done.");

    cpl_image_save(image_id, "idimage.fits", CPL_TYPE_UNSPECIFIED, NULL,
            CPL_IO_CREATE);
    cpl_test_eq( cpl_image_get_max( image_id),
            sph_ifs_lenslet_model_get_nlens(model));
    cpl_image_save(image_ill, "illimage.fits", CPL_TYPE_UNSPECIFIED, NULL,
            CPL_IO_CREATE);
    cpl_test_abs( cpl_image_get_max( image_ill), 1.0, 0.0000001);
    cpl_test_abs( cpl_image_get_min( image_ill), 0.0, 0.0000001);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( image_wav );
    cpl_test_nonnull( image_id );
    cpl_test_nonnull( image_dw );
    cpl_test_nonnull( image_ill );
    cpl_image_delete(specidim);
    cpl_image_delete(image_wav);
    cpl_image_delete(image_dw);
    cpl_image_delete(image_ill);
    cpl_image_delete(image_id);
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_get_index function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_get_index(void) {
    sph_ifs_lenslet_model* model = NULL;
    int index = 0;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );
    index = sph_ifs_lenslet_model_get_index(model, 0, -145);
    cpl_test_eq(index, 1);
    index = sph_ifs_lenslet_model_get_index(model, 0, 145);
    cpl_test_eq(index, sph_ifs_lenslet_model_get_nlens( model ));

    sph_ifs_lenslet_model_delete(model);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_delete(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( model );
    sph_ifs_lenslet_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_set_lenslet_distortion(void) {
    sph_ifs_lenslet_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    sph_distortion_model* dist = NULL;
    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    dist = sph_distortion_model_new(NULL, NULL);
    sph_ifs_lenslet_model_set_lenslet_distortion(model, dist);

    /*Verify */
    cpl_test_nonnull( model->lenslet_distortion );

    sph_ifs_lenslet_model_delete(model);
    //sph_distortion_model_delete(dist);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ifs_lenslet_model_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_lenslet_model_set_get_hex_w_lenslet_distortion(void) {
    sph_ifs_lenslet_model* model_distorted = NULL;
    sph_ifs_lenslet_model* model_undistorted = NULL;
    int rerr = CPL_ERROR_NONE;
    sph_distortion_model* dist = NULL;
    cpl_size pows[2];
    sph_polygon* hex = NULL;
    int u = 0;
    int v = 0;
    double x1 = 0.0;
    double x2 = 0.0;
    double y1 = 0.0;
    double y2 = 0.0;
    double dx = 0.0;
    double dy = 0.0;
    double cx0 = 0.0;
    double cy0 = 0.0;
    double cx1 = 0.01;
    double cy1 = 0.01;
    double xx1 = 0.0;
    double yy1 = 0.0;
    /* Setup and run ...*/
    cpl_error_reset();

    model_distorted = sph_ifs_lenslet_model_new();
    model_undistorted = sph_ifs_lenslet_model_new();

    cpl_test_nonnull( model_distorted );
    cpl_test_nonnull( model_undistorted );
    dist = sph_distortion_model_new(NULL, NULL);
    pows[0] = 1;
    pows[1] = 0;
    cpl_polynomial_set_coeff(dist->polyx, pows, cx1);
    pows[0] = 0;
    pows[1] = 1;
    cpl_polynomial_set_coeff(dist->polyy, pows, cy1);
    pows[0] = 0;
    pows[1] = 0;
    cpl_polynomial_set_coeff(dist->polyx, pows, cx0);
    cpl_polynomial_set_coeff(dist->polyy, pows, cy0);
    sph_ifs_lenslet_model_set_lenslet_distortion(model_distorted, dist);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    u = 10;
    v = 10;
    hex = sph_ifs_lenslet_model_get_poly(model_distorted, u, v);
    x1 = sph_polygon_get_midx(hex);
    y1 = sph_polygon_get_midy(hex);
    sph_polygon_get_midxy_(hex, &xx1, &yy1);
    cpl_test_rel(x1, xx1, DBL_EPSILON);
    cpl_test_rel(y1, yy1, DBL_EPSILON);

    sph_ifs_lenslet_model_get_centre(model_distorted, u, v, &x2, &y2);
    /*Verify */
    cpl_test_abs(x1, x2, 0.0001);
    cpl_test_abs(y1, y2, 0.0001);
    sph_ifs_lenslet_model_get_centre(model_undistorted, u, v, &x1, &y1);
    dx = x1 * cx1 + cx0;
    dy = y1 * cy1 + cy0;
    cpl_test_abs(x1-x2, -dx, 0.0001);
    cpl_test_abs(y1-y2, -dy, 0.0001);
    sph_polygon_delete(hex);
    hex = NULL;

    sph_ifs_lenslet_model_delete(model_distorted);
    sph_ifs_lenslet_model_delete(model_undistorted);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

static
int cutest_init_ifs_lenslet_model_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static
int cutest_clean_ifs_lenslet_model_testsuite(void) {
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_ifs_lenslet_model_test",
            cutest_init_ifs_lenslet_model_testsuite,
            cutest_clean_ifs_lenslet_model_testsuite);

    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    cutest_ifs_lenslet_model_predict_spec_loc_pp_model();
    cutest_ifs_lenslet_model_predict_spec_loc_pp();

    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_new",
                    cutest_ifs_lenslet_model_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_create_pdt_images",
                    cutest_ifs_lenslet_model_create_pdt_images)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ifs_lenslet_model_new_from_propertylist",
                    cutest_ifs_lenslet_model_new_from_propertylist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ifs_lenslet_model_new_from_propertylist_distmodel",
                    cutest_ifs_lenslet_model_new_from_propertylist_distmodel)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_save",
                    cutest_ifs_lenslet_model_save)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_load",
                    cutest_ifs_lenslet_model_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_get_lambda_vector",
                    cutest_ifs_lenslet_model_get_lambda_vector)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_get_as_propertylist",
                    cutest_ifs_lenslet_model_get_as_propertylist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_get_coords",
                    cutest_ifs_lenslet_model_get_coords)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_get_index",
                    cutest_ifs_lenslet_model_get_index)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_delete",
                    cutest_ifs_lenslet_model_delete)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_get_centre",
                    cutest_ifs_lenslet_model_get_centre)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ifs_lenslet_model_get_poly",
                    cutest_ifs_lenslet_model_get_poly)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ifs_lenslet_model_set_lenslet_distortion",
                    cutest_ifs_lenslet_model_set_lenslet_distortion)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ifs_lenslet_model_set_get_hex_w_lenslet_distortion",
                    cutest_ifs_lenslet_model_set_get_hex_w_lenslet_distortion)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
