/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_ird_instrument_model.h"
#include "sph_dataset.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"
#include "sph_test.h"


#include "sph_test.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_image_grid.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include "sph_test_image_tools.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_ird_instrument_model    unit tests for sph_ird_instrument_model
 * and associated functions.
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_ird_instrument_model_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

static
int cutest_clean_ird_instrument_model_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_new function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_new(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( model );
    cpl_test_abs( model->detsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_MICRONS, 0.0);
    cpl_test_eq( model->detsize_pixels_x,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_X);
    cpl_test_eq( model->detsize_pixels_y,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_Y);
    cpl_test_abs( model->pixsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS, 0.0);
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_save function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_save(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    rerr = sph_ird_instrument_model_save(model,
            "ird_instrument_model_test.ini");

    /*Verify */
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_load function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_load(void) {
    sph_ird_instrument_model* model = NULL;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_load("ird_instrument_model_test.ini");

    cpl_test_nonnull( model );
    /*Verify */
    cpl_test_abs( model->detsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_MICRONS, 0.0);
    cpl_test_eq( model->detsize_pixels_x,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_X);
    cpl_test_eq( model->detsize_pixels_y,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_Y);
    cpl_test_abs( model->pixsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS, 0.0);
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_new_from_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_new_from_propertylist(void) {
    sph_ird_instrument_model* model = NULL;
    cpl_propertylist* pl = NULL;

    /* Setup and run ...*/
    cpl_error_reset();
    pl = cpl_propertylist_new();
    cpl_test_nonnull( pl );
    cpl_propertylist_append_int(pl, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME,
            2048);
    cpl_propertylist_append_int(pl, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME,
            1024);
    cpl_propertylist_append_double(pl, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME,
            18.0);

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_ird_instrument_model_new_from_propertylist(pl);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( model );
    /*Verify */
    cpl_test_abs( model->detsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_MICRONS, 0.0);
    cpl_test_eq( model->detsize_pixels_x,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_X);
    cpl_test_eq( model->detsize_pixels_y,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_Y);
    cpl_test_abs( model->pixsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS, 0.0);
    cpl_propertylist_delete(pl);
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_get_as_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_get_centres(void) {
    sph_ird_instrument_model* model = NULL;
    cpl_propertylist* pl = NULL;
    double x = 0.0;
    double y = 0.0;

    /* Setup and run ...*/
    cpl_error_reset();
    pl = cpl_propertylist_new();
    cpl_test_nonnull( pl );
    cpl_propertylist_append_int(pl, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME,
            2048);
    cpl_propertylist_append_int(pl, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME,
            1024);
    cpl_propertylist_append_double(pl, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME,
            18.0);

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_ird_instrument_model_new_from_propertylist(pl);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(sph_ird_instrument_model_get_centre_left(model,&x, &y),
            CPL_ERROR_NONE);
    cpl_test_abs(x, 512.0, 0.0000001);
    cpl_test_abs(y, 512.0, 0.0000001);
    cpl_test_eq(sph_ird_instrument_model_get_centre_right(model,&x, &y),
            CPL_ERROR_NONE);
    cpl_test_abs(x, 1024+512.0, 0.0000001);
    cpl_test_abs(y, 512.0, 0.0000001);
    cpl_test_eq(
            sph_ird_instrument_model_get_centre_left_subframe_coords(model,&x,&y),
            CPL_ERROR_NONE);
    cpl_test_abs(x, 512.0, 0.0000001);
    cpl_test_abs(y, 512.0, 0.0000001);
    cpl_test_eq(
            sph_ird_instrument_model_get_centre_right_subframe_coords(model,&x,&y),
            CPL_ERROR_NONE);
    cpl_test_abs(x, 512.0, 0.0000001);
    cpl_test_abs(y, 512.0, 0.0000001);
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_propertylist_delete(pl);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_get_as_propertylist function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_get_as_propertylist(void) {
    sph_ird_instrument_model* model = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl2 = NULL;

    /* Setup and run ...*/
    cpl_error_reset();
    pl = cpl_propertylist_new();
    cpl_test_nonnull( pl );
    cpl_propertylist_append_int(pl, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME,
            2048);
    cpl_propertylist_append_int(pl, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME,
            1024);
    cpl_propertylist_append_double(pl, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME,
            18.0);

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_ird_instrument_model_new_from_propertylist(pl);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull( model );
    pl2 = sph_ird_instrument_model_get_as_propertylist(model);
    cpl_test_nonnull( pl2 );

    /*Verify */
    cpl_test_abs( model->detsize_microns, 2048 * 18.0, 0.0);
    cpl_test_eq(model->detsize_pixels_x, 2048);
    cpl_test_eq(model->detsize_pixels_y, 1024);
    cpl_test_abs( model->pixsize_microns, 18.0, 0.0);

    cpl_test_abs( model->detsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_MICRONS, 0.0);
    cpl_test_eq( model->detsize_pixels_x,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_X);
    cpl_test_eq( model->detsize_pixels_y,
            SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_PIX_Y);
    cpl_test_abs( model->pixsize_microns,
            SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS, 0.0);

    cpl_test(
            cpl_propertylist_has(pl2, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME));
    cpl_test(
            cpl_propertylist_has(pl2, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME));
    cpl_test_eq(
            cpl_propertylist_get_int( pl2, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_X_NAME),
            2048);
    cpl_test_eq(
            cpl_propertylist_get_int( pl2, SPH_IRD_INSTRUMENT_MODEL_DET_SIZE_Y_NAME),
            1024);
    cpl_test_abs(
            cpl_propertylist_get_double( pl2, SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_NAME),
            18.0, 0.0);

    cpl_propertylist_delete(pl);
    cpl_propertylist_delete(pl2);
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_extract_master_frame.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_extract_left_master_frame(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    sph_master_frame* mframein = NULL;
    sph_master_frame* mframeout = NULL;
    double totval = 0.0;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( model );

    image = sph_test_image_tools_create_flat_image_double(2048, 2048, 121.0);
    cpl_test_nonnull( image );
    mframein = sph_master_frame_new_from_cpl_image(image);
    cpl_test_nonnull( mframein );
    mframeout = sph_ird_instrument_model_extract_left_master_frame(model,
            mframein);
    /*Verify */
    cpl_test_nonnull( mframeout );
    totval = sph_master_frame_get_absflux(mframeout, NULL);
    cpl_test_abs( totval, 121.0 * 1024 * 1024, 0.001);

    sph_ird_instrument_model_delete(model);
    cpl_image_delete(image);
    cpl_test_error(CPL_ERROR_NONE);
    sph_master_frame_delete(mframein);
    sph_master_frame_delete(mframeout);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_extract_master_frame.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_extract_left_master_frame_window_mode(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    sph_master_frame* mframein = NULL;
    sph_master_frame* mframeout = NULL;
    double totval = 0.0;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    rerr = sph_ird_instrument_model_set_windows(model, 128);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( model );

    image = sph_test_image_tools_create_flat_image_double(2048, 2048, 121.0);
    cpl_test_nonnull( image );
    mframein = sph_master_frame_new_from_cpl_image(image);
    cpl_test_nonnull( mframein );
    mframeout = sph_ird_instrument_model_extract_left_master_frame(model,
            mframein);
    /*Verify */
    cpl_test_nonnull( mframeout );
    totval = sph_master_frame_get_absflux(mframeout, NULL);
    cpl_test_abs( totval, 121.0 * 128 * 128, 0.001);
    sph_master_frame_save(mframeout, "blblblb.fits", NULL);
    sph_ird_instrument_model_delete(model);
    cpl_image_delete(image);
    cpl_test_error(CPL_ERROR_NONE);
    sph_master_frame_delete(mframein);
    sph_master_frame_delete(mframeout);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_extract_master_frame.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_extract_right_master_frame_window_mode(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    sph_master_frame* mframein = NULL;
    sph_master_frame* mframeout = NULL;
    double totval = 0.0;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    rerr = sph_ird_instrument_model_set_windows(model, 128);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( model );

    image = sph_test_image_tools_create_flat_image_double(2048, 2048, 121.0);
    cpl_test_nonnull( image );
    mframein = sph_master_frame_new_from_cpl_image(image);
    cpl_test_nonnull( mframein );
    mframeout = sph_ird_instrument_model_extract_right_master_frame(model,
            mframein);
    /*Verify */
    cpl_test_nonnull( mframeout );
    totval = sph_master_frame_get_absflux(mframeout, NULL);
    cpl_test_abs( totval, 121.0 * 128 * 128, 0.001);

    sph_ird_instrument_model_delete(model);
    cpl_image_delete(image);
    cpl_test_error(CPL_ERROR_NONE);
    sph_master_frame_delete(mframein);
    sph_master_frame_delete(mframeout);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_extract_master_frame.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_adjust_centre_window(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    double cxnew = 0.0;
    double cynew = 0.0;
    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq(
            sph_ird_instrument_model_adjust_centre_coords( model,512.0,512.0,&cxnew,&cynew),
            CPL_ERROR_NONE);
    cpl_test_abs(cxnew, 512.0, 0.001);
    cpl_test_abs(cynew, 512.0, 0.001);

    rerr = sph_ird_instrument_model_set_windows(model, 1024);

    cpl_test_eq(
            sph_ird_instrument_model_adjust_centre_coords( model,512.0,512.0,&cxnew,&cynew),
            CPL_ERROR_NONE);
    cpl_test_abs(cxnew, 512.0, 0.001);
    cpl_test_abs(cynew, 512.0, 0.001);

    rerr = sph_ird_instrument_model_set_windows(model, 128);

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( model );

    cpl_test_eq(
            sph_ird_instrument_model_adjust_centre_coords( model,512.0,512.0,&cxnew,&cynew),
            CPL_ERROR_NONE);
    cpl_test_abs(cxnew, 64.0, 0.001);
    cpl_test_abs(cynew, 64.0, 0.001);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_eq(
            sph_ird_instrument_model_adjust_centre_coords( model,511.0,515.0,&cxnew,&cynew),
            CPL_ERROR_NONE);
    cpl_test_abs(cxnew, 63.0, 0.001);
    cpl_test_abs(cynew, 67.0, 0.001);
    cpl_test_error(CPL_ERROR_NONE);

    sph_ird_instrument_model_delete(model);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_extract_master_frame.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_adjust_centre_window_v(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_vector* cxin = NULL;
    cpl_vector* cyin = NULL;
    cpl_vector* cxnew = NULL;
    cpl_vector* cynew = NULL;
    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    cxin = cpl_vector_new(10);
    cyin = cpl_vector_new(10);
    cpl_vector_fill(cxin, 512.0);
    cpl_vector_fill(cyin, 512.0);

    cpl_test_eq(
            sph_ird_instrument_model_adjust_centre_coords_v( model,cxin,cyin,&cxnew,&cynew),
            CPL_ERROR_NONE);
    cpl_test_nonnull( cxnew );
    cpl_test_nonnull( cynew );
    cpl_test_abs(cpl_vector_get(cxnew,1), 512.0, 0.001);
    cpl_test_abs(cpl_vector_get(cynew,1), 512.0, 0.001);
    cpl_vector_delete(cxnew);
    cxnew = NULL;
    cpl_vector_delete(cynew);
    cynew = NULL;

    rerr = sph_ird_instrument_model_set_windows(model, 1024);

    cpl_test_eq(
            sph_ird_instrument_model_adjust_centre_coords_v( model,cxin,cyin,&cxnew,&cynew),
            CPL_ERROR_NONE);
    cpl_test_nonnull( cxnew );
    cpl_test_nonnull( cynew );
    cpl_test_abs(cpl_vector_get(cxnew,1), 512.0, 0.001);
    cpl_test_abs(cpl_vector_get(cynew,1), 512.0, 0.001);
    cpl_vector_delete(cxnew);
    cxnew = NULL;
    cpl_vector_delete(cynew);
    cynew = NULL;

    cpl_test_error(CPL_ERROR_NONE);

    sph_ird_instrument_model_delete(model);
    cpl_vector_delete(cxin);
    cxin = NULL;
    cpl_vector_delete(cyin);
    cyin = NULL;

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_extract_master_frame.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_extract_right_master_frame(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;
    cpl_image* image = NULL;
    sph_master_frame* mframein = NULL;
    sph_master_frame* mframeout = NULL;
    double totval = 0.0;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_test_nonnull( model );

    image = sph_test_image_tools_create_flat_image_double(2048, 2048, 121.0);
    cpl_test_nonnull( image );
    mframein = sph_master_frame_new_from_cpl_image(image);
    cpl_test_nonnull( mframein );
    mframeout = sph_ird_instrument_model_extract_right_master_frame(model,
            mframein);
    /*Verify */
    cpl_test_nonnull( mframeout );
    totval = sph_master_frame_get_absflux(mframeout, NULL);
    cpl_test_abs( totval, 121.0 * 1024 * 1024, 0.001);

    sph_ird_instrument_model_delete(model);
    cpl_image_delete(image);
    cpl_test_error(CPL_ERROR_NONE);
    sph_master_frame_delete(mframein);
    sph_master_frame_delete(mframeout);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_ird_instrument_model_delete function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ird_instrument_model_delete(void) {
    sph_ird_instrument_model* model = NULL;
    int rerr = CPL_ERROR_NONE;

    /* Setup and run ...*/
    cpl_error_reset();

    model = sph_ird_instrument_model_new();

    cpl_test_nonnull( model );

    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    /*Verify */
    cpl_test_nonnull( model );
    sph_ird_instrument_model_delete(model);
    cpl_test_error(CPL_ERROR_NONE);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("sph_ird_instrument_model_test",
            cutest_init_ird_instrument_model_testsuite,
            cutest_clean_ird_instrument_model_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_ird_instrument_model_new",
                    cutest_ird_instrument_model_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ird_instrument_model_new_from_propertylist",
                    cutest_ird_instrument_model_new_from_propertylist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ird_instrument_model_save",
                    cutest_ird_instrument_model_save)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ird_instrument_model_load",
                    cutest_ird_instrument_model_load)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ird_instrument_model_get_as_propertylist",
                    cutest_ird_instrument_model_get_as_propertylist)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ird_instrument_model_extract_left_master_frame",
                    cutest_ird_instrument_model_extract_left_master_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ird_instrument_model_extract_right_master_frame",
                    cutest_ird_instrument_model_extract_right_master_frame)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_ird_instrument_model_extract_left_master_frame_window",
                    cutest_ird_instrument_model_extract_left_master_frame_window_mode)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "sph_ird_instrument_model_extract_right_master_frame_window",
                    cutest_ird_instrument_model_extract_right_master_frame_window_mode)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ird_instrument_adjust_centre_windows",
                    cutest_ird_instrument_model_adjust_centre_window)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ird_instrument_model_get_centres",
                    cutest_ird_instrument_model_get_centres)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_ird_instrument_model_adjust_centre_win_v",
                    cutest_ird_instrument_model_adjust_centre_window_v)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_ird_instrument_model_delete",
                    cutest_ird_instrument_model_delete)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
