/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>
#include <math.h>



#include "sph_master_frame.h"
#include "sph_common_keywords.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup cutest_sph_master_frame Unit test for the sph_master_frame module            
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

sph_master_frame* __mframe = NULL;

static sph_master_frame*
create_mframe(void) {
    int nx = 256;
    int ny = 256;
    if (__mframe)
        return __mframe;
    __mframe = sph_master_frame_new(nx, ny);
    cpl_image_delete(__mframe->badpixelmap);
    __mframe->badpixelmap = NULL;
    __mframe->badpixelmap = sph_test_image_tools_create_flat_image_int(nx, ny,
            0);
    cpl_image_delete(__mframe->rmsmap);
    __mframe->rmsmap = NULL;
    __mframe->rmsmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    cpl_image_delete(__mframe->ncombmap);
    __mframe->ncombmap = NULL;
    __mframe->ncombmap = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    cpl_image_delete(__mframe->image);
    __mframe->image = NULL;
    __mframe->image = sph_test_image_tools_create_flat_image_double(nx, ny,
            10.0);
    return __mframe;
}

static
int cutest_init_testsuite(void) {
    sph_test_nop_code();
    __mframe = create_mframe();
    return __mframe == NULL ? 1 : 0;
}

static
int cutest_clean_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Close and dump errors.
     * -------------------------------------------------------------------*/
    sph_master_frame_delete(__mframe);
    sph_error_dump(SPH_ERROR_ERROR);

    return sph_end_test();
}
static
void cutest_null_params(void) {
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_test_nonnull( __mframe );
    rerr = sph_master_frame_save_dfs(__mframe, "master_frame_dfs.fits", NULL,
            NULL, NULL, "TEST_TAG", NULL, NULL, NULL);
    cpl_test_noneq(rerr, CPL_ERROR_NONE);
    SPH_RAISE_CPL_RESET;
}
static
void cutest_happy_case(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frameset* allframes = NULL;
    cpl_parameterlist* params = NULL;
    cpl_image* imA = NULL;

    allframes = cpl_frameset_new();
    imA = sph_test_image_tools_create_flat_image_double(256, 256, 10.0);
    cpl_frameset_insert(
            allframes,
            sph_test_frame_image(imA, CPL_BPP_32_SIGNED, "ATAG",
                    CPL_FRAME_GROUP_RAW));

    params = cpl_parameterlist_new();
    cpl_parameterlist_append(
            params,
            cpl_parameter_new_value("ANAME", CPL_TYPE_INT, "Descr", "Conext",
                    1.0));

    cpl_test_nonnull( __mframe );
    cpl_test_nonnull( allframes );
    cpl_test_nonnull( params );

    rerr = sph_master_frame_save_dfs(__mframe, "master_frame_dfs.fits",
            allframes, cpl_frameset_get_first(allframes), params, "TEST_TAG",
            "NORECIPE", "NOPIPE", NULL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    cpl_frameset_delete(allframes);
    cpl_parameterlist_delete(params);
    cpl_image_delete(imA);
}

static
void cutest_happy_case_extnames(void) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frameset* allframes = NULL;
    cpl_parameterlist* params = NULL;
    cpl_image* imA = NULL;

    allframes = cpl_frameset_new();
    imA = sph_test_image_tools_create_flat_image_double(256, 256, 10.0);
    cpl_frameset_insert(
            allframes,
            sph_test_frame_image(imA, CPL_BPP_32_SIGNED, "ATAG",
                    CPL_FRAME_GROUP_RAW));

    params = cpl_parameterlist_new();
    cpl_parameterlist_append(
            params,
            cpl_parameter_new_value("ANAME", CPL_TYPE_INT, "Descr", "Conext",
                    1.0));

    cpl_test_nonnull( __mframe );
    cpl_test_nonnull( allframes );
    cpl_test_nonnull( params );

    rerr = sph_master_frame_save_dfs(__mframe, "master_frame_dfs.fits",
            allframes, cpl_frameset_get_first(allframes), params, "TEST_TAG",
            "NORECIPE", "NOPIPE", NULL);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);
    //check extnames of the double image extensions saved in master frame
    cpl_test_eq(
            0,
            cpl_fits_find_extension("master_frame_dfs.fits", SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_IMAGE_EXTNAME));
    cpl_test_eq(
            1,
            cpl_fits_find_extension("master_frame_dfs.fits", SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_BADPIXMAP_EXTNAME));
    cpl_test_eq(
            2,
            cpl_fits_find_extension("master_frame_dfs.fits", SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_NCOMBMAP_EXTNAME));
    cpl_test_eq(
            3,
            cpl_fits_find_extension("master_frame_dfs.fits", SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_RMSMAP_EXTNAME));

    cpl_frameset_delete(allframes);
    cpl_parameterlist_delete(params);
    cpl_image_delete(imA);
}

/*----------------------------------------------------------------------------*/
/**
 @brief   Main to run the tests.
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    (void)sph_test_init();

    pSuite = sph_add_suite("cutest_sph_master_frame_save_dfs",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "testing NULL params", cutest_null_params)) {
        return sph_test_get_error();
    }
    if (NULL == sph_test_do(pSuite, "testing happy case", cutest_happy_case)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "testing happy case extnames check",
                    cutest_happy_case_extnames)) {
        return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
