/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_point_pattern.h"
#include "sph_utils.h"
#include "sph_test_image_tools.h"
#include "sph_fitting.h"

#include <gsl/gsl_rng.h>
#include <math.h>
#include <float.h>

/*----------------------------------------------------------------------------*/
/*-                            Declarations of private functions              */
/*----------------------------------------------------------------------------*/

static
int cutest_init_testsuite();
static
int cutest_clean_testsuite();
static
void cutest_sph_point_pattern_new();
static
void cutest_sph_point_pattern_new_from_table();
static
void cutest_sph_point_pattern_new_from_image();

static
void cutest_sph_point_pattern_rotate();

static
void cutest_sph_point_pattern_scale();

static
void cutest_sph_point_pattern_centralise_tiny_shifts_bug();
static
void cutest_sph_point_pattern_new_from_image_noisy();
static
void cutest_sph_point_pattern_find_closest();
static
void cutest_sph_point_pattern_centralise();
static
void cutest_sph_point_pattern_load_ascii();
static
void cutest_sph_point_pattern_find_closest_neighbour();
static
void cutest_sph_point_pattern_create_image();
static
void cutest_sph_point_pattern_create_image_regular();
static
void cutest_sph_point_pattern_create_image_accurate();
static
void cutest_sph_point_pattern_make_pairs();
static
void cutest_sph_point_pattern_find_square();
static
void cutest_sph_point_pattern_find_square_in_random_points();
static
void cutest_sph_point_pattern_score_square_perfect_square_gives_zero();

static sph_point_pattern*
cutest_sph_point_patter_create_pp__(int np);

static cpl_frame*
cutest_sph_point_pattern_helper_create_table__(sph_point_pattern* pp,
        const char* filename);

static cpl_image*
cutest_sph_point_pattern_helper_create_image__(int npx, int npy, double psize);


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    //sph_error_dump( SPH_ERROR_ERROR );
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_new(void) {
    sph_point_pattern* pp = NULL;
    pp = sph_point_pattern_new();
    cpl_test_nonnull( pp );
    sph_point_pattern_delete(pp);

    pp = sph_point_pattern_new_(17);
    cpl_test_nonnull( pp );
    sph_point_pattern_delete(pp);

}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_new_from_table(void) {
    sph_point_pattern* pp = cutest_sph_point_patter_create_pp__(50);
    cpl_frame* frame = cutest_sph_point_pattern_helper_create_table__(pp,
            "testtab.fits");

    sph_point_pattern_delete(pp);

    sph_filemanager_add_tmp_file(cpl_frame_get_filename(frame));
    pp = sph_point_pattern_new_from_table(frame, 1);
    cpl_test_nonnull( pp );
    cpl_test_eq(sph_point_pattern_get_size(pp), 50);
    sph_point_pattern_delete(pp);
    cpl_frame_delete(frame);
    sph_filemanager_clean();
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_rotate(void) {
    sph_point_pattern* pp = NULL;

    pp = sph_point_pattern_new_(2);
    cpl_test_nonnull( pp );
    sph_point_pattern_add_point(pp, 0.0, 0.0);
    sph_point_pattern_add_point(pp, 1.0, 1.0);
    cpl_test_eq(sph_point_pattern_get_size(pp), 2);

    cpl_test_eq(sph_point_pattern_rotate(pp, 0.0, 0.0, 45.0),
            CPL_ERROR_NONE);

    cpl_test_abs( sph_point_pattern_get_x(pp, 0), 0.0, 0.000000001);
    cpl_test_abs( sph_point_pattern_get_y(pp, 0), 0.0, 0.000000001);

    cpl_test_abs( sph_point_pattern_get_x(pp, 1), 0.0, 0.000000001);
    cpl_test_abs( sph_point_pattern_get_y(pp, 1), sqrt(1.+1.),
            0.000000001);

    sph_point_pattern_save_ascii(pp, "pprot.txt");
    sph_point_pattern_delete(pp);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_scale(void) {
    sph_point_pattern* pp = NULL;

    pp = sph_point_pattern_new_(2);
    cpl_test_nonnull( pp );
    sph_point_pattern_add_point(pp, 0.0, 0.0);
    sph_point_pattern_add_point(pp, 1.0, 1.0);
    cpl_test_eq(sph_point_pattern_get_size(pp), 2);

    cpl_test_eq(sph_point_pattern_scale_around(pp, 0.0, 0.0, 2.0),
            CPL_ERROR_NONE);

    cpl_test_abs( sph_point_pattern_get_x(pp, 0), 0.0, 0.000000001);
    cpl_test_abs( sph_point_pattern_get_y(pp, 0), 0.0, 0.000000001);

    cpl_test_abs( sph_point_pattern_get_x(pp, 1), 2.0, 0.000000001);
    cpl_test_abs( sph_point_pattern_get_y(pp, 1), 2.0, 0.000000001);

    sph_point_pattern_save_ascii(pp, "pprot.txt");
    sph_point_pattern_delete(pp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_new_from_image(void) {
    sph_point_pattern* pp = NULL;
    cpl_image* im = cutest_sph_point_pattern_helper_create_image__(5, 5, 3.0);
    pp = sph_point_pattern_new_from_image(im, NULL, 5.0, 3.0, 3.0, 3.0, 25,
                                          NULL);
    cpl_test_nonnull( pp );
    cpl_test_eq(sph_point_pattern_get_size(pp), 25);
    sph_point_pattern_delete(pp);
    cpl_image_delete(im);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_centralise_tiny_shifts_bug(void) {
    sph_point_pattern* pp = NULL;
    sph_point_pattern* pp2 = NULL;
    cpl_image* im = NULL;
    cpl_image* testim = NULL;
    cpl_image* imorig = NULL;
    int nx = 256;
    int ny = 256;
    double mid[2];
    cpl_size midi;
    double doffx0 = 0.0;
    double doffy0 = 0.0;
    cpl_polynomial* polyx = cpl_polynomial_new(2);
    cpl_polynomial* polyy = cpl_polynomial_new(2);
    sph_distortion_model* model_no_dist =
            sph_distortion_model_new(polyx, polyy);
    int np = 5;

    pp = sph_distortion_model_create_point_pattern(model_no_dist, nx,
            ny, np);
    cpl_test_nonnull( pp );
    cpl_test_eq(sph_point_pattern_get_size(pp), np * np);

    im = sph_point_pattern_create_image(pp, nx, ny, 3.0);
    sph_point_pattern_delete(pp);
    pp = sph_point_pattern_new_from_image(im, NULL, 5.0, 3.0, 3.0, 3.0, 0,
                                          NULL);
    cpl_image_delete(im);

    im = sph_point_pattern_create_image(pp, nx, ny, 6.0);

    cpl_test_nonnull( im );
    cpl_image_save(im, "im_in.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sph_point_pattern_centralise(pp);

    pp2 = sph_point_pattern_new_from_image(im, NULL, 5.0, 3.0, 3.0, 3.0, 0,
                                          NULL);

    mid[0] = cpl_image_get_size_x(im) / 2.0;
    mid[1] = cpl_image_get_size_y(im) / 2.0;

    midi = sph_point_pattern_find_closest(pp2, mid);

    doffx0 = sph_point_pattern_get_x(pp2, midi);
    doffy0 = sph_point_pattern_get_y(pp2, midi);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_ERROR,
            "Offset: %6.3f, %6.3f", doffx0, doffy0);

    sph_point_pattern_add_offset(pp, doffx0, doffy0);

    testim = sph_point_pattern_create_image(pp, cpl_image_get_size_x(im),
            cpl_image_get_size_y(im), 6.0);

    cpl_image_save(testim, "im_out.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    cpl_image_subtract(im, testim);
    cpl_test_abs( cpl_image_get_absflux(im), 0.0, 0.01);
    sph_point_pattern_delete(pp);
    sph_point_pattern_delete(pp2);
    sph_distortion_model_delete(model_no_dist);
    cpl_image_delete(im);
    cpl_image_delete(imorig);
    cpl_image_delete(testim);
    cpl_polynomial_delete(polyx);
    cpl_polynomial_delete(polyy);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_new_from_image_noisy(void) {
    sph_point_pattern* pp = NULL;
    sph_point_pattern* ppnoise = NULL;
    cpl_image* im = cutest_sph_point_pattern_helper_create_image__(6, 6, 4.5);
    gsl_rng* pRNG = gsl_rng_alloc(gsl_rng_taus);
    int ii = 0;

    cpl_image_save(im, "testimA.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    pp = sph_point_pattern_new_from_image(im, NULL, 5.0, 4.0, 3.0, 3.0, 0,
                                          NULL);
    cpl_test_nonnull( pp );
    cpl_test_eq(sph_point_pattern_get_size(pp), 36);
    sph_test_image_tools_add_noise(im, 0.0001, pRNG);
    cpl_image_save(im, "testimAnoise.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    ppnoise = sph_point_pattern_new_from_image(im, NULL, 5.0, 4.0, 3.0, 3.0, 0,
                                          NULL);
    cpl_test_nonnull( ppnoise );
    cpl_test_eq(sph_point_pattern_get_size(ppnoise), 36);
    for (ii = 0; ii < 36; ++ii) {
        cpl_test_abs(
                sph_point_pattern_get_x(pp, ii)- sph_point_pattern_get_x(ppnoise, ii),
                0.0, 0.1);
        cpl_test_abs(
                sph_point_pattern_get_y(pp, ii)- sph_point_pattern_get_y(ppnoise, ii),
                0.0, 0.1);
    }
    cpl_image_delete(im);
    im = NULL;
    im = sph_point_pattern_create_image(ppnoise, 256, 256, 1.5);
    cpl_image_save(im, "testimB.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sph_point_pattern_delete(pp);
    pp = sph_point_pattern_new_from_image(im, NULL, 5.0, 3.0, 3.0, 3.0, 0,
                                          NULL);
    cpl_test_nonnull( pp );
    cpl_test_eq(sph_point_pattern_get_size(pp), 36);
    cpl_image_delete(im);
    im = NULL;
    im = sph_point_pattern_create_image(pp, 256, 256, 1.5);
    cpl_image_save(im, "testimC.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sph_point_pattern_delete(pp);
    sph_point_pattern_delete(ppnoise);
    cpl_image_delete(im);
    gsl_rng_free(pRNG);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_find_closest(void) {
    sph_point_pattern* pp = cutest_sph_point_patter_create_pp__(10);
    const double   point[2] = {50.1, 50.1};
    const double * p;
    cpl_size i;

    cpl_test_nonnull( pp );
    i = sph_point_pattern_find_closest(pp, point);
    cpl_test_leq(0, i);

    p = sph_point_pattern_get(pp, i);

    cpl_test_abs( p[0], 50.0, 0.001);
    cpl_test_abs( p[1], 50.0, 0.001);

    sph_point_pattern_delete(pp);

}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_centralise(void) {
    sph_point_pattern* pp = NULL;
    sph_point* p = NULL;
    pp = cutest_sph_point_patter_create_pp__(11);
    cpl_test_nonnull( pp );
    cpl_test_eq_error(sph_point_pattern_centralise(pp), CPL_ERROR_NONE);
    p = sph_point_pattern_get_point(pp, 5);
    cpl_test_nonnull( p );
    cpl_test_abs(p->x, 0.0, 0.01);
    cpl_test_abs(p->y, 0.0, 0.01);
    sph_point_pattern_delete(pp);
    sph_point_delete(p);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_make_pairs(void) {
    sph_point_pattern* pp = NULL;
    cpl_vector* pairsA = NULL;
    cpl_vector* pairsB = NULL;
    cpl_vector* distances = NULL;
    cpl_vector* dotprods = NULL;

    pp = sph_point_pattern_new_(1);
    sph_point_pattern_add_point(pp, 0.0, 0.0);
    sph_point_pattern_add_point(pp, 1.0, 1.0);
    sph_point_pattern_add_point(pp, 0.0, 1.0);
    sph_point_pattern_add_point(pp, 1.0, 0.0);
    cpl_test_nonnull( pp );
    cpl_test_eq(
            sph_point_pattern_make_pairs(pp,&pairsA,&pairsB,&distances,&dotprods),
            CPL_ERROR_NONE);
    cpl_test_nonnull( pairsA );
    cpl_test_nonnull( pairsB );
    cpl_test_nonnull( distances );
    cpl_test_eq(cpl_vector_get_size(pairsA), 6);
    cpl_test_eq(cpl_vector_get_size(pairsB), 6);
    cpl_test_eq(cpl_vector_get_size(distances), 6);
    cpl_test_eq(cpl_vector_get_size(dotprods), 6);
    cpl_test_zero(cpl_vector_get(pairsA, 0 ));
    cpl_test_eq(cpl_vector_get(pairsB, 0 ), 1);
    cpl_test_abs( cpl_vector_get(distances, 0 ), sqrt(2.0), 0.001);
    cpl_test_abs( cpl_vector_get(dotprods, 0 ), 1.0/sqrt(2.0), 0.001);
    cpl_test_zero(cpl_vector_get(pairsA, 1 ));
    cpl_test_eq(cpl_vector_get(pairsB, 1 ), 2);
    cpl_test_abs( cpl_vector_get(distances, 1 ), 1.0, 0.001);
    cpl_test_abs( cpl_vector_get(dotprods, 1 ), 0.0, 0.001);
    cpl_test_zero(cpl_vector_get(pairsA, 2 ));
    cpl_test_eq(cpl_vector_get(pairsB, 2 ), 3);
    cpl_test_abs( cpl_vector_get(distances, 2 ), 1.0, 0.001);
    cpl_test_abs( cpl_vector_get(dotprods, 2 ), 0.0, 0.001);
    cpl_test_eq(cpl_vector_get(pairsA, 3 ), 1);
    cpl_test_eq(cpl_vector_get(pairsB, 3 ), 2);
    cpl_test_abs( cpl_vector_get(distances, 3 ), 1.0, 0.001);
    cpl_test_abs( cpl_vector_get(dotprods, 3 ), 0.0, 0.001);
    cpl_test_eq(cpl_vector_get(pairsA, 4 ), 1);
    cpl_test_eq(cpl_vector_get(pairsB, 4 ), 3);
    cpl_test_abs( cpl_vector_get(distances, 4 ), 1.0, 0.001);
    cpl_test_abs( cpl_vector_get(dotprods, 4 ), 0.0, 0.001);
    cpl_test_eq(cpl_vector_get(pairsA, 5 ), 2);
    cpl_test_eq(cpl_vector_get(pairsB, 5 ), 3);
    cpl_test_abs( cpl_vector_get(distances, 5 ), sqrt(2.0), 0.001);
    cpl_test_abs( cpl_vector_get(dotprods, 5 ), 1.0/sqrt(2.0), 0.001);
    cpl_vector_delete(pairsA);
    cpl_vector_delete(pairsB);
    cpl_vector_delete(distances);
    cpl_vector_delete(dotprods);
    sph_point_pattern_delete(pp);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_score_square_perfect_square_gives_zero(void) {
    sph_point_pattern* pp = NULL;
    int p[4] = { 0, 1, 2, 3 };

    pp = sph_point_pattern_new_(1);
    sph_point_pattern_add_point(pp, 0.0, 0.0);
    sph_point_pattern_add_point(pp, 1.0, 1.0);
    sph_point_pattern_add_point(pp, 0.0, 1.0);
    sph_point_pattern_add_point(pp, 1.0, 0.0);
    cpl_test_nonnull( pp );

    cpl_test_abs(sph_point_pattern_score_square( pp, p ), 0.0,
            DBL_EPSILON);
    sph_point_pattern_delete(pp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_find_square(void) {
    sph_point_pattern* pp = NULL;
    sph_point_pattern* result = NULL;

    pp = sph_point_pattern_new_(1);
    sph_point_pattern_add_point(pp, 0.0, 0.0);
    sph_point_pattern_add_point(pp, 1.0, 1.0);
    sph_point_pattern_add_point(pp, 0.0, 1.0);
    sph_point_pattern_add_point(pp, 1.0, 0.0);
    //sph_point_pattern_add_point(pp,0.5,0.5);

    cpl_test_nonnull( pp );
    result = sph_point_pattern_find_square(pp);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( result );
    cpl_test_eq(sph_point_pattern_get_size(result), 4);
    sph_point_pattern_save_ascii(result, "testpp.txt");
    cpl_test_abs(sph_point_pattern_get_x(result, 0), 0.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 0), 0.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_x(result, 1), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 1), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_x(result, 2), 0.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 2), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_x(result, 3), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 3), 0.0, 0.001);
    sph_point_pattern_delete(pp);
    sph_point_pattern_delete(result);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_find_square_in_random_points(void) {
    sph_point_pattern* pp = NULL;
    sph_point_pattern* result = NULL;
    gsl_rng* pRNG = NULL;
    int i = 0;
    pRNG = gsl_rng_alloc(gsl_rng_taus);
    gsl_rng_set(pRNG, 1);

    pp = sph_point_pattern_new_(12);

    for (i = 0; i < 25; ++i) {
        sph_point_pattern_add_point(pp, gsl_ran_flat(pRNG, -10.0, 10.0),
                gsl_ran_flat(pRNG, -10.0, 10.0));
    }
    sph_point_pattern_add_point(pp, 0.0, 0.0);
    sph_point_pattern_add_point(pp, 1.0, 1.0);
    sph_point_pattern_add_point(pp, 0.0, 1.0);
    sph_point_pattern_add_point(pp, 1.0, 0.0);
    for (i = 0; i < 25; ++i) {
        sph_point_pattern_add_point(pp, gsl_ran_flat(pRNG, -10.0, 10.0),
                gsl_ran_flat(pRNG, -10.0, 10.0));
    }cpl_test_nonnull( pp );
    result = sph_point_pattern_find_square(pp);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( result );
    cpl_test_eq(sph_point_pattern_get_size(result), 4);
    sph_point_pattern_save_ascii(result, "testpp.txt");
    cpl_test_abs(sph_point_pattern_get_x(result, 0), 0.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 0), 0.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_x(result, 1), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 1), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_x(result, 2), 0.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 2), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_x(result, 3), 1.0, 0.001);
    cpl_test_abs(sph_point_pattern_get_y(result, 3), 0.0, 0.001);
    sph_point_pattern_delete(pp);
    sph_point_pattern_delete(result);
    gsl_rng_free(pRNG);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_load_ascii(void) {
    sph_point_pattern* self = NULL;
    FILE* f = NULL;

    f = fopen("point_pattern.txt", "w");
    fprintf(f, "%s\n", "# Hello this is a comment");
    fprintf(f, "%s\n", "# Hello this is a comment too");
    fprintf(f, "  5.0     2.0\n");
    fprintf(f, "  6.0     3.0\n");
    fprintf(f, "  7.0     4.0\n");
    fprintf(f, "  8.0     5.0\n");
    fclose(f);
    self = sph_point_pattern_load_ascii("point_pattern.txt");
    cpl_test_nonnull( self );
    cpl_test_eq(sph_point_pattern_get_size(self), 4);
    cpl_test_abs(sph_point_pattern_get_x(self, 0), 5.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_y(self, 0), 2.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_x(self, 1), 6.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_y(self, 1), 3.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_x(self, 2), 7.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_y(self, 2), 4.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_x(self, 3), 8.0, 0.00001);
    cpl_test_abs(sph_point_pattern_get_y(self, 3), 5.0, 0.00001);
    sph_point_pattern_delete(self);
    self = NULL;
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_find_closest_neighbour(void) {
    sph_point_pattern* pp = NULL;
    sph_point* p = NULL;
    double dist = 0.0;
    int n = 0;
    pp = cutest_sph_point_patter_create_pp__(11);
    cpl_test_nonnull( pp );
    cpl_test_eq_error(sph_point_pattern_centralise(pp), CPL_ERROR_NONE);
    n = sph_point_pattern_find_closest_neighbour(pp, 6, &dist);
    cpl_test_lt(0, n);
    p = sph_point_pattern_get_point(pp, n);
    cpl_test_nonnull( p );
    cpl_test_abs(p->x, 0.0, 0.01);
    cpl_test_abs(p->y, 0.0, 0.01);
    cpl_test_abs(dist, 14.1, 0.5);
    sph_point_pattern_delete(pp);
    sph_point_delete(p);
    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_create_image(void) {
    sph_point_pattern* pp = NULL;
    cpl_image* im = NULL;
    cpl_apertures* aps = NULL;
    pp = cutest_sph_point_patter_create_pp__(10);
    cpl_test_nonnull( pp );
    im = sph_point_pattern_create_image(pp, 256, 256, 2.0);
    cpl_test_nonnull( im );

    aps = cpl_apertures_extract_sigma(im, 5.0);
    cpl_test_eq(cpl_apertures_get_size(aps), 10);
    cpl_apertures_delete(aps);
    cpl_image_delete(im);
    sph_point_pattern_delete(pp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_create_image_regular(void) {
    sph_point_pattern* pp = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    pp = cutest_sph_point_patter_create_pp__(25);
    cpl_test_nonnull( pp );
    im = sph_point_pattern_create_image(pp, 256, 256, 2.0);
    cpl_test_nonnull( im );
    sph_point_pattern_add_offset(pp, 10.0, 10.0);
    im2 = sph_point_pattern_create_image(pp, 256, 256, 2.0);
    cpl_test_nonnull( im2 );

    cpl_image_subtract(im, im2);

    cpl_test_abs( cpl_image_get_max_window(im,50,50,150,150), 0.0,
            0.0001);
    cpl_test_abs( cpl_image_get_min_window(im,50,50,150,150), 0.0,
            0.0001);
    cpl_image_save(im, "imregular.fits", CPL_TYPE_UNSPECIFIED, NULL,
            CPL_IO_CREATE);
    cpl_image_delete(im);
    cpl_image_delete(im2);
    sph_point_pattern_delete(pp);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_point_pattern_create_image_accurate(void) {
    sph_point_pattern* pp = NULL;
    cpl_image* im = NULL;
    cpl_image* im2 = NULL;
    cpl_image* imerr = NULL;
    double xpos = 129.333;
    double ypos = 128.133;
    double xw = 3.0;
    double yw = 3.0;

    cpl_test_error(CPL_ERROR_NONE);
    pp = sph_point_pattern_new();
    cpl_test_nonnull( pp );
    sph_point_pattern_add_point(pp, 128.333, 128.333);

    im = sph_point_pattern_create_image(pp, 256, 256, 3.0);
    cpl_test_nonnull( im );
    cpl_test_error(CPL_ERROR_NONE);
    sph_point_pattern_add_offset(pp, 10.0, 10.0);
    im2 = sph_point_pattern_create_image(pp, 256, 256, 3.0);
    cpl_test_nonnull( im2 );
    cpl_test_error(CPL_ERROR_NONE);

    //sph_test_image_tools_apply_poisson_noise(im,NULL);
    imerr = cpl_image_power_create(im, 0.5);
    cpl_image_add_scalar(imerr, 0.01);
    cpl_image_add_scalar(im, 0.1);
    sph_fitting_fit_gauss2D(im, imerr, &xpos, &ypos, &xw, &yw, 3, 3);
    cpl_image_save(im, "imsingle.fits", CPL_TYPE_UNSPECIFIED, NULL,
            CPL_IO_CREATE);
    cpl_image_fill_gaussian(im2, 128.0, 128.0, 1000.0, 4.0, 4.0);
    cpl_image_save(im2, "imsingle2.fits", CPL_TYPE_UNSPECIFIED, NULL,
            CPL_IO_CREATE);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Position: %6.3f, %6.3f, width: %6.3f, %6.3f", xpos, ypos, xw, yw);
    cpl_test_abs(xpos, 128.333, 0.01);
    cpl_test_abs(ypos, 128.333, 0.01);
    cpl_image_delete(im);
    cpl_image_delete(im2);
    cpl_image_delete(imerr);
    sph_point_pattern_delete(pp);
    SPH_RAISE_CPL_RESET
    return;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/
static sph_point_pattern* cutest_sph_point_patter_create_pp__(int np) {
    sph_point_pattern* pp = sph_point_pattern_new();
    int ii;
    for (ii = 0; ii < np; ++ii) {
        sph_point_pattern_add_point(pp, ii * 10.0, ii * 10.0);
    }
    return pp;
}

static cpl_frame* cutest_sph_point_pattern_helper_create_table__(
        sph_point_pattern* pp, const char* filename) {
    cpl_frame* result = cpl_frame_new();

    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result, "POINT PATTERN TABLE");
    sph_point_pattern_save(pp, cpl_frame_get_filename(result));
    return result;
}

static cpl_image* cutest_sph_point_pattern_helper_create_image__(int npx,
        int npy, double psize) {
    cpl_image* image = cpl_image_new(256, 256, CPL_TYPE_DOUBLE);
    cpl_image* smallim = NULL;
    cpl_image* extractim = NULL;

    int size0 = (int) psize * 20;
    int size = size0;
    int nx = 256;
    int ny = 256;
    int yy = 0;
    int xx = 0;
    int ddx = 0;
    int ddy = 0;
    double xpos = 0.0;
    double ypos = 0.0;
    double xpos0 = 0.0;
    double ypos0 = 0.0;
    double stepx = (double) nx / (npx + 1);
    double stepy = (double) ny / (npy + 1);
    for (xx = 0; xx < npx; ++xx) {
        for (yy = 0; yy < npy; ++yy) {
            sph_error_code rerr;
            size = size0;
            xpos = stepx * (xx + 1);
            ypos = stepy * (yy + 1);
            xpos0 = xpos - size * 0.5;
            ypos0 = ypos - size * 0.5;
            if ((int) xpos0 + size >= nx) {
                size = nx - (int) xpos0;
            }
            if ((int) ypos0 + size >= ny)
                size = ny - (int) ypos0;
            if ((int) xpos0 < 0)
                xpos0 = 0;
            if ((int) ypos0 < 0)
                ypos0 = 0;
            ddx = xpos - (int) (xpos0);
            ddy = ypos - (int) (ypos0);
            smallim = cpl_image_new(size, size, CPL_TYPE_DOUBLE);
            rerr = cpl_image_fill_gaussian(smallim, ddx, ddy, 1000.0, psize,
                    psize);
            cpl_test_eq_error(rerr, CPL_ERROR_NONE);
            extractim = cpl_image_extract(image, (int) (xpos0) + 1,
                    (int) (ypos0) + 1, (int) (xpos0) + size,
                    (int) (ypos0) + size);
            rerr = cpl_image_add(extractim, smallim);
            cpl_test_eq_error(rerr, CPL_ERROR_NONE);
            rerr = cpl_image_copy(image, extractim, (int) (xpos0) + 1,
                    (int) (ypos0) + 1);
            cpl_test_eq_error(rerr, CPL_ERROR_NONE);
            cpl_image_delete(smallim);
            smallim = NULL;
            cpl_image_delete(extractim);
            extractim = NULL;
        }
    }
    return image;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Point pattern modules tests", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing default constructor",
                    cutest_sph_point_pattern_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing centralise tiny shift bug",
                    cutest_sph_point_pattern_centralise_tiny_shifts_bug)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing default constructor from table",
                    cutest_sph_point_pattern_new_from_table)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing constructor noisy image",
                    cutest_sph_point_pattern_new_from_image_noisy)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing constructor with real image",
                    cutest_sph_point_pattern_new_from_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing find_closest",
                    cutest_sph_point_pattern_find_closest)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing centralise",
                    cutest_sph_point_pattern_centralise)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing finding closest neighbour",
                    cutest_sph_point_pattern_find_closest_neighbour)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing crate_image",
                    cutest_sph_point_pattern_create_image)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing crate_image accurate",
                    cutest_sph_point_pattern_create_image_accurate)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing crate_image is regular",
                    cutest_sph_point_pattern_create_image_regular)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing load ascii",
                    cutest_sph_point_pattern_load_ascii)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing rotation",
                    cutest_sph_point_pattern_rotate)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing scaling",
                    cutest_sph_point_pattern_scale)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Make pairs",
                    cutest_sph_point_pattern_make_pairs)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Find square",
                    cutest_sph_point_pattern_find_square)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Find square in random points",
                    cutest_sph_point_pattern_find_square_in_random_points)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "Score square gives zero on perfect square",
                    cutest_sph_point_pattern_score_square_perfect_square_gives_zero)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    sph_filemanager_delete(1);
    return result;
}

/**@}*/
