/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#ifdef _OPENMP
#include <omp.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "sph_polygon.h"
#include "sph_common_keywords.h"
#include <math.h>
#include "sph_error.h"


#include "sph_test.h"
#include "sph_time.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/
static
int cutest_init_polygon_testsuite(void) {
    return sph_test_nop_code();
}

static
int cutest_clean_polygon_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test for the sph_polygon_clipping function.
 */
/*----------------------------------------------------------------------------*/
static
void cutest_polygon_calculate_overlap_performance(void) {
    sph_polygon* target = NULL;
    sph_error_code rerr = 0;
    double result = 0.0;
    sph_polygon_workspace* ws;
    int ii = 0;
    int npolys = 10000;
    sph_polygon* polys[npolys];

    for (ii = 0; ii < npolys; ++ii) {
        polys[ii] = sph_polygon_new();
        rerr = sph_polygon_add_point(polys[ii], 1.0, 1.0);
        rerr |= sph_polygon_add_point(polys[ii], 3.0, 1.0);
        rerr |= sph_polygon_add_point(polys[ii], 3.0, 3.0);
        rerr |= sph_polygon_add_point(polys[ii], 1.0, 3.0);
    }

    target = sph_polygon_new();

    cpl_test_nonnull( target );

    rerr = sph_polygon_add_point(target, 0.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 0.0);
    rerr |= sph_polygon_add_point(target, 2.0, 2.0);
    rerr |= sph_polygon_add_point(target, 0.0, 2.0);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Start timer %d: ", sph_time_start_timer(1));

    ws = sph_polygon_workspace_new();
    for (ii = 0; ii < npolys; ++ii) {
        result = sph_polygon_calculate_overlap(polys[ii], target, ws);
        if (fabs(result - 1.0) > 1.0e-10) {
            sph_test_fail("Incorrect");
        }
    }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Stop time %f: ", sph_time_stop_timer(1));
    for (ii = 0; ii < npolys; ++ii) {
        sph_polygon_delete(polys[ii]);
    }
    sph_polygon_workspace_delete(ws);
    sph_polygon_delete(target);
    target = NULL;
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {

    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("polygon_test_performance",
            cutest_init_polygon_testsuite, cutest_clean_polygon_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_polygon_calculate_overlap performance",
                    cutest_polygon_calculate_overlap_performance)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
