/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_transform.h"
#include "sph_cube.h"
#include "sph_utils.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    sph_end_test();
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_transform_test_new(void) {
    sph_transform* trans = NULL;

    trans = sph_transform_new_default();
    cpl_test_nonnull( trans );
    sph_transform_delete(trans);
    return;
}
static
void cutest_sph_transform_test_apply_transfrom_cpl_scale(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    int bp = 0;
    double dx = 0.0;
    double dy = 0.0;
    double angle = 0.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0 + 0.0);

    sph_point_rotate_around(p0, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);
    sph_point_rotate_around(p1, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
            p0->y + 0.5, 1.0, 1000.0);
    sph_test_image_tools_add_gauss(input_mframe->image, p1->x + 0.5,
            p1->y + 0.5, 2.0, 500.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + dy, 0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy + 10, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + 10 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + 10 + dy, 0);

    actual_result = sph_master_frame_duplicate(input_mframe);

    trans = sph_transform_new_cpl_warp();

    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.3, NULL);
    sph_master_frame_save(actual_result, "cutest_sph_transform_actual_result_cpl_scale.fits", NULL);
    sph_master_frame_save(input_mframe, "cutest_sph_transform_input_mframe_cpl_scale.fits", NULL);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(actual_result->badpixelmap,nx/2,ny/2,&bp), 1);
    cpl_test_abs(cpl_image_get(actual_result->ncombmap,nx/2,ny/2,&bp),
            1.0, 0.2);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,nx/2,ny/2,&bp),
            SPH_MASTER_FRAME_BAD_RMS, 5.0e9);

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}

static
void cutest_sph_transform_test_apply_transfrom_scale_conserves_total(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    double dx = 0.0;
    double dy = 0.0;
    double angle = 0.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
            p0->y + 0.5, 10.0, 1000.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);

    actual_result = sph_master_frame_duplicate(input_mframe);

    trans = sph_transform_new_cpl_warp();

    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.3, NULL);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Fluxes: %f,%f", cpl_image_get_absflux( actual_result->image), cpl_image_get_absflux(input_mframe->image));
    cpl_test_eq( cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_test_abs( cpl_image_get_absflux(actual_result->image),
            cpl_image_get_absflux(input_mframe->image),
            0.01 * cpl_image_get_absflux(input_mframe->image));

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}

static
void cutest_sph_transform_test_apply_transfrom_scale_conserves_total_fft(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    double dx = 0.0;
    double dy = 0.0;
    double angle = 0.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
            p0->y + 0.5, 10.0, 1000.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);

    actual_result = sph_master_frame_duplicate(input_mframe);

    trans = sph_transform_new_default();

    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.3, NULL);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Fluxes: %f,%f", cpl_image_get_absflux( actual_result->image), cpl_image_get_absflux(input_mframe->image));
    cpl_test_eq( cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_test_abs( cpl_image_get_absflux(actual_result->image),
            cpl_image_get_absflux(input_mframe->image),
            0.01 * cpl_image_get_absflux(input_mframe->image));

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}

static
void cutest_sph_transform_test_apply_transfrom_cpl(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    int bp = 0;
    double dx = 10.32;
    double dy = 3.6;
    double angle = 30.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0 + 0.0);

    sph_point_rotate_around(p0, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);
    sph_point_rotate_around(p1, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
            p0->y + 0.5, 1.0, 1000.0);
    sph_test_image_tools_add_gauss(input_mframe->image, p1->x + 0.5,
            p1->y + 0.5, 2.0, 500.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + dy, 0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy + 10, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + 10 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + 10 + dy, 0);

    actual_result = sph_master_frame_duplicate(input_mframe);

    trans = sph_transform_new_cpl_warp();

    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.0, NULL);
    sph_master_frame_save(actual_result, "cutest_sph_transform_actual_result_cpl.fits", NULL);
    sph_master_frame_save(input_mframe, "cutest_sph_transform_input_mframe_cpl.fits", NULL);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(actual_result->badpixelmap,nx/2,ny/2,&bp), 1);
    cpl_test_abs(cpl_image_get(actual_result->ncombmap,nx/2,ny/2,&bp),
            1.0, 0.2);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,nx/2,ny/2,&bp),
            SPH_MASTER_FRAME_BAD_RMS, 5.0e9);

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
static
void cutest_sph_transform_test_apply_transfrom_cpl_shift_only(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    double dx = 10.32;
    double dy = 3.6;
    double angle = 0.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;
    int bp = 0;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0 + 0.0);

    sph_point_rotate_around(p0, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);
    sph_point_rotate_around(p1, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x, p0->y, 1.0,
            1000.0);
    sph_test_image_tools_add_gauss(input_mframe->image, p1->x, p1->y, 2.0,
            500.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
    cpl_image_set(input_mframe->badpixelmap, 50, 50, 1);
    cpl_image_set(input_mframe->rmsmap, 50, 50, SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, 50, 50, 0);

    actual_result = sph_master_frame_duplicate(input_mframe);
    trans = sph_transform_new_cpl_warp();
    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.0, NULL);
    sph_master_frame_save(actual_result, "cutest_sph_transform_actual_result_cpl_so.fits", NULL);
    sph_master_frame_save(input_mframe, "cutest_sph_transform_input_mframe_cpl_so.fits", NULL);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(actual_result->badpixelmap,50-dx,50-dy,&bp),
            1);
    cpl_test_zero(cpl_image_get(actual_result->badpixelmap,50,50,&bp));
    cpl_test_abs(
            cpl_image_get(actual_result->ncombmap,50-dx,50-dy,&bp), 1.0, 0.001);
    cpl_test_abs(cpl_image_get(actual_result->ncombmap,50,50,&bp), 1,
            0.00001);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,50-dx,50-dy,&bp),
            SPH_MASTER_FRAME_BAD_RMS, 0.1);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,50,50,&bp), 0.0,
            0.0000001);

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
static
void cutest_sph_transform_test_apply_transfrom_cpl_rot_only(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    int nx = 256;
    int ny = 256;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;
    sph_point* p2 = NULL;
    sph_cube* result_cube = NULL;
    int ii;
    int nframes = 5;
    cpl_error_code code;
    //double dx[] = {1.0, 1.0, 1.0, 1.0, 1.0 };
    //double dy[] = {1.0, -1.0, 2.0, -2.0, 3.0 };
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };

    result_cube = sph_cube_new("cutest_sph_transform_result_cube_rot_cpl.fits");
    cpl_test_nonnull( result_cube );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0);
    p2 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0);
    sph_test_image_tools_add_gauss(expected_result->image, p0->x + 0.5,
            p0->y + 0.5, 1.0, 10000.0);
    sph_test_image_tools_add_gauss(expected_result->image, p1->x + 0.5,
            p1->y + 0.5, 2.0, 5000.0);
    trans = sph_transform_new_cpl_warp();
    for (ii = 0; ii < nframes; ++ii) {
        p2->x = p1->x;
        p2->y = p1->y;
        input_mframe = sph_master_frame_new(nx, ny);
        cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
        sph_point_rotate_around(p2, nx / 2.0, ny / 2.0,
                cos(angle[ii] * CPL_MATH_RAD_DEG),
                sin(angle[ii] * CPL_MATH_RAD_DEG), dx[ii], dy[ii]);
        sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
                p0->y + 0.5, 1.0, 10000.0);
        sph_test_image_tools_add_gauss(input_mframe->image, p2->x + 0.5,
                p2->y + 0.5, 2.0, 5000.0);

        sph_transform_apply(trans, input_mframe, NULL, nx / 2.0 + dx[ii],
                ny / 2.0 + dy[ii], angle[ii], 1.0, NULL);

        sph_master_frame_subtract_master_frame(input_mframe, expected_result);
        code = sph_master_frame_multiply_double(input_mframe, 2.0);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        code = sph_master_frame_divide_double(input_mframe,
                cpl_image_get_max(expected_result->image));
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        code = sph_master_frame_multiply_double(input_mframe, 0.5);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        sph_cube_append_master(result_cube, input_mframe, ii);
        cpl_test_abs(cpl_image_get_max(input_mframe->image), 0.0,
                0.2);
        cpl_test_abs(sph_master_frame_get_mean(input_mframe,NULL),
                0.0, 0.001);
        sph_master_frame_delete(input_mframe);
        input_mframe = NULL;
    }
    sph_transform_delete(trans);
    trans = NULL;

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    sph_cube_finalise_file(result_cube->filename);
    sph_cube_delete(result_cube);
    result_cube = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_point_delete(p2);
    p2 = NULL;
}

static
void cutest_sph_transform_test_apply_transfrom(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    int bp = 0;
    double dx = 10.32;
    double dy = 3.6;
    double angle = 30.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0 + 0.0);

    sph_point_rotate_around(p0, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);
    sph_point_rotate_around(p1, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
            p0->y + 0.5, 1.0, 1000.0);
    sph_test_image_tools_add_gauss(input_mframe->image, p1->x + 0.5,
            p1->y + 0.5, 2.0, 500.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + dy, 0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy + 10, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + 10 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + 10 + dy, 0);

    actual_result = sph_master_frame_duplicate(input_mframe);

    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);

    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.0, NULL);
    sph_master_frame_save(actual_result, "cutest_sph_transform_actual_result.fits", NULL);
    sph_master_frame_save(input_mframe, "cutest_sph_transform_input_mframe.fits", NULL);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(actual_result->badpixelmap,nx/2,ny/2,&bp), 1);
    cpl_test_abs(cpl_image_get(actual_result->ncombmap,nx/2,ny/2,&bp),
            1.0, 0.2);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,nx/2,ny/2,&bp),
            SPH_MASTER_FRAME_BAD_RMS, 5.0e9);

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_transform_test_apply_transfrom_rot_only(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    int nx = 256;
    int ny = 256;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;
    sph_point* p2 = NULL;
    sph_cube* result_cube = NULL;
    int ii;
    int nframes = 5;
    //double dx[] = {1.0, 1.0, 1.0, 1.0, 1.0 };
    //double dy[] = {1.0, -1.0, 2.0, -2.0, 3.0 };
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };

    result_cube = sph_cube_new("cutest_sph_transform_result_cube.fits");
    cpl_test_nonnull( result_cube );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0);
    p2 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0);
    sph_test_image_tools_add_gauss(expected_result->image, p0->x + 0.5,
            p0->y + 0.5, 1.0, 10000.0);
    sph_test_image_tools_add_gauss(expected_result->image, p1->x + 0.5,
            p1->y + 0.5, 2.0, 5000.0);
    for (ii = 0; ii < nframes; ++ii) {
        p2->x = p1->x;
        p2->y = p1->y;
        input_mframe = sph_master_frame_new(nx, ny);
        cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
        sph_point_rotate_around(p2, nx / 2.0, ny / 2.0,
                cos(angle[ii] * CPL_MATH_RAD_DEG),
                sin(angle[ii] * CPL_MATH_RAD_DEG), dx[ii], dy[ii]);
        sph_test_image_tools_add_gauss(input_mframe->image, p0->x + 0.5,
                p0->y + 0.5, 1.0, 10000.0);
        sph_test_image_tools_add_gauss(input_mframe->image, p2->x + 0.5,
                p2->y + 0.5, 2.0, 5000.0);
        trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);

        sph_transform_apply(trans, input_mframe, NULL, nx / 2.0 + dx[ii],
                ny / 2.0 + dy[ii], angle[ii], 1.0, NULL);
        sph_master_frame_subtract_master_frame(input_mframe, expected_result);
        sph_master_frame_divide_double(input_mframe,
                cpl_image_get_max(expected_result->image));
        sph_cube_append_master(result_cube, input_mframe, ii);
        cpl_test_abs(cpl_image_get_max(input_mframe->image), 0.0,
                0.01);
        cpl_test_abs(sph_master_frame_get_mean(input_mframe,NULL),
                0.0, 0.001);
        sph_master_frame_delete(input_mframe);
        input_mframe = NULL;
        sph_transform_delete(trans);
        trans = NULL;
    }

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);

    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    sph_cube_finalise_file(result_cube->filename);

    sph_cube_delete(result_cube);
    result_cube = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_point_delete(p2);
    p2 = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_transform_test_apply_transfrom_images(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    int bp = 0;
    double dx = 10.0;
    double dy = 3.0;
    double angle = 30.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;
    cpl_error_code code;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0 + 0.0);

    sph_point_rotate_around(p0, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);
    sph_point_rotate_around(p1, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x, p0->y, 1.0,
            1000.0);
    sph_test_image_tools_add_gauss(input_mframe->image, p1->x, p1->y, 2.0,
            500.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + dy, 0);
    cpl_image_set(input_mframe->badpixelmap, nx / 2 + dx, ny / 2 + dy + 10, 1);
    cpl_image_set(input_mframe->rmsmap, nx / 2 + dx, ny / 2 + 10 + dy,
            SPH_MASTER_FRAME_BAD_RMS);
    code = cpl_image_set(input_mframe->ncombmap, nx / 2 + dx, ny / 2 + 10 + dy, 0);
    cpl_test_eq_error(CPL_ERROR_NONE, code);

    actual_result = sph_master_frame_duplicate(input_mframe);
    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE, 0.1, 0.05);
    code = sph_transform_apply_to_images(trans, &actual_result->image,
            &actual_result->badpixelmap, &actual_result->ncombmap,
            &actual_result->rmsmap, NULL, nx / 2.0 + dx, ny / 2.0 + dy, angle,
            1.0, NULL);
    cpl_test_eq_error(CPL_ERROR_NONE, code);
    code = sph_master_frame_save(actual_result, "cutest_sph_transform_actual_result_ims.fits", NULL);
    cpl_test_eq_error(CPL_ERROR_NONE, code);
    code = sph_master_frame_save(input_mframe, "cutest_sph_transform_input_mframe_ims.fits", NULL);
    cpl_test_eq_error(CPL_ERROR_NONE, code);

    cpl_test_eq(cpl_image_get(actual_result->badpixelmap,nx/2,ny/2,&bp), 1);
    cpl_test_abs(cpl_image_get(actual_result->ncombmap,nx/2,ny/2,&bp),
            0, 0.4);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,nx/2,ny/2,&bp),
            SPH_MASTER_FRAME_BAD_RMS, 5.0e9);

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        A test function of the testsuite
 *
 * This is a standard test case -- with ideal data.
 *
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_transform_test_apply_transfrom_shift_only(void) {
    sph_transform* trans = NULL;
    sph_master_frame* input_mframe = NULL;
    sph_master_frame* expected_result = NULL;
    sph_master_frame* actual_result = NULL;
    int nx = 256;
    int ny = 256;
    double dx = 10.32;
    double dy = 3.6;
    double angle = 0.0;
    sph_point* p0 = NULL;
    sph_point* p1 = NULL;
    int bp = 0;

    input_mframe = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( input_mframe );
    expected_result = sph_master_frame_new(nx, ny);
    cpl_test_nonnull( expected_result );
    p0 = sph_point_new(nx / 2.0, ny / 2.0);
    p1 = sph_point_new(nx / 2.0 + 20.0, ny / 2.0 + 0.0);

    sph_point_rotate_around(p0, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);
    sph_point_rotate_around(p1, nx / 2.0, ny / 2.0,
            cos(angle * CPL_MATH_RAD_DEG), sin(angle * CPL_MATH_RAD_DEG), dx,
            dy);

    sph_test_image_tools_add_gauss(input_mframe->image, p0->x, p0->y, 1.0,
            1000.0);
    sph_test_image_tools_add_gauss(input_mframe->image, p1->x, p1->y, 2.0,
            500.0);
    cpl_image_add_scalar(input_mframe->ncombmap, 1.0);
    cpl_image_set(input_mframe->badpixelmap, 50, 50, 1);
    cpl_image_set(input_mframe->rmsmap, 50, 50, SPH_MASTER_FRAME_BAD_RMS);
    cpl_image_set(input_mframe->ncombmap, 50, 50, 0);

    actual_result = sph_master_frame_duplicate(input_mframe);
    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE, 0.5, 0.0);
    sph_transform_apply(trans, actual_result, NULL, nx / 2.0 + dx,
            ny / 2.0 + dy, angle, 1.0, NULL);
    sph_master_frame_save(actual_result, "cutest_sph_transform_actual_result_so.fits", NULL);
    sph_master_frame_save(input_mframe, "cutest_sph_transform_input_mframe_so.fits", NULL);
    cpl_test_eq(cpl_error_get_code(), CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(actual_result->badpixelmap,50-dx,50-dy,&bp),
            1);
    cpl_test_zero(cpl_image_get(actual_result->badpixelmap,50,50,&bp));
    cpl_test_abs(
            cpl_image_get(actual_result->ncombmap,50-dx,50-dy,&bp), 1.0, 0.001);
    cpl_test_abs(cpl_image_get(actual_result->ncombmap,50,50,&bp), 1,
            0.00001);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,50-dx,50-dy,&bp),
            SPH_MASTER_FRAME_BAD_RMS, 0.1);
    cpl_test_abs(cpl_image_get(actual_result->rmsmap,50,50,&bp), 0.0,
            0.0000001);

    sph_master_frame_delete(actual_result);
    actual_result = NULL;
    sph_master_frame_delete(input_mframe);
    input_mframe = NULL;
    sph_master_frame_delete(expected_result);
    expected_result = NULL;
    sph_point_delete(p0);
    p0 = NULL;
    sph_point_delete(p1);
    p1 = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_transform_remap_badpix_no_transform_w_irdmodel(void) {
    sph_transform* trans = NULL;
    cpl_image* badpix = NULL;
    cpl_image* nbadpix = NULL;
    sph_ird_instrument_model* irdmodel = NULL;

    irdmodel = sph_ird_instrument_model_new();
    irdmodel->detsize_pixels_x = 512 * 2;
    irdmodel->detsize_pixels_y = 512;
    irdmodel->split_pixel_x = 512;
    irdmodel->window_size_y = 512;
    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);

    badpix = sph_test_image_tools_create_flat_image_int(512, 512, 0);
    sph_test_image_tools_add_badpixles(badpix, 100, NULL);
    cpl_image_fill_rejected(badpix, 1.0);
    cpl_image_accept_all(badpix);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Input n: %f", cpl_image_get_flux(badpix));
    cpl_test_abs( cpl_image_get_flux(badpix), 100, 0.1);
    nbadpix = sph_transform_remap_badpix(trans, badpix, 0.0, 256, 256, 0.0, 0.0,
            NULL, 1.0, irdmodel);
    cpl_test_nonnull( nbadpix );
    cpl_image_save(badpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(nbadpix, "cutest_sph_transform_nbadpix.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_test_eq( cpl_image_get_flux(badpix), 100);
    cpl_test_eq( cpl_image_get_flux(nbadpix), 100);
    cpl_image_delete(badpix);
    badpix = NULL;
    cpl_image_delete(nbadpix);
    nbadpix = NULL;
    sph_ird_instrument_model_delete(irdmodel);
    irdmodel = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_transform_remap_badpix_no_transform(void) {
    sph_transform* trans = NULL;
    cpl_image* badpix = NULL;
    cpl_image* nbadpix = NULL;

    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);
    badpix = sph_test_image_tools_create_flat_image_int(512, 512, 0);
    sph_test_image_tools_add_badpixles(badpix, 1000, NULL);
    cpl_image_fill_rejected(badpix, 1.0);
    cpl_image_accept_all(badpix);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Input n: %f", cpl_image_get_flux(badpix));
    cpl_test_abs( cpl_image_get_flux(badpix), 1000, 0.1);
    nbadpix = sph_transform_remap_badpix(trans, badpix, 0.0, 256, 256, 0.0, 0.0,
            NULL, 1.0, NULL);
    cpl_test_nonnull( nbadpix );
    cpl_image_save(badpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(nbadpix, "cutest_sph_transform_nbadpix.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_test_eq( cpl_image_get_flux(badpix), 1000);
    cpl_test_eq( cpl_image_get_flux(nbadpix), 1000);
    cpl_image_delete(badpix);
    badpix = NULL;
    cpl_image_delete(nbadpix);
    nbadpix = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}

static
void cutest_sph_transform_remap_badpix_ok_with_double_badpix(void) {
    sph_transform* trans = NULL;
    cpl_image* badpix = NULL;
    cpl_image* nbadpix = NULL;

    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);
    badpix = sph_test_image_tools_create_flat_image_double(512, 512, 0);
    sph_test_image_tools_add_badpixles(badpix, 1000, NULL);
    cpl_image_fill_rejected(badpix, 1.0);
    cpl_image_accept_all(badpix);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Input n: %f", cpl_image_get_flux(badpix));
    cpl_test_abs( cpl_image_get_flux(badpix), 1000, 0.1);
    nbadpix = sph_transform_remap_badpix(trans, badpix, 0.0, 256, 256, 0.0, 0.0,
            NULL, 1.0, NULL);
    cpl_test_nonnull( nbadpix );
    cpl_image_save(badpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(nbadpix, "cutest_sph_transform_nbadpix.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
    cpl_test_eq( cpl_image_get_flux(badpix), 1000);
    cpl_test_eq( cpl_image_get_flux(nbadpix), 1000);
    cpl_image_delete(badpix);
    badpix = NULL;
    cpl_image_delete(nbadpix);
    nbadpix = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_transform_test_rot_badpix_only_shift(void) {
    sph_transform* trans = NULL;
    cpl_image* badpix = NULL;
    cpl_image* nbadpix = NULL;
    int bpix = 0;
    gsl_rng* pRNG = NULL;
    int nx = 2048;
    int ny = 2048;

    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);

    badpix = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    cpl_image_set(badpix, nx / 2, ny / 2, 1);
    cpl_image_set(badpix, nx / 2 + 100, ny / 2, 1);
    nbadpix = sph_transform_remap_badpix(trans, badpix, 0.0, nx / 2, ny / 2,
            10.0, 0.0, NULL, 1.0, NULL);
    cpl_test_nonnull( nbadpix );
    cpl_image_save(badpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(nbadpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_test_abs(cpl_image_get(nbadpix,nx/2+10,ny/2,&bpix), 1.0,
            0.01);
    cpl_test_abs(cpl_image_get(nbadpix,nx/2+110,ny/2,&bpix), 1.0,
            0.01);
    cpl_test_abs(cpl_image_get(nbadpix,nx/2,ny/2,&bpix), 0.0, 0.01);
    cpl_test_abs(cpl_image_get(nbadpix,nx/2+100,ny/2,&bpix), 0.0,
            0.01);
    cpl_image_delete(badpix);
    badpix = NULL;
    cpl_image_delete(nbadpix);
    nbadpix = NULL;

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    badpix = sph_test_image_tools_create_flat_image_int(nx, ny, 0);
    sph_test_image_tools_add_hotpixels(badpix, 100000, 1.0, pRNG);
    nbadpix = sph_transform_remap_badpix(trans, badpix, 40.0, nx / 2, ny / 2,
            10.0, 0.0, NULL, 1.0, NULL);
    cpl_test_nonnull( nbadpix );
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "NBadpix was: %f", cpl_image_get_absflux(nbadpix));
    gsl_rng_free(pRNG);
    pRNG = NULL;
    cpl_image_delete(badpix);
    badpix = NULL;
    cpl_image_delete(nbadpix);
    nbadpix = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

static
void cutest_sph_transform_test_rot_badpix_zero_ang(void) {
    sph_transform* trans = NULL;
    cpl_image* badpix = NULL;
    cpl_image* nbadpix = NULL;
    int bpix = 0;

    trans = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH, 0.5, 0.0);
    badpix = sph_test_image_tools_create_flat_image_int(512, 512, 0);
    cpl_image_set(badpix, 256, 256, 1);
    cpl_image_set(badpix, 300, 256, 1);
    nbadpix = sph_transform_remap_badpix(trans, badpix, 0.0, 256, 256, 0.0, 0.0,
            NULL, 1.0, NULL);
    cpl_test_nonnull( nbadpix );
    cpl_image_save(badpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(nbadpix, "cutest_sph_transform_badpix.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_EXTEND);
    cpl_test_abs(cpl_image_get(nbadpix,256,256,&bpix), 1.0, 0.01);
    cpl_test_abs(cpl_image_get(nbadpix,300,256,&bpix), 1.0, 0.01);
    cpl_image_delete(badpix);
    badpix = NULL;
    cpl_image_delete(nbadpix);
    nbadpix = NULL;
    sph_transform_delete(trans);
    trans = NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing transform suite", cutest_init_testsuite,
            cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing new", cutest_sph_transform_test_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply_transform",
                    cutest_sph_transform_test_apply_transfrom)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply_transform_images",
                    cutest_sph_transform_test_apply_transfrom_images)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply_transform_images rot only",
                    cutest_sph_transform_test_apply_transfrom_rot_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing apply_transform_images shift only",
                    cutest_sph_transform_test_apply_transfrom_shift_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing remap no transform",
                    cutest_sph_transform_remap_badpix_no_transform)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing remap no transform w irdmodel",
                    cutest_sph_transform_remap_badpix_no_transform_w_irdmodel)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing remap shift",
                    cutest_sph_transform_test_rot_badpix_only_shift)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing remap rot no angle",
                    cutest_sph_transform_test_rot_badpix_zero_ang)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing transform cpl",
                    cutest_sph_transform_test_apply_transfrom_cpl)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing transform cpl scale",
                    cutest_sph_transform_test_apply_transfrom_cpl_scale)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing transform cpl shift only",
                    cutest_sph_transform_test_apply_transfrom_cpl_shift_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing transform cpl rot only",
                    cutest_sph_transform_test_apply_transfrom_cpl_rot_only)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "Testing transform scaling conserves total",
                    cutest_sph_transform_test_apply_transfrom_scale_conserves_total)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(
                    pSuite,
                    "Testing transform scaling conserves total fft",
                    cutest_sph_transform_test_apply_transfrom_scale_conserves_total_fft)) {
        return sph_test_get_error();
    }
    if (NULL
    		== sph_test_do(
    				pSuite,
					"Testing remap badpix on double format badpix",
					cutest_sph_transform_remap_badpix_ok_with_double_badpix)) {
    	return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
