/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>

#include <string.h>
#include <stdio.h>
#include <math.h>

#ifdef HAVE_UNISTD_H
/* Needed for access() */
#include <unistd.h>
#endif

/*-----------------------------------------------------------------------------
 Define
 -----------------------------------------------------------------------------*/

#ifndef SPH_STD_ZPL_POL_HIGH_CATG
#define SPH_STD_ZPL_POL_HIGH_CATG "spher-calib/zpl/sph_zpl_polhigh_star_table.fits"
#endif

#ifndef SPH_STD_ZPL_POL_ZERO_CATG
#define SPH_STD_ZPL_POL_ZERO_CATG "spher-calib/zpl/sph_zpl_polzero_star_table.fits"
#endif


#ifndef SPH_STD_ZPL_POL_CORRECTIONS_CATG
#define SPH_STD_ZPL_POL_CORRECTIONS_CATG "spher-calib/zpl/sph_zpl_pol_correct_table.fits"
#endif

#define PRO_RECi_PIPE_ID          "ESO PRO REC1 PIPE ID"
#define PRO_RECi_PIPE_ID_COMMENT "Pipeline (unique) identifier"

#define PRO_RECi_ID              "ESO PRO REC1 ID"
#define PRO_RECi_ID_COMMENT      "Pipeline recipe (unique)"


/*-----------------------------------------------------------------------------
 Function declarations
 -----------------------------------------------------------------------------*/

typedef struct {
	const char * hd;
	cpl_size Qity;
	double RA;
	double DEC;
	double V;
	double p_V;
	double theta_V;
	double p_R;
	double theta_R;
	double p_I;
	double theta_I;
	const char * SpType;
	double PM_RA;
	double PM_DEC;
}catg_star;

typedef struct{
	const catg_star * els;
	cpl_size num_els;
}stars_pol_catg;

typedef enum{
	sph_zpl_test_pol_high,
	sph_zpl_test_pol_zero
}sph_zpl_test_pol;

static cpl_table * convert_catg_data_to_table(const catg_star * stars,
		const cpl_size num_stars);

static cpl_boolean sph_zpl_polar_catg_create(const char * path,
									  sph_zpl_test_pol pol_catg,
									  const char * pipeid);

static cpl_boolean sph_zpl_polar_correction_catg_create(const char * path,
												const char * pipeid);

static
void plist_append_double_value_and_comment(cpl_propertylist * plist, const char * key,
		const double value, const char * comment);

static
void plist_append_string_value_and_comment(cpl_propertylist * plist,
		const char * key, const char * value, const char * comment);
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(int argc, char* argv[]) {

    char * base;
    char *end;
    int ndir = 5; /* Depth of source directory */

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    cpl_test_lt(0, argc);

    base = cpl_strdup(argv[0]);

    while ((end = strrchr(base, '/')) > base && --ndir >= 0) {
        *end = 0;
    }
    if (end != NULL) {
        const char *pipeid = strrchr(argv[0], '/');

        end[1] = 0; /* Truncate after '/' */

        const cpl_boolean doit = argc == 2;

        if(doit){

        	const cpl_boolean success_catg = sph_zpl_polar_catg_create(base, sph_zpl_test_pol_high,
                            pipeid ? pipeid + 1 : NULL);
        	cpl_test_eq(success_catg, CPL_TRUE);

        	const cpl_boolean success_corr_catg = sph_zpl_polar_correction_catg_create(base,
                            pipeid ? pipeid + 1 : NULL);
        	cpl_test_eq(success_corr_catg, CPL_TRUE);
        }
        else
        	cpl_msg_info(cpl_func, "Creation of the table disabled, test skipped");
    }

    cpl_free(base);

    return cpl_test_end(0);
}

/**@}*/



static cpl_table * convert_catg_data_to_table(const catg_star * stars,
		const cpl_size num_stars){

	const char * HD_CLMN = "HD";
	const char * QITY_CLMN = "Qity";
	const char * RA_CLMN = "RA";
	const char * DEC_CLMN = "DEC";
	const char * V_CLMN= "V";
	const char * P_V_CLMN = "p_V";
	const char * THETA_V_CLMN = "theta_V";
	const char * P_R_CLMN = "p_R";
	const char * THETA_R_CLMN = "theta_R";
	const char * P_I_CLMN = "p_I";
	const char * THETA_I_CLMN = "theta_I";
	const char * SP_TYPE_CLMN = "SpType";
	const char * PM_RA_CLMN = "PM_RA";
	const char * PM_DEC_CLMN = "PM_Dec";

	cpl_table * tb = cpl_table_new(num_stars);

	cpl_table_new_column(tb, HD_CLMN, CPL_TYPE_STRING);
	cpl_table_new_column(tb, QITY_CLMN, CPL_TYPE_LONG_LONG);
	cpl_table_new_column(tb, RA_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, DEC_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, V_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, P_V_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, THETA_V_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, P_R_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, THETA_R_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, P_I_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, THETA_I_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, SP_TYPE_CLMN, CPL_TYPE_STRING);
	cpl_table_new_column(tb, PM_RA_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, PM_DEC_CLMN, CPL_TYPE_DOUBLE);

	for(cpl_size i = 0; i < num_stars; ++i){
		cpl_table_set_string(tb, HD_CLMN,  i, stars[i].hd);
		cpl_table_set_long_long(tb, QITY_CLMN,  i, stars[i].Qity);

		cpl_table_set_double(tb, RA_CLMN,  i, stars[i].RA);
		cpl_table_set_double(tb, DEC_CLMN,  i, stars[i].DEC);

		cpl_table_set_double(tb, V_CLMN,  i, stars[i].V);

		if(!isnan(stars[i].p_V))
			cpl_table_set_double(tb, P_V_CLMN,  i, stars[i].p_V);
		if(!isnan(stars[i].theta_V))
			cpl_table_set_double(tb, THETA_V_CLMN,  i, stars[i].theta_V);

		if(!isnan(stars[i].p_R))
			cpl_table_set_double(tb, P_R_CLMN,  i, stars[i].p_R);
		if(!isnan(stars[i].theta_R))
			cpl_table_set_double(tb, THETA_R_CLMN,  i, stars[i].theta_R);

		if(!isnan(stars[i].p_I))
			cpl_table_set_double(tb, P_I_CLMN,  i, stars[i].p_I);
		if(!isnan(stars[i].theta_I))
			cpl_table_set_double(tb, THETA_I_CLMN,  i, stars[i].theta_I);

		cpl_table_set_string(tb, SP_TYPE_CLMN,  i, stars[i].SpType);

		cpl_table_set_double(tb, PM_RA_CLMN,  i, stars[i].PM_RA);
		cpl_table_set_double(tb, PM_DEC_CLMN,  i, stars[i].PM_DEC);
	}


	return tb;
}

#define get_vect_size(v) sizeof(v) / sizeof(v[0])

static stars_pol_catg get_catgs_high(void){

	static const catg_star high_stars [] = {
		{ "23512", 2,  56.641667, +23.62389, 8.11, 2.26,  29.9, 2.29,  29.6, 2.13,  29.3,     "A0V",  21.34, -42.79},
		{ "42379", 3,  92.825000, +21.56361, 7.43, 2.90, 169.5, 2.83,   NAN,  NAN,   NAN,    "B1II",   0.66,  -0.15},
		{ "62150", 3, 115.304167, -32.64389, 7.72, 2.60,   NAN,  NAN,   NAN,  NAN,   NAN,    "B3Ia",  -2.23,   3.25},
		{ "79186", 3, 137.766667, -44.86778, 5.02, 2.60,  47.7, 2.40,   NAN,  NAN,   NAN,    "B5Ia",  -4.92,   4.75},
		{ "94414", 2, 162.729167, -77.12444, 8.01, 5.20, 121.0, 5.38, 122.0, 5.11, 123.0,     "B4V", -20.54,   0.27},
		{ "94909", 3, 164.100000, -57.55111, 7.35, 5.01,   NAN, 5.24,   NAN,  NAN,   NAN,    "B1Ia",  -4.99,   5.71},
		{ "98143", 2, 168.816667, -77.51833, 7.61, 8.03, 133.0, 8.04, 130.0, 7.10, 130.0,   "B8III", -14.11,  -5.78},
		{"111613", 3, 192.820833, -60.32972, 5.75, 3.10,  81.1, 3.10,   NAN,  NAN,   NAN,   "A2Iab",  -4.50,  -0.97},
		{"118522", 3, 205.000000, -70.78778, 6.60, 3.10,   NAN,  NAN,   NAN,  NAN,   NAN,   "K0III", -25.09,  -8.96},
		{"147084", 1, 245.158333, -24.16917, 4.54, 4.15,  31.7, 4.44,  32.2, 4.40,  31.5,    "A4II",  -4.32, -14.15},
		{"149757", 2, 249.287500, -10.56694, 2.58, 1.44, 126.0, 1.35, 124.0, 1.18, 124.0,     "O9V",  15.26,  24.79},
		{"154445", 1, 256.383333, - 0.89194, 5.64, 3.72,  89.9, 3.63,  89.5, 3.29,  90.5,     "B1V",   4.09,  -1.15},
		{"160529", 3, 265.495833, -33.50361, 6.77, 7.31,  20.4, 7.04,  20.8, 6.53,  21.3,   "A3Iab",  -1.75,  -1.49},
		{"161056", 2, 265.945833, - 7.07944, 6.30, 4.03,  66.9, 4.01,  67.3, 3.57,  67.8,   "B1.5V",  -4.57, -10.72},
		{"183143", 2, 291.858333, +18.29583, 6.92, 6.15, 179.2, 5.81, 178.9, 5.36, 179.0,   "B7Iae",  -1.02,  -5.65},
		{"185859", 2, 295.116667, +20.47694, 6.52, 2.32,   NAN, 2.02,   NAN,  NAN,   NAN, "B0.5Iae",  -0.12,  -5.29},
		{"188220", 3, 298.666667, -15.91361, 8.02, 2.27,   NAN, 2.40,   NAN, 2.24,   NAN,     "A0V",   4.92, -18.34}
	};


	stars_pol_catg to_ret = {high_stars, get_vect_size(high_stars)};
	return to_ret;
}

/*
{ 87901,  152.091667, +11.96694, " B8IV",  1.40,  "alp Leo",     3982,   248.73,     5.59},
{102647,  177.262500, +14.57194, "  A3V",  2.13,  "bet Leo",     4534,  -497.68,  -114.67},
{121370,  208.670833, +18.39750, " G0IV",  2.68,  "eta Boo",     5235,   -60.95,  -356.29},
{124897,  213.912500, +19.18222, "K1III", -0.04,  "alp Boo",     5340, -1093.39, -2000.06},
{140573,  236.066667, +06.42556, "K2III",  2.63,  "apl Ser",     5854,   133.84,    44.81},
{159561,  263.733333, +12.56000, "A5III",  2.10,  "alp Oph",     6556,   108.07,  -221.57},
{161096,  265.866667, +04.56722, "K2III",  2.75,  "bet Oph",     6603,   -41.45,   159.34},
{187642,  297.691667, +08.86806, "  A7V",  0.76,  "alp Aql",     7557,   536.23,   385.29},
{  1581,    5.016667, -64.87472, "  G0V",  4.20,  "Zet Tuc",       77,  1707.42,  1164.30},
{  4391,   11.437500, -47.55194, "  G5V",  5.80,  "       ",      209,   183.99,    78.81},
{  4628,   12.091667, +05.28056, "  K2V",  5.74,  "       ",      222,   757.11, -1141.33},
{  9540,   23.312500, -24.17778, "  K0V",  6.96,  "       ",         ,   272.56,  -158.66},
{ 10476,   25.620833, +20.26833, "  K1V",  5.20,  "107 Psc",      493,  -302.42,  -678.88},
{ 14412,   34.741667, -25.94556, "  G8V",  6.34,  "       ",      683,  -217.55,   444.44},
{ 20630,   49.837500, +03.37000, "  G5V",  4.85, k"ap1 Cet",      996,   269.30,    93.75},
{ 20794,   49.979167, -43.06972, "  G8V",  4.26,  " 82 Eri",     1008,  3038.34,   726.58},
{ 30495,   71.900000, -16.93444, "  G3V",  5.50,  " 58 Eri",     1532,   130.04,   169.27},
{ 32147,   75.200000, -05.75361, "  K3V",  6.22,  "       ",     1614,   550.12, -1109.23},
{ 38393,   86.112500, -22.44833, "  F6V",  3.60,  "gam Lep",     1983,  -291.67,  -368.97},
{ 43834,   92.558333, -74.75278, "  G5V",  5.09,  "alp Men",     2261,   121.80,  -212.34},
{ 68456,   122.25000, -61.30222, "  F6V",  4.76,  "       ",     3220,  -150.05,  -296.87},
{ 72673,   128.21250, -31.50083, "  K0V",  6.38,  "       ",     3384, -1113.81,   762.43},
{ 98281,  169.591667, -05.06722, "  G8V",  7.23,  "       ",         ,   795.08,  -150.84},
{100623,  173.620833, -32.83111, "  K0V",  5.98,  "       ",     4458,  -670.79,   822.91},
{102438,  176.812500, -30.28639, "  G6V",  6.48,  "       ",         ,  -266.08,  -227.88},
{102870,  177.670833, +01.76444, "  F8V",  3.61,  "bet Vir",     4540,   740.23,  -270.43},
{115617,  199.600000, -18.30667, "  K7V",  4.74,  " 61 Vir",     5019, -1070.36, -1063.69},
{125184,  214.500000, -07.54222, " G5IV",  6.50,  "       ",         ,   268.45,  -235.92},
{136352,  230.450000, -48.31750, "  G2V",  5.65,  "v2 Lupi",     5699, -1623.12,  -275.70},
{141004,  236.608333, +07.35306, "  G0V",  4.42,  "lam Ser",     5868,  -224.00,   -70.64},
{146233,  243.904167, -08.36917, "  G1V",  5.50,  " 18 Sco",     6060,   230.77,  -495.53},
{149661,  249.087500, -02.32444, "  K0V",  5.77,  " 12 Oph",     6171,   456.04,  -309.63},
{160691,  266.033333, -51.83389, "  G5V",  5.10,  "mu Arae",     6585,   -16.85,  -190.60},
{172051,  279.720833, -21.05167, "  G6V",  5.85,  "       ",     6998,   -74.85,  -152.00},
{188512,  298.825000, +06.40667, " G9IV",  3.71,  "bet Aql",     7602,    45.27,  -481.91},
{192310,  303.820833, -27.03278, "  K1V",  5.72,  "       ",     7722,  1241.85,  -180.96},
{188088,  298.570833, -23.94083, "  K3V",  6.20,  "       ",     7578,  -123.15,  -409.70},
{196761,  310.045833, -23.77361, "  G8V",  6.37,  "       ",     7898,   501.45,   461.36},
{209100,  330.837500, -56.78583, "  K5V",  4.70,  "eps Ind",     8387,  3960.93, -2539.23},
{216803,  344.100000, -31.56556, "  K4V",  6.48,  "       ",     8721,   331.11,  -158.98}
}
static stars_pol_catg get_catgs_zero(){
	stars_pol_catg to_ret = {&zero_stars, get_vect_size(zero_stars)};
	return to_ret;
}
*/

static stars_pol_catg get_catgs(sph_zpl_test_pol pol_type){

	stars_pol_catg def_ret = {NULL, 0};
	cpl_ensure(pol_type == sph_zpl_test_pol_high, CPL_ERROR_ILLEGAL_INPUT, def_ret);

	return get_catgs_high();
}

static cpl_propertylist * get_pol_plist(const char * catg_tag){

    cpl_propertylist * plist = cpl_propertylist_new();

    plist_append_string_value_and_comment(plist, "INSTRUME", "SPHERE", "Instrument used");

    plist_append_double_value_and_comment(plist, "MJD-OBS",
    		57113.0, "This calib cannot be used before this date");

    plist_append_string_value_and_comment(plist, "ESO PRO CATG",
    		catg_tag, NULL);

	plist_append_string_value_and_comment(plist, "ESO SEQ ARM",
			"ZIMPOL", "Name of the sub-system");

	return plist;
}

typedef struct{
	const char * filter_name;
	double delta_theta;
	double delta_p;
}pol_corrections;

static cpl_boolean sph_zpl_polar_correction_catg_create(const char * path,
									  const char * pipeid){

	const char * FILTER_CLMN = "Filter";
	const char * D_THETA_CLMN = "d_Theta";
	const char * D_P_CLMN = "d_P";

    char * file = cpl_sprintf("%s%s", path, SPH_STD_ZPL_POL_CORRECTIONS_CATG);

    pol_corrections corrs[] = {
    		{"V",    6.7, 0.57},
			{"N_R", 12.6, 0.55},
			{"N_I", 68.1, 0.42}};

    cpl_size sz = get_vect_size(corrs);

    cpl_table * tb = cpl_table_new(sz);

	cpl_table_new_column(tb, FILTER_CLMN, CPL_TYPE_STRING);
	cpl_table_new_column(tb, D_THETA_CLMN, CPL_TYPE_DOUBLE);
	cpl_table_new_column(tb, D_P_CLMN, CPL_TYPE_DOUBLE);

	for(cpl_size i = 0; i < sz; ++i){
		cpl_table_set_string(tb, FILTER_CLMN,  i, corrs[i].filter_name);
		cpl_table_set_double(tb, D_THETA_CLMN,  i, corrs[i].delta_theta);
		cpl_table_set_double(tb, D_P_CLMN,  i, corrs[i].delta_p);
	}

	cpl_propertylist * plist = get_pol_plist("ZPL_POL_CORRECT_TABLE");

	cpl_error_code code = cpl_table_save(tb, plist, NULL, file, CPL_IO_CREATE);
    cpl_test_fits(file);
    if(code){
    	cpl_msg_error(cpl_func, "Failed to write %s aborting. (pipeid=%s)", file,
                      pipeid ? pipeid : "<NULL>");
    	cpl_test_eq(code, CPL_ERROR_NONE);
    }

	cpl_table_delete(tb);
    cpl_free(file);
    cpl_propertylist_delete(plist);
    return cpl_error_get_code() == CPL_ERROR_NONE;
}

static cpl_boolean sph_zpl_polar_catg_create(const char * path,
									  sph_zpl_test_pol pol_catg,
									  const char * pipeid)
{
	const char * base = pol_catg == sph_zpl_test_pol_high ? SPH_STD_ZPL_POL_HIGH_CATG : SPH_STD_ZPL_POL_ZERO_CATG;
	const char * catg = pol_catg == sph_zpl_test_pol_high ? "ZPL_POLHIGH_STAR_TABLE" : "ZPL_POLZERO_STAR_TABLE";
    char * file = cpl_sprintf("%s%s", path, base);

    stars_pol_catg catgs = get_catgs(pol_catg);

    if(catgs.els == NULL) return CPL_FALSE;

    cpl_table * tb = convert_catg_data_to_table(catgs.els, catgs.num_els);

    if(tb == NULL) return CPL_FALSE;

    cpl_propertylist * plist = get_pol_plist(catg);


	cpl_error_code code = cpl_table_save(tb, plist, NULL, file, CPL_IO_CREATE);
    cpl_test_fits(file);
    if(code){
    	cpl_msg_error(cpl_func, "Failed to write %s aborting. (pipeid=%s)", file,
                      pipeid ? pipeid : "<NULL>");
    	cpl_test_eq(code, CPL_ERROR_NONE);
    }

    cpl_table_delete(tb);
    cpl_propertylist_delete(plist);

    cpl_free(file);
    return cpl_error_get_code() == CPL_ERROR_NONE;
}

static
void plist_append_string_value_and_comment(cpl_propertylist * plist, const char * key,
		const char * value, const char * comment){

    cpl_error_code err = cpl_propertylist_update_string(plist, key, value);
    cpl_test_eq_error(err, CPL_ERROR_NONE);

    if(!comment) return;

	err = cpl_propertylist_set_comment(plist, key, comment);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
}


static
void plist_append_double_value_and_comment(cpl_propertylist * plist, const char * key,
		const double value, const char * comment){

    cpl_error_code err = cpl_propertylist_update_double(plist, key, value);
    cpl_test_eq_error(err, CPL_ERROR_NONE);

    if(!comment) return;

	err = cpl_propertylist_set_comment(plist, key, comment);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
}
