from edps import (
    task, subworkflow,
    FilterMode, ReportInput,
    qc0, qc1calib, calchecker, science,
)

from .sphere_task_functions import (
    which_readout_type, which_exptime, is_fast_pol, is_long_exp_SlowPol
)
from .zimpol_datasources import *

"""
# Naming conventions

- Classification rules for raw frames start with `cls_`
- Classification rules for products have the same name as the PRO.CATG
- DataSources start with `raw_`
- Tasks start with `task_zpl_`
"""


# --- Processing Tasks -----------------------------------------------------------------------------

# flat field and modemeff


@subworkflow("zimpol_internal_calibrations_polarimetry", "")
def process_zpl_internal_calib_pol():
    # --- Determine bias level and read - out noise(needed for FastPolarimetry readout mode and monitoring) -----------
    task_zpl_bias_pol = (task("zimpol_bias_polarimetry")
                         .with_recipe("sph_zpl_master_bias")
                         .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                         .with_report("sphere_master_bias", ReportInput.RECIPE_INPUTS_OUTPUTS)
                         .with_main_input(zpl_raw_bias_pol)
                         .with_meta_targets([qc1calib])
                         .build())

    # --- Determine dark current (needed for SlowPolarimetry readout mode with exposure times >= 30 sec and monitoring)
    task_zpl_dark_pol = (task("zimpol_dark_polarimetry")
                         .with_recipe("sph_zpl_master_dark")
                         .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                         .with_report("sphere_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
                         .with_dynamic_parameter("fast_or_slow", which_readout_type)
                         .with_main_input(zpl_raw_dark_pol)
                         .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                condition=is_fast_pol, min_ret=0)
                         .with_meta_targets([qc1calib])
                         .build())

    # --- Determination of pixel-to-pixel variation in sensitivity (quantum efficiency, filter defects, etc) ----------
    # different raw data may be needed to match the filters used in CAM2 and CAM2, therefore we have 2 tasks and
    # filter the products per camera
    task_zpl_flat_pol_qc = (task("zimpol_flat_polarimetry_qc")
                            .with_recipe("sph_zpl_intensity_flat")
                            .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                            .with_report("sphere_img_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                            .with_dynamic_parameter("fast_or_slow", which_readout_type)
                            .with_dynamic_parameter("exptime", which_exptime)
                            .with_main_input(zpl_raw_flat_pol)
                            .with_associated_input(task_zpl_bias_pol,
                                                   [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                   condition=is_fast_pol, min_ret=0)
                            .with_associated_input(task_zpl_dark_pol,
                                                   [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                   condition=is_long_exp_SlowPol, min_ret=0)
                            .with_meta_targets([qc1calib])
                            .build())

    task_zpl_flat_pol_cam1 = (task("zimpol_flat_polarimetry_cam1")
                              .with_recipe("sph_zpl_intensity_flat")
                              .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                              .with_report("sphere_img_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                              .with_dynamic_parameter("fast_or_slow", which_readout_type)
                              .with_dynamic_parameter("exptime", which_exptime)
                              .with_main_input(zpl_raw_flat_pol)
                              .with_associated_input(task_zpl_bias_pol,
                                                     [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                     condition=is_fast_pol, min_ret=0)
                              .with_associated_input(task_zpl_dark_pol,
                                                     [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                     condition=is_long_exp_SlowPol, min_ret=0)
                              .with_output_filter(ZPL_INT_FLAT_FIELD_MASTER_CAM1)
                              .build())

    task_zpl_flat_pol_cam2 = (task("zimpol_flat_polarimetry_cam2")
                              .with_recipe("sph_zpl_intensity_flat")
                              .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                              .with_report("sphere_img_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
                              .with_dynamic_parameter("fast_or_slow", which_readout_type)
                              .with_dynamic_parameter("exptime", which_exptime)
                              .with_main_input(zpl_raw_flat_pol)
                              .with_associated_input(task_zpl_bias_pol,
                                                     [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                     condition=is_fast_pol, min_ret=0)
                              .with_associated_input(task_zpl_dark_pol,
                                                     [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                     condition=is_long_exp_SlowPol, min_ret=0)
                              .with_output_filter(ZPL_INT_FLAT_FIELD_MASTER_CAM2)
                              .build())

    # modem_eff = efficiency of the modulation-demodulation cycle of rows in ZIMPOL detector
    # different raw data may be needed to match the filters used in CAM2 and CAM2, therefore we have 2 tasks and
    # filter the products per camera
    task_zpl_modem_eff_qc = (task("zimpol_modem_efficiency_qc")
                             .with_recipe("sph_zpl_modem_efficiency")
                             .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                             .with_dynamic_parameter("fast_or_slow", which_readout_type)
                             .with_dynamic_parameter("exptime", which_exptime)
                             .with_main_input(zpl_raw_modem_eff)
                             .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                    condition=is_fast_pol, min_ret=0)
                             .with_associated_input(task_zpl_dark_pol,
                                                    [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                    condition=is_long_exp_SlowPol, min_ret=0)
                             .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1],
                                                    min_ret=0, match_rules=rules.zpl_ff_cam1)
                             .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2],
                                                    min_ret=0, match_rules=rules.zpl_ff_cam2)
                             .with_meta_targets([qc1calib])
                             .build())

    task_zpl_modem_eff_cam1 = (task("zimpol_modem_efficiency_cam1")
                               .with_recipe("sph_zpl_modem_efficiency")
                               .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                               .with_dynamic_parameter("fast_or_slow", which_readout_type)
                               .with_dynamic_parameter("exptime", which_exptime)
                               .with_main_input(zpl_raw_modem_eff)
                               .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                      condition=is_fast_pol, min_ret=0)
                               .with_associated_input(task_zpl_dark_pol,
                                                      [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                      condition=is_long_exp_SlowPol, min_ret=0)
                               .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1],
                                                      min_ret=0, match_rules=rules.zpl_ff_cam1)
                               .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2],
                                                      min_ret=0, match_rules=rules.zpl_ff_cam2)
                               .with_output_filter(ZPL_MODEM_EFF_CAM1)
                               .build())

    task_zpl_modem_eff_cam2 = (task("zimpol_modem_efficiency_cam2")
                               .with_recipe("sph_zpl_modem_efficiency")
                               .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                               .with_dynamic_parameter("fast_or_slow", which_readout_type)
                               .with_dynamic_parameter("exptime", which_exptime)
                               .with_main_input(zpl_raw_modem_eff)
                               .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                      condition=is_fast_pol, min_ret=0)
                               .with_associated_input(task_zpl_dark_pol,
                                                      [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                      condition=is_long_exp_SlowPol, min_ret=0)
                               .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1],
                                                      min_ret=0, match_rules=rules.zpl_ff_cam1)
                               .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2],
                                                      min_ret=0, match_rules=rules.zpl_ff_cam2)
                               .with_output_filter(ZPL_MODEM_EFF_CAM2)
                               .build())

    return task_zpl_bias_pol, task_zpl_dark_pol, task_zpl_flat_pol_cam1, task_zpl_flat_pol_cam2, \
        task_zpl_modem_eff_cam1, task_zpl_modem_eff_cam2, task_zpl_flat_pol_qc, task_zpl_modem_eff_qc


@subworkflow("zimpol_standard_polarimetry", "")
def process_zpl_standard_pol(task_zpl_bias_pol, task_zpl_dark_pol, task_zpl_flat_pol_cam1, task_zpl_flat_pol_cam2,
                             task_zpl_modem_eff_cam1,
                             task_zpl_modem_eff_cam2):
    # Determination of instrumental polarization from unpolarized stars
    # Not needed for science processing, monitoring only
    task_zpl_std_pol0 = (task("zimpol_standard_polarimetry_unpolarized")
                         .with_recipe("sph_zpl_science_p1")
                         .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                         .with_dynamic_parameter("fast_or_slow", which_readout_type)
                         .with_dynamic_parameter("exptime", which_exptime)
                         .with_main_input(zpl_raw_std_pol0)
                         .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                min_ret=1)
                         .with_associated_input(task_zpl_dark_pol, [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                condition=is_long_exp_SlowPol, min_ret=0)
                         .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1], min_ret=0,
                                                match_rules=rules.zpl_ff_cam1)
                         .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2], min_ret=0,
                                                match_rules=rules.zpl_ff_cam2)
                         .with_associated_input(task_zpl_modem_eff_cam1, [ZPL_MODEM_EFF_CAM1], min_ret=0,
                                                match_rules=rules.zpl_modemeff1_cal)
                         .with_associated_input(task_zpl_modem_eff_cam2, [ZPL_MODEM_EFF_CAM2], min_ret=0,
                                                match_rules=rules.zpl_modemeff2_cal)
                         .with_associated_input(zpl_pol_correct_table, min_ret=1)
                         .with_associated_input(zpl_filter_table, min_ret=1)
                         .with_meta_targets([qc1calib, calchecker])
                         .build())

    # Determination of offsets in polarization level and angle from highly polarized stars
    # Not needed for science processing, monitoring only
    task_zpl_std_polhigh = (task("zimpol_standard_polarimetry_polarized")
                            .with_recipe("sph_zpl_science_p1")
                            .with_report("sphere_rawdisp", ReportInput.RECIPE_INPUTS)
                            .with_dynamic_parameter("fast_or_slow", which_readout_type)
                            .with_dynamic_parameter("exptime", which_exptime)
                            .with_main_input(zpl_raw_std_polhigh)
                            .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                   min_ret=1)
                            .with_associated_input(task_zpl_dark_pol, [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                   condition=is_long_exp_SlowPol, min_ret=0)
                            .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1], min_ret=0,
                                                   match_rules=rules.zpl_ff_cam1)
                            .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2], min_ret=0,
                                                   match_rules=rules.zpl_ff_cam2)
                            .with_associated_input(task_zpl_modem_eff_cam1, [ZPL_MODEM_EFF_CAM1], min_ret=0,
                                                   match_rules=rules.zpl_modemeff1_cal)
                            .with_associated_input(task_zpl_modem_eff_cam2, [ZPL_MODEM_EFF_CAM2], min_ret=0,
                                                   match_rules=rules.zpl_modemeff2_cal)
                            .with_associated_input(zpl_pol_correct_table, min_ret=1)
                            .with_associated_input(zpl_filter_table, min_ret=1)
                            .with_associated_input(zpl_polhigh_star_table, min_ret=1)
                            .with_meta_targets([qc1calib, calchecker])
                            .build())

    return task_zpl_std_pol0, task_zpl_std_polhigh


@subworkflow("zimpol_science_polarimetry_p1", "")
def process_zpl_science_p1(task_zpl_bias_pol, task_zpl_dark_pol, task_zpl_flat_pol_cam1, task_zpl_flat_pol_cam2,
                           task_zpl_modem_eff_cam1,
                           task_zpl_modem_eff_cam2, task_zpl_coronagraph_center_pol):
    # Determine flux of science target for coronagraphic data (telescope moved so that the star can be observed without
    # coronagraph) taken in mode P1 (no field derotation)
    # Not used for science processing
    task_zpl_science_flux_p1 = (task("zimpol_science_flux_p1")
                                .with_recipe("sph_zpl_science_p1")
                                .with_dynamic_parameter("fast_or_slow", which_readout_type)
                                .with_dynamic_parameter("exptime", which_exptime)
                                .with_main_input(zpl_raw_science_flux_p1)
                                .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                       condition=is_fast_pol, min_ret=0)
                                .with_associated_input(task_zpl_dark_pol, [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                       condition=is_long_exp_SlowPol, min_ret=0)
                                .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1],
                                                       min_ret=0, match_rules=rules.zpl_ff_cam1)
                                .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2],
                                                       min_ret=0, match_rules=rules.zpl_ff_cam2)
                                .with_associated_input(task_zpl_modem_eff_cam1, [ZPL_MODEM_EFF_CAM1],
                                                       min_ret=0, match_rules=rules.zpl_modemeff1_flx)
                                .with_associated_input(task_zpl_modem_eff_cam2, [ZPL_MODEM_EFF_CAM2],
                                                       min_ret=0, match_rules=rules.zpl_modemeff2_flx)
                                .with_meta_targets([qc0, calchecker])
                                .build())

    # Processing of science observations (object behind coronagraph or taken without coronagraph)
    # taken in mode P1 (no field derotation)
    task_zpl_science_p1 = (task("zimpol_science_p1")
                           .with_recipe("sph_zpl_science_p1")
                           .with_dynamic_parameter("fast_or_slow", which_readout_type)
                           .with_dynamic_parameter("exptime", which_exptime)
                           .with_main_input(zpl_raw_science_p1)
                           .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                  condition=is_fast_pol, min_ret=0)
                           .with_associated_input(task_zpl_dark_pol, [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                  condition=is_long_exp_SlowPol, min_ret=0)
                           .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1], min_ret=0,
                                                  match_rules=rules.zpl_ff_cam1)
                           .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2], min_ret=0,
                                                  match_rules=rules.zpl_ff_cam2)
                           .with_associated_input(task_zpl_modem_eff_cam1, [ZPL_MODEM_EFF_CAM1], min_ret=0,
                                                  match_rules=rules.zpl_modemeff1_sci)
                           .with_associated_input(task_zpl_modem_eff_cam2, [ZPL_MODEM_EFF_CAM2], min_ret=0,
                                                  match_rules=rules.zpl_modemeff2_sci)
                           .with_associated_input(task_zpl_coronagraph_center_pol, [ZPL_STAR_CENTER_POL_CAM1,
                                                                                    ZPL_STAR_CENTER_POL_CAM2],
                                                  min_ret=0)
                           .with_associated_input(task_zpl_science_flux_p1, min_ret=0)
                           .with_input_filter(ZPL_SCIENCE_P1_REDUCED_QPLUS_CAM1, ZPL_SCIENCE_P1_REDUCED_QPLUS_CAM2,
                                              mode=FilterMode.REJECT)
                           .with_meta_targets([qc0, science, calchecker])
                           .build())

    return task_zpl_science_flux_p1, task_zpl_science_p1


@subworkflow("zimpol_science_polarimetry_p2", "")
def process_zpl_science_p2(task_zpl_bias_pol, task_zpl_dark_pol, task_zpl_flat_pol_cam1, task_zpl_flat_pol_cam2,
                           task_zpl_modem_eff_cam1, task_zpl_modem_eff_cam2, task_zpl_coronagraph_center_pol):
    # Determine flux of science target for coronagraphic data (telescope moved so that the star can be observed without
    # coronagraph) taken in mode P2 (active field derotation)
    # Not used for science processing
    task_zpl_science_flux_p2 = (task("zimpol_science_flux_p2")
                                .with_recipe("sph_zpl_science_p23")
                                .with_dynamic_parameter("fast_or_slow", which_readout_type)
                                .with_dynamic_parameter("exptime", which_exptime)
                                .with_main_input(zpl_raw_science_flux_p2)
                                .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                       condition=is_fast_pol, min_ret=0)
                                .with_associated_input(task_zpl_dark_pol, [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                       condition=is_long_exp_SlowPol, min_ret=0)
                                .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1],
                                                       min_ret=0, match_rules=rules.zpl_ff_cam1)
                                .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2],
                                                       min_ret=0, match_rules=rules.zpl_ff_cam2)
                                .with_associated_input(task_zpl_modem_eff_cam1, [ZPL_MODEM_EFF_CAM1], min_ret=0,
                                                       match_rules=rules.zpl_modemeff1_flx)
                                .with_associated_input(task_zpl_modem_eff_cam2, [ZPL_MODEM_EFF_CAM2], min_ret=0,
                                                       match_rules=rules.zpl_modemeff2_flx)
                                .with_meta_targets([qc0, calchecker])
                                .build())

    # Processing of science observations (object behind coronagraph or taken without coronagraph)
    # taken in mode P2 (active field derotation)
    task_zpl_science_p2 = (task("zimpol_science_p2")
                           .with_recipe("sph_zpl_science_p23")
                           .with_dynamic_parameter("fast_or_slow", which_readout_type)
                           .with_dynamic_parameter("exptime", which_exptime)
                           .with_main_input(zpl_raw_science_p2)
                           .with_associated_input(task_zpl_bias_pol, [ZPL_MASTER_BIAS_CAM1, ZPL_MASTER_BIAS_CAM2],
                                                  condition=is_fast_pol, min_ret=0)
                           .with_associated_input(task_zpl_dark_pol, [ZPL_MASTER_DARK_CAM1, ZPL_MASTER_DARK_CAM2],
                                                  condition=is_long_exp_SlowPol, min_ret=0)
                           .with_associated_input(task_zpl_flat_pol_cam1, [ZPL_INT_FLAT_FIELD_MASTER_CAM1], min_ret=0,
                                                  match_rules=rules.zpl_ff_cam1)
                           .with_associated_input(task_zpl_flat_pol_cam2, [ZPL_INT_FLAT_FIELD_MASTER_CAM2], min_ret=0,
                                                  match_rules=rules.zpl_ff_cam2)
                           .with_associated_input(task_zpl_modem_eff_cam1, [ZPL_MODEM_EFF_CAM1], min_ret=0,
                                                  match_rules=rules.zpl_modemeff1_sci)
                           .with_associated_input(task_zpl_modem_eff_cam2, [ZPL_MODEM_EFF_CAM2], min_ret=0,
                                                  match_rules=rules.zpl_modemeff2_sci)
                           .with_associated_input(task_zpl_coronagraph_center_pol, [ZPL_STAR_CENTER_POL_CAM1,
                                                                                    ZPL_STAR_CENTER_POL_CAM2],
                                                  min_ret=0)
                           .with_associated_input(task_zpl_science_flux_p2, min_ret=0)
                           .with_input_filter(ZPL_SCIENCE_P23_REDUCED_QPLUS_CAM1, ZPL_SCIENCE_P23_REDUCED_QPLUS_CAM2,
                                              mode=FilterMode.REJECT)
                           .with_meta_targets([qc0, science, calchecker])
                           .build())

    return task_zpl_science_flux_p2, task_zpl_science_p2
