from dataclasses import dataclass

from edps import Task, ReportInput
from edps import task, subworkflow, qc1calib, match_rules, FilterMode, qc0

from .visir_datasources import *


@dataclass
class VisirSpectra:
    repack: Task
    undistort: Task
    destripe: Task
    processed: Task


def process_spectra(raw_input, data_type, data_format):
    # This function contains the tasks needed to process spectroscopic data
    # of standard stars and science exposures.
    # The input variables data_format (echelle or long-slit) and data_type (science
    # or standard) defines the name of the task.

    # The data_type defines the metatarget list of the last task in the subworkflow.
    if data_type == "science":
        metatargets = [qc0]
        renamed = SPEC_OBS_LMR_PREPROCESSED if data_format == "long_slit" else SPEC_OBS_HRG_PREPROCESSED
    else:  # standard
        metatargets = [qc1calib]
        renamed = SPEC_CAL_PHOT_PREPROCESSED if data_format == "long_slit" else SPEC_CAL_PHOT_HRG_PREPROCESSED

    # The input variables data_format (echelle or long-slit) and data_type (science
    # or standard) defines the name of the task.
    suffix = data_format + '_' + data_type
    repack_building = (task("repack_" + suffix)
                       .with_recipe("visir_util_repack")
                       .with_main_input(raw_input)
                       .with_associated_input(static_mask, min_ret=0)
                       .with_meta_targets(metatargets))

    if data_type == "standard" and (data_format == "echelle" or data_format == "long_slit"):
        repack = (repack_building
                  .with_report("visir_rawdisp", ReportInput.RECIPE_INPUTS)
                  .build())
    else:
        repack = repack_building.build()

    undistort = (task("undistort_" + suffix)
                 .with_recipe("visir_util_undistort")
                 .with_main_input(repack)
                 .with_associated_input(linearity_table, min_ret=0)
                 .with_associated_input(static_mask, min_ret=0)
                 .with_meta_targets(metatargets)
                 .build())

    destripe = (task("destripe_" + suffix)
                .with_recipe("visir_old_util_destripe")
                .with_main_input(undistort)
                .with_associated_input(linearity_table, min_ret=0)
                .with_associated_input(static_mask, min_ret=0)
                .with_meta_targets(metatargets)
                .build())

    assoc_undistort = (match_rules()
                       .with_match_keywords([kwd.mjd]))

    process_spectra_building = (task("process_" + suffix)
                                .with_recipe("visir_old_spc_obs")
                                .with_main_input(destripe)
                                .with_associated_input(undistort, match_rules=assoc_undistort)
                                .with_associated_input(spectroscopic_efficiency)
                                .with_associated_input(telluric_lines)
                                .with_associated_input(spectroscopic_catalog)
                                .with_input_map({DESTRIPED: renamed})
                                .with_input_filter(UNDISTORTED, mode=FilterMode.REJECT)
                                .with_meta_targets(metatargets))

    if suffix == 'long_slit_standard':
        process_spectra = (process_spectra_building
                           .with_report("visir_specphot_std", ReportInput.RECIPE_INPUTS_OUTPUTS)
                           .build())
    else:
        process_spectra = process_spectra_building.build()

    return VisirSpectra(repack=repack,
                        undistort=undistort,
                        destripe=destripe,
                        processed=process_spectra)


@subworkflow("process_science_echelle", "")
# This workflow calls the function reduce_spectra to reduce science
#  observations taken in echelle mode.
def process_science_echelle(raw_input):
    return process_spectra(raw_input, "science", "echelle")


@subworkflow("process_standard_echelle", "")
# This workflow calls the function reduce_spectra to reduce standard
# star observations taken in echelle mode.
def process_standard_echelle(raw_input):
    return process_spectra(raw_input, "standard", "echelle")


@subworkflow("process_science_longslit", "")
# This workflow calls the function reduce_spectra to reduce science
# observations taken in long-slit mode.
def process_science_longslit(raw_input):
    return process_spectra(raw_input, "science", "long_slit")


@subworkflow("process_standard_longslit", "")
# This workflow calls the function reduce_spectra to reduce  standard
# star observations taken in long-slit mode.
def process_standard_longslit(raw_input):
    return process_spectra(raw_input, "standard", "long_slit")
