/*
 * This file is part of the CR2RES Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <cr2res_dfs.h>
#include <cr2res_calib.h>
#include <cr2res_io.h>
#include "cr2res_pfits.h"
#include "cr2res_utils.h"

#define MODE_FLAT 0
#define MODE_DARK 1
#define MODE_BPM 2
#define MODE_DETLIN 3

#define pow2(x) (x) * (x)
#define detlin(x, a, b, c) (x) * ((a) + (x) * ((b) + (x) * (c)))
#define deterr(x, a, b, c, sx, sa, sb, sc) sqrt(detlin(pow2(x), pow2(sa), pow2(sb), pow2(sc)) + pow2(sx) * pow2((a) + 2 * (b) * (x) + 3 * (c) * pow2(x)))

#define localdir "." //getenv("srcdir")

static hdrl_image * cr2res_create_hdrl(int nx, int ny, double value, double error);

static void test_cr2res_calib_image(void);
static void test_cr2res_calib_cosmic(void);
static void test_cr2res_calib_flat(void);
static void test_cr2res_calib_dark(void);
static void test_cr2res_calib_bpm(void);
static void test_cr2res_calib_detlin(void);


static void create_empty_fits()
{
    // Many tests require a empty fits file, since crires code checks if a file exists
    // when adding a filename to a frame
	char *my_path = cpl_sprintf("%s/TEST_empty.fits", localdir);
    cpl_propertylist * list = cpl_propertylist_new();
    cpl_propertylist_append_string(list, CR2RES_HEADER_DRS_TYPE, "DEBUG");
    cpl_propertylist_save(list, my_path, CPL_IO_CREATE);
    cpl_propertylist_delete(list);
    cpl_free(my_path);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save the given hdrl image, in the right way depending on what we want
 */
/*----------------------------------------------------------------------------*/
static void save_hdrl(char * filename, hdrl_image * hdrl, int mode, double dit)
{
    // Empty structures needed for master flat frame, but not for the test
    // Need an empty fits file with header for the framesets
    cpl_frame * empty = cpl_frame_new();

	char *my_path = cpl_sprintf("%s/TEST_empty.fits", localdir);
    cpl_frame_set_filename(empty, my_path);
    cpl_frame_set_tag(empty, "DEBUG");
    cpl_frame_set_group(empty, CPL_FRAME_GROUP_CALIB);

    cpl_parameterlist * parlist = cpl_parameterlist_new();
    cpl_propertylist * ext1 = cpl_propertylist_new();
    cpl_propertylist_append_double(ext1, CR2RES_HEADER_DIT, dit);
    cpl_propertylist * ext[] = {ext1, NULL, NULL};


    cpl_frameset * all = cpl_frameset_new();
    cpl_frameset * in = cpl_frameset_new();
    cpl_frameset_insert(all, empty);
    empty = cpl_frame_duplicate(empty);
    cpl_frameset_insert(in, empty);

    hdrl_image * list[] = {hdrl, NULL, NULL};

    if (mode == MODE_FLAT)
        cr2res_io_save_MASTER_FLAT(filename, all, in, parlist, list, ext1, ext, CR2RES_CAL_FLAT_MASTER_PROCATG, "debug");
    if (mode == MODE_DARK)
        cr2res_io_save_MASTER_DARK(filename, all, in, parlist, list, ext1, ext, CR2RES_CAL_DARK_MASTER_PROCATG, "debug");
    if (mode == MODE_BPM)
    {
        cpl_image * list2[] = {hdrl_image_get_image(hdrl), NULL, NULL};
        cr2res_io_save_BPM(filename, all, in, parlist, list2, ext1, ext, CR2RES_CAL_FLAT_BPM_PROCATG, "debug");    
    }
    if (mode == MODE_DETLIN){
        hdrl_imagelist * list3 = hdrl_imagelist_new();
        hdrl_imagelist_set(list3, hdrl, 0);
        hdrl_imagelist_set(list3, hdrl, 1);
        hdrl_imagelist_set(list3, hdrl, 2);

        hdrl_imagelist * list4[] = {list3, NULL, NULL};

        cr2res_io_save_DETLIN_COEFFS(filename, all, in, parlist, list4, ext1, ext, CR2RES_CAL_DETLIN_COEFFS_PROCATG, "debug");

        // hdrl_imagelist_unset(list3, 0);
        // hdrl_imagelist_unset(list3, 1);
        // hdrl_imagelist_unset(list3, 2);
        hdrl_imagelist_delete(list3);
    }
    cpl_frameset_delete(all);
    cpl_frameset_delete(in);
    cpl_free(my_path) ;
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(ext1);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a Flat Field fits file and cpl_frame
 */
/*----------------------------------------------------------------------------*/
static cpl_frame * create_master_flat(char * filename, int nx, int ny, double value, double error, cpl_mask ** bpm)
{
    cpl_frame * out = cpl_frame_new();
    cpl_frame_set_filename(out, filename);
    cpl_frame_set_tag(out, "FLAT");
    cpl_frame_set_group(out, CPL_FRAME_GROUP_CALIB);

    hdrl_image * hdrl = cr2res_create_hdrl(nx, ny, value, error);

    if (bpm != NULL){
        hdrl_image_reject(hdrl, 1, 1);
        *bpm = cpl_mask_duplicate(hdrl_image_get_mask(hdrl));
    }

    save_hdrl(filename, hdrl, MODE_FLAT, 0);
    hdrl_image_delete(hdrl);

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a Dark Current fits file and cpl_frame
 */
/*----------------------------------------------------------------------------*/
static cpl_frame * create_master_dark(char * filename, int nx, int ny, double value, double error, double dit, cpl_mask ** bpm)
{

    cpl_frame * out = cpl_frame_new();
    cpl_frame_set_filename(out, filename);
    cpl_frame_set_tag(out, "DARK");
    cpl_frame_set_group(out, CPL_FRAME_GROUP_CALIB);

    hdrl_image * hdrl = cr2res_create_hdrl(nx, ny, value, error);

    if (bpm != NULL){
        hdrl_image_reject(hdrl, 1, 1);
        *bpm = cpl_mask_duplicate(hdrl_image_get_mask(hdrl));
    }

    save_hdrl(filename, hdrl, MODE_DARK, dit);
    hdrl_image_delete(hdrl);

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a Bad Pixel Map (BPM) fits file and cpl_frame
 */
/*----------------------------------------------------------------------------*/
static cpl_frame * create_bpm(char * filename, int nx, int ny, double value)
{
    cpl_frame * out = cpl_frame_new();
    cpl_frame_set_filename(out, filename);
    cpl_frame_set_tag(out, "BPM");
    cpl_frame_set_group(out, CPL_FRAME_GROUP_CALIB);

    hdrl_image * hdrl = cr2res_create_hdrl(nx, ny, value, 0);
    save_hdrl(filename, hdrl, MODE_BPM, 0);
    hdrl_image_delete(hdrl);

    return out;   
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a Determine Linearity (DetLin) fits file and cpl_frame
 */
/*----------------------------------------------------------------------------*/
static cpl_frame * create_detlin(char * filename, hdrl_image * a, hdrl_image * b, hdrl_image * c)
{
	char *my_path = cpl_sprintf("%s/TEST_empty.fits", localdir);

    // Empty structures needed for master flat frame, but not for the test
    // Need an empty fits file with header for the framesets
    cpl_frame * empty = cpl_frame_new();
    cpl_frame_set_filename(empty, my_path);
    cpl_frame_set_tag(empty, "DEBUG");
    cpl_frame_set_group(empty, CPL_FRAME_GROUP_CALIB);

    cpl_parameterlist * parlist = cpl_parameterlist_new();
    cpl_propertylist * ext1 = cpl_propertylist_new();
    cpl_propertylist * ext[] = {ext1, NULL, NULL};

    cpl_frameset * all = cpl_frameset_new();
    cpl_frameset * in = cpl_frameset_new();
    cpl_frameset_insert(all, empty);
    empty = cpl_frame_duplicate(empty);
    cpl_frameset_insert(in, empty);

    hdrl_imagelist * list3 = hdrl_imagelist_new();
    hdrl_imagelist_set(list3, a, 0);
    hdrl_imagelist_set(list3, b, 1);
    hdrl_imagelist_set(list3, c, 2);

    hdrl_imagelist * list4[] = {list3, NULL, NULL};

    cr2res_io_save_DETLIN_COEFFS(filename, all, in, parlist, list4, ext1, ext, CR2RES_CAL_DETLIN_COEFFS_PROCATG, "debug");

    hdrl_imagelist_unset(list3, 2);
    hdrl_imagelist_unset(list3, 1);
    hdrl_imagelist_unset(list3, 0);

    hdrl_imagelist_delete(list3);
    
    cpl_frameset_delete(all);
    cpl_frameset_delete(in);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(ext1);

    cpl_frame * out = cpl_frame_new();
    cpl_frame_set_filename(out, filename);
    cpl_frame_set_tag(out, "DETLIN");
    cpl_frame_set_group(out, CPL_FRAME_GROUP_CALIB);

    cpl_free(my_path);

    return out;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a hdrl image with constant value and error
 */
/*----------------------------------------------------------------------------*/
static hdrl_image * cr2res_create_hdrl(int nx, int ny, double value, double error)
{
    hdrl_image * out = hdrl_image_new(nx, ny);
    hdrl_value hv;
    hv.data = value;
    hv.error = error;
    hdrl_image_add_scalar(out, hv);
    return out;
}

static double calc_read_noise(double dit, int ndit, int chip)
{
    double min_dit = 1.427;
    double lim_dit = 50.0;
    double min_rn;
    if (chip ==1){
        min_rn = 11.0;
    } else {
        min_rn = 12.0;
    }
    double lim_rn = 6.0;
    if(dit <= min_dit){
        return min_rn/CR2RES_GAIN_CHIP1;
    } else if(dit >= lim_dit) {
        return lim_rn/CR2RES_GAIN_CHIP1;
    } else {
        double rn = min_rn + (lim_rn-min_rn)*(dit-min_dit)/(lim_dit-min_dit);
        return rn/CR2RES_GAIN_CHIP1;
    }
}

static void add_read_noise(hdrl_image * im, double dit, int chip)
{
    double read_noise = calc_read_noise(dit, 1, chip);
    cpl_image_add_scalar(hdrl_image_get_error(im), read_noise);
}

static void add_read_noise_cpl(cpl_image * im, double dit, int chip)
{
    double read_noise = calc_read_noise(dit, 1, chip);
    cpl_image_add_scalar(im, read_noise);
}

static void test_cr2res_calib_image()
{
    int nx = 5;
    int ny = 5;
    //int badpix;

    double img_value = 100;
    double img_error = 1;

    hdrl_image * in = cr2res_create_hdrl(nx, ny, img_value, img_error);
    int chip = 1;

	char *my_path1 = cpl_sprintf("%s/TEST_master_flat.fits", localdir);
	char *my_path2 = cpl_sprintf("%s/TEST_master_dark.fits", localdir);
	char *my_path3 = cpl_sprintf("%s/TEST_bpm.fits", localdir);
    cpl_frame * flat = create_master_flat(my_path1, nx, ny, 1, 0, NULL);
    cpl_frame * dark = create_master_dark(my_path2, nx, ny, 10, 1, 10, NULL);
    cpl_frame * bpm = create_bpm(my_path3, nx, ny, 0);
    cpl_frame * detlin = NULL;
    double dit = 10;

    hdrl_image * out;
    //hdrl_image * cmp;

    // NULL input / output
    out = cr2res_calib_image(NULL, chip, 0, 0, 1, 0, NULL, NULL, NULL, NULL,dit,1);
    cpl_test_null(out);

    out = cr2res_calib_image(in, 0, 0, 0, 1, 0, NULL, NULL, NULL, NULL, dit, 1);
    cpl_test_null(out);

    out = cr2res_calib_image(in, CR2RES_NB_DETECTORS + 1, 0, 0, 1, 0, 
        NULL, NULL, NULL, NULL, dit, 1);
    cpl_test_null(out);

    // No correction
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, NULL, NULL, NULL, NULL, dit, 1);
    cpl_test_nonnull(out);
    cpl_test_image_abs(hdrl_image_get_image(in), hdrl_image_get_image(out), DBL_EPSILON);

    // add the shot noise error
    cpl_image_add_scalar(hdrl_image_get_error(in), sqrt(img_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(in, dit, chip);

    cpl_test_image_abs(hdrl_image_get_error(in), hdrl_image_get_error(out), DBL_EPSILON);
    hdrl_image_delete(out);


    hdrl_image_delete(in);
    cpl_frame_delete(flat);
    cpl_frame_delete(dark);
    cpl_frame_delete(bpm);
    cpl_frame_delete(detlin);
    cpl_free(my_path1) ;
    cpl_free(my_path2) ;
    cpl_free(my_path3) ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test Cosmic Correction
 */
/*----------------------------------------------------------------------------*/
static void test_cr2res_calib_cosmic()
{
    int nx = 5;
    int ny = 5;
    //int badpix;

    double img_value = 100;
    double img_error = 1;
    //double out_value, out_error;
    //double flat_value, flat_error;

    hdrl_image * in, * out;// * cmp;
    //cpl_frame * flat;
    int chip = 1;
    //int cosmics_corr;
    double dit = 10;

    in = cr2res_create_hdrl(nx, ny, img_value, img_error);

    // Case 1: No Cosmic Correction, i.e. nothing happens
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, NULL, NULL, NULL, NULL, dit, 1);
    cpl_test_image_abs(hdrl_image_get_image(in), hdrl_image_get_image(out), DBL_EPSILON);
    // add the shot noise
    cpl_image_add_scalar(hdrl_image_get_error(in), sqrt(img_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(in, dit, chip);

    cpl_test_image_abs(hdrl_image_get_error(in), hdrl_image_get_error(out), DBL_EPSILON);

    // Case 2: Cosmic Correction
    // TODO when cosmic correction is implemented

    hdrl_image_delete(in);
    hdrl_image_delete(out);

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Only test the flat field calibration
 */
/*----------------------------------------------------------------------------*/
static void test_cr2res_calib_flat()
{
    int nx = 5;
    int ny = 5;
    //int badpix;

    double img_value = 100;
    double img_error = 1;
    double tmp_error = 0;
    double out_value, out_error;
    double flat_value, flat_error;

    hdrl_image * in, * out, * cmp;
    //cpl_mask * bpm;
    cpl_frame * flat;
    int chip = 1;
    double dit = 10;
    //cpl_error_code error;

    in = cr2res_create_hdrl(nx, ny, img_value, img_error);

    // Case 1: Flat is just 1 and no error, i.e. no change
    flat_value = 1;
    flat_error = 0;
	char *my_path1 = cpl_sprintf("%s/TEST_master_flat.fits", localdir);
    flat = create_master_flat(my_path1, nx, ny, flat_value, flat_error, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    
    out_value = img_value / flat_value;
    out_error = sqrt( pow(img_error / flat_value, 2) + pow(img_value * flat_error/ (flat_value * flat_value), 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    // add the shot noise
    cpl_image_add_scalar(hdrl_image_get_error(cmp), sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(flat);

    // Case 2: Flat is constant == 2, error 1 
    flat_value = 2;
    flat_error = 1;
    flat = create_master_flat(my_path1, nx, ny, flat_value, flat_error, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    
    out_value = img_value / flat_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);

    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);

    out_error = sqrt( pow(tmp_error / flat_value, 2) + pow(img_value * flat_error/ (flat_value * flat_value), 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(flat);

    // Case 3: Flat is 0, i.e. all pixels become bad
    flat_value = 0;
    flat_error = 1;
    flat = create_master_flat(my_path1, nx, ny, flat_value, flat_error, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    
    cpl_test_eq(nx * ny, hdrl_image_count_rejected(out));

    hdrl_image_delete(out);
    cpl_frame_delete(flat);

    // Case 4: Flat file does not exist
    flat = cpl_frame_new();
    cpl_frame_set_filename(flat, "TEST_tobeornottobe.fits");
    cpl_frame_set_tag(flat, "FLAT");
    cpl_frame_set_group(flat, CPL_FRAME_GROUP_CALIB);

    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    cpl_test_error(CPL_ERROR_FILE_IO);
    hdrl_image_delete(out);
    cpl_frame_delete(flat);

    // Case 5: image is in a wrong group
    flat = create_master_dark(my_path1, nx, ny, 1, 0, 10, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(flat);

    // Case 6: No Filename set
    flat = cpl_frame_new();
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_null(out);
    cpl_frame_delete(flat);

    // Case 7: DataFile is empty, i.e. only header
    flat = cpl_frame_new();
	char *my_path2 = cpl_sprintf("%s/TEST_empty.fits", localdir);
    cpl_frame_set_filename(flat, my_path2);
    out = cr2res_calib_image(in, chip, 0, 0, 1, 0, flat, NULL, NULL, NULL, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(flat);

    // // Case 8: BPM is set 
    // flat_value = 1;
    // flat_error = 1;
    // flat = create_master_flat("TEST_master_flat.fits", nx, ny, flat_value, flat_error, &bpm);
    // out = cr2res_calib_image(in, chip, 0, 0, 0, flat, NULL, NULL, NULL, dit, 1);
    
    // cpl_test_eq_mask(bpm, hdrl_image_get_mask(out));

    // hdrl_image_delete(out);
    // cpl_frame_delete(flat);
    // cpl_mask_delete(bpm);


    cpl_free(my_path1) ;
    cpl_free(my_path2) ;
    hdrl_image_delete(in);
}

static void test_cr2res_calib_dark()
{
    int nx = 5;
    int ny = 5;
    //int badpix;

    double img_value = 100;
    double img_error = 1;
    double tmp_error = 0;
    double out_value, out_error;
    double dark_value, dark_error, dark_dit;

    hdrl_image * in, * out, * cmp;
    //cpl_mask *bpm;
    cpl_frame * dark;
    int chip = 1;
    double dit = 10;

    in = cr2res_create_hdrl(nx, ny, img_value, img_error);
	char *my_path = cpl_sprintf("%s/TEST_master_dark.fits", localdir);
	char *my_path2 = cpl_sprintf("%s/TEST_empty.fits", localdir);


    // Case 1: Dark is 0, dit is the same as the image, i.e. no correction
    dark_value = 0;
    dark_error = 0;
    dark_dit = dit;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 2: Dark is 0, dit is different, i.e. still nothing
    dark_value = 0;
    dark_error = 0;
    dark_dit = dit * 2.3;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 3: Dark is 0, dit is same, but dark has error now
    dark_value = 0;
    dark_error = 2;
    dark_dit = dit;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 4: Dark is constant 10, dit is same, and has error
    dark_value = 10;
    dark_error = 2;
    dark_dit = dit;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 5: Dark is constant 10, dit is different, and has error
    dark_value = 10;
    dark_error = 2;
    dark_dit = dit * 2.3;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 6: Dark is negative, dit positive
    dark_value = -10;
    dark_error = 2;
    dark_dit = dit * 2.3;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 7: Dark is positive, dit negative
    dark_value = 10;
    dark_error = 2;
    dark_dit = -dit * 2.3;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 8: Dark and dit negative
    dark_value = -10;
    dark_error = 2;
    dark_dit = -dit * 2.3;
    dark = create_master_dark(my_path, nx, ny, dark_value, dark_error, dark_dit, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    out_value = img_value - dit/dark_dit * dark_value;
    tmp_error = img_error + sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1);
    tmp_error = tmp_error + calc_read_noise(dit, 1, chip);
    out_error = sqrt(pow(tmp_error, 2) + pow(dit/dark_dit * dark_error, 2));
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);

    cpl_test_image_abs(hdrl_image_get_image(cmp), hdrl_image_get_image(out), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(cmp), hdrl_image_get_error(out), DBL_EPSILON);

    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(dark);

    // Case 9: Dark file does not exist
    dark = cpl_frame_new();
    cpl_frame_set_filename(dark, "TEST_tobeornottobe.fits");
    cpl_frame_set_tag(dark, "DARK");
    cpl_frame_set_group(dark, CPL_FRAME_GROUP_CALIB);

    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    cpl_test_error(CPL_ERROR_FILE_IO);
    cpl_test_null(out);
    cpl_frame_delete(dark);

    // Case 10: image is in a wrong group
    dark = create_master_flat(my_path, nx, ny, 1, 0, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(dark);

    // Case 11: No Filename set
    dark = cpl_frame_new();
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_null(out);
    cpl_frame_delete(dark);



    // Case 12: DataFile is empty, i.e. only header
    dark = cpl_frame_new();
    cpl_frame_set_filename(dark, my_path2);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(dark);

    // // Case 13: Check that bpm of dark is passed on to out
    // dark_value = 10;
    // dark_error = 0;
    // dark_dit = 10;
    // dark = create_master_dark("TEST_master_dark.fits", nx, ny, dark_value, dark_error, dark_dit, &bpm);
    // out = cr2res_calib_image(in, chip, 0, 0, 0, NULL, dark, NULL, NULL, dit, 1);
    
    // cpl_test_eq_mask(bpm, hdrl_image_get_mask(out));

    // cpl_mask_delete(bpm);
    // hdrl_image_delete(out);
    // cpl_frame_delete(dark);

    cpl_free(my_path);
    cpl_free(my_path2);

    hdrl_image_delete(in);
}

static void test_cr2res_calib_bpm()
{
    int nx = 5;
    int ny = 5;
    int badpix;

    double img_value = 100;
    double img_error = 1;
    //double out_value, out_error;
    //double bpm_value;
    hdrl_image * hdrl;
    hdrl_value value;
    cpl_image * tmp;

    hdrl_image * in, * out, * cmp;
    cpl_frame * bpm;
    int chip = 1;
    double dit = 10;

    in = cr2res_create_hdrl(nx, ny, img_value, img_error);
    value.data = 10; value.error = img_error;
    hdrl_image_set_pixel(in, 2, 2, value);

	char *my_path = cpl_sprintf("%s/TEST_master_bpm.fits", localdir);
	char *my_path2 = cpl_sprintf("%s/TEST_empty.fits", localdir);
    

    // Case 1: BPM is all 0, i.e. all good pixels
    bpm = create_bpm(my_path, nx, ny, 0);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cmp = hdrl_image_duplicate(in);
    tmp = cpl_image_power_create(hdrl_image_get_image(cmp), 0.5);
    cpl_image_divide_scalar(tmp, sqrt(CR2RES_GAIN_CHIP1));
    add_read_noise_cpl(tmp, dit, chip);
    cpl_image_add(hdrl_image_get_error(cmp), tmp);
    cpl_image_delete(tmp);

    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), DBL_EPSILON);
    cpl_frame_delete(bpm);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);

    // Case 2: BPM is all 1, i.e all pixels are bad
    bpm = create_bpm(my_path, nx, ny, 1);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cpl_test_nonnull(out);
    cpl_test_eq(hdrl_image_count_rejected(out), nx * ny);
    cpl_frame_delete(bpm);
    hdrl_image_delete(out);

    // Case 3: One Bad Pixel
    bpm = cpl_frame_new();
    cpl_frame_set_filename(bpm, my_path);
    cpl_frame_set_tag(bpm, "BPM");
    cpl_frame_set_group(bpm, CPL_FRAME_GROUP_CALIB);

    hdrl = cr2res_create_hdrl(nx, ny, 0, 0);
    value.data = 1;
    hdrl_image_set_pixel(hdrl, 2, 2, value);
    save_hdrl(my_path, hdrl, MODE_BPM, 0);
    hdrl_image_delete(hdrl);

    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cmp = hdrl_image_duplicate(in);
    tmp = cpl_image_power_create(hdrl_image_get_image(cmp), 0.5);
    cpl_image_divide_scalar(tmp, sqrt(CR2RES_GAIN_CHIP1));
    add_read_noise_cpl(tmp, dit, chip);
    cpl_image_add(hdrl_image_get_error(cmp), tmp);
    cpl_image_delete(tmp);

    cpl_test_eq(hdrl_image_count_rejected(out), 1);
    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);
    // cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), DBL_EPSILON);

    for (size_t i = 1; i <= (size_t) nx; i++)
    {
        for (size_t j = 1; j <= (size_t) ny; j++)
        {
            if (!hdrl_image_is_rejected(out, i, j)){
                cpl_test_abs(cpl_image_get(hdrl_image_get_error(out), i, j, &badpix), 
                        cpl_image_get(hdrl_image_get_error(cmp), i, j, &badpix),
                        DBL_EPSILON);
            }
        }       
        
    }

    cpl_frame_delete(bpm);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);

    // Case 4: Only one good pixel
    bpm = cpl_frame_new();
    cpl_frame_set_filename(bpm, my_path);
    cpl_frame_set_tag(bpm, "BPM");
    cpl_frame_set_group(bpm, CPL_FRAME_GROUP_CALIB);

    hdrl = cr2res_create_hdrl(nx, ny, 1, 0);
    value.data = 0;
    hdrl_image_set_pixel(hdrl, 2, 2, value);
    save_hdrl(my_path, hdrl, MODE_BPM, 0);
    hdrl_image_delete(hdrl);

    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cmp = hdrl_image_duplicate(in);
    tmp = cpl_image_power_create(hdrl_image_get_image(cmp), 0.5);
    cpl_image_divide_scalar(tmp, sqrt(CR2RES_GAIN_CHIP1));
    add_read_noise_cpl(tmp, dit, chip);
    cpl_image_add(hdrl_image_get_error(cmp), tmp);
    cpl_image_delete(tmp);

    cpl_test_eq(hdrl_image_count_rejected(out), nx*ny - 1);
    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);    
    // cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), DBL_EPSILON);

    for (size_t i = 1; i <= (size_t) nx; i++)
    {
        for (size_t j = 1; j <= (size_t) ny; j++)
        {
            if (!hdrl_image_is_rejected(out, i, j)){
                cpl_test_abs(cpl_image_get(hdrl_image_get_error(out), i, j, &badpix), 
                        cpl_image_get(hdrl_image_get_error(cmp), i, j, &badpix),
                        DBL_EPSILON);
            }
        }       
        
    }
    

    cpl_frame_delete(bpm);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);

    // Case 5: BPM file does not exist
    bpm = cpl_frame_new();
    cpl_frame_set_filename(bpm, "TEST_tobeornottobe.fits");
    cpl_frame_set_tag(bpm, "BPM");
    cpl_frame_set_group(bpm, CPL_FRAME_GROUP_CALIB);

    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cpl_test_error(CPL_ERROR_FILE_IO);
    cpl_test_null(out);
    cpl_frame_delete(bpm);

    // Case 6: image is in a wrong group
    bpm = create_master_flat(my_path, nx, ny, 1, 0, NULL);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(bpm);

    // Case 7: No Filename set
    bpm = cpl_frame_new();
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_null(out);
    cpl_frame_delete(bpm);


    // Case 8: DataFile is empty, i.e. only header
    bpm = cpl_frame_new();
    cpl_frame_set_filename(bpm, my_path2);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, bpm, NULL, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(bpm);

    cpl_free(my_path);
    cpl_free(my_path2);

    hdrl_image_delete(in);
}

static void test_cr2res_calib_detlin()
{
    int nx = 5;
    int ny = 5;
    //int badpix;

    double img_value = 20000;
    double img_error = 200;
    double out_value, out_error;
    //double bpm_value;
    //hdrl_image * hdrl;
    //hdrl_value value;

    hdrl_image * in, * out, * cmp;
    hdrl_image * ima, * imb, * imc;
    double a, b, c, sa, sb, sc;
    double i, si;
    cpl_frame * detlin;
    int chip = 1;
    double dit = 10;

    in = cr2res_create_hdrl(nx, ny, img_value, img_error);
    i = img_value; si = img_error;

	char *my_path = cpl_sprintf("%s/TEST_master_detlin.fits", localdir);
	char *my_path2 = cpl_sprintf("%s/TEST_empty.fits", localdir);

    // DetLin images aren't actually images, but polynomial coefficients for each pixel
    // The new image is the old multiplied by the
    // evaluation of that polynomial
    // new image = old_image * detlin_poly(old_image)

    // Case 1: DetLin b = c = 0, a = 1, doesn't change anything
    a = 1; c = 0; b = 0;
    sa = sb = sc = 0;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);

    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    cmp = hdrl_image_duplicate(in);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(img_value) / sqrt(CR2RES_GAIN_CHIP1));
    
    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), DBL_EPSILON);
    hdrl_image_delete(out);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);
    hdrl_image_delete(cmp);

    // Case 2: DetLin a = 1, c = 0, b = 1, x + x**2
    a = b = 1; c = 0;
    sa = sb = sc = 0;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = detlin(img_value, a, b, c);
    out_error = deterr(img_value, a, b, c, img_error, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), 
        hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), 
        hdrl_image_get_error(cmp), 1e-9);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);


    // Case 3: DetLin a = b = c = 0, everything is 0
    a = c = b = 0;
    sa = sb = sc = 0;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    cmp = cr2res_create_hdrl(nx, ny, 0, 0);
    add_read_noise(cmp, dit, chip);
    // Shot noise is also 0
    cpl_test_image_abs(hdrl_image_get_image(out), 
        hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), 
        hdrl_image_get_error(cmp), 1e-14);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 4: DetLin a = 0.5, c = b = 0, divide all values by 2, including the error
    a = 0.5; b = c = 0;
    sa = sb = sc = 0;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = img_value * a;
    out_error = img_error * a;
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));
    
    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), 
        hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), 
        hdrl_image_get_error(cmp), DBL_EPSILON);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 5: DetLin a = 1, c = 0, b = 2
    // x = a + y * b
    a = 2; b = 0; c = 0;
    sa = sb = sc = 0;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = detlin(i, a, b, c);
    out_error = deterr(i, a, b, c, si, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), 
        hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), 
        hdrl_image_get_error(cmp), DBL_EPSILON);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 6: DetLin a = 1, c = 1, b = 2
    // x = a + y * b + y**2 * c
    a = c = 1; b = 2;
    sa = sb = sc = 0;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = detlin(i, a, b, c);
    out_error = deterr(i, a, b, c, si, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), 
        hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), 
        hdrl_image_get_error(cmp), DBL_EPSILON);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 7: DetLin a = 0, c = 0, b = 1, but now with error
    // x = b * y
    a = c = 0; b = 1;
    sa = sc = 0; sb = 1;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = pow2(img_value);
    out_error = deterr(i, a, b, c, si, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);
    
    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), 1e-6);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 8: DetLin a = 1, c = 0, b = 1, but now with error
    a = 1; c = 0; b = 1;
    sa = 1 ; sc = 0; sb = 1;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = detlin(i, a, b, c);
    out_error = deterr(i, a, b, c, si, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value / CR2RES_GAIN_CHIP1));
    
    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), 
        hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), 
        hdrl_image_get_error(cmp), DBL_EPSILON);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 9: DetLin a = 1, c = 1, b = 0, but now with error
    // x = a + c * y**2
    a = 1; c = 1; b = 0;
    sa = 1 ; sc = 1; sb = 0;
    i = img_value; si = img_error;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = detlin(i, a, b, c);
    out_error = deterr(i, a, b, c, si, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), DBL_EPSILON);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 10: DetLin a = 1, c = 1, b = 1, but now with error
    // x = a + b * y + c * y**2
    a = 1; c = 1; b = 1;
    sa = 1 ; sc = 1; sb = 1;
    i = img_value; si = img_error;
    ima = cr2res_create_hdrl(nx, ny, a, sa);
    imb = cr2res_create_hdrl(nx, ny, b, sb);
    imc = cr2res_create_hdrl(nx, ny, c, sc);
    detlin = create_detlin(my_path, ima, imb, imc);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    out_value = detlin(i, a, b, c);
    out_error = deterr(i, a, b, c, si, sa, sb, sc);
    cmp = cr2res_create_hdrl(nx, ny, out_value, out_error);
    cpl_image_add_scalar(hdrl_image_get_error(cmp), 
        sqrt(out_value)/sqrt(CR2RES_GAIN_CHIP1));

    add_read_noise(cmp, dit, chip);

    cpl_test_image_abs(hdrl_image_get_image(out), hdrl_image_get_image(cmp), DBL_EPSILON);
    cpl_test_image_abs(hdrl_image_get_error(out), hdrl_image_get_error(cmp), 1e-6);
    hdrl_image_delete(out);
    hdrl_image_delete(cmp);
    cpl_frame_delete(detlin);
    hdrl_image_delete(ima);
    hdrl_image_delete(imb);
    hdrl_image_delete(imc);

    // Case 11: No Filename set
    detlin = cpl_frame_new();
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_null(out);
    cpl_frame_delete(detlin);


    // Case 12: DataFile is empty, i.e. only header
    detlin = cpl_frame_new();
    cpl_frame_set_filename(detlin, my_path2);
    out = cr2res_calib_image(in, chip, 0, 0, 0, 0, NULL, NULL, NULL, detlin, dit, 1);
    cpl_test_null(out);
    cpl_frame_delete(detlin);

    cpl_free(my_path);
    cpl_free(my_path2);

    hdrl_image_delete(in);

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Run the Unit tests
 */
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_error_reset();
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);

    create_empty_fits();

    test_cr2res_calib_image();
    test_cr2res_calib_cosmic();
    test_cr2res_calib_flat();
    test_cr2res_calib_dark();
    test_cr2res_calib_bpm();
    test_cr2res_calib_detlin();

    return cpl_test_end(0);
}
