/* $Id: efosc_qc.c,v 1.2 2008-12-27 14:48:49 cizzo Exp $
 *
 * This file is part of the EFOSC2 Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cizzo $
 * $Date: 2008-12-27 14:48:49 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <math.h>

#include <cpl.h>
#include <efosc_utils.h>
#include <efosc_paf.h>
#include <efosc_dfs.h>
#include <efosc_qc.h>

#define DICT_LINE_LENGTH    (80)
#define MAX_PAF_NAME_LENGTH (80)
#define PAF_ROOT_NAME       "qc"

/**
 * @defgroup efoscqc Quality Control Utilities
 *
 * The module collects utility functions for quality control operations.
 */

/**@{*/
const char *const efosc_qc_dic_version = "2.0";


/**
 * @brief
 *   Write an integer value to a header.
 *
 * @return @c CPL_ERROR_NONE on success
 *
 * @param header     Product header
 * @param value      Value to write.
 * @param name       QC1 PAF entry name.
 * @param unit       Optional unit to be associated to value.
 * @param comment    Comment to be associated to value.
 * @param instrument Instrument name
 *
 *   This function writes the header entries directly to the header
 *   of the FITS file written to disk, using the cpl calls.
 *   From @em name, the name of the QC keyword that
 *   should be written to header is derived prepending the string "ESO "
 *   and replacing all '.' with a blank (e.g., "QC.BIAS.MASTER.MEAN"
 *   becomes "ESO QC BIAS MASTER MEAN"). Finally, the new keyword
 *   is written to the header.
 */

cpl_error_code efosc_qc_write_qc_double(cpl_propertylist *header, double value, 
                                      const char *name,
                                      const char *comment)
{

  const char func[] = "efosc_qc_write_qc_double";

  char *header_name;
  int   i;

  header_name = cpl_malloc((strlen(name) + 6) * sizeof(char *));

  strcpy(header_name, "ESO ");
  strcat(header_name, name);

  for (i = 0; header_name[i] != '\0'; i++)
    if (header_name[i] == '.')
      header_name[i] = ' ';

  if (cpl_propertylist_update_double(header, header_name, value)) {
      cpl_free(header_name);
      cpl_error_set_where(func);
      return cpl_error_get_code();
  }

  cpl_propertylist_set_comment(header, header_name, comment);

  cpl_free(header_name);

  return CPL_ERROR_NONE;

}


cpl_error_code efosc_qc_write_qc_int(cpl_propertylist *header, int value,
                                   const char *name,
                                   const char *comment)
{

  const char func[] = "efosc_qc_write_qc_int";

  char *header_name;
  int   i;


  header_name = cpl_malloc((strlen(name) + 6) * sizeof(char *));

  strcpy(header_name, "ESO ");
  strcat(header_name, name);

  for (i = 0; header_name[i] != '\0'; i++)
    if (header_name[i] == '.')
      header_name[i] = ' ';

  if (cpl_propertylist_update_int(header, header_name, value)) {
      cpl_free(header_name);
      cpl_error_set_where(func);
      return cpl_error_get_code();
  }

  cpl_propertylist_set_comment(header, header_name, comment);

  cpl_free(header_name);

  return CPL_ERROR_NONE;

}

/**@}*/
