/* $Id: efosc_bias_impl.c,v 1.3 2012-09-24 16:17:50 cgarcia Exp $
 *
 * This file is part of the EFOSC2 Data Reduction Pipeline
 * Copyright (C) 2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2012-09-24 16:17:50 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <efosc_bias_impl.h>

#include <efosc_stack.h>
#include <efosc_qc.h>
#include <efosc_tools.h>
#include <efosc_dfs.h>
#include <efosc_utils.h>
#include <moses.h>

#include <cpl.h>

#include <string.h>
#include <math.h>

/**
 * @addtogroup efosc_bias
 */

/**@{*/

const char *const efosc_bias_name = "efosc_bias";
const char *const efosc_bias_description_short = "Compute the master bias frame";
const char *const efosc_bias_author = "Jonas M. Larsen, Carlo Izzo";
const char *const efosc_bias_email = PACKAGE_BUGREPORT;
const char *const efosc_bias_description =
"This recipe is used to combine input raw BIAS frames into a master bias\n"
"frame. The overscan regions, if present, are removed from the result.\n\n"
"Input files:\n\n"
"  DO category:               Type:       Explanation:         Required:\n"
"  BIAS                       Raw         Bias frame              Y\n\n"
"Output files:\n\n"
"  DO category:               Data type:  Explanation:\n"
"  MASTER_BIAS                FITS image  Master bias frame\n\n";


static void
write_qc(cpl_propertylist *qc,
         const efosc_setting *setting,
         const efosc_image_list *bias,
         const efosc_image *master_bias,
         const stack_method *sm);
/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 */
void efosc_bias_define_parameters(cpl_parameterlist *parameters)
{
    char *context = cpl_sprintf("efosc.%s", efosc_bias_name);
    char *name    = cpl_sprintf("efosc.%s.qc", efosc_bias_name);
    cpl_parameter *p;
    
    efosc_stack_define_parameters(parameters, context, "minmax");

    /*
     * Computation of QC1 parameters
     */

    p = cpl_parameter_new_value(name,
                                CPL_TYPE_BOOL,
                                "Compute QC1 parameters",
                                context,
                                TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "qc");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);

    cpl_free((void *)context);
    cpl_free((void *)name);

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(bias_frames); \
    efosc_stack_method_delete(&sm); \
    cpl_free((void *)context); \
    efosc_image_list_delete_const(&bias, efosc_image_delete); \
    efosc_image_delete(&master_bias); \
    efosc_setting_delete(&setting); \
    cpl_propertylist_delete(qc); \
} while (0)
/**
 * @brief    Do the processing
 *
 * @param    frames         input frames
 * @param    parameters     recipe parameters
 *
 * @return   0 if everything is ok
 */

void efosc_bias(cpl_frameset *frames, const cpl_parameterlist *parameters)
{
    /* Raw */
    cpl_frameset *bias_frames      = NULL;
    const efosc_image_list *bias    = NULL;

    /* Product */
    efosc_image *master_bias = NULL;
    cpl_propertylist *qc = cpl_propertylist_new();

    /* Parameters */
    stack_method *sm    = NULL;

    /* Other */
    efosc_setting *setting = NULL;
    char *context = cpl_sprintf("efosc.%s", efosc_bias_name);

    /* Get parameters */
    sm = efosc_stack_method_new(parameters, context);
    assure( !cpl_error_get_code(), return, "Could not get stacking method");
    
    /* Find raw */
    bias_frames = efosc_frameset_extract(frames, BIAS);
    assure( cpl_frameset_get_size(bias_frames) > 0, return, 
            "No %s provided", BIAS);

    /* Get instrument setting */
    setting = efosc_setting_new(cpl_frameset_get_position(bias_frames, 0));
    assure( !cpl_error_get_code(), return, "Could not get instrument setting" );

    /* Load bias */
    bias = efosc_image_load_list_const(bias_frames, NULL, setting, NULL);
    assure( !cpl_error_get_code(), return, "Could not load bias images");

    /* Stack */
    master_bias = efosc_stack_const(bias, sm);
    assure( !cpl_error_get_code(), return, "Bias stacking failed");
    
    /* QC */
    if (dfs_get_parameter_bool_const(parameters, "efosc.efosc_bias.qc")) {
        write_qc(qc, setting,
                 bias, master_bias, sm);
    }

    /* Save product */
    efosc_dfs_save_image(frames, master_bias, MASTER_BIAS,
                        qc, parameters, efosc_bias_name, 
                        cpl_frameset_get_position(bias_frames, 0));
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            MASTER_BIAS);
    
    cleanup;
    return;
}


#undef cleanup
#define cleanup \
do { \
    efosc_image_delete(&image); \
} while (0)


static void
write_qc(cpl_propertylist *qc,
         const efosc_setting *setting,
         const efosc_image_list *bias,
         const efosc_image *master_bias,
         const stack_method *sm)
{
    const efosc_image *first_raw  = efosc_image_list_first_const(bias);
    const efosc_image *second_raw = efosc_image_list_next_const(bias);
    efosc_image *image = NULL;

    efosc_qc_write_qc_double(qc,
                            efosc_image_get_median(first_raw, NULL),
                            "QC.BIAS.LEVEL",
                            "Bias level");
    double ron;
    double fpn;
    if (second_raw != NULL) {

        image = efosc_image_duplicate(first_raw);
        efosc_image_subtract(image, second_raw);

        ron = efosc_image_get_stdev_robust(image, 50, NULL) / sqrt(2.0);

        fpn = efosc_fixed_pattern_noise_bias(first_raw,
                                            second_raw,
                                            ron);
/*
        fpn = efosc_fixed_pattern_noise(first_raw,
                                       1.0,
                                       ron);
*/
        assure( !cpl_error_get_code(), return, 
                "Could not compute fixed pattern noise" );
    }
    else {
        cpl_msg_warning(cpl_func,
                        "Only %d bias frame(s) provided, "
                        "cannot compute readout noise", 
                        efosc_image_list_size(bias));
        ron = -1;
        fpn = -1;
    }

    efosc_qc_write_qc_double(qc,
                            ron,
                            "QC.RON",
                            "Readout noise");

    efosc_qc_write_qc_double(qc,
                            fpn,
                            "QC.BIAS.FPN",
                            "Bias fixed pattern noise");

    double structure = efosc_image_get_stdev_robust(first_raw, 50, NULL);
    if (structure*structure >= ron*ron + fpn*fpn) {
        structure = sqrt(structure*structure - ron*ron - fpn*fpn);
    }
    else {
        cpl_msg_warning(cpl_func,
                        "Overall bias standard deviation (%f ADU) is less "
                        "than combined readout and fixed pattern noise "
                        "(%f ADU), setting structure to zero",
                        structure , sqrt(ron*ron + fpn*fpn));
        structure = 0;
    }
    
    
    efosc_qc_write_qc_double(qc,
                            structure,
                            "QC.BIAS.STRUCT",
                            "Bias structure");

    /* Master bias QC */

    efosc_qc_write_qc_double(qc,
                            efosc_image_get_median(master_bias, NULL),
                            "QC.MBIAS.LEVEL",
                            "Master bias level");

    double ron_expect = -1;
    if (ron > 0) {

        int N = efosc_image_list_size(bias);

        /*
          When median stacking and N >= 3, we need to
          take into account the fact that the median is more noisy than
          the mean.
        */

        if (sm->method == MEDIAN) {
            ron_expect = efosc_utils_median_corr(N) * ron / sqrt(N);
        }
        else {
            ron_expect = ron / sqrt(N);
        }
    }
    else cpl_msg_warning(cpl_func,
                         "Cannot compute expected master bias readout noise");
    
    efosc_qc_write_qc_double(qc,
                            ron_expect,
                            "QC.MBIAS.RONEXP",
                            "Expected master bias readout noise");
    
    double mbias_noise = -1;
    if (ron_expect > 0) {
        mbias_noise =
            efosc_image_get_stdev_robust(master_bias, 3*ron_expect, NULL);
    }
    else {
        mbias_noise = -1;
    }
    
    efosc_qc_write_qc_double(qc,
                            mbias_noise,
                            "QC.MBIAS.NOISE",
                            "Master bias readout noise");

    efosc_qc_write_qc_double(qc,
                            mbias_noise / ron_expect,
                            "QC.MBIAS.NRATIO",
                            "Master bias observed/expected noise");
    
    double mbias_struct = efosc_image_get_stdev(master_bias, NULL);

    if (mbias_struct * mbias_struct > mbias_noise * mbias_noise) {

        cpl_msg_debug(cpl_func, "Overall standard deviation is %f ADU",
                      mbias_struct);

        mbias_struct = sqrt(mbias_struct * mbias_struct - 
                            mbias_noise * mbias_noise);
    }
    else {
        cpl_msg_warning(cpl_func,
                        "Master bias overall standard deviation (%f ADU) is "
                        "greater than master bias noise (%f ADU), "
                        "cannot compute master bias structure",
                        mbias_struct, mbias_noise);
        mbias_struct = -1;
    }

    efosc_qc_write_qc_double(qc,
                            mbias_struct,
                            "QC.MBIAS.STRUCT",
                            "Structure of master bias");
    
    cleanup;
    return;
}
/**@}*/
