/* $Id: efosc_image.h,v 1.2 2011-02-07 12:47:47 cizzo Exp $
 *
 * This file is part of the EFOSC2 Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cizzo $
 * $Date: 2011-02-07 12:47:47 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifndef EFOSC_IMAGE_H
#define EFOSC_IMAGE_H

#include <efosc_setting.h>
#include <cpl.h>

typedef struct _efosc_image efosc_image;

extern const cpl_type EFOSC_IMAGE_TYPE;

/* Container */
#undef LIST_ELEM
#define LIST_ELEM efosc_image
#include <list.h>

/* Constructors */
efosc_image *efosc_image_new(cpl_image *data, cpl_image *weights);

efosc_image *efosc_image_duplicate(const efosc_image *image);

/* Desctructors */
void efosc_image_delete(efosc_image **image);
void efosc_image_delete_const(const efosc_image **image);


/* I/O */
efosc_image *efosc_image_load(const cpl_frame *frame, const efosc_image *bias,
			    const efosc_setting *setting,
			    double *saturated);


efosc_image_list *efosc_image_load_list(const cpl_frameset *frames, 
				      const efosc_image *bias,
				      const efosc_setting *setting,
				      double *saturated);

const efosc_image_list *
efosc_image_load_list_const(const cpl_frameset *frames, const efosc_image *bias,
                           const efosc_setting *setting,
                           double *saturated);

void
efosc_image_save(const efosc_image *image, const cpl_propertylist *header,
		const char *filename);

void
efosc_image_save_sex(const efosc_image *image, const cpl_propertylist *header,
                    const char *filename_dat,
                    const char *filename_var,
                    int radius);

/* Other */
int efosc_image_get_size_x(const efosc_image *image);
int efosc_image_get_size_y(const efosc_image *image);
const float *efosc_image_get_data_const(const efosc_image *image);

void efosc_image_draw(efosc_image *image, int type,
		     double x, double y,
		     int radius, double color);

void efosc_image_crop(efosc_image *image,
		     int xlo, int ylo,
		     int xhi, int yhi);

/* Arithmetic */
void efosc_image_subtract(efosc_image *left, const efosc_image *right);
void efosc_image_multiply(efosc_image *left, const efosc_image *right);
void efosc_image_multiply_noerr(efosc_image *left, const cpl_image *right);
void efosc_image_divide(efosc_image *left, const efosc_image *right);
void efosc_image_divide_noerr(efosc_image *left, cpl_image *right);
void efosc_image_abs(efosc_image *image);
void efosc_image_square(efosc_image *image);

void efosc_image_exponential(efosc_image *image, double b, double db);
void efosc_image_multiply_scalar(efosc_image *image, double s, double ds);
void efosc_image_divide_scalar(efosc_image *image, double s, double ds);
void efosc_image_subtract_scalar(efosc_image *image, double s, double ds);

efosc_image *efosc_image_collapse_create(const efosc_image_list *images);
efosc_image *efosc_image_collapse_median_create(const efosc_image_list *images);
efosc_image *efosc_image_collapse_minmax_create(const efosc_image_list *images, 
                                                int low, int high);
efosc_image *efosc_image_collapse_ksigma_create(const efosc_image_list *images,
                                                int low, int high, int iter);

cpl_image *
efosc_image_filter_median_create(const efosc_image *image, 
                                int xradius,
                                int yradius,
                                int xstart, 
                                int ystart,
                                int xend,
                                int yend,
                                int xstep,
                                int ystep,
                                bool use_data);

cpl_image *
efosc_image_flat_fit_create(efosc_image *image,
                           int step,
                           int degree,
                           float level);
cpl_image *
efosc_image_filter_max_create(const efosc_image *image,
                             int xradius,
                             int yradius,
                             bool use_data);

/* Statistics */
double efosc_image_get_mean(const efosc_image *image, double *dmean);
double efosc_image_get_median(const efosc_image *image, double *dmedian);

double efosc_image_get_stdev(const efosc_image *image, double *dstdev);
double efosc_image_get_stdev_robust(const efosc_image *image, 
				   double cut,
				   double *dstdev);
double efosc_image_get_error_mean(const efosc_image *image, double *dmean);

double efosc_image_get_min(const efosc_image *image);
double efosc_image_get_max(const efosc_image *image);

#endif
