/* $Id: efosc_img_sky_flat_impl.c,v 1.1 2009-07-09 12:27:18 cizzo Exp $
 *
 * This file is part of the EFOSC Library
 * Copyright (C) 2009 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cizzo $
 * $Date: 2009-07-09 12:27:18 $
 * $Revision: 1.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <efosc_img_sky_flat_impl.h>

#include <efosc_stack.h>
#include <efosc_dfs.h>
#include <efosc_qc.h>
#include <efosc_utils.h>

#include <cpl.h>

/**
 * @addtogroup efosc_img_sky_flat
 */

/**@{*/

const char *const efosc_img_sky_flat_name = "efosc_img_sky_flat";
const char *const efosc_img_sky_flat_description_short = "Compute master img_sky_flat frame";
const char *const efosc_img_sky_flat_author = "Jonas M. Larsen";
const char *const efosc_img_sky_flat_email = PACKAGE_BUGREPORT;
const char *const efosc_img_sky_flat_description = 
"Input files:\n"
"  DO category:               Type:       Explanation:             Number:\n"
"  SKY_FLAT_IMG               Raw         Jittered sky flat fields    1+\n"
"  MASTER_BIAS                FITS image  Master bias                 1\n"
"\n"
"Output files:\n"
"  DO category:               Data type:  Explanation:\n"
"  MASTER_SKY_FLAT_IMG        FITS image  Master sky flat field\n";
/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 */

void efosc_img_sky_flat_define_parameters(cpl_parameterlist *parameters)
{
    char *context = cpl_sprintf("efosc.%s", efosc_img_sky_flat_name);
    
    efosc_stack_define_parameters(parameters, context, "median");

    cpl_free((void *)context);

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_frameset_delete(sflat_frames); \
    cpl_frameset_delete(master_bias_frame); \
    efosc_stack_method_delete(&sm); \
    cpl_free((void *)context); \
    efosc_image_delete_const(&master_bias); \
    efosc_image_delete(&master_sky_flat); \
    efosc_image_list_delete(&sflats, efosc_image_delete); \
    cpl_propertylist_delete(qc); \
    efosc_setting_delete(&setting); \
} while (0)

/**
 * @brief    Do the processing
 *
 * @param    frames         input frames
 * @param    parameters     The parameters list
 *
 * @return   0 if everything is ok
 */
void efosc_img_sky_flat(cpl_frameset *frames, const cpl_parameterlist *parameters)
{
    /* Raw */
    cpl_frameset *sflat_frames      = NULL;
    efosc_image_list *sflats         = NULL;

    /* Calibration */
    cpl_frameset *master_bias_frame = NULL;
    const efosc_image *master_bias   = NULL; 

    /* Products */
    efosc_image *master_sky_flat = NULL;
    cpl_propertylist *qc = cpl_propertylist_new();
    double saturation;

    /* Parameters */
    stack_method *sm = NULL;

    /* Other */
    char *context = cpl_sprintf("efosc.%s", efosc_img_sky_flat_name);
    efosc_setting *setting = NULL;

    /* QCs about sky level */
    double skylevel, skylevmax, skylevmin;
    int    first_frame;

    /* Get parameters */
    sm = efosc_stack_method_new(parameters, context);
    assure( !cpl_error_get_code(), return, "Could not get stacking method");

    /* Eliminate irrelevant RAW frames (header would be inherited from them) */
    cpl_frameset_erase(frames, "TEST");
    
    /* Find raw */
    sflat_frames = efosc_frameset_extract(frames, SKY_FLAT_IMG);
    assure( cpl_frameset_get_size(sflat_frames) > 0, return, 
            "No %s provided", SKY_FLAT_IMG);

    /* Find calibration */
    master_bias_frame = efosc_frameset_extract(frames, MASTER_BIAS);
    assure( cpl_frameset_get_size(master_bias_frame) == 1, return, 
            "One %s required. %"CPL_SIZE_FORMAT" found", 
            MASTER_BIAS, cpl_frameset_get_size(master_bias_frame));

    /* Get setting */
    setting = efosc_setting_new(cpl_frameset_get_position_const(sflat_frames, 0));
    assure( !cpl_error_get_code(), return, "Could not get instrument setting" );
    
    master_bias = efosc_image_load(cpl_frameset_get_position(master_bias_frame, 0), 
                                  NULL, setting, NULL);
    assure( !cpl_error_get_code(), return, 
            "Could not load master bias");

    /* Load raw frames, subtract bias */
    sflats = efosc_image_load_list(sflat_frames, master_bias, setting, &saturation);
    assure( !cpl_error_get_code(), return, "Could not load sky flat images");

    /* Normalize to median = 1.
     *
     * (The input sky flats generally have different normalization because of
     *  the time dependent sky level. We must bring the input flats
     *  to the same normalization before stacking.)
     */
    {
        efosc_image *image;

        skylevel = skylevmax = skylevmin = 0.0;

        first_frame = 1;
        
        for (image = efosc_image_list_first(sflats);
             image != NULL;
             image = efosc_image_list_next(sflats)) {

            skylevel = efosc_image_get_median(image, NULL);

            efosc_image_divide_scalar(image, skylevel, -1.0);
            
            assure( !cpl_error_get_code(), return, 
                    "Raw sky flat normalization failed");

            if (first_frame) {
                first_frame = 0;
                skylevmax = skylevmin = skylevel;
            }
            else {
                if (skylevmax < skylevel)
                    skylevmax = skylevel;
                if (skylevmin > skylevel)
                    skylevmin = skylevel;
            }
        }
    }


    /* Stack */
    master_sky_flat = efosc_stack(sflats, sm);
    assure( !cpl_error_get_code(), return, "Sky flat stacking failed");

    /* QC */
    efosc_qc_write_qc_double(qc,
                            saturation,
                            "QC.OVEREXPO",
                            "Percentage of overexposed pixels");

    efosc_qc_write_qc_double(qc,
                            skylevmin,
                            "QC.SKYFLAT.FLUX.MIN",
                            "Median level of dimmest input flat");

    efosc_qc_write_qc_double(qc,
                            skylevmax,
                            "QC.SKYFLAT.FLUX.MAX",
                            "Median level of brightest input flat");
    
    /* Save product */
    efosc_dfs_save_image(frames, master_sky_flat, MASTER_SKY_FLAT_IMG,
                        qc, parameters, efosc_img_sky_flat_name, 
                        cpl_frameset_get_position(sflat_frames, 0));
    assure( !cpl_error_get_code(), return, "Saving %s failed",
            MASTER_SKY_FLAT_IMG);
    
    cleanup;
    return;
}


/**@}*/
