/* $Id: efosc_stack.c,v 1.2 2011-02-07 12:48:55 cizzo Exp $
 *
 * This file is part of the EFOSC2 Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cizzo $
 * $Date: 2011-02-07 12:48:55 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <efosc_stack.h>

#include <efosc_dfs.h>
#include <efosc_utils.h>

#include <cpl.h>

#include <string.h>
#include <stdbool.h>

/**
 * @defgroup efosc_stack Image stacking
 */

/**@{*/

/**
 * @brief    Define recipe parameters
 * @param    parameters     parameter list to fill
 * @param    context        parameters context
 * @param    default_method default stack method
 */

void efosc_stack_define_parameters(cpl_parameterlist *parameters, 
                                  const char *context,
                                  const char *default_method)
{
    cpl_parameter *p;
    const char *full_name = NULL;
    const char *name;

    name = "stack_method";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_enum(full_name,
                               CPL_TYPE_STRING,
                               "Frames combination method",
                               context,
                               default_method, 4,
                               "average", "median", "minmax", "ksigma");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);


    /* minmax */
    name = "minrejection";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_INT,
                                "Number of lowest values to be rejected",
                                context,
                                1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    name = "maxrejection";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_INT,
                                "Number of highest values to be rejected",
                                context,
                                1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    /* ksigma */
    name = "klow";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "Low threshold in ksigma method",
                                context,
                                3.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    name = "khigh";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_DOUBLE,
                                "High threshold in ksigma method",
                                context,
                                3.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    name = "kiter";
    full_name = cpl_sprintf("%s.%s", context, name);
    p = cpl_parameter_new_value(full_name,
                                CPL_TYPE_INT,
                                "Max number of iterations in ksigma method",
                                context,
                                999);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(parameters, p);
    cpl_free((void *)full_name);

    return;
}

#undef cleanup
#define cleanup \
do { \
    cpl_free((void *)name); \
} while (0)
/**
 * @brief    Get stack method from parameter list
 * @param    parameters     recipe parameter list
 * @param    context        read stack method from this context
 * @return   newly allocated opaque stack method
 *
 * The parameter list should have been previously created using
 * efosc_stack_define_parameters()
 */
stack_method *
efosc_stack_method_new(const cpl_parameterlist *parameters, const char *context)
{
    stack_method *sm = cpl_malloc(sizeof(stack_method));
    const char *name = NULL;

    cpl_msg_info(cpl_func, "Stack method parameters:");

    cpl_msg_indent_more();
    name = cpl_sprintf("%s.%s", context, "stack_method");
    sm->method_name = dfs_get_parameter_string_const(parameters, 
                                            name);
    cpl_free((void *)name); name = NULL;
    cpl_msg_indent_less();

    assure( !cpl_error_get_code(), return NULL, NULL );
    assure( sm->method_name != NULL, return NULL, NULL );

    if (strcmp(sm->method_name, "average") == 0) {
        sm->method = AVERAGE;
    }
    else if (strcmp(sm->method_name, "median") == 0) {
        sm->method = MEDIAN;
    }
    else if (strcmp(sm->method_name, "minmax") == 0) {
/*
        assure( false, return NULL, "Unsupported stack method %s", sm->method_name);
*/
        sm->method = MINMAX;
    }
    else if (strcmp(sm->method_name, "ksigma") == 0) {
        sm->method = KSIGMA;
    }
    else {
        assure( false, return NULL, "Unknown stack method '%s'", sm->method_name);
    }

    switch (sm->method) {
    case AVERAGE: break;
    case MEDIAN: break;
    case MINMAX:

        cpl_msg_indent_more();
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "minrejection");
        sm->pars.minmax.min_reject = dfs_get_parameter_int_const(parameters, 
                                                                 name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
        cpl_msg_indent_less();
        assure( !cpl_error_get_code(), return NULL, NULL );
        
        cpl_msg_indent_more();
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "maxrejection");
        sm->pars.minmax.max_reject = dfs_get_parameter_int_const(parameters, 
                                                                 name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
        cpl_msg_indent_less();
        assure( !cpl_error_get_code(), return NULL, NULL );

        break;
    case KSIGMA:
        cpl_msg_indent_more();
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "klow");
        sm->pars.ksigma.klow = dfs_get_parameter_double_const(parameters, 
                                                              name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
        cpl_msg_indent_less();
        assure( !cpl_error_get_code(), return NULL, NULL );
        
        cpl_msg_indent_more();
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "khigh");
        sm->pars.ksigma.khigh = dfs_get_parameter_double_const(parameters, 
                                                               name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
        cpl_msg_indent_less();
        assure( !cpl_error_get_code(), return NULL, NULL );

        cpl_msg_indent_more();
        cpl_msg_indent_more();
        name = cpl_sprintf("%s.%s", context, "kiter");
        sm->pars.ksigma.kiter = dfs_get_parameter_int_const(parameters, 
                                                            name);
        cpl_free((void *)name); name = NULL;
        cpl_msg_indent_less();
        cpl_msg_indent_less();
        assure( !cpl_error_get_code(), return NULL, NULL );
        
        break;
    default:
        passure( false, return NULL );
        break;
    } /* switch sm->method */

    cleanup;
    return sm;
}

/**
 * @brief    Destructor
 * @param    sm            object to delete
 */
void
efosc_stack_method_delete(stack_method **sm)
{
    if (sm && *sm) {
        cpl_free(*sm); *sm = NULL;
    }
    return;
}

#undef cleanup
#define cleanup
/**
 * @brief   Stack method as string
 * @param   sm            stack method
 * @return  textual representation of the provided stack method
 */
static const char *efosc_stack_method_get_string(const stack_method *sm)
{
    assure( sm != NULL, return "Null", NULL );

    return sm->method_name;
}

#undef cleanup
#define cleanup \
do { \
} while (0)
/**
 * @brief    Stack images
 * @param    images        list of images to stack
 * @param    sm            stacking method
 * @return   master images stacked using the specified method
 */
efosc_image *
efosc_stack_const(const efosc_image_list *images, const stack_method *sm)
{
    efosc_image *master = NULL;

    assure( images != NULL, return master, NULL );
    assure( efosc_image_list_size(images) > 0, return master, 
           "No images to collapse");
    
    cpl_msg_info(cpl_func, "Stacking images (method = %s)",
                 efosc_stack_method_get_string(sm)); 

    switch (sm->method) {
    case AVERAGE: 
        master = efosc_image_collapse_create(images);
        break;
    case MEDIAN: 
        master = efosc_image_collapse_median_create(images);
        break;
    case MINMAX:
        master = efosc_image_collapse_minmax_create(images, 
                                  sm->pars.minmax.min_reject,
                                  sm->pars.minmax.max_reject);
        break;
    case KSIGMA: 
        master = efosc_image_collapse_ksigma_create(images,
                                  sm->pars.ksigma.klow,
                                  sm->pars.ksigma.khigh,
                                  sm->pars.ksigma.kiter);
        break;
    default:
        assure( false, return NULL, "Unknown stack method '%s' (%d)",
                efosc_stack_method_get_string(sm), sm->method);
        break;
    }    

    return master;
}

/**
 * @brief    Same as efosc_stack_const()
 */
efosc_image *
efosc_stack(efosc_image_list *images, const stack_method *sm)
{
    return efosc_stack_const((const efosc_image_list *)images, sm);
}


/**@}*/
