/* $Id: efosc_star.h,v 1.2 2009-07-10 12:28:40 cizzo Exp $
 *
 * This file is part of the EFOSC Library
 * Copyright (C) 2009 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: cizzo $
 * $Date: 2009-07-10 12:28:40 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifndef EFOSC_STAR_H
#define EFOSC_STAR_H

#include <efosc_std_star.h>

#include <cpl.h>

typedef struct _efosc_star
{
    efosc_point *pixel;
    double semi_major;
    double semi_minor;
    double fwhm;
    double stellarity_index;    /* in [0; 1] */
    double orientation;         /* radians */

    double magnitude;           /* instrumental magnitude (integrated ADU) */
    double dmagnitude;          /* 1 sigma error bar */

    double magnitude_corr;      /* magnitude corrected for gain, atmospheric
                                   extinction, exposure time */
    double dmagnitude_corr;

    double weight;              /* for the user of this module to define the
                                   meaning this */

    const efosc_std_star *id;    /* copy of identified, or NULL */

} efosc_star;

#undef LIST_ELEM
#define LIST_ELEM efosc_star
#include <list.h>

efosc_star *efosc_star_new(                   double x, double y,
                                                        double fwhm,
                                            double smajor, double sminor,
                                                        double orientation,
                                                        double m, double dm,
                                            double si);

efosc_star *efosc_star_new_from_table(        const cpl_table *tab,
                                            unsigned int    row,
                                            const char      *x_col,
                                            const char      *y_col,
                                            const char      *fwhm_col,
                                            const char      *smaj_col,
                                            const char      *smin_col,
                                            const char      *theta_col,
                                            const char      *mag_col,
                                            const char      *dmag_col,
                                            const char      *stlndx_col);

bool efosc_star_check_values(                const efosc_star *star);

void efosc_star_delete(efosc_star **star);
void efosc_star_delete_but_standard(efosc_star **star);

efosc_star *
efosc_star_duplicate(const efosc_star *star);

bool
efosc_star_equal(const efosc_star *s,
                const efosc_star *t);

double efosc_star_distsq(const efosc_star *s, const efosc_star *t);
double efosc_star_extension(const efosc_star *s, void *data);
double efosc_star_stellarity(const efosc_star *s, void *data);
double efosc_star_ellipticity(const efosc_star *s, void *data);

bool
efosc_star_brighter_than(const efosc_star *s1,
                        const efosc_star *s2,
                        void *data);

void
efosc_star_print(cpl_msg_severity level, const efosc_star *s);

void
efosc_star_print_list(cpl_msg_severity level, const efosc_star_list *sl);

double
efosc_star_get_x(const efosc_star *s, void *data);

double
efosc_star_get_y(const efosc_star *s, void *data);

double
efosc_star_get_zeropoint(const efosc_star *s, void *data);

double
efosc_star_get_zeropoint_err(const efosc_star *s, void *data);

bool
efosc_star_is_identified(const efosc_star *s, void *data);

#endif
