/* $Id: efosc_calib.c,v 1.10 2011-02-10 10:56:53 cizzo Exp $
 *
 * This file is part of the EFOSC Data Reduction Pipeline
 * Copyright (C) 2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cizzo $
 * $Date: 2011-02-10 10:56:53 $
 * $Revision: 1.10 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <string.h>
#include <cpl.h>
#include <moses.h>
#include <efosc_stack.h>
#include <efosc_dfs.h>
#include <efosc_qc.h>

static int efosc_calib_create(cpl_plugin *);
static int efosc_calib_exec(cpl_plugin *);
static int efosc_calib_destroy(cpl_plugin *);
static int efosc_calib(cpl_parameterlist *, cpl_frameset *);

static char efosc_calib_description[] =
"This recipe is used to identify reference lines on MOS arc lamp\n"
"exposures, and trace the spectral edges on the corresponding flat field\n"
"exposures. This information is used to determine the spectral extraction\n"
"mask to be applied in the scientific data reduction, performed with the\n"
"recipe efosc_science. The input arc lamp and flat field exposures are\n"
"assumed to be obtained quasi-simultaneously, so that they would be\n" 
"described by exactly the same instrument distortions.\n"
"A line catalog must be specified, containing the wavelengths of the\n"
"reference arc lamp lines used for the wavelength calibration. A grism\n"
"table (typically depending on the instrument mode, and in particular on\n"
"the grism used) may also be specified: this table contains a default\n"
"recipe parameter setting to control the way spectra are extracted for\n"
"a specific instrument mode, as it is used for automatic run of the\n"
"pipeline on Paranal and in Garching. If this table is specified, it\n"
"will modify the default recipe parameter setting, with the exception of\n"
"those parameters which have been explicitly modified on the command line.\n"
"If a grism table is not specified, the input recipe parameters values\n"
"will always be read from the command line, or from an esorex configuration\n"
"file if present, or from their generic default values (that are rarely\n"
"meaningful). Finally a master bias frame must be input to this recipe.\n" 
"In the table below the CURV_COEFF_MOS, CURV_TRACES_MOS, SPATIAL_MAP_MOS\n"
"REDUCED_LAMP_MOS, SPECTRA_DETECTION_MOS, SPATIAL_MAP_MOS, and\n" 
"SLIT_LOCATION_MOS, are never created in case of long-slit-like data.\n" 
"The products SPECTRA_DETECTION_MOS, SPATIAL_MAP_MOS, and DISP_RESIDUALS_MOS,\n" 
"are just created if the --check parameter is set to true. The product\n"
"GLOBAL_DISTORTION_TABLE is just created if more than 6 separate spectra\n"
"are identified in the CCD.\n\n"
"Input files:\n\n"
"  DO category:               Type:       Explanation:         Required:\n"
"  SCREEN_FLAT_MOS            Raw         Flat field exposures    Y\n"
"  LAMP_MOS                   Raw         Arc lamp exposure       Y\n"
"  MASTER_BIAS or BIAS        Calib       Bias frame              Y\n"
"  LINE_CATALOG               Calib       Line catalog            Y\n"
"  GRISM_TABLE                Calib       Grism table             .\n\n"
"Output files:\n\n"
"  DO category:               Data type:  Explanation:\n"
"  MASTER_SCREEN_FLAT_MOS     FITS image  Combined (sum) flat field\n"
"  MASTER_NORM_FLAT_MOS       FITS image  Normalised flat field\n"
"  MAPPED_SCREEN_FLAT_MOS     FITS image  Wavelength calibrated flat field\n"
"  REDUCED_LAMP_MOS           FITS image  Wavelength calibrated arc spectrum (Not in LSS)\n"
"  DISP_COEFF_MOS             FITS table  Inverse dispersion coefficients\n"
"  DISP_RESIDUALS_MOS         FITS image  Residuals in wavelength calibration\n"
"  DISP_RESIDUALS_TABLE_MOS   FITS table  Residuals in wavelength calibration\n"
"  DELTA_IMAGE_MOS            FITS image  Offset vs linear wavelength calib\n"
"  WAVELENGTH_MAP_MOS         FITS image  Wavelength for each pixel on CCD\n"
"  SPECTRA_DETECTION_MOS      FITS image  Check for preliminary detection (Not in LSS)\n"
"  SLIT_MAP_MOS               FITS image  Map of central wavelength on CCD\n"
"  CURV_TRACES_MOS            FITS table  Spectral curvature traces (Not in LSS)\n"
"  CURV_COEFF_MOS             FITS table  Spectral curvature coefficients (Not in LSS)\n"
"  SPATIAL_MAP_MOS            FITS image  Spatial position along slit on CCD (Not in LSS)\n"
"  SPECTRAL_RESOLUTION_MOS    FITS table  Resolution at reference arc lines\n"
"  SLIT_LOCATION_MOS          FITS table  Slits on product frames and CCD (Not in LSS)\n"
"  GLOBAL_DISTORTION_TABLE    FITS table  Global distortions table\n\n";

#define efosc_calib_exit(message)             \
{                                             \
if (message) cpl_msg_error(recipe, message);  \
cpl_free(instrume);                           \
cpl_free(pipefile);                           \
cpl_free(fiterror);                           \
cpl_free(fitlines);                           \
cpl_image_delete(bias);                       \
cpl_image_delete(master_bias);                \
cpl_image_delete(coordinate);                 \
cpl_image_delete(checkwave);                  \
cpl_image_delete(flat);                       \
cpl_image_delete(master_flat);                \
cpl_image_delete(added_flat);                 \
cpl_image_delete(norm_flat);                  \
cpl_image_delete(rainbow);                    \
cpl_image_delete(rectified);                  \
cpl_image_delete(residual);                   \
cpl_image_delete(smo_flat);                   \
cpl_image_delete(spatial);                    \
cpl_image_delete(spectra);                    \
cpl_image_delete(wavemap);                    \
cpl_image_delete(delta);                      \
cpl_image_delete(rect_flat);                  \
cpl_image_delete(mapped_flat);                \
cpl_mask_delete(refmask);                     \
cpl_propertylist_delete(header);              \
cpl_propertylist_delete(save_header);         \
cpl_propertylist_delete(qclist);              \
cpl_table_delete(grism_table);                \
cpl_table_delete(idscoeff);                   \
cpl_table_delete(restable);                   \
cpl_table_delete(maskslits);                  \
cpl_table_delete(overscans);                  \
cpl_table_delete(traces);                     \
cpl_table_delete(polytraces);                 \
cpl_table_delete(slits);                      \
cpl_table_delete(restab);                     \
cpl_table_delete(global);                     \
cpl_table_delete(wavelengths);                \
cpl_vector_delete(lines);                     \
cpl_frameset_delete(no_bias_set);             \
cpl_msg_indent_less();                        \
return -1;                                    \
}

#define efosc_calib_exit_memcheck(message)       \
{                                               \
if (message) cpl_msg_info(recipe, message);     \
printf("free instrume (%p)\n", instrume);       \
cpl_free(instrume);                             \
printf("free pipefile (%p)\n", pipefile);       \
cpl_free(pipefile);                             \
printf("free fiterror (%p)\n", fiterror);       \
cpl_free(fiterror);                             \
printf("free fitlines (%p)\n", fitlines);       \
cpl_free(fitlines);                             \
printf("free bias (%p)\n", bias);               \
cpl_image_delete(bias);                         \
printf("free master_bias (%p)\n", master_bias); \
cpl_image_delete(master_bias);                  \
printf("free coordinate (%p)\n", coordinate);   \
cpl_image_delete(coordinate);                   \
printf("free checkwave (%p)\n", checkwave);     \
cpl_image_delete(checkwave);                    \
printf("free flat (%p)\n", flat);               \
cpl_image_delete(flat);                         \
printf("free master_flat (%p)\n", master_flat); \
cpl_image_delete(master_flat);                  \
printf("free norm_flat (%p)\n", norm_flat);     \
cpl_image_delete(norm_flat);                    \
printf("free rainbow (%p)\n", rainbow);         \
cpl_image_delete(rainbow);                      \
printf("free rectified (%p)\n", rectified);     \
cpl_image_delete(rectified);                    \
printf("free residual (%p)\n", residual);       \
cpl_image_delete(residual);                     \
printf("free smo_flat (%p)\n", smo_flat);       \
cpl_image_delete(smo_flat);                     \
printf("free spatial (%p)\n", spatial);         \
cpl_image_delete(spatial);                      \
printf("free spectra (%p)\n", spectra);         \
cpl_image_delete(spectra);                      \
printf("free wavemap (%p)\n", wavemap);         \
cpl_image_delete(wavemap);                      \
printf("free delta (%p)\n", delta);             \
cpl_image_delete(delta);                        \
printf("free refmask (%p)\n", refmask);         \
cpl_mask_delete(refmask);                       \
printf("free header (%p)\n", header);           \
cpl_propertylist_delete(header);                \
printf("free save_header (%p)\n", save_header); \
cpl_propertylist_delete(save_header);           \
printf("free qclist (%p)\n", qclist);           \
cpl_propertylist_delete(qclist);                \
printf("free grism_table (%p)\n", grism_table); \
cpl_table_delete(grism_table);                  \
printf("free idscoeff (%p)\n", idscoeff);       \
cpl_table_delete(idscoeff);                     \
printf("free restable (%p)\n", restable);       \
cpl_table_delete(restable);                     \
printf("free overscans (%p)\n", overscans);     \
cpl_table_delete(overscans);                    \
printf("free traces (%p)\n", traces);           \
cpl_table_delete(traces);                       \
printf("free polytraces (%p)\n", polytraces);   \
cpl_table_delete(polytraces);                   \
printf("free slits (%p)\n", slits);             \
cpl_table_delete(slits);                        \
printf("free slits (%p)\n", maskslits);         \
cpl_table_delete(maskslits);                    \
printf("free restab (%p)\n", restab);           \
cpl_table_delete(restab);                       \
printf("free global (%p)\n", global);           \
cpl_table_delete(global);                       \
printf("free wavelengths (%p)\n", wavelengths); \
cpl_table_delete(wavelengths);                  \
printf("free lines (%p)\n", lines);             \
cpl_vector_delete(lines);                       \
printf("free no_bias_set (%p)\n", no_bias_set); \
cpl_frameset_delete(no_bias_set);               \
cpl_msg_indent_less();                          \
return 0;                                       \
}


/**
 * @brief    Build the list of available plugins, for this module. 
 *
 * @param    list    The plugin list
 *
 * @return   0 if everything is ok, -1 otherwise
 *
 * Create the recipe instance and make it available to the application 
 * using the interface. This function is exported.
 */

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
    cpl_plugin *plugin = &recipe->interface;

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    EFOSC_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    "efosc_calib",
                    "Determination of the extraction mask",
                    efosc_calib_description,
                    "Carlo Izzo",
                    "cizzo@eso.org",
    "This file is currently part of the EFOSC2 Instrument Pipeline\n"
    "Copyright (C) 2008-2009 European Southern Observatory\n\n"
    "This program is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public License as published by\n"
    "the Free Software Foundation; either version 2 of the License, or\n"
    "(at your option) any later version.\n\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the\n"
    "GNU General Public License for more details.\n\n"
    "You should have received a copy of the GNU General Public License\n"
    "along with this program; if not, write to the Free Software Foundation,\n"
    "Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA\n",
                    efosc_calib_create,
                    efosc_calib_exec,
                    efosc_calib_destroy);

    cpl_pluginlist_append(list, plugin);
    
    return 0;
}


/**
 * @brief    Setup the recipe options    
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 *
 * Defining the command-line/configuration parameters for the recipe.
 */

static int efosc_calib_create(cpl_plugin *plugin)
{
    cpl_recipe    *recipe;
    cpl_parameter *p;


    /* 
     * Check that the plugin is part of a valid recipe 
     */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    /* 
     * Create the parameters list in the cpl_recipe object 
     */

    recipe->parameters = cpl_parameterlist_new(); 


    /*
     * Dispersion
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.dispersion",
                                CPL_TYPE_DOUBLE,
                                "Expected spectral dispersion (Angstrom/pixel)",
                                "efosc.efosc_calib",
                                0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "dispersion");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Peak detection level
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.peakdetection",
                                CPL_TYPE_DOUBLE,
                                "Initial peak detection threshold (ADU)",
                                "efosc.efosc_calib",
                                0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "peakdetection");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /* 
     * Degree of wavelength calibration polynomial
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.wdegree",
                                CPL_TYPE_INT,
                                "Degree of wavelength calibration polynomial",
                                "efosc.efosc_calib",
                                0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wdegree");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Reference lines search radius
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.wradius",
                                CPL_TYPE_INT,
                                "Search radius if iterating pattern-matching "
                                "with first-guess method",
                                "efosc.efosc_calib",
                                4);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wradius");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Rejection threshold in dispersion relation polynomial fitting
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.wreject",
                                CPL_TYPE_DOUBLE,
                                "Rejection threshold in dispersion "
                                "relation fit (pixel)",
                                "efosc.efosc_calib",
                                0.7);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wreject");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Wavelength solution interpolation (for LSS data)
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.wmode",
                                CPL_TYPE_INT,
                                "Interpolation mode of wavelength solution "
                                "applicable to LSS-like data (0 = no "
                                "interpolation, 1 = fill gaps, 2 = global "
                                "model)",
                                "efosc.efosc_calib",
                                2);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wmode");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Wavelength solution interpolation (for generic MOS data)
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.wmosmode",
                                CPL_TYPE_INT,
                                "Interpolation mode of wavelength solution "
                                "(0 = no interpolation, 1 = local (slit) "
                                "solution, 2 = global model)",
                                "efosc.efosc_calib",
                                0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wmosmode");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Line catalog table column containing the reference wavelengths
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.wcolumn",
                                CPL_TYPE_STRING,
                                "Name of line catalog table column "
                                "with wavelengths",
                                "efosc.efosc_calib",
                                "WLEN");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "wcolumn");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Degree of spectral curvature polynomial
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.cdegree",
                                CPL_TYPE_INT,
                                "Degree of spectral curvature polynomial",
                                "efosc.efosc_calib",
                                0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdegree");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Curvature solution interpolation (for MOS-like data)
     */
 
    p = cpl_parameter_new_value("efosc.efosc_calib.cmode",
                                CPL_TYPE_INT,
                                "Interpolation mode of curvature solution "
                                "applicable to MOS-like data (0 = no "
                                "interpolation, 1 = fill gaps, 2 = global "
                                "model)",
                                "efosc.efosc_calib",
                                1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cmode");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Start wavelength for spectral extraction
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.startwavelength",
                                CPL_TYPE_DOUBLE,
                                "Start wavelength in spectral extraction",
                                "efosc.efosc_calib",
                                0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "startwavelength");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * End wavelength for spectral extraction
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.endwavelength",
                                CPL_TYPE_DOUBLE,
                                "End wavelength in spectral extraction",
                                "efosc.efosc_calib",
                                0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "endwavelength");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Reference wavelength for wavelength calibration
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.reference",
                                CPL_TYPE_DOUBLE,
                                "Reference wavelength for calibration",
                                "efosc.efosc_calib",
                                0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "reference");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Try slit identification
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.slit_ident",
                                CPL_TYPE_BOOL,
                                "Attempt slit identification",
                                "efosc.efosc_calib",
                                TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "slit_ident");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Flat field frames stack parameters
     */

    efosc_stack_define_parameters(recipe->parameters, "efosc.efosc_calib",
                                 "average");

    /*
     * Degree of flat field fitting polynomial along spatial direction 
     * (used for LSS data)
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.sdegree",
                                CPL_TYPE_INT,
                                "Degree of flat field fitting polynomial "
                                "along spatial direction (used for LSS-like "
                                "data only)",
                                "efosc.efosc_calib",
                                4);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sdegree");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Degree of flat field fitting polynomial along dispersion direction
     * (used for MOS data)
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.ddegree",
                                CPL_TYPE_INT,
                                "Degree of flat field fitting polynomial "
                                "along dispersion direction (not used for "
                                "long-slit-like data)",
                                "efosc.efosc_calib",
                                -1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "ddegree");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Smooth box radius for flat field along dispersion direction
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.dradius",
                                CPL_TYPE_INT,
                                "Smooth box radius for flat field along "
                                "dispersion direction",
                                "efosc.efosc_calib",
                                10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "dradius");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Smooth box radius for flat field along spatial direction
     * (used for LSS data only)
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.sradius",
                                CPL_TYPE_INT,
                                "Smooth box radius for flat field along "
                                "spatial direction",
                                "efosc.efosc_calib",
                                10);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sradius");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    /*
     * Create check products
     */

    p = cpl_parameter_new_value("efosc.efosc_calib.check",
                                CPL_TYPE_BOOL,
                                "Create intermediate products",
                                "efosc.efosc_calib",
                                FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "check");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    return 0;
}


/**
 * @brief    Execute the plugin instance given by the interface
 *
 * @param    plugin  the plugin
 *
 * @return   0 if everything is ok
 */

static int efosc_calib_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    return efosc_calib(recipe->parameters, recipe->frames);
}


/**
 * @brief    Destroy what has been created by the 'create' function
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 */

static int efosc_calib_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else 
        return -1;

    cpl_parameterlist_delete(recipe->parameters); 

    return 0;
}


/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    parlist     The parameters list
 * @param    frameset    The set-of-frames
 *
 * @return   0 if everything is ok
 */

static int efosc_calib(cpl_parameterlist *parlist, cpl_frameset *frameset)
{

    const char *recipe = "efosc_calib";


    /*
     * Input parameters
     */

    double      dispersion;
    double      peakdetection;
    int         wdegree;
    int         wradius;
    double      wreject;
    int         wmode;
    int         wmosmode;
    const char *wcolumn;
    int         cdegree;
    int         cmode;
    double      startwavelength;
    double      endwavelength;
    double      reference;
    int         slit_ident;
    int         sdegree;
    int         ddegree;
    int         sradius;
    int         dradius;
    int         check;
    const char *stack_method;
    int         min_reject;
    int         max_reject;
    double      klow;
    double      khigh;
    int         kiter;

    /*
     * CPL objects
     */

    cpl_imagelist    *biases      = NULL;
    cpl_image        *bias        = NULL;
    cpl_image        *master_bias = NULL;
    cpl_image        *multi_bias  = NULL;
    cpl_image        *flat        = NULL;
    cpl_image        *master_flat = NULL;
    cpl_image        *added_flat  = NULL;
    cpl_image        *trace_flat  = NULL;
    cpl_image        *smo_flat    = NULL;
    cpl_image        *norm_flat   = NULL;
    cpl_image        *spectra     = NULL;
    cpl_image        *wavemap     = NULL;
    cpl_image        *delta       = NULL;
    cpl_image        *residual    = NULL;
    cpl_image        *checkwave   = NULL;
    cpl_image        *rectified   = NULL;
    cpl_image        *dummy       = NULL;
    cpl_image        *add_dummy   = NULL;
    cpl_image        *refimage    = NULL;
    cpl_image        *coordinate  = NULL;
    cpl_image        *rainbow     = NULL;
    cpl_image        *spatial     = NULL;
    cpl_image        *rect_flat   = NULL;
    cpl_image        *mapped_flat = NULL;

    cpl_mask         *refmask     = NULL;

    cpl_table        *grism_table = NULL;
    cpl_table        *overscans   = NULL;
    cpl_table        *wavelengths = NULL;
    cpl_table        *idscoeff    = NULL;
    cpl_table        *restable    = NULL;
    cpl_table        *slits       = NULL;
    cpl_table        *positions   = NULL;
    cpl_table        *maskslits   = NULL;
    cpl_table        *traces      = NULL;
    cpl_table        *polytraces  = NULL;
    cpl_table        *restab      = NULL;
    cpl_table        *global      = NULL;

    cpl_vector       *lines       = NULL;

    cpl_propertylist *header      = NULL;
    cpl_propertylist *save_header = NULL;
    cpl_propertylist *qclist      = NULL;

    cpl_frame        *frame       = NULL;
    cpl_frameset     *no_bias_set = NULL;

    /*
     * Auxiliary variables
     */

    char    version[80];
    char   *arc_tag;
    char   *flat_tag;
    char   *master_screen_flat_tag;
    char   *master_norm_flat_tag;
    char   *reduced_lamp_tag;
    char   *disp_residuals_tag;
    char   *disp_coeff_tag;
    char   *wavelength_map_tag;
    char   *spectra_detection_tag;
    char   *spectral_resolution_tag;
    char   *slit_map_tag;
    char   *curv_traces_tag;
    char   *curv_coeff_tag;
    char   *spatial_map_tag;
    char   *slit_location_tag;
    char   *global_distortion_tag = "GLOBAL_DISTORTION_TABLE";
    char   *disp_residuals_table_tag;
    char   *delta_image_tag;
    char   *mapped_screen_flat_tag;
    char   *keyname;
    char   *key_gris_name;
    char   *key_filt_name;
    char   *key_conad;
    char    mode;
    int     mos = 0;
    int     lss = 0;
    int     treat_as_lss = 0;
    int     nslits;
    float  *data;
    double *xpos;
    double  mxpos;
    double  mean_rms;
    double  alltime;
    int     nflats;
    int     nbias;
    int     narc;
    int     nlines;
    int     rebin;
    double *line;
    double *fiterror = NULL;
    int    *fitlines = NULL;
    int     nx, ny;
    double  gain;
    int     ccd_xsize, ccd_ysize;
    int     rotate = 1;
    int     rotate_back = -1;
    int     i;

    char   *instrume = NULL;
    char   *pipefile = NULL;


    snprintf(version, 80, "%s-%s", PACKAGE, PACKAGE_VERSION);

    cpl_msg_set_indentation(2);

    /* 
     * Get configuration parameters
     */

    cpl_msg_info(recipe, "Recipe %s configuration parameters:", recipe);
    cpl_msg_indent_more();

    if (cpl_frameset_count_tags(frameset, "GRISM_TABLE") > 1)
        efosc_calib_exit("Too many in input: GRISM_TABLE");

    grism_table = dfs_load_table(frameset, "GRISM_TABLE", 1);

    dispersion = dfs_get_parameter_double(parlist, 
                    "efosc.efosc_calib.dispersion", grism_table);

    if (dispersion <= 0.0)
        efosc_calib_exit("Invalid spectral dispersion value");

    peakdetection = dfs_get_parameter_double(parlist, 
                    "efosc.efosc_calib.peakdetection", grism_table);
    if (peakdetection <= 0.0)
        efosc_calib_exit("Invalid peak detection level");

    wdegree = dfs_get_parameter_int(parlist, 
                    "efosc.efosc_calib.wdegree", grism_table);

    if (wdegree < 1)
        efosc_calib_exit("Invalid polynomial degree");

    if (wdegree > 5)
        efosc_calib_exit("Max allowed polynomial degree is 5");

    wradius = dfs_get_parameter_int(parlist, 
                                    "efosc.efosc_calib.wradius", NULL);

    if (wradius < 0)
        efosc_calib_exit("Invalid search radius");

    wreject = dfs_get_parameter_double(parlist, 
                                       "efosc.efosc_calib.wreject", NULL);

    if (wreject <= 0.0)
        efosc_calib_exit("Invalid rejection threshold");

    wmode = dfs_get_parameter_int(parlist, "efosc.efosc_calib.wmode", NULL);

    if (wmode < 0 || wmode > 2)
        efosc_calib_exit("Invalid wavelength solution interpolation mode");

    wmosmode = dfs_get_parameter_int(parlist,
                                     "efosc.efosc_calib.wmosmode", NULL);

    if (wmosmode < 0 || wmosmode > 2)
        efosc_calib_exit("Invalid wavelength solution interpolation mode");

    wcolumn = dfs_get_parameter_string(parlist, 
                                       "efosc.efosc_calib.wcolumn", NULL);

    cdegree = dfs_get_parameter_int(parlist, 
                    "efosc.efosc_calib.cdegree", grism_table);

    if (cdegree < 1)
        efosc_calib_exit("Invalid polynomial degree");

    if (cdegree > 5)
        efosc_calib_exit("Max allowed polynomial degree is 5");

    cmode = dfs_get_parameter_int(parlist, "efosc.efosc_calib.cmode", NULL);

    if (cmode < 0 || cmode > 2)
        efosc_calib_exit("Invalid curvature solution interpolation mode");

    startwavelength = dfs_get_parameter_double(parlist, 
                    "efosc.efosc_calib.startwavelength", grism_table);
    if (startwavelength > 1.0)
        if (startwavelength < 3000.0 || startwavelength > 13000.0)
            efosc_calib_exit("Invalid wavelength");

    endwavelength = dfs_get_parameter_double(parlist, 
                    "efosc.efosc_calib.endwavelength", grism_table);
    if (endwavelength > 1.0) {
        if (endwavelength < 3000.0 || endwavelength > 13000.0)
            efosc_calib_exit("Invalid wavelength");
        if (startwavelength < 1.0)
            efosc_calib_exit("Invalid wavelength interval");
    }

    if (startwavelength > 1.0)
        if (endwavelength - startwavelength <= 0.0)
            efosc_calib_exit("Invalid wavelength interval");

    reference = dfs_get_parameter_double(parlist,
                "efosc.efosc_calib.reference", grism_table);

    if (reference < startwavelength || reference > endwavelength)
        efosc_calib_exit("Invalid reference wavelength");

    slit_ident = dfs_get_parameter_bool(parlist, 
                    "efosc.efosc_calib.slit_ident", NULL);

    stack_method = dfs_get_parameter_string(parlist,
                                            "efosc.efosc_calib.stack_method",
                                            NULL);

    if (strcmp(stack_method, "minmax") == 0) {
        min_reject = dfs_get_parameter_int(parlist,
                                       "efosc.efosc_calib.minrejection", NULL);
        if (min_reject < 0)
            efosc_calib_exit("Invalid number of lower rejections");

        max_reject = dfs_get_parameter_int(parlist,
                                       "efosc.efosc_calib.maxrejection", NULL);
        if (max_reject < 0)
            efosc_calib_exit("Invalid number of upper rejections");
    }

    if (strcmp(stack_method, "ksigma") == 0) {
        klow  = dfs_get_parameter_double(parlist, 
                                         "efosc.efosc_calib.klow", NULL);
        if (klow < 0.1)
            efosc_calib_exit("Invalid lower K-sigma");

        khigh = dfs_get_parameter_double(parlist, 
                                         "efosc.efosc_calib.khigh", NULL);
        if (khigh < 0.1)
            efosc_calib_exit("Invalid lower K-sigma");

        kiter = dfs_get_parameter_int(parlist, 
                                      "efosc.efosc_calib.kiter", NULL);
        if (kiter < 1)
            efosc_calib_exit("Invalid number of iterations");
    }

    sdegree = dfs_get_parameter_int(parlist, "efosc.efosc_calib.sdegree", NULL);
    ddegree = dfs_get_parameter_int(parlist, "efosc.efosc_calib.ddegree", NULL);
    sradius = dfs_get_parameter_int(parlist, "efosc.efosc_calib.sradius", NULL);
    dradius = dfs_get_parameter_int(parlist, "efosc.efosc_calib.dradius", NULL);

    if (sradius < 1 || dradius < 1)
        efosc_calib_exit("Invalid smoothing box radius");

    check = dfs_get_parameter_bool(parlist, "efosc.efosc_calib.check", NULL);

    cpl_table_delete(grism_table); grism_table = NULL;

    if (cpl_error_get_code())
        efosc_calib_exit("Failure getting the configuration parameters");


    /* 
     * Check input set-of-frames
     */

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Check input set-of-frames:");
    cpl_msg_indent_more();


    /*
     * Ensure that first frame is not a BIAS, to avoid wrong
     * header inheritance. If it is, move the BIAS frames at
     * the end of SOF.
     */

    if (cpl_frameset_count_tags(frameset, "BIAS")) {
        no_bias_set = cpl_frameset_duplicate(frameset);
        cpl_frameset_erase(frameset, "BIAS");
        frame = cpl_frameset_find(no_bias_set, "BIAS");
        while (frame) {
            cpl_frameset_insert(frameset, cpl_frame_duplicate(frame));
            frame = cpl_frameset_find(no_bias_set, NULL);
        }
        cpl_frameset_delete(no_bias_set); no_bias_set = NULL;
    }

    key_gris_name = "ESO INS GRIS1 NAME";
    key_filt_name = "ESO INS FILT1 NAME";

    no_bias_set = cpl_frameset_duplicate(frameset);
    cpl_frameset_erase(no_bias_set, "BIAS");
    cpl_frameset_erase(no_bias_set, "MASTER_BIAS");

    if (!dfs_equal_keyword(no_bias_set, key_gris_name)) 
        efosc_calib_exit("Input frames are not from the same grism");

    cpl_frameset_erase(no_bias_set, "GRISM_TABLE");
    if (!dfs_equal_keyword(no_bias_set, key_filt_name)) 
        efosc_calib_exit("Input frames are not from the same filter");

    if (!dfs_equal_keyword(no_bias_set, "ESO INS SLIT1 NAME")) 
        efosc_calib_exit("Input frames are not from the same mask");

    cpl_frameset_delete(no_bias_set); no_bias_set = NULL;

    if (!dfs_equal_keyword(frameset, "ESO DET READ SPEED")) 
        efosc_calib_exit("Input frames are not read with the same speed");

    if (!dfs_equal_keyword(frameset, "ESO DET WIN1 BINX")) 
        efosc_calib_exit("Input frames have not the same X rebin factor");

    if (!dfs_equal_keyword(frameset, "ESO DET WIN1 BINY")) 
        efosc_calib_exit("Input frames have not the same Y rebin factor");

    arc_tag                  = "LAMP_MOS";
    flat_tag                 = "SCREEN_FLAT_MOS";
    master_screen_flat_tag   = "MASTER_SCREEN_FLAT_MOS";
    master_norm_flat_tag     = "MASTER_NORM_FLAT_MOS";
    reduced_lamp_tag         = "REDUCED_LAMP_MOS";
    disp_residuals_tag       = "DISP_RESIDUALS_MOS";
    disp_coeff_tag           = "DISP_COEFF_MOS";
    wavelength_map_tag       = "WAVELENGTH_MAP_MOS";
    spectra_detection_tag    = "SPECTRA_DETECTION_MOS";
    spectral_resolution_tag  = "SPECTRAL_RESOLUTION_MOS";
    slit_map_tag             = "SLIT_MAP_MOS";
    curv_traces_tag          = "CURV_TRACES_MOS";
    curv_coeff_tag           = "CURV_COEFF_MOS";
    spatial_map_tag          = "SPATIAL_MAP_MOS";
    slit_location_tag        = "SLIT_LOCATION_MOS";
    disp_residuals_table_tag = "DISP_RESIDUALS_TABLE_MOS";
    delta_image_tag          = "DELTA_IMAGE_MOS";
    mapped_screen_flat_tag   = "MAPPED_SCREEN_FLAT_MOS";

    narc = cpl_frameset_count_tags(frameset, arc_tag);

    if (narc == 0)
        efosc_calib_exit("Missing input arc lamp frame");

    if (narc > 1)
        efosc_calib_exit("Just one input arc lamp frame is allowed"); 

    header = dfs_load_header(frameset, arc_tag, 0);

    if (header == NULL)
        efosc_calib_exit("Cannot load arc lamp frame header");

    mode = cpl_propertylist_get_string(header, "ESO INS SLIT1 NAME")[0];

    if (cpl_error_get_code() != CPL_ERROR_NONE)
        efosc_calib_exit("Missing INS SLIT1 NAME in arc lamp frame header");

    cpl_propertylist_delete(header); header = NULL;

    if (mode == 'M') {
        cpl_msg_info(recipe, "MOS data found");
        mos = 1;
    }
    else {
        cpl_msg_info(recipe, "LSS data found");
        lss = 1;
    }

    nbias = 0;
    if (cpl_frameset_count_tags(frameset, "MASTER_BIAS") == 0) {
        if (cpl_frameset_count_tags(frameset, "BIAS") == 0)
            efosc_calib_exit("Missing required input: MASTER_BIAS or BIAS");
        nbias = cpl_frameset_count_tags(frameset, "BIAS");
    }

    if (cpl_frameset_count_tags(frameset, "MASTER_BIAS") > 1)
        efosc_calib_exit("Too many in input: MASTER_BIAS");

    if (cpl_frameset_count_tags(frameset, "LINE_CATALOG") == 0)
        efosc_calib_exit("Missing required input: LINE_CATALOG");

    if (cpl_frameset_count_tags(frameset, "LINE_CATALOG") > 1)
        efosc_calib_exit("Too many in input: LINE_CATALOG");

    nflats = cpl_frameset_count_tags(frameset, flat_tag);

    if (nflats < 1) {
        cpl_msg_error(recipe, "Missing required input: %s", flat_tag);
        efosc_calib_exit(NULL);
    }

    cpl_msg_indent_less();

    if (nflats > 1)
        cpl_msg_info(recipe, "Load %d flat field frames and sum them...",
                     nflats);
    else
        cpl_msg_info(recipe, "Load flat field exposure...");

    cpl_msg_indent_more();

    header = dfs_load_header(frameset, flat_tag, 0);

    if (header == NULL)
        efosc_calib_exit("Cannot load flat field frame header");

    alltime = cpl_propertylist_get_double(header, "EXPTIME");

    if (cpl_error_get_code() != CPL_ERROR_NONE)
        efosc_calib_exit("Missing keyword EXPTIME in flat field frame header");

    cpl_propertylist_delete(header);

    for (i = 1; i < nflats; i++) {

        header = dfs_load_header(frameset, NULL, 0);

        if (header == NULL)
            efosc_calib_exit("Cannot load flat field frame header");

        alltime += cpl_propertylist_get_double(header, "EXPTIME");

        if (cpl_error_get_code() != CPL_ERROR_NONE)
            efosc_calib_exit("Missing keyword EXPTIME in flat field "
                            "frame header");

        cpl_propertylist_delete(header);

    }

    master_flat = dfs_load_image(frameset, flat_tag, CPL_TYPE_FLOAT, 0, 0);

    if (master_flat == NULL)
        efosc_calib_exit("Cannot load flat field");

    if (nflats > 1) {
        if (strcmp(stack_method, "average") == 0) {
            for (i = 1; i < nflats; i++) {
                flat = dfs_load_image(frameset, NULL, CPL_TYPE_FLOAT, 0, 0);
                if (flat) {
                    cpl_image_add(master_flat, flat);
                    cpl_image_delete(flat); flat = NULL;
                }
                else
                    efosc_calib_exit("Cannot load flat field");
            }
        }
        else {
            cpl_imagelist *flatlist = NULL;
            double rflux, flux;

            /*
             * added_flat is needed for tracing (masters obtained with
             * rejections are not suitable for tracing)
             */

            added_flat = cpl_image_duplicate(master_flat);

            flatlist = cpl_imagelist_new();
            cpl_imagelist_set(flatlist, master_flat,
                              cpl_imagelist_get_size(flatlist));
            master_flat = NULL;

            /*
             * Stacking with rejection requires normalization
             * at the same flux. We normalise according to mean
             * flux. This is equivalent to determining the
             * flux ration for each image as the average of the
             * flux ratio of all pixels weighted on the actual
             * flux of each pixel.
             */

            rflux = cpl_image_get_mean(added_flat);

            for (i = 1; i < nflats; i++) {
                flat = dfs_load_image(frameset, NULL, CPL_TYPE_FLOAT, 0, 0);
                if (flat) {
                    cpl_image_add(added_flat, flat);
                    flux = cpl_image_get_mean(flat);
                    cpl_image_multiply_scalar(flat, rflux / flux);
                    cpl_imagelist_set(flatlist, flat,
                                      cpl_imagelist_get_size(flatlist));
                    flat = NULL;
                }
                else {
                    efosc_calib_exit("Cannot load flat field");
                }
            }

            if (strcmp(stack_method, "median") == 0) {
                master_flat = cpl_imagelist_collapse_median_create(flatlist);
            }

            if (strcmp(stack_method, "minmax") == 0) {
                master_flat = cpl_imagelist_collapse_minmax_create(flatlist,
                                                                   min_reject,
                                                                   max_reject);
            }

            if (strcmp(stack_method, "ksigma") == 0) {
                master_flat = mos_ksigma_stack(flatlist,
                                               klow, khigh, kiter, NULL);
            }

            cpl_imagelist_delete(flatlist);
        }
    }

    /*
     * Get some info from arc lamp header
     */

    header = dfs_load_header(frameset, arc_tag, 0);

    if (header == NULL)
        efosc_calib_exit("Cannot load arc lamp header");

    instrume = (char *)cpl_propertylist_get_string(header, "INSTRUME");
    if (instrume == NULL)
        efosc_calib_exit("Missing keyword INSTRUME in arc lamp header");
    instrume = cpl_strdup(instrume);

    rebin = cpl_propertylist_get_int(header, "ESO DET WIN1 BINX");

    if (cpl_error_get_code() != CPL_ERROR_NONE)
        efosc_calib_exit("Missing keyword ESO DET WIN1 BINX in arc lamp "
                        "frame header");

    if (rebin != 1) {
        dispersion *= rebin;
        cpl_msg_warning(recipe, "The rebin factor is %d, and therefore the "
                        "working dispersion used is %f A/pixel", rebin,
                        dispersion);
    }

    key_conad = "ESO DET OUT1 CONAD";

    if (!cpl_propertylist_has(header, key_conad))
        key_conad = "ESO DET OUT2 CONAD";
    
    gain = cpl_propertylist_get_double(header, key_conad);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(recipe, "Missing keyword %s in arc lamp frame header", 
                      key_conad);
        efosc_calib_exit(NULL);
    }

    cpl_msg_info(recipe, "The gain factor is: %.2f e-/ADU", gain);

    if (mos) {

        cpl_msg_info(recipe, "Produce mask slit position table...");

        maskslits = mos_load_slits_efosc_mos(header);
    
        if (maskslits) {
    
            /*
             * Check if all slits have the same X offset: in such case, 
             * treat the observation as a long-slit one!
             */
    
            mxpos = cpl_table_get_column_median(maskslits, "ytop");
            xpos = cpl_table_get_data_double(maskslits, "ytop");
            nslits = cpl_table_get_nrow(maskslits);
    
            treat_as_lss = 1;
            for (i = 0; i < nslits; i++) {
                if (fabs(mxpos-xpos[i]) > 0.01) {
                    treat_as_lss = 0;
                    break;
                }
            }
        
            if (treat_as_lss) {
                cpl_msg_warning(recipe, 
                          "All MOS slits have the same offset: %.2f\n"
                          "The long-slit data reduction strategy is applied!", 
                          mxpos);
                cpl_table_delete(maskslits); maskslits = NULL;
            }
        
            if (slit_ident == 0) {
                cpl_table_delete(maskslits); maskslits = NULL;
            }
    
        }
        else {
            cpl_msg_warning(recipe, "No slits coordinates found in header.");
            if (slit_ident) {
                efosc_calib_exit("Cannot identify slits!");
            }
            cpl_error_reset();
        }
    }
    
    /* Leave the header on for the next step... */


    /*
     * Remove the master bias
     */

    if (nbias) {

        /*
         * Set of raw BIASes in input, need to create master bias!
         */

        cpl_msg_info(recipe, "Generate the master from input raw biases...");

        if (nbias > 1) {

            biases = cpl_imagelist_new();

            bias = dfs_load_image(frameset, "BIAS", CPL_TYPE_FLOAT, 0, 0);
    
            if (bias == NULL)
                efosc_calib_exit("Cannot load bias frame");

            cpl_imagelist_set(biases, bias, 0); bias = NULL;
    
            for (i = 1; i < nbias; i++) {
                bias = dfs_load_image(frameset, NULL, CPL_TYPE_FLOAT, 0, 0);
                if (bias) {
                    cpl_imagelist_set(biases, bias, i); bias = NULL;
                }
                else
                    efosc_calib_exit("Cannot load bias frame");
            }
    
            master_bias = cpl_imagelist_collapse_median_create(biases);

            cpl_imagelist_delete(biases);
        }
        else {
            master_bias = dfs_load_image(frameset, "BIAS", 
                                         CPL_TYPE_FLOAT, 0, 1);
            if (master_bias == NULL)
                efosc_calib_exit("Cannot load bias");
        }

    }
    else {
        master_bias = dfs_load_image(frameset, "MASTER_BIAS", 
                                     CPL_TYPE_FLOAT, 0, 1);
        if (master_bias == NULL)
            efosc_calib_exit("Cannot load master bias");
    }

    cpl_msg_info(recipe, "Remove the master bias...");

    overscans = mos_load_overscans_efosc(header);
    cpl_propertylist_delete(header); header = NULL;

    if (nbias) {
        int xlow = cpl_table_get_int(overscans, "xlow", 0, NULL);
        int ylow = cpl_table_get_int(overscans, "ylow", 0, NULL);
        int xhig = cpl_table_get_int(overscans, "xhig", 0, NULL);
        int yhig = cpl_table_get_int(overscans, "yhig", 0, NULL);
        dummy = cpl_image_extract(master_bias, xlow+1, ylow+1, xhig, yhig);
        cpl_image_delete(master_bias); master_bias = dummy;

        if (dfs_save_image(frameset, master_bias, "MASTER_BIAS",
                           NULL, parlist, recipe, version))
            efosc_calib_exit(NULL);
    }

    if (nflats > 1) {
        multi_bias = cpl_image_multiply_scalar_create(master_bias, nflats);
        dummy = mos_remove_bias(master_flat, multi_bias, overscans);
        if (added_flat)
            add_dummy = mos_remove_bias(added_flat, multi_bias, overscans);
        cpl_image_delete(multi_bias);
    }
    else {
        dummy = mos_remove_bias(master_flat, master_bias, overscans);
    }
    cpl_image_delete(master_flat);
    master_flat = dummy;

    if (master_flat == NULL)
        efosc_calib_exit("Cannot remove bias from flat field");

    if (added_flat) {
        cpl_image_delete(added_flat);
        added_flat = add_dummy;

        if (added_flat == NULL)
            efosc_calib_exit("Cannot remove bias from added flat field");

        cpl_image_turn(added_flat, rotate);

        trace_flat = added_flat;
    }
    else
        trace_flat = master_flat;

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Load arc lamp exposure...");
    cpl_msg_indent_more();

    spectra = dfs_load_image(frameset, arc_tag, CPL_TYPE_FLOAT, 0, 0);

    if (spectra == NULL)
        efosc_calib_exit("Cannot load arc lamp exposure");

    cpl_msg_info(recipe, "Remove the master bias...");

    dummy = mos_remove_bias(spectra, master_bias, overscans);
    cpl_table_delete(overscans); overscans = NULL;
    cpl_image_delete(master_bias); master_bias = NULL;
    cpl_image_delete(spectra); spectra = dummy;

    if (spectra == NULL)
        efosc_calib_exit("Cannot remove bias from arc lamp exposure");

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Load input line catalog...");
    cpl_msg_indent_more();

    wavelengths = dfs_load_table(frameset, "LINE_CATALOG", 1);

    if (wavelengths == NULL)
        efosc_calib_exit("Cannot load line catalog");


    /*
     * Cast the wavelengths into a (double precision) CPL vector
     */

    nlines = cpl_table_get_nrow(wavelengths);

    if (nlines == 0)
        efosc_calib_exit("Empty input line catalog");

    if (cpl_table_has_column(wavelengths, wcolumn) != 1) {
        cpl_msg_error(recipe, "Missing column %s in input line catalog table",
                      wcolumn);
        efosc_calib_exit(NULL);
    }

    line = cpl_malloc(nlines * sizeof(double));
    
    for (i = 0; i < nlines; i++)
        line[i] = cpl_table_get(wavelengths, wcolumn, i, NULL);

    cpl_table_delete(wavelengths); wavelengths = NULL;

    lines = cpl_vector_wrap(nlines, line);


    /*
     * Rotate frames horizontally with red to the right
     */

    cpl_image_turn(spectra, rotate);
    cpl_image_turn(master_flat, rotate);

    if (treat_as_lss || lss) {

        /* FIXME:
         * The LSS data calibration is still dirty: it doesn't apply
         * any spatial rectification, and only in future an external
         * spectral curvature model would be provided in input. Here
         * and there temporary solutions are adopted, such as accepting
         * the preliminary wavelength calibration.
         */

        /*
         * Flat field normalisation is done directly on the master flat
         * field (without spatial rectification first). The spectral
         * curvature model may be provided in input, in future releases.
         */

        cpl_msg_indent_less();
        cpl_msg_info(recipe, "Perform flat field normalisation...");
        cpl_msg_indent_more();

        norm_flat = cpl_image_duplicate(master_flat);

        smo_flat = mos_normalise_longflat(norm_flat, sradius, dradius,
                                          sdegree);

        cpl_image_delete(smo_flat); smo_flat = NULL; /* It may be a product */

        save_header = dfs_load_header(frameset, flat_tag, 0);
        cpl_propertylist_update_int(save_header, "ESO PRO DATANCOM", nflats);

        cpl_image_turn(master_flat, rotate_back);
        if (dfs_save_image(frameset, master_flat, master_screen_flat_tag,
                           save_header, parlist, recipe, version))
            efosc_calib_exit(NULL);
//%%%        cpl_image_delete(master_flat); master_flat = NULL;
        cpl_image_turn(master_flat, rotate);

        cpl_image_turn(norm_flat, rotate_back);
        if (dfs_save_image(frameset, norm_flat, master_norm_flat_tag,
                           save_header, parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_propertylist_delete(save_header); save_header = NULL;
//%%%        cpl_image_delete(norm_flat); norm_flat = NULL;


        /*
         * In the case of LSS data, extract the spectra directly
         * on the first attempt. The spectral curvature model may
         * be provided in input, in future releases.
         */

        cpl_msg_indent_less();
        cpl_msg_info(recipe, "Perform wavelength calibration...");
        cpl_msg_indent_more();

        nx = cpl_image_get_size_x(spectra);
        ny = cpl_image_get_size_y(spectra);

        wavemap = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
        if (check)
            residual = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);

        fiterror = cpl_calloc(ny, sizeof(double));
        fitlines = cpl_calloc(ny, sizeof(int));
        idscoeff = cpl_table_new(ny);
        restable = cpl_table_new(nlines);

        rectified = mos_wavelength_calibration_raw(spectra, lines, dispersion,
                                                   peakdetection, wradius,
                                                   wdegree, wreject, reference,
                                                   &startwavelength, 
                                                   &endwavelength, fitlines, 
                                                   fiterror, idscoeff, wavemap,
                                                   residual, restable, NULL);

        if (rectified == NULL)
            efosc_calib_exit("Wavelength calibration failure.");

        if (!cpl_table_has_valid(idscoeff, "c0"))
            efosc_calib_exit("Wavelength calibration failure.");

        if (dfs_save_table(frameset, restable, disp_residuals_table_tag, NULL,
                           parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_table_delete(restable); restable = NULL;

        if (wmode) {
            cpl_image_delete(rectified); rectified = NULL;
            cpl_image_delete(wavemap); wavemap = NULL;
            mos_interpolate_wavecalib(idscoeff, wavemap, wmode, 2);
            wavemap = mos_map_idscoeff(idscoeff, nx, reference,
                                       startwavelength, endwavelength);
            rectified = mos_wavelength_calibration(spectra, reference,
                                                   startwavelength, 
                                                   endwavelength, dispersion, 
                                                   idscoeff, 0);
        }

        cpl_table_wrap_double(idscoeff, fiterror, "error"); fiterror = NULL;
        cpl_table_set_column_unit(idscoeff, "error", "pixel");
        cpl_table_wrap_int(idscoeff, fitlines, "nlines"); fitlines = NULL;

        for (i = 0; i < ny; i++)
            if (!cpl_table_is_valid(idscoeff, "c0", i))
                cpl_table_set_invalid(idscoeff, "error", i);

        delta = mos_map_pixel(idscoeff, reference, startwavelength,
                              endwavelength, dispersion, 2);

//        dummy = cpl_image_extract(master_flat, 1, ylow, nx, yhig);
//        cpl_image_delete(master_flat); master_flat = dummy;

        mapped_flat = mos_wavelength_calibration(master_flat, reference,
                                      startwavelength, endwavelength,
                                      dispersion, idscoeff, 0);

        cpl_image_delete(master_flat); master_flat = NULL;

        header = cpl_propertylist_new();
        cpl_propertylist_update_double(header, "CRPIX1", 1.0);
        cpl_propertylist_update_double(header, "CRPIX2", 1.0);
        cpl_propertylist_update_double(header, "CRVAL1",
                                       startwavelength + dispersion/2);
        cpl_propertylist_update_double(header, "CRVAL2", 1.0);
        /* cpl_propertylist_update_double(header, "CDELT1", dispersion);
        cpl_propertylist_update_double(header, "CDELT2", 1.0); */
        cpl_propertylist_update_double(header, "CD1_1", dispersion);
        cpl_propertylist_update_double(header, "CD1_2", 0.0);
        cpl_propertylist_update_double(header, "CD2_1", 0.0);
        cpl_propertylist_update_double(header, "CD2_2", 1.0);
        cpl_propertylist_update_string(header, "CTYPE1", "LINEAR");
        cpl_propertylist_update_string(header, "CTYPE2", "PIXEL");

        if (dfs_save_image(frameset, delta, delta_image_tag,
                           header, parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_image_delete(delta); delta = NULL;

        if (dfs_save_image(frameset, mapped_flat, mapped_screen_flat_tag,
                           header, parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_image_delete(mapped_flat); mapped_flat = NULL;

        cpl_propertylist_delete(header); header = NULL;

        cpl_msg_info(recipe, "Valid solutions found: "CPL_SIZE_FORMAT" out of %d rows", 
                     ny - cpl_table_count_invalid(idscoeff, "c0"), ny);

        cpl_image_delete(spectra); spectra = NULL;

        mean_rms = mos_distortions_rms(rectified, lines, startwavelength,
                                       dispersion, 6, 0);

        cpl_msg_info(recipe, "Mean residual: %f pixel", mean_rms);

        mean_rms = cpl_table_get_column_mean(idscoeff, "error");

        cpl_msg_info(recipe, "Mean model accuracy: %f pixel (%f A)",
                     mean_rms, mean_rms * dispersion);

        restab = mos_resolution_table(rectified, startwavelength, dispersion,
                                      60000, lines);

        if (restab) {
            cpl_msg_info(recipe, "Mean spectral resolution: %.2f",
                  cpl_table_get_column_mean(restab, "resolution"));
            cpl_msg_info(recipe, 
                  "Mean reference lines FWHM: %.2f +/- %.2f pixel",
                  cpl_table_get_column_mean(restab, "fwhm") / dispersion,
                  cpl_table_get_column_mean(restab, "fwhm_rms") / dispersion);

            qclist = cpl_propertylist_new();

            /*
             * QC1 parameters
             */

            keyname = "QC.MOS.RESOLUTION";

            if (efosc_qc_write_qc_double(qclist, 
                    cpl_table_get_column_mean(restab,
                            "resolution"),
                            keyname,
                            "Mean spectral resolution")) {
                efosc_calib_exit("Cannot write mean spectral resolution to "
                        "QC log file");
            }

            keyname = "QC.MOS.RESOLUTION.RMS"; 

            if (efosc_qc_write_qc_double(qclist,
                    cpl_table_get_column_stdev(restab,
                            "resolution"),
                            keyname,
                            "Scatter of spectral resolution")) {
                efosc_calib_exit("Cannot write spectral resolution scatter "
                        "to QC log file");
            }

            keyname = "QC.MOS.RESOLUTION.NLINES";

            if (efosc_qc_write_qc_int(qclist, cpl_table_get_nrow(restab) -
                    cpl_table_count_invalid(restab,
                            "resolution"),
                            keyname,
                            "Number of lines for spectral "
                            "resolution computation")) {
                efosc_calib_exit("Cannot write number of lines used in "
                        "spectral resolution computation "
                        "to QC log file");
            }

            if (dfs_save_table(frameset, restab, spectral_resolution_tag, 
                               qclist, parlist, recipe, version))
                efosc_calib_exit(NULL);

            cpl_table_delete(restab); restab = NULL;
            cpl_propertylist_delete(qclist); qclist = NULL;

        }
        else
            efosc_calib_exit("Cannot compute the spectral resolution table");

        cpl_vector_delete(lines); lines = NULL;


        /*
         * Save rectified arc lamp spectrum to disk
         */

        header = cpl_propertylist_new();
        cpl_propertylist_update_double(header, "CRPIX1", 1.0);
        cpl_propertylist_update_double(header, "CRPIX2", 1.0);
        cpl_propertylist_update_double(header, "CRVAL1", 
                                       startwavelength + dispersion/2);
        cpl_propertylist_update_double(header, "CRVAL2", 1.0);
        /* cpl_propertylist_update_double(header, "CDELT1", dispersion);
        cpl_propertylist_update_double(header, "CDELT2", 1.0); */
        cpl_propertylist_update_double(header, "CD1_1", dispersion);
        cpl_propertylist_update_double(header, "CD1_2", 0.0);
        cpl_propertylist_update_double(header, "CD2_1", 0.0);
        cpl_propertylist_update_double(header, "CD2_2", 1.0);
        cpl_propertylist_update_string(header, "CTYPE1", "LINEAR");
        cpl_propertylist_update_string(header, "CTYPE2", "PIXEL");
        cpl_propertylist_update_int(header, "ESO PRO DATANCOM", 1);

        if (dfs_save_image(frameset, rectified, reduced_lamp_tag, header, 
                           parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_image_delete(rectified); rectified = NULL;
        cpl_propertylist_delete(header); header = NULL;

        if (dfs_save_table(frameset, idscoeff, disp_coeff_tag, NULL, 
                           parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_table_delete(idscoeff); idscoeff = NULL;

        header = dfs_load_header(frameset, arc_tag, 0);

        if (header == NULL)
            efosc_calib_exit("Cannot reload arc lamp header");

        /*
         * QC1 parameters
         */

        data = cpl_image_get_data(wavemap);

        if (efosc_qc_write_qc_double(header, data[nx/2 + ny*nx/2],
                "QC.MOS.CENTRAL.WAVELENGTH",
                "Wavelength at CCD center")) {
            efosc_calib_exit("Cannot write central wavelength to QC "
                    "log file");
        }

        cpl_image_turn(wavemap, rotate_back);
        if (dfs_save_image(frameset, wavemap, wavelength_map_tag, header,
                           parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_image_delete(wavemap); wavemap = NULL;

        if (check) {
            cpl_propertylist_update_double(header, "CRPIX2", 1.0);
            cpl_propertylist_update_double(header, "CRVAL2", 1.0);
            /* cpl_propertylist_update_double(header, "CDELT2", 1.0); */
            cpl_propertylist_update_double(header, "CD1_1", 1.0);
            cpl_propertylist_update_double(header, "CD1_2", 0.0);
            cpl_propertylist_update_double(header, "CD2_1", 0.0);
            cpl_propertylist_update_double(header, "CD2_2", 1.0);
            cpl_propertylist_update_string(header, "CTYPE1", "LINEAR");
            cpl_propertylist_update_string(header, "CTYPE2", "PIXEL");

            if (dfs_save_image(frameset, residual, disp_residuals_tag, header,
                               parlist, recipe, version))
                efosc_calib_exit(NULL);

            cpl_image_delete(residual); residual = NULL;
        }

        cpl_propertylist_delete(header); header = NULL;
        cpl_free(instrume); instrume = NULL;

        return 0;         /* Successful LSS data reduction */

    }   /* End of LSS data reduction section */


    /*
     * Here the MOS calibration is carried out.
     */

    /*
     * Detecting spectra on the CCD
     */

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Detecting spectra on CCD...");
    cpl_msg_indent_more();

    ccd_xsize = nx = cpl_image_get_size_x(spectra);
    ccd_ysize = ny = cpl_image_get_size_y(spectra);

    refmask = cpl_mask_new(nx, ny);

    checkwave = mos_wavelength_calibration_raw(spectra, lines, dispersion, 
                                               peakdetection, wradius, 
                                               wdegree, wreject, reference,
                                               &startwavelength, &endwavelength,
                                               NULL, NULL, NULL, NULL, NULL, 
                                               NULL, refmask);

    if (checkwave == NULL)
        efosc_calib_exit("Wavelength calibration failure.");

    if (check) {

        /*
         * Save check image to disk
         */

        header = cpl_propertylist_new();
        cpl_propertylist_update_double(header, "CRPIX1", 1.0);
        cpl_propertylist_update_double(header, "CRPIX2", 1.0);
        cpl_propertylist_update_double(header, "CRVAL1", 
                                       startwavelength + dispersion/2);
        cpl_propertylist_update_double(header, "CRVAL2", 1.0);
        /* cpl_propertylist_update_double(header, "CDELT1", dispersion);
        cpl_propertylist_update_double(header, "CDELT2", 1.0); */
        cpl_propertylist_update_double(header, "CD1_1", dispersion);
        cpl_propertylist_update_double(header, "CD1_2", 0.0);
        cpl_propertylist_update_double(header, "CD2_1", 0.0);
        cpl_propertylist_update_double(header, "CD2_2", 1.0);
        cpl_propertylist_update_string(header, "CTYPE1", "LINEAR");
        cpl_propertylist_update_string(header, "CTYPE2", "PIXEL");
        if (dfs_save_image(frameset, checkwave, spectra_detection_tag, header, 
                           parlist, recipe, version))
            efosc_calib_exit(NULL);
        cpl_propertylist_delete(header); header = NULL;
    }

    cpl_image_delete(checkwave); checkwave = NULL;

    cpl_msg_info(recipe, "Locate slits at reference wavelength on CCD...");
    slits = mos_locate_spectra(refmask);

    if (!slits) {
        cpl_msg_error(cpl_error_get_where(), "%s", cpl_error_get_message());
        efosc_calib_exit("No slits could be detected!");
    }

    refimage = cpl_image_new_from_mask(refmask);
    cpl_mask_delete(refmask); refmask = NULL;

    if (check) {
        save_header = dfs_load_header(frameset, arc_tag, 0);
        cpl_image_turn(refimage, rotate_back);
        if (dfs_save_image(frameset, refimage, slit_map_tag, NULL,
                           parlist, recipe, version))
            efosc_calib_exit(NULL);
        cpl_propertylist_delete(save_header); save_header = NULL;
    }

    cpl_image_delete(refimage); refimage = NULL;

    if (slit_ident) {

        /*
         * Attempt slit identification: this recipe may continue even
         * in case of failed identification (i.e., the position table is 
         * not produced, but an error is not set). In case of failure,
         * the spectra would be still extracted, even if they would not
         * be associated to slits on the mask.
         * 
         * The reason for making the slit identification an user option 
         * (via the parameter slit_ident) is to offer the possibility 
         * to avoid identifications that are only apparently successful, 
         * as it would happen in the case of an incorrect slit description 
         * in the data header.
         */

        cpl_msg_indent_less();
        cpl_msg_info(recipe, "Attempt slit identification (optional)...");
        cpl_msg_indent_more();

        mos_rotate_slits(maskslits, -rotate, 0, 0);
        positions = mos_identify_slits(slits, maskslits, NULL);

        if (positions) {
            cpl_table_delete(slits);
            slits = positions;

            /*
             * Eliminate slits which are _entirely_ outside the CCD
             */

            cpl_table_and_selected_double(slits, 
                                          "ybottom", CPL_GREATER_THAN, ny-1);
            cpl_table_or_selected_double(slits, 
                                          "ytop", CPL_LESS_THAN, 0);
            cpl_table_erase_selected(slits);

            nslits = cpl_table_get_nrow(slits);

            if (nslits == 0)
                efosc_calib_exit("No slits found on the CCD");

            cpl_msg_info(recipe, "%d slits are entirely or partially "
                         "contained in CCD", nslits);

        }
        else {
            slit_ident = 0;
            cpl_msg_info(recipe, "Global distortion model cannot be computed");
            if (cpl_error_get_code() != CPL_ERROR_NONE) {
                efosc_calib_exit(NULL);
            }
        }
    }


    /*
     * Determination of spectral curvature
     */

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Determining spectral curvature...");
    cpl_msg_indent_more();

    cpl_msg_info(recipe, "Tracing master flat field spectra edges...");
    traces = mos_trace_flat(trace_flat, slits, reference, 
                            startwavelength, endwavelength, dispersion);

    if (!traces)
        efosc_calib_exit("Tracing failure");

    cpl_image_delete(added_flat); added_flat = NULL;

    cpl_msg_info(recipe, "Fitting flat field spectra edges...");
    polytraces = mos_poly_trace(slits, traces, cdegree);

    if (!polytraces)
        efosc_calib_exit("Trace fitting failure");

    if (cmode) {
        cpl_msg_info(recipe, "Computing global spectral curvature model...");
        mos_global_trace(slits, polytraces, cmode);
    }

    if (dfs_save_table(frameset, traces, curv_traces_tag, NULL, parlist,
                       recipe, version))
        efosc_calib_exit(NULL);

    cpl_table_delete(traces); traces = NULL;

    coordinate = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    spatial = mos_spatial_calibration(spectra, slits, polytraces, reference, 
                                      startwavelength, endwavelength, 
                                      dispersion, 0, coordinate);

    if (!slit_ident) {
        cpl_image_delete(spectra); spectra = NULL;
    }

    /*
     * Flat field normalisation is done directly on the master flat
     * field (without spatial rectification first). The spectral
     * curvature model may be provided in input, in future releases.
     */

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Perform flat field normalisation...");
    cpl_msg_indent_more();

    norm_flat = cpl_image_duplicate(master_flat);

    smo_flat = mos_normalise_flat(norm_flat, coordinate, slits, polytraces, 
                                  reference, startwavelength, endwavelength,
                                  dispersion, dradius, ddegree);

    cpl_image_delete(smo_flat); smo_flat = NULL;  /* It may be a product */
 
    save_header = dfs_load_header(frameset, flat_tag, 0);
    cpl_propertylist_update_int(save_header, "ESO PRO DATANCOM", nflats);
//%%%
    rect_flat = mos_spatial_calibration(master_flat, slits, polytraces,
                                        reference, startwavelength,
                                        endwavelength, dispersion, 0, NULL);

    cpl_image_turn(master_flat, rotate_back);
    if (dfs_save_image(frameset, master_flat, master_screen_flat_tag,
                       save_header, parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(master_flat); master_flat = NULL;

    cpl_image_turn(norm_flat, rotate_back);
    if (dfs_save_image(frameset, norm_flat, master_norm_flat_tag,
                       save_header, parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(norm_flat); norm_flat = NULL;
    cpl_propertylist_delete(save_header); save_header = NULL;


    /*
     * Final wavelength calibration of spectra having their curvature
     * removed
     */

    cpl_msg_indent_less();
    cpl_msg_info(recipe, "Perform final wavelength calibration...");
    cpl_msg_indent_more();

    nx = cpl_image_get_size_x(spatial);
    ny = cpl_image_get_size_y(spatial);

    idscoeff = cpl_table_new(ny);
    restable = cpl_table_new(nlines);
    rainbow = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    if (check)
        residual = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    fiterror = cpl_calloc(ny, sizeof(double));
    fitlines = cpl_calloc(ny, sizeof(int));

    rectified = mos_wavelength_calibration_final(spatial, slits, lines, 
                                                 dispersion, peakdetection, 
                                                 wradius, wdegree, wreject,
                                                 reference, &startwavelength, 
                                                 &endwavelength, fitlines, 
                                                 fiterror, idscoeff, rainbow, 
                                                 residual, restable);



    if (rectified == NULL)
        efosc_calib_exit("Wavelength calibration failure.");

    if (dfs_save_table(frameset, restable, disp_residuals_table_tag, NULL,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_table_delete(restable); restable = NULL;

    cpl_table_wrap_double(idscoeff, fiterror, "error"); fiterror = NULL;
    cpl_table_set_column_unit(idscoeff, "error", "pixel");
    cpl_table_wrap_int(idscoeff, fitlines, "nlines"); fitlines = NULL;

    for (i = 0; i < ny; i++)
        if (!cpl_table_is_valid(idscoeff, "c0", i))
            cpl_table_set_invalid(idscoeff, "error", i);

    if (wmosmode > 0) {
        mos_interpolate_wavecalib_slit(idscoeff, slits, 1, wmosmode - 1);

        cpl_image_delete(rectified);

        rectified = mos_wavelength_calibration(spatial, reference,
                                           startwavelength, endwavelength,
                                           dispersion, idscoeff, 0);
    }

    cpl_image_delete(spatial); spatial = NULL;

    delta = mos_map_pixel(idscoeff, reference, startwavelength,
                          endwavelength, dispersion, 2);

    header = cpl_propertylist_new();
    cpl_propertylist_update_double(header, "CRPIX1", 1.0);
    cpl_propertylist_update_double(header, "CRPIX2", 1.0);
    cpl_propertylist_update_double(header, "CRVAL1",
                                   startwavelength + dispersion/2);
    cpl_propertylist_update_double(header, "CRVAL2", 1.0);
    /* cpl_propertylist_update_double(header, "CDELT1", dispersion);
    cpl_propertylist_update_double(header, "CDELT2", 1.0); */
    cpl_propertylist_update_double(header, "CD1_1", dispersion);
    cpl_propertylist_update_double(header, "CD1_2", 0.0);
    cpl_propertylist_update_double(header, "CD2_1", 0.0);
    cpl_propertylist_update_double(header, "CD2_2", 1.0);
    cpl_propertylist_update_string(header, "CTYPE1", "LINEAR");
    cpl_propertylist_update_string(header, "CTYPE2", "PIXEL");

    if (dfs_save_image(frameset, delta, delta_image_tag,
                       header, parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(delta); delta = NULL;
    cpl_propertylist_delete(header); header = NULL;

    mean_rms = mos_distortions_rms(rectified, lines, startwavelength, 
                                   dispersion, 6, 0);

    cpl_msg_info(recipe, "Mean residual: %f pixel", mean_rms);

    mean_rms = cpl_table_get_column_mean(idscoeff, "error");

    cpl_msg_info(recipe, "Mean model accuracy: %f pixel (%f A)", 
                 mean_rms, mean_rms * dispersion);

    restab = mos_resolution_table(rectified, startwavelength, dispersion, 
                                  60000, lines);

    if (restab) {
        cpl_msg_info(recipe, "Mean spectral resolution: %.2f", 
                   cpl_table_get_column_mean(restab, "resolution"));
        cpl_msg_info(recipe, "Mean reference lines FWHM: %.2f +/- %.2f pixel",
                   cpl_table_get_column_mean(restab, "fwhm") / dispersion,
                   cpl_table_get_column_mean(restab, "fwhm_rms") / dispersion);

        qclist = cpl_propertylist_new();


        /*
         * QC1 parameters
         */

        keyname = "QC.MOS.RESOLUTION";

        if (efosc_qc_write_qc_double(qclist, 
                cpl_table_get_column_mean(restab,
                        "resolution"),
                        keyname,
                        "Mean spectral resolution")) {
            efosc_calib_exit("Cannot write mean spectral resolution to QC "
                    "log file");
        }

        keyname = "QC.MOS.RESOLUTION.RMS";

        if (efosc_qc_write_qc_double(qclist, 
                cpl_table_get_column_stdev(restab, 
                        "resolution"),
                        "QC.MOS.RESOLUTION.RMS",
                        "Scatter of spectral resolution")) {
            efosc_calib_exit("Cannot write spectral resolution scatter "
                    "to QC log file");
        }

        keyname = "QC.MOS.RESOLUTION.NLINES";

        if (efosc_qc_write_qc_int(qclist, cpl_table_get_nrow(restab) -
                cpl_table_count_invalid(restab, 
                        "resolution"),
                        "QC.MOS.RESOLUTION.NLINES",
                        "Number of lines for spectral resolution "
                        "computation")) {
            efosc_calib_exit("Cannot write number of lines used in "
                    "spectral resolution computation "
                    "to QC log file");
        }

        if (dfs_save_table(frameset, restab, spectral_resolution_tag, qclist,
                           parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_table_delete(restab); restab = NULL;
        cpl_propertylist_delete(qclist); qclist = NULL;

    }
    else
        efosc_calib_exit("Cannot compute the spectral resolution table");

    cpl_free(instrume); instrume = NULL;
    cpl_vector_delete(lines); lines = NULL;

    if (dfs_save_table(frameset, idscoeff, disp_coeff_tag, NULL,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    mapped_flat = mos_wavelength_calibration(rect_flat, reference,
                                      startwavelength, endwavelength,
                                      dispersion, idscoeff, 0);

    cpl_image_delete(rect_flat); rect_flat = NULL;


    /*
     * Global distortion models
     */

    if (slit_ident) {

        cpl_msg_info(recipe, "Computing global distortions model");
        global = mos_global_distortion(slits, maskslits, idscoeff, 
                                       polytraces, reference);

        if (global && 0) {
            cpl_table *stest;
            cpl_table *ctest;
            cpl_table *dtest;
            cpl_image *itest;

            stest = mos_build_slit_location(global, maskslits, ccd_ysize);

            ctest = mos_build_curv_coeff(global, maskslits, stest);
            if (dfs_save_table(frameset, ctest, "CURVS", NULL,
                               parlist, recipe, version))
                efosc_calib_exit(NULL);

            itest = mos_spatial_calibration(spectra, stest, ctest, 
                                            reference, startwavelength, 
                                            endwavelength, dispersion, 
                                            0, NULL);
            cpl_table_delete(ctest); ctest = NULL;
            cpl_image_delete(itest); itest = NULL;
            if (dfs_save_table(frameset, stest, "SLITS", NULL,
                               parlist, recipe, version))
                efosc_calib_exit(NULL);

            dtest = mos_build_disp_coeff(global, stest);
            if (dfs_save_table(frameset, dtest, "DISPS", NULL,
                               parlist, recipe, version))
                efosc_calib_exit(NULL);

            cpl_table_delete(dtest); dtest = NULL;
            cpl_table_delete(stest); stest = NULL;
        }

        if (global) {
            if (dfs_save_table(frameset, global, global_distortion_tag, NULL,
                               parlist, recipe, version))
                efosc_calib_exit(NULL);
            cpl_table_delete(global); global = NULL;
        }

        cpl_image_delete(spectra); spectra = NULL;
        cpl_table_delete(maskslits); maskslits = NULL;
    }

    cpl_table_delete(idscoeff); idscoeff = NULL;

    header = cpl_propertylist_new();
    cpl_propertylist_update_double(header, "CRPIX1", 1.0);
    cpl_propertylist_update_double(header, "CRPIX2", 1.0);
    cpl_propertylist_update_double(header, "CRVAL1", 
                                   startwavelength + dispersion/2);
    cpl_propertylist_update_double(header, "CRVAL2", 1.0);
    /* cpl_propertylist_update_double(header, "CDELT1", dispersion);
    cpl_propertylist_update_double(header, "CDELT2", 1.0); */
    cpl_propertylist_update_double(header, "CD1_1", dispersion);
    cpl_propertylist_update_double(header, "CD1_2", 0.0);
    cpl_propertylist_update_double(header, "CD2_1", 0.0);
    cpl_propertylist_update_double(header, "CD2_2", 1.0);
    cpl_propertylist_update_string(header, "CTYPE1", "LINEAR");
    cpl_propertylist_update_string(header, "CTYPE2", "PIXEL");
    cpl_propertylist_update_int(header, "ESO PRO DATANCOM", 1);

    if (dfs_save_image(frameset, rectified, reduced_lamp_tag, header,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(rectified); rectified = NULL;

//%%%
    cpl_propertylist_update_int(header, "ESO PRO DATANCOM", nflats);

    if (dfs_save_image(frameset, mapped_flat, mapped_screen_flat_tag, header,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(mapped_flat); mapped_flat = NULL;

    cpl_propertylist_delete(header); header = NULL;

    if (check) {
        save_header = dfs_load_header(frameset, arc_tag, 0);

        cpl_propertylist_update_double(save_header, "CRPIX2", 1.0);
        cpl_propertylist_update_double(save_header, "CRVAL2", 1.0);
        /* cpl_propertylist_update_double(save_header, "CDELT2", 1.0); */
        cpl_propertylist_update_double(save_header, "CD1_1", 1.0);
        cpl_propertylist_update_double(save_header, "CD1_2", 0.0);
        cpl_propertylist_update_double(save_header, "CD2_1", 0.0);
        cpl_propertylist_update_double(save_header, "CD2_2", 1.0);
        cpl_propertylist_update_string(save_header, "CTYPE1", "LINEAR");
        cpl_propertylist_update_string(save_header, "CTYPE2", "PIXEL");

        if (dfs_save_image(frameset, residual, disp_residuals_tag, save_header,
                           parlist, recipe, version))
            efosc_calib_exit(NULL);

        cpl_image_delete(residual); residual = NULL;
        cpl_propertylist_delete(save_header); save_header = NULL;
    }

    wavemap = mos_map_wavelengths(coordinate, rainbow, slits, polytraces, 
                                  reference, startwavelength, endwavelength, 
                                  dispersion);

    cpl_image_delete(rainbow); rainbow = NULL;

    save_header = dfs_load_header(frameset, arc_tag, 0);

    cpl_image_turn(wavemap, rotate_back);
    if (dfs_save_image(frameset, wavemap, wavelength_map_tag, save_header,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(wavemap); wavemap = NULL;

    cpl_image_turn(coordinate, rotate_back);
    if (dfs_save_image(frameset, coordinate, spatial_map_tag, save_header,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_image_delete(coordinate); coordinate = NULL;
    cpl_propertylist_delete(save_header); save_header = NULL;

    if (dfs_save_table(frameset, polytraces, curv_coeff_tag, NULL,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_table_delete(polytraces); polytraces = NULL;

    mos_rotate_slits(slits, rotate, ccd_ysize, ccd_xsize);
    if (dfs_save_table(frameset, slits, slit_location_tag, NULL,
                       parlist, recipe, version))
        efosc_calib_exit(NULL);

    cpl_table_delete(slits); slits = NULL;

    if (cpl_error_get_code()) {
        cpl_msg_error(cpl_error_get_where(), "%s", cpl_error_get_message());
        efosc_calib_exit(NULL);
    }

    return 0;
}
