/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <string.h>
#include <cpl.h>
#include "eris_pfits.h"
#include "eris_ifu_error.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_combine_static.h"

/**
   @name   eris_combine_jittered_cubes_range()
   @param  cubesData:       List of jittered cubes to mosaic (over a given range
                            with kappa-sigma clipping of outliers)
   @param  cubesError:      Associated errors to cubesData
   @param  mergedCubeData:  Resulting merged cube containing the jittered cubes
   @param  mergedCubeError: Associated errors to mergedCubeData
   @param  mergedCubeDIT:   Image of the same size as mergedCubeData containing 0
                            for blank (ZERO pixels) and the summed integration
                            times for overlapping regions
   @param n_cubes:          Number of cubes in the list to merge
   @param cumoffsetx:       Array of relative x pixel offsets with respect to the
                            first frame in the same sequence as the cube list.
   @param cumoffsety:       Array of relative y pixel offsetsm with respect to the
                            first frame in the same sequence as the cube list.
   @param exptimes:         Exposure times array giving the time in the same
                            sequence as the cube list
   @param kernel_type:      The name of the interpolation kernel that you want to
                            generate using the eclipse routine
                            sinfo_generate_interpolation_kernel()
                            Supported kernels are:
                                    "tanh":    Hyperbolic tangent
                                    "sinc2":   Square sinc
                                    "lanczos": Lanczos2 kernel
                                    "hamming": Hamming kernel
                                    "hann":    Hann kernel
   @param z_min             Minimum cube's plane processed
   @param z_max             Maximum cube's plane processed
   @param kappa             Value for kappa-sigma clipping
   @param compute_mode      MEAN or MEDIAN mode for the combined cube
   @param pclip             Percentile clip to be initially applied.

   @doc
   Merges jittered data cubes to one bigger cube by averaging the overlap regions
   weighted by the integration times. The x, y size of the final data cube is user
   given, and should be between 32 and 64 pixels, while the relative pixel-offset
   (sub-pixel accuracy) of the single cubes with respect to the first cube in the
   list is read from the SEQ CUMOFFSETX,Y fits header keyword.

   Copied from sinfo_new_combine_jittered_cubes_thomas_range()
 */
cpl_error_code eris_ifu_combine_jittered_images(
                                            cpl_image     **imagesData,
                                            cpl_image     **imagesError,
                                            int           nx_out,
                                            int           ny_out,
                                            cpl_image     **mergedImageData,
                                            cpl_image     **mergedImageError,
                                            cpl_image     **mergedImageDIT,
                                            int           n_cubes,
                                            const float   *offsetx,
                                            const float   *offsety,
                                            const double  *exptimes,
                                            const double  kappa,
                                            const char    *compute_mode,
                                            const int     pclip)
{
    int             llx0                    = 0,
                    lly0                    = 0,
                    nx_in                   = 0,
                    ny_in                   = 0,
                    min_lx                  = INT_MAX,
                    min_ly                  = INT_MAX,
                    *llx                    = NULL,
                    *lly                    = NULL;
    float           *sub_offsetx            = NULL,
                    *sub_offsety            = NULL;
    cpl_image       **imagesDataShifted     = NULL,
                    **imagesErrorShifted    = NULL;
    cpl_error_code  ret_err                 = CPL_ERROR_NONE;

    cpl_ensure_code(imagesData      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(imagesError     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mergedImageData  != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mergedImageError != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mergedImageDIT   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(offsetx      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(offsety      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(exptimes        != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(compute_mode    != NULL, CPL_ERROR_NULL_INPUT);

    TRY {
        nx_in = cpl_image_get_size_x(imagesData[0]);
        ny_in = cpl_image_get_size_y(imagesData[0]);

        /* Center the cubes within the allocated big cube.
         * That means to define the (0,0) positions of the cubes in the image planes
         * to sub-pixel accuracy by using cumoffsetx/y and the reference cube. */

        /* Position of first reference frame, centered in big cube */
        llx0 = (int)((double)(nx_out - nx_in) / 2.0 + 0.5);
        lly0 = (int)((double)(ny_out - ny_in) / 2.0 + 0.5);

        /* Go through the frame list and determine the lower left edge position
         * of the shifted cubes. Additionnally, the sub-pixel offsets are
         * determined. */
        llx = cpl_calloc(n_cubes, sizeof(int));
        lly = cpl_calloc(n_cubes, sizeof(int));
        sub_offsetx = cpl_calloc(n_cubes, sizeof(float));
        sub_offsety = cpl_calloc(n_cubes, sizeof(float));
        CHECK_ERROR_STATE();

        for (int i = 0; i < n_cubes; i++) {
            llx[i] = llx0 - eris_ifu_combine_nearest_int(offsetx[i]);

            sub_offsetx[i] = (float)eris_ifu_combine_nearest_int(offsetx[i]) - offsetx[i];
            lly[i] = lly0 - eris_ifu_combine_nearest_int(offsety[i]);
            sub_offsety[i] = (float)eris_ifu_combine_nearest_int(offsety[i]) - offsety[i];

            if (llx[i] < min_lx) {
                min_lx = llx[i];
            }
            if (lly[i] < min_ly) {
                min_ly = lly[i];
            }
        }
        CHECK_ERROR_STATE();

        /* "Normalize" the shift - minimum should be 0 */
        if (min_lx != 0) {
            for (int i = 0 ; i < n_cubes ; i++ ) {
                llx[i] = llx[i] - min_lx;
            }
        }
        if (min_ly != 0) {
            for (int i = 0 ; i < n_cubes ; i++ ) {
                lly[i] = lly[i] - min_ly;
            }
        }

        /* Shift the cubes according to the computed sub-pixel offsets
         * that means shift the single image planes of each cube
         * first determine an interpolation kernel
         */
        BRK_IF_NULL(
            imagesDataShifted  = (cpl_image**)cpl_calloc(n_cubes, sizeof(cpl_imagelist*)));
        BRK_IF_NULL(
            imagesErrorShifted = (cpl_image**)cpl_calloc(n_cubes, sizeof(cpl_imagelist*)));

        for (int i = 0; i < n_cubes; i++) {
            BRK_IF_NULL(
                imagesDataShifted[i] = eris_ifu_combine_shift_image_kmos(
                                                imagesData[i],
                                                sub_offsetx[i], sub_offsety[i],
                                                "NN", NONE_NANS));
            BRK_IF_NULL(
                imagesErrorShifted[i] = eris_ifu_combine_shift_image_kmos(
                                                imagesError[i],
                                                sub_offsetx[i], sub_offsety[i],
                                                "NN", NONE_NANS));
        }

        /* Build the DIT cube */
        BRK_IF_ERROR(
            eris_ifu_combine_build_mask_cube(imagesDataShifted,
                                             mergedImageDIT,
                                             llx, lly, exptimes,
                                             n_cubes,
                                             nx_out, ny_out));

        if ((kappa != -1) && (pclip != -1)) {
            // ks-clipping
            BRK_IF_ERROR(
                eris_ifu_combine_coadd_ks_clip(
                                            n_cubes,
                                            kappa, llx, lly, exptimes,
                                            mergedImageData, mergedImageError, *mergedImageDIT,
                                            imagesDataShifted, imagesErrorShifted,
                                            compute_mode, pclip, nx_out, ny_out));
        } else {
            /* Calculate a weighted average using the exposure time of the
             * single frames of the overlapping regions of the cubes */
            BRK_IF_ERROR(
                eris_ifu_combine_coadd(n_cubes,
                                       mergedImageData, mergedImageError, *mergedImageDIT,
                                       imagesDataShifted, imagesErrorShifted,
                                       exptimes,
                                       llx, lly, compute_mode, nx_out, ny_out));
        }
    } CATCH {
        CATCH_MSGS();
        ret_err = cpl_error_get_code();
        eris_ifu_free_image(mergedImageData);
        eris_ifu_free_image(mergedImageError);
        eris_ifu_free_image(mergedImageDIT);
    }

    // free memory
    for (int i = 0; i < n_cubes; i++) {
        eris_ifu_free_image(&imagesDataShifted[i]);
        eris_ifu_free_image(&imagesErrorShifted[i]);
    }

    cpl_free(imagesDataShifted); imagesDataShifted = NULL;
    cpl_free(imagesErrorShifted); imagesErrorShifted = NULL;
    cpl_free(llx); llx = NULL;
    cpl_free(lly); lly = NULL;
    cpl_free(sub_offsetx); sub_offsetx = NULL;
    cpl_free(sub_offsety); sub_offsety = NULL;

    return ret_err;
}

/**
 * @brief eris_ifu_combine_divide_DIT
 * @param cubesData List of Data cubes
 * @param n_cubes   Number of data cubes
 * @param exptimes  Vector with exposuretimes for each cube
 * @return Error code
 *
 * Divides each cube by its exposure time.
 */
cpl_error_code eris_ifu_combine_divide_DIT( cpl_imagelist **cubesData,
                                            const int     n_cubes,
                                            const double *exptimes)
{
	cpl_ensure_code(cubesData,CPL_ERROR_NULL_INPUT);
    TRY {
        for (int i = 0; i < n_cubes; i++) {
            BRK_IF_ERROR(
                cpl_imagelist_divide_scalar(cubesData[i], exptimes[i]));
        }
    } CATCH {

    }

    return cpl_error_get_code();
}

/**
  @name eris_ifu_combine_auto_size_cube
  @brief    Computes size of coadded cube
  @param    offsetx  Input offset list
  @param    offsety  Input offset list
  @param    nframes  Input number of values
  @param    ref_offx Input reference offset array
  @param    ref_offy Input reference offset array
  @param    size_x   Input/output coadded cube x size
  @param    size_y   Input/output coadded cube y size
  @note This routine differs from the sinfo_auto_size_cube5 one because
  allows to input cubes of size different from the one of a single cube
  component (64).

  Copied from sinfo_auto_size_cube
*/
cpl_error_code eris_ifu_combine_auto_size_cube(const float *offsetx,
                                               const float *offsety,
                                               const int nframes,
                                               float *ref_offx,
                                               float *ref_offy,
                                               int *size_x,
                                               int *size_y)
{
    float   min_offx = 0,
            max_offx = 0,
            min_offy = 0,
            max_offy = 0;

    cpl_ensure_code(offsetx  != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(offsety  != NULL, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(nframes > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*size_x >= 64, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(*size_y >= 64, CPL_ERROR_ILLEGAL_INPUT);

    cpl_msg_info (cpl_func, "Computation of output cube size") ;
    eris_ifu_combine_get_xy_min_max(nframes,
                                    offsetx,   offsety,
                                    &min_offx, &max_offx,
                                    &min_offy, &max_offy);

    cpl_msg_info(cpl_func, "   max_offx = %f, max_offy = %f", max_offx, max_offy);
    cpl_msg_info(cpl_func, "   min_offx = %f, min_offy = %f", min_offx, min_offy);

    *ref_offx = (min_offx+max_offx)/2;
    *ref_offy = (min_offy+max_offy)/2;

    /* The following to allow to compute the size of output cube in case the input
    * has size different than 64x64
    */
    *size_x += 2*floor(max_offx-min_offx+0.5);
    *size_y += 2*floor(max_offy-min_offy+0.5);

    cpl_msg_info(cpl_func, "   Output cube size: %d x %d",*size_x,*size_y);
    cpl_msg_info(cpl_func, "   Ref. offset x: %f, y: %f",*ref_offx,*ref_offy);
    cpl_msg_info(cpl_func, "   Max. offset x: %f, y: %f",max_offx,max_offy);
    cpl_msg_info(cpl_func, "   Min. offset x: %f, y: %f",min_offx,min_offy);

    return cpl_error_get_code();
}

/**
   @name eris_ifu_combine_build_mask_cube
   @brief Build the mask data image.

   @param z_min          minimum cube's plane processed
   @param z_max          maximum cube's plane processed
   @param olx            output cube x size
   @param oly            output cube y size
   @param n_cubes        number of cubes in the list to merge
   @param llx            lower left edge x position of the shifted cubes.
   @param lly            lower left edge y position of the shifted cubes.
   @param exptimes       exposure times array giving the time
                         in the same sequence as the cube list
   @param  cubesData        list of jittered cubes to mosaic
   @param  cubesDataShifted list of shifted jittered cubes to mosaic
   @param  mergedCubeDIT    image of the same size as combinedCube containing 0
                            for blank (ZERO pixels) and the summed integration times for
                            overlapping regions

   @doc Build the mask data cube.
        The mask is 0 where no data is available, otherwise the
        integration time of one frame, respectively the summed integration
        times in the overlapping regions are inserted

    This version is based on eris_build_mask_cube_thomas() which in fact does
    pretty the same than eris_build_mask_cube(). Just the loops are sorted
    differently. This function has been added by Alex Agudo (MPE) in order not
    to delete old (somehow) working and to create a consistent workflow for all
    cases (mean/median and ks-clipping/no-clipping).

    Copied from sinfo_build_mask_cube
 */
cpl_error_code eris_ifu_combine_build_mask(cpl_imagelist **cubesDataShifted,
                                                cpl_image     *mergedImgDIT,
                                                const int     n_cubes,
                                                const int     *llx,
                                                const int     *lly,
                                                const double  *exptimes)
{
    int             nx_in                   = 0,
                    ny_in                   = 0;
    cpl_size        nx_out                  = 0,
                    ny_out                  = 0;
    cpl_image       *imgCubesDataShifted    = NULL;
    double          *pimgMergedCubeDIT      = NULL;

    cpl_ensure_code(llx != NULL,              CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lly != NULL,              CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(exptimes != NULL,         CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cubesDataShifted != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mergedImgDIT != NULL,     CPL_ERROR_NULL_INPUT);

    nx_out = cpl_image_get_size_x(mergedImgDIT);
    ny_out = cpl_image_get_size_y(mergedImgDIT);

    pimgMergedCubeDIT = cpl_image_get_data_double(mergedImgDIT);

    for (int i = 0; i < n_cubes; i++) {
        imgCubesDataShifted = cpl_imagelist_get(cubesDataShifted[i], 0);
        nx_in = cpl_image_get_size_x(imgCubesDataShifted);
        ny_in = cpl_image_get_size_y(imgCubesDataShifted);

        //go through the first image plane of the big data cube
        for (int y = 0; y < ny_out; y++) {
            for (int  x = 0; x < nx_out; x++) {
                // find the position of the present cube and go
                // through the single spectra
                if ((y >= lly[i]) && (y < lly[i]+ny_in) &&
                    (x >= llx[i]) && (x < llx[i]+nx_in))
                {
                    pimgMergedCubeDIT[x+y*nx_out] += exptimes[i];
                }
            } // end for(x)
        } // end for(y)
    } // end for(i)
    return cpl_error_get_code();
}

/**
   @name eris_ifu_combine_build_mask_cube
   @brief Build the mask data cube.

   @param z_min          minimum cube's plane processed
   @param z_max          maximum cube's plane processed
   @param olx            output cube x size
   @param oly            output cube y size
   @param n_cubes        number of cubes in the list to merge
   @param llx            lower left edge x position of the shifted cubes.
   @param lly            lower left edge y position of the shifted cubes.
   @param exptimes       exposure times array giving the time
                         in the same sequence as the cube list
   @param  cubesData        list of jittered cubes to mosaic
   @param  cubesDataShifted list of shifted jittered cubes to mosaic
   @param  mergedCubeDIT    image of the same size as combinedCube containing 0
                            for blank (ZERO pixels) and the summed integration times for
                            overlapping regions

    This is a variation of eris_ifu_combine_build_mask which is doing just one
    mask for a single slice. But just the first one. NaNs are not taken into account.
    Here we do hence a mask for every slice is calculated.
 */
cpl_error_code eris_ifu_combine_build_mask_cube(cpl_image     **imagesDataShifted,
                                                cpl_image     **mergedImgDIT,
                                                const int     *llx,
                                                const int     *lly,
                                                const double  *exptimes,
                                                int           n_cubes,
                                                cpl_size      nx_out,
                                                cpl_size      ny_out)
{
    int             nx_in                   = 0,
                    ny_in                   = 0,
                    x_in                    = 0,
                    y_in                    = 0;
    double          *pimgMergedCubeDIT      = NULL,
                    *pimgCubesDataShifted   = NULL;
    cpl_image       *imgCubesDataShifted    = NULL;
    cpl_error_code  ret_err                 = CPL_ERROR_NONE;

    cpl_ensure_code(llx != NULL,              CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lly != NULL,              CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(exptimes != NULL,         CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(imagesDataShifted != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mergedImgDIT != NULL,    CPL_ERROR_NULL_INPUT);

    TRY {
        BRK_IF_NULL(
            *mergedImgDIT = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE));
        BRK_IF_NULL(
            pimgMergedCubeDIT = cpl_image_get_data_double(*mergedImgDIT));

        for (int i = 0; i < n_cubes; i++) {
            imgCubesDataShifted = imagesDataShifted[i];
            BRK_IF_NULL(
                pimgCubesDataShifted = cpl_image_get_data_double(imgCubesDataShifted));
            nx_in = cpl_image_get_size_x(imgCubesDataShifted);
            ny_in = cpl_image_get_size_y(imgCubesDataShifted);
            x_in = 0;
            y_in = 0;
            for (int y = 0; y < ny_out; y++) {
                for (int  x = 0; x < nx_out; x++) {
                    // find the position of the present cube and go
                    // through the single spectra
                    if ((y >= lly[i]) && (y < lly[i]+ny_in) &&
                        (x >= llx[i]) && (x < llx[i]+nx_in))
                    {
                        if (!eris_ifu_is_nan_or_inf(pimgCubesDataShifted[x_in+nx_in*y_in]) &&
                            !cpl_image_is_rejected(imgCubesDataShifted, x_in+1, y_in+1))
                        {
                            pimgMergedCubeDIT[x+y*nx_out] += exptimes[i];
                        }
                        x_in++;
                        if (x_in >= nx_in) {
                            x_in = 0;
                            y_in++;
                        }
                    }
                } // end for(x)
            } // end for(y)
        } // end for(i)
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_image(mergedImgDIT);
        ret_err = cpl_error_get_code();
    }

    return ret_err;
}
/**
 * @brief eris_ifu_combine_coadd_ks_clip
 * @param z_min             Min cube's plane to be processed
 * @param z_max             Max cube's plane to be processed
 * @param n_cubes           Number of cubes to be coadded
 * @param kappa             Value for kappa-sigma clip rejection
 * @param llx               Array holding lower left X for each input cube
 * @param lly               Array holding lower left Y for each input cube
 * @param exptimes          Array holding exposure times for each input cube
 * @param mergedCubeData    Output coadded cube data
 * @param mergedCubeError   Output coadded cube errors
 * @param mergedCubeDIT     Output coadded cube DIT contributions
 * @param cubesDataShifted  Input cubes data to be coadded
 * @param cubesErrorShifted Input cubes error to be coadded
 * @param compute_mode      MEAN or MEDIAN, for the combined cube
 * @param pclip             xxx
 * @return
 *
 *  MEAN:
 *  data_exp_sum = sum(cubesDataShifted * mergedCubeDIT)
 *  exp_sum = sum(cubesDataShifted * mergedCubeDIT)
 *  mergedCubeData = data_exp_sum / exp_sum
 *
 *  MEDIAN:
 *  No division by exptime! All data cubes should have the same exposure time!
 */
cpl_error_code eris_ifu_combine_coadd_ks_clip(const int n_frames,
                                              const double kappa,
                                              int *llx,
                                              int *lly,
                                              const double *exptimes,
                                              cpl_image **imgMergedCubeData,
                                              cpl_image **imgMergedCubeError,
                                              cpl_image *mergedImgDIT,
                                              cpl_image **imagesDataShifted,
                                              cpl_image **imagesErrorShifted,
                                              const char *compute_mode,
                                              const int pclip,
                                              const int nx_out,
                                              const int ny_out)
{
    int         n_contributes           = 0,
                nx_in                   = 0,
                ny_in                   = 0,
                posx                    = 0,
                posy                    = 0,
                ovr                     = 0;    //overlap count
    double      *pimgCubesDataShifted   = NULL,
                *pimgCubesErrorShifted  = NULL,
                *pimgMergedCubeData     = NULL,
                *pimgMergedCubeError    = NULL,
                *pmergedImgDIT          = NULL;
    cpl_vector  *msk                    = NULL,
                *val                    = NULL; // vector to compute median of an output pixel
    eris_ifu_vector *data_vec           = NULL,
                    *err_vec            = NULL;

    TRY {
        nx_in = cpl_image_get_size_x(imagesDataShifted[0]);
        ny_in = cpl_image_get_size_y(imagesDataShifted[0]);

        if (
            (nx_out != cpl_image_get_size_x(mergedImgDIT)) ||
            (ny_out != cpl_image_get_size_y(mergedImgDIT))
            )
        {
            cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
            return cpl_error_get_code();
        }

        BRK_IF_NULL(
            *imgMergedCubeData = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE));
        BRK_IF_NULL(
            pimgMergedCubeData  = cpl_image_get_data_double(*imgMergedCubeData));

        BRK_IF_NULL(
            *imgMergedCubeError = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE));
        BRK_IF_NULL(
            pimgMergedCubeError = cpl_image_get_data_double(*imgMergedCubeError));

        BRK_IF_NULL(
            pmergedImgDIT = cpl_image_get_data_double(mergedImgDIT));

        for (int y = 0; y < ny_out; y++) {
            for (int x = 0; x < nx_out; x++) {
                n_contributes = eris_ifu_combine_calc_contributions(imagesDataShifted,
                                                                    n_frames,
                                                                    llx, lly, x, y);
                if (n_contributes > 0) {
                    msk = cpl_vector_new(n_frames);
                    for (int i = 0; i < n_frames; i++) {
                        cpl_vector_set(msk, i, 1);
                    }

                    BRK_IF_ERROR(
                        eris_ifu_combine_coadd_ks_clip_internal(imagesDataShifted, n_frames,
                                                                n_contributes, x, y,
                                                                llx, lly, kappa, &msk,
                                                                compute_mode, pclip));

                    if (!strcmp(compute_mode, "MEDIAN")){
                        // median
                        val = cpl_vector_new(n_frames);
                        ovr = 0;
                    }

                    data_vec = eris_ifu_vector_new(n_frames);
                    err_vec = eris_ifu_vector_new(n_frames);

                    for (int i = 0; i < n_frames; i++) {
                        if ((y >= lly[i]) && (y < lly[i]+ny_in) &&
                            (x >= llx[i]) && (x < llx[i]+nx_in))
                        {
                            pimgCubesDataShifted = cpl_image_get_data_double(imagesDataShifted[i]);
                            pimgCubesErrorShifted = cpl_image_get_data_double(imagesErrorShifted[i]);

                            posx = x - llx[i];
                            posy = y - lly[i];

                            if (!isnan(pimgCubesDataShifted[posx+posy*nx_in]) &&
                                (fabs(pimgCubesDataShifted[posx+posy*nx_in]) > DBL_ZERO_TOLERANCE) &&
                                (fabs(pmergedImgDIT[x+y*nx_out]) > DBL_ZERO_TOLERANCE) &&
                                (cpl_vector_get(msk, i) != 0))
                            {
                                if (!strcmp(compute_mode, "MEDIAN")) {
                                    // median
                                    cpl_vector_set(val, ovr, pimgCubesDataShifted[posx+posy*nx_in]);
                                    ovr++;
                                } else {
                                    // mean
                                    pimgMergedCubeData[x+y*nx_out] += pimgCubesDataShifted[posx+posy*nx_in] * exptimes[i];
                                }
                                eris_ifu_vector_set(data_vec, i, pimgCubesDataShifted[posx+posy*nx_in]);
                                eris_ifu_vector_set(err_vec, i, pimgCubesErrorShifted[posx+posy*nx_in]);
                            } else {
                                // invalid data value
                                eris_ifu_vector_reject(data_vec, i);
                                eris_ifu_vector_reject(err_vec, i);
                            }
                        } else {
                            // invalid data value
                            eris_ifu_vector_reject(data_vec, i);
                            eris_ifu_vector_reject(err_vec, i);
                        }
                    } // end for(i)

                    // calculate error
                    pimgMergedCubeError[x+y*nx_out] = eris_ifu_combine_calc_error(data_vec, err_vec, compute_mode);
                    eris_ifu_vector_delete(data_vec); data_vec = NULL;
                    eris_ifu_vector_delete(err_vec); err_vec = NULL;

                    if (!strcmp(compute_mode, "MEDIAN")) {
                        // median
                        if (ovr > 0) {
                            cpl_vector *tval = cpl_vector_extract(val, 0, ovr-1, 1);
                            pimgMergedCubeData[x+y*nx_out] = cpl_vector_get_median_const(tval);
                            eris_ifu_free_vector(&tval);
                        }
                        eris_ifu_free_vector(&val);
                    } else {
                        // mean
                        pimgMergedCubeData[x+y*nx_out] /= pmergedImgDIT[x+y*nx_out];
                    }

                    eris_ifu_free_vector(&msk);

                } // end check if overlap nc > 0
            } // end loop over x
        } // end loop over y
    } CATCH {
        eris_ifu_free_vector(&msk);
        eris_ifu_free_vector(&val);
        eris_ifu_free_image(imgMergedCubeData);
        eris_ifu_free_image(imgMergedCubeError);
    }

    return cpl_error_get_code();
}

/**
 * @name eris_ifu_combine_coadd_ks_clip_internal
 * @param cubesDataShifted
 * @param n_cubes
 * @param n_contributions
 * @param x
 * @param y
 * @param m
 * @param llx
 * @param lly
 * @param kappa
 * @param msk
 * @param compute_mode
 * @param pclip
 * @return
 *
 * @brief inner routine called from eris_ifu_combine_coadd_ks_clip()
 */
cpl_error_code eris_ifu_combine_coadd_ks_clip_internal(cpl_image** imagesDataShifted,
                                            const int n_frames,
                                            const int n_contributions,
                                            const int x,
                                            const int y,
                                            int *llx,
                                            int *lly,
                                            const double kappa,
                                            cpl_vector **msk,
                                            const char *compute_mode,
                                            const int pclip)
{
    int             nclip       = 0,
                    lox         = 0,
                    loy         = 0,
                    upx         = 0,
                    upy         = 0,
                    posx        = 0,
                    posy        = 0,
                    pos         = 0,
                    ovr         = 0,
                    pk          = 0;
    double          sig         = 0.,
                    med         = 0.,
                    *pimg_in    = NULL;
    cpl_size        nx          = 0,
                    ny          = 0;
    cpl_vector      *val        = NULL;
    cpl_error_code  ret_err     = CPL_ERROR_NONE;

    TRY {
        nx = cpl_image_get_size_x(imagesDataShifted[0]);
        ny = cpl_image_get_size_y(imagesDataShifted[0]);

        for (int ks = 0; ks < n_contributions; ks++) {  //for large nc: may optimize for efficiency
            sig = 0;
            med = 0;
            ovr = 0;

            if (n_contributions-nclip > 0) {
                val = cpl_vector_new(n_contributions-nclip);
            } else {
                break;
            }

            // fill val
            for (int i = 0; i < n_frames; i++) {
                pimg_in = cpl_image_get_data_double(imagesDataShifted[i]);

                lox = llx[i];
                loy = lly[i];
                upx = llx[i]+nx;
                upy = lly[i]+ny;

                if ((y >= loy) && (y < upy) && (x >= lox) && (x < upx)) {
                    posx = x - lox;
                    posy = y - loy;
                    pos = posx+posy*nx;

                    if (!isnan(pimg_in[pos]) &&
                        (fabs(pimg_in[pos]) > DBL_ZERO_TOLERANCE) &&
                        (cpl_vector_get(*msk, i) != 0))
                    {
                        BRK_IF_ERROR(
                            cpl_vector_set(val, ovr, pimg_in[pos]));
                        ovr++;
                    }
                }
            }

            // get avg, med, sig
            if (ovr > 2) {
                if (!strcmp(compute_mode, "MEDIAN")) {
                    med=cpl_vector_get_median(val);
                } else {
                    med = cpl_vector_get_mean(val);
                }

                //Initial percentile cut
                if ((ks == 0) && (pclip < 100) && (pclip > 0)) {
                    med = cpl_vector_get_median(val);
                    cpl_vector_subtract_scalar(val, med);
                    cpl_vector_power(val, 2);
                    cpl_vector_sqrt(val);
                    cpl_vector_sort(val, CPL_SORT_ASCENDING);
                    pk = round(pclip/100.0*(cpl_vector_get_size(val)-1));
                    sig = cpl_vector_get(val, pk) / kappa;
                } else {
                    if (!strcmp(compute_mode, "MEDIAN")){ //Median: sigma = 1.4826*MAD
                        cpl_vector_subtract_scalar(val, med);
                        cpl_vector_power(val, 2);
                        cpl_vector_sqrt(val);
                        sig = 1.4826 * cpl_vector_get_median(val);
                    } else {
                        sig=cpl_vector_get_stdev(val);
                    }
                }

                for (int i = 0 ; i < n_frames; i++) {
                    pimg_in = cpl_image_get_data_double(imagesDataShifted[i]);

                    lox = llx[i];
                    loy = lly[i];
                    upx = llx[i]+nx;
                    upy = lly[i]+ny;

                    // Do k-s clipping at each pixel
                    if ((y >= loy) && (y < upy) && (x >= lox) && (x < upx)) {
                        posx = x - lox;
                        posy = y - loy;
                        pos = posx+posy*nx;

                        if (!isnan(pimg_in[pos]) &&
                            (fabs(pimg_in[pos]) > DBL_ZERO_TOLERANCE) &&
                            (cpl_vector_get(*msk,i) != 0))
                        {
                            if(fabs((pimg_in[pos]-med)) > kappa*sig)
                            {
                                // clipping pixel here
                                pimg_in[pos] = NAN;
                                // agudo: added this in, hopefully with correct coordinates
                                cpl_image_reject(imagesDataShifted[i], posx+1, posy+1);
                                cpl_vector_set(*msk, i, 0);
                                nclip++;
                            }
                        }
                    }
                }
            }
            eris_ifu_free_vector(&val);
        } // end for(ks)
    } CATCH {
        eris_ifu_free_vector(&val);
        ret_err = cpl_error_get_code();
    }

    return ret_err;
}

/**
   @brief   copied from sinfo_compute_weight_average
   @name    eris_compute_weight_average
   @short compute weighted mean of shifted cubes
   @param z_min  minimum cube's plane processed
   @param z_max  maximum cube's plane processed
   @param ilx    input cube component x size
   @param ily    input cube component y size
   @param n_cubes: number of cubes in the list to merge
   @param  cubes: list of jittered cubes to mosaic
   @param  mergedCube: resulting merged cube containing the
                                      jittered cubes
   @param  mask: cube of the same size as combinedCube
                 containing 0 for blank (ZERO pixels) and
                 the summed integration times for
                 overlapping regions

   @param  tmpcubes: shifted list of jittered cubes to mosaic

   @param exptimes: exposure times array giving the time
                    in the same sequence as the cube list
   @param compute_mode: MEDIAN calculate median, otherwise the weighted mean.
   @doc calculate a weighted average using the exposure time of the
        single frames of the overlapping regions of the cubes
 */
cpl_error_code eris_ifu_combine_coadd( const int n_cubes,
                                       cpl_image **imgMergedCubeData,
                                       cpl_image **imgMergedCubeError,
                                       cpl_image *mergedImgDIT,
                                       cpl_image **imagesDataShifted,
                                       cpl_image **imagesErrorShifted,
                                       const double *exptimes,
                                       int *llx,
                                       int *lly,
                                       const char *compute_mode,
                                       const int nx_out,
                                       const int ny_out)
{
    int         nx_in                   = 0,
                ny_in                   = 0,
                posx                    = 0,
                posy                    = 0,
                ovr                     = 0;    //overlap count
    double      *pimgCubesDataShifted   = NULL,
                *pimgCubesErrorShifted  = NULL,
                *pimgMergedCubeData     = NULL,
                *pimgMergedCubeError    = NULL,
                *pmergedImgDIT          = NULL;
    cpl_vector  *val                    = NULL; // vector to compute median of an output pixel
    eris_ifu_vector *data_vec           = NULL,
                    *err_vec            = NULL;

    nx_in = cpl_image_get_size_x(imagesDataShifted[0]);
    ny_in = cpl_image_get_size_y(imagesDataShifted[0]);

    if (
        (nx_out != cpl_image_get_size_x(mergedImgDIT)) ||
        (ny_out != cpl_image_get_size_y(mergedImgDIT))
        )
    {
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_INPUT);
        return cpl_error_get_code();
    }

    *imgMergedCubeData   = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE);
    pimgMergedCubeData  = cpl_image_get_data_double(*imgMergedCubeData);

    *imgMergedCubeError  = cpl_image_new(nx_out, ny_out, CPL_TYPE_DOUBLE);
    pimgMergedCubeError = cpl_image_get_data_double(*imgMergedCubeError);

    pmergedImgDIT = cpl_image_get_data_double(mergedImgDIT);

    for (int y = 0; y < ny_out; y++) {
        for (int x = 0; x < nx_out; x++) {
            if (!strcmp(compute_mode, "MEDIAN")) {
                // median
                val = cpl_vector_new(n_cubes);
                ovr = 0;
            }

            data_vec = eris_ifu_vector_new(n_cubes);
            err_vec = eris_ifu_vector_new(n_cubes);

            for (int i = 0; i < n_cubes; i++) {
                if ((y >= lly[i]) && (y < lly[i]+ny_in) &&
                    (x >= llx[i]) && (x < llx[i]+nx_in))
                {
                    pimgCubesDataShifted  = cpl_image_get_data_double(imagesDataShifted[i]);
                    pimgCubesErrorShifted = cpl_image_get_data_double(imagesErrorShifted[i]);

                    posx = x - llx[i];
                    posy = y - lly[i];

                    if (!isnan(pimgCubesDataShifted[posx+posy*nx_in]) &&
                        (fabs(pimgCubesDataShifted[posx+posy*nx_in]) > DBL_ZERO_TOLERANCE) &&
                        (fabs(pmergedImgDIT[x+y*nx_out]) > DBL_ZERO_TOLERANCE))
                    {
                        // valid data value
                        if (!strcmp(compute_mode, "MEDIAN")) {
                            // median
                            cpl_vector_set(val, ovr, pimgCubesDataShifted[posx+posy*nx_in]);
                            ovr++;
                        } else {
                            // mean
                            pimgMergedCubeData[x+y*nx_out] += pimgCubesDataShifted[posx+posy*nx_in] * exptimes[i];
                        }
                        eris_ifu_vector_set(data_vec, i, pimgCubesDataShifted[posx+posy*nx_in]);
                        eris_ifu_vector_set(err_vec, i, pimgCubesErrorShifted[posx+posy*nx_in]);
                    } else {
                        // invalid data value
                        eris_ifu_vector_reject(data_vec, i);
                        eris_ifu_vector_reject(err_vec, i);
                    }
                } else {
                    // invalid data value
                    eris_ifu_vector_reject(data_vec, i);
                    eris_ifu_vector_reject(err_vec, i);
                }
            } // end for(i)

            // calculate error
            pimgMergedCubeError[x+y*nx_out] = eris_ifu_combine_calc_error(data_vec, err_vec, compute_mode);

            eris_ifu_vector_delete(data_vec); data_vec = NULL;
            eris_ifu_vector_delete(err_vec); err_vec = NULL;

            if (!strcmp(compute_mode, "MEDIAN")){
                // median
                if (ovr > 0) {
                    cpl_vector *tval = cpl_vector_extract(val, 0, ovr-1, 1);
                    pimgMergedCubeData[x+y*nx_out] = cpl_vector_get_median_const(tval);
                    eris_ifu_free_vector(&tval);
                }
                eris_ifu_free_vector(&val);
            } else {
                // mean
                pimgMergedCubeData[x+y*nx_out] /= pmergedImgDIT[x+y*nx_out];
            }
        } // end for x
    } // end for y

    return cpl_error_get_code();
}

int eris_ifu_combine_calc_contributions(cpl_image **imagesDataShifted,
                                        const int n_frames,
                                        const int* llx, const int* lly,
                                        const int x, const int y)
{
    int         n_contributes   = 0,
                pos             = 0,
                posx            = 0,
                posy            = 0,
                lox             = 0,
                loy             = 0,
                upx             = 0,
                upy             = 0;
    double      *pimg           = NULL;
    cpl_size    nx              = 0,
                ny              = 0;

    nx = cpl_image_get_size_x(imagesDataShifted[0]);
    ny = cpl_image_get_size_y(imagesDataShifted[0]);

    // Count the number of shifted input cubes to be used for
    // each overlapping point at x,y
    for (int i = 0; i < n_frames; i++) {
        pimg = cpl_image_get_data_double(imagesDataShifted[i]);

        lox = llx[i];
        loy = lly[i];
        upx = llx[i] + nx;
        upy = lly[i] + ny;

        if ((y >= loy) && (y < upy) && (x >= lox) && (x < upx)) {
            posx = x - lox;
            posy = y - loy;
            pos = posx+posy*nx;

            if (!isnan(pimg[pos]) && (fabs(pimg[pos]) > DBL_ZERO_TOLERANCE)) {
                n_contributes++;
            }
        }
    }

    return n_contributes;
}

/**
   @brief Subtract spatial median to each cube plane
   @param inp

   Copied from sinfo_new_sinfoni_correct_median_it
 */
cpl_error_code eris_ifu_combine_subtract_background(cpl_image *img,
                                                    bool      *warn)
{
    double          local_median    = 0.;
    cpl_error_code  err             = CPL_ERROR_NONE;

    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);

    TRY {
        BRK_IF_ERROR(
            cpl_image_reject_value(img, CPL_VALUE_NOTFINITE));

        local_median = cpl_image_get_median(img);
        if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
            local_median = NAN;
            RECOVER();
        }
        CHECK_ERROR_STATE();

        if (!isnan(local_median)) {
            BRK_IF_ERROR(
                cpl_image_subtract_scalar(img, local_median));
            *warn = FALSE;
        }  else {
            *warn = TRUE;
        }
    } CATCH {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }
    return err;
}

/**
 * @brief eris_ifu_combine_get_xy_min_max
 * @param nframes  Input number of values
 * @param offsetx  Input offset list
 * @param offsety  Input offset list
 * @param min_offx Minimum offset in offsetx
 * @param max_offx Maximum offset in offsetx
 * @param min_offy Minimum offset in offsety
 * @param max_offy Maximum offset in offsety
 *
 * Copied from sinfo_get_xy_min_max
 */
void eris_ifu_combine_get_xy_min_max(const int nframes,
                                     const float *offsetx, const float *offsety,
                                     float *min_offx, float *max_offx,
                                     float *min_offy, float *max_offy)
{
    float offx = 0.,
        offy = 0.;

    for (int n = 0; n < nframes; n++) {
        offx = offsetx[n];
        offy = offsety[n];
        if (n == 0) {
            *min_offx = offx;
            *min_offy = offy;
            *max_offx = offx;
            *max_offy = offy;
        } else {
            if(offx > *max_offx)
                *max_offx=offx;
            if(offy > *max_offy)
                *max_offy=offy;
            if(offx < *min_offx)
                *min_offx=offx;
            if(offy < *min_offy)
                *min_offy=offy;
        }
    }
}

/**
   @brief  Determines the nearest integer to a specified real value
   @name   eris_ifu_combine_nearest_int()
   @param  x Double value to convert to int
   @return Nearest integer to specified real value

   Copied from sinfo_new_nint()
 */
int eris_ifu_combine_nearest_int(const double x)
{
    int k = x;

    if (x >= 0.) {
        if ((x - (double)k) <= 0.5) {
            return k;
        } else {
            return k + 1;
        }
    } else {
        if ((x - (double)k) <= -0.5) {
            return k - 1;
        } else {
            return k;
        }
    }
}

/**
  @name     eris_ifu_combine_shift_image
  @memo     Shift an image by a given (non-integer) 2d offset
  @param    img_in        Image to shift.
  @param    shift_x       Shift in x.
  @param    shift_y       Shift in y.
  @param    interp_kernel Interpolation kernel to use
  @return   Newly allocated image with shifted input image

  @see      sinfo_generate_interpolation_kernel
  @doc

  This function is a conversion to CPL of the ECLIPSE function shift_image()
  but slightly changed. If a blank (ZERO) pixel appears the blank pixel is
  shifted but preserved as blank.
  If a blank (ZERO) pixel appears within the interpolation kernel the blank
  pixel is set to 0.

  This function shifts an image by a non-integer offset, using
  interpolation. You can either generate an interpolation kernel once and
  pass it to this function, or let it generate a default kernel. In the
  former case, use sinfo_generate_interpolation_kernel() to generate an
  appropriate kernel. In the latter case, pass NULL as last argument. A
  default interpolation kernel is then generated then discarded before this
  function returns.

  The returned image is a newly allocated object, it must be deallocated
  using cpl_image_delete().
*/
cpl_image* eris_ifu_combine_shift_image(const cpl_image *img_in,
                                        const double    shift_x,
                                        const double    shift_y,
                                        const double    *kernel)
{
    cpl_image       *img_shifted    = NULL;
    cpl_size        nx              = 0,
                    ny              = 0;
    int             samples         = KERNEL_SAMPLES,
                    mid             = samples / 2,
                    px              = 0,
                    py              = 0,
                    pos             = 0,
                    tabx            = 0,
                    taby            = 0;
    double          norm            = 0.,
                    value           = 0.,
                    fx              = 0.,
                    rx              = 0.,
                    fy              = 0.,
                    ry              = 0.,
                    *first_pass     = NULL,
                    *pimg_shifted   = NULL;
    const double    *pimg_in        = NULL;

    cpl_ensure(img_in != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(kernel != NULL, CPL_ERROR_NULL_INPUT, NULL);

    /* Shifting by a zero offset returns a copy of the input image */
    if ((fabs(shift_x) < 1e-2) && (fabs(shift_y) < 1e-2))
        return cpl_image_duplicate(img_in);

    nx = cpl_image_get_size_x(img_in);
    ny = cpl_image_get_size_y(img_in);

    pimg_in = cpl_image_get_data_double_const(img_in);
    if (pimg_in != NULL) {
        first_pass = cpl_calloc(nx, ny*sizeof(double));
        img_shifted = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        pimg_shifted = cpl_image_get_data_double(img_shifted);
        for (int y = 0; y < (int)ny; y++) {
            for (int x = 1; x < (int)nx-2; x++) {
                fx = (double)x - shift_x;
                px = (int)fx;
                rx = fx - (double)px;
                pos = px + y * (int)nx;

                if ((px>1) && (px<(nx-3))) {
                    tabx = (int)(fabs(mid * rx));

                    // Sum up over 4 closest pixel values,
                    // weighted by interpolation kernel values
                    value = pimg_in[pos-1] * kernel[mid+tabx] +
                            pimg_in[pos]   * kernel[tabx] +
                            pimg_in[pos+1] * kernel[mid-tabx] +
                            pimg_in[pos+2] * kernel[samples-tabx-1];

                    // Also sum up interpolation kernel coefficients
                    // for further normalization
                    norm = kernel[mid+tabx] +
                           kernel[tabx] +
                           kernel[mid-tabx] +
                           kernel[samples-tabx-1];

                    if (fabs(norm) > 1e-4) {
                        value /= norm;
                    }
                } else {
                    value = 0.;
                }
                // There may be a problem of rounding here if pixelvalue
                // has not enough bits to sustain the accuracy.
                first_pass[x+y*nx] = value;
            }
        }

        for (int x = 0; x < (int)nx; x++) {
            for (int y = 1; y < (int)ny-3; y++) {
                fy = (double)y - shift_y;
                py = (int)fy ;
                ry = fy - (double)py ;
                pos = x + py * nx ;

                if ((py > 1) && (py < ((int)ny-2))) {
                    taby = (int)(fabs((double)mid * ry));

                    // Sum up over 4 closest pixel values,
                    // weighted by interpolation kernel values
                    value = first_pass[pos-nx] * kernel[mid+taby] +
                            first_pass[pos] * kernel[taby] +
                            first_pass[pos+nx] * kernel[mid-taby] +
                            first_pass[pos+2*nx]*kernel[samples-taby-1];

                    // Also sum up interpolation kernel coefficients
                    // for further normalization
                    norm = kernel[mid+taby] +
                           kernel[taby] +
                           kernel[mid-taby] +
                           kernel[samples-taby-1];

                    if (fabs(norm) > 1e-4) {
                        value /= norm;
                    }
                } else {
                    value = 0.0;
                }
                pimg_shifted[x+y*nx] = value;
            }
        }
    } else {
        cpl_error_set(cpl_func, CPL_ERROR_ILLEGAL_OUTPUT);
        cpl_msg_error(cpl_func, "Cannot get a data from an image");
        return NULL;
    }

    // free memory
    cpl_free(first_pass); first_pass = NULL;

    return img_shifted;
}

/**
 * @brief eris_ifu_combine_convert_0_to_NaN_img
 * @param img all 0 in the image are converted to NaN
 */
void eris_ifu_combine_convert_0_to_NaN_img(cpl_image *img)
{
    cpl_size    nx      = 0,
                ny      = 0;
    double      *pimg   = NULL;

    if (img != NULL) {
        pimg = cpl_image_get_data_double(img);

        nx = cpl_image_get_size_x(img);
        ny = cpl_image_get_size_y(img);

        for (int i = 0 ; i < nx*ny; i++) {
            if (fabs(pimg[i]) < DBL_ZERO_TOLERANCE) {
                pimg[i] = NAN;
            }
        }
    }
}

double eris_ifu_combine_calc_error(eris_ifu_vector *data_vec,
                                   eris_ifu_vector *err_vec,
                                   const char *compute_mode)
{
    double  std_err     = 0.,
            std_dev     = 0.;
    int     vec_size    = 0;
    cpl_ensure_code(data_vec != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(err_vec  != NULL, CPL_ERROR_NULL_INPUT);

    vec_size = eris_ifu_vector_count_non_rejected(data_vec);

    if (vec_size > 2) {
        // recalculate error from input data (regardless if there is
        // input error or not)

        if (!strcmp(compute_mode, "MEDIAN")) {
            std_dev = eris_ifu_vector_get_stdev_median(data_vec);
        } else {
            std_dev = eris_ifu_vector_get_stdev(data_vec);
        }
        std_err = std_dev / sqrt(vec_size);
    } else {
        vec_size = eris_ifu_vector_count_non_rejected(err_vec);
        /* error propagation from input error*/
        if (vec_size == 1) {
            int i = 0;
            while (eris_ifu_vector_is_rejected(err_vec, i)) {
                i++;
            }
            std_err = eris_ifu_vector_get(err_vec, i);
        } else if (vec_size == 2) {
            double tmp_dbl = 0.;
            int i = 0;
            while (eris_ifu_vector_is_rejected(err_vec, i)) {
                i++;
            }

            int j = i+1;
            while (eris_ifu_vector_is_rejected(err_vec, j)) {
                j++;
            }

            tmp_dbl = sqrt(pow(eris_ifu_vector_get(err_vec, i), 2) +
                           pow(eris_ifu_vector_get(err_vec, j), 2));

            std_dev = tmp_dbl / 2;
            std_err = std_dev / sqrt(2);
        }
    }
    return std_err;
}

/* --- simple shifting from KMOS pipeline --- */

enum boundary_mode {
    NATURAL,
    EXACT,
    ESTIMATED1,
    ESTIMATED2
};

double** matrix(int nrow, int ncol) {
    double **matrix = (double**)cpl_malloc((size_t)((nrow)*sizeof(double*)));

    for(int i = 0; i < nrow; i++) {
        matrix[i] = (double *) cpl_malloc((size_t)((ncol)*sizeof(double)));
    }

    return matrix;
}

void free_matrix(double **matrix, int nrow) {
    for(int i = 0; i < nrow; i++) {
        cpl_free(matrix[i]); matrix[i] = NULL;
    }
    cpl_free(matrix); matrix = NULL;
}

/**
    @brief
        Rejects NaN values in the internal badpixelmask.

    @param img The image to reject.

    @return CPL_ERROR_NONE or the relevant _cpl_error_code_ on error

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT     if @c img  is NULL.
*/
cpl_error_code eris_ifu_reject_nan(cpl_image *img)
{
    cpl_error_code  err     = CPL_ERROR_NONE;
    int             nx      = 0,
                    ny      = 0,
                    is_rej  = 0;
    float           tmp     = 0.0;

    cpl_ensure_code(img != NULL, CPL_ERROR_NULL_INPUT);

    TRY {

        nx = cpl_image_get_size_x(img);
        ny = cpl_image_get_size_y(img);
        CHECK_ERROR_STATE();

        for (int ix = 1; ix <= nx; ix++) {
            for (int iy = 1; iy <= ny; iy++) {
                tmp = cpl_image_get(img, ix, iy, &is_rej);
                CHECK_ERROR_STATE();

                if (!is_rej && isnan(tmp)) {
                    BRK_IF_ERROR(
                        cpl_image_reject(img, ix, iy));
                }
            }
        }
    } CATCH {
        err = cpl_error_get_code();
    }

    return err;
}

/**
    @brief  Shifts each image of an image cube
    @param  inputCube      List of images
    @param  xshift         Fraction of pixel (< 1) to be shifted in x direction
    @param  yshift         Fraction of pixel (< 1) to be shifted in y direction
    @param  method         Interpolation method: either "BCS" for bicubic splines
                           or "NN" for nearest neighbor
    @param  extrapolation  How to handle extrapolation, see description

    @return An image cube with the shifted images

    was kmclipm_shift() in KMOS pipeline

    This function is written as recipe as defined in the ESO Document
    VLT-TRE-KMO-146611-003 (KMOS Data Reduction Library Design), it will also be
    used in the Data Reduction Pipeline.

    Extrapolation is done in following ways as requested by the "extrapolation"
    parameter:
        - NONE_NANS: <br>
          no extrapolation will be done, points outside the input images will be
          set to NaN
        - NONE_CLIPPING <br>
          no extrapolation will be done, points outside the input images will be
          removed, the output image will be smaller
        - BCS_NATURAL: <br>
          only valid for the "BCS" interpolation, which will be of type natural,
          i.e. at the image edge the first derivate is assumed to be zero
        - BCS_ESTIMATED <br>
          only valid for the "BCS" interpolation, the second derivate at the
          image egde will be estimated by interpolation using the last three
          points.

    The returned imagelist has to be deallocated with @c cpl_imagelist_delete().

    Possible _cpl_error_code_ set in this function:

    @li CPL_ERROR_ILLEGAL_INPUT if the input cube is NULL
    @li CPL_ERROR_ILLEGAL_INPUT if the input cube is empty
    @li CPL_ERROR_ILLEGAL_INPUT if either xshift or yshift is greater than 1
    @li CPL_ERROR_ILLEGAL_INPUT if interpolation method is neither "BCS" nore "NN
    @li CPL_ERROR_ILLEGAL_INPUT if extrapolation type is unknown

    @em FUTURE: First an empty rectilinear cube of the required dimensions is
    created and then the values from the input data are interpolated, conserving
    the flux from the 2D frame into the 3D cube if requested.The errors are
    propagated into a new noise cube if the input noise map is provided.
*/
cpl_image* eris_ifu_combine_shift_image_kmos(const cpl_image *img_in,
                                            double xshift,
                                            double yshift,
                                            const char *method,
                                            const enum extrapolationType extrapolation)
{
    const double    *pimg_in    = NULL;
    double          **array_in  = NULL,
                    **array_out = NULL,
                    *pimg_out   = NULL;
    cpl_image       *img_out    = NULL;
    int             xdim        = 0,
                    ydim        = 0,
                    xdim_new    = 0,
                    ydim_new    = 0;

    cpl_ensure(img_in != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure((extrapolation == NONE_NANS) ||
               (extrapolation == NONE_CLIPPING)/* ||
               (extrapolation == BCS_NATURAL) ||
               (extrapolation == BCS_ESTIMATED)*/, CPL_ERROR_ILLEGAL_INPUT, NULL);

    TRY {
        if ((xshift == 0.0) && (yshift == 0.0)) {
            // duplicate image
            BRK_IF_NULL(
                img_out = cpl_image_duplicate(img_in));
        } else {
            xdim = cpl_image_get_size_x(img_in);
            ydim = cpl_image_get_size_y(img_in);

                pimg_in = cpl_image_get_data_double_const(img_in);

                array_in = matrix(xdim, ydim);
                for (int j = 0; j < ydim ; j++) {
                    for (int i = 0; i < xdim ; i++) {
                        if (isnan(pimg_in[i + j*xdim]))
                            /* agudo: fix because of NaN-input */
                            array_in[i][j] = 0;
                        else
                            array_in[i][j] = pimg_in[i + j*xdim];
                    }
                }

                /*if (strcmp(method, "BCS") == 0) {
                    xdim_new = xdim;
                    ydim_new = ydim;
                    xstart_new = xshift;
                    ystart_new = yshift;

                    enum boundary_mode boundaryMode = NATURAL;
                    if (extrapolation == BCS_ESTIMATED) {
                        boundaryMode = ESTIMATED2;
                    }

                    array_out = bicubicspline_reg_reg(
                        xdim, 0.0, 1.0,
                        ydim, 0.0, 1.0,
                        array_in,
                        xdim_new, xstart_new, 1.0,
                        ydim_new, ystart_new, 1.0,
                        boundaryMode);
                } else */if (strcmp(method, "NN") == 0) {
                    if (fabs(xshift) > 1.0 || fabs(yshift) > 1.0) {
                        BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                           "Neither xshift nor yshift are allowed to "
                                           "be greater than 1 pixel for NN");
                    }
                    int xoffset = 0,
                        yoffset = 0;
                    if (xshift > 0.5) {
                        xoffset = 1;
                    } else if (xshift < -0.5) {
                        xoffset = -1;
                    } else {
                        xoffset = 0;
                    }
                    if (yshift > 0.5) {
                        yoffset = 1;
                    } else if (yshift < -0.5) {
                        yoffset = -1;
                    } else {
                        yoffset = 0;
                    }
                    array_out = matrix(xdim,ydim);
                    for (int j = 0; j < ydim; j++) {
                        for (int i = 0; i < xdim; i++) {
                            int xix = i+xoffset,
                                yix = j+yoffset;
                            if (xix < 0 || xix >= xdim || yix < 0 || yix >= ydim) {
                                array_out[i][j] = NAN;
                            } else {
                                array_out[i][j] = array_in[i+xoffset][j+yoffset];
                            }
                        }
                    }
                } else {
                    BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                       "Unknown value for interpolation method");
                }
                free_matrix(array_in, xdim);

                switch (extrapolation) {
//                case BCS_NATURAL:
//                case BCS_ESTIMATED:
//                    BRK_IF_NULL(
//                        image_out = cpl_malloc(xdim_new * ydim_new * sizeof(double)));
//                    for (int j = 0; j < ydim_new; j++) {
//                        for (int i = 0; i < xdim_new; i++) {
//                            image_out[i+j*xdim_new] = array_out[i][j];
//                        }
//                    }
//                    break;
                case NONE_CLIPPING:
                    xdim_new = xdim - lrintf(ceill(fabsl(xshift)));
                    ydim_new = xdim - lrintf(ceill(fabsl(yshift)));
                    BRK_IF_NULL(
                        pimg_out = cpl_malloc(xdim_new * ydim_new * sizeof(double)));
                    int xmin = - floorl(xshift);
                    if (xmin < 0)
                        xmin = 0;
                    int xmax = xdim - ceill(xshift);
                    if (xmax > xdim)
                        xmax = xdim;
                    int ymin = - floorl(yshift);
                    if (ymin < 0)
                        ymin = 0;
                    int ymax = ydim - ceill(yshift);
                    if (ymax > ydim)
                        ymax = ydim;
                    for (int j = ymin; j < ymax; j++) {
                        for (int i = xmin; i < xmax; i++) {
                            pimg_out[(i - xmin) + (j - ymin) * xdim_new] = array_out[i][j];
                        }
                    }
                    break;
                case NONE_NANS:
                    xdim_new = xdim;
                    ydim_new = ydim;
                    BRK_IF_NULL(
                        pimg_out = cpl_malloc(xdim * ydim * sizeof(double)));
                    float xxmin = 0 - xshift;
                    float xxmax = xdim - 1 - xshift;
                    float yymin = 0 - yshift;
                    float yymax = ydim - 1 - yshift;
                    for (int j = 0; j < ydim; j++) {
                        for (int i = 0; i < xdim; i++) {
                            if ((i < xxmin ) || (i > xxmax) ||
                                (j < yymin ) || (j > yymax))
                            {
                                pimg_out[i+j*xdim] = NAN;
                            } else {
                                pimg_out[i+j*xdim] = array_out[i][j];
                            }
                        }
                    }
                    break;
                default:
                    BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                       "Unknown value for extrapolation type");
                }

                free_matrix(array_out, xdim);
                img_out = cpl_image_wrap_double(xdim_new, ydim_new, pimg_out);

                /* reject NaN values here */
                BRK_IF_ERROR(
                    eris_ifu_reject_nan(img_out));
        }
    }
    CATCH
    {
        CATCH_MSGS();
        eris_ifu_free_image(&img_out);
    }

    return img_out;
}

cpl_error_code eris_ifu_combine_read_image_planes(const cpl_frameset *frameset,
                                                  cpl_image **imagesData,
                                                  cpl_image **imagesError,
                                                  int z,
                                                  int edge_trim,
                                                  bool subtract_background)
{
    cpl_size            nframes         = 0,
                        llx             = 0,
                        lly             = 0,
                        urx             = 0,
                        ury             = 0;
    cpl_error_code      err             = CPL_ERROR_NONE;
    const cpl_frame     *frame          = NULL;
    cpl_mask            *tmp_mask       = NULL;
    cpl_image           *cube_qual_tmp  = NULL;
    cpl_propertylist    *plist          = NULL;
    const char          *name           = NULL;
    int                 tmp_size_x      = 0,
                        tmp_size_y      = 0;
    bool                warn            = FALSE,
                        warned          = FALSE;

    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);

    TRY {
        nframes = cpl_frameset_get_size(frameset);

        /* Read in cubes */
        for (cpl_size n = 0; n < nframes; n++) {
            BRK_IF_NULL(
                frame = cpl_frameset_get_position_const(frameset, n));
            BRK_IF_NULL(
                name = cpl_frame_get_filename(frame));
            if (edge_trim > 0) {
                BRK_IF_NULL(
                    plist = cpl_propertylist_load(name, 1));
                tmp_size_x = eris_pfits_get_naxis1(plist);
                tmp_size_y = eris_pfits_get_naxis2(plist);
                CHECK_ERROR_STATE();

                eris_ifu_free_propertylist(&plist);
                llx = 1 + edge_trim;
                lly = 1 + edge_trim;
                urx = tmp_size_x - edge_trim;
                ury = tmp_size_y - edge_trim;

                BRK_IF_NULL(
                    imagesData[n]  = cpl_image_load_window(name, CPL_TYPE_DOUBLE, z, 1, llx, lly, urx, ury));
                BRK_IF_NULL(
                    imagesError[n] = cpl_image_load_window(name, CPL_TYPE_DOUBLE, z, 2, llx, lly, urx, ury));
                BRK_IF_NULL(
                    cube_qual_tmp = cpl_image_load_window(name, CPL_TYPE_DOUBLE, z, 3, llx, lly, urx, ury));
            }
            else{
                BRK_IF_NULL(
                    imagesData[n]  = cpl_image_load(name, CPL_TYPE_DOUBLE, z, 1));
                BRK_IF_NULL(
                    imagesError[n] = cpl_image_load(name, CPL_TYPE_DOUBLE, z, 2));
                BRK_IF_NULL(
                    cube_qual_tmp = cpl_image_load(name, CPL_TYPE_DOUBLE, z, 3));
            }

            /* Reject mask */
            BRK_IF_NULL(
                tmp_mask = cpl_mask_threshold_image_create(cube_qual_tmp, 0, INT_MAX));
            BRK_IF_ERROR(
                cpl_image_reject_from_mask(imagesData[n], tmp_mask));
            BRK_IF_ERROR(
                cpl_image_reject_from_mask(imagesError[n], tmp_mask));
            eris_ifu_free_mask(&tmp_mask);
            eris_ifu_free_image(&cube_qual_tmp);

            if (subtract_background) {
                BRK_IF_ERROR(
                    eris_ifu_combine_subtract_background(imagesData[n], &warn));
                if (warn && !warned) {
                    warned = TRUE;
                    cpl_msg_warning(cpl_func, "      local_median is NAN in %dth plane", z);
                }
            }
            CHECK_ERROR_STATE();
        } // end for nframes
    } CATCH {
        CATCH_MSGS();
        err = cpl_error_get_code();
    }
    return err;
}

/**
 * @brief eris_ifu_combine_min_cube_size
 * @param fs
 * @return The minimum number of planes in all cubes
 *
 * It is checked as well if CRPIX3, CRVAL3 and CD3_3 are all the same. If these
 * conditions are met, it is safe just to omit any extra planes on individual cubes
 * at the top/far end.
 */
int eris_ifu_combine_min_cube_size(const cpl_frameset *fs) {
    int                     naxis3      = 0,
                            naxis3_tmp  = 0;
    double                  crval3      = 0.,
                            crpix3      = 0,
                            cd3_3       = 0.,
                            crval3_tmp  = 0.,
                            crpix3_tmp  = 0.,
                            cd3_3_tmp   = 0.;
    const char              *name   = NULL;
    const cpl_frame         *fr     = NULL;
    cpl_propertylist        *pl     = NULL;

    cpl_ensure(fs != NULL, CPL_ERROR_NULL_INPUT, -1);

    TRY {
        // initialize with 1st frame
        cpl_frameset_iterator *iter = NULL;
        BRK_IF_NULL(
            iter = cpl_frameset_iterator_new(fs));
        fr = cpl_frameset_iterator_get(iter);
        CHECK_ERROR_STATE();
        BRK_IF_NULL(
            name = cpl_frame_get_filename(fr));

        // load header of data extension
        BRK_IF_NULL(
            pl = cpl_propertylist_load(name, 1));
        naxis3 = cpl_propertylist_get_int(   pl, NAXIS3);
        crpix3 = cpl_propertylist_get_double(pl, CRPIX3);
        crval3 = cpl_propertylist_get_double(pl, CRVAL3);
        cd3_3  = cpl_propertylist_get_double(pl, CD3_3);
        CHECK_ERROR_STATE();
        eris_ifu_free_propertylist(&pl);

        BRK_IF_ERROR(
            cpl_frameset_iterator_advance(iter, 1));
        fr = cpl_frameset_iterator_get(iter);
        CHECK_ERROR_STATE();

        while (fr != NULL) {
            BRK_IF_NULL(
                name = cpl_frame_get_filename(fr));
            // load header of data extension
            BRK_IF_NULL(
                pl = cpl_propertylist_load(name, 1));

            naxis3_tmp = cpl_propertylist_get_int(   pl, NAXIS3);
            crpix3_tmp = cpl_propertylist_get_double(pl, CRPIX3);
            crval3_tmp = cpl_propertylist_get_double(pl, CRVAL3);
            cd3_3_tmp  = cpl_propertylist_get_double(pl, CD3_3);
            CHECK_ERROR_STATE();
            eris_ifu_free_propertylist(&pl);

            if ((fabs(crpix3 - crpix3_tmp) > DBL_ZERO_TOLERANCE) ||
                (fabs(crval3 - crval3_tmp) > DBL_ZERO_TOLERANCE) ||
                (fabs(cd3_3 - cd3_3_tmp)   > DBL_ZERO_TOLERANCE))
            {
                BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                                   "WCS not matching for all frames!");
            }

            if (naxis3_tmp < naxis3) {
                naxis3 = naxis3_tmp;
            }

            BRK_IF_ERROR(
                cpl_frameset_iterator_advance(iter, 1));
            fr = cpl_frameset_iterator_get(iter);
            CHECK_ERROR_STATE();
        } // end while
    } CATCH {
        CATCH_MSGS();
        naxis3 = -1;
    }
    return naxis3;
}
