/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cpl.h"

#include <stdbool.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_interp2d.h>
#include <gsl/gsl_spline2d.h>

#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_dfs.h"
#include "eris_ifu_extract_spec_static.h"
#include "eris_ifu_jitter_static.h"
#include "eris_ifu_wavecal_static.h"
#include "eris_ifu_distortion_static.h"
#include "eris_ifu_lambda_corr.h"
#include "eris_ifu_sky_tweak.h"
#include "eris_ifu_efficiency_response.h"
#include "eris_ifu_lambda_corr.h"
#include "eris_utils.h"
#include "eris_pfits.h"
#include "skycorr/eris_ifu_skycorr.h"
#include "skycorr/sc_skycorr.h"
#include "skycorr/sc_lines.h"

///*static */void
//eris_ifu_jitter_free_exposureEntry(struct exposureEntry* exp_entry){

//	if(exp_entry != NULL) {
//		if(exp_entry->frame != NULL) cpl_frame_delete(exp_entry->frame);
//		if(exp_entry->hdr != NULL) cpl_propertylist_delete(exp_entry->hdr);
//		if(exp_entry->rawImage != NULL) hdrl_image_delete(exp_entry->rawImage);
//		if(exp_entry->cube != NULL) hdrl_imagelist_delete(exp_entry->cube);
//		if(exp_entry->cubeBpm != NULL) cpl_imagelist_delete(exp_entry->cubeBpm);
//		if(exp_entry->cubeHdr != NULL) cpl_propertylist_delete(exp_entry->cubeHdr);
//		if(exp_entry->badPixelMask != NULL) cpl_mask_delete(exp_entry->badPixelMask);
//		if(exp_entry->darkSubtrImage != NULL) hdrl_image_delete(exp_entry->darkSubtrImage);
//		if(exp_entry->skySubtrImage != NULL) hdrl_image_delete(exp_entry->skySubtrImage);
//		exp_entry = NULL;
//	};
//	return;
//}

#define ERIS_IFU_SKIP_SKY_CORR_PLANES 75
void eris_ifu_jitter_free_sofStruct(struct sofStruct* sof_struct){

	/* The following commented lines of code creates seg faults */
	if(sof_struct != NULL) {
		if(sof_struct->exposureTable != NULL) {
			for(cpl_size i = 0; i < sof_struct->exposureTableCnt; i++) {
				if (sof_struct->exposureTable != NULL) {

					if(sof_struct->exposureTable[i].frame != NULL) {
						//cpl_frame_delete(sof_struct->exposureTable[i].frame);
					}
					if(sof_struct->exposureTable[i].hdr != NULL) {
						cpl_propertylist_delete(sof_struct->exposureTable[i].hdr);
					}
					if(sof_struct->exposureTable[i].rawImage != NULL) {
						//hdrl_image_delete(sof_struct->exposureTable[i].rawImage);
					}
					if(sof_struct->exposureTable[i].cube != NULL){
						hdrl_imagelist_delete(sof_struct->exposureTable[i].cube);
					}
					if(sof_struct->exposureTable[i].cubeBpm  != NULL){
						cpl_imagelist_delete(sof_struct->exposureTable[i].cubeBpm);
					}
					if(sof_struct->exposureTable[i].cubeHdr != NULL) {
						cpl_propertylist_delete(sof_struct->exposureTable[i].cubeHdr);
					}
					if(sof_struct->exposureTable[i].badPixelMask != NULL) {
						cpl_mask_delete(sof_struct->exposureTable[i].badPixelMask);
					}
					if(sof_struct->exposureTable[i].darkSubtrImage != NULL){
						hdrl_image_delete(sof_struct->exposureTable[i].darkSubtrImage);
					}
					if(sof_struct->exposureTable[i].skySubtrImage != NULL){
						hdrl_image_delete(sof_struct->exposureTable[i].skySubtrImage);
					}

				}
			}
			//eris_ifu_jitter_free_exposureEntry(sof_struct->exposureTable);
		}
		if(sof_struct->masterDark != NULL) hdrl_image_delete(sof_struct->masterDark);
		if(sof_struct->masterFlat != NULL) hdrl_image_delete(sof_struct->masterFlat);
		if(sof_struct->waveMap != NULL) cpl_image_delete(sof_struct->waveMap);
		if(sof_struct->badPixelMask != NULL) cpl_mask_delete(sof_struct->badPixelMask);
		if(sof_struct->distortion != NULL) {
                    for (int sx = 0; sx < SLITLET_CNT; sx++) {
                        cpl_polynomial_delete(sof_struct->distortion[sx]);
                    }
                    cpl_free(sof_struct->distortion);
                }
		if(sof_struct->dqi != NULL) {
			cpl_image_delete(sof_struct->dqi);
		}
		if(sof_struct->oh_ref_peaks != NULL) cpl_vector_delete(sof_struct->oh_ref_peaks);
		if(sof_struct->oh_ref_frame != NULL) cpl_frame_delete(sof_struct->oh_ref_frame);
		if(sof_struct->borders != NULL) cpl_table_delete(sof_struct->borders);
		if(sof_struct->poly_u != NULL) cpl_polynomial_delete(sof_struct->poly_u);
		if(sof_struct->poly_v != NULL) cpl_polynomial_delete(sof_struct->poly_v);
		if(sof_struct->distances != NULL) cpl_vector_delete(sof_struct->distances);
		if(sof_struct->positions != NULL) cpl_bivector_delete(sof_struct->positions);
		sof_struct = NULL;
	}
	return;
}


void
eris_ifu_jitter_get_cube_type_string(cubeType type)
{

	switch(type) {
	case OBJECT_CUBE:
		cpl_msg_info(cpl_func,"type: OBJECT_CUBE");
		break;
	case STD_CUBE:
		cpl_msg_info(cpl_func,"type: STD_CUBE");
		break;
	case STD_FLUX_CUBE:
		cpl_msg_info(cpl_func,"type: STD_FLUX_CUBE");
		break;
	case STD_CUBE_NOFLAT:
		cpl_msg_info(cpl_func,"type: STD_CUBE_NOFLAT");
		break;
	case STD_FLUX_CUBE_NOFLAT:
		cpl_msg_info(cpl_func,"type: STD_FLUX_CUBE_NOFLAT");
		break;
	case PSF_CUBE:
		cpl_msg_info(cpl_func,"type: PSF_CUBE");
		break;
	case SKY_OBJECT_CUBE:
		cpl_msg_info(cpl_func,"type: SKY_OBJECT_CUBE");
		break;
	case SKY_STD_CUBE:
		cpl_msg_info(cpl_func,"type: SKY_STD_CUBE");
		break;
	case SKY_STD_FLUX_CUBE:
		cpl_msg_info(cpl_func,"type: SKY_STD_FLUX_CUBE");
		break;
	case SKY_PSF_CUBE:
		cpl_msg_info(cpl_func,"type: SKY_PSF_CUBE");
		break;
	case TWEAKED_CUBE:
		cpl_msg_info(cpl_func,"type: TWEAKED_CUBE");
		break;
	case DAR_CUBE:
		cpl_msg_info(cpl_func,"type: DAR_CUBE");
		break;
	case JITTER_CUBE:
		cpl_msg_info(cpl_func,"type: JITTER_CUBE");
		break;
	case BPM_CUBE:
		cpl_msg_info(cpl_func,"type: BPM_CUBE");
		break;
	case OBJECT_CUBE_COADD:
		cpl_msg_info(cpl_func,"type: OBJECT_CUBE_COADD");
		break;
	case STD_CUBE_COADD:
		cpl_msg_info(cpl_func,"type: STD_CUBE_COADD");
		break;
	case PSF_CUBE_COADD:
		cpl_msg_info(cpl_func,"type: PSF_CUBE_COADD");
		break;
	case STD_FLUX_CUBE_COADD:
		cpl_msg_info(cpl_func,"type: STD_FLUX_CUBE_COADD");
		break;
	case DAR_STD_FLUX_CUBE:
		cpl_msg_info(cpl_func,"type: DAR_STD_FLUX_CUBE");
		break;
	case TWEAKED_CUBE_COADD:
	    cpl_msg_info(cpl_func,"type: TWEAKED_CUBE_COADD");
	    break;
    case DAR_CUBE_COADD:
        cpl_msg_info(cpl_func,"type: DAR_CUBE_COADD");
        break;
	case STD_FLUX_CUBE_COADD_NOFLAT:
		cpl_msg_info(cpl_func,"type: STD_FLUX_CUBE_COADD_NOFLAT");
		break;
	case STD_CUBE_COADD_NOFLAT:
		cpl_msg_info(cpl_func,"type: STD_CUBE_COADD_NOFLAT");
		break;
	default:
		cpl_msg_info(cpl_func,"type: DEFAULT %d",type);
		break;
	}

	return;
}
cubeType
eris_ifu_jitter_get_coadd_obj_type(cubeType type)
{
	cubeType coadd_type;
	switch(type) {
	case OBJECT_CUBE:
		coadd_type = OBJECT_CUBE_COADD;
		break;
	case OBJECT_CUBE_COADD:
			coadd_type = OBJECT_CUBE_COADD;
			break;
	case DAR_CUBE:
			coadd_type = DAR_CUBE_COADD;
			break;
	case STD_CUBE:
		coadd_type = STD_CUBE_COADD;
		break;
	case STD_CUBE_COADD:
		coadd_type = STD_CUBE_COADD;
		break;
	case STD_FLUX_CUBE:
		coadd_type = STD_FLUX_CUBE_COADD;
		break;
	case STD_FLUX_CUBE_COADD:
		coadd_type = STD_FLUX_CUBE_COADD;
		break;
	case STD_FLUX_CUBE_COADD_NOFLAT:
		coadd_type = STD_FLUX_CUBE_COADD_NOFLAT;
		break;
	case STD_CUBE_NOFLAT:
		coadd_type = STD_CUBE_COADD_NOFLAT;
		break;
	case STD_CUBE_COADD_NOFLAT:
		coadd_type = STD_CUBE_COADD_NOFLAT;
		break;
	case PSF_CUBE:
		coadd_type = PSF_CUBE_COADD;
		break;
	case DAR_CUBE_COADD:
		coadd_type = DAR_CUBE_COADD;
		break;
	case DAR_STD_CUBE_COADD:
		coadd_type = STD_CUBE_COADD;
		break;
	case TWEAKED_CUBE_COADD:
		coadd_type = TWEAKED_CUBE_COADD;
		break;
	default:
		cpl_msg_warning(cpl_func,"case %d not found switch to default case",type);
		coadd_type = OBJECT_CUBE_COADD;
		break;
	}

	return coadd_type;
}
cubeType
eris_ifu_jitter_get_obj_type(sofModes mode)
{
	cubeType type;
	switch(mode) {
	case SCIENCE:
		type = OBJECT_CUBE;
		break;
	case STD:
		type = STD_CUBE;
		break;
	case STD_FLUX:
		type = STD_FLUX_CUBE;
		break;
	case PSF:
		type = PSF_CUBE;
		break;
	default:
		type = OBJECT_CUBE;
		break;
	}

	return type;
}
cubeType
eris_ifu_jitter_get_sky_type(sofModes mode)
{
	cubeType type;
	switch(mode) {
	case SCIENCE:
		type = SKY_OBJECT_CUBE;
		break;
	case STD:
		type = SKY_STD_CUBE;
		break;
	case STD_FLUX:
		type = SKY_STD_FLUX_CUBE;
		break;
	case PSF:
		type = SKY_PSF_CUBE;
		break;
	default:
		type = SKY_OBJECT_CUBE;
		break;
	}
	return type;
}



static cpl_error_code
eris_ifu_jitter_save_cpl_cube(
		cpl_frameset *allframes,
		cpl_frameset *usedframes,
		cpl_frame *inherit,
		const cpl_parameterlist * parlist,
		cpl_imagelist *cube,
		hdrl_imagelist *hdrlCube,
		cpl_imagelist *bpmCube,
		cpl_propertylist *hdr,
		struct paramStruct params,
		cubeType type,
		int counter,
		const char* recipe_name)
{
	cpl_error_code retVal   = CPL_ERROR_NONE;
	char    *proCatg        = NULL;
	char    *filenamePrefix = NULL;
	char    *filename       = NULL;

	TRY
	{
		ASSURE(cube != NULL || hdrlCube != NULL,
				CPL_ERROR_ILLEGAL_INPUT, "One of cube or hdrlCube must be not NULL");
		ASSURE(!(cube != NULL && hdrlCube != NULL),
				CPL_ERROR_ILLEGAL_INPUT, "One of cube or hdrlCube must be NULL");
		eris_ifu_jitter_get_procatg_and_filename(type,&proCatg,&filenamePrefix);

		if (counter >= 0) {
			filename = cpl_sprintf("%s_%s_%3.3d.fits", recipe_name, filenamePrefix, counter);
		} else {
			filename = cpl_sprintf("%s_%s.fits", recipe_name, filenamePrefix);
		}
		cpl_propertylist_erase(hdr,"RADECSYS");
		if (cube != NULL) {
			eris_ifu_save_imagelist(allframes,
					inherit,
					hdr,
					parlist,
					recipe_name,
					proCatg,
					filename,
					CPL_TYPE_FLOAT,
					cube);
			if (bpmCube != NULL) {
				cpl_imagelist_save(bpmCube, filename, CPL_TYPE_INT,
						NULL,CPL_IO_EXTEND);
			}

		} else {
			cpl_imagelist *datacube  = cpl_imagelist_new();
			cpl_imagelist *errorcube = cpl_imagelist_new();
			cpl_propertylist *applist = NULL;

			//            BRK_IF_NULL(applist = cpl_propertylist_new());
			applist = cpl_propertylist_duplicate(hdr);
			cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, proCatg);

			//TODO: Why adding CPL_DFS_PRO_TYPE, CPL_DFS_PRO_TECH ???
			cpl_propertylist_update_string(applist, CPL_DFS_PRO_TYPE, "CUBE");
			cpl_propertylist_update_string(applist, CPL_DFS_PRO_TECH, "CUBE");

			eris_ifu_split_hdrl_imagelist(hdrlCube, datacube, errorcube);

			cpl_imagelist *bpmMaskZero;
			bpmMaskZero = eris_ifu_interpolatedMask_to_maskZero(
					bpmCube, params.bpmThreshold);

			/*
			cpl_imagelist *bpmMaskZero = cpl_imagelist_new();
			for (cpl_size iz=0; iz < cpl_imagelist_get_size(datacube); iz++) {
			    cpl_imagelist_set(bpmMaskZero,
			        cpl_image_new_from_mask(
			            hdrl_image_get_mask(
			                    hdrl_imagelist_get(hdrlCube, iz)
			                    )), iz);
			    CHECK_ERROR_STATE();
			}
			 */

			eris_ifu_save_deq_cube(allframes,
                                    NULL,
                                    parlist,
                                    usedframes,
                                    inherit,
                                    recipe_name,
                                    applist,
                                    "RADECSYS",
                                    filename,
                                    datacube,
                                    errorcube,
                                    rmse,
                                    bpmMaskZero,
                                    maskzero/*,
                                    CPL_TYPE_FLOAT */);

			//cpl_imagelist_unwrap(datacube);
			//cpl_imagelist_unwrap(errorcube);
			cpl_imagelist_delete(datacube);
			cpl_imagelist_delete(errorcube);
			cpl_propertylist_delete(applist);
			cpl_imagelist_delete(bpmMaskZero);
		}
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}

	eris_ifu_free_string(&proCatg);
	eris_ifu_free_string(&filenamePrefix);
	eris_ifu_free_string(&filename);
	eris_check_error_code("eris_ifu_jitter_save_cpl_cube");
	return retVal;
}

/*---------------------------------------------------------------------------*/
/**
   @brief generate sof containing obj and sky frame(s), if present,
   corresponding to index i,  and master and calib frames from input set and put
   them in an output one
   @param    set   input frame set
   @param    sof input sofStructure
   @param    i   index corresponding to a give set

   @return   output frame-set or NULL
 */
/*---------------------------------------------------------------------------*/
static cpl_frameset*
eris_ifu_frameset_extract_obj_set(const cpl_frameset* set, struct sofStruct* sof,
		const cpl_size ix)
{
	//cpl_msg_info(cpl_func,"check inputs");
	cpl_ensure(set, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(sof, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ix >= 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    //cpl_ensure_code(out, CPL_ERROR_NULL_INPUT);

	cpl_frameset * obj_set = NULL;
	cpl_frameset * sky_set = NULL;
	cpl_frameset * cdb_set = NULL;
    cpl_frameset * out_set = NULL;

	//cpl_msg_info(cpl_func,"select obj");
	obj_set = cpl_frameset_new();
	eris_ifu_extract_obj_frames(set, obj_set);
    //cpl_frameset_dump(obj_set, stdout);

	//cpl_msg_info(cpl_func,"select sky");
    sky_set = cpl_frameset_new();
	eris_ifu_extract_sky_frames(set, sky_set);
	//cpl_frameset_dump(sky_set, stdout);

	//cpl_msg_info(cpl_func,"select cdb");
	cdb_set = cpl_frameset_new();
	eris_ifu_extract_mst_frames(set, cdb_set);
	//cpl_frameset_dump(cdb_set, stdout);

	cpl_frame* frame = sof->exposureTable[ix].frame;
//	const char* fname = cpl_frame_get_filename(frame);

	//cpl_msg_info(cpl_func,"put all together");
	out_set = cpl_frameset_new();
	cpl_frameset_insert(out_set, cpl_frame_duplicate(frame));
	cpl_frameset_join(out_set, sky_set);
	cpl_frameset_join(out_set, cdb_set);
    //cpl_msg_info(cpl_func,"dumping");
	//cpl_frameset_dump(out_set, stdout);

	cpl_frameset_delete(obj_set);
	cpl_frameset_delete(sky_set);
	cpl_frameset_delete(cdb_set);
	eris_check_error_code("eris_frameset_extract_obj_set");

	return out_set;
}

cpl_error_code eris_ifu_jitter_process_exposures(struct sofStruct *sof,
		struct stdParamStruct stdParams,
		struct paramStruct params,
		cpl_frameset *frameset,
		const cpl_parameterlist * parlist,
		const char* recipe_name,
		const char* context)
{
	cpl_error_code  retVal  = CPL_ERROR_NONE;
	hdrl_image   *objImg = NULL;
	hdrl_image   *tmpImg = NULL;
	struct exposureEntry *objEntry;
	cpl_polynomial *ohLambdaCorrection = NULL;
	productDepthType productDepth = stdParams.productDepth;
	bool        doOHbasedLambdaCorrection = FALSE;
	cubeType type = OBJECT_CUBE;
	const char  *expType = NULL;
	double cdelt = 0.;
	double centralLambda = 0.0;
	double shift = 0.0;
	bool isSky = FALSE;

	TRY
    {
        cpl_msg_info(cpl_func,"Build OBJ and SKY data cubes, may correct wavecal by OH lines, subtract sky background");

        if ((sof->oh_ref_frame != NULL) && !params.skip_oh_align && !(isSky  && params.skip_sky_oh_align)) {
			doOHbasedLambdaCorrection = TRUE;
		} else {
			doOHbasedLambdaCorrection = FALSE;
        }
        cpl_msg_debug(cpl_func,"doOHbasedLambdaCorrection: %d, isSky: %d, "
                                "params.skip_sky_oh_align: %d, params.skip_oh_align: %d",
                                doOHbasedLambdaCorrection, isSky,
                                params.skip_sky_oh_align, params.skip_oh_align);

		if (getenv("USE_OCS") != NULL) {
	        eris_ifu_update_wcs_with_OCS_keywords(sof);
	    }

		for (int ix=0; ix < sof->exposureTableCnt; ix++) {
			if (sof->exposureTable[ix].isObject) {
				expType = "OBJ";
				isSky = FALSE;
				type = eris_ifu_jitter_get_obj_type(sof->mode);
			} else {
				expType = "SKY";
				isSky = TRUE;
				type = eris_ifu_jitter_get_sky_type(sof->mode);
			}
			cpl_msg_info(__func__, "Process exposure #\%3.3d %s %s", ix, expType,
					cpl_frame_get_filename(sof->exposureTable[ix].frame));

			objEntry = &sof->exposureTable[ix];
			objImg = objEntry->rawImage;

			// build OBJECT cube
            BRK_IF_NULL(
                    objEntry->cube = eris_ifu_jitter_build_cube(
                            objImg, ix, sof, params, false,
                            ohLambdaCorrection, productDepth));

			eris_ifu_free_polynomial(&ohLambdaCorrection);
			// TODO: do lambda correct. with OH lines also for PS or STD OBS?

            if (doOHbasedLambdaCorrection) {
				// compute OH based lambda correction

                /* to prevent memory leak delete cpl_polynomial structure */
                eris_ifu_free_polynomial(&ohLambdaCorrection);

                int pfit_order = 0;
                char* param_name = cpl_sprintf("%s.oh_align_poly_order", context);
                cpl_msg_info(cpl_func,"pname: %s",param_name);
                pfit_order = cpl_parameter_get_int(cpl_parameterlist_find_const(parlist, param_name));
                cpl_free(param_name);


				BRK_IF_NULL( ohLambdaCorrection = eris_ifu_lcorr_get(NULL,
						objEntry->cube, objEntry->cubeHdr, sof->oh_ref_peaks, pfit_order));

				cdelt = cpl_propertylist_get_double(objEntry->cubeHdr, "CDELT3");
				centralLambda = cpl_propertylist_get_double(
						objEntry->cubeHdr, "CRVAL3") +
								hdrl_imagelist_get_size(objEntry->cube) * cdelt * 0.5;
				shift = cpl_polynomial_eval_1d(
						ohLambdaCorrection, centralLambda, NULL);
				cpl_msg_info(__func__,
						"Overall shift of wavelength due OH lambda correction: %f [micron]  %f [pixel]",
						 shift, shift/cdelt);

				eris_ifu_free_hdrl_imagelist(&objEntry->cube);

                tmpImg = eris_ifu_jitter_subtract_background(
                        params.skyTweak, ix, sof, productDepth);

                objEntry->cube = eris_ifu_jitter_build_cube(
                        tmpImg, ix, sof, params, true,
                        ohLambdaCorrection, productDepth);

	            eris_ifu_free_polynomial(&ohLambdaCorrection);

				BRK_IF_ERROR(eris_ifu_append_qc_float(
						objEntry->cubeHdr, "LAMBDA SHIFT UM", shift,
						"[um] OH based lambda shift"));
				BRK_IF_ERROR(eris_ifu_append_qc_float(
						objEntry->cubeHdr, "LAMBDA SHIFT PIXEL", shift/cdelt,
                        "[px] OH based lambda shift"));
            } else {
                if (params.skip_oh_align && (ix == 0)) {
                    cpl_msg_warning(cpl_func, " |------------------------------------------------------|");
                    cpl_msg_warning(cpl_func, " | WARNING: with parameter --sky_oh_align the spectral  |");
                    cpl_msg_warning(cpl_func, " |          calibration of the outputs is expected to   |");
                    cpl_msg_warning(cpl_func, " |          be off for several pixels due to flexure of |");
                    cpl_msg_warning(cpl_func, " |          the instrument                              |");
                    cpl_msg_warning(cpl_func, " |------------------------------------------------------|");
                }
            }

			if (productDepth >= PD_AUXILLIARY) {
/*
				cpl_imagelist *datacube  = cpl_imagelist_new();
				cpl_imagelist *errorcube = cpl_imagelist_new();
				BRK_IF_ERROR(
						eris_ifu_split_hdrl_imagelist(objEntry->cube, datacube, errorcube));
*/

/*
				cpl_imagelist *bpmMaskZero;
				BRK_IF_NULL(
						bpmMaskZero = eris_ifu_interpolatedMasK_to_maskZero(
								objEntry->cubeBpm, params.bpmThreshold));
				for (cpl_size zx=0; zx<cpl_imagelist_get_size(datacube); zx++){
				    cpl_image *tmpImg = cpl_imagelist_get(datacube, zx);
				    cpl_mask *tmpMask = cpl_mask_threshold_image_create(
				        cpl_imagelist_get(objEntry->cubeBpm, zx), 0.5, 1.5);
				    cpl_image_reject_from_mask(tmpImg, tmpMask);
				    eris_ifu_free_mask(&tmpMask);
		            CHECK_ERROR_STATE();
				}
*/
				//                eris_ifu_jitter_save_cpl_cube(frameset, objEntry->frame, parlist,
						//                    datacube, NULL, bpmMaskOne, objEntry->cubeHdr, params, type, ix, recipe_name);
				cpl_frameset* obj_set =
		        eris_ifu_frameset_extract_obj_set(frameset, sof, ix);
				eris_ifu_jitter_save_cpl_cube(frameset, obj_set, objEntry->frame, parlist,
						NULL, objEntry->cube, objEntry->cubeBpm, objEntry->cubeHdr, params, type, ix, recipe_name);
				eris_ifu_jitter_save_cpl_cube(frameset, obj_set, objEntry->frame, parlist,
						objEntry->cubeBpm, NULL, NULL, objEntry->cubeHdr, params, BPM_CUBE, ix, recipe_name);
			    cpl_frameset_delete(obj_set);
			}


			cpl_boolean chop_nan = CPL_FALSE;
			char* param_name = cpl_sprintf("%s.chop-nan", context);
			//cpl_msg_info(cpl_func,"pname: %s",param_name);
			chop_nan = cpl_parameter_get_bool(
							cpl_parameterlist_find_const(parlist, param_name));
		    cpl_free(param_name);
		    if(chop_nan) {
		    	ifsBand ifu_band = eris_ifu_get_band(objEntry->cubeHdr);
		    	cpl_size zmin = 0;
		    	cpl_size zmax = 0;
		    	eris_ifu_get_plane_cut_min_max(ifu_band, &zmin, &zmax);
		    	//cpl_msg_info(cpl_func,"check zmin: %lld zmax: %lld", zmin, zmax);

		    	eris_ifu_cube_trim_nans(zmin, zmax, &(objEntry->cube),
		    			&(objEntry->cubeBpm), objEntry->cubeHdr);
		    }


			eris_ifu_free_hdrl_image(&tmpImg);
		}
		//AMO TODO STRANGE: added line: this fixes a leak but creates another problem
		//eris_ifu_free_hdrl_imagelist(&objEntry->cube);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_ifu_free_polynomial(&ohLambdaCorrection);
	eris_ifu_free_hdrl_image(&objImg);
	 eris_check_error_code("eris_ifu_jitter_process_exposures");
	return retVal;
}
static cpl_error_code
eris_ifu_jitter_skycorr_from_mask(struct exposureEntry *objEntry,
		const cpl_mask*mask_obj, const int sky_est_method)
{

    cpl_image* ima;
    cpl_image* err;
    cpl_mask* bpm;
    hdrl_image* himg;
    cpl_size skip = ERIS_IFU_SKIP_SKY_CORR_PLANES;
    cpl_size size_z = hdrl_imagelist_get_size(objEntry->cube);

    hdrl_value sky = {0, 0};
    for (cpl_size zx = skip; zx < size_z - skip; zx++){
    	//cpl_msg_info(cpl_func,"zx: %lld",zx);

    	himg = hdrl_imagelist_get(objEntry->cube, zx);
    	ima = hdrl_image_get_image(himg);
    	err = hdrl_image_get_error(himg);
    	bpm = hdrl_image_get_mask(himg);

    	hdrl_image_reject_from_mask(himg, mask_obj);
    	if(sky_est_method==0) {
    		sky = hdrl_image_get_median(himg);
    	} else {
    		sky = hdrl_image_get_mean(himg);
    	}

    	himg = hdrl_image_create(ima, err);
    	hdrl_image_accept_all(himg);
    	hdrl_image_sub_scalar(himg, sky);

    	hdrl_image_reject_from_mask(himg, bpm);
    	hdrl_imagelist_set(objEntry->cube, hdrl_image_duplicate(himg), zx);
    	hdrl_image_delete(himg);

    	if(cpl_error_get_code() != CPL_ERROR_NONE) {
    		 eris_check_error_code("eris_ifu_jitter_skycorr_from_mask");
    		 cpl_msg_error(cpl_func,"Error found in eris_ifu_jitter_skycorr_from_mask. Exit!");
    		 exit(0);
    	}
    }
    eris_check_error_code("eris_ifu_jitter_skycorr_from_mask");
    return cpl_error_get_code();

}
static cpl_error_code
eris_ifu_jitter_get_split_param_int(const cpl_parameterlist * parlist,
		const char* pname, const int min_x, const int min_y,
		const int max_x, const int max_y, int* px, int* py)
{

	const cpl_parameter* par = NULL;
	char* param_name;
	const char* context = "eris.eris_ifu_jitter";

	const char* pname_string_value;
	param_name = cpl_sprintf("%s.%s", context, pname);
	par = cpl_parameterlist_find_const(parlist, param_name);
	pname_string_value = cpl_parameter_get_string(par);
	cpl_free(param_name);

	int nFields;
	unsigned  int end;

	nFields = sscanf(pname_string_value, "%d,%d%n", px,  py, &end);

	if (nFields != 2 || end != strlen(pname_string_value)) {
		cpl_msg_error(cpl_func, "The %s parameter must be "
				"a list of two integers separated by a comma", pname);
		printf("Error reading sky-box-center string\n");
	}

	if (*px < min_x) {
		cpl_msg_warning(cpl_func,"%s_x: %d set it to 0",pname, *px);
		*px = 0;
	}
	if (*py < min_y) {
		cpl_msg_warning(cpl_func,"%s_y: %d set it to 0",pname, *py);
		*py = 0;
	}
	if (*px > max_x) {
		cpl_msg_warning(cpl_func,"%s_x: %d set it to 10",pname, *px);
		*px = 10;
	}
	if (*py > max_y) {
		cpl_msg_warning(cpl_func,"%s_y: %d set it to 10", pname, *py);
		*py = 10;
	}
        /*
	cpl_msg_debug(cpl_func,"%s_x: %d, %s_y: %d",
				pname, *px, pname, *py);
         */
	eris_check_error_code("eris_ifu_jitter_get_split_param_int");
	return cpl_error_get_code();
}

static cpl_error_code
eris_ifu_jitter_skycor_from_box(const cpl_parameterlist * parlist,
		struct exposureEntry *objEntry, const int aj_method)
{

	int sky_box_center_x, sky_box_center_y,
	    sky_box_width_x, sky_box_width_y,
		sky_box_edges_margin_x, sky_box_edges_margin_y;

	eris_ifu_jitter_get_split_param_int(parlist,"sky-box-center", 0, 0, 64, 64,
			&sky_box_center_x, &sky_box_center_y);
	eris_ifu_jitter_get_split_param_int(parlist,"sky-box-width", 0, 0, 10, 10,
				&sky_box_width_x, &sky_box_width_y);
	eris_ifu_jitter_get_split_param_int(parlist,"sky-box-edges-margin", 0, 0, 10, 10,
					&sky_box_edges_margin_x, &sky_box_edges_margin_y);


	cpl_msg_info(cpl_func,"sky_box_center_x: %d sky_box_center_y: %d",
	                		sky_box_center_x, sky_box_center_y);
	cpl_msg_info(cpl_func,"sky_box_width_x: %d sky_box_width_y: %d",
		                		sky_box_width_x, sky_box_width_y);
    cpl_msg_info(cpl_func,"sky_box_edges_margin_x: %d sky_box_edges_margin_y: %d",
    		sky_box_edges_margin_x, sky_box_edges_margin_y);



    char* param_name;
    int sky_est_method = -1;
    const char* context = "eris.eris_ifu_jitter";
    const cpl_parameter* p = NULL;

    param_name = cpl_sprintf("%s.sky-est-method", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    if(p != NULL) {
    	sky_est_method = cpl_parameter_get_int(p);
    }
    cpl_free(param_name);

    int niter = 5;

    param_name = cpl_sprintf("%s.sky-est-niter", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    if(p != NULL) {
    	niter = cpl_parameter_get_int(p);
    }
    cpl_free(param_name);

    int kappa = 3;
    param_name = cpl_sprintf("%s.sky-est-kappa", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    if(p != NULL) {
    	kappa = cpl_parameter_get_double(p);
    }
    cpl_free(param_name);

    /* apply correction based on user defined parameters */
    cpl_image* img = NULL;
    hdrl_image* himg = NULL;
    hdrl_value sky = {0, 0};
    cpl_size llx = 0 , urx = 0, lly = 0 , ury = 0 ;
    cpl_msg_info(cpl_func,"llx, lly, urx, ury, [%lld,%lld,%lld,%lld]",
    		llx,lly,urx,ury);
    /* TODO: Handle start/end wave ranges where object is NAN */
    cpl_size skip = ERIS_IFU_SKIP_SKY_CORR_PLANES;
    /*
    cpl_imagelist* before = cpl_imagelist_new();
    cpl_imagelist* after = cpl_imagelist_new();
    for (cpl_size zx = 0;zx<hdrl_imagelist_get_size(objEntry->cube);zx++){
    	himg = hdrl_imagelist_get(objEntry->cube, zx);
    	img = hdrl_image_get_image(himg);
    	cpl_imagelist_set(before, img, zx);
    }
    cpl_imagelist_save(before, "cube_before.fits", CPL_TYPE_INT,
    		NULL, CPL_IO_DEFAULT);
    		*/
    cpl_bivector* xy_pos = NULL;
    cpl_vector* sky_data = NULL;
    double* x_pos = NULL;
    double* y_pos = NULL;

    cpl_size size_x = hdrl_imagelist_get_size_x(objEntry->cube);
    cpl_size size_y = hdrl_imagelist_get_size_y(objEntry->cube);
    if(aj_method == 5) {
    	xy_pos = cpl_bivector_new(4);
    	sky_data = cpl_vector_new(4);
    	x_pos = cpl_bivector_get_x_data(xy_pos);
    	y_pos = cpl_bivector_get_y_data(xy_pos);

    	x_pos[0] = sky_box_edges_margin_x + sky_box_width_x;
    	y_pos[0] = sky_box_edges_margin_y + sky_box_width_y;

    	x_pos[1] = sky_box_edges_margin_x + sky_box_width_x;
    	y_pos[1] = size_y - sky_box_edges_margin_y - sky_box_width_y;

    	x_pos[2] = size_x - sky_box_edges_margin_x - sky_box_width_x;
    	y_pos[2] = sky_box_edges_margin_y + sky_box_width_y;

    	x_pos[3] = size_x - sky_box_edges_margin_x - sky_box_width_x;
    	y_pos[3] = size_y - sky_box_edges_margin_y - sky_box_width_y;

    }

    if(aj_method == 6) {
    	xy_pos = cpl_bivector_new(8);
    	sky_data = cpl_vector_new(8);
    	x_pos = cpl_bivector_get_x_data(xy_pos);
    	y_pos = cpl_bivector_get_y_data(xy_pos);


    	x_pos[0] = sky_box_edges_margin_x + sky_box_width_x;
    	y_pos[0] = sky_box_edges_margin_y + sky_box_width_y;

    	x_pos[1] = sky_box_edges_margin_x + sky_box_width_x;
    	y_pos[1] = size_y - sky_box_edges_margin_y - sky_box_width_y;

    	x_pos[2] = size_x - sky_box_edges_margin_x - sky_box_width_x;
    	y_pos[2] = sky_box_edges_margin_y + sky_box_width_y;

    	x_pos[3] = size_x - sky_box_edges_margin_x - sky_box_width_x;
    	y_pos[3] = size_y - sky_box_edges_margin_y - sky_box_width_y;



    	x_pos[4] = floor(0.5 * size_x);
    	y_pos[4] = sky_box_edges_margin_y + sky_box_width_y;

    	x_pos[5] = sky_box_edges_margin_x + sky_box_width_x;
    	y_pos[5] = floor(0.5 * size_y);

    	x_pos[6] = size_x - sky_box_edges_margin_x - sky_box_width_x;
    	y_pos[6] = floor(0.5 * size_y);

    	x_pos[7] = floor(0.5 * size_x);
    	y_pos[7] = size_y - sky_box_edges_margin_y - sky_box_width_y;

    }

    cpl_size size_z = hdrl_imagelist_get_size(objEntry->cube);
    double sky_tmp_val = 0;
    if(aj_method == 4) {
    	llx = sky_box_center_x - sky_box_width_x;
    	urx = sky_box_center_x + sky_box_width_x;
    	lly = sky_box_center_y - sky_box_width_y;
    	ury = sky_box_center_y + sky_box_width_y;
    	for (cpl_size zx = skip; zx < size_z - skip; zx++){
    		//cpl_msg_info(cpl_func,"zx: %lld",zx);
    		himg = hdrl_imagelist_get(objEntry->cube, zx);
    		img = hdrl_image_get_image(himg);
    		/* the following may fail at start end wave ranges */
    		if(sky_est_method==0) {
    			sky.data = cpl_image_get_median_window(img, llx, lly, urx, ury);
    		} else {
    			sky.data = cpl_image_get_mean_window(img, llx, lly, urx, ury);
    		}
    		hdrl_image_sub_scalar(himg, sky);
    		hdrl_imagelist_set(objEntry->cube, himg, zx);
    	}
    } else if(aj_method == 5 || aj_method == 6) {
    	cpl_size xy_pos_size = cpl_bivector_get_size(xy_pos);
    	double* psky_data = cpl_vector_get_data(sky_data);
    	for (cpl_size zx = skip; zx < size_z - skip; zx++){

    		himg = hdrl_imagelist_get(objEntry->cube, zx);
    		img = hdrl_image_get_image(himg);
    		/* the following may fail at start end wave ranges */
    		for (cpl_size k = 0; k < xy_pos_size; k++){
    			llx = ((cpl_size) x_pos[k] - sky_box_width_x);
    			lly = ((cpl_size) y_pos[k] - sky_box_width_y);
    			urx = ((cpl_size) x_pos[k] + sky_box_width_x);
    			ury = ((cpl_size) y_pos[k] + sky_box_width_y);
    			//cpl_msg_info(cpl_func,"llx, lly, urx, ury: [%lld,%lld,%lld,%lld]",llx, lly, urx, ury);
    			if(sky_est_method==0) {
    				sky_tmp_val = cpl_image_get_median_window(img, llx, lly, urx, ury);
    				if(cpl_error_get_code() != CPL_ERROR_NONE) {
    					cpl_error_reset();
    					continue;
    				} else {
    					psky_data[k] = sky_tmp_val;
    				}

    			} else {
    				sky_tmp_val = cpl_image_get_mean_window(img, llx, lly, urx, ury);
    				if(cpl_error_get_code() != CPL_ERROR_NONE) {
    					cpl_error_reset();
    					continue;
    				} else {
    					psky_data[k] = sky_tmp_val;
    				}
    			}
    			if(cpl_error_get_code() != CPL_ERROR_NONE) {
    				cpl_msg_error(cpl_func,"Some error during sky modeling: "
    				"try to use a different region or other aj-method");
    				eris_check_error_code("eris_ifu_jitter_skycor_from_box");
    				exit(0);
    			}
    		}
    		if(sky_est_method==0) {
    		    sky.data = cpl_vector_get_median(sky_data);
    		} else {
    			sky.data = cpl_vector_get_mean(sky_data);
    		}
    		hdrl_image_sub_scalar(himg, sky);
    		hdrl_imagelist_set(objEntry->cube, himg, zx);
    	}
    } else if(aj_method == 7) {

		hdrl_image* cube_collapsed = NULL;
		cpl_image* cube_cmap = NULL;
		hdrl_imagelist_collapse_mean(objEntry->cube, &cube_collapsed, &cube_cmap);
		/*
		cpl_image_save(hdrl_image_get_image(cube_collapsed), "collapsed_mean.fits",
		                    		CPL_TYPE_FLOAT, NULL, CPL_IO_DEFAULT);
		cpl_image_save(cube_cmap, "cube_cmap.fits", CPL_TYPE_FLOAT, NULL,
				CPL_IO_DEFAULT);
				*/
		cpl_mask* mask_obj;

		mask_obj = eris_ifu_hima_get_obj_mask(cube_collapsed, niter, kappa);

		hdrl_image_delete(cube_collapsed);
		cpl_image_delete(cube_cmap);

		eris_ifu_jitter_skycorr_from_mask(objEntry, mask_obj, sky_est_method);

		cpl_mask_delete(mask_obj);
    } else if(aj_method == 8) {
		hdrl_image* cube_mean = NULL;
		cpl_image* cube_cmap = NULL;
		hdrl_imagelist_collapse_mean(objEntry->cube, &cube_mean, &cube_cmap);
		cpl_mask* mask_obj = eris_ifu_hima_get_obj_mask_percent(cube_mean, 0.25);

		hdrl_image_delete(cube_mean);
		cpl_image_delete(cube_cmap);
		eris_ifu_jitter_skycorr_from_mask(objEntry, mask_obj, sky_est_method);
		cpl_mask_delete(mask_obj);
    }

    /*
    for (cpl_size zx = 0;zx<hdrl_imagelist_get_size(objEntry->cube);zx++){
    	himg = hdrl_imagelist_get(objEntry->cube, zx);
    	img = hdrl_image_get_image(himg);
    	cpl_imagelist_set(after, img, zx);
    }

    cpl_imagelist_save(after, "cube_after.fits",
                    		CPL_TYPE_INT, NULL, CPL_IO_DEFAULT);
                    		*/


    eris_check_error_code("eris_ifu_jitter_skycor_from_box");
    return cpl_error_get_code();

}

cpl_error_code eris_ifu_jitter_process_cubes(struct sofStruct *sof,
                                             struct stdParamStruct stdParams,
                                             struct paramStruct params,
                                             cpl_frameset *frameset,
                                             const cpl_parameterlist * parlist,
                                             const char* recipe_name,
                                             cubeType *obj_type)
{
    cpl_error_code          retVal          = CPL_ERROR_NONE,
                            err             = CPL_ERROR_NONE;
    struct exposureEntry    *objEntry       = NULL,
                            *skyEntry       = NULL;
    cpl_mask                *tmpMask        = NULL;
    hdrl_image              *tmpImg         = NULL;
    cpl_size                imgSize         = 0;
    double                  *pObjErr        = NULL,
                            *pSkyErr        = NULL,
                            *pTweakedErr    = NULL;
    cpl_binary              *pObjMask       = NULL,
                            *pErrMask       = NULL,
                            *pTweakedMask   = NULL;

	TRY
    {
        cpl_msg_info(cpl_func, "Process Jitter Cubes: split Obj & Sky, correct Sky as user specified");

        for (int ix = 0; ix < sof->exposureTableCnt; ix++) {
			if (!sof->exposureTable[ix].isObject) {
				continue; // loop only over OBJECT cubes, skip SKY
			}
			objEntry = &sof->exposureTable[ix];
			cpl_frameset* obj_set =
			                eris_ifu_frameset_extract_obj_set(frameset, sof, ix);
            // apply skytweak if desired
			if (params.skyTweak != NONE) {
				int skyIndex = objEntry->skyIndex;
				if (skyIndex < 0) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
                                       "no vaild sky frame found");
				}
				skyEntry = &sof->exposureTable[skyIndex];

				if (skyEntry->cube == NULL) {
					cpl_msg_error(__func__,
							"Missing sky image for input frame %s",
                            cpl_frame_get_filename(objEntry->frame));
                } else {
                    cpl_imagelist *dataCubeObj = cpl_imagelist_new(),
                                  *errCubeObj  = cpl_imagelist_new(),
                                  *dataCubeSky = cpl_imagelist_new(),
                                  *errCubeSky  = cpl_imagelist_new();

                    BRK_IF_ERROR(
                        eris_ifu_split_hdrl_imagelist(objEntry->cube,
                                                      dataCubeObj, errCubeObj));
                    BRK_IF_ERROR(
                        eris_ifu_split_hdrl_imagelist(skyEntry->cube,
                                                      dataCubeSky, errCubeSky));

                    if (params.skyTweak == DAVIES) {
                        cpl_imagelist           *skycorrObj     = NULL,
                                                *skycorrSky     = NULL,
                                                *skycorrErr     = NULL;

                        skycorrObj = eris_ifu_sky_tweak(dataCubeObj, dataCubeSky,
                                                        objEntry->cubeHdr, .3,
                                                        params.tbsub,
                                                        params.discard_subband,
                                                        params.stretch_sky,
                                                        params.stretch_degree,
                                                        params.stretch_resampling,
                                                        0, &skycorrSky);

                        /*
                         * simple error calculation for sky tweaked cube
                         * as sky tweaked cube is basically objject cube - sky cube
                         * the error is sqrt(objectError^2 + skyError^2)
                         */
                        BRK_IF_NULL(
                            skycorrErr = cpl_imagelist_duplicate(errCubeObj));

                        imgSize = cpl_image_get_size_x(cpl_imagelist_get(skycorrErr, 0)) *
                                  cpl_image_get_size_y(cpl_imagelist_get(skycorrErr, 0));

                        for (cpl_size tx = 0; tx < cpl_imagelist_get_size(skycorrErr); tx++) {
                           pObjErr = cpl_image_get_data_double(
                                            cpl_imagelist_get(errCubeObj,tx));
                           pSkyErr = cpl_image_get_data_double(
                                            cpl_imagelist_get(errCubeSky,tx));
                           pTweakedErr = cpl_image_get_data_double(
                                            cpl_imagelist_get(skycorrErr,tx));
                           pObjMask = cpl_mask_get_data(
                                            cpl_image_get_bpm(cpl_imagelist_get(errCubeObj,tx)));
                           pErrMask = cpl_mask_get_data(
                                            cpl_image_get_bpm(cpl_imagelist_get(errCubeSky,tx)));
                           pTweakedMask = cpl_mask_get_data(
                                            cpl_image_get_bpm(cpl_imagelist_get(skycorrErr,tx)));

                           for (cpl_size rx = 0; rx < imgSize; rx++) {
                               if ((pObjMask[rx] != BAD_PIX) &&
                                   (pErrMask[rx] != BAD_PIX))
                               {
                                   pTweakedMask[rx] = GOOD_PIX;
                                   pTweakedErr[rx] = hypot(pObjErr[rx], pSkyErr[rx]);
                               } else {
                                   pTweakedMask[rx] = BAD_PIX;
                               }
                           }
                        }

                        // agudo: Possible memory leak here when objEntry->cube is overwritten?
                        //        It isn't deleted anywhere beforehand
                        BRK_IF_NULL(
                            objEntry->cube = hdrl_imagelist_create(skycorrObj, skycorrErr));

                        eris_ifu_free_imagelist(&skycorrObj);
                        eris_ifu_free_imagelist(&skycorrErr);
                        //To  delete the following two creates a seg fault
                        eris_ifu_free_imagelist(&dataCubeSky);
                        eris_ifu_free_imagelist(&errCubeSky);
                        eris_ifu_free_imagelist(&dataCubeObj);
                        eris_ifu_free_imagelist(&errCubeObj);

                        if ((stdParams.productDepth >= PD_DEBUG) && (skycorrSky != NULL)) {
                            char *fname = cpl_sprintf(
                                            "eris_ifu_jitter_dbg_skycoorsky_%3.3d.fits", skyIndex);
                            cpl_imagelist_save(skycorrSky, fname,
                                               CPL_TYPE_FLOAT, NULL, CPL_IO_CREATE);
                            eris_ifu_free_string(&fname);
                        }

                        eris_ifu_free_imagelist(&skycorrSky);
                        CHECK_ERROR_STATE();
                    }
                    else if (params.skyTweak == AUSTRIAN) {
                        cpl_table               *objspec        = NULL,
                                                *skyspec        = NULL;
                        cpl_parameterlist       *parlist_skycorr = NULL;
                        double ts = cpl_test_get_walltime();

                        // loop through all pixels of the image and apply skycorr
                        cpl_size nx = hdrl_imagelist_get_size_x(objEntry->cube);
                        cpl_size ny = hdrl_imagelist_get_size_y(objEntry->cube);

                        // create parameterlist for skycorr
                        BRK_IF_NULL(
                            parlist_skycorr = eris_ifu_skycorr_create_parlist(skyEntry->cubeHdr));

// agudo: couldn't successfully extract this code from sc_skycorr, so keep reading in txt files for every spectrum
//                        /* Read line groups from ASCII file and adapt it to observing conditions */
//                        cpl_table *groups = cpl_table_new(0);
//                        BRK_IF_ERROR(
//                            sc_lines(groups, parlist_skycorr));

                        // in y we increment by 2 because the iamge is already scaled by 2 here
                        for (cpl_size y = 1; y <= ny; y += 2) {
                            if (stdParams.productDepth == 3) {
                                cpl_msg_debug(cpl_func, ">>>>> line %d", (int)y);
                            }
                            for (cpl_size x = 1; x <= nx; x++) {
                                int success = CPL_FALSE;
                                if (stdParams.productDepth == 3) {
                                    cpl_msg_debug(cpl_func, ">>>>>    pix (%d/%d)", (int)x, (int)y);
                                }
                                BRK_IF_NULL(
                                    objspec = eris_ifu_skycorr_extract_spectrum(objEntry->cube, objEntry->cubeHdr, x, y));
                                BRK_IF_NULL(
                                    skyspec = eris_ifu_skycorr_extract_spectrum(skyEntry->cube, skyEntry->cubeHdr, x, y));

                                if (stdParams.productDepth == 3) {
                                    char* fn = NULL;

//                                    cpl_table *t = NULL;
//                                    t = cpl_table_duplicate(objspec);
//                                    cpl_table_erase_column(t, "weight");
//                                    cpl_table_erase_column(t, "mask_I");
//                                    fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_objspec.fits", (int)x, (int)y);
//                                    cpl_table_save(t, NULL, objEntry->cubeHdr, fn, CPL_IO_CREATE);
//                                    cpl_free(fn);

//                                    fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_objspec_vector.fits", (int)x, (int)y);
//                                    double *pt = cpl_table_get_data_double (t, "flux");
//                                    cpl_vector *vt = cpl_vector_wrap(cpl_table_get_nrow(t), pt);
//                                    eris_ifu_save_vector_dbg(vt, fn, CPL_IO_CREATE, objEntry->cubeHdr);
//                                    cpl_free(fn);

//                                    cpl_table_delete(t); t = NULL;
//                                    t = cpl_table_duplicate(skyspec);
//                                    cpl_table_erase_column(t, "weight");
//                                    cpl_table_erase_column(t, "mask_I");
//                                    fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_skyspec.fits", (int)x, (int)y);
//                                    cpl_table_save(t, NULL, skyEntry->cubeHdr, fn, CPL_IO_CREATE);
//                                    cpl_free(fn);
//                                    cpl_table_delete(t); t = NULL;

                                    fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_objspec_1_orig.fits", (int)x, (int)y);
                                    cpl_table_save(objspec, NULL, objEntry->cubeHdr, fn, CPL_IO_CREATE);
                                    cpl_free(fn);
                                    fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_skyspec_1_orig.fits", (int)x, (int)y);
                                    cpl_table_save(skyspec, NULL, skyEntry->cubeHdr, fn, CPL_IO_CREATE);
                                    cpl_free(fn);
                                }

                                /* prepare skycorr code */
                                double mean_lambda = 0.5 * (cpl_table_get(objspec, "lambda", 0, NULL) +
                                                            cpl_table_get(objspec, "lambda", cpl_table_get_nrow(objspec)-1, NULL));
                                cpl_parameter *p = cpl_parameterlist_find(parlist_skycorr, "meanlam");
                                cpl_parameter_set_double(p, mean_lambda);

                                double mean_obj_flux =  cpl_table_get_column_mean(objspec, "flux"),
                                       mean_sky_flux =  cpl_table_get_column_mean(skyspec, "flux");

                                // ignore spectra which have thoroughly a flux of zero
                                if ((fabs(mean_obj_flux) > 0.00001) && (fabs(mean_sky_flux) > 0.00001)) {
                                    // tweak outliers that are larger than 10 times than average
                                    int outlier_factor = 10;
                                    BRK_IF_ERROR(
                                        eris_ifu_skycorr_flatten_outliers(objspec, skyspec, outlier_factor));

                                    // alternatively remove outliers as good as possible (but we risk to remove as well OH lines etc.)
                                    // double min_frac = 0.5;
                                    // BRK_IF_ERROR(
                                    //   eris_ifu_skycorr_flatten_outliers2(objspec, skyspec, min_frac));

                                   if (stdParams.productDepth == 3) {
                                       char* fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_objspec_2_flatten1.fits", (int)x, (int)y);
                                       cpl_table_save(objspec, NULL, objEntry->cubeHdr, fn, CPL_IO_CREATE);
                                       cpl_free(fn);
                                       fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_skyspec_2_flatten1.fits", (int)x, (int)y);
                                       cpl_table_save(skyspec, NULL, skyEntry->cubeHdr, fn, CPL_IO_CREATE);
                                       cpl_free(fn);
                                   }

                                    err = sc_skycorr(parlist_skycorr, objspec, skyspec, /*groups,*/ stdParams.productDepth);
                                    if (err != CPL_ERROR_NONE) {
                                        // reset error
                                        cpl_errorstate_set(CPL_ERROR_NONE);
                                        err = CPL_ERROR_NONE;
                                        CHECK_ERROR_STATE();

                                        // tweak outliers again that are larger than 8 times than average (seems not really to help...)
                                        outlier_factor = 8;
                                        BRK_IF_ERROR(
                                            eris_ifu_skycorr_flatten_outliers(objspec, skyspec, outlier_factor));

                                        if (stdParams.productDepth == 3) {
                                            char* fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_objspec_3_flatten2.fits", (int)x, (int)y);
                                            cpl_table_save(objspec, NULL, objEntry->cubeHdr, fn, CPL_IO_CREATE);
                                            cpl_free(fn);
                                            fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_skyspec_3_flatten2.fits", (int)x, (int)y);
                                            cpl_table_save(skyspec, NULL, skyEntry->cubeHdr, fn, CPL_IO_CREATE);
                                            cpl_free(fn);
                                        }

                                        err = sc_skycorr(parlist_skycorr, objspec, skyspec, /*groups, */stdParams.productDepth);
                                        if (err != CPL_ERROR_NONE) {
                                            // give up sky correcting...
                                            cpl_errorstate_set(CPL_ERROR_NONE);
                                            err = CPL_ERROR_NONE;
                                            CHECK_ERROR_STATE();
                                        } else {
                                            success = CPL_TRUE;
                                        }
                                    } else {
                                        success = CPL_TRUE;
                                    }
                                }

                                if (success == CPL_FALSE) {
                                    // do what? -> insert blank spectrum
                                    if (stdParams.productDepth == 3) {
                                        cpl_msg_debug(cpl_func, ">>>>>       inserted blank spectrum");
                                        char* fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_objspec_4_failed.fits", (int)x, (int)y);
                                        cpl_table_save(objspec, NULL, objEntry->cubeHdr, fn, CPL_IO_CREATE);
                                        cpl_free(fn);
                                        fn = cpl_sprintf("%d_%d_eris_ifu_jitter_dbg_skyspec_4_failed.fits", (int)x, (int)y);
                                        cpl_table_save(skyspec, NULL, skyEntry->cubeHdr, fn, CPL_IO_CREATE);
                                        cpl_free(fn);
                                    }
                                    eris_ifu_free_table(&objspec);
                                    BRK_IF_NULL(
                                        objspec = eris_ifu_skycorr_create_nan_spectrum(objEntry->cube));
                                }
                                BRK_IF_ERROR(
                                    eris_ifu_skycorr_insert_spectrum(objEntry->cube, objspec, x, y));

                                eris_ifu_free_table(&objspec);
                                eris_ifu_free_table(&skyspec);
                                CHECK_ERROR_STATE();
                            } // end for: x <= nx
                        } // end for: y <= ny
                        eris_ifu_free_parameterlist(&parlist_skycorr);

                        double te = cpl_test_get_walltime();
                        cpl_msg_debug(cpl_func,">>>>> >>>>> skycorr took %g seconds", te - ts);
//                        eris_ifu_free_table(&groups);
                    } // end if: AUSTRIAN
                } // end else: (skyEntry->cube != NULL)

                if (!cpl_propertylist_has(objEntry->cubeHdr,"PRODCATG")) {
                    cpl_propertylist_append_string(objEntry->cubeHdr, "PRODCATG",
                                                   PRODCATG_CUBE_OBJ);
                }

                BRK_IF_ERROR(
                    eris_ifu_jitter_save_cpl_cube(frameset, obj_set, objEntry->frame, parlist,
                                              NULL, objEntry->cube, objEntry->cubeBpm, objEntry->cubeHdr,
                                              params, TWEAKED_CUBE, ix, recipe_name));
			} /* end sky tweak handling */
			cpl_frameset_delete(obj_set);
            CHECK_ERROR_STATE();

			char* param_name;
			int aj_method = -1;
			const char* context = "eris.eris_ifu_jitter";
			param_name = cpl_sprintf("%s.aj-method", context);
			const cpl_parameter* p = NULL;
			p = cpl_parameterlist_find_const(parlist, param_name);
            if (p != NULL) {
			   aj_method = cpl_parameter_get_int(p);
			}
			cpl_free(param_name);

            if((objEntry->skyIndex < 0) && (aj_method > 3)) {
				/* Handle sky subtraction in cas no sky frame is present */
				eris_ifu_jitter_skycor_from_box(parlist, objEntry, aj_method);
			}

			if (params.darCorrection) {
                for (cpl_size zx = 0;zx<hdrl_imagelist_get_size(objEntry->cube);zx++){
                    tmpImg= hdrl_imagelist_get(objEntry->cube, zx);
                    tmpMask = cpl_mask_threshold_image_create(cpl_imagelist_get(objEntry->cubeBpm, zx), params.bpmThreshold, 1.5);;
                    hdrl_image_reject_from_mask(tmpImg, tmpMask);
                    eris_ifu_free_mask(&tmpMask);
                    CHECK_ERROR_STATE(); 
                }
                BRK_IF_ERROR(
                    eris_ifu_dar_correction(objEntry->cube,
                                            objEntry->cubeHdr, /*objEntry->cubeBpm,*/
                                            params.darShiftMethod,
                                            params.darShiftWidth, params.darShiftLength));
                cpl_frameset* obj_set2 =
                eris_ifu_frameset_extract_obj_set(frameset, sof, ix);

                eris_ifu_jitter_save_cpl_cube(frameset, obj_set2,
                		sof->exposureTable[ix].frame, parlist, NULL,
						objEntry->cube, objEntry->cubeBpm, objEntry->cubeHdr,
                        params, DAR_CUBE, ix, recipe_name);
                cpl_frameset_delete(obj_set2);
			}
        } // end for: ix < sof->exposureTableCnt

		cubeType sky_type = SKY_OBJECT_CUBE;
		/* TODO: In case of only one OBJ (or STD or PSF) input we need to create
		 * a cube object. Why? */
		char *proCatg = NULL;
        char *filenamePrefix = NULL;
		*obj_type = eris_ifu_jitter_get_obj_type(sof->mode);
		eris_ifu_jitter_get_procatg_and_filename(*obj_type, &proCatg, &filenamePrefix);
		eris_ifu_free_string(&filenamePrefix);
		cpl_frameset* cube_set;
		if ((cube_set = eris_ifu_extract_frameset(frameset, proCatg )) == NULL){
			for(cpl_size i = 0; i < sof->exposureTableCnt; i++) {
                if (!cpl_propertylist_has(sof->exposureTable[i].cubeHdr,"PRODCATG")) {
                    cpl_propertylist_append_string(sof->exposureTable[i].cubeHdr,
                                                   "PRODCATG", "ANCILLARY.EXPMAP");
                }

				if (sof->exposureTable[i].isObject) {
					cpl_frameset* obj_set =
					                eris_ifu_frameset_extract_obj_set(frameset, sof, i);
					*obj_type = eris_ifu_jitter_get_obj_type(sof->mode);
					eris_ifu_jitter_save_cpl_cube(frameset, obj_set,
                                                  sof->exposureTable[i].frame,
                                                  parlist, NULL, sof->exposureTable[i].cube,
                                                  sof->exposureTable[i].cubeBpm,
                                                  sof->exposureTable[i].cubeHdr,
                                                  params, *obj_type, i, recipe_name);
					cpl_frameset_delete(obj_set);
				} else {
					sky_type = eris_ifu_jitter_get_sky_type(sof->mode);
					eris_ifu_jitter_save_cpl_cube(frameset, frameset,
                                                  sof->exposureTable[i].frame,
                                                  parlist, NULL, sof->exposureTable[i].cube,
                                                  sof->exposureTable[i].cubeBpm,
                                                  sof->exposureTable[i].cubeHdr,
                                                  params, sky_type, i, recipe_name);
				}
			}
		}
        cpl_free(proCatg);
	}
	CATCH
	{
		retVal = cpl_error_get_code();
        CATCH_MSGS();
	}

	return retVal;
}


hdrl_image *eris_ifu_jitter_subtract_background(
        skyTweakModes sky_tweak,
		int objIdx,
		struct sofStruct *sof,
		productDepthType productDepth)
{
	hdrl_image *objImg = NULL;
	cpl_mask *inputMask = NULL;
    cpl_mask *outputMask = NULL;
    cpl_image *dqi = NULL;
	char *filename = NULL;
    char *filename2 = NULL;
	bool darkSkySubtracted = false;

	cpl_ensure(sof != NULL,CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{

		objImg = hdrl_image_duplicate(sof->exposureTable[objIdx].rawImage);
		inputMask = cpl_mask_duplicate(
				hdrl_image_get_mask(sof->exposureTable[objIdx].rawImage));

		if (sky_tweak == NONE) {
			const int skyIdx = sof->exposureTable[objIdx].skyIndex;
			if (skyIdx >=0) {
				hdrl_image *skyImg = hdrl_image_duplicate(sof->exposureTable[skyIdx].rawImage);
				BRK_IF_NULL(skyImg);
				BRK_IF_ERROR(hdrl_image_sub_image(objImg, skyImg));
				darkSkySubtracted = true;
				hdrl_image_delete(skyImg);
			} else if (sof->masterDark != NULL) {
				hdrl_image *rawImg = hdrl_image_duplicate(sof->masterDark);
				BRK_IF_ERROR(hdrl_image_sub_image(objImg, sof->masterDark));
                darkSkySubtracted = true;
                hdrl_image_delete(rawImg);
			}

			if (darkSkySubtracted) {
			    outputMask = cpl_mask_duplicate(hdrl_image_get_mask(objImg));
			    // detect new bad pixels
			    BRK_IF_ERROR(cpl_mask_xor(outputMask, inputMask));
			    BRK_IF_NULL(dqi = cpl_image_new_from_mask(outputMask));
			    BRK_IF_ERROR(cpl_image_multiply_scalar(dqi, ERIS_DQI_SKY));
			    BRK_IF_ERROR(cpl_image_add(
			        sof->exposureTable[objIdx].dqiImage, dqi));
			    if (productDepth >= PD_DEBUG) {
				    BRK_IF_NULL(
						filename = cpl_sprintf("%s_%3.3d",
								ERIS_IFU_PRO_JITTER_DBG_DARK_FN, objIdx));
                    BRK_IF_NULL(
                        filename2 = cpl_sprintf("%s_%3.3d.fits",
                                ERIS_IFU_PRO_JITTER_DBG_DARK_FN, objIdx));
                    BRK_IF_ERROR(
                        eris_ifu_save_hdrl_image_dbg(objImg, filename, 1, NULL));
                    BRK_IF_ERROR(
                        cpl_image_save(sof->exposureTable[objIdx].dqiImage,
                            filename2, CPL_TYPE_INT, NULL, CPL_IO_EXTEND));
			    }
			}
		}
		cpl_mask_delete(inputMask);
	}
	CATCH
	{
	}

    eris_ifu_free_image(&dqi);
    eris_ifu_free_mask(&outputMask);
	eris_ifu_free_string(&filename);
    eris_ifu_free_string(&filename2);
	return objImg;
}

/**
 * @brief function to generate a 3D data cube
 * @param  inputImage input image (brick-wall pattern, before re-sampling)
 * @param  sofIdx index of image in input sofStruct structure
 * @param  sofStruct data structure containing all input data
 * @param  params structure with all parameters required to reduce sci data
 * @param  doVelocityCorrection boolean switch to apply or not velocity corr
 * @param  ohLambdaCorrection polynomial coefficients with OH wave-cal corr
 * @param  productDepth product depth parameter
 *
 */
hdrl_imagelist *eris_ifu_jitter_build_cube(
		hdrl_image *inputImage,
		int sofIdx,
		struct sofStruct *sof,
		struct paramStruct params,
		bool doVelocityCorrection,
		cpl_polynomial *ohLambdaCorrection,
		productDepthType productDepth)
{
	hdrl_imagelist *cube = NULL;
	cpl_imagelist  *dataCube = NULL;
	cpl_imagelist  *errCube = NULL;
	hdrl_image *image1 = NULL;
	hdrl_image *image2 = NULL;
	hdrl_image *bpmImg1 = NULL;
	cpl_image *bpmImg2 = NULL;
	hdrl_image *badPixelMaskImg = NULL;
	cpl_image *badPixelMaskImg1 = NULL;
	cpl_image *badPixelMaskImg2 = NULL;

	hdrl_image *waveImg = NULL;
	cpl_image *waveDataImg = NULL;
	cpl_image *waveErrImg = NULL;
	cpl_image *waveMap = NULL;
	struct exposureEntry *objEntry;
	char *filename;
    char *filename2;
	double velocityCorr;
	cpl_size lsize;
	double *pWaveMap;
	double minLambda = 0.;
	double maxLambda = 0.;
	bool buildCubeForOHbasedLambdaCorrection = false;

	cpl_ensure(inputImage != NULL,CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(sof != NULL,CPL_ERROR_NULL_INPUT, NULL);
	//cpl_ensure(ohLambdaCorrection != NULL,CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
	    if (doVelocityCorrection == false && ohLambdaCorrection == NULL) {
	        buildCubeForOHbasedLambdaCorrection = true;
	    }

		objEntry = &sof->exposureTable[sofIdx];
		BRK_IF_NULL(image1 = hdrl_image_duplicate(inputImage));

		//divide by flat image (if available)
		if (sof->masterFlat != NULL) {
			//BRK_IF_ERROR(
			hdrl_image_div_image(image1, sof->masterFlat);
			if (productDepth >= PD_DEBUG) {
			    if (buildCubeForOHbasedLambdaCorrection) {
			        BRK_IF_NULL(
						filename = cpl_sprintf("%sB_%3.3d",
						    ERIS_IFU_PRO_JITTER_DBG_FLAT_FN, sofIdx));
			    } else {
			        BRK_IF_NULL(
			            filename = cpl_sprintf("%s_%3.3d",
			                ERIS_IFU_PRO_JITTER_DBG_FLAT_FN, sofIdx));
			    }
				eris_ifu_save_hdrl_image_dbg(image1, filename, 1, NULL);
				eris_ifu_free_string(&filename);
			}
		}

		//create a bad pixel mask image
		BRK_IF_NULL(
				badPixelMaskImg = hdrl_image_new(
						cpl_mask_get_size_x(objEntry->badPixelMask),
						cpl_mask_get_size_y(objEntry->badPixelMask)));
		BRK_IF_NULL(
				badPixelMaskImg1 = cpl_image_new_from_mask(objEntry->badPixelMask));
		BRK_IF_NULL(
				badPixelMaskImg2 = cpl_image_cast(
						badPixelMaskImg1, CPL_TYPE_DOUBLE));
		BRK_IF_ERROR(
				hdrl_image_insert(badPixelMaskImg,
						badPixelMaskImg2, NULL, 1, 1));
		eris_ifu_free_image(&badPixelMaskImg1);
        eris_ifu_free_image(&badPixelMaskImg2);

        cpl_msg_info(__func__, "No. bad pixels: %d", (int)cpl_image_count_rejected(hdrl_image_get_image(image1))); 

        // Correct the flatfielded image1 according to the badPixelMaskImg

        int bpc_iter = params.bpc_iter;
        for (int i = 0; i < bpc_iter; i++){
            eris_ifu_bpm_correction(image1, badPixelMaskImg);

            if (productDepth >= PD_DEBUG) {
                if (buildCubeForOHbasedLambdaCorrection) {
                    BRK_IF_NULL(
                        filename = cpl_sprintf("%sB_%3.3d_%d",
                            ERIS_IFU_PRO_JITTER_DBG_FLAT_FN"-corr", sofIdx, i));
                } else {
                    BRK_IF_NULL(
                        filename = cpl_sprintf("%s_%3.3d_%d",
                            ERIS_IFU_PRO_JITTER_DBG_FLAT_FN"-corr", sofIdx, i));
                }
                eris_ifu_save_hdrl_image_dbg(image1, filename, 1, NULL);
                eris_ifu_free_string(&filename);
            }
        }


        // apply derotator correction
        double derot_corr; 
        if (params.derot_corr!=params.derot_corr)
            derot_corr = objEntry->derot_corr;
        else
            derot_corr = params.derot_corr; 
        cpl_msg_info(cpl_func, "First column shifted (derot_corr): %.2f", derot_corr); 

        for (int i = 0; i < SLITLET_CNT; i++) {
            if (sof->distortion[i] != NULL) {
                BRK_IF_ERROR(
                    cpl_polynomial_shift_1d(sof->distortion[i], 0, derot_corr));
            }
        }

		//correct for distortion
		if (sof->distortion != NULL) {
			BRK_IF_NULL(
					image2 = eris_ifu_dist_warp_image(image1,
							sof->distortion, sof->borders));
			BRK_IF_NULL(
                    bpmImg1 = eris_ifu_dist_warp_bpm(badPixelMaskImg,
                            sof->distortion, sof->borders, productDepth));
        } else { //AGUDO3: we will never get here, since eris_ifu_jitter_processSof() will exit when distortion==NULL
			BRK_IF_NULL(
					image2 = eris_ifu_warp_polynomial_image(image1,
							sof->poly_u, sof->poly_v));
			BRK_IF_NULL(
					bpmImg1 = eris_ifu_warp_polynomial_image(badPixelMaskImg,
							sof->poly_u, sof->poly_v));
		}

        // shift distortion back for next iteration 
        for (int i = 0; i < SLITLET_CNT; i++) {
            if (sof->distortion[i] != NULL) {
                BRK_IF_ERROR(
                    cpl_polynomial_shift_1d(sof->distortion[i], 0, -1*derot_corr));
            }
        }

		CHECK_ERROR_STATE();

		if (productDepth >= PD_DEBUG) {
            if (buildCubeForOHbasedLambdaCorrection) {
                BRK_IF_NULL(
					filename = cpl_sprintf("%sB_%3.3d",
					    ERIS_IFU_PRO_JITTER_DBG_WRAP_FN, sofIdx));
            } else {
                BRK_IF_NULL(
                    filename = cpl_sprintf("%s_%3.3d",
                        ERIS_IFU_PRO_JITTER_DBG_WRAP_FN, sofIdx));
            }
			eris_ifu_save_hdrl_image_dbg(image2, filename, 1, NULL);
			if (bpmImg1 != NULL) {
				BRK_IF_NULL(
						filename2 = cpl_sprintf("%s.fits", filename));
                BRK_IF_ERROR(
                    cpl_image_save(hdrl_image_get_image(bpmImg1), filename2,
                        CPL_TYPE_FLOAT, NULL, CPL_IO_EXTEND));
                eris_ifu_free_string(&filename2);
			}
            eris_ifu_free_string(&filename);
		}

		waveMap = cpl_image_duplicate(sof->waveMap);
		lsize = cpl_image_get_size_x(waveMap) * cpl_image_get_size_y(waveMap);
		BRK_IF_NULL(pWaveMap = cpl_image_get_data_double(waveMap));
		maxLambda = cpl_image_get_max(waveMap);
		minLambda = cpl_image_get_min(waveMap);

		// velocity shift
		if (doVelocityCorrection) {
			velocityCorr = 1. + params.velocityOffset * 1000. / CPL_PHYS_C;
			for (int ix = 0; ix < lsize; ix++) {
				pWaveMap[ix] *= velocityCorr;
			}

		}
		// OH line lambda correction
		if (ohLambdaCorrection != NULL) {
			for (int ix = 0; ix < lsize; ix++) {
				pWaveMap[ix] -= cpl_polynomial_eval_1d(
						ohLambdaCorrection, pWaveMap[ix], NULL);
			}
		}

		//wavelength resampling
		/* to prevent memory leaks delete header before duplication */
		cpl_propertylist_delete(objEntry->cubeHdr);
	        objEntry->cubeHdr = cpl_propertylist_duplicate(objEntry->hdr);
		BRK_IF_NULL(
				waveDataImg = eris_ifu_wave_resampled_arc_image(
						hdrl_image_get_image(image2),
						waveMap, sof->band, minLambda, maxLambda,
						objEntry->cubeHdr, 3));
		BRK_IF_NULL(
				waveErrImg = eris_ifu_wave_resampled_arc_image(
						hdrl_image_get_error(image2),
						waveMap, sof->band, minLambda, maxLambda,
						objEntry->cubeHdr, 3));

		BRK_IF_NULL(
				bpmImg2 = eris_ifu_wave_resampled_arc_image(
						hdrl_image_get_image(bpmImg1),
						waveMap, sof->band, minLambda, maxLambda,
						objEntry->cubeHdr, 3));

		cpl_binary *pImgMask = cpl_mask_get_data(cpl_image_get_bpm(waveDataImg));
		float *pDqiData = cpl_image_get_data_float(bpmImg2);
		cpl_size size = cpl_image_get_size_x(waveDataImg)
		        * cpl_image_get_size_y(waveDataImg);
		for (cpl_size px = 0; px < size; px++) {
		    if (pDqiData[px] > params.bpmThreshold) {
                pImgMask[px] = BAD_PIX;
		    }
		}
        waveImg = hdrl_image_create(waveDataImg,waveErrImg);
        CHECK_ERROR_STATE();

		if (productDepth >= PD_DEBUG) {
			cpl_propertylist *pl2save = cpl_propertylist_duplicate(objEntry->hdr);
			// call eris_ifu_wave_resampled_arc_image again to get property list
			// entries for image (instead of cube)
			cpl_image *image2save = eris_ifu_wave_resampled_arc_image(
					hdrl_image_get_image(image2),
					waveMap, sof->band, minLambda, maxLambda, pl2save, 2);
            if (buildCubeForOHbasedLambdaCorrection) {
                BRK_IF_NULL(
					filename = cpl_sprintf("%sB_%3.3d",
					    ERIS_IFU_PRO_JITTER_DBG_WAVE_FN, sofIdx));
            } else {
                BRK_IF_NULL(
                    filename = cpl_sprintf("%s_%3.3d",
                        ERIS_IFU_PRO_JITTER_DBG_WAVE_FN, sofIdx));
            }
			cpl_propertylist_erase(pl2save,"RADECSYS");
			eris_ifu_save_hdrl_image_dbg(waveImg, filename, 1, pl2save);
			eris_ifu_free_image(&image2save);

			image2save = eris_ifu_wave_resampled_arc_image(
					hdrl_image_get_image(bpmImg1),
					waveMap, sof->band, minLambda, maxLambda, pl2save, 2);
			BRK_IF_NULL(
					filename2 = cpl_sprintf("%s.fits", filename));
			BRK_IF_ERROR(
			        cpl_image_save(image2save, filename2,
			            CPL_TYPE_FLOAT, pl2save, CPL_IO_EXTEND));
			eris_ifu_free_string(&filename);
            eris_ifu_free_string(&filename2);
			eris_ifu_free_image(&image2save);
			eris_ifu_free_propertylist(&pl2save);
			CHECK_ERROR_STATE();
		}

		BRK_IF_NULL(
				dataCube = eris_ifu_jitter_reconstruct_cube(
						waveDataImg,
						params.slitletDetectionMode,
						params.fineTuneMode,
						params.firstCol,
						sof->distances,
						sof->positions));
		cpl_image_reject_from_mask(waveErrImg, cpl_image_get_bpm_const(waveDataImg));
		BRK_IF_NULL(
				errCube = eris_ifu_jitter_reconstruct_cube(
						waveErrImg,
						params.slitletDetectionMode,
						params.fineTuneMode,
						params.firstCol,
						sof->distances,
						sof->positions));
		BRK_IF_ERROR(eris_ifu_mask_nans_in_cube(dataCube));
		cube = hdrl_imagelist_create(dataCube, errCube);
        cpl_imagelist_delete(dataCube);
        cpl_imagelist_delete(errCube);
		cpl_propertylist_update_int(objEntry->cubeHdr,NAXIS1,64);
		cpl_propertylist_update_int(objEntry->cubeHdr,NAXIS2,64);
		cpl_propertylist_update_int(objEntry->cubeHdr,NAXIS3,
				hdrl_imagelist_get_size(cube));

		if(objEntry->cubeBpm != NULL) {
			cpl_imagelist_delete(objEntry->cubeBpm);
		}
            // AMO: commented to fix PIPE-11106  TODO: Why a multiplication by 2??? That was creating a NAN in collapsed cube
	    cpl_image_multiply_scalar(bpmImg2, 2.0);
		
        BRK_IF_NULL(
				objEntry->cubeBpm = eris_ifu_jitter_reconstruct_cube(
						bpmImg2,
						params.slitletDetectionMode,
						params.fineTuneMode,
						params.firstCol,
						sof->distances,
						sof->positions));

	}
	CATCH
	{
	}
        /* TODO: AMo does not understand why objEntry is created and cannot be
         * erased. In particular objEntry->cubeBpm is an object created but not
         * erased
         */
	//eris_free_exposureEntry(objEntry);
	eris_ifu_free_hdrl_image(&waveImg);
	eris_ifu_free_hdrl_image(&image1);
	eris_ifu_free_hdrl_image(&image2);
	eris_ifu_free_hdrl_image(&badPixelMaskImg);
	eris_ifu_free_image(&badPixelMaskImg1);
	eris_ifu_free_image(&badPixelMaskImg2);
	eris_ifu_free_image(&waveDataImg);
	eris_ifu_free_image(&waveErrImg);
	eris_ifu_free_image(&waveMap);
	eris_ifu_free_hdrl_image(&bpmImg1);
	eris_ifu_free_image(&bpmImg2);
	eris_check_error_code("eris_ifu_jitter_build_cube");
	return cube;
}

cpl_error_code
eris_image_get_valid_data_range(cpl_image* image,
		cpl_size* min, cpl_size* max)
{
	cpl_size sx = cpl_image_get_size_x(image);
	cpl_size sy = cpl_image_get_size_y(image);
	float* pdata = cpl_image_get_data(image);
	cpl_size rows2check = 200;
	cpl_size nan_counter = 0;
	double nan_thresh = 0.5;
	*min = 0;
	*max = sy;
	for(cpl_size j = 0;  j < rows2check; j++) {
		nan_counter = 0;
		for(cpl_size i = 0;  i < sx; i++) {
			if(isnan(pdata[i + j * sx])) {
				nan_counter++;
			}
		}

		if(nan_counter > nan_thresh * sx) {
			*min = j;
		}
	}

	for(cpl_size j = sy-1;  j >= sy - rows2check; j--) {
		nan_counter = 0;
		for(cpl_size i = 0;  i < sx; i++) {
			if(isnan(pdata[i + j * sx])) {
				nan_counter++;
			}
		}
		if(nan_counter > nan_thresh * sx) {
			*max = j;
		}
	}
	cpl_msg_info(cpl_func,"min: %lld max: %lld", *min, *max);

	return cpl_error_get_code();
}
/**
  @brief  Function to create a 3D-data cube from a  re-sampled image

  @param  resampledDetImage  the re-sampled image
  @param  mode parameter to control the slitlet detection algorithm used
  @param  fineTuneMode Specifies the row interpolation mode using GSL routines
          0: no interpolation
          >2: polynomial interpolation, polynomial degree is interType-1
           2: linear interpolation
          -1: Cubic spline with natural boundary conditions
          -2: cubic spline with periodic boundary conditions
          -3: Non-rounded Akima spline with natural boundary conditions.
          -4: Non-rounded Akima spline with periodic boundary conditions.
          -5: Steffen's method guarantees the monotonicity of the
              interpolating function between the given data points
    @param firsCol  first column of each slitlet used to re-sample image.
    @param distancesV output computed slitlets distances
    @param positionsVV output computed slutlets positions
    @return reconstructed 3D data cube (cpl_imagelist object)
  */

cpl_imagelist * eris_ifu_jitter_reconstruct_cube(
		cpl_image *resampledDetImage,
		slitletDetectionModes mode,
		int fineTuneMode,
		double firstCol,
		cpl_vector  *distancesV,
		cpl_bivector *positionsVV)
{

	cpl_ensure(resampledDetImage, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure((mode >= UNSET) && (mode <= GRID), CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(fineTuneMode >= -5, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(firstCol >= 0 && firstCol <= ERIS_IFU_DETECTOR_SIZE_X, CPL_ERROR_ILLEGAL_INPUT, NULL);
	//cpl_ensure(distancesV, CPL_ERROR_NULL_INPUT, NULL);
	//cpl_ensure(positionsVV, CPL_ERROR_NULL_INPUT, NULL);

	cpl_imagelist *cube = NULL;
	cpl_image *slide = NULL;
	cpl_size inx;
	cpl_size onx;
	cpl_size ony;
	cpl_size onz;
	int beginCol[SLITLET_CNT] = {0};
	double start[SLITLET_CNT] = {0.};
	// variables needed for mode DISTANCES
	double di;
	double *distances = NULL;
	// variables needed for mode EDGES
	double center;
	double *left_edges = NULL;
	double *right_edges = NULL;
	float *pi = NULL;
    float *po = NULL;
    cpl_binary *mi = NULL;
    cpl_binary *mo = NULL;
	double *pt1 = NULL;
	double *pt2 = NULL;
	double *xi = NULL;
	double *xt = NULL;
	int icol;
	float tmp;

	TRY
	{
		inx = cpl_image_get_size_x(resampledDetImage);
		onx = inx / SLITLET_CNT;
		ony = SLITLET_CNT * 2;
		onz = cpl_image_get_size_y(resampledDetImage);
		cube = cpl_imagelist_new();
		/*
		cpl_size iz_min = 0;
		cpl_size iz_max = onz;
		eris_image_get_valid_data_range(resampledDetImage, &iz_min, &iz_max);
		*/
		if (distancesV == NULL) { // new ERIS way of simple re-sampling
			pi = cpl_image_get_data_float(resampledDetImage);
			mi = cpl_mask_get_data(cpl_image_get_bpm(resampledDetImage));

			for (int iz=0; iz<onz; iz++) {
				BRK_IF_NULL(
						slide = cpl_image_new(onx, ony, CPL_TYPE_FLOAT));
				po = cpl_image_get_data_float(slide);
				mo =  cpl_mask_get_data(cpl_image_get_bpm(slide));

				// flip x-axis to correct east direction (shall go to the left)
				for (int slitlet=0; slitlet<SLITLET_CNT; slitlet++) {
					for (int col=0; col < onx; col++) {
						tmp = (float) pi[onx-1-col + slitlet*onx + iz*inx];
                        /* this step expands of a factor 2 along Y the
						 * rectangular spaxel to a make it appear as square
						 * as it is on the Sky. To conserve flux we need to
						 * divide by 2 (or multiply by 0.5)
                         */
                        tmp *= 0.5;
						po[col + onx*(rowIndices[slitlet]*2)]   = tmp;
						po[col + onx*(rowIndices[slitlet]*2+1)] = tmp;
                        if (mi[onx-1-col + slitlet*onx + iz*inx] == BAD_PIX) {
                            mo[col + onx*(rowIndices[slitlet]*2)] = BAD_PIX;
                            mo[col + onx*(rowIndices[slitlet]*2+1)] = BAD_PIX;
						}

					}
				}

				BRK_IF_ERROR(
						cpl_imagelist_set(cube, slide, iz));
			}
		} else { // old SINFONI way
			if (mode == DISTANCES) {
				if (distancesV == NULL) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
							"missing data about slitlet distances");
				}
				if (cpl_vector_get_size(distancesV) != SLITLET_CNT-1) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_INCOMPATIBLE_INPUT,
							"distances vector has wrong number of elements "
							"(got %lld but exepected %d)",
							cpl_vector_get_size(distancesV), SLITLET_CNT-1);
				}
				distances = cpl_vector_get_data(distancesV);
			} else if (mode == EDGES) {
				if (positionsVV == NULL) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
							"missing data about slitlet positions");
				}
				if (cpl_bivector_get_size(positionsVV) != SLITLET_CNT) {
					BRK_WITH_ERROR_MSG(CPL_ERROR_INCOMPATIBLE_INPUT,
							"slit pos bivector has wrong number of elements "
							"(got %lld but exepected %d)",
							cpl_bivector_get_size(positionsVV), SLITLET_CNT);
				}
				left_edges = cpl_vector_get_data(
						cpl_bivector_get_x(positionsVV));
				right_edges = cpl_vector_get_data(
						cpl_bivector_get_y(positionsVV));
			}

			di = 0;
			for (int slitlet=0; slitlet<SLITLET_CNT; slitlet++) {
				if (mode == DISTANCES) {
					if (slitlet == 0) {
						start[slitlet] = firstCol;
					} else {
						di += distances[slitlet-1];
						start[slitlet] = firstCol + di;
					}
				} else if (mode == EDGES) {
					center = (left_edges[slitlet] + right_edges[slitlet]) / 2.;
					start[slitlet] = center - (double) (onx -1 )/2.;
				}else if (mode == GRID) {
					start[slitlet] = (double) (onx * slitlet);
				}
				beginCol[slitlet] =
						start[slitlet] > 0 ?
								(int)(start[slitlet]+.5) : (int)(start[slitlet]-.5);
				//            double shift[SLITLET_CNT];
				//            shift[rowIndices[slitlet]] = start[slitlet] -
						//                    (double) beginCol[slitlet];
			}

			BRK_IF_NULL(
					xi = cpl_calloc(onx, sizeof(double)));
			BRK_IF_NULL(
					xt = cpl_calloc(onx, sizeof(double)));
			BRK_IF_NULL(
					pt1 = cpl_calloc(onx, sizeof(double)));
			BRK_IF_NULL(
					pt2 = cpl_calloc(onx, sizeof(double)));
			pi = cpl_image_get_data_float(resampledDetImage);

			for (int iz=0; iz<onz; iz++) {
				BRK_IF_NULL(
						slide = cpl_image_new(onx, ony, CPL_TYPE_FLOAT));
				po = cpl_image_get_data_float(slide);

				for (int slitlet=0; slitlet<SLITLET_CNT; slitlet++) {
					for (int col=0; col < onx; col++) {
						xi[col] = (double) (beginCol[slitlet] + col);
						xt[col] = start[slitlet] + col;
						icol =  beginCol[slitlet] + col;
						if (icol < 0) {
							pt1[col] = pi[0 + iz * inx];
						} else if (icol >= inx) {
							pt1[col] = pi[(inx - 1) + iz * inx];
						} else {
							pt1[col] = pi[icol + iz * inx];
						}
					}
					if (fineTuneMode == 0) {
						for (int col=0; col < onx; col++) {
							pt2[col] = pt1[col];
						}
					} else {
						BRK_IF_ERROR(
								eris_ifu_1d_interpolation(xi, pt1, (int) onx,
										xt, pt2, (int) onx, fineTuneMode));
					}
					for (int col=0; col < onx; col++) {
						/* this step expands of a factor 2 along Y the
						 * rectangular spaxel to a make it appear as square
						 * as it is on the Sky. To conserve flux we need to
						 * divide by 2 (or multiply by 0.5)
						 * TODO: why cast to float? It should be all in double
						 * precision
						 */
                         tmp = (float) (pt2[col] * 0.5);
						 po[col + onx*(rowIndices[slitlet]*2)]   = tmp;
						 po[col + onx*(rowIndices[slitlet]*2+1)] = tmp;
					}
				}
				BRK_IF_ERROR(
						cpl_imagelist_set(cube, slide, iz));
			}
		}
		BRK_IF_ERROR(eris_ifu_mask_nans_in_cube(cube));
	}
	CATCH
	{
		eris_ifu_free_imagelist(&cube);
	}
	return cube;
}
/**
  @brief  Get the value of the PRO.CATG and the filename as function of the type of cube

  @param  type  the cube type
  @param[out]  proCatg cube PRO.CATG
  @param[out]  filenamePrefix cube filename
  */
cpl_error_code
eris_ifu_jitter_get_procatg_and_filename(cubeType type, char **proCatg,
		char    **filenamePrefix)
{
	cpl_ensure_code(proCatg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(filenamePrefix, CPL_ERROR_NULL_INPUT);

	switch (type) {
	case OBJECT_CUBE_COADD:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD_FN);
		break;
    case TWEAKED_CUBE_COADD:
        *proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD);
        *filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD_FN);
        break;
    case DAR_CUBE_COADD:
        *proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE_COADD);
        *filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE_COADD_FN);
        break;
    case DAR_STD_CUBE_COADD:
           *proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_COADD);
           *filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_FN);
           break;
	case STD_CUBE_COADD:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_COADD);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_FN);
		break;
	case STD_FLUX_CUBE_COADD:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_FN);
		break;
	case STD_FLUX_CUBE_COADD_NOFLAT:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT_FN);
		break;
	case STD_CUBE_COADD_NOFLAT:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_NOFLAT);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_NOFLAT_FN);
		break;
	case PSF_CUBE_COADD:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD_FN);
		break;
	case OBJECT_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_OBJ_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_OBJ_CUBE_FN);
		break;
	case STD_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_FN);
		break;
	case STD_FLUX_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_FN);
		break;
	case STD_CUBE_NOFLAT:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_NOFLAT);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_NOFLAT_FN);
		break;
	case PSF_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE_FN);
		break;
	case SKY_OBJECT_CUBE:
	case SKY_STD_CUBE:
	case SKY_STD_FLUX_CUBE:
	case SKY_PSF_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_SKY_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_SKY_CUBE_FN);
		break;
	case TWEAKED_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_TWK_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_TWK_CUBE_FN);
		break;
	case TWEAKED_STD_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_FN);
		break;
	case DAR_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE_FN);
		break;
		//PIPPO
	case DAR_PSF_CUBE_COADD:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD_FN);
		break;
	case DAR_STD_CUBE:
		//*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE);
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_CUBE_FN);
		break;
	case DAR_STD_FLUX_CUBE:
		//*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE);
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_FN);
		break;
	case DAR_PSF_CUBE:
		//*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_DAR_CUBE);
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_PSF_CUBE_FN);
		break;
	case BPM_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_BPM_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_BPM_CUBE_FN);
		break;
	case JITTER_CUBE:
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_CUBE_FN);
		break;
	default:
		cpl_msg_warning(cpl_func,"default case %d",type);
		*proCatg = cpl_sprintf(ERIS_IFU_PRO_JITTER_CUBE);
		*filenamePrefix = cpl_sprintf(ERIS_IFU_PRO_JITTER_CUBE_FN);
		break;
	}
	cpl_msg_debug(cpl_func,"proCatg: %s",*proCatg);
	cpl_msg_debug(cpl_func,"filenamePrefix: %s",*filenamePrefix);
	return cpl_error_get_code();
}



/**
  @brief Extract a spectrum

  @param  data_in  input 3D data cube
  @param  noise_in  input 3D errors cube
  @param  mask      input pixel mask
  @param[out]  spec_data_out extracted spectrum
  @param[out]  spec_noise_out error associated to spec_data_out
  @return cpl_error_code error status
  */
cpl_error_code eris_ifu_extract_spec(
		const cpl_imagelist *data_in,
		const cpl_imagelist *noise_in,
		cpl_image     *mask,
		cpl_vector    **spec_data_out,
		cpl_vector    **spec_noise_out)
{
	cpl_error_code  ret_error       = CPL_ERROR_NONE;

	const cpl_image  *tmp_img        = NULL;
	const cpl_mask   *bpm            = NULL;
	const cpl_binary *bpm_data       = NULL;
	cpl_mask         *empty_mask     = NULL;

	int             nx              = 0,
			ny              = 0,
			nz              = 0,
			i               = 0,
			j               = 0,
			g               = 0;

	const float    *pmask          = NULL;
	const double   *ptmp_img       = NULL;

	double          sum             = 0.0,
			weights         = 0.0,
			*pvec           = NULL,
			*pvecn          = NULL;

	TRY
	{
		ASSURE((data_in != NULL) &&
				(spec_data_out != NULL),
				CPL_ERROR_NULL_INPUT,
				"Not all input data is provided!");

		BRK_IF_NULL(
				tmp_img = cpl_imagelist_get_const(data_in, 0));

		nx = cpl_image_get_size_x(tmp_img);
		ny = cpl_image_get_size_y(tmp_img);
		nz = cpl_imagelist_get_size(data_in);

		if (mask != NULL) {
			ASSURE((nx == cpl_image_get_size_x(mask)) &&
					(ny == cpl_image_get_size_y(mask)),
					CPL_ERROR_ILLEGAL_INPUT,
					"Data and mask don't have same dimensions!");

			BRK_IF_NULL(
					pmask = cpl_image_get_data_float_const(mask));
		}

		if (noise_in != NULL) {
			ASSURE(spec_noise_out != NULL,
					CPL_ERROR_NULL_INPUT,
					"Not all input data is provided!");

			BRK_IF_NULL(
					tmp_img = cpl_imagelist_get_const(noise_in, 0));

			ASSURE((nx == cpl_image_get_size_x(tmp_img)) &&
					(ny == cpl_image_get_size_y(tmp_img)) &&
					(nz == cpl_imagelist_get_size(noise_in)),
					CPL_ERROR_ILLEGAL_INPUT,
					"Data and noise don't have same dimensions!");
		}

		BRK_IF_NULL(
				*spec_data_out = cpl_vector_new(nz));

		BRK_IF_NULL(
				pvec = cpl_vector_get_data(*spec_data_out));

		if (noise_in != NULL) {
			BRK_IF_NULL(
					*spec_noise_out = cpl_vector_new(nz));

			BRK_IF_NULL(
					pvecn = cpl_vector_get_data(*spec_noise_out));
		}

		BRK_IF_NULL(
				empty_mask = cpl_mask_new(nx,ny));

		// loop over all spatial slices
		for (g = 0; g < nz; g++) {
			BRK_IF_NULL(
					tmp_img = cpl_imagelist_get_const(data_in, g));
			BRK_IF_NULL(
					ptmp_img = cpl_image_get_data_double_const(tmp_img));

			bpm = cpl_image_get_bpm_const(tmp_img);
			if (bpm == NULL) {
				bpm = empty_mask;
			}
			BRK_IF_NULL(
					bpm_data = cpl_mask_get_data_const(bpm));

			// extract spectrum for data
			sum = 0.0;
			weights = 0.0;
			for (j = 0; j < ny; j++) {
				for (i = 0; i < nx; i++) {
					// sum weighted pixels in spatial plane
                    if ((bpm_data[i+j*nx] == GOOD_PIX) && isfinite(ptmp_img[i+j*nx])) {
						if (mask != NULL) {
							sum += ptmp_img[i+j*nx] * pmask[i+j*nx];
							weights += pmask[i+j*nx];
						} else {
							sum += ptmp_img[i+j*nx];
							weights += 1.;
						}
					}
				}
			}
			pvec[g] = sum/weights;


			// extract spectrum for noise
			if (noise_in != NULL) {
				BRK_IF_NULL(
						tmp_img = cpl_imagelist_get_const(noise_in, g));
				BRK_IF_NULL(
						ptmp_img = cpl_image_get_data_double_const(tmp_img));

				sum = 0.0;
				weights = 0.0;
				for (j = 0; j < ny; j++) {
					for (i = 0; i < nx; i++) {
						// combine weighted pixels in spatial plane
                        if ((bpm_data[i+j*nx] == GOOD_PIX) && isfinite(ptmp_img[i+j*nx])) {
							if (mask != NULL) {
								sum += pow(ptmp_img[i+j*nx], 2) * pow(pmask[i+j*nx], 2);
								weights += pow(pmask[i+j*nx], 2);
							} else {
								sum += pow(ptmp_img[i+j*nx], 2);
								weights += 1.;
							}
						}
					}
				}
				pvecn[g] = sqrt(sum/weights);
			}
		}
	}
	CATCH
	{
		CATCH_MSG();
		ret_error = cpl_error_get_code();
	}

	if (empty_mask != NULL) {cpl_mask_delete(empty_mask);}

	return ret_error;
}

/**
  @brief function to save products of jitter recipes

  @param  frames  input frame set
  @param  parlist  input parameterlist
  @param  sof      eris data structure containing inputs
  @param  spectrum extracted spectrum
  @param  error  error associated at the spectrum
  @param  totalFlux  total extracted flux (sum)
  @param  productDepth  parameter controlling how many products to create
  @param  is_sdp_format is the product to be saved as phase3 ?
  @param  recipe_name recipe name

  @return cpl_error_code error status
  */
static
cpl_error_code eris_ifu_jitter_spec_save_products(
		cpl_frameset* frames,
		const cpl_parameterlist * parlist,
		struct esSofStruct sof,
		cpl_bivector *spectrum,
		cpl_vector *error,
        cpl_vector *totalFlux,
		productDepthType productDepth,
		cpl_boolean is_sdp_format,
		const char* recipe_name)
{
	cpl_table           *pro_tab = NULL;
	cpl_propertylist    *pl;
	cpl_vector          *lambda = NULL;
	cpl_vector          *data = NULL;
	cpl_array           *waveArray = NULL;
	cpl_array           *fluxArray = NULL;
	cpl_array           *errArray = NULL;
    cpl_array           *totalFluxArray = NULL;
	double              firstLambda;
	double              deltaLambda;
	cpl_size            spectrumSize;

	TRY
	{
		spectrumSize = cpl_bivector_get_size(spectrum);
		BRK_IF_NULL(lambda = cpl_bivector_get_x(spectrum));
		BRK_IF_NULL(data = cpl_bivector_get_y(spectrum));
		firstLambda = cpl_vector_get(lambda,0);
		deltaLambda = cpl_vector_get(lambda,1) - firstLambda;

		if (productDepth >= PD_AUXILLIARY) {
			pl = cpl_propertylist_duplicate(sof.header);
			cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG, "SPECTRUM");
			cpl_propertylist_update_string(pl, "CTYPE1", "WAVE");
			cpl_propertylist_update_double(pl, "CRPIX1", 1.);
			cpl_propertylist_update_double(pl, "CRVAL1", firstLambda);
			cpl_propertylist_update_double(pl, "CDELT1", deltaLambda);
			cpl_propertylist_erase(pl, "CTYPE2");
			cpl_propertylist_erase(pl, "CRPIX2");
			cpl_propertylist_erase(pl, "CRVAL2");
			cpl_propertylist_erase(pl, "CDELT2");
			cpl_propertylist_erase(pl, "CTYPE3");
			cpl_propertylist_erase(pl, "CRPIX3");
			cpl_propertylist_erase(pl, "CRVAL3");
			cpl_propertylist_erase(pl, "CDELT3");
			cpl_propertylist_update_double(pl, "CD1_1", 0.);
			cpl_propertylist_update_double(pl, "CD1_2", deltaLambda);
			cpl_propertylist_erase(pl, "CD1_3");
			cpl_propertylist_erase(pl, "CD2_1");
			cpl_propertylist_erase(pl, "CD2_2");
			cpl_propertylist_erase(pl, "CD2_3");
			cpl_propertylist_erase(pl, "CD3_1");
			cpl_propertylist_erase(pl, "CD3_2");
			cpl_propertylist_erase(pl, "CD3_3");

			CHECK_ERROR_STATE();
			BRK_IF_ERROR(
					cpl_vector_save(data, "spectrum_vector.fits",
							CPL_TYPE_DOUBLE, pl, CPL_IO_CREATE));
			cpl_propertylist_delete(pl);
		}
		pl = cpl_propertylist_duplicate(sof.header);

		if(strstr(recipe_name,"no_flat") != NULL) {
			cpl_msg_info(cpl_func,"recipe_name: %s",recipe_name);
			cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG,
					ERIS_IFU_PRO_JITTER_SPECTRUM_NOFLAT);
		} else {
			cpl_msg_info(cpl_func,"recipe_name: %s",recipe_name);
			cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG,
					ERIS_IFU_PRO_JITTER_SPECTRUM);
		}



		cpl_propertylist_update_string(pl, "ESO PRO REC1 ID","eris_ifu_jitter");
		if(is_sdp_format) {
			waveArray = cpl_array_wrap_double(cpl_vector_get_data(lambda),
                                              spectrumSize);
			fluxArray = cpl_array_wrap_double(cpl_vector_get_data(data),
                                              spectrumSize);
			errArray = cpl_array_wrap_double(cpl_vector_get_data(error),
                                              spectrumSize);
            if (totalFlux != NULL) {
                totalFluxArray = cpl_array_wrap_double(cpl_vector_get_data(totalFlux),
                                              spectrumSize);
            }


			pro_tab = cpl_table_new(1);
			cpl_table_new_column_array(pro_tab, "WAVE",
					CPL_TYPE_DOUBLE, spectrumSize);
			cpl_table_new_column_array(pro_tab, "FLUX",
					CPL_TYPE_DOUBLE, spectrumSize);
			cpl_table_new_column_array(pro_tab, "ERR",
					CPL_TYPE_DOUBLE, spectrumSize);
            cpl_table_new_column_array(pro_tab, "TOT_FLUX",
                    CPL_TYPE_DOUBLE, spectrumSize);
			cpl_table_set_array(pro_tab, "WAVE", 0, waveArray);
			cpl_table_set_array(pro_tab, "FLUX", 0, fluxArray);
			cpl_table_set_array(pro_tab, "ERR",  0, errArray);
            if (totalFlux != NULL) {
                cpl_table_set_array(pro_tab, "TOT_FLUX",  0, totalFluxArray);
            }
			//cpl_table_save(pro_tab, pl, NULL, "spectrum.fits", CPL_IO_CREATE);
		} else {
			pro_tab = cpl_table_new(spectrumSize);
			cpl_table_wrap_double(pro_tab, cpl_vector_get_data(lambda), "WAVE");
            if (totalFlux != NULL) {
                cpl_table_wrap_double(pro_tab, cpl_vector_get_data(data), "FLUX");
            } else {
                cpl_table_wrap_double(pro_tab, cpl_vector_get_data(data), "TOT_FLUX");
            }
			cpl_table_wrap_double(pro_tab, cpl_vector_get_data(error), "ERR");
            if (totalFlux != NULL) {
                cpl_table_wrap_double(pro_tab, cpl_vector_get_data(totalFlux), "TOT_FLUX");
            }
		}
		char *filename = cpl_sprintf("%s_spectrum.fits", recipe_name);
        cpl_propertylist_append_string(pl, "PRODCATG", "SCIENCE.SPECTRUM");
		cpl_dfs_save_table(frames, NULL, parlist, frames, NULL,
				pro_tab, NULL, recipe_name, pl,
				NULL, PACKAGE "/" PACKAGE_VERSION, filename);

		cpl_free(filename);
		cpl_propertylist_delete(pl);
		if(is_sdp_format) {
	        cpl_array_unwrap(waveArray);
	        cpl_array_unwrap(fluxArray);
	        cpl_array_unwrap(errArray);
            if (totalFlux != NULL) {
                cpl_array_unwrap(totalFluxArray);
            }
		} else {
			cpl_table_unwrap(pro_tab,"WAVE");
            if (totalFlux != NULL) {
                cpl_table_unwrap(pro_tab,"FLUX");
            } else {
                cpl_table_unwrap(pro_tab,"TOT_FLUX");
            }
			cpl_table_unwrap(pro_tab,"ERR");
            if (totalFlux != NULL) {
                cpl_table_unwrap(pro_tab,"TOT_FLUX");
            }
		}
		cpl_table_delete(pro_tab);
		CHECK_ERROR_STATE();

	} CATCH{
	}
	eris_check_error_code("eris_ifu_jitter_spec_save_products");
	return cpl_error_get_code();
}


/**
  @brief fetch parameters controlling the extraction

  @param  parlist  input parameterlist
  @param  params   output structure with extraction parameters
  @param  context  value indicating the parameter prefix (recipe dependent)

  @return cpl_error_code error status
  */
cpl_error_code eris_ifu_extract_spec_fetch_params(
		const cpl_parameterlist * parlist,
		struct esParamStruct *params,
		const char* context)
{
	TRY
	{
		const char* methodString;
		char* param_name = NULL;

		param_name = cpl_sprintf("%s.mask_method", context);

		methodString = cpl_parameter_get_string(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);


		if (strncasecmp(methodString, "mask", strlen("mask")) == 0) {
			params->mask_method = MASK;
		} else if (strncasecmp(methodString, "position", strlen("position")) == 0) {
			params->mask_method = POSITION;
		} else if (strncasecmp(methodString, "max", strlen("max")) == 0) {
			params->mask_method = MAX;
		} else if (strncasecmp(methodString, "fit", strlen("fit")) == 0) {
			params->mask_method = FIT;
		} else if (strncasecmp(methodString, "optimal", strlen("optimal")) == 0) {
			params->mask_method = OPTIMAL;
		} else {
			cpl_msg_error(cpl_func, "The maks_method parameter must be one "
					"of the list: mask, position, max, fit");
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Error reading recipe parameter, unknown mask method %s",
					methodString);

		}

		const char* center;
		param_name = cpl_sprintf("%s.center", context);
		center = cpl_parameter_get_string(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		int nFields;
		int end;
		nFields = sscanf(center, "%d,%d%n",
				&params->center_x,  &params->center_y, &end);
		if (nFields != 2 || end != (int)strlen(center)) {
			cpl_msg_error(cpl_func, "The center parameter must be "
					"a list of two integers separated by a comma");
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Error reading recipe parameter, cannot properly read center spec %s",
					center);
			printf("Error reading center string\n");
		}

		param_name = cpl_sprintf("%s.radius", context);
		params->radius = cpl_parameter_get_double(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.product_depth", context);
		params->productDepth = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		CHECK_ERROR_STATE();
	} CATCH
	{
	}

	return cpl_error_get_code();
}


/**
  @brief function to extract a spectrum from a 3D data cube
  @param  frameset input frameset
  @param  parlist  input parameterlist
  @param  obj_type input cube type
  @param  stdParams input structure containing std parameters
  @param  pipefile_prefix  prefix of recipe data products
  @param  context  value indicating the parameter prefix (recipe dependent)

  @return cpl_error_code error status
  */
cpl_error_code
eris_ifu_jitter_extract(cpl_frameset  * frameset,
                        const cpl_parameterlist *parlist,
                        cubeType obj_type,
                        const char* pcatg,
                        struct stdParamStruct stdParams,
                        const char* pipefile_prefix,
                        const char* context)
{
    double                  startLambda         = 0.,
                            deltaLambda         = 0.,
                            refPixel            = 0.;
    cpl_frame               *frame_cube_coadd   = NULL;
    cpl_image               *mask               = NULL;
    hdrl_image              *collapsedCube      = NULL;
    cpl_bivector            *spectrum           = NULL;
    cpl_vector              *error              = NULL;
    cpl_vector              *totalFlux          = NULL;
    cpl_propertylist        *plist              = NULL;
    char                    *fnamePrefix        = NULL,
                            *proCatg            = NULL;
    const char              *fname              = NULL;
    struct esParamStruct    esParams;
    struct esSofStruct      esSof;

    TRY {
    	cpl_msg_info(cpl_func,"Extracting data source from %s", pcatg);

        frame_cube_coadd = cpl_frameset_find(frameset, pcatg);
        fname = cpl_frame_get_filename(frame_cube_coadd);

        /* WCS are in header of data extention, first one */
        plist = cpl_propertylist_load(fname, 1);
        refPixel = cpl_propertylist_get_double(plist, "CRPIX3");
        startLambda = cpl_propertylist_get_double(plist, "CRVAL3");
        if(cpl_propertylist_has(plist, "CDELT3")) {
            deltaLambda = cpl_propertylist_get_double(plist, "CDELT3");
        } else {
            deltaLambda = cpl_propertylist_get_double(plist, "CD3_3");
        }

        startLambda = startLambda - (refPixel - 1.) * deltaLambda;

        BRK_IF_ERROR(
            eris_ifu_extract_spec_fetch_params(parlist, &esParams, context));

        //eris_ifu_jitter_get_cube_type_string(obj_type);
        cubeType coadd_obj_type = eris_ifu_jitter_get_coadd_obj_type(obj_type);
        eris_ifu_jitter_get_cube_type_string(coadd_obj_type);
        BRK_IF_ERROR(
            eris_ifu_jitter_extract_spec_processSof(obj_type, frameset,
                esParams, &esSof));

        eris_ifu_jitter_get_procatg_and_filename(obj_type, &proCatg, &fnamePrefix);

        cpl_image *contribMap = NULL;
        collapsedCube = eris_ifu_extract_spec_collapse(esSof.cube, &contribMap);
        eris_ifu_free_image(&contribMap);

        char* pro_catg = cpl_sprintf("%s_MEDIAN",proCatg);
        cpl_free(proCatg);
        cpl_free(fnamePrefix);
        cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG, pro_catg);
        char *filename = cpl_sprintf("%s_cube_median.fits", pipefile_prefix);
        /* compute QC */
        if( strstr(pro_catg, "STD") != NULL ||
        		strstr(pro_catg, "PSF") != NULL ){
        	cpl_propertylist* qcheader =
        			eris_compute_psf_qc(collapsedCube, parlist, context);

        	cpl_propertylist_append(plist, qcheader);
        	cpl_propertylist_delete(qcheader);
        }

        cpl_image* mask_ima = cpl_image_new_from_mask(hdrl_image_get_mask(collapsedCube));

        eris_ifu_save_deq_image(frameset, NULL, parlist,frameset, NULL,
                pipefile_prefix, plist, "RADECSYS", filename,
                hdrl_image_get_image(collapsedCube),
                hdrl_image_get_error(collapsedCube),
                rmse, mask_ima, flag16bit, UNIT_ADU);
        cpl_image_delete(mask_ima);

        cpl_free(filename);
        cpl_free(pro_catg);

//            const cpl_parameter *par = cpl_parameterlist_find_const(parlist, "edge-trim");
//            if (par != NULL) {
//                edge_trim = cpl_parameter_get_int(par);
//            }
// agudo: eventually take edge_trim in respect when creating mask?
        BRK_IF_NULL(
            mask = eris_ifu_extract_spec_create_mask(esParams, esSof,
                                                     collapsedCube,
                                                     esParams.productDepth));

        cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
                                       ERIS_IFU_PRO_JITTER_EXTRACTION_MASK);
        filename = cpl_sprintf("%s_extraction_mask.fits", pipefile_prefix);
        cpl_dfs_save_image(frameset, plist, parlist, frameset, NULL,
                           mask, CPL_TYPE_FLOAT, pipefile_prefix,
                           plist, "RADECSYS",
                           PACKAGE "/" PACKAGE_VERSION,
                           filename);

        cpl_free(filename);

        if (esParams.mask_method == OPTIMAL) {
            BRK_IF_NULL(
                spectrum = eris_ifu_optimal_extraction(esSof.cube,
                                                       esSof.qualImagelist,
                                                       mask,
                                                       startLambda, deltaLambda,
                                                       esParams.productDepth,
                                                       &error));
        } else {
            BRK_IF_NULL(
                spectrum = eris_ifu_extract_spectrum(esSof.cube, mask,
                                                     startLambda, deltaLambda,
                                                     &error, &totalFlux));
        }

        eris_ifu_jitter_spec_save_products(frameset, parlist, esSof, spectrum,
                error, totalFlux, stdParams.productDepth, CPL_TRUE,
                pipefile_prefix);
    } CATCH
    {
        CATCH_MSG();
    }

    //The following free make crush
    if (esSof.mask != mask) {
        eris_ifu_free_image(&mask);
    }
    eris_ifu_free_propertylist(&plist);
    eris_ifu_extract_free_esSofStruct(&esSof);
    eris_ifu_free_hdrl_image(&collapsedCube);
    eris_ifu_free_bivector(&spectrum);
    eris_ifu_free_vector(&error);
    eris_ifu_free_vector(&totalFlux);
    eris_check_error_code("eris_ifu_jitter_extract");
    return cpl_error_get_code();
}

/**
  @brief function to extract a spectrum from a 3D data cube
  @param  obj_type input cube type
  @param  frames input frameset
  @param  params  input eris structure to control extraction
  @param  sof     output eris structure to hold input data

  @return cpl_error_code error status
  */
cpl_error_code
eris_ifu_jitter_extract_spec_processSof(
		cubeType obj_type,
		cpl_frameset* frames,
		struct esParamStruct params,
		struct esSofStruct *sof)
{
	cpl_frame           *frame = NULL;

	TRY
	{

		if (frames == NULL) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"missing frameset");
		}

		if (cpl_frameset_is_empty(frames)) {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"SOF file is empty or missing");
		}
		CHECK_ERROR_STATE();

		

		// get data cube
		char *proCatg = NULL;
		char    *fnamePrefix = NULL;
		eris_ifu_jitter_get_procatg_and_filename(obj_type, &proCatg, &fnamePrefix);
		frame = cpl_frameset_find(frames, proCatg);
		cpl_free(fnamePrefix);

		CHECK_ERROR_STATE();
		if (frame != NULL) {
			BRK_IF_NULL(
					sof->cube = eris_ifu_load_deq_hdrl_imagelist(
							cpl_frame_get_filename(frame),
							&sof->header, &sof->qualImagelist, &sof->qualityType));
			sof->nx = hdrl_imagelist_get_size_x(sof->cube);
			sof->ny = hdrl_imagelist_get_size_y(sof->cube);
			sof->nz = hdrl_imagelist_get_size(sof->cube);
		} else {
			BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
					"missing \"%s\" tag in the SOF, input data cube",
					proCatg);

		}
		cpl_free(proCatg);

		sof->mask = NULL;
		if (params.mask_method == MASK) {
		    frame = cpl_frameset_find(frames, ERIS_IFU_UTIL_MASK);
		    CHECK_ERROR_STATE();
		    if (frame != NULL) {
		        BRK_IF_NULL(
		            sof->mask = cpl_image_load(
		                cpl_frame_get_filename(frame),
		                CPL_TYPE_DOUBLE, 0, 0));
		    } else {
		        BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
		            "missing \"%s\" tag in the SOF, extraction mask",
		            ERIS_IFU_UTIL_MASK);
		    }
		}
	} CATCH
	{
		//    	CATCH_MSGS();
	}
	eris_check_error_code("eris_ifu_jitter_extract_spec_processSof");
	return cpl_error_get_code();

}

/**
  @brief function to apply Differential Atmospheric Refraction correction
  @param  cube input 3D data cube
  @param  hdr  input FITS header
  @param  method  method to perform DAR correction
  @param  radius  radius controlling DAR correction CPL_KERNEL_DEF_WIDTH
  @param  length  to control DAR correction CPL_KERNEL_DEF_SAMPLES

  @return cpl_error_code error status
  */
cpl_error_code eris_ifu_dar_correction(
        hdrl_imagelist *cube,
        cpl_propertylist *hdr,
//        cpl_imagelist *cubeBpm,
        int method,
        double radius,
        int length)
{
	cpl_ensure_code(cube,CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdr,CPL_ERROR_NULL_INPUT);
	//cpl_ensure_code(radius > 0,CPL_ERROR_ILLEGAL_INPUT);
	//cpl_ensure_code(length > 0,CPL_ERROR_ILLEGAL_INPUT);

    TRY {
        double airmass = (
                cpl_propertylist_get_double(hdr, "ESO TEL AIRM START") +
                cpl_propertylist_get_double(hdr, "ESO TEL AIRM END")) / 2.;
        double parang = (
                cpl_propertylist_get_double(hdr, "ESO TEL PARANG START") +
                cpl_propertylist_get_double(hdr, "ESO TEL PARANG END")) / 2.;
        double posang =
                cpl_propertylist_get_double(hdr, "ESO ADA POSANG");
        double temp =
                cpl_propertylist_get_double(hdr, "ESO TEL AMBI TEMP");
        double rhum =
                cpl_propertylist_get_double(hdr, "ESO TEL AMBI RHUM");
        double pres = (
                cpl_propertylist_get_double(hdr, "ESO TEL AMBI PRES START") +
                cpl_propertylist_get_double(hdr, "ESO TEL AMBI PRES END")) / 2.;

        cpl_wcs *wcs = cpl_wcs_new_from_propertylist(hdr);
        CHECK_ERROR_STATE();

        hdrl_parameter *darParams = NULL;
        darParams = hdrl_dar_parameter_create(
            (hdrl_value) {airmass, 0.},
            (hdrl_value) {parang, 0.},
            (hdrl_value) {posang, 0.},
            (hdrl_value) {temp, 0.},
            (hdrl_value) {rhum, 0.},
            (hdrl_value) {pres, 0.},
            wcs);
        //BRK_IF_ERROR(hdrl_dar_parameter_verify(darParams));

        cpl_vector *lambdaIn = NULL;
        double lambdaRef;
        lambdaIn = eris_ifu_lcorr_create_lambda_vector(hdr);
        cpl_vector_multiply_scalar(lambdaIn, 10000.); // convert microns to A
        lambdaRef = cpl_vector_get(lambdaIn, cpl_vector_get_size(lambdaIn)/2);
        int naxis3 = cpl_propertylist_get_int(hdr, NAXIS3);
        cpl_vector *xShift = cpl_vector_new(naxis3);
        cpl_vector *yShift = cpl_vector_new(naxis3);
        cpl_vector *xShiftErr = cpl_vector_new(naxis3);
        cpl_vector *yShiftErr = cpl_vector_new(naxis3);
        CHECK_ERROR_STATE();

        BRK_IF_ERROR(hdrl_dar_compute( darParams,
            (hdrl_value) {lambdaRef, 0.}, lambdaIn,
            xShift, yShift, xShiftErr, yShiftErr ));
       cpl_msg_info(cpl_func, "DAR correction shift in X min/max: %.1f/%.1f, "
                "in Y min/max: %.1f/%.1f",
                cpl_vector_get_min(xShift), cpl_vector_get_max(xShift),
                cpl_vector_get_min(yShift), cpl_vector_get_max(yShift));

        if (radius <= 0) {
            radius = CPL_KERNEL_DEF_WIDTH;
        }
        if (length == 0) {
            length = CPL_KERNEL_DEF_SAMPLES;
        }

        const char *kernelName[] = {
                "TANH",
                "SINC",
                "SINC2",
                "LANCZOS",
                "HAMMING",
                "HANN",
                "NEAREST",
                "bilinear",
                "bicubic"};
        cpl_msg_info(cpl_func,"DAR correction: shift method %s, radius %.1f, length %d",
                kernelName[method], radius, length);

        for (cpl_size sx=0; sx<naxis3; sx++) {
            hdrl_image *hdrlImg = hdrl_imagelist_get(cube, sx);
            cpl_image *inImg = hdrl_image_get_image(hdrlImg);
            cpl_image *errImg = hdrl_image_get_error(hdrlImg);
            cpl_image *err2Img = cpl_image_duplicate(errImg);

            cpl_size nx = cpl_image_get_size_x(inImg);
            cpl_size ny = cpl_image_get_size_y(inImg);
            cpl_image *outImg = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
            cpl_image *out2Img = NULL;

            double shiftX = cpl_vector_get(xShift, sx);
            double shiftY = cpl_vector_get(yShift, sx);
            if (method <= 6) {
                out2Img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
                BRK_IF_ERROR(eris_ifu_dar_cpl_shift(inImg, outImg, out2Img, shiftX, shiftY,
                    method, radius, length));
            } else if (method == 7) {
                eris_ifu_dar_gsl_shift(inImg, outImg, shiftX, shiftY, gsl_interp2d_bilinear);
            } else if (method == 8) {
                eris_ifu_dar_gsl_shift(inImg, outImg, shiftX, shiftY, gsl_interp2d_bicubic);
            }
            cpl_image_reject_from_mask(err2Img, cpl_image_get_bpm(outImg));
            hdrl_image *hdrl2Img = hdrl_image_create(outImg, err2Img);

            hdrl_imagelist_set(cube, hdrl2Img, sx);
            // the previous hdrl_image is deallocated by hdrl_imagelist_set
            //hdrl_image_delete(hdrlImg);
            //eris_ifu_free_image(&inImg);
            //eris_ifu_free_image(&errImg);
            eris_ifu_free_image(&out2Img);
            cpl_image_delete(outImg);
            cpl_image_delete(err2Img);
        }

        hdrl_parameter_delete(darParams);
        eris_ifu_free_vector(&lambdaIn);
        eris_ifu_free_vector(&xShift);
        eris_ifu_free_vector(&xShiftErr);
        eris_ifu_free_vector(&yShift);
        eris_ifu_free_vector(&yShiftErr);
        cpl_wcs_delete(wcs);
    } CATCH
    {
        //      CATCH_MSGS();
    }
    return cpl_error_get_code();
}
cpl_error_code eris_ifu_dar_cpl_shift(
        cpl_image *inImg,
        cpl_image *out1Img,
        cpl_image *out2Img,
        double xShift,
        double yShift,
        cpl_kernel kernelType,
        double width,
        cpl_size kernelSize)
{

	cpl_ensure_code(inImg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(out1Img, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(out2Img, CPL_ERROR_NULL_INPUT);


    TRY {
        cpl_size nx, ny;
        cpl_vector  *xprofile = NULL;
        double      xradius;
        cpl_vector  *yprofile = NULL;
        double      yradius;
        cpl_image *correction_map = NULL;

        nx = cpl_image_get_size_x(inImg);
        ny = cpl_image_get_size_x(inImg);
        cpl_image   *deltax = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image   *deltay = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(deltax, xShift);
        cpl_image_add_scalar(deltay, yShift);

        xprofile = cpl_vector_new(kernelSize);
        cpl_vector_fill_kernel_profile(xprofile, kernelType, width);
        xradius = width;
        yprofile = cpl_vector_new(kernelSize);
        cpl_vector_fill_kernel_profile(yprofile, kernelType, width);
        yradius = width;
        CHECK_ERROR_STATE();

        cpl_image_warp(out1Img, inImg, deltax, deltay, xprofile, xradius, yprofile, yradius);
        CHECK_ERROR_STATE();

        correction_map = cpl_image_new(cpl_image_get_size_x(out1Img),
            cpl_image_get_size_y(out1Img),
            cpl_image_get_type(out1Img));
        cpl_image_fill_jacobian(correction_map, deltax, deltay);
        cpl_image_copy(out2Img, out1Img, 1, 1);
        cpl_image_multiply(out2Img, correction_map);
        CHECK_ERROR_STATE();

        eris_ifu_free_image(&deltax);
        eris_ifu_free_image(&deltay);
        eris_ifu_free_image(&correction_map);
        eris_ifu_free_vector(&xprofile);
        eris_ifu_free_vector(&yprofile);

    } CATCH
    {
        //      CATCH_MSGS();
    }

    return cpl_error_get_code();
}

cpl_error_code eris_ifu_dar_gsl_shift(
        cpl_image *inImg,
        cpl_image *outImg,
        double xShift,
        double yShift,
        const gsl_interp2d_type *T)
{
    cpl_size nx, ny;

    //cpl_ensure_code(inImg, CPL_ERROR_NULL_INPUT);
    //cpl_ensure_code(outImg, CPL_ERROR_NULL_INPUT);
    //cpl_ensure_code(T, CPL_ERROR_NULL_INPUT);

    gsl_set_error_handler_off();
    const double xa[] = {
            0.,1.,2.,3.,4.,5.,6.,7.,8.,9.,10.,11.,12.,13.,14.,15.,
            16.,17.,18.,19.,20.,21.,22.,23.,24.,25.,26.,27.,28.,29.,30.,
            31.,32.,33.,34.,35.,36.,37.,38.,39.,40.,41.,42.,43.,44.,45.,
            46.,47.,48.,49.,50.,51.,52.,53.,54.,55.,56.,57.,58.,59.,60.,
            61.,62.,63.
    };
    const double ya[]= {
            0.,1.,2.,3.,4.,5.,6.,7.,8.,9.,10.,11.,12.,13.,14.,15.,
            16.,17.,18.,19.,20.,21.,22.,23.,24.,25.,26.,27.,28.,29.,30.,
            31.,32.,33.,34.,35.,36.,37.,38.,39.,40.,41.,42.,43.,44.,45.,
            46.,47.,48.,49.,50.,51.,52.,53.,54.,55.,56.,57.,58.,59.,60.,
            61.,62.,63.
    };
    double *za = cpl_image_get_data_double(inImg);

    /* initialize interpolation */
    nx = cpl_image_get_size_x(outImg);
    ny = cpl_image_get_size_y(outImg);
    gsl_spline2d *spline = gsl_spline2d_alloc(T, nx, ny);
    gsl_interp_accel *xacc = gsl_interp_accel_alloc();
    gsl_interp_accel *yacc = gsl_interp_accel_alloc();
    gsl_spline2d_init(spline, xa, ya, za, nx, ny);
    double z;
    //TODO: why error is computed but not used?
//    int err;
    for (int i = 0; i < nx; ++i)
      {
        double xi = i - xShift;

        for (int j = 0; j < nx; ++j)
          {
            double yj = j - yShift;
            if (xi > nx-1 || xi < 0 || yj > ny-1 || yj < 0) {
                z = 0.001;
            }/* else {
                err = gsl_spline2d_eval_e(spline, xi, yj, xacc, yacc, &z);
            }*/
            cpl_image_set(outImg, i+1, j+1, z);
//            printf("%d: %d %d %f --> %f %f %f\n", err, i, j, za[j+nx*j], xi, yj, z);
          }
//        printf("\n");
      }

    gsl_spline2d_free(spline);
    gsl_interp_accel_free(xacc);
    gsl_interp_accel_free(yacc);
    return cpl_error_get_code();
}
/**
 * @brief count frames with a given PRO.CATG
 * @param sof input frameset
 * @param tag  frame tag
 * @return number of frames with a given tag
 */
int
eris_frameset_count_tag(cpl_frameset * sof, const char*  tag)
{

	cpl_ensure(sof, CPL_ERROR_NULL_INPUT, -1);
	cpl_ensure(tag, CPL_ERROR_NULL_INPUT, -1);

    int count = 0;
    cpl_frameset* loc_sof = cpl_frameset_duplicate(sof);
    cpl_frame* frame = cpl_frameset_find(loc_sof, tag);
    if(frame != NULL) {
        while(frame) {
        	cpl_frameset_erase_frame(loc_sof, frame);
            frame = cpl_frameset_find(loc_sof, tag);
            count++;
        }
    }
    cpl_frameset_delete(loc_sof);
    return count;

}
/**
 * @brief duplicate a cube of a given TAG to a COADDED corresponding cube
 * @param frameset input frameset containing data cube
 * @param pcatg  cube PRO.CATG
 * @param apply_flat was the cube corrected by flat?
 */
cpl_error_code
eris_frameset_duplicate_cube_tag(cpl_frameset* frameset, const char* pcatg,
		cpl_boolean apply_flat)
{
	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(pcatg, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	//cpl_msg_warning(cpl_func,"pcatg: %s", pcatg);
	//cpl_frameset_dump(frameset,stdout);
	cpl_frame* frm_tmp = cpl_frameset_find(frameset, pcatg);
	cpl_frame* frm_dup = cpl_frame_duplicate(frm_tmp);
	const char* fname = cpl_frame_get_filename(frm_tmp);
	const char* fname_new = NULL;
	cpl_propertylist* plist = cpl_propertylist_load(fname, 0);
	//cpl_msg_info(cpl_func,"pcatg: %s",pcatg);
	if(strcmp(pcatg,ERIS_IFU_PRO_JITTER_OBJ_CUBE) == 0) {
		//cpl_msg_info(cpl_func,"case OBJ");
		fname_new = "eris_ifu_jitter_obj_cube_coadd.fits";
		cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD);
		cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD);
	} else if(strcmp(pcatg,ERIS_IFU_PRO_JITTER_DAR_CUBE) == 0) {
		fname_new = "eris_ifu_jitter_dar_cube_coadd.fits";
		cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_COADD);
		cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_COADD);
	} else if(strcmp(pcatg,ERIS_IFU_PRO_JITTER_TWK_CUBE) == 0) {
		fname_new = "eris_ifu_jitter_twk_cube_coadd.fits";
		cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD);
		cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD);        
	} else if(strcmp(pcatg,ERIS_IFU_PRO_JITTER_PSF_CUBE) == 0) {
		//cpl_msg_info(cpl_func,"case PSF");
		fname_new = "eris_ifu_stdstar_psf_cube_coadd.fits";
		cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD);
		cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD);
	} else if(strcmp(pcatg,ERIS_IFU_PRO_JITTER_STD_CUBE) == 0){
		//cpl_msg_info(cpl_func,"case STD");
		if(apply_flat) {
			fname_new = "eris_ifu_stdstar_std_cube_coadd.fits";
			cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD);
			cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD);
		} else {
			fname_new = "eris_ifu_stdstar_no_flat_std_cube_coadd.fits";
			cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_NOFLAT);
			cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_NOFLAT);
		}
	} else if(strcmp(pcatg,ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE) == 0){
		//cpl_msg_info(cpl_func,"case STD_FLUX");
		if(apply_flat) {
			fname_new = "eris_ifu_stdstar_std_flux_cube_coadd.fits";
			cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD);
			cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD);
		} else {
			fname_new = "eris_ifu_stdstar_no_flat_std_flux_cube_coadd.fits";
			cpl_propertylist_update_string(plist, FHDR_PRO_CATG, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT);
			cpl_frame_set_tag(frm_dup, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT);
		}
	}

	char* cmd = cpl_sprintf("cp %s %s",fname, fname_new);
    cpl_msg_debug(cpl_func, "Copying %s", cmd);
	int status = system(cmd);
	if(status == -1) {
		cpl_msg_warning(cpl_func,"call to system() failed");
	}
	cpl_frame_set_filename(frm_dup, fname_new);
    cpl_frame_set_group(frm_dup, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(frm_dup, CPL_FRAME_LEVEL_FINAL);

	cpl_propertylist_save(plist, fname_new, CPL_IO_CREATE);
	cpl_propertylist_delete(plist);

	plist = cpl_propertylist_load(fname, 1);
	cpl_imagelist* data = cpl_imagelist_load(fname,CPL_TYPE_FLOAT, 1);
	cpl_imagelist_save(data, fname_new, CPL_TYPE_FLOAT, plist,  CPL_IO_EXTEND);
	cpl_imagelist_delete(data);
	cpl_propertylist_delete(plist);

	plist = cpl_propertylist_load(fname, 2);
	data = cpl_imagelist_load(fname, CPL_TYPE_FLOAT, 2);
	cpl_imagelist_save(data, fname_new, CPL_TYPE_FLOAT, plist,  CPL_IO_EXTEND);
	cpl_imagelist_delete(data);
	cpl_propertylist_delete(plist);

	plist = cpl_propertylist_load(fname, 3);
	data = cpl_imagelist_load(fname, CPL_TYPE_INT, 3);
	cpl_imagelist_save(data, fname_new, CPL_TYPE_INT, plist,  CPL_IO_EXTEND);
	cpl_imagelist_delete(data);
	cpl_propertylist_delete(plist);


    //cpl_msg_warning(cpl_func, "insert frame %s with tag %s",
    //		cpl_frame_get_filename(frm_tmp), cpl_frame_get_tag(frm_tmp));
	cpl_frameset_insert(frameset, frm_dup);
	cpl_free(cmd);


	return cpl_error_get_code();
}

cpl_error_code eris_ifu_update_wcs_with_OCS_keywords(
        struct sofStruct *sof)
{
    cpl_error_code retval = CPL_ERROR_NONE;
    cpl_propertylist *header = NULL;
    //const char *filename;
    double  crval1,
            crval2,
            crval1New,
            crval2New,
            crval1Ref,
            crval2Ref,
            cumOffset1,
            cumOffset2;
    int     sx,
            rx;


    if (sof->exposureTableCnt < 2) { // nothing to do!
        return CPL_ERROR_NONE;
    }

    TRY{
        /*
         * find reference exposure
         * either the first one with CUMOFFS1 and CUMOFFS1 set to 0.0
         * or the the first exposure
         */
        rx = -1;
        for (sx = 0; sx < sof->exposureTableCnt; sx++) {
            header = sof->exposureTable[sx].hdr;
            cumOffset1 = cpl_propertylist_get_double(header, FHDR_E_CUMOFFS1);
            cumOffset2 = cpl_propertylist_get_double(header, FHDR_E_CUMOFFS2);
            CHECK_ERROR_STATE();
            if (cumOffset1 == 0.0 && cumOffset2 == 0.0) {
                rx = sx;
            }
            cpl_msg_info("update_WCS_with_OCS_keywords",
                    "%d %s : %f/%f  %.1f/%.1f %g %g %g %g",
                    sx, cpl_frame_get_filename(sof->exposureTable[sx].frame),
                    cpl_propertylist_get_double(header, FHDR_E_CRVAL1),
                    cpl_propertylist_get_double(header, FHDR_E_CRVAL2),
                    cpl_propertylist_get_double(header, "CRPIX1"),
                    cpl_propertylist_get_double(header, "CRPIX2"),
                    cpl_propertylist_get_double(header, "CD1_1"),
                    cpl_propertylist_get_double(header, "CD1_2"),
                    cpl_propertylist_get_double(header, "CD2_1"),
                    cpl_propertylist_get_double(header, "CD2_2")
                    );
            CHECK_ERROR_STATE();
        }
        if (rx == -1) {
            rx = 0;
        }

        header = sof->exposureTable[rx].hdr;
        crval1 = cpl_propertylist_get_double(header, FHDR_E_CRVAL1);
        crval2 = cpl_propertylist_get_double(header, FHDR_E_CRVAL2);
        cumOffset1 = cpl_propertylist_get_double(header, FHDR_E_CUMOFFS1);
        cumOffset2 = cpl_propertylist_get_double(header, FHDR_E_CUMOFFS2);
        CHECK_ERROR_STATE();

        crval1Ref = crval1 - cumOffset1 / 3600.;
        crval2Ref = crval2 - cumOffset2 / 3600.;

        cpl_msg_info("update_WCS_with_OCS_keywords","ref: %d %f %f",
                rx, crval1Ref, crval2Ref);

        for (sx = 0; sx < sof->exposureTableCnt; sx++) {
            header = sof->exposureTable[sx].hdr;
            crval1 = cpl_propertylist_get_double(header, FHDR_E_CRVAL1);
            crval2 = cpl_propertylist_get_double(header, FHDR_E_CRVAL2);
            cumOffset1 = cpl_propertylist_get_double(header, FHDR_E_CUMOFFS1);
            cumOffset2 = cpl_propertylist_get_double(header, FHDR_E_CUMOFFS2);

            crval1New = crval1Ref + cumOffset1 / 3600.;
            crval2New = crval2Ref + cumOffset2 / 3600.;
            cpl_propertylist_set_double(header, FHDR_E_CRVAL1, crval1New);
            cpl_propertylist_set_double(header, FHDR_E_CRVAL2, crval2New);
            CHECK_ERROR_STATE();

            cpl_msg_info("update_WCS_with_OCS_keywords",
                    "%d  %f -> %f (%f)   %f -> %f (%f)   %f / %f   %f / %f  %f // %f",
                    sx,
                    crval1, crval1New, (crval1-crval1New)*3600.,
                    crval2, crval2New, (crval2-crval2New)*3600.,
                    cumOffset1, cumOffset2,
                    (crval1New-crval1Ref)*3600., (crval2New-crval2Ref)*3600.,
                    (crval1-crval1Ref)*3600., (crval2-crval2Ref)*3600. );
        }
    } CATCH {
    }
    return retval;
}
