/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <hdrl.h>
#include "eris_utils.h"
#include "eris_ifu_jitter_static.h"
#include "eris_ifu_efficiency_response.h"
#include "eris_ifu_resample.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_strehl.h"
#include "eris_ifu_error.h"
#include "eris_ifu_stdstar_static.h"

/**
  @brief    function to do common steps of data reduction
  @param    frameset input frames
  @param    parlist input parameters
  @param    apply_flat_field parameter to apply master flat (or not)
  @param    context      recipe context (eris)
  @param    recipe_name  recipe name
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */

cpl_error_code
eris_stdstar_reduction_common(cpl_frameset *frameset,
                              const cpl_parameterlist *parlist,
                              cpl_boolean apply_flat_field,
                              const char *context,
                              const char *recipe_name)
{
    struct stdParamStruct   stdParams   = stdParamStructInit;
    struct sofStruct        sof;
    struct paramStruct      params;
    int                     obj_number  = 0;
    char                    *pipefile_prefix = NULL;

	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(parlist,  CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(context,  CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(recipe_name, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    TRY {
        if (apply_flat_field) {
            cpl_msg_info(cpl_func, "=============================================");
            cpl_msg_info(cpl_func, "Data reduction with flat field");
            cpl_msg_info(cpl_func, "---------------------------------------------");
            pipefile_prefix = cpl_sprintf("%s", recipe_name);
        } else {
            cpl_msg_info(cpl_func, "=============================================");
            cpl_msg_info(cpl_func, "Data reduction without flat field");
            cpl_msg_info(cpl_func, "---------------------------------------------");
            pipefile_prefix = cpl_sprintf("%s_%s", recipe_name, "no_flat");
        }

        cpl_boolean is_std_flux = CPL_FALSE;
        cpl_boolean is_std_norm = CPL_FALSE;
        cpl_boolean is_pupil_lamp = CPL_FALSE;
        cpl_boolean is_pupil_sky = CPL_FALSE;
        cpl_boolean is_psf = CPL_FALSE;
        if (cpl_frameset_find(frameset,"STD") != NULL){
            is_std_norm = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"PUPIL_SKY") != NULL){
            is_pupil_sky = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"PUPIL_LAMP") != NULL) {
            is_pupil_lamp = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"STD_FLUX") != NULL) {
            is_std_flux = CPL_TRUE;
        }
        else if (cpl_frameset_find(frameset,"PSF_CALIBRATOR") != NULL) {
            is_psf = CPL_TRUE;
        }


        if (is_std_norm) {
            obj_number = eris_frameset_count_tag(frameset, "STD");
        } else if (is_std_flux) {
            obj_number = eris_frameset_count_tag(frameset, "STD_FLUX");
        } else if (is_pupil_lamp) {
                obj_number = eris_frameset_count_tag(frameset, "PUPIL_LAMP");
        } else if (is_pupil_sky) {
                obj_number = eris_frameset_count_tag(frameset, "PUPIL_SKY");
        } else if (is_psf) {
            obj_number = eris_frameset_count_tag(frameset, "PSF_CALIBRATOR");
        }

        (void)memset(&sof, 0, sizeof(sof)); /* Rough init so free works on error */

		cpl_msg_info(cpl_func, "Reading recipe parameters");
		BRK_IF_ERROR(
            eris_ifu_jitter_fetch_params(context, recipe_name, parlist,
                                         &stdParams, &params));
		cpl_msg_info(cpl_func,
                    "Instrument is %d, requested product level is %d",
                    stdParams.instrument, stdParams.productDepth);

		BRK_IF_ERROR(
            eris_ifu_jitter_processSof(frameset,
                                       stdParams,
                                       params,
                                       &sof));

		cpl_msg_info(cpl_func,"Instrument %s, band %s, scale %s",
                    eris_ifu_get_instrumentString(sof.instrument),
                    eris_ifu_get_bandString(sof.band),
                    eris_ifu_get_preopticsScaleString(sof.scale));

		BRK_IF_ERROR(
            eris_ifu_jitter_process_exposures(&sof, stdParams, params,
                                              frameset, parlist, pipefile_prefix,
                                              context));

		cubeType obj_type = OBJECT_CUBE;

		BRK_IF_ERROR(
            eris_ifu_jitter_process_cubes(&sof, stdParams, params, frameset,
                                          parlist, pipefile_prefix, &obj_type));

		cubeType resampled_obj_type = STD_CUBE_COADD;
		const char *input_cube_pro_catg = NULL;
		if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_JITTER_DAR_CUBE)
				> 0) {
			cpl_msg_info(cpl_func,"case1");
			if(is_std_norm) {
				obj_type = DAR_STD_CUBE;
				resampled_obj_type = DAR_STD_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
			} else if(is_std_flux) {
				obj_type = DAR_STD_FLUX_CUBE;
				if (apply_flat_field) {
					resampled_obj_type = STD_FLUX_CUBE_COADD;
					input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD;
				} else {
					resampled_obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
					input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT;
				}
			} else if(is_psf) {
				obj_type = DAR_PSF_CUBE;
				resampled_obj_type = DAR_PSF_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
			} else if(is_pupil_lamp || is_pupil_sky) {
				obj_type = DAR_STD_CUBE;
				resampled_obj_type = DAR_PUPIL_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
			}
		} else if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_JITTER_TWK_CUBE)
				> 0 ) {
			cpl_msg_info(cpl_func,"case2");
			if(is_std_norm) {
				obj_type = TWEAKED_STD_CUBE;
				resampled_obj_type = STD_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_CUBE_COADD;
			} else if(is_std_flux) {
				obj_type = STD_FLUX_CUBE;
				if (apply_flat_field) {
				    resampled_obj_type = STD_FLUX_CUBE_COADD;
				    input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD;
				} else {
					resampled_obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
				    input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT;
				}
			} else if(is_psf) {
				obj_type = PSF_CUBE;
				resampled_obj_type = PSF_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_TWK_CUBE;
			} else if(is_pupil_lamp || is_pupil_sky) {
				obj_type = PUPIL_CUBE;
				resampled_obj_type = PUPIL_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_TWK_CUBE;
			}
		} else {
			cpl_msg_info(cpl_func,"case3");
			if(is_std_norm) {
				obj_type = STD_CUBE;
				resampled_obj_type = STD_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_CUBE_COADD;
			} else if(is_std_flux) {
				obj_type = STD_FLUX_CUBE;
				if (apply_flat_field) {
				    resampled_obj_type = STD_FLUX_CUBE_COADD;
				    input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD;
				} else {
					resampled_obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
					input_cube_pro_catg = ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_NOFLAT;
				}

			} else if(is_psf) {
				obj_type = PSF_CUBE;
				resampled_obj_type = PSF_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD;
			} else if(is_pupil_lamp ||is_pupil_sky) {
				obj_type = PUPIL_CUBE;
				resampled_obj_type = PUPIL_CUBE_COADD;
				input_cube_pro_catg = ERIS_IFU_PRO_JITTER_PUPIL_CUBE_COADD;
			}
		}

		cpl_msg_info(cpl_func,"input cube pro catg: %s",input_cube_pro_catg);

		/* combine cubes with HDRL functionality */
		if (params.combine && obj_number > 1) {
			// eris_ifu_combine(obj_type, stdParams, params, frameset, parlist,
			//         recipe_name, pipefile_prefix);
			char *combDoCatg =  NULL;
			char   *filenameSpec = NULL;
			eris_ifu_jitter_get_procatg_and_filename(obj_type, &combDoCatg,
					&filenameSpec);

            cpl_msg_debug(cpl_func,"GENERATE COMBINED DATA CUBE %s",combDoCatg);
			eris_ifu_combine_pbp(frameset, parlist, combDoCatg, filenameSpec,
					NULL, NULL, NULL, recipe_name, pipefile_prefix);

			eris_ifu_free_string(&combDoCatg);
			eris_ifu_free_string(&filenameSpec);
		} else if (obj_number == 1) {
			char *combDoCatg =  NULL;
			char   *filenameSpec = NULL;
		    eris_ifu_jitter_get_procatg_and_filename(obj_type, &combDoCatg,
								&filenameSpec);
			if(is_std_norm) {
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_STD_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_STD_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec, */apply_flat_field, CPL_FALSE);
				//cpl_frameset_dump(frameset, stdout);
			}
			else if(is_std_flux) {
				cpl_msg_debug(cpl_func,"duplicate FLUX cube tag");
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec,*/ apply_flat_field, CPL_FALSE);

			} else if(is_pupil_lamp || is_pupil_sky) {
				cpl_msg_debug(cpl_func,"duplicate PUPIL cube tag");
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_OBJ_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_OBJ_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec, */apply_flat_field, CPL_TRUE);
			} else if(is_psf) {
				cpl_msg_debug(cpl_func,"duplicate PSF cube tag");
				eris_frameset_duplicate_cube_tag(frameset,
						ERIS_IFU_PRO_JITTER_PSF_CUBE, apply_flat_field);
				eris_ifu_cube_collapse_mean_and_save(ERIS_IFU_PRO_JITTER_PSF_CUBE,
                        frameset, parlist, recipe_name, /*filenameSpec, */apply_flat_field, CPL_FALSE);
			}
			cpl_free(combDoCatg);
		    cpl_free(filenameSpec);
		}

		if( is_pupil_lamp || is_pupil_sky) {
//			hdrl_imagelist* himl = NULL;
			cpl_table* qclog_tbl = eris_qclog_init();
			eris_get_pupil_shift(sof.exposureTable[0].cube, 0, &qclog_tbl);
			cpl_free(qclog_tbl);
		}

		/* extract spectra and for STD star computes response and efficiency */
        if (apply_flat_field  == CPL_FALSE) {
            if (is_std_norm) {
				obj_type = STD_CUBE_COADD_NOFLAT;
			} else if (is_std_flux) {
				obj_type = STD_FLUX_CUBE_COADD_NOFLAT;
			}
		}
		if (params.extractSource && eris_can_extract(frameset)) {
			eris_print_rec_status(102);
			cpl_msg_info(cpl_func,"input pro catg: %s obj_type: %d",
					input_cube_pro_catg, resampled_obj_type);

			eris_ifu_jitter_extract(frameset, parlist, resampled_obj_type,
					input_cube_pro_catg, stdParams, pipefile_prefix, context);
			eris_print_rec_status(103);
		}
		//cpl_frameset_dump(frameset, stdout);
		char* param_name = NULL;

		param_name = cpl_sprintf("%s.compute-strehl", context);
		cpl_boolean compute_strehl = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		param_name = cpl_sprintf("%s.flux-calibrate", context);
		cpl_boolean flux_calibrate = cpl_parameter_get_bool(
				cpl_parameterlist_find_const(parlist, param_name));
		cpl_free(param_name);

		eris_ifu_jitter_get_cube_type_string(obj_type);
		if(obj_type == STD_CUBE || obj_type == STD_CUBE_COADD_NOFLAT ||
		   obj_type == STD_FLUX_CUBE || obj_type == STD_FLUX_CUBE_COADD_NOFLAT||
		   obj_type == DAR_STD_FLUX_CUBE || obj_type == DAR_STD_CUBE_COADD) {

			param_name = cpl_sprintf("%s.compute-efficiency", context);
			cpl_boolean compute_efficiency = cpl_parameter_get_bool(
					cpl_parameterlist_find_const(parlist, param_name));
			cpl_free(param_name);

			param_name = cpl_sprintf("%s.compute-response", context);
			cpl_boolean compute_response = cpl_parameter_get_bool(
					cpl_parameterlist_find_const(parlist, param_name));
			cpl_free(param_name);

			cpl_msg_debug(cpl_func,"compute_response: %d eris_can_compute_response: %d apply_flat_field: %d",
					compute_response,eris_can_compute_response(frameset),
					apply_flat_field);
			if (compute_response && eris_can_compute_response(frameset) &&
					apply_flat_field == CPL_TRUE) {
				cpl_msg_info(cpl_func,"Compute response");
				eris_response_compute(recipe_name, parlist, frameset, frameset);
			}

			cpl_msg_debug(cpl_func, "compute_efficiency: %d eris_can_compute_efficiency: %d",
					compute_efficiency,eris_can_compute_efficiency(frameset));

			if (compute_efficiency && apply_flat_field == CPL_FALSE) {
				if (eris_can_compute_efficiency(frameset)) {
					cpl_msg_info(cpl_func,"Compute efficiency");
					cpl_frame * spectrum_frame = NULL;
					cpl_frame * extcoeff_frame = NULL;
					cpl_frame * fluxstdcat_frame = NULL;

					spectrum_frame = cpl_frameset_find(frameset,
							ERIS_IFU_PRO_JITTER_SPECTRUM_NOFLAT);
					extcoeff_frame = cpl_frameset_find(frameset,
							ERIS_IFU_CALIB_EXTCOEFF_TABLE);
					fluxstdcat_frame = cpl_frameset_find(frameset,
							ERIS_IFU_CALIB_FLUX_STD_CATALOG);

					cpl_table* tab_eff = eris_efficiency_compute(spectrum_frame,
							fluxstdcat_frame, extcoeff_frame, frameset, parlist,
							pipefile_prefix);

					cpl_table_delete(tab_eff);
				}
			}
		}

		//cpl_frameset_dump(frameset, stdout);
		cpl_msg_debug(cpl_func,"flux_calibrate: %d eris_can_flux_calibrate: %d, apply_flat_field:%d",
				flux_calibrate,eris_can_flux_calibrate(frameset),apply_flat_field);
        if (flux_calibrate && apply_flat_field == CPL_TRUE) {
            if (eris_can_flux_calibrate(frameset)) {
				cpl_msg_info(cpl_func,"Flux calibrate extracted spectrum and combined data cube");

				eris_flux_calibrate_spectra(pipefile_prefix, recipe_name,
						parlist, frameset, frameset);
				char* cube_pro_catg = NULL;
                if (is_std_norm) {
					cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_STD_CUBE_COADD);
                } else if (is_std_flux) {
					cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD);
				} else {
					cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD);
				}

				eris_flux_calibrate_cube2(cube_pro_catg, pipefile_prefix,
                        recipe_name, parlist, /*frameset, */frameset);
				cpl_free(cube_pro_catg);
			}
		}

		//if(compute_strehl && !is_std_flux) {
        if (compute_strehl && apply_flat_field) {
			/* for PSF standards we can compute the Strehl */
			cpl_msg_info(cpl_func,"===============================================");
			cpl_msg_info(cpl_func,"===========STREHL COMPUTATION==================");
			cpl_msg_info(cpl_func,"===============================================");
            eris_ifu_stdstar_strehl_compute(frameset, parlist, context/*, recipe_name*/);
		}
		//cpl_frameset_dump(frameset, stdout);
    } CATCH {

	}

	cpl_free(pipefile_prefix);
	eris_ifu_jitter_free_sofStruct(&sof);
	eris_ifu_free_std_param(&stdParams);
	eris_check_error_code("eris_stdstar_reduction_common");

	return cpl_error_get_code();
}
