/* $Id: eris_ifu_strehl.c,v 1.9 2013-03-26 17:00:44 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <eris_ifu_strehl.h>
#include <eris_ifu_jitter_interface.h>
#include <eris_ifu_wavecal_static.h>
#include <eris_utils.h>
#include <eris_pfits.h>
#include <eris_ifu_utils.h>
#include <string.h>
#include <cpl.h>
#include <hdrl.h>

#define M1_RADIUS 8.15
#define M2_RADIUS 0.9
#define ERIS_IFU_RAW_BPM                    "RAW_BPM"
#define ERIS_IFU_RAW_ERROR                  "RAW_ERROR"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_strehl Strehl computation related functions
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define HDRL_PARAMETER_HEAD void * base
/** @cond PRIVATE */
typedef struct {
    HDRL_PARAMETER_HEAD;
    cpl_size    llx ;
    cpl_size    lly ;
    cpl_size    urx ;
    cpl_size    ury ;
} hdrl_rect_region_parameter;

/*----------------------------------------------------------------------------*/
/**
  @brief    save propertylist
  @param    procatg  product category
  @param    filename   filename
  @param    header   FITS header
  @param    parlist  parameter list
  @param    frameset frameset
  @return  cpl_error_code

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_propertylist_save(
        const char              *   procatg,
        const char              *   filename,
        const cpl_propertylist  *   header,
        const cpl_parameterlist *   parlist,
        cpl_frameset            *   frameset)
{
    /* Add a QC parameter  */
    cpl_propertylist * applist = cpl_propertylist_new();

    /* Add the product category and save image */
    cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procatg);

    cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset, NULL,
            "ERIS_IFU_STREHL", header, NULL, PACKAGE "/" PACKAGE_VERSION,
            filename);
    cpl_propertylist_delete(applist);
    eris_check_error_code("eris_ifu_propertylist_save");
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
  @brief    wrap negative or zero coordinates around full image size
  @param    rect_region  rect region to wrap
  @param    nx   image size in x, nx is added to entries < 1
  @param    ny   image size in y, ny is added to entries < 1
  @return  cpl_error_code

  allows reverse indexing: 0 would be nx, -2 would be nx - 2 etc
  Wrapping is based in FITS convention: 1 first pixel, nx last pixel inclusive
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code hdrl_rect_region_fix_negatives(
        hdrl_parameter    *     rect_region,
        const cpl_size          nx,
        const cpl_size          ny)
{
    hdrl_rect_region_parameter * rr_loc =
        (hdrl_rect_region_parameter *)rect_region ;

    cpl_error_ensure(rect_region != 0, CPL_ERROR_NULL_INPUT,
            return CPL_ERROR_NULL_INPUT, "region input must not be NULL");
    cpl_error_ensure(hdrl_rect_region_parameter_check(rect_region),
            CPL_ERROR_ILLEGAL_INPUT, return CPL_ERROR_ILLEGAL_INPUT,
            "Expected Rect Region parameter") ;

    if (nx > 0 && rr_loc->llx < 1) rr_loc->llx += nx;
    if (ny > 0 && rr_loc->lly < 1) rr_loc->lly += ny;
    if (nx > 0 && rr_loc->urx < 1) rr_loc->urx += nx;
    if (ny > 0 && rr_loc->ury < 1) rr_loc->ury += ny;

    return hdrl_rect_region_parameter_verify(rect_region, nx, ny);
}

/*----------------------------------------------------------------------------*/
/**
  @brief   compute photon count error in [ADU]
  @param   ima_data in [ADU]
  @param   gain detector's gain in [e- / ADU]
  @param   ron  detector's read out noise in [ADU]
  @param   ima_errs output error image in [ADU]
  @return  cpl_error_code
  @note ima_errs need to be deallocated
        ima_data must contain the photon counts with no offsets
        this usually means the image must be overscan and bias corrected
        Then the shot noise can be calculated from the poissonian distribution
        as sqrt(electron-counts). To this (transformed back into ADUs) the
        readout noise is added in quadrature.
  @doc
  error is computed with standard formula

  \f$ err_{ADU} = \sqrt{ \frac{ counts }{ gain } + ron^{ 2 } } \f$

  If an image value is negative the associated error is set to RON
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
eris_ifu_detector_shotnoise_model(const cpl_image* ima_data, const double gain,
                              const double ron, cpl_image ** ima_errs)
{
    cpl_ensure_code(ima_data, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ima_errs, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(gain > 0., CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(ron > 0., CPL_ERROR_ILLEGAL_INPUT);

    *ima_errs = cpl_image_duplicate(ima_data);
    /* set negative values (= zero measurable electrons) to read out noise */
    cpl_image_threshold(*ima_errs, 0., INFINITY, ron, ron);

    /* err_ADU = sqrt(counts/gain + ron * ron)*/

    cpl_image_divide_scalar(*ima_errs, gain);
    cpl_image_add_scalar(*ima_errs, ron * ron);
    cpl_image_power(*ima_errs, 0.5);
    eris_check_error_code("eris_ifu_detector_shotnoise_model");
    return cpl_error_get_code();
}



/* ---------------------------------------------------------------------------*/
/**
 * @brief load header of frame and wrap negative region parameters
 *
 * @param frm    frame to load from
 * @param extnum extension number to load from
 * @param par    region parameter whose ranges should be wrapped by the
 *               image size
 */
/* ---------------------------------------------------------------------------*/
static cpl_error_code
eris_ifu_fix_neg_region(const cpl_frame * frm, cpl_size extnum,
                        hdrl_parameter * par)
{
    cpl_propertylist * plist =
        cpl_propertylist_load(cpl_frame_get_filename(frm), extnum);
    if (!plist) {
        return cpl_error_get_code();
    }
    cpl_size nx = eris_pfits_get_naxis1(plist);
    cpl_size ny = eris_pfits_get_naxis2(plist);
    hdrl_rect_region_fix_negatives(par, nx, ny);
    cpl_propertylist_delete(plist);
    eris_check_error_code("eris_ifu_fix_neg_region");
    return cpl_error_get_code();
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief load image corresponding to a given extension and region
 *
 * @param frm    frame to load from
 * @param eidx    extension number to load from
 * @param region_params parameter specifying region to load
 */
/* ---------------------------------------------------------------------------*/
static cpl_image * eris_ifu_image_mef_load_region(
        cpl_frame       *   frm,
        cpl_size            eidx,
        hdrl_parameter  *   region_params)
{
    cpl_image   *   out = NULL ;

    if (region_params == NULL) {
        out = cpl_image_load(cpl_frame_get_filename(frm),
                CPL_TYPE_DOUBLE, 0, eidx);
    } else {
        eris_ifu_fix_neg_region(frm, eidx, region_params);
        out = cpl_image_load_window(cpl_frame_get_filename(frm),
                CPL_TYPE_DOUBLE, 0, eidx,
                hdrl_rect_region_get_llx(region_params),
                hdrl_rect_region_get_lly(region_params),
                hdrl_rect_region_get_urx(region_params),
                hdrl_rect_region_get_ury(region_params)) ;
    }
    eris_check_error_code("eris_ifu_image_mef_load_region");
    return out ;
}


/*----------------------------------------------------------------------------*/
/**
  @brief Load an hdrl_image from some frames.
         If missing, an error is determined using a simple detector model.
  @param in             Input frame
  @param ext_num        Extension to load the image from
  @param in_err         Input frame containing errors (optional)
  @param ext_num_err    Extension to load the error from
  @param in_bpm         Input frame containing BPMs (optional)
  @param ext_num_bpm    Extension to load the BPM from
  @param region_params  Region to Load (if NULL, load all)
  @param ron            Read out noise
  @param gain           Detector gain
  @param out_hdrl_ima   Output hdrl_image object
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_hdrl_image_load(
        cpl_frame       *   in,
        cpl_size            ext_num,
        cpl_frame       *   in_err,
        cpl_size            ext_num_err,
        cpl_frame       *   in_bpm,
        cpl_size            ext_num_bpm,
        hdrl_parameter  *   region_params,
        double              ron,
        double              gain,
        hdrl_image      **  out_hdrl_ima)
{
    cpl_image       *   img ;

    /* Check Entries */
    cpl_ensure_code(in, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_hdrl_ima, CPL_ERROR_NULL_INPUT);

    /* Load in image */
    img = eris_ifu_image_mef_load_region(in, ext_num, region_params);
    if (img == NULL) return CPL_ERROR_ILLEGAL_INPUT ;

    /* Load BPM */
    if (in_bpm != NULL) {
        cpl_image * bpm = eris_ifu_image_mef_load_region(in_bpm, ext_num_bpm, region_params);
        cpl_mask * bpm_mask = cpl_mask_threshold_image_create(bpm,
                -0.5, 0.5) ;
        cpl_image_delete(bpm) ;
        cpl_mask_not(bpm_mask) ;
        cpl_image_reject_from_mask(img, bpm_mask);
        cpl_mask_delete(bpm_mask) ;
    }

    /* Load error */
    cpl_image * err =NULL;
    if (in_err != NULL) {
        err = eris_ifu_image_mef_load_region(in_err, ext_num_err, region_params);
    } else if (ron < 0. && gain < 0.) {
        /* no error */
        err = NULL;
    } else {
        /* No passed error -> Create uniform error */
        eris_ifu_detector_shotnoise_model(img, gain, ron, &err);
    }

    /* Create out himage */
    *out_hdrl_ima = hdrl_image_create(img, err);

    /* Cleanup */
    cpl_image_delete(img);
    cpl_image_delete(err);
    eris_check_error_code("eris_ifu_hdrl_image_load");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * TODO: NOT USED
  @brief compute Strehl
  @param hima input image
* @param    wavelength      Nominal filter wavelength [m]
 * @param    m1_radius       primary mirror radius [m]
 * @param    m2_radius       obstruction radius [m]
 * @param    pixel_scale_x   image X pixel scale in [arcsec]
 * @param    pixel_scale_y   image Y pixel scale in [arcsec]
 * @param    flux_radius     radius used to sum the flux [arcsec]
  @param bkg_radius_low  radius used to determine the background [arcsec]
  @param bkg_radius_high radius used to determine the background [arcsec]
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
/*TODO: NOT USED
static cpl_error_code
eris_ifu_strehl_calculate(hdrl_image* hima,
		const double wavelength,
		const double m1_radius,
		const double m2_radius,
		const double pixel_scale_x,
		const double pixel_scale_y,
		const double flux_radius,
		const double bkg_radius_low,
		const double bkg_radius_high)
{


	hdrl_parameter * strehl_p = hdrl_strehl_parameter_create(wavelength, m1_radius,
			m2_radius, pixel_scale_x, pixel_scale_y, flux_radius, bkg_radius_low,
			bkg_radius_high);
	return cpl_error_get_code();
}
*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Set the group as RAW or CALIB in a frameset
  @param    frameset the frameset to process
  @param    parlist  the parameters controlling the data reduction
  @param    recipe_name the id of the recipe calling this step
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_strehl_compute(cpl_frameset * frameset, const cpl_parameterlist * parlist,
		const char* recipe_name)
{

    int                         extnum_raw = 0;
    int                         extnum_err = 1;
    int                         extnum_bpm = 2;
    hdrl_strehl_result          res;

    cpl_frame               *   in_frm ;
    cpl_frame               *   err_frm ;
    cpl_frame               *   bpm_frm ;
    hdrl_image              *   hima ;
    hdrl_parameter          *   region_params = NULL;

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe_name, CPL_ERROR_NULL_INPUT);
    /* Check initial Entries */
    if (eris_ifu_dfs_set_groups(frameset) != CPL_ERROR_NONE) {
    	return cpl_error_get_code();
    }

    /* Get parameters*/
    /* those params are not needed
     *
     *  char* param_name = NULL;
     *  const cpl_parameter     *   par = NULL;
    param_name = cpl_sprintf("%s.ext-nb-raw", recipe_name);
    par = cpl_parameterlist_find_const(parlist, param_name);
    extnum_raw = cpl_parameter_get_int(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.ext-nb-raw-err", recipe_name);
    par = cpl_parameterlist_find_const(parlist, param_name);
    extnum_err = cpl_parameter_get_int(par);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.ext-nb-raw-bpm", recipe_name);
    par = cpl_parameterlist_find_const(parlist, param_name);
    extnum_bpm = cpl_parameter_get_int(par);
    cpl_free(param_name);
    */

    /* Parse the Strehl Parameters */
    hdrl_parameter * p = hdrl_strehl_parameter_parse_parlist(parlist,
                                                             recipe_name);
    if (p == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                "Parsing of the strehl parameters failed");
    }

    /* Parse the Region Parameters
    region_params=hdrl_rect_region_parameter_parse_parlist(parlist,
            recipe_name, "region-") ;
    if (region_params == NULL) {
        hdrl_parameter_delete(p);
        return cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED,
                "Parsing of the region parameters failed");
    }
    */

    /* Load INPUT Data */
    /* Get the first Required frame */
    if ((in_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_CUBE))==NULL) {
        //if (region_params) hdrl_parameter_delete(region_params) ;
        if (p) hdrl_parameter_delete(p) ;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Missing RAW file");
    }
    //bpm_frm = cpl_frameset_find(frameset, ERIS_IFU_RAW_BPM) ;
    //err_frm = cpl_frameset_find(frameset, ERIS_IFU_RAW_ERROR) ;
    int sx = 0;
    int sy = 0;
    const char* fname = cpl_frame_get_filename(in_frm);
    cpl_image* in_ima = cpl_image_load(fname,CPL_TYPE_FLOAT, 0, 0);
    sx = cpl_image_get_size_x(in_ima);
    sy = cpl_image_get_size_y(in_ima);
    bpm_frm = in_frm;
    err_frm = in_frm;
    cpl_propertylist* plist = cpl_propertylist_load(fname,0);
    double ron = cpl_propertylist_get_double(plist,"ESO DET CHIP RON");
    double gain = cpl_propertylist_get_double(plist,"ESO DET CHIP GAIN");
    cpl_propertylist_delete(plist);
    cpl_image_delete(in_ima);
    region_params = hdrl_rect_region_parameter_create(1, 1, sx, sy);
    /* Load the image */
    if (eris_ifu_hdrl_image_load(in_frm, extnum_raw, err_frm, extnum_err,
                bpm_frm, extnum_bpm, region_params, ron, gain,
                &hima) != CPL_ERROR_NONE) {
        if (region_params) hdrl_parameter_delete(region_params) ;
        if (p) hdrl_parameter_delete(p) ;
        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                "Cannot load RAW image");
    }
    if (region_params) hdrl_parameter_delete(region_params) ;


    /*  This is not needed as ERRORS are provided by the input data
    if(err_frm == NULL) {
        // Replace the error image created by the ron/gain by the mad-scaled
        // rms of the image
       double dmad = 0.;
       cpl_image * img = cpl_image_load(cpl_frame_get_filename(in_frm),
    		   	   	   	   	   	   	    CPL_TYPE_DOUBLE, 0, extnum_raw);
       cpl_image_get_mad(img, &dmad);
       cpl_image_delete(img);

       cpl_image_multiply_scalar(hdrl_image_get_error(hima), 0.);
       cpl_image_add_scalar(hdrl_image_get_error(hima),
                            (dmad * CPL_MATH_STD_MAD));
    }
    */
    cpl_msg_info(cpl_func,"Handles NANs");
    eris_ifu_mask_nans_in_hdrlimage(&hima);
    /* Strehl COMPUTATION */
    res = hdrl_strehl_compute(hima, p);
    hdrl_parameter_delete(p);
    cpl_msg_info(cpl_func,"strehl=%g+-%g", res.strehl_value.data,
                 res.strehl_value.error);
    /* expected difference sqrt(pi / 2)  due to median */
    cpl_msg_info(cpl_func, "star peak at %g/%g: %g +- %g", res.star_x,
                 res.star_y, res.star_peak.data, res.star_peak.error);
    cpl_msg_info(cpl_func, "star flux %g+-%g", res.star_flux.data,
                 res.star_flux.error);
    cpl_msg_info(cpl_func,"median estimated background=%g+-%g "
                 "(computed error %g)",
                 res.star_background.data, res.star_background.error,
                 res.computed_background_error);
    cpl_propertylist* header=cpl_propertylist_new();
    cpl_propertylist_update_double(header, "ESO QC STREHL",
                                   res.strehl_value.data);
    cpl_propertylist_update_double(header, "ESO QC STREHL ERROR",
                                   res.strehl_value.error);
    cpl_propertylist_update_string(header, CPL_DFS_PRO_CATG, "ERIS_IFU_STREHL");

    eris_ifu_propertylist_save("ERIS_IFU_STREHL","eris_ifu_strehl.fits", header,
                         parlist,frameset);


    /* Cleanup */
    cpl_propertylist_delete(header);
    hdrl_image_delete(hima);
    /* In case of Poisson error model: */
    eris_check_error_code("eris_ifu_strehl_compute");
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    compute Strehl
  @param    frameset the frameset to process
  @param    parlist  the parameters controlling the data reduction
  @param    context  the recipe parameter context (instrument+recipe) prefixes
  @param    recipe_name the id of the recipe calling this step
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_stdstar_strehl_compute(cpl_frameset * frameset,
                                const cpl_parameterlist * parlist,
                                const char* context/*,
                                const char* recipe_name*/)
{
    //const cpl_parameter     *   par = NULL;
    int                         extnum_raw = 1;
    int                         extnum_err = 2;
    int                         extnum_bpm = 3;
    hdrl_strehl_result          res;

    cpl_frame               *   in_frm = NULL ;
    cpl_frame               *   err_frm ;
    cpl_frame               *   bpm_frm ;
    hdrl_image              *   hima ;
    hdrl_parameter          *   region_params = NULL;
    char* param_name = NULL;

    double wavelength = 0;
    //ifsPreopticsScale = eris_ifu_get_spiffier_preoptics_scale(sof->band);
    double pixel_scale_x = 0;
    double pixel_scale_y = 0;
    double flux_radius_pix = 1;
    double bkg_radius_low = 1;
    double bkg_radius_high = 1.5;

    cpl_ensure_code(frameset,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(context,CPL_ERROR_NULL_INPUT);

    param_name = cpl_sprintf("%s.strehl_flux_radius", context);
    flux_radius_pix = cpl_parameter_get_double(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


    param_name = cpl_sprintf("%s.strehl_bkg-radius-low", context);
    bkg_radius_low = cpl_parameter_get_double(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


    param_name = cpl_sprintf("%s.strehl_bkg-radius-high", context);
    bkg_radius_high = cpl_parameter_get_double(
    		cpl_parameterlist_find_const(parlist, param_name));
    cpl_free(param_name);


    ifsBand             band = UNDEFINED_BAND;
    ifsPreopticsScale   scale = UNDEFINED_SCALE;
    ifsInstrument       instrument = UNSET_INSTRUMENT;
    eris_ifu_jitter_get_instrument_settings(frameset, &instrument, &band, &scale);

    switch(scale) {
    case S250MAS:
    	pixel_scale_x = 0.250;
    	break;

    case S100MAS:
    	pixel_scale_x = 0.100;
    	break;

    case S25MAS:
    	pixel_scale_x = 0.025;
    	break;

    case UNDEFINED_SCALE:
    	pixel_scale_x = 0.025;
    	break;
    //TODO: FOLLOWING TO BE VERIFIED
    case PUPIL:
       	pixel_scale_x = 0.100;
       	break;
    }
    eris_ifu_get_central_lambda(band, &wavelength);
    wavelength *= 1.e-6; /* convert to [m] */
    pixel_scale_y = pixel_scale_x;
    cpl_msg_info(cpl_func,"pixel_scale: %g [arcsec]",pixel_scale_x);
    /* Determine flux_radius, bkg_radius_low, bkg_radius_high from QC.FWHM
     * on cube collapsed median*/
    cpl_frame* std_frm;
    cpl_frame* std_flux_frm;
    cpl_frame* psf_frm;
    //cpl_frameset_dump(frameset,stdout);
    std_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_CUBE_COADD_MEDIAN);
    std_flux_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_COADD_MEDIAN);
    psf_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD_MEDIAN);
    if ( (std_frm == NULL)  && (psf_frm == NULL) && (std_flux_frm == NULL)) {
    	return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
    			"Missing ERIS_IFU_PRO_JITTER_PSF_CUBE_COADD_MEDIAN file, Cannot compute Strehl,");
    } else {
    	if(std_frm != NULL) {
    		in_frm = std_frm;
    	} else if(psf_frm != NULL) {
    		in_frm = psf_frm;
    	} else if(std_flux_frm != NULL) {
    		in_frm = std_flux_frm;
    	}
    }

    cpl_propertylist* plist = cpl_propertylist_load(cpl_frame_get_filename(in_frm),0);
    double fwhm_maj = 9;
    double fwhm_min = 9;

    if(cpl_propertylist_has(plist, "ESO QC FWHM MAJ")) {
        fwhm_maj = cpl_propertylist_get_double(plist, "ESO QC FWHM MAJ");
        if(fwhm_maj == 0) {
        	if(cpl_propertylist_has(plist, "ESO QC FWHMX")) {
        		fwhm_maj = cpl_propertylist_get_double(plist, "ESO QC FWHMX");
        	}
        }
    }

    if(cpl_propertylist_has(plist, "ESO QC FWHM MIN")) {
    	fwhm_min = cpl_propertylist_get_double(plist, "ESO QC FWHM MIN");
    	if(fwhm_min == 0) {
    		if(cpl_propertylist_has(plist, "ESO QC FWHMY")) {
    			fwhm_min = cpl_propertylist_get_double(plist, "ESO QC FWHMY");
    		}
    	}
    }

    cpl_propertylist_delete(plist);
    //cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    //cpl_msg_info(cpl_func,"fwhm_min: %g fwhm_maj: %g",fwhm_min, fwhm_maj);
    double fwhm_sup = (fwhm_maj > fwhm_min) ? fwhm_maj : fwhm_min;
    if(flux_radius_pix == -1) {
    	cpl_msg_info(cpl_func,"pipeline setting flux_radius_pix: 3 * PSF sigmas");
        flux_radius_pix = 3. * fwhm_sup / CPL_MATH_FWHM_SIG; // 3 sigmas
    }
    cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    double flux_radius_arc = flux_radius_pix * pixel_scale_x;
    //cpl_msg_info(cpl_func,"bkg_radius_low: %g",bkg_radius_low);
    if(bkg_radius_low == -1) {
    	cpl_msg_info(cpl_func,"pipeline setting bkg_radius_low: 1.5 * flux_radius");
        bkg_radius_low = 1.5 * flux_radius_arc;
    } else {
    	bkg_radius_low *= pixel_scale_x;
    }
    //cpl_msg_info(cpl_func,"bkg_radius_high: %g",bkg_radius_high);
    if(bkg_radius_high == -1) {
    	cpl_msg_info(cpl_func,"pipeline setting bkg_radius_high: 2.0 * flux_radius");
        bkg_radius_high = 2.0 * flux_radius_arc;
    } else {
    	bkg_radius_high *= pixel_scale_x;
    }

    cpl_msg_info(cpl_func,"flux_radius: %g [arcsec] bkg_radius_low: %g [arcsec] bkg_radius_high: %g [arcsec]",
        		flux_radius_arc, bkg_radius_low,bkg_radius_high);

    cpl_msg_info(cpl_func,"flux_radius: %g [pix] bkg_radius_low: %g [pix] bkg_radius_high: %g [pix]",
    		flux_radius_arc / pixel_scale_x,bkg_radius_low / pixel_scale_x,bkg_radius_high / pixel_scale_x);


    /* Load INPUT Data for Strehl Computation: we use the cube collapsed mean */
    std_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_CUBE_MEAN);
    psf_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_PSF_CUBE_MEAN);
    std_flux_frm = cpl_frameset_find(frameset, ERIS_IFU_PRO_JITTER_STD_FLUX_CUBE_MEAN);
    if ( (std_frm == NULL) && (std_flux_frm == NULL)  && (psf_frm == NULL) ) {

        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                "Missing ERIS_IFU_PRO_JITTER_PSF_CUBE_MEAN file, Cannot compute Strehl,");
    } else {
    	if(std_frm != NULL) {
    		in_frm = std_frm;
    	} else if(psf_frm != NULL) {
    		in_frm = psf_frm;
    	}
    }

    int sx = 0;
    int sy = 0;
    const char* fname = cpl_frame_get_filename(in_frm);
    cpl_msg_info(cpl_func,"strehl computed on: %s",fname);
    cpl_image* in_ima = cpl_image_load(fname,CPL_TYPE_FLOAT, 0, 1);

    sx = cpl_image_get_size_x(in_ima);
    sy = cpl_image_get_size_y(in_ima);
    bpm_frm = in_frm;
    err_frm = in_frm;

    plist = cpl_propertylist_load(fname,0);
    double ron = cpl_propertylist_get_double(plist,"ESO DET CHIP RON");
    double gain = cpl_propertylist_get_double(plist,"ESO DET CHIP GAIN");

    //cpl_propertylist_delete(plist);
    cpl_image_delete(in_ima);
    region_params = hdrl_rect_region_parameter_create(1, 1, sx, sy);

    /* Load the image */
    if (eris_ifu_hdrl_image_load(in_frm, extnum_raw, err_frm, extnum_err,
                bpm_frm, extnum_bpm, region_params, ron, gain,
                &hima) != CPL_ERROR_NONE) {
        if (region_params) hdrl_parameter_delete(region_params) ;

        return cpl_error_set_message(cpl_func, CPL_ERROR_FILE_NOT_FOUND,
                "Cannot load RAW image");
    }
    if (region_params) hdrl_parameter_delete(region_params) ;
    

    //cpl_msg_info(cpl_func,"Handles NANs");
    eris_ifu_mask_nans_in_hdrlimage(&hima);

    cpl_size cx = 0;
    cpl_size cy = 0;
    cpl_size margin_x = 4;
    cpl_size margin_y = 4;
    cpl_size llx = margin_x;
    cpl_size lly = margin_y;
    cpl_size urx = sx - margin_x;
    cpl_size ury = sy - margin_y;

    cpl_image_get_maxpos_window(hdrl_image_get_image(hima), llx, lly, urx, ury, &cx, &cy);
    cpl_msg_info(cpl_func,"Image max detected at cx: %lld, cy: %lld",cx, cy);
    /*
    cpl_image_save(hdrl_image_get_image(hima),"ima.fits", CPL_TYPE_DOUBLE, NULL,
    		CPL_IO_DEFAULT);
    */
    cpl_msg_info(cpl_func,"check Strehl parameters");
    /*
    cpl_msg_info(cpl_func,"flux_radius[pix]: %g bkg_radius_low[pix]: %g bkg_radius_high[pix]: %g",
        	        		flux_radius_pix ,bkg_radius_low / pixel_scale_x,bkg_radius_high / pixel_scale_x );
    */
    res.strehl_value.data = 0;
    res.strehl_value.error = 0;
    hdrl_parameter * p;
    cpl_msg_info(cpl_func,"wavelength: %g M1: %g M2: %g",wavelength, M1_RADIUS,M2_RADIUS);
    cpl_msg_info(cpl_func,"flux_radius_pix: %g cx: %lld cy: %lld sx: %d sy:%d",flux_radius_pix, cx, cy, sx, sy);
    if( (flux_radius_pix + cx) > (sx -2) ||
    	(flux_radius_pix + cy) > (sy -2) ||
		(cx-flux_radius_pix) < 2 ||
		(cy-flux_radius_pix) < 2){
    	//cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    	cpl_msg_info(cpl_func,"correct Strehl parameters");
    	flux_radius_pix = ( (flux_radius_pix + cx) > (sx -2) ) ?  flux_radius_pix : sx -2;
    	flux_radius_pix = ( (flux_radius_pix + cy) > (sy -2) ) ?  flux_radius_pix : sy -2;
    	flux_radius_pix = ( (cx - flux_radius_pix) < 2 ) ?  cx -2 : flux_radius_pix;
    	flux_radius_pix = ( (cy - flux_radius_pix) < 2 ) ?  cy -2 : flux_radius_pix;
    	//cpl_msg_info(cpl_func,"flux_radius_pix %g pixel_scale_x %g", flux_radius_pix, pixel_scale_x);
    	flux_radius_arc = flux_radius_pix * pixel_scale_x;
    	bkg_radius_low = flux_radius_arc;
    	bkg_radius_high = (flux_radius_pix + 1) * pixel_scale_x;
    	cpl_msg_info(cpl_func,"flux_radius: %g bkg_radius_low: %g bkg_radius_high: %g",
    	        		flux_radius_arc ,bkg_radius_low ,bkg_radius_high );

    	if(bkg_radius_low > 0) {
    	p = hdrl_strehl_parameter_create(wavelength,
    	    		M1_RADIUS, M2_RADIUS, pixel_scale_x, pixel_scale_y,
    				flux_radius_arc, bkg_radius_low, bkg_radius_high) ;
    	  /* Strehl COMPUTATION */
    	    res = hdrl_strehl_compute(hima, p);
    	    hdrl_parameter_delete(p);
    	} else {
    		cpl_msg_warning(cpl_func,"bkg_radius_low: %g. Skip Strehl computation.", bkg_radius_low);
    	}


    } else {
    	//cpl_msg_info(cpl_func,"fwhm_sup: %g",fwhm_sup);
    	//for (flux_radius_pix=fwhm_sup; flux_radius_pix < 3 * fwhm_sup; flux_radius_pix += 0.2) {
    		flux_radius_arc = flux_radius_pix * pixel_scale_x;
    		cpl_msg_info(cpl_func,"flux_radius_pix: %g",flux_radius_pix);
    		if(bkg_radius_low > 0) {
    		p = hdrl_strehl_parameter_create(wavelength,
    				M1_RADIUS, M2_RADIUS, pixel_scale_x, pixel_scale_y,
					flux_radius_arc, bkg_radius_low, bkg_radius_high) ;

    		res = hdrl_strehl_compute(hima, p);
    		hdrl_parameter_delete(p);
    		cpl_msg_info(cpl_func,"strehl=%g+-%g", res.strehl_value.data,
    		    		    			res.strehl_value.error);
    		} else {
    			cpl_msg_warning(cpl_func,"bkg_radius_low: %g. Skip Strehl computation.", bkg_radius_low);
    		}

    	//}
    }



    if(cpl_error_get_code() != CPL_ERROR_NONE) {
       cpl_msg_error(cpl_func,"Error computing Strehl. Temporarily suppress it.");
       cpl_error_set(cpl_func,CPL_ERROR_NONE);
       hdrl_image_delete(hima);
       return CPL_ERROR_NONE;
    } else {
    	if (res.strehl_value.data > 0) {

    		cpl_msg_info(cpl_func,"strehl=%g+-%g", res.strehl_value.data,
    				res.strehl_value.error);
    		/* expected difference sqrt(pi / 2)  due to median */
    		cpl_msg_info(cpl_func, "star peak at %g/%g: %g +- %g", res.star_x,
    				res.star_y, res.star_peak.data, res.star_peak.error);
    		cpl_msg_info(cpl_func, "star flux %g+-%g", res.star_flux.data,
    				res.star_flux.error);
    		cpl_msg_info(cpl_func,"median estimated background=%g+-%g "
    				"(computed error %g)",
					res.star_background.data, res.star_background.error,
					res.computed_background_error);

    		double qc_strehl = 0;
    		double qc_strehl_err = 0;
    		if(isnan(res.strehl_value.data)){
    			qc_strehl=999.;
    			qc_strehl_err=999.;
    		} else {
    			qc_strehl=res.strehl_value.data;
    			qc_strehl_err=res.strehl_value.error;
    		}

    		cpl_image* data = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, extnum_raw);

    		cpl_image* errs = cpl_image_load(fname, CPL_TYPE_DOUBLE, 0, extnum_err);
    		cpl_propertylist* eheader = cpl_propertylist_load(fname, extnum_err);


    		cpl_image* qual = cpl_image_load(fname, CPL_TYPE_INT, 0, extnum_bpm);
    		cpl_propertylist* qheader = cpl_propertylist_load(fname, extnum_bpm);



    		cpl_propertylist_append_double(plist, "ESO QC STREHL", qc_strehl);
    		cpl_propertylist_append_double(plist, "ESO QC STREHL ERROR", qc_strehl_err);
    		cpl_image_save(data, fname, CPL_TYPE_DOUBLE, plist, CPL_IO_DEFAULT);
    		//cpl_propertylist_dump(plist,stdout);
    		/*
        const char* pcatg = cpl_propertylist_get_string(plist, FHDR_PRO_CATG);
    	eris_ifu_save_image(frameset, plist, parlist, recipe_name, pcatg,
    			fname, CPL_TYPE_DOUBLE, data);
    		 */

    		cpl_image_save(errs, fname, CPL_TYPE_DOUBLE, eheader, CPL_IO_EXTEND);
    		cpl_image_save(qual, fname, CPL_TYPE_INT, qheader, CPL_IO_EXTEND);
    		cpl_image_delete(data);
    		cpl_image_delete(errs);
    		cpl_image_delete(qual);

    		cpl_propertylist_delete(eheader);
    		cpl_propertylist_delete(qheader);
    		//cpl_msg_info(cpl_func,"ok5");
    	}
    }
    cpl_propertylist_delete(plist);
    /* Cleanup */

    hdrl_image_delete(hima);
    eris_check_error_code("eris_ifu_stdstar_strehl_compute");

    return cpl_error_get_code();
}

/** @endcond */

/**@}*/
