/* $Id: eris_ifu_utils.c,v 1.12 2013-03-25 11:46:49 cgarcia Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-03-25 11:46:49 $
 * $Revision: 1.12 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"
#include "eris_ifu_functions.h"
#include "eris_utils.h"
#include "eris_pfits.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_utils     IFU Miscellaneous Utilities
 * @{
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/**
    @brief
        Implements the where-function knownm from IDL.

    @param data    The vector.
    @param val     The value.
    @param op      The operator.

    @return
        A vector with the indices to the input vector which fulfill the given
        condition. When the condition is never met NULL is returned.

    The vector is checked elementwise like 'data[i] op val'.
    The operator can be any of following enums:
        eq: data[i] == val
        ne: data[i] != val
        ge: data[i] >= val
        gt: data[i] > val
        le: data[i] <= val
        lt: data[i] @c < val

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c data is NULL or if an illegal @c op is
                                provided.
 */
cpl_vector* eris_ifu_idl_where(const cpl_vector *data, double val, int op)
{
	cpl_vector      *ret_vec    = NULL;
	double          *pret_vec   = NULL;
	const double    *pdata      = NULL;
	int             j           = 0,
			size        = 0;

	cpl_ensure(data, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		size = cpl_vector_get_size(data);

		// create vector of same size as input data and set default values
		// to -1.0 (will be shortened at the end)
		ret_vec = cpl_vector_new(size);
		pret_vec = cpl_vector_get_data(ret_vec);

		cpl_vector_fill(ret_vec, -1.0);

		// identify valid values in data and copy inidices to temp vector
		pdata = cpl_vector_get_data_const(data);

		j = 0;
		for (int i = 0; i < size; i++) {
			switch (op) {
			case eq:
				if (pdata[i] == val)
					pret_vec[j++] = i;
				break;
			case ne:
				if (fabs(pdata[i]-val) > 0.0001)
					pret_vec[j++] = i;
				break;
			case ge:
				if (pdata[i] >= val)
					pret_vec[j++] = i;
				break;
			case gt:
				if (pdata[i] > val)
					pret_vec[j++] = i;
				break;
			case le:
				if (pdata[i] <= val)
					pret_vec[j++] = i;
				break;
			case lt:
				if (pdata[i] < val)
					pret_vec[j++] = i;
				break;
			default:
				SET_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
						"illegal operator");
				break;
			}
		}

		//cut trailing -1
		eris_ifu_cut_endings(&ret_vec, NULL, NULL, TRUE);
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_vector(&ret_vec);
	}
	eris_check_error_code("eris_ifu_idl_where");
	return ret_vec;
}

/**
    @brief
        Returns a vector of given indices.

    @param data     The data vector.
    @param indices  The indices to the data vector.

    @return
        A (possibly shorter) vector with the desired values.

    The returned vector contains the values of @c data indicated by @c indices .
    @c indices needn't be orderd, so will also have the returned vector the same
    order.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c data is NULL or if an illegal @c op is
                                provided.
 */
cpl_vector* eris_ifu_idl_values_at_indices(const cpl_vector *data,
                                           const cpl_vector *indices)
{
	cpl_vector*     ret_vec     = NULL;
	double          *pret_vec   = NULL;
	const double    *pdata      = NULL,
                    *pindices   = NULL;
	int             size        = 0;

	cpl_ensure(data, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(indices, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		pdata = cpl_vector_get_data_const(data);
		pindices = cpl_vector_get_data_const(indices);

		size = cpl_vector_get_size(indices);

		ret_vec = cpl_vector_new(size);
		pret_vec = cpl_vector_get_data(ret_vec);

		for (int i = 0; i < size; i++) {
			if ((int)pindices[i] < 0) {
				SET_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
						"One of the indices is < 0!");
			} else {
				pret_vec[i] = pdata[(int)pindices[i]];
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_vector(&ret_vec);
	}
	eris_check_error_code("eris_ifu_idl_values_at_indices");
	return ret_vec;
}

/**
    @brief
        Cut leading and trailing -1 of a vector.

    @param vec    (Input)  Vector to trim.
                  (Output) Trimmed vector, if cut is TRUE.
    @param begin  NULL or variable to hold determined beginning
                  position (zero based). Set to zero in case of error.
    @param end    NULL or variable to hold determined ending
                  position (zero based). Set to zero in case of error.
    @param cut    TRUE if @c vec should be trimmed,
                  FALSE if just the start and end positions should be determined.
                  Setting @c begin and @c end to NULL and @c cut to FALSE is
                  senseless.

    @return
        The function returns CPL_ERROR_NONE on success or a CPL error code
        otherwise.

    This function only cuts -1. Other negative numbers are classified as ok.

    Possible cpl_error_code set in this function:

    @li CPL_ERROR_NULL_INPUT    if @c vec is NULL.
 */
cpl_error_code eris_ifu_cut_endings(cpl_vector** vec,
                                    int *begin, int *end, int cut)
{
    cpl_error_code  err         = CPL_ERROR_NONE;
	int             min         = 0,
                    max         = 0;
    double          *pvec       = NULL;
    cpl_vector      *tmp_vec    = NULL;

    cpl_ensure_code(vec, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(*vec, CPL_ERROR_NULL_INPUT);

	TRY
	{
		pvec = cpl_vector_get_data(*vec);
        min = 0;                            // min_pos
		max = cpl_vector_get_size(*vec)-1;  // max_pos

		// find beginning
		for (int i = 0; i < cpl_vector_get_size(*vec); i++) {
			if (pvec[i] == -1) {
				min = i+1;
			} else {
				break;
			}
		}

        if (min != cpl_vector_get_size(*vec)) {
			// find ending
			for (int i = cpl_vector_get_size(*vec)-1; i >= 0; i--) {
				if (pvec[i] == -1) {
					max = i-1;
				} else {
					break;
				}
			}

			if (cut == TRUE) {
				// extract appropriate part of vector
				tmp_vec = cpl_vector_extract(*vec, min, max, 1);

				cpl_vector_delete(*vec); *vec= NULL;

				*vec = tmp_vec;
			}
		} else {
			if (cut == TRUE) {
				// all values are -1 return NULL
				cpl_vector_delete(*vec); *vec = NULL;
			}

			min = 0;
			max = 0;
		}

        // set return values if wanted
		if (begin != NULL) {
			*begin = min;
		}

		if (end != NULL) {
			*end = max;
		}
	}
	CATCH
	{

		err = cpl_error_get_code();
//		eris_print_rec_status(9);
//		cpl_msg_info(cpl_func,"ok9");
		if (begin != NULL) {
			*begin = 0;
		}

		if (end != NULL) {
			*end = 0;
		}

		eris_ifu_free_vector(vec);
	}
	eris_check_error_code("eris_ifu_cut_endings");
	return err;
}

/**
  @brief    Return HDRL imagelist by searching tagged frames in frameset
  @param    frameset The input frameset (of the recipe)
  @param    tag The tag name of the frames to be selected
  @param   exposureCorrectionMode: data correction to be applied
  @return   a pointer to the HDRL imagelist or NULL in case of an error

  The function returns an HDRL imagelist (and, if requested, a CPL imagelist)
  by searching an input frameset for frames with the proper tag name. If no
  frames with the tag are found a NULL pointer will be returned. No error images
  in the HDRL imagelist will be created.

 */
hdrl_imagelist * eris_ifu_get_hdrlimagelist_by_tag(cpl_frameset *frameset,
		const char *tag,
		int exposureCorrectionMode)
{
	hdrl_imagelist  *hdrl_list  = NULL;
	cpl_frameset    *frames     = NULL;
	hdrl_image      *tmp_img    = NULL;
	const cpl_frame     *fr             = NULL;
	const char          *fn             = NULL;

	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		frames = eris_ifu_get_frameset_by_tag(frameset, tag);
		hdrl_list = hdrl_imagelist_new();

		/*
		 * Loop through all FLAT_LAMP frames and save on- and off-frames
		 * accordingly in their cpl_imagelists
		 */
		fr = cpl_frameset_find_const(frames, tag);

		while (fr != NULL) {
			fn = cpl_frame_get_filename(fr);
			eris_ifu_file_exists(fn);

			/* If the frame has a tag we process it. Else it is an object and
			 * is ignored */
			if (cpl_frame_get_tag(fr) != NULL) {
						tmp_img = eris_ifu_load_exposure_file(fn,
								exposureCorrectionMode, NULL);

						hdrl_imagelist_set(hdrl_list, tmp_img,
								hdrl_imagelist_get_size(hdrl_list));
			} else {
				/* if (tag == NULL) do nothing */
			}

			/* next frame */
			fr = cpl_frameset_find_const(frames, NULL);
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_hdrl_imagelist(&hdrl_list);
	}

	eris_ifu_free_frameset(&frames);
	eris_check_error_code("eris_ifu_get_hdrlimagelist_by_tag");
	return hdrl_list;
}

/**
 * @brief  Get frames with given tag from frameset
 * @param  frameset    frame set
 * @param  tag         to search for
 * @return newly allocated, possibly empty, frameset, or NULL on error
 */
cpl_frameset* eris_ifu_get_frameset_by_tag(
		const cpl_frameset *frameset,
		const char *tag)
{
	cpl_frameset    *frames = NULL;
	const cpl_frame *curFrame ;
	cpl_frame         *locFrame ;
	cpl_size         frameCnt;
	int i;

	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		/* Check whether there are frames with the requested tag */
		frameCnt = cpl_frameset_count_tags(frameset, tag);
		CHECK_ERROR_STATE();
		if (frameCnt == 0)
		{
			BRK_WITH_ERROR_MSG(CPL_ERROR_DATA_NOT_FOUND,
					"frameset contains no frames with tag %s", tag);
		}

		/* Create the output frameset */
		frames = cpl_frameset_new() ;

		/* Extract frames with proper tag into the output frameset */
		frameCnt = cpl_frameset_get_size(frameset) ;
		for (i=0 ; i<frameCnt ; i++) {
			curFrame = cpl_frameset_get_position_const(frameset, i) ;
			if (!strcmp(cpl_frame_get_tag(curFrame), tag)) {
				locFrame = cpl_frame_duplicate(curFrame) ;
				cpl_frameset_insert(frames, locFrame) ;
			}
		}
	}
	CATCH
	{
		frames = NULL;
	}
	eris_check_error_code("eris_ifu_get_frameset_by_tag");
	return frames;
}

/**
  @brief    Append a parameterlist to another one
  @param    p1 The 1st list to be extended
  @param    p2 The 2ns list to be appendend
  @return   CPL_ERROR_NONE in case everything went fine
 */
cpl_error_code eris_ifu_parameterlist_append_list(
		cpl_parameterlist       *p1,
		const cpl_parameterlist *p2)
{
	cpl_error_code      err = CPL_ERROR_NONE;
	const cpl_parameter *p  = NULL;

	cpl_ensure_code(p1, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(p2, CPL_ERROR_NULL_INPUT);

	TRY
	{
		p = cpl_parameterlist_get_first_const(p2);
		cpl_parameterlist_append(p1, cpl_parameter_duplicate(p));
		while ((p = cpl_parameterlist_get_next_const(p2)) != NULL) {
			cpl_parameterlist_append(p1, cpl_parameter_duplicate(p));
		}
		CHECK_ERROR_STATE();
	}
	CATCH
	{
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_parameterlist_append_list");
	return err;
}

/**
  @brief    free memory and set pointer to null
  @param    item
  @return   void
 */
void eris_ifu_free_hdrl_image(hdrl_image **item) {
	if ((item != NULL) && (*item != NULL)) {
		hdrl_image_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_hdrl_imagelist(hdrl_imagelist **item) {
	if ((item != NULL) && (*item != NULL)) {
		hdrl_imagelist_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_image(cpl_image **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_image_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_imagelist(cpl_imagelist **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_imagelist_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_mask(cpl_mask **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_mask_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_matrix(cpl_matrix **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_matrix_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_propertylist(cpl_propertylist **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_propertylist_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_vector(cpl_vector **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_vector_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_ifu_vector(eris_ifu_vector **item) {
	if ((item != NULL) && (*item != NULL)) {
		eris_ifu_vector_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_string(char **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_free(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_frameset(cpl_frameset **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_frameset_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_frame(cpl_frame **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_frame_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_hdrl_parameter(hdrl_parameter **item) {
	if ((item != NULL) && (*item != NULL)) {
		hdrl_parameter_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_parameter(cpl_parameter **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_parameter_delete(*item);
		*item = NULL;
	}
}

/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_parameterlist(cpl_parameterlist **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_parameterlist_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_table(cpl_table **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_table_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_double_array(double **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_free(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_float_array(float **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_free(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_int_array(int**item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_free(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_polynomial(cpl_polynomial **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_polynomial_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_bivector(cpl_bivector **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_bivector_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    free memory and set pointer to null
  @param    item
 */
void eris_ifu_free_apertures(cpl_apertures **item) {
	if ((item != NULL) && (*item != NULL)) {
		cpl_apertures_delete(*item);
		*item = NULL;
	}
}
/**
  @brief    save image
  @param    fs input frameset
  @param    plist input propertylist
  @param    parlist input propertylist
  @param    recipe input recipe name
  @param    procatg product category
  @param    filename product file name
  @param    type of image
  @param    image image to save
  @return   CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_image(cpl_frameset *fs,
		const cpl_propertylist *plist,
		const cpl_parameterlist *parlist,
		const char *recipe,
		const char *procatg,
		const char *filename,
		cpl_type type,
		const cpl_image *image)
{
	cpl_error_code  err         = CPL_ERROR_NONE;
	char            *pckgName   = NULL;
	cpl_propertylist *plistdup  = NULL;

	cpl_ensure_code(fs, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);

	TRY
	{
		plistdup = cpl_propertylist_duplicate(plist);

		pckgName = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);

		cpl_propertylist_update_string(plistdup, CPL_DFS_PRO_CATG, procatg);

		cpl_dfs_save_image(fs, NULL, parlist, fs, NULL, image, type,
				recipe, plistdup, "RADECSYS", pckgName, filename);

		cpl_propertylist_delete(plistdup);
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}

	eris_ifu_free_string(&pckgName);
	eris_check_error_code("eris_ifu_save_image");
	return err;
}
/**
  @brief    save imagelist
  @param    fs input frameset
  @param    inherit  frame from which to inherit header
  @param    plist input propertylist
  @param    parlist input propertylist
  @param    recipe input recipe name
  @param    procatg product category
  @param    filename product file name
  @param    type of imagelist
  @param    imglist imagelist to save
  @return   CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_imagelist(cpl_frameset *fs,
		cpl_frame *inherit,
		const cpl_propertylist *plist,
		const cpl_parameterlist *parlist,
		const char *recipe,
		const char *procatg,
		const char *filename,
		cpl_type type,
		const cpl_imagelist *imglist)
{
	cpl_error_code  err         = CPL_ERROR_NONE;
	char            *pckgName   = NULL;
	cpl_propertylist *plistdup  = NULL;

	cpl_ensure_code(fs, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(plist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(imglist, CPL_ERROR_NULL_INPUT);

	TRY
	{
		plistdup = cpl_propertylist_duplicate(plist);
		pckgName = cpl_sprintf("%s%s%s", PACKAGE, "/", PACKAGE_VERSION);
		cpl_propertylist_update_string(plistdup, CPL_DFS_PRO_CATG, procatg);
		// TODO: why adding CPL_DFS_PRO_TYPE, CPL_DFS_PRO_TECH ???
		cpl_propertylist_update_string(plistdup, CPL_DFS_PRO_TYPE, "CUBE");
		cpl_propertylist_update_string(plistdup, CPL_DFS_PRO_TECH, "CUBE");
		//cpl_propertylist_append_string(plist, "PRODCATG", "SCIENCE.CUBE.IFS");
		cpl_propertylist_erase(plistdup, "RADECSYS");
		//eris_ifu_plist_erase_wcs(plistdup);
		CHECK_ERROR_STATE();

		cpl_dfs_save_imagelist(fs, NULL, parlist, fs, inherit, imglist, type,
				recipe, plistdup, "RADECSYS", pckgName, filename);

	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}

	eris_ifu_free_string(&pckgName);
	cpl_propertylist_delete(plistdup);
	eris_check_error_code("eris_ifu_save_imagelist");
	return err;
}

/**
  @brief    'Add' the mask to the image
  @param    img     Image to alter
  @param    mask    Mask to add
  @return   CPL error code



  Possible cpl_error_code set in this function:\n
  - CPL_ERROR_NULL_INPUT The CPL image is a NULL pointer
  - CPL_TYPE_INVALID     The image has less two times the border size columns

 */
cpl_error_code eris_ifu_hdrl_image_reject_mask(hdrl_image *img,
		const cpl_mask *mask)
{
	cpl_error_code      err         = CPL_ERROR_NONE;
	const cpl_binary    *pmask      = NULL;
	cpl_mask            *img_mask   = NULL;
	cpl_binary          *pimg_mask  = NULL;
	cpl_size            nRows       = 0;
	cpl_size            nCols       = 0;

	cpl_ensure_code(img, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(mask, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(hdrl_image_get_size_x(img)==cpl_mask_get_size_x(mask),
			CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(hdrl_image_get_size_y(img)==cpl_mask_get_size_y(mask),
			CPL_ERROR_ILLEGAL_INPUT);

	TRY
	{
		// reject from image
		img_mask = hdrl_image_get_mask(img);
		pimg_mask = cpl_mask_get_data(img_mask);
		pmask = cpl_mask_get_data_const(mask);

		nCols = hdrl_image_get_size_x(img);
		nRows = hdrl_image_get_size_y(img);
		for (cpl_size y = 0; y < nRows; y++) {
			for (cpl_size x = 0; x < nCols; x++) {
				if ((pmask[x+y*nCols] == BAD_PIX) &&
						(pimg_mask[x+y*nCols] == GOOD_PIX))
				{
					pimg_mask[x+y*nCols] = BAD_PIX;
				}
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_hdrl_image_reject_mask");
	return err;
}
/**
 * @brief save vector
 * @param vec input vector
 * @param filename file name of product
 * @param create code to indicate to create file
 * @param pl propertylist
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_vector_dbg(const cpl_vector *vec,
		const char* filename, int create,
		const cpl_propertylist *pl)
{
	cpl_ensure_code(vec,CPL_ERROR_NULL_INPUT);

	return cpl_vector_save(vec, filename, CPL_TYPE_DOUBLE, pl, create);
}
/**
 * @brief save vector
 * @param vec input vector
 * @param filename file name of product
 * @param create code to indicate to create file
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_ifu_vector_dbg(const eris_ifu_vector *vec,
		const char* filename, int create)
{
	return eris_ifu_vector_save(vec, filename, CPL_TYPE_DOUBLE, NULL, create, -10.0);
}

/**
  col   1 for x only, 2 for y only, 0 for both
 */
/**
 * @brief save bivector as a vector (col=1 or col=2) or as a table
 * @param bivec input bivector
 * @param filename file name of product
 * @param col parameter to indicate what to save
 * @param create code to indicate to create file
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_bivector_dbg(const cpl_bivector *bivec,
		const char* filename, int col, int create)
{
//	int             size    = cpl_bivector_get_size(bivec);
//	const double    *px     = cpl_bivector_get_x_data_const(bivec),
//			*py     = cpl_bivector_get_y_data_const(bivec);
    const cpl_vector    *x = NULL,
                        *y = NULL;

	if (col == 1) {
        x = cpl_bivector_get_x_const(bivec);
        return eris_ifu_save_vector_dbg(x, filename, create, NULL);
	} else if (col == 2) {
        y = cpl_bivector_get_x_const(bivec);
        return eris_ifu_save_vector_dbg(y, filename, create, NULL);
    } else if (col == 0) {
        x = cpl_bivector_get_x_const(bivec);
        y = cpl_bivector_get_y_const(bivec);
        eris_ifu_save_vector_dbg(x, filename, create, NULL);
        return eris_ifu_save_vector_dbg(y, filename, CPL_IO_EXTEND, NULL);
    } else {
		return CPL_ERROR_ILLEGAL_INPUT;
	}
}
/**
 * @brief eris_ifu_save_image_dbg
 * @param img input image
 * @param filename file name of product
 * @param create code to indicate to create file
 * @param pl input propertylist
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_image_dbg(const cpl_image *img,
		const char* filename, int create,
		const cpl_propertylist *pl)
{
	cpl_ensure_code(img,CPL_ERROR_NULL_INPUT);

	if (img == NULL) {
		return cpl_image_save(img, filename,
				CPL_TYPE_INT, pl, create);
	} else if (cpl_image_get_type(img) == CPL_TYPE_FLOAT) {
		return cpl_image_save(img, filename,
				CPL_TYPE_FLOAT, pl, create);
	} else if (cpl_image_get_type(img) == CPL_TYPE_DOUBLE) {
		return cpl_image_save(img, filename,
				CPL_TYPE_DOUBLE, pl, create);
	} else if (cpl_image_get_type(img) == CPL_TYPE_INT) {
		return cpl_image_save(img, filename,
				CPL_TYPE_INT, pl, create);
	} else {
		return CPL_ERROR_ILLEGAL_INPUT;
	}
}
/**
 * @brief eris_ifu_save_table_dbg
 * @param tbl input table
 * @param filename file name of product
 * @param create code to indicate to create file
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_table_dbg(const cpl_table *tbl,
		const char* filename, int create)
{
	return cpl_table_save(tbl, NULL, NULL, filename, create);
}
/**
 * @brief eris_ifu_save_mask_dbg
 * @param mask input mask
 * @param filename file name of product
 * @param create code to indicate to create file
 * @param pl input propertylist
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_mask_dbg(const cpl_mask *mask,
		const char* filename, int create,
		const cpl_propertylist *pl)
{
	return cpl_mask_save(mask, filename, pl, create);
}

/**
 * @brief save CPL imagelist for debug purposes
 * @param imglist input image list
 * @param filename file name of product
 * @param create code to indicate to create file
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_imagelist_dbg(const cpl_imagelist *imglist,
		const char* filename, int create)
{
	cpl_ensure_code(imglist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(cpl_imagelist_get_size(imglist) >= 0,
			CPL_ERROR_ILLEGAL_INPUT);

	const cpl_image *img = cpl_imagelist_get_const(imglist, 0);
	eris_check_error_code("eris_ifu_save_imagelist_dbg");
	if (cpl_image_get_type(img) == CPL_TYPE_FLOAT) {
		return cpl_imagelist_save(imglist, filename,
				CPL_TYPE_FLOAT, NULL, create);
	} else if (cpl_image_get_type(img) == CPL_TYPE_DOUBLE) {
		return cpl_imagelist_save(imglist, filename,
				CPL_TYPE_DOUBLE, NULL, create);
	} else {
		return CPL_ERROR_ILLEGAL_INPUT;
	}
}
/**
 * @brief save CPL image for debug purposes
 * @param img input image
 * @param name file name of product
 * @param singlefile if HDRL image content is saved as a single file (MEF)
 * @param pl input propertylist
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_cpl_image_dbg(const cpl_image *img,
		const char* name, int singlefile,
		const cpl_propertylist *pl)
{
	char                *fn     = NULL;
	const cpl_mask      *mask   = NULL;
	cpl_error_code      err     = CPL_ERROR_NONE;
	int                 create  = CPL_IO_CREATE;

	TRY
	{
		if (singlefile) {
			fn = cpl_sprintf("%s.fits", name);
		} else {
			fn = cpl_sprintf("%s_data.fits", name);
		}
		eris_ifu_save_image_dbg(img, fn, create, pl);
		cpl_free(fn); fn = NULL;

		if (singlefile) {
			fn = cpl_sprintf("%s.fits", name);
			create = CPL_IO_EXTEND;
		} else {
			fn = cpl_sprintf("%s_mask.fits", name);
			create = CPL_IO_CREATE;
		}
		mask = cpl_image_get_bpm_const(img);
		eris_ifu_save_mask_dbg(mask, fn, create, pl);
		cpl_free(fn); fn = NULL;
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	cpl_free(fn); fn = NULL;
	eris_check_error_code("eris_ifu_save_cpl_image_dbg");
	return err;
}
/**
 * @brief  save CPL imagelist for debugging purposes
 * @param imglist input imagelist
 * @param name file name of product
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_cpl_imagelist_dbg(const cpl_imagelist *imglist,
		const char* name)
{
	char            *fn   = NULL;
	cpl_error_code  err   = CPL_ERROR_NONE;

	TRY
	{
		fn = cpl_sprintf("%s.fits", name);
		eris_ifu_save_imagelist_dbg(imglist, fn, CPL_IO_CREATE);
		eris_ifu_free_string(&fn);
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_save_cpl_imagelist_dbg");
	return err;
}
/**
 * @brief save HDRL imagelist for debugging purposes
 * @param hdrl_img input image
 * @param filename file name of product
 * @param singlefile if HDRL image content is saved as a single file (MEF)
 * @param pl input propertylist
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_hdrl_image_dbg(const hdrl_image *hdrl_img,
		const char* filename, int singlefile,
		const cpl_propertylist *pl)
{
	char                *fn     = NULL;
	const cpl_image     *img    = NULL;
	const cpl_mask      *mask   = NULL;
	cpl_error_code      err     = CPL_ERROR_NONE;
	int                 create  = CPL_IO_CREATE;

	TRY
	{
		if (singlefile) {
			fn = cpl_sprintf("%s.fits", filename);
		} else {
			fn = cpl_sprintf("%s_data.fits", filename);
		}
		img = hdrl_image_get_image_const(hdrl_img);
		eris_ifu_save_image_dbg(img, fn, create, pl);
		cpl_free(fn); fn = NULL;

		if (singlefile) {
			fn = cpl_sprintf("%s.fits", filename);
			create = CPL_IO_EXTEND;
		} else {
			fn = cpl_sprintf("%s_err.fits", filename);
			create = CPL_IO_CREATE;
		}
		img = hdrl_image_get_error_const(hdrl_img);
		eris_ifu_save_image_dbg(img, fn, create, pl);
		cpl_free(fn); fn = NULL;

		if (singlefile) {
			fn = cpl_sprintf("%s.fits", filename);
			create = CPL_IO_EXTEND;
		} else {
			fn = cpl_sprintf("%s_mask.fits", filename);
			create = CPL_IO_CREATE;
		}
		mask = hdrl_image_get_mask_const(hdrl_img);
		if(mask != NULL) {
		    eris_ifu_save_mask_dbg(mask, fn, create, pl);
		}
		cpl_free(fn); fn = NULL;
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_save_hdrl_image_dbg");
	return err;
}
/**
 * @brief eris_ifu_save_hdrl_imagelist_dbg
 * @param hdrl_img_list input image list
 * @param filename file name of product
 * @param singlefile if HDRL imagelist content is saved as a single file (MEF)
 * @return CPL_ERROR_NONE if OK
 */
cpl_error_code eris_ifu_save_hdrl_imagelist_dbg(
		const hdrl_imagelist *hdrl_img_list,
		const char* filename, int singlefile)
{
	char                *fn             = NULL;
	cpl_imagelist       *img_list       = NULL;
	const hdrl_image    *tmp_hdrl_img   = NULL;
	const cpl_mask      *mask           = NULL;
	const cpl_binary    *pmask          = NULL;
	cpl_image           *img            = NULL;
	cpl_error_code      err             = CPL_ERROR_NONE;
	float               *pimg           = NULL;
	int                 nx              = 0,
			ny              = 0,
			create          = CPL_IO_CREATE;

	TRY
	{
		/* save data */
		if (singlefile) {
			fn = cpl_sprintf("%s.fits", filename);
		} else {
			fn = cpl_sprintf("%s_data.fits", filename);
		}
		img_list = eris_ifu_hdrl_get_imagelist(hdrl_img_list,
				eris_hdrl_data);
		eris_ifu_save_imagelist_dbg(img_list, fn, create);
		eris_ifu_free_imagelist(&img_list);
		cpl_free(fn); fn = NULL;

		/* save error */
		if (singlefile) {
			fn = cpl_sprintf("%s.fits", filename);
			create = CPL_IO_EXTEND;
		} else {
			fn = cpl_sprintf("%s_err.fits", filename);
			create = CPL_IO_CREATE;
		}
		img_list = eris_ifu_hdrl_get_imagelist(hdrl_img_list,
				eris_hdrl_error);
		eris_ifu_save_imagelist_dbg(img_list, fn, create);
		eris_ifu_free_imagelist(&img_list);
		cpl_free(fn); fn = NULL;

		/* save bp-mask */
		if (singlefile) {
			fn = cpl_sprintf("%s.fits", filename);
			create = CPL_IO_EXTEND;
		} else {
			fn = cpl_sprintf("%s_mask.fits", filename);
			create = CPL_IO_CREATE;
		}
		img_list = cpl_imagelist_new();

		for (int i = 0; i < hdrl_imagelist_get_size(hdrl_img_list); i++) {
			tmp_hdrl_img = hdrl_imagelist_get_const(hdrl_img_list, i);
			mask = hdrl_image_get_mask_const(tmp_hdrl_img);

			nx = cpl_mask_get_size_x(mask);
			ny = cpl_mask_get_size_y(mask);

			img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
			pimg = cpl_image_get_data_float(img);
			pmask = cpl_mask_get_data_const(mask);

			for (int j = 0; j < nx*ny; j++) {
				if (pmask[j] == BAD_PIX) {
					pimg[j] = BAD_PIX;
				} else {
					pimg[j] = GOOD_PIX;
				}
			}

			cpl_imagelist_set(img_list, img, i);
		}
		eris_ifu_save_imagelist_dbg(img_list, fn, create);
		eris_ifu_free_imagelist(&img_list);
		cpl_free(fn); fn = NULL;
	}
	CATCH
	{
		CATCH_MSGS();
		err = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_save_hdrl_imagelist_dbg");
	return err;
}

///**
//  @brief Loading image data of a given category.
//  @param frameset The input set-of-frames.
//  @param category The category of the image to load. If NULL, the
//                  next frame with same keyword as accessed right before will
//                  be returned
//  @param ext      The number of the extension. 0 for primary HDU
//  @return The loaded image.
//  Data type of CPL_TYPE_FLOAT is assumed.
//*/
//hdrl_image* eris_ifu_load_image_ctg(
//        cpl_frameset    *frameset,
//        const char      *category,
//        int             ext)
//{
//    cpl_frame   *frame  = NULL;   /* must not be deleted at the end */
//    hdrl_image  *imgHdrl    = NULL;
//
//    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
//    cpl_ensure(category, CPL_ERROR_NULL_INPUT, NULL);
//
//    TRY
//    {
//        ASSURE(ext >= 0,
//                            CPL_ERROR_ILLEGAL_INPUT,
//                "ext must be 0 or larger!");
//
//        frame = cpl_frameset_find(frameset, category);
//        CHECK_ERROR_STATE();
//
//        if (frame != NULL) {
//                imgHdrl = eris_ifu_load_image(cpl_frame_get_filename(frame),
//                                          CPL_TYPE_FLOAT, 0, ext);
//        }
//    }
//    CATCH
//    {
//        CATCH_MSG();
//        eris_ifu_free_hdrl_image(&imgHdrl);
//    }
//    return imgHdrl;
//}

/**
  @brief Loading badpixel mask of a given category.
  @param frameset The input set-of-frames.
  @param category The category of the badpixelmask to load. If NULL, the
                  next frame with same keyword as accessed right before will
                  be returned
  @param ext      The number of the extension. 0 for primary HDU
  @return The loaded basdpixel mask.
 */
cpl_mask* eris_ifu_load_badpixel_mask(
		const cpl_frameset  *frameset,
		const char          *category,
		int                 ext)
{
	const cpl_frame *frame      = NULL;   /* must not be deleted at the end */
	cpl_mask        *bp_mask    = NULL;

	cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(category, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		ASSURE(ext >= 0,
				CPL_ERROR_ILLEGAL_INPUT,
				"ext must be 0 or larger!");

		frame = cpl_frameset_find_const(frameset, category);
		CHECK_ERROR_STATE();

		if (frame != NULL) {
			bp_mask = cpl_mask_load(cpl_frame_get_filename(frame), 0, ext);
		}
	}
	CATCH
	{
		CATCH_MSG();
		eris_ifu_free_mask(&bp_mask);
	}
	eris_check_error_code("eris_ifu_load_badpixel_mask");
	return bp_mask;
}

///**
//    @brief
//      Override for cpl_image_load().

//    @param filename   Name of the file to load from.
//    @param im_type       Type of the created image
//    @param pnum       Plane number in the Data Unit (0 for first)
//    @param xtnum       Extension number in the file (0 for primary HDU)

//    @return The function returns the newly created image or NULL if an
//            error occurred.

//    This is an override for cpl_image_load(), just giving a proper error
//    message if the input file isn't found and rejecting all infinite values
//    (NaN, +/-Inf). The errors returned are the same as with @c cpl_image_load
//*/
//hdrl_image* eris_ifu_load_image(const char *filename,
//                                cpl_type im_type,
//                                int pnum,
//                                int xtnum)
//{
//    hdrl_image   *imgHdrl   = NULL;
//    cpl_image    *img       = NULL;
//    float        *pimg      = NULL;
//    cpl_size     nx         = 0,
//                 ny         = 0;
//    int             ix         = 0,
//                 iy         = 0;

//    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

//    TRY
//    {
//        /* for the health of any developer:
//           check if any pre-existing error is set */
//        if (cpl_error_get_code() != CPL_ERROR_NONE) {
//            cpl_msg_error("","An already existing error has been detected. "
//                             "Aborting now.");
//            CHECK_ERROR_STATE();
//        }

//        /* load image */
//        img = cpl_image_load(filename, im_type, pnum, xtnum);
//        if (cpl_error_get_code() != CPL_ERROR_NONE) {
//            if (cpl_error_get_code() == CPL_ERROR_FILE_IO) {
//                cpl_msg_error("", "File not found: %s", filename);
//            } else {
//                cpl_msg_debug("", "Problem loading file '%s' (%s --> Code %d)",
//                              filename, cpl_error_get_message(),
//                              cpl_error_get_code());
//            }
//        }
//        CHECK_ERROR_STATE();

//        /* reject invalid values for internal bad pixel mask */
//            pimg = cpl_image_get_data(img);
//        nx = cpl_image_get_size_x(img);
//        ny = cpl_image_get_size_y(img);
//        for (iy = 0; iy < ny; iy++) {
//            for (ix = 0; ix < nx; ix++) {
//                if (!isfinite(pimg[ix+iy*nx])) {
//                    cpl_image_reject(img, ix+1, iy+1);
//                }
//            }
//        }

//        /* create noise map*/
//            imgHdrl = hdrl_image_create(img, NULL);
//    }
//    CATCH
//    {
//        eris_ifu_free_hdrl_image(&imgHdrl);
//    }

//    eris_ifu_free_image(&img);

//    return imgHdrl;
//}

/**
  @name     eris_ifu_file_exists
  @param    filename    Name of the file to look up.
  @return   true file exists, false otherwise
 */
cpl_error_code eris_ifu_file_exists(const char *filename)
{
	cpl_error_code  err = CPL_ERROR_NONE;
	FILE *fh    = NULL;

	cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);

	TRY
	{
		fh = fopen(filename, "r");
	}
	CATCH
	{
		err = cpl_error_get_code();
	}

	if(fh != NULL) {
		fclose(fh);
	}
	eris_check_error_code("eris_ifu_file_exists");
	return err;
}

/**
  @brief    Extract from a HDRL-imglist a specific CPL-imglist
  @param    hdrl_imglist  The input hdrl_imglist
  @param    type          Either data, error or badpix
  @return   The desired cpl_imagelist
 */
cpl_imagelist* eris_ifu_hdrl_get_imagelist(const hdrl_imagelist *hdrl_imglist,
		enum hdrl_t type)
{
	cpl_imagelist       *data           = NULL;
	const hdrl_image    *tmp_hdrl_img   = NULL;
	cpl_image           *tmp_img        = NULL;

	cpl_ensure(hdrl_imglist, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		data = cpl_imagelist_new();

		for (int i = 0; i < hdrl_imagelist_get_size(hdrl_imglist); i++) {
			tmp_hdrl_img = hdrl_imagelist_get(hdrl_imglist, i);

			if (type == eris_hdrl_data) {
				/* type == data */
						tmp_img = cpl_image_duplicate(
								hdrl_image_get_image_const(tmp_hdrl_img));
			} else if (type == eris_hdrl_error) {
				/* type == error */
				tmp_img = cpl_image_duplicate(
						hdrl_image_get_error_const(tmp_hdrl_img));
			} else {
				/* type == badpix */

				/* doesn't make sense yet */
				// tmp_img = cpl_image_duplicate(
				//            hdrl_image_get_mask(tmp_hdrl_img));
			}

			cpl_imagelist_set(data, tmp_img, i);
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_imagelist(&data);
		eris_ifu_free_image(&tmp_img);
	}
	eris_check_error_code("eris_ifu_hdrl_get_imagelist");
	return data;
}
/**
 * @brief eris_ifu_get_lampString
 * @param lampStatus
 * @return Returns a string indicating the lamps switched ON
 */
char* eris_ifu_get_lampString(int lampStatus)
{
	char *lampString;
	const char *arLampString = "";
	const char *krLampString = "";
	const char *neLampString = "";
	const char *xeLampString = "";

	if ((lampStatus & AR_LAMP) == AR_LAMP) {arLampString = "Ar";}
	if ((lampStatus & KR_LAMP) == KR_LAMP) {krLampString = "Kr";}
	if ((lampStatus & NE_LAMP) == NE_LAMP) {neLampString = "Ne";}
	if ((lampStatus & XE_LAMP) == XE_LAMP) {xeLampString = "Xe";}
	lampString = cpl_sprintf("%s%s%s%s",
			arLampString, krLampString, neLampString, xeLampString);
	eris_check_error_code("eris_ifu_get_lampString");
	return lampString;
}

/**
 * @brief eris_ifu_get_bandString
 * @param band
 * @return Returns the band-setting as string
 */
const char* eris_ifu_get_bandString(ifsBand band)
{
	const char *bandString;
	switch (band) {
	case UNDEFINED_BAND:
		bandString = "";
		break;
	case J_LOW:
		bandString = "J_LOW";
		break;
	case H_LOW:
		bandString = "H_LOW";
		break;
	case K_LOW:
		bandString = "K_LOW";
		break;
	case J_SHORT:
		bandString = "J_SHORT";
		break;
	case J_MIDDLE:
		bandString = "J_MIDDLE";
		break;
	case J_LONG:
		bandString = "J_LONG";
		break;
	case H_SHORT:
		bandString = "H_SHORT";
		break;
	case H_MIDDLE:
		bandString = "H_MIDDLE";
		break;
	case H_LONG:
		bandString = "H_LONG";
		break;
	case K_SHORT:
		bandString = "K_SHORT";
		break;
	case K_MIDDLE:
		bandString = "K_MIDDLE";
		break;
	case K_LONG:
		bandString = "K_LONG";
		break;
	case J_SPIFFI:
		bandString = "J_SPIFFI";
		break;
	case H_SPIFFI:
		bandString = "H_SPIFFI";
		break;
	case K_SPIFFI:
		bandString = "K_SPIFFI";
		break;
	case HK_SPIFFI:
		bandString = "HK_SPIFFI";
		break;
	default:
		bandString = "Unknown";
		break;
	}
	eris_check_error_code("eris_ifu_get_bandString");
	return bandString;
}
/**
 * @brief eris_ifu_get_band_resolution
 * @param band
 * @return Returns the nominal resolution corresponding to a given band
 */
double eris_ifu_get_band_resolution(ifsBand band)
{
	double resolution;
	switch (band) {
	case UNDEFINED_BAND:   resolution =     0; break;
	case J_LOW:            resolution =  5000; break;
	case H_LOW:            resolution =  5200; break;
	case K_LOW:            resolution =  5600; break;
	case J_SHORT:          resolution = 10000; break;
	case J_MIDDLE:         resolution = 10000; break;
	case J_LONG:           resolution = 10000; break;
	case H_SHORT:          resolution = 10400; break;
	case H_MIDDLE:         resolution = 10400; break;
	case H_LONG:           resolution = 10400; break;
	case K_SHORT:          resolution = 11200; break;
	case K_MIDDLE:         resolution = 11200; break;
	case K_LONG:           resolution = 11200; break;
	case J_SPIFFI:         resolution =  5000; break;
	case H_SPIFFI:         resolution =  5000; break;
	case K_SPIFFI:         resolution =  5000; break;
	case HK_SPIFFI:        resolution =  3000; break;
	default:               resolution =     0; break;
	}
	eris_check_error_code("eris_ifu_get_band_resolution");
	return resolution;
}


/**
 * @brief eris_ifu_get_plane_cut_min_max
 * @param band
 * @param zmin output plane that still has more than half pixels with NANs
 * @param zmax output plane that still has more than half pixels with NANs
 * @return Returns the min/max values corresponding to NAN for a given band
 */
cpl_error_code
eris_ifu_get_plane_cut_min_max(ifsBand band, cpl_size* zmin, cpl_size* zmax)
{

	cpl_ensure_code(zmin >= 0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(zmin <= zmax, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(zmin != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(zmax != NULL, CPL_ERROR_NULL_INPUT);

	switch (band) {
	case UNDEFINED_BAND: break;
	case J_LOW:           *zmin = 46;  *zmax = 2015; break;
	case H_LOW:           *zmin = 48;  *zmax = 2023; break;
	case K_LOW:           *zmin = 62;  *zmax = 2022; break;
	case J_SHORT:         *zmin = 50;  *zmax = 2034; break;
	case J_MIDDLE:        *zmin = 57;  *zmax = 2040; break;
	case J_LONG:          *zmin = 48;  *zmax = 2033; break;
	case H_SHORT:         *zmin = 52;  *zmax = 2034; break;//
	case H_MIDDLE:        *zmin = 47;  *zmax = 2034; break;
	case H_LONG:          *zmin = 56;  *zmax = 2042; break;//
	case K_SHORT:         *zmin = 54;  *zmax = 2036; break;
	case K_MIDDLE:        *zmin = 49;  *zmax = 2029; break;
	case K_LONG:          *zmin = 50;  *zmax = 2034; break;
	case J_SPIFFI:        *zmin = 32;  *zmax = 2058; break;
	case H_SPIFFI:        *zmin = 32;  *zmax = 2058; break;
	case K_SPIFFI:        *zmin = 32;  *zmax = 2058; break;
	case HK_SPIFFI:       *zmin = 32;  *zmax = 2058; break;
	default:              break;
	}
	eris_check_error_code("eris_ifu_get_plane_cut_min_max");
	return cpl_error_get_code();
}



/**
 * @brief remove planes from input iml and bpm in the range [0, zmin), (zmax,sz]
 * @param zmin  planes in range [0,zmin] are removed
 * @param zmax  planes in range [zmax,sz] are removed
 * @param iml   input imagelist
 * @param bpm   input bad pixel map
 * @param header input FITS header
 * @return cpl_error_code
 */
//TODO: Why hvalue.data = k * 10.;  ?
cpl_error_code
eris_ifu_cube_set_values(hdrl_imagelist** iml/*, cpl_imagelist** bpm*/) {


	cpl_size size = hdrl_imagelist_get_size(*iml);
	cpl_msg_info(cpl_func,"org size: %lld", size);
	hdrl_image* hima = NULL;
//	cpl_image* ima = NULL;
    hdrl_value hvalue = {0, 0};
	for(cpl_size k = 0; k < size; k++) {
		hima =  hdrl_imagelist_get(*iml, k);
		for(cpl_size j = 10;  j < 20; j++) {
			for(cpl_size i = 10;  i < 20; i++) {
				hvalue.data = k * 10.;                         //TODO: why this?
				hdrl_image_set_pixel(hima, i, j, hvalue);
			}
		}
		//hdrl_imagelist_unset(*iml, k);
		//hdrl_imagelist_set(*iml, k, hima);
		//cpl_imagelist_unset(*bpm, k);
	}

	eris_check_error_code("eris_ifu_cube_set_values");

	return cpl_error_get_code();
}

/**
 * @brief remove planes from input iml and bpm in the range [0, zmin), (zmax,sz]
 * @param zmin  planes in range [0,zmin] are removed
 * @param zmax  planes in range [zmax,sz] are removed
 * @param iml   input imagelist
 * @param bpm   input bad pixel map
 * @param header input FITS header
 * @return cpl_error_code
 */
cpl_error_code
eris_ifu_cube_trim_nans(const cpl_size zmin, const cpl_size zmax,
		hdrl_imagelist** iml, cpl_imagelist** bpm, cpl_propertylist* header) {

    cpl_ensure_code(iml,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(bpm,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(header,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(zmin <= zmax,CPL_ERROR_ILLEGAL_INPUT);
	cpl_size size = hdrl_imagelist_get_size(*iml);
	cpl_msg_debug(cpl_func,"org size: %lld", size);
	/* we peel-off the cube starting from the end thus the index we touch
	 * is always decreasing (maximum)
	 */
	for(cpl_size i = size-1; i > zmax; i--) {
		hdrl_imagelist_unset(*iml, i);
		cpl_imagelist_unset(*bpm, i);
	}
	cpl_msg_debug(cpl_func,"size1: %lld", hdrl_imagelist_get_size(*iml));
	/* we peel-off the cube starting from the bottom thus the index we touch
	 * is always 0 (minimum)
	 */
	for(cpl_size i = 0; i < zmin; i++) {
		hdrl_imagelist_unset(*iml, 0);
		cpl_imagelist_unset(*bpm, 0);
	}
	cpl_msg_debug(cpl_func,"size2: %lld", hdrl_imagelist_get_size(*iml));
	double crval3 = cpl_propertylist_get_double(header,"CRVAL3");
	const double cd3_3 = cpl_propertylist_get_double(header,"CD3_3");
	int naxis3 = cpl_propertylist_get_int(header,"NAXIS3");

	crval3 += cd3_3 * zmin;
	naxis3 = zmax - zmin +1;
	cpl_propertylist_set_double(header, "CRVAL3", crval3);
	cpl_propertylist_set_int(header, "NAXIS3", naxis3);

	eris_check_error_code("eris_ifu_cube_trim_nans");
	return cpl_error_get_code();
}


/**
 * @brief eris_ifu_get_instrumentString
 * @param instrument
 * @return Returns the instrument-setting as string
 */
const char* eris_ifu_get_instrumentString(ifsInstrument instrument)
{
	const char *instrumentString;
	switch (instrument) {
	case OTHER_INSTRUMENT:  instrumentString = "unknown instrument"; break;
	case SPIFFI:            instrumentString = "SPIFFI"; break;
	case SPIFFIER:          instrumentString = "SPIFFIER"; break;
	case NIX:               instrumentString = "NIX"; break;
	case UNSET_INSTRUMENT:  instrumentString = "unset instrument"; break;
	default:                instrumentString = "unset instrument"; break;
	}
	eris_check_error_code("eris_ifu_get_instrumentString");
	return instrumentString;
}

/**
 * @brief eris_ifu_get_preopticsScaleString
 * @param scale
 * @return Returns the scale-setting as string
 */
const char* eris_ifu_get_preopticsScaleString(ifsPreopticsScale scale)
{
	const char *scaleString;
	switch (scale) {
	case UNDEFINED_SCALE:
		scaleString = "";
		break;
	case S250MAS:
		scaleString = "250mas";
		break;
	case S100MAS:
		scaleString = "100mas";
		break;
	case S25MAS:
		scaleString = "25mas";
		break;
	case PUPIL:
		scaleString = "PUPIL";
		break;
	default:
		scaleString = "Unknown";
		break;
	}
	eris_check_error_code("eris_ifu_get_preopticsScaleString");
	return scaleString;
}

/*
 * removes badpixborder around img
 * @param img input image
 * @return sub-image with removed border
 */
cpl_image* eris_ifu_image_create_window(const cpl_image* img)
{
	cpl_size            nx              = 0,
			ny              = 0;
	double              *pimg_window    = NULL;
	const double        *pimg           = NULL;
	cpl_image           *img_window     = NULL;
	const cpl_mask      *mask           = NULL;
	cpl_mask            *mask_window    = NULL;
	const cpl_binary    *pmask          = NULL;
	cpl_binary          *pmask_window   = NULL;

	cpl_ensure(img, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		nx = cpl_image_get_size_x(img);
		ny = cpl_image_get_size_y(img);
		CHECK_ERROR_STATE();
		img_window = cpl_image_new(nx-2*ERIS_IFU_DETECTOR_BP_BORDER,
				ny-2*ERIS_IFU_DETECTOR_BP_BORDER,
				cpl_image_get_type(img));

		pimg = cpl_image_get_data_const(img);
		mask = cpl_image_get_bpm_const(img);
		pmask = cpl_mask_get_data_const(mask);
		pimg_window = cpl_image_get_data(img_window);
		mask_window = cpl_image_get_bpm(img_window);
		pmask_window = cpl_mask_get_data(mask_window);

		for (int x = 0; x < nx-2*ERIS_IFU_DETECTOR_BP_BORDER; x++) {
			for (int y = 0; y < ny-2*ERIS_IFU_DETECTOR_BP_BORDER; y++) {
				cpl_size nx_new = nx-2*ERIS_IFU_DETECTOR_BP_BORDER,
						x_old  = x+ERIS_IFU_DETECTOR_BP_BORDER,
						y_old  = y+ERIS_IFU_DETECTOR_BP_BORDER;
				pimg_window[x+y*nx_new] = pimg[x_old+y_old*nx];
				pmask_window[x+y*nx_new] = pmask[x_old+y_old*nx];
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_image(&img_window);
	}
	eris_check_error_code("eris_ifu_image_create_window");
	return img_window;
}

/**
 * @brief eris_ifu_mask_create_border
 * @param mask
 * @return
 *
 * Add badpixborder around img
 */
cpl_mask* eris_ifu_mask_create_border(const cpl_mask* mask)
{
	cpl_size            nx              = 0,
			ny              = 0;
	cpl_binary          *pmask_border    = NULL;
	const cpl_binary    *pmask           = NULL;
	cpl_mask            *mask_border     = NULL;

	cpl_ensure(mask, CPL_ERROR_NULL_INPUT, NULL);

	TRY
	{
		nx = cpl_mask_get_size_x(mask);
		ny = cpl_mask_get_size_y(mask);
		CHECK_ERROR_STATE();

		mask_border = cpl_mask_new(nx+2*ERIS_IFU_DETECTOR_BP_BORDER,
				ny+2*ERIS_IFU_DETECTOR_BP_BORDER);

		pmask = cpl_mask_get_data_const(mask);
		pmask_border = cpl_mask_get_data(mask_border);

		for (int x = 0; x < nx; x++) {
			for (int y = 0; y < ny; y++) {
				cpl_size x_new = x+ERIS_IFU_DETECTOR_BP_BORDER,
						y_new = y+ERIS_IFU_DETECTOR_BP_BORDER,
						nx_new = nx+2*ERIS_IFU_DETECTOR_BP_BORDER;
				pmask_border[x_new+y_new*nx_new] = pmask[x+y*nx];
			}
		}
	}
	CATCH
	{
		CATCH_MSGS();
		eris_ifu_free_mask(&mask_border);
	}
	eris_check_error_code("eris_ifu_mask_create_border");
	return mask_border;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Flag NANs in image
  @param hima input image
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_ifu_mask_nans_in_hdrlimage(hdrl_image** hima){

	double* pdata = NULL;
	double* perrs = NULL;
	cpl_binary*    pbpm = NULL;
	cpl_size sx = 0;
	cpl_size sy = 0;
	cpl_size npoints = 0;
	cpl_ensure(hima != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);


	sx = hdrl_image_get_size_x( *hima );
	sy = hdrl_image_get_size_y( *hima );
	npoints = sx * sy;

	pdata = cpl_image_get_data(hdrl_image_get_image(*hima));
	perrs = cpl_image_get_data(hdrl_image_get_error(*hima));
	pbpm = cpl_mask_get_data(hdrl_image_get_mask(*hima));

	cpl_size nan_pix = 0;
	for(cpl_size i = 0; i < npoints; i++) {
		//cpl_msg_error(cpl_func,"pdata[%lld]:%g perrs[%lld]:%g",i,pdata[i],i,perrs[i]);
		//if(pdata[i] == CPL_VALUE_NOTFINITE || perrs[i] == CPL_VALUE_NOTFINITE ){
		if(isnan(pdata[i]) || isnan(perrs[i]) ){
			//cpl_msg_info(cpl_func,"found NAN at %lld",i);
            pbpm[i] = BAD_PIX;
			nan_pix++;
		}
	}

	eris_check_error_code("eris_ifu_mask_nans_in_hdrlimage");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   mask NANs in input data cube
 * @param   cube input imagelist
 * @return  CPL_ERROR_NONE if OK
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_mask_nans_in_cube(cpl_imagelist *cube)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_size        nx = 0,
			ny = 0,
			nz = 0,
			idx = 0;
	cpl_image       *tmpImg = NULL;
	cpl_mask        *mask = NULL;
	cpl_binary      *maskData = NULL;
	const double    *imgDataD = NULL;
	const float     *imgDataF = NULL;
	cpl_type        imgType = CPL_TYPE_INVALID;

	cpl_ensure(cube, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	TRY
	{
		nz = cpl_imagelist_get_size(cube);
		if (nz > 0) {
			tmpImg = cpl_imagelist_get(cube,0);
			imgType = cpl_image_get_type(tmpImg);
			if (! (imgType == CPL_TYPE_FLOAT || imgType == CPL_TYPE_DOUBLE)) {
				BRK_WITH_ERROR_MSG(CPL_ERROR_INVALID_TYPE,
						"unsupported image type within cube");
			}
			nx = cpl_image_get_size_x(tmpImg);
			ny = cpl_image_get_size_y(tmpImg);
		}
		CHECK_ERROR_STATE();
		for (cpl_size z=0; z < nz; z++) {
			tmpImg = cpl_imagelist_get(cube,z);
			switch (imgType) {
			case CPL_TYPE_FLOAT:
				imgDataF = cpl_image_get_data_float_const(tmpImg);
				break;
			case CPL_TYPE_DOUBLE:
				imgDataD = cpl_image_get_data_double_const(tmpImg);
				break;
			default: break;
			}
			mask = cpl_image_get_bpm(tmpImg);
			maskData = cpl_mask_get_data(mask);
			for (cpl_size y=0; y < ny; y++) {
				for (cpl_size x=0; x < nx; x++) {
					idx = y * nx + x;
					switch (imgType) {
					case CPL_TYPE_FLOAT:
                        if (!isfinite(imgDataF[idx])) {
                            maskData[idx] = BAD_PIX;
						}
						break;
					case CPL_TYPE_DOUBLE:
                        if (!isfinite(imgDataD[idx])) {
                            maskData[idx] = BAD_PIX;
						}
						break;
					default: break;
					}
				}
			}
			CHECK_ERROR_STATE();

		}
	}
	CATCH {
		retVal = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_mask_nans_in_cube");
	return retVal;
}

// /**
// * @brief eris_ifu_mask_reset
// * @param mask
// * @return
// */
//cpl_error_code eris_ifu_mask_reset(cpl_mask* mask)
//{
//    cpl_error_code  err = CPL_ERROR_NONE;
//    cpl_size        nx  = 0,
//                    ny  = 0;
//
//    cpl_ensure_code(mask, CPL_ERROR_NULL_INPUT);
//
//    TRY
//    {
//        nx = cpl_mask_get_size_x(mask);
//        ny = cpl_mask_get_size_y(mask);
//        CHECK_ERROR_STATE();
//
//        for (int x = 1; x <= nx; x++) {
//            for (int y = 1; y <= ny; y++) {
//                    cpl_mask_set(mask, x, y, GOOD_PIX);
//            }
//        }
//    }
//    CATCH
//    {
//        CATCH_MSGS();
//        err = cpl_error_get_code();
//    }
//    return err;
//}
/*----------------------------------------------------------------------------*/
/* TODO: document bit code. Why bit code is double and not integer? */
/**
 * @brief   transform input image (quality mask) to output mask with given bit code
 * @param   qualityMask input imagelist
 * @param   qualityType BIT code
 * @return  mask of given type.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_mask* eris_ifu_quality2bp_mask(const cpl_image *qualityMask,
		deqQualityType qualityType)
{
	cpl_mask *bpmMask = NULL;
	TRY
	{
		ASSURE((qualityMask != NULL), CPL_ERROR_NULL_INPUT, NULL);
				bpmMask = cpl_mask_threshold_image_create(
						qualityMask, .5, 4294967296.5 );
		switch(qualityType) {
		case maskzero:
			cpl_mask_not(bpmMask);
			break;
		case maskone:
			break;
		case flag32bit:
			break;
		case flag16bit:
			break;
		case unspecified:
			cpl_mask_delete(bpmMask);
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Unspecified DEQ quality type");

			break;
		default:
			BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
					"Unknown DEQ quality type");
		}
	}
	CATCH {
		bpmMask = NULL;
	}
	eris_check_error_code("eris_ifu_quality2bp_mask");
	return bpmMask;

}
/*----------------------------------------------------------------------------*/
/**
 * @brief   return cube flagging (as 1) pixels above a given threshold
 * @param   bpmCube input imagelist
 * @param   threshold threshold
 * @return  cube with detected threshold pixels
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist* eris_ifu_interpolatedMask_to_maskZero(cpl_imagelist *bpmCube,
		double threshold)
{
	cpl_imagelist   *maskOneCube = NULL;
	cpl_image *img = NULL;
	cpl_image       *maskOneImg = NULL;
	int             *maskOneImgPtr = NULL;
	float           *floatPtr = NULL;
	double          *doublePtr = NULL;
	float           floatThreshold;
	cpl_size imgPixelCnt = 0,
			nx = 0,
			ny = 0;
	cpl_type bpmType;

	TRY
	{
		ASSURE(bpmCube != NULL, CPL_ERROR_NULL_INPUT, "bpmCube must not be NULL");

		maskOneCube = cpl_imagelist_new();

		img = cpl_imagelist_get(bpmCube, 0);
		bpmType = cpl_image_get_type(img);
		ASSURE(bpmType == CPL_TYPE_FLOAT || bpmType == CPL_TYPE_DOUBLE,
				CPL_ERROR_ILLEGAL_INPUT, "bpmCube must be of type float or double");

		nx = cpl_image_get_size_x(img);
		ny = cpl_image_get_size_y(img);
		imgPixelCnt = nx * ny;

		floatThreshold = (float) threshold;

		for (cpl_size cx=0; cx < cpl_imagelist_get_size(bpmCube); cx++) {
			img = cpl_imagelist_get(bpmCube, cx);

			maskOneImg = cpl_image_new(nx, ny, CPL_TYPE_INT);
			maskOneImgPtr = cpl_image_get_data_int(maskOneImg);

			if (bpmType == CPL_TYPE_FLOAT) {
				floatPtr = cpl_image_get_data_float(img);
				for (cpl_size px=0; px < imgPixelCnt; px++) {
					if (floatPtr[px] > floatThreshold) {
						maskOneImgPtr[px] = 1;
					} else {
						maskOneImgPtr[px] = 0;
					}
				}
			} else if (bpmType == CPL_TYPE_DOUBLE) {
				doublePtr = cpl_image_get_data_double(img);
				for (cpl_size px=0; px < imgPixelCnt; px++) {
					if (doublePtr[px] > threshold) {
						maskOneImgPtr[px] = 1;
					} else {
						maskOneImgPtr[px] = 0;
					}
				}

			}

			cpl_imagelist_set(maskOneCube, maskOneImg, cx);
		}
	}
	CATCH {
		maskOneCube = NULL;
	}
	eris_check_error_code("eris_ifu_interpolatedMask_to_maskZero");
	return maskOneCube;

}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   extract from hdrl_imagelist the data and error information
 * @param   cube input imagelist
 * @param   dataCube output data
 * @param   errorCube output errs
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_split_hdrl_imagelist(hdrl_imagelist *cube,
		cpl_imagelist *dataCube,
		cpl_imagelist *errorCube)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	hdrl_image *image = NULL;
	TRY
	{

		cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
		cpl_ensure(dataCube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
		cpl_ensure(errorCube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
		for (cpl_size ix=0; ix < hdrl_imagelist_get_size(cube); ix++) {
			image = hdrl_imagelist_get(cube, ix);
			cpl_imagelist_set(dataCube, cpl_image_duplicate(hdrl_image_get_image(image)), ix);
			cpl_imagelist_set(errorCube, cpl_image_duplicate(hdrl_image_get_error(image)), ix);

		}
	}
	CATCH {
	}
	eris_check_error_code("eris_ifu_split_hdrl_imagelist");
	return retVal;

}

/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   extract from hdrl_imagelist the data and error information
 * @param   cube input imagelist
 * @param   dataCube output data
 * @param   errorCube output errs
 * @param   qualCube output  quality
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_split3_hdrl_imagelist(hdrl_imagelist *cube,
		cpl_imagelist *dataCube,
		cpl_imagelist *errorCube,
		cpl_imagelist *qualCube)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	hdrl_image *image = NULL;
	TRY
	{
		cpl_ensure(cube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
		cpl_ensure(dataCube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
		cpl_ensure(errorCube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
		cpl_ensure(qualCube != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

		for (cpl_size ix=0; ix < hdrl_imagelist_get_size(cube); ix++) {
			image = hdrl_imagelist_get(cube, ix);
			cpl_imagelist_set(dataCube,
					hdrl_image_get_image(image), ix);
			cpl_imagelist_set(errorCube,
					hdrl_image_get_error(image), ix);
			cpl_imagelist_set(qualCube,
					cpl_image_new_from_mask(hdrl_image_get_mask(image)), ix);

		}
	}
	CATCH {
	}
	eris_check_error_code("eris_ifu_split3_hdrl_imagelist");
	return retVal;

}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   Initialize QC table
 * @return   cpl_table
 *
 */
/*----------------------------------------------------------------------------*/
cpl_table *
eris_qclog_init(void)
{

	cpl_table *table;

	table = cpl_table_new(0);
	cpl_table_new_column(table,"key_name", CPL_TYPE_STRING);
	cpl_table_new_column(table,"key_type", CPL_TYPE_STRING);
	cpl_table_new_column(table,"key_value", CPL_TYPE_STRING);
	cpl_table_new_column(table,"key_help", CPL_TYPE_STRING);
	eris_check_error_code("eris_qclog_init");
	return table;
}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   add QC int info to table
 * @param   table input table
 * @param   key_name keyword name
 * @param   value keyword value
 * @param   key_help keyword help info
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_qclog_add_int(cpl_table* table,
		const char*  key_name,
		const int    value,
		const char*  key_help)
{

	cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_name != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_help != NULL, CPL_ERROR_NULL_INPUT);

	int sz = cpl_table_get_nrow(table);
	int raw = sz;
	char* key_value;
	char* key_type;

	key_value = cpl_sprintf("%d",value);
	key_type = cpl_sprintf("%s","CPL_TYPE_INT");

	cpl_table_set_size(table,sz+1);

	cpl_table_set_string(table,"key_name" ,raw,key_name);
	cpl_table_set_string(table,"key_type" ,raw,key_type);
	cpl_table_set_string(table,"key_value",raw,key_value);
	cpl_table_set_string(table,"key_help" ,raw,key_help);

	cpl_free(key_value);
	cpl_free(key_type);
	eris_check_error_code("eris_qclog_add_int");
	return cpl_error_get_code();

}

/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   add QC boolean info to table
 * @param   table input table
 * @param   key_name keyword name
 * @param   key_help keyword help info
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/

cpl_error_code
eris_qclog_add_bool(cpl_table* table,
		const char*  key_name,
		const char*  key_help)
{
	cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_name != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_help != NULL, CPL_ERROR_NULL_INPUT);

	int sz = cpl_table_get_nrow(table);
	int raw = sz;
	char* key_value;
	char* key_type;

	key_value = cpl_sprintf("%s",key_name);
	key_type = cpl_sprintf("%s","CPL_TYPE_BOOL");

	cpl_table_set_size(table,sz+1);

	cpl_table_set_string(table,"key_name" ,raw,key_name);
	cpl_table_set_string(table,"key_type" ,raw,key_type);
	cpl_table_set_string(table,"key_value",raw,key_value);
	cpl_table_set_string(table,"key_help" ,raw,key_help);
	cpl_free(key_value);
	cpl_free(key_type);
	eris_check_error_code("eris_qclog_add_bool");
	return cpl_error_get_code();

}

/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   add QC double info to table
 * @param   table input table
 * @param   key_name keyword name
 * @param   value keyword value
 * @param   key_help keyword help info
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_qclog_add_double(cpl_table* table,
		const char*  key_name,
		const double value,
		const char*  key_help)
{
	cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_name != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_help != NULL, CPL_ERROR_NULL_INPUT);

	int sz = cpl_table_get_nrow(table);

	int raw = sz;
	char* key_value;
	char* key_type;

	key_value = cpl_sprintf("%g",value);

	key_type = cpl_sprintf("%s","CPL_TYPE_DOUBLE");

	cpl_table_set_size(table,sz+1);

	cpl_table_set_string(table,"key_name" ,raw,key_name);
	cpl_table_set_string(table,"key_type" ,raw,key_type);
	cpl_table_set_string(table,"key_value",raw,key_value);
	cpl_table_set_string(table,"key_help" ,raw,key_help);

	cpl_free(key_value);
	cpl_free(key_type);
	eris_check_error_code("eris_qclog_add_double");
	return cpl_error_get_code();

}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   add QC float info to table
 * @param   table input table
 * @param   key_name keyword name
 * @param   value keyword value
 * @param   key_help keyword help info
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_qclog_add_double_f(cpl_table* table,
		const char*  key_name,
		const double value,
		const char*  key_help)
{

	cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_name != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_help != NULL, CPL_ERROR_NULL_INPUT);

	int sz = cpl_table_get_nrow(table);
	int raw = sz;
	char* key_value;
	char* key_type;

	key_value = cpl_sprintf("%f",value);
	key_type = cpl_sprintf("%s","CPL_TYPE_DOUBLE");

	cpl_table_set_size(table,sz+1);

	cpl_table_set_string(table,"key_name" ,raw,key_name);
	cpl_table_set_string(table,"key_type" ,raw,key_type);
	cpl_table_set_string(table,"key_value",raw,key_value);
	cpl_table_set_string(table,"key_help" ,raw,key_help);
	cpl_free(key_value);
	cpl_free(key_type);
	eris_check_error_code("eris_qclog_add_double_f");
	return cpl_error_get_code();

}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   add QC double (with format) info to table
 * @param   table input table
 * @param   key_name keyword name
 * @param   value keyword value
 * @param   key_help keyword help info
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_qclog_add_double_format(cpl_table* table,
		const char*  key_name,
		const double value,
		const char*  key_help)
{

	cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_name != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_help != NULL, CPL_ERROR_NULL_INPUT);

	int sz = cpl_table_get_nrow(table);
	int raw = sz;
	char* key_value;
	char* key_type;

	key_value = cpl_sprintf("%13.6f",value);
	key_type = cpl_sprintf("%s","CPL_TYPE_DOUBLE");

	cpl_table_set_size(table,sz+1);

	cpl_table_set_string(table,"key_name" ,raw,key_name);
	cpl_table_set_string(table,"key_type" ,raw,key_type);
	cpl_table_set_string(table,"key_value",raw,key_value);
	cpl_table_set_string(table,"key_help" ,raw,key_help);
	cpl_free(key_value);
	cpl_free(key_type);
	eris_check_error_code("eris_qclog_add_double_format");
	return cpl_error_get_code();

}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   add QC string info to table
 * @param   table input table
 * @param   key_name keyword name
 * @param   value keyword value
 * @param   key_help keyword help info
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_qclog_add_string(cpl_table* table,
		const char*  key_name,
		const char*  value,
		const char*  key_help)
{

	cpl_ensure_code(table != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_name != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(key_help != NULL, CPL_ERROR_NULL_INPUT);

	int sz = cpl_table_get_nrow(table);
	int raw = sz;
	char* key_value;
	char* key_type;

	key_value = cpl_sprintf("%s",value);
	key_type = cpl_sprintf("%s","CPL_TYPE_STRING");

	cpl_table_set_size(table,sz+1);

	cpl_table_set_string(table,"key_name" ,raw,key_name);
	cpl_table_set_string(table,"key_type" ,raw,key_type);
	cpl_table_set_string(table,"key_value",raw,key_value);
	cpl_table_set_string(table,"key_help" ,raw,key_help);
	cpl_free(key_value);
	cpl_free(key_type);
	eris_check_error_code("eris_qclog_add_string");
	return cpl_error_get_code();

}

/*----------------------------------------------------------------------------*/
/**
 *
 * @brief   convert table with QC parameter information to a propertylist
 * @param    plist  output property list
 * @param    qclog  input qclog table
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_pfits_put_qc(
		cpl_propertylist       *   plist,
		cpl_table          *   qclog)
{


	cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qclog != NULL, CPL_ERROR_NULL_INPUT);
	char* key_name = NULL;
	char* key_value = NULL;
	char* key_type = NULL;
	char* key_help = NULL;
	/*
    char            key_name[FILE_NAME_SZ];
    char            key_value[FILE_NAME_SZ];
    char            key_type[FILE_NAME_SZ];
    char            key_help[FILE_NAME_SZ] ;
	 */

	int i = 0;
	int n = 0;
	/* Test entries */
	if (plist == NULL) {
		cpl_msg_error(cpl_func,"plist=NULL, something strange");
		return CPL_ERROR_NULL_INPUT ;
	}
	/* Parameter Name:    PIPEFILE */

	//cpl_table_save(qclog,NULL,NULL,"qclog.fits",CPL_IO_CREATE);
	n = cpl_table_get_nrow(qclog);
	for(i = 0; i < n; i++) {

		key_name = cpl_sprintf("ESO %s",cpl_table_get_string(qclog, "key_name", i));
		//strcat(key_name, cpl_table_get_string(qclog, "key_name", i));
		key_type = cpl_sprintf("%s", cpl_table_get_string(qclog, "key_type", i));
		//strcpy(key_type, cpl_table_get_string(qclog, "key_type", i));
		key_value = cpl_sprintf("%s",  cpl_table_get_string(qclog, "key_value", i));
		//strcpy(key_value, cpl_table_get_string(qclog, "key_value", i));
		key_help = cpl_sprintf("%s",  cpl_table_get_string(qclog, "key_help", i));
		//strcpy(key_help, cpl_table_get_string(qclog, "key_help", i));
		/*
        cpl_msg_info(cpl_func,"raw %i, name=%s type=%s value=%s\n",
                i,key_name,key_type,key_value);
		 */
		if( !cpl_propertylist_has(plist, key_name) ) {
			if(strcmp(key_type, "CPL_TYPE_STRING") == 0) {
				cpl_propertylist_append_string(plist, key_name, key_value) ;
				cpl_propertylist_set_comment(plist, key_name, key_help) ;
			} else if(strcmp(key_type, "CPL_TYPE_BOOL") == 0) {
				cpl_propertylist_append_bool(plist, key_name, atoi(key_value)) ;
				cpl_propertylist_set_comment(plist, key_name, key_help) ;
			} else if(strcmp(key_type, "CPL_TYPE_INT") == 0) {
				cpl_propertylist_append_int(plist, key_name, atoi(key_value)) ;
				cpl_propertylist_set_comment(plist, key_name, key_help) ;
			} else if(strcmp(key_type, "CPL_TYPE_FLOAT") == 0) {
				cpl_propertylist_append_float(plist, key_name, (float) atof(key_value)) ;
				cpl_propertylist_set_comment(plist, key_name, key_help) ;
			} else if(strcmp(key_type, "CPL_TYPE_DOUBLE") == 0) {
				cpl_propertylist_append_double(plist, key_name, atof(key_value)) ;
				cpl_propertylist_set_comment(plist, key_name, key_help) ;
			}
		}
		cpl_free(key_name);
		cpl_free(key_type);
		cpl_free(key_value);
		cpl_free(key_help);

	}
	eris_check_error_code("eris_pfits_put_qc");
	return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 *
 * @brief    compute QC keyword with number of bad pixels and fraction to total
 * @param    image  input image (a bad pixel map, intensity [0,1]
 * @param    qc_list  output property list to add QC keywords
 * @param    prefix   prefix of QC keyword
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_ifu_get_badpix_qc_from_ima(const cpl_image* image,
		cpl_propertylist* qc_list,
		const char* prefix)
{

	cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qc_list != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(prefix != NULL, CPL_ERROR_NULL_INPUT);
	cpl_size  sx = cpl_image_get_size_x(image);
	cpl_size  sy = cpl_image_get_size_y(image);
	cpl_image* ima = cpl_image_duplicate(image);
	cpl_image_threshold(ima, 0, 0, 0, 1);
	double flux = cpl_image_get_flux(ima);
	cpl_image_delete(ima);
	cpl_size  nBadPix = (cpl_size) flux;

	cpl_size  npix = sx *  sy;
	double  fracBadPix = flux / npix;
	char* kname;
	cpl_msg_warning(cpl_func,"nBadPix: %lld",nBadPix);
	kname = cpl_sprintf("%s NBADPIX",prefix);
	cpl_msg_info(cpl_func,"kname: %s prefix: %s",kname, prefix);
	eris_ifu_append_qc_int(qc_list, kname, nBadPix, "Number of bad pixels");
	cpl_free(kname);

	kname = cpl_sprintf("%s BPIXFRAC",prefix);
	eris_ifu_append_qc_double(qc_list, kname, fracBadPix,
			"Fraction of bad pixels to total");
	cpl_free(kname);
	eris_check_error_code("eris_ifu_get_badpix_qc_from_ima");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 *
 * @brief    compute QC keyword with number of bad pixels and fraction to total
 * @param    bp_mask  input bad pixel mask
 * @param    qc_list  output property list to add QC keywords
 * @param    prefix   prefix of QC keyword
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_ifu_get_badpix_qc_from_mask(const cpl_mask* bp_mask,
		cpl_propertylist* qc_list,
		const char* prefix)
{
	cpl_ensure_code(bp_mask != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qc_list != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(prefix != NULL, CPL_ERROR_NULL_INPUT);
	cpl_size  sx = cpl_mask_get_size_x(bp_mask);
	cpl_size  sy = cpl_mask_get_size_y(bp_mask);
	cpl_size  nBadPix = cpl_mask_count(bp_mask);

	cpl_size  npix = sx *  sy;
	double  fracBadPix = (double) nBadPix / npix;
	char* kname;

	kname = cpl_sprintf("%s NBADPIX",prefix);
	cpl_msg_info(cpl_func,"kname: %s prefix: %s",kname, prefix);
	eris_ifu_append_qc_int(qc_list, kname, nBadPix, "Number of bad pixels");
	cpl_free(kname);

	kname = cpl_sprintf("%s BPIXFRAC",prefix);
	eris_ifu_append_qc_double(qc_list, kname, fracBadPix,
			"Fraction of bad pixels to total");
	cpl_free(kname);
	eris_check_error_code("eris_ifu_get_badpix_qc_from_mask");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief    Find the aperture(s) with the greatest flux
 * @param    self   The aperture object
 * @param    ind  The aperture-indices in order of decreasing flux
 * @param    nfind  Number of indices to find
 * @return   CPL_ERROR_NONE or the relevant _cpl_error_code_ on error
 *
 * nfind must be at least 1 and at most the size of the aperture object.
 *
 * The ind array must be able to hold (at least) nfind integers.
 * On success the first nfind elements of ind point to indices of the
 * aperture object.
 *
 * To find the single ind of the aperture with the maximum flux use simply:
 * int ind;
 * eris_apertures_find_max_flux(self, &ind, 1);
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_apertures_find_max_flux(const cpl_apertures * self,
		int * ind, int nfind)
{



	cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

	const int    nsize = cpl_apertures_get_size(self);
	int          ifind;


	cpl_ensure_code(nsize > 0,      CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(ind,          CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(nfind > 0,      CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(nfind <= nsize, CPL_ERROR_ILLEGAL_INPUT);

	for (ifind=0; ifind < nfind; ifind++) {
		double maxflux = -1;
		int maxind = -1;
		int i;
		for (i=1; i <= nsize; i++) {
			int k;

			/* The flux has to be the highest among those not already found */
			for (k=0; k < ifind; k++) if (ind[k] == i) break;

			if (k == ifind) {
				/* i has not been inserted into ind */
				const double flux = cpl_apertures_get_flux(self, i);

				if (maxind < 0 || flux > maxflux) {
					maxind = i;
					maxflux = flux;
				}
			}
		}
		ind[ifind] = maxind;
	}
	eris_check_error_code("eris_apertures_find_max_flux");
	return CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
 * @internal
 * @brief    Find the peak flux, peak sum and position of a Gaussian
 * @param    self        Image to process
 * @param    sigma       The initial detection level  [ADU]
 * @param    pxpos       On success, the refined X-position [pixel]
 * @param    pypos       On success, the refined Y-position [pixel]
 * @param    ppeak       On success, the refined peak flux  [ADU]
 * @param    sigma_x     sigma_x of PSF
 * @param    sigma_y     sigma_y of PSF
 * @return CPL_ERROR_NONE or the relevant CPL error code on error
 *
 * The routine initially determines the approximate position and flux value of
 * the PSF with a robust Gaussian fit: first are identified all sources that lie
 * 5 sigmas above the median of the image, then is determined the position of
 * the barycenter of the region with highest peak. Finally is performed the fit
 * of a Gaussian centered on the found barycenter position.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
eris_gaussian_maxpos(const cpl_image * self,
		double sigma,
		double  * pxpos,
		double  * pypos,
		double  * ppeak,
		double  * sigma_x,
		double  * sigma_y)
{
	/* copied from irplib_strehl.c r163170 */


	cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(sigma > 0, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);



	const cpl_size  nx = cpl_image_get_size_x(self);
	const cpl_size  ny = cpl_image_get_size_y(self);
	int             iretry = 3; /* Number retries with decreasing sigma */
	int             ifluxapert = 0;
	double          med_dist;
	const double    median = cpl_image_get_median_dev(self, &med_dist);
	cpl_mask      * selection = NULL;
	cpl_size        nlabels = 0;
	cpl_image     * labels = NULL;
	cpl_apertures * aperts;
	cpl_size        npixobj;
	double          objradius;
	cpl_size        winsize;
	cpl_size        xposmax, yposmax;
	double          xposcen, yposcen;
	double          valmax, valfit = -1.0;
	cpl_array     * gauss_parameters = NULL;
	cpl_errorstate  prestate = cpl_errorstate_get();
	cpl_error_code  code = CPL_ERROR_NONE;


	cpl_ensure_code( sigma > 0.0, CPL_ERROR_ILLEGAL_INPUT);

	selection = cpl_mask_new(nx, ny);

	/* find aperture with signal larger than sigma * median deviation */
	for (; iretry > 0 && nlabels == 0; iretry--, sigma *= 0.5) {

		/* Compute the threshold */
		const double threshold = median + sigma * med_dist;

		/* Select the pixel above the threshold */
		code = cpl_mask_threshold_image(selection, self, threshold, DBL_MAX,
                                        BAD_PIX);

		if (code) break;

		/* Labelise the thresholded selection */
		cpl_image_delete(labels);
		labels = cpl_image_labelise_mask_create(selection, &nlabels);
	}
	sigma *= 2.0; /* reverse last iteration that found no labels */

	cpl_mask_delete(selection);

	if (code) {
		cpl_image_delete(labels);
		return cpl_error_set_where(cpl_func);
	} else if (nlabels == 0) {
		cpl_image_delete(labels);
		return cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
	}

	aperts = cpl_apertures_new_from_image(self, labels);

	/* Find the aperture with the greatest flux */
	code = eris_apertures_find_max_flux(aperts, &ifluxapert, 1);

	if (code) {
		cpl_apertures_delete(aperts);
		cpl_image_delete(labels);
		return cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
	}

	npixobj = cpl_apertures_get_npix(aperts, ifluxapert);
	objradius = sqrt((double)npixobj * CPL_MATH_1_PI);
	winsize = CX_MIN(CX_MIN(nx, ny), (3.0 * objradius));

	xposmax = cpl_apertures_get_maxpos_x(aperts, ifluxapert);
	yposmax = cpl_apertures_get_maxpos_y(aperts, ifluxapert);
	xposcen = cpl_apertures_get_centroid_x(aperts, ifluxapert);
	yposcen = cpl_apertures_get_centroid_y(aperts, ifluxapert);
	valmax  = cpl_apertures_get_max(aperts, ifluxapert);

	cpl_apertures_delete(aperts);
	cpl_image_delete(labels);

	cpl_msg_debug(cpl_func, "Object radius at S/R=%g: %g (window-size=%u)",
			sigma, objradius, (unsigned)winsize);
	cpl_msg_debug(cpl_func, "Object-peak @ (%d, %d) = %g", (int)xposmax,
			(int)yposmax, valmax);

	/* fit gaussian to get subpixel peak position */

	gauss_parameters = cpl_array_new(7, CPL_TYPE_DOUBLE);
	cpl_array_set_double(gauss_parameters, 0, median);

	code = cpl_fit_image_gaussian(self, NULL, xposmax, yposmax,
			winsize, winsize, gauss_parameters,
			NULL, NULL, NULL,
			NULL, NULL, NULL,
			NULL, NULL, NULL);
	if (!code) {
		const double M_x = cpl_array_get_double(gauss_parameters, 3, NULL);
		const double M_y = cpl_array_get_double(gauss_parameters, 4, NULL);
		const double SIGMA_x = cpl_array_get_double(gauss_parameters, 5, NULL);
		const double SIGMA_y = cpl_array_get_double(gauss_parameters, 6, NULL);
		valfit = cpl_gaussian_eval_2d(gauss_parameters, M_x, M_y);
		if (!cpl_errorstate_is_equal(prestate)) {
			code = cpl_error_get_code();
		} else {
			*pxpos        = M_x;
			*pypos        = M_y;
			*ppeak        = valfit;
			*sigma_x       = SIGMA_x;
			*sigma_y       = SIGMA_y;

			cpl_msg_debug(cpl_func, "Gauss-fit @ (%g, %g) = %g, SIGMA: (%g, %g)",
					M_x, M_y, valfit,
					SIGMA_x,
					SIGMA_y);

		}
	}
	cpl_array_delete(gauss_parameters);

	if (code || valfit < valmax) {
		cpl_errorstate_set(prestate);
		*pxpos   = xposcen;
		*pypos   = yposcen;
		*ppeak   = valmax;
	}
	eris_check_error_code("eris_gaussian_maxpos");
	return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief   compute QC parameters on a PSF STD image
 @param   hima    input image
 @param   parlist input parameters
 @param   context parameter context (prefix+recipe name)
 @return  propertylist with computed QC parameters
 */
/*----------------------------------------------------------------------------*/

cpl_propertylist*
eris_compute_psf_qc(hdrl_image* hima, const cpl_parameterlist* parlist,
		const char* context){

    cpl_ensure(hima != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(parlist != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(context != NULL, CPL_ERROR_NULL_INPUT, NULL);

	//cpl_errorstate clean_state = cpl_errorstate_get();
	cpl_size ima_szx = hdrl_image_get_size_x(hima);
	cpl_size ima_szy = hdrl_image_get_size_y(hima);

	double peak = 0;
	double max_x = 0;
	double max_y = 0;
	double sigma_x = 0;
	double sigma_y = 0;
	double FWHM_x = 0;
	double FWHM_y = 0;
	cpl_image* image = hdrl_image_get_image(hima);
	cpl_image* error = hdrl_image_get_image(hima);

	cpl_propertylist* qchead = cpl_propertylist_new();
	eris_gaussian_maxpos(image, 5, &max_x, &max_y, &peak, &sigma_x, &sigma_y);
	FWHM_x = sigma_x * CPL_MATH_FWHM_SIG;
	FWHM_y = sigma_y * CPL_MATH_FWHM_SIG;
	cpl_msg_info(cpl_func, "max_x: %g max_y: %g FWHM_x:%g FWHM_y: %g",
			max_x, max_y, FWHM_x, FWHM_y);

	if(cpl_error_get_code() != CPL_ERROR_NONE) {
		cpl_msg_error(cpl_func,"Error fitting 2D Gaussian to object for QC. Exit");
		cpl_error_reset();
		return qchead;
	}

	int max_ima_x = (int) (max_x + 0.5);
	int max_ima_y = (int) (max_y + 0.5);

	int psf_sz = 40;
	int wllx = ((max_ima_x - psf_sz)>0)       ? (max_ima_x - psf_sz) : 1;
	int wlly = ((max_ima_y - psf_sz)>0)       ? (max_ima_y - psf_sz) : 1;
	int wurx = ((max_ima_x + psf_sz)<ima_szx) ? (max_ima_x + psf_sz) : ima_szx ;
	int wury = ((max_ima_y + psf_sz)<ima_szy) ? (max_ima_y + psf_sz) : ima_szy ;

	cpl_table* qclog_tbl = eris_qclog_init();

	/* not used code:
	double max_ima_cx = 0;
	max_ima_cx = cpl_image_get_centroid_x_window(image, wllx, wlly, wurx, wury);
	double max_ima_cy = 0;
	max_ima_cy = cpl_image_get_centroid_y_window(image, wllx, wlly, wurx, wury);
	 */

	int llx = 8;
	int lly = 8;
	int halfbox_x = 16;
	int halfbox_y = 16;
	eris_qclog_add_int(qclog_tbl,"QC FWHM LLX", llx, "[pix] STD star FWHM LLX");
	eris_qclog_add_int(qclog_tbl,"QC FWHM LLY", lly, "[pix] STD star FWHM LLY");
	eris_qclog_add_int(qclog_tbl,"QC FWHM HBX", halfbox_x, "[pix] STD star FWHM HBX");
	eris_qclog_add_int(qclog_tbl,"QC FWHM HBX", halfbox_y, "[pix] STD star FWHM HBY");


	int check1 = 0;
	double* pdata = cpl_image_get_data_double(image);
	cpl_size npix = cpl_image_get_size_x(image) * cpl_image_get_size_y(image);
	cpl_boolean has_nan = CPL_FALSE;
	for(cpl_size i =0; i < npix; i++ ) {
		if(eris_ifu_is_nan_or_inf(pdata[i])) {
			has_nan  = CPL_TRUE;
			break;
		}
	}

	cpl_bivector * biv = NULL;
	if(has_nan == CPL_FALSE) {
		cpl_image* fimage = cpl_image_cast(image, CPL_TYPE_FLOAT);
		//cpl_image_reject_from_mask(image, mask);
		biv = cpl_image_iqe(fimage, wllx, wlly, wurx, wury);
		cpl_image_delete(fimage);
	}

	if( (cpl_error_get_code() == CPL_ERROR_NONE) && (has_nan == CPL_FALSE)) {
		const double * x = cpl_bivector_get_x_data_const(biv);
		eris_qclog_add_double(qclog_tbl, "QC CENTERX", x[0],
				"[pix] STD star X centroid position");
		eris_qclog_add_double(qclog_tbl, "QC CENTERY", x[1],
				"[pix] STD star Y centroid position");
		eris_qclog_add_double(qclog_tbl, "QC FWHM MAJ", x[2],
				"[pix] STD star FWHM on major axis");
		eris_qclog_add_double(qclog_tbl, "QC FWHM MIN", x[3],
				"[pix] STD star FWHM on minor axis");
		eris_qclog_add_double(qclog_tbl, "QC THETA", x[4],
				"[deg] STD star ellipsis angle theta");
		cpl_bivector_delete(biv);
	} else {
		eris_qclog_add_double(qclog_tbl, "QC CENTERX", max_x,
				"[pix] STD star X centroid position");
		eris_qclog_add_double(qclog_tbl, "QC CENTERY", max_y,
				"[pix] STD star Y centroid position");
		eris_qclog_add_double(qclog_tbl, "QC FWHM MAJ", FWHM_x,
				"[pix] STD star FWHM on major axis");
		eris_qclog_add_double(qclog_tbl, "QC FWHM MIN", FWHM_y,
				"[pix] STD star FWHM on minor axis");

		cpl_error_reset();
	}

	double xcen = max_x;
	double ycen = max_y;
	if (has_nan == CPL_FALSE) {
//		double norm = 0;

//		double sig_x = 0;
//		double sig_y = 0;
		double fwhm_x = 0;
		double fwhm_y = 0;
		//double disp = 0;

		double fwhm_factor = 5.0;

		char* pname = cpl_sprintf("%s.fwhm-factor",context);
		eris_parameters_get_double(parlist, pname, &fwhm_factor);
		cpl_free(pname);
		double rms = 0;
		double red_chisq = 0;
		cpl_matrix* covariance = NULL;
		double major = 0;
		double minor = 0;
		double angle = 0;
		cpl_array* fit_params = NULL;
		cpl_array* gauss2d_fit_params = cpl_array_new(7, CPL_TYPE_DOUBLE);
		cpl_array* gauss2d_err_params = cpl_array_new(7, CPL_TYPE_DOUBLE);
		cpl_matrix* phys_cov = NULL;
		int status = 0;

		if(CPL_ERROR_NONE == cpl_fit_image_gaussian(image, error,
				max_ima_x, max_ima_y, psf_sz, psf_sz, gauss2d_fit_params,
			    gauss2d_err_params, fit_params, &rms, &red_chisq, &covariance,
			    &major, &minor, &angle, &phys_cov) ) {
			sigma_x = cpl_array_get(gauss2d_fit_params, 5, &status);
			sigma_y = cpl_array_get(gauss2d_fit_params, 6, &status);

			fwhm_x = sigma_x * CPL_MATH_FWHM_SIG;
	        fwhm_y = sigma_y * CPL_MATH_FWHM_SIG;
			eris_qclog_add_double_f(qclog_tbl, "QC FWHMX", fwhm_x,
					"[pix] STD star FWHM on X");
			eris_qclog_add_double_f(qclog_tbl, "QC FWHMY", fwhm_y,
					"[pix] STD star FWHM on Y");

			halfbox_x =  (floor)(0.5 * (fwhm_x + fwhm_y) * fwhm_factor + 0.5);

			halfbox_y =  (floor)(0.5 * (fwhm_x + fwhm_y) * fwhm_factor + 0.5);

		} else {
			//TODO remove irplib
			//irplib_error_recover(clean_state, "Problem fitting Gaussian");
            cpl_array_delete(gauss2d_fit_params);
            cpl_array_delete(gauss2d_err_params);
			cpl_error_reset();

		}
	} else {

		eris_qclog_add_double_f(qclog_tbl, "QC FWHMX", FWHM_x,
				"[pix] STD star FWHM on X");
		eris_qclog_add_double_f(qclog_tbl, "QC FWHMY", FWHM_y,
				"[pix] STD star FWHM on Y");
	}

	/* we use a large value of kappa to be sure to flag well the object */
	//cpl_mask* obj_mask=NULL;
	//int pfx = 3;
	//int pfy = 3;
	//const char* pfm;
	//TODO: maybe use the mask defined elsewhere for extraction
	//if(strcmp(extract_method, "optimal") != 0) {
	//	double kappa = 5.;
	//	obj_mask = eris_object_mask(image, pfm, pfx, pfy, kappa);
	//}


	llx = (int)(xcen - halfbox_x);
	llx = (llx  > 0 ) ? llx  : 1;

	if((llx + 2 * halfbox_x) >= ima_szx) {
		halfbox_x = (int) ((ima_szx - llx - 1) / 2);
		check1++;
	}

	lly = (int)(ycen - halfbox_y);
	lly = (lly  > 0 ) ? lly  : 1;
	if((lly + 2 * halfbox_y) >= ima_szy) {
		halfbox_y = (int) ((ima_szy - lly - 1) / 2);
		check1++;
	}
	eris_qclog_add_int(qclog_tbl,"QC CHECK1", check1,
			"Check on evaluation box");
	//cpl_table_save(qclog_tbl,NULL,NULL,"qclog_tbl.fits",CPL_IO_DEFAULT);
	eris_pfits_put_qc(qchead, qclog_tbl);
	cpl_table_delete(qclog_tbl);
	eris_check_error_code("eris_compute_psf_qc");
	return qchead;

}

static cpl_error_code
eris_imagelist_reject_value(hdrl_imagelist* iml/*, cpl_value value*/)
{

	cpl_ensure(iml != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);

    cpl_size sz = hdrl_imagelist_get_size(iml);

    for(cpl_size i = 0; i < sz; i++) {
      hdrl_image* hima = hdrl_imagelist_get(iml, i);
      hdrl_image_reject_value(hima, CPL_VALUE_NAN);
    }
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 @brief   find mask flagging pixels above percent(%) pixels with lower intensity
 @param   hima    input hdrl image
 @param   percent  input percent threshold
 @return  cpl_mask* mask flagging points where objects have been found
 @note  percent must be such that:  0.25 <= percent <= 0.75
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
eris_ifu_hima_get_obj_mask_percent(hdrl_image* hima, const double percent)
{
	cpl_mask* mask_obj = NULL;

	cpl_size sx = hdrl_image_get_size_x(hima);
	cpl_size sy = hdrl_image_get_size_y(hima);

	cpl_image* image = hdrl_image_get_image(hima);
	cpl_image* ima_tmp = cpl_image_duplicate(image);


	cpl_ensure(hima != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(0.25 <= percent && percent <= 0.75, CPL_ERROR_ILLEGAL_INPUT, NULL);
	double* data = cpl_image_get_data(ima_tmp);

	cpl_size npoints = sx * sy;
	cpl_ensure(npoints > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_vector* v = cpl_vector_new(npoints);
	v = cpl_vector_wrap(npoints, data);

	cpl_vector_sort(v, CPL_SORT_ASCENDING);

	cpl_size thresh_pos = percent * npoints;
	double threshold_max = cpl_vector_get(v, thresh_pos);
	//cpl_msg_info(cpl_func,"threshold_max: %g",threshold_max);
	cpl_vector_unwrap(v);
	cpl_image_delete(ima_tmp);
	mask_obj = cpl_mask_threshold_image_create(image, DBL_MIN, threshold_max);
	cpl_mask_not(mask_obj);
    if (v != NULL) {
    	cpl_vector_delete(v);
    }
	return mask_obj;
}
/*----------------------------------------------------------------------------*/
/**
 @brief   find mask flagging pixels above a threshold set by ks-clip algorithm
 @param   hima    input hdrl image
 @param   niter   number of kapa-sigma iterations
 @param   kappa   kappa used in kappa-sigma clip of (object) data
 @return  cpl_mask* mask flagging points where objects have been found
 @note  niter must be such that:  1 <= niter <= 100
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
eris_ifu_hima_get_obj_mask(hdrl_image* hima, const cpl_size niter,
		const double kappa)
{

	cpl_mask* mask_tmp = NULL;
    cpl_mask* mask_obj = NULL;
    hdrl_value median = {0, 0};
    double max = 0;
    double stdev = 0;

	cpl_ensure(hima != NULL, CPL_ERROR_NULL_INPUT, NULL);
	cpl_ensure(1 <= niter && niter <= 100, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(kappa > 0., CPL_ERROR_ILLEGAL_INPUT, NULL);

	max = cpl_image_get_max(hdrl_image_get_image(hima));

    for (cpl_size i = 0; i < niter; i++) {
		median = hdrl_image_get_median(hima);
        if (!isnan(median.data)) {
			stdev = hdrl_image_get_stdev(hima);
			double thresh_max = median.data + kappa * stdev;
			cpl_msg_info(cpl_func, "median: %g  stdev: %g thresh: %g max: %g",
					median.data, stdev, thresh_max, max);
			thresh_max = (thresh_max < max) ? 	thresh_max : 0.99 * max;
			cpl_msg_info(cpl_func, "median: %g  stdev: %g thresh: %g max: %g",
					median.data, stdev, thresh_max, max);
            if (stdev > 0) {

				cpl_image* ima_tmp = hdrl_image_get_image(hima);

				mask_tmp = cpl_mask_threshold_image_create(ima_tmp, DBL_MIN, thresh_max);
				cpl_mask_not(mask_tmp);

				hdrl_image_reject_from_mask(hima, mask_tmp);

                if (i == (niter -1))  {
					mask_obj = cpl_mask_duplicate(mask_tmp);
				}
				cpl_mask_delete(mask_tmp);
			}
		}
	}

	eris_check_error_code("eris_ifu_hima_get_obj_mask");
	return mask_obj;
}

/**
   @brief    Get what object shift
   @param    iml imagelist
   @param    n   imagelist sequence
   @param    qclog_tbl qclog table
   @return   0 if standard 0 if pixel scale changes in input Strehl data
 */
cpl_error_code
eris_get_pupil_shift(hdrl_imagelist* iml, const int n, cpl_table** qclog_tbl)
{


	cpl_ensure(iml != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(qclog_tbl != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
	cpl_ensure(*qclog_tbl != NULL, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);
    cpl_ensure(n >= 0, CPL_ERROR_ILLEGAL_INPUT, CPL_ERROR_ILLEGAL_INPUT);

    cpl_size max_ima_x = 0;
    cpl_size max_ima_y = 0;
    int nx = 0;
    int ny = 0;

    double xshift = 0;
    double yshift = 0;

    double max_ima_cx = 0;
    double max_ima_cy = 0;

    hdrl_image* himg = NULL;
    cpl_image* img_dup = NULL;
    cpl_image* contrib_map = NULL;
    char* key_name;

    eris_imagelist_reject_value(iml/*, CPL_VALUE_NAN*/);

    hdrl_imagelist_collapse_median(iml, &himg, &contrib_map);
    nx = hdrl_image_get_size_x(himg);
    ny = hdrl_image_get_size_y(himg);

    img_dup = hdrl_image_get_image(himg);

    cpl_image_get_maxpos(img_dup, &max_ima_x, &max_ima_y);
    max_ima_cx = cpl_image_get_centroid_x_window(img_dup, 1, 1, nx, ny);
    max_ima_cy = cpl_image_get_centroid_y_window(img_dup, 1, 1, nx, ny);
    //cpl_image_delete(img_dup);


    xshift = max_ima_cx - (double)nx/2;
    yshift = max_ima_cy - (double)ny/2;

    key_name = cpl_sprintf("QC PUPIL%d SHIFTX", n);

    eris_qclog_add_double_f(*qclog_tbl, key_name, xshift,
                           "X shift centroid - center image");
    cpl_free(key_name);
    key_name = cpl_sprintf("QC PUPIL%d SHIFTY", n);
    eris_qclog_add_double_f(*qclog_tbl, key_name, yshift,
                           "Y shift centroid - center image");
    cpl_msg_info(cpl_func,"xshift: %g yshift: %g",xshift, yshift);
    cpl_free(key_name);
    hdrl_image_delete(himg);
    cpl_image_delete(contrib_map);
    eris_check_error_code("eris_get_pupil_shift");
    return cpl_error_get_code();
}

/**
 * @brief eris_ifu_mask_from_image
 * @param img_mask
 * @return
 *
 *  convert image to mask
    img:  1: good, 0:bad
    mask: 0: good, 1: bad
 */
cpl_mask* eris_ifu_mask_from_image(const cpl_image* img_mask)
{
    cpl_size        nx          = 0,
                    ny          = 0;
    cpl_mask        *mask       = NULL;
    const double    *pimg_mask  = NULL;
    cpl_binary      *pmask      = NULL;

    cpl_ensure(img_mask != NULL, CPL_ERROR_NULL_INPUT, NULL);

    TRY {
        nx = cpl_image_get_size_x(img_mask);
        ny = cpl_image_get_size_y(img_mask);

        BRK_IF_NULL(
            mask = cpl_mask_new(nx,ny));
        BRK_IF_NULL(
            pimg_mask = cpl_image_get_data_double_const(img_mask));
        BRK_IF_NULL(
            pmask = cpl_mask_get_data(mask));

        for (cpl_size x = 0; x < nx; x++) {
            for (cpl_size y = 0; y < ny; y++) {
                if (pimg_mask[x+nx*y] > 0.5) {
                    // pimg_mask = 1 -> CPL_BINARY_0
                    pmask[x+nx*y] = GOOD_PIX;
                } else {
                    // pimg_mask = 0 -> CPL_BINARY_1
                    pmask[x+nx*y] = BAD_PIX;
                }
            }
        }


    } CATCH {
        CATCH_MSG();
        eris_ifu_free_mask(&mask);
    }

    return mask;
}

/**@}*/
