/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef ERISP_ERIS_ERIS_IFU_WAVE_H_
#define ERISP_ERIS_ERIS_IFU_WAVE_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <math.h>

#include <cpl.h>
#include <hdrl.h>
#include "eris_ifu_functions.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/
// DEGREE: degree of the polynomial
// FIRST_FIT:  fit of isolated lines in collapsed slitlet spectra
// COLUMN_FIT: final fit of reference lines in wavecal image columns
// SMOOTH_FIT: cross fitting of the column fits
#define FIRST_FIT_DEGREE   2
#define COLUMN_FIT_DEGREE  3
#define SMOOTH_FIT_DEGREE  2

//define central wavelengths and pixel dispersions for each grating

#define CENTRALLAMBDA_K_SPIFFI    2.2
#define DISPERSION_K_SPIFFI       0.000245
#define CENTRALLAMBDA_H_SPIFFI    1.65
#define DISPERSION_H_SPIFFI       0.000195
#define CENTRALLAMBDA_J_SPIFFI    1.25
#define DISPERSION_J_SPIFFI       0.000145
#define CENTRALLAMBDA_HK_SPIFFI   1.95
#define DISPERSION_HK_SPIFFI      0.0005

#define CENTRALLAMBDA_J_LOW       1.253
#define DISPERSION_J_LOW          1.52e-04
#define CENTRALLAMBDA_J_SHORT     1.185
#define DISPERSION_J_SHORT        7.85e-05
#define CENTRALLAMBDA_J_MIDDLE    1.264
#define DISPERSION_J_MIDDLE       7.83e-05
#define CENTRALLAMBDA_J_LONG      1.344
#define DISPERSION_J_LONG         7.80e-05
#define CENTRALLAMBDA_H_LOW       1.656
#define DISPERSION_H_LOW          2.02e-04
#define CENTRALLAMBDA_H_SHORT     1.561
#define DISPERSION_H_SHORT        9.79e-05
#define CENTRALLAMBDA_H_MIDDLE    1.667
#define DISPERSION_H_MIDDLE       9.75e-05
#define CENTRALLAMBDA_H_LONG      1.760
#define DISPERSION_H_LONG         9.72e-05
#define CENTRALLAMBDA_K_LOW       2.207
#define DISPERSION_K_LOW          2.54e-04
#define CENTRALLAMBDA_K_SHORT     2.074
#define DISPERSION_K_SHORT        1.31e-04
#define CENTRALLAMBDA_K_MIDDLE    2.197
#define DISPERSION_K_MIDDLE       1.31e-04
#define CENTRALLAMBDA_K_LONG      2.328
#define DISPERSION_K_LONG         1.30e-04


// FITS header keywords in reference linelist table extensions
#define ERIS_IFU_REFTABLE_INSTR_HDR		"INSTRUME"
#define ERIS_IFU_REFTABLE_LAMPS_HDR		"LAMPS"
#define ERIS_IFU_REFTABLE_SPIFFI_VAL	"SINFONI"
#define ERIS_IFU_REFTABLE_SPIFFIER_VAL	"ERIS"
#define ERIS_IFU_REFTABLE_ARGON_VAL 	"Ar"
#define ERIS_IFU_REFTABLE_NEON_VAL	 	"Ne"
#define ERIS_IFU_REFTABLE_KRYPTON_VAL 	"Kr"
#define ERIS_IFU_REFTABLE_XEON_VAL 		"Xe"
#define ERIS_IFU_REFTABLE_IGNORED_COLUMN	"ignored"
#define ERIS_IFU_REFTABLE_LAMBDA_COLUMN		"wavelength"
#define ERIS_IFU_REFTABLE_INTENSITY_COLUMN	"intensity"

// FITS table columns in firstFit table
#define ERIS_IFU_FIRSTFIT_INSTRUMENT	"instrument"
#define ERIS_IFU_FIRSTFIT_BAND			"band"
#define ERIS_IFU_FIRSTFIT_LAMPS			"lamps"
#define ERIS_IFU_FIRSTFIT_BLOCK			"block"
#define ERIS_IFU_FIRSTFIT_POSITION		"position"
#define ERIS_IFU_FIRSTFIT_WAVELENGTH	"wavelength"

#define ERIS_IFU_ARCFRAMR_IDX   "FRAMESET_IDX"
#define ERIS_IFU_ARCFRAME_DIT   "DIT"
#define ERIS_IFU_ARCFRAME_LAMP  "LAMP_STATUS"
#define ERIS_IFU_ARCFRAME_FILE  "FILENAME"

#define ERIS_IFU_FITTABLE_SLITLET      "slitlet"
#define ERIS_IFU_FITTABLE_INDEX        "index"
#define ERIS_IFU_FITTABLE_POSITION     "position"
#define ERIS_IFU_FITTABLE_ERRORCODE    "errorcode"
#define ERIS_IFU_FITTABLE_WAVELENGTH   "wavelength"


struct waveSetupStruct {
        double  fwhm;
        double  sigma;
        int     s_range;
        int     c_range;
        double  ri_dispersion;
        double  ri_centralLambda;
};
struct waveTablesStruct {
        cpl_table *slitletFitting;
        cpl_table *columnFitting;
        cpl_table *slitletCoeff;
        cpl_table *columnCoeffRaw;
        cpl_table *columnCoeffSmoothed;
        cpl_table *smoothingCoeff;
};

/*-----------------------------------------------------------------------------
                                Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
@brief 	Extract and preprocess arc images from frameset
@param	arcFrames: frameset containing arc image frames only
@param  arcImgCnt (OUT): number of returned arc images
@param  arcImages (OUT): arc image image list
@param  lampStates (OUT): lamp status for each arc image
@param	band (OUT): detected wavelength band (from the raw frames)
@param  instrument (OUT): detected instrument (from the raw frames)
@return CPL error code: CPL_ERROR_NONE in case of success

The input frames are sorted according their lamp/DIT status. Then for each
detected lamp status all images will be collected and collapsed to single
images. When dark images with the same DIT are avvailable too, they will be
collapsed and the resulting image will be subtracted from the arc image.

Image list collapsing and dark subtraction is implemented by using
the function @ref eris_ifu_wave_collapse_arc_images.

Possible cpl_error_code set in this function:\n
- CPL_ERROR_NULL_INPUT input frame set is NULL
- other error codes returned by internally used CPL functions

*/
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_wave_get_arc_images (
        cpl_frameset *arcFrames,
        int exposureCorrectionMode,
        int *arcImgCnt,
        hdrl_imagelist **arcImages,
        int **lampStates,
        ifsBand *band,
        ifsPreopticsScale *scale,
        ifsInstrument *instrument,
		double saturation_threshold,
		cpl_table** qclog);

/*----------------------------------------------------------------------------*/
/**
@brief
@param	darkTable: table holding holding data about all dark frames
@param	litTable: table holding holding data about all lamp-on frames
@param	lampState: requested lamp-on pattern
@param	dit: requested DIT
@return a collapsed and dark subtracted image

The input tables shall have following columns:
 - index position of the frame in the input frameset
 - lamp-on pattern
 - DIT
 - filename of the frame

The input tables are searched for rows which match the requested lamp-on
pattern and DIT value. The images of the frames found are collapsed and
the dark images (if one was found) is subtracted from the lamp-on images.

If there are just two images are found the image collapsing is done by using
the mean value. For three or more images the median value is used.

Possible cpl_error_code set in this function:\n
- CPL_ERROR_NULL_INPUT One or both input tables are NULL
- CPL_ERROR_DATA_NOT_FOUND No frames with the lamp-on pattern found
*/
/*----------------------------------------------------------------------------*/
hdrl_image *eris_ifu_wave_collapse_arc_images(
        cpl_table *darkTable,
        cpl_table *litTable,
        int lampState,
        float dit,
        int exposureCorrectionMode,
		double saturation_threshold,
		cpl_table** qclog);

/*----------------------------------------------------------------------------*/
/**
 @brief Return the wavelength calibration image
@param arcImgCnt: Number of preprocessed arc lamp exposures
@param *arcImages: list of preprocessed arc lamp exposures
@param *lampStates: list of lamp states, one for each arc lamp exposures
@param band: instrument band selection
@param instrument: current instrument (SPIFFI of  SPIFFIER)
@param refLineTableFileName: file name for reference line table
@param firstFitTableFileName file name of FITS table holding firs fit positions
@param *slitPos: slitlet left and right border positions

@return the wavelength calibration image

Possible cpl_error_code set in this function:\n
- CPL_ERROR_NULL_INPUT The

*/
/*----------------------------------------------------------------------------*/
cpl_image * eris_ifu_wave_get_calImg(int arcImagesCnt,
        hdrl_imagelist *arcImages,
        int *lampStates,
        ifsBand band,
        ifsInstrument instrument,
        struct waveSetupStruct waveSetup,
        const char* refLineTableFileName,
        const char* firstFitTableFileName,
        cpl_bivector *slitPos,
        struct waveTablesStruct *tables,
        int productDepth,
        cpl_frameset *fs,
        const cpl_parameterlist* parlist,
		cpl_table* qclog);
/*----------------------------------------------------------------------------*/
/**
@brief  Return a bivector (lambda, intensitry) holding the arc lamp reference
lines
@param	refLineTableFileName: file name of the FITS file holding the
reference line tables
@param	instrument: the requested instrument
@param	lampState: the requested lamp-on pattern
@return	a bivector (lambda, intensitry) holding the arc lamp reference


Possible cpl_error_code set in this function:\n
- CPL_ERROR_NULL_INPUT The

*/
/*----------------------------------------------------------------------------*/
cpl_bivector *eris_ifu_wave_get_refLines(
		const char * refLineTableFileName,
		ifsInstrument instrument,
		int lampState);

/*----------------------------------------------------------------------------*/
/**
@brief  Return a CPL vector which is an collapsed spectrum of a slitlet
@param	dataImg: a detector frame image
@param  center: center coordinate of the slitlet

@return	a CPL vector holding the collapsed spectrum

Possible cpl_error_code set in this function:\n
- CPL_ERROR_NULL_INPUT

*/
/*----------------------------------------------------------------------------*/
cpl_vector * eris_ifu_wave_collapse_slitlet(const cpl_image *waveImg,
                                            int center);

/*----------------------------------------------------------------------------*/
/**
@brief  Internal helper function to collect and save collapsed spectra
@param	collapsedSpectrum: a collapsed spectrum of a slitlet
@param  sIdx: slitlet index [0...SLITLET_CNT-1]
@param  nRows: size of spectrum
@param lampStatus: arc lamp status
@param band: instrument band selection
@param instrument: current instrument (SPIFFI of  SPIFFIER)
@param refLine: arc lamp reference lines

@return	void

Possible cpl_error_code set in this function:

*/
/*----------------------------------------------------------------------------*/
void eris_ifu_wave_save_spectrum(cpl_image **collapsedSpectra,
        int aIdx,
        cpl_vector *collapsedSpectrum,
        int sIdx,
        cpl_size nRows,
        int lampState,
        ifsBand band,
        ifsInstrument instrument,
        cpl_bivector *refLines,
        int productDepth);

/*----------------------------------------------------------------------------*/
/**
@brief
@param
@param
@param

@return

Possible cpl_error_code set in this function:

*/
/*----------------------------------------------------------------------------*/
cpl_polynomial* eris_ifu_get_first_fit(
        cpl_image **spectra,
        int *lampStates,
        int spectrumCnt,
        int slitlet,
        int ffOffsetIn,
        int *ffOffsetOut,
        struct waveSetupStruct waveSetup,
        cpl_table *firstFitTable,
        cpl_table *fittingDumpTable,
        cpl_table *coeffDumpTable);

/*----------------------------------------------------------------------------*/
/**
@brief
@param
@param
@param

@return

Possible cpl_error_code set in this function:

*/
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_fit_all_lines(const char *recipe_name,
        int sIdx,
        const double *slitletStart,
        const double *slitletEnd,
        int arcImgCnt,
        const cpl_image **dataImg,
        cpl_bivector **refLines,
        struct waveSetupStruct waveSetup,
        cpl_polynomial *firstFit,
        cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y],
        cpl_table *dumpTable,
        cpl_table *columnCoeffRawTable,
        cpl_table *columnCoeffSmoothedTable,
        cpl_table *smoothingCoeffTable);
/*----------------------------------------------------------------------------*/
/**
@brief
@param
@param
@param

@return

Possible cpl_error_code set in this function:

*/
/*----------------------------------------------------------------------------*/
cpl_table *eris_ifu_wave_get_firstFitTable(
		const char* firstFitTableFileName,
		ifsInstrument instrument,
//		int lampStatus,
		ifsBand band);

cpl_image *eris_ifu_wave_resampled_arc_image(
        const cpl_image *arcImg,
        const cpl_image *waveCalImg,
        ifsBand band,
        double minLambda,
        double maxLambda,
        cpl_propertylist *plist,
        int axisNumber);

cpl_error_code eris_ifu_save_resampled_arc_images(
        int arcImgCnt,
        const cpl_image **dataImg,
        const cpl_image *waveCalImg,
        ifsBand band,
        int *lampStates,
        cpl_frameset* frameset,
        const cpl_parameterlist* parlist);

cpl_error_code eris_ifu_wave_smooth_coeffs(
        int sIdx,
        int start,
        int end,
        int fitDegree,
        cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y],
        cpl_table *columnCoeffSmoothedTable,
        cpl_table *smoothingCoeffTable);

cpl_error_code eris_ifu_wave_pos_err(
        cpl_polynomial *allFits[ERIS_IFU_DETECTOR_SIZE_Y],
        cpl_table *fitTable, ifsBand band);

cpl_error_code eris_ifu_read_wave_setup(
        const char* filename,
        ifsBand band,
        struct waveSetupStruct *waveSetup);

cpl_table *eris_ifu_wave_create_fitting_table( void );

cpl_error_code eris_ifu_wave_fill_fitting_table(
		cpl_table *table,
		int row,
		int slitlet,
		int index,
        int arcImgIdx,
		int position,
		double wavelength,
		struct gaussParStruct *gaussPar);

cpl_table *eris_ifu_wave_create_coeff_table(
        int size,
        int polynomialDegree);

cpl_error_code eris_ifu_wave_fill_coeff_table(
        cpl_table *table,
        int row,
        int idx,
        int nLines,
        cpl_polynomial *polynomial);

cpl_error_code eris_ifu_wave_save_fitting_tables(
        struct waveTablesStruct *tables,
        ifsInstrument           instrument,
        ifsBand                 band,
        struct waveSetupStruct waveSetup,
	cpl_frameset* frameset,
	const cpl_parameterlist* parlist,
	const char* recipe_name);

cpl_error_code eris_ifu_wave_init_tables(
        struct waveTablesStruct *tables);

void eris_ifu_wave_free_tables(
        struct waveTablesStruct *tables);

void eris_ifu_wave_clear_tables(
        struct waveTablesStruct *tables);
cpl_error_code
eris_ifu_get_dispersion(ifsBand band, double* dispersion);

cpl_error_code
eris_ifu_get_central_lambda(ifsBand band, double* centralLambda);

#endif


