/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <libgen.h>
#include <math.h>
#include <string.h>
#include "eris_nix_casu_utils.h"
#include <hdrl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_casu_utils     Utilities to ease the use of CASU 
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Translate a located_imagelist to arrays of casu_fits structs 
  @return   A CPL error code

  The function translates a located_imagelist to 3 arrays of casu_fits structs.
  If the cpl_error_code on entry is not good or if the pointer to the 
  located_imagelist is NULL then the routine will return immediately.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code encu_limlist_to_casu_fits(located_imagelist * limlist,
                                         casu_fits *** indata,
                                         casu_fits *** inconf,
                                         casu_fits *** invar) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    if (!limlist) return CPL_ERROR_NONE;

    cpl_size nimages = limlist->size;
    *indata = cpl_malloc(nimages * sizeof(casu_fits *));
    *inconf = cpl_malloc(nimages * sizeof(casu_fits *));
    *invar = cpl_malloc(nimages * sizeof(casu_fits *));

    for (int i = 0; i < (int)nimages; i++) {
        enu_check(limlist->limages[i]->himage != NULL, 
                  CPL_ERROR_NULL_INPUT,
                  "located image has no himage");
        (*indata)[i] = casu_fits_wrap(cpl_image_cast(hdrl_image_get_image(
                                      limlist->limages[i]->himage),
                                      CPL_TYPE_FLOAT), NULL,
                                      limlist->limages[i]->plist,
                                      limlist->limages[i]->plist);
        (*indata)[i]->fname = cpl_strdup(cpl_frame_get_filename(
                                         limlist->limages[i]->frame));
        (*indata)[i]->nexten = 1;
        enu_check(limlist->limages[i]->confidence != NULL, 
                  CPL_ERROR_NULL_INPUT,
                  "located image has no confidence array");
        (*inconf)[i] = casu_fits_wrap(cpl_image_cast(
                                      limlist->limages[i]->confidence,
                                      CPL_TYPE_FLOAT), NULL,
                                      limlist->limages[i]->plist, NULL);
        (*inconf)[i]->fname = cpl_strdup(cpl_frame_get_filename(
                                         limlist->limages[i]->frame));
        (*inconf)[i]->nexten = 4;
        (*invar)[i] = casu_fits_wrap(cpl_image_cast(hdrl_image_get_error(
                                     limlist->limages[i]->himage),
                                     CPL_TYPE_FLOAT), NULL,
                                     limlist->limages[i]->plist,
                                     limlist->limages[i]->plist);
        (*invar)[i]->fname = cpl_strdup(cpl_frame_get_filename(
                                        limlist->limages[i]->frame));
        (*invar)[i]->nexten = 2;
    } 

 cleanup:

    /* memory cleanup tbd */

    return cpl_error_get_code();;
}

/**@}*/

