/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "casu_utils.h"
#include "casu_wcsutils.h"
#include "eris_nix_catalogue.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_catalogue        Utilities relating to catalogues 
 */
/*----------------------------------------------------------------------------*/


/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Rename essential columns to standard names
  @return   An error code or CPL_ERROR_NONE if all goes well

  TBD
 */
/*----------------------------------------------------------------------------*/

cpl_error_code en_catalogue_conform_naming(cpl_frameset * frameset,
                                           const char * tag) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* parameter check */

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tag, CPL_ERROR_NULL_INPUT);
 
    /* Look for tagged file in frameset */

    const cpl_frame * target_frame = cpl_frameset_find_const(frameset, tag);
    if (target_frame != NULL) {
        const char * filename = cpl_frame_get_filename(target_frame);

        /* Read the whole input file. It's assumed this isn't so big that it's
           going to take very long. */

        cpl_table * whole = cpl_table_load(filename, 1, CPL_FALSE);

        /* Naming problems. HDRL expects columns RA and DEC. If not present 
           in table look for ra, dec or Dec, if these are found then rename. */

        en_catalogue_name_conformance(whole);

        /* Replace the FITS table with the updated version */

        cpl_propertylist * pheader = cpl_propertylist_load(filename, 0);
        cpl_propertylist * header = cpl_propertylist_load(filename, 1);
        cpl_table_save(whole, pheader, header, filename, CPL_IO_CREATE);
        cpl_propertylist_delete(pheader);
        cpl_propertylist_delete(header);

        cpl_table_delete(whole);

    } else {
        cpl_msg_warning(cpl_func, "SoF has no file tagged %s", tag);
    }

    return cpl_error_get_code();
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Estimate the coverage of a set of images.
  @return   A CPL error code.

  The function finds the RA and Dec limits of the area covered by a list
  of images.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code en_catalogue_coverage(const located_imagelist * jitters,
                                     double * ramin, double * ramax,
                                     double * decmin, double * decmax) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_ensure_code(jitters, CPL_ERROR_NULL_INPUT);

    for (cpl_size j=0; j < jitters->size; j++) {
        double ra1 = 0.0;
        double ra2 = 0.0;
        double dec1 = 0.0;
        double dec2 = 0.0;
        int status = CASU_OK;
        casu_coverage(jitters->limages[j]->plist, 1, &ra1, &ra2, &dec1,
                      &dec2, &status);
        enu_check(status==CASU_OK, CPL_ERROR_INCOMPATIBLE_INPUT,
                  "error in casu_coverage: status=%d", status);

        if (j==0) {
            *ramin = ra1;
            *ramax = ra2;
            *decmin = dec1;
            *decmax = dec2;
        } else {
            if (ra1 < *ramin) *ramin = ra1;
            if (ra2 > *ramax) *ramax = ra2;
            if (dec1 < *decmin) *decmin = dec1;
            if (dec2 < *decmax) *decmax = dec2;
        }
    }

    cpl_msg_info(cpl_func, "...coverage %5.3f %5.3f %5.3f %5.3f", *ramin,
                 *ramax, *decmin, *decmax);
  cleanup:
    return cpl_error_get_code();
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Extract standards from a named FITS table
  @return   A table structure with the extracted catalogue objects

  The function loads a catalogue from a named FITS table, then selects 
  sources in the given RA, Dec window into a cpl_table and 
  returns it.
 */
/*----------------------------------------------------------------------------*/

cpl_table * en_catalogue_load_from_file(const char * filename,
                                        const double ramin,
                                        const double ramax,
                                        const double decmin,
                                        const double decmax) {


    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_table * whole = NULL;
    cpl_table * result = NULL;

    /* parameter check */

    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

    /* Read the whole input file. It's assumed this isn't so big that it's
       going to take very long. */

    whole = cpl_table_load(filename, 1, CPL_FALSE);

    /* Enforce column naming compliance */

    en_catalogue_name_conformance(whole);

    cpl_size nrows = cpl_table_get_nrow(whole);
    cpl_msg_info(cpl_func, "read rows %d %s", (int)nrows, cpl_error_get_message());

    /* Is there a wrap around problem? */

    if (ramin < 0.0 && ramax > 0.0) {

        /* yes, so we need to break problem into 2 queries */

        double ramin1 = ramin + 360.0;
        double ramax1 = 360.0;

        cpl_table_unselect_all(whole);
        cpl_table_or_selected_double(whole, "RA", CPL_NOT_LESS_THAN, ramin1);
        cpl_table_and_selected_double(whole, "RA", CPL_NOT_GREATER_THAN,
                                      ramax1);
        cpl_table_and_selected_double(whole, "DEC", CPL_NOT_LESS_THAN, decmin);
        cpl_table_and_selected_double(whole, "DEC", CPL_NOT_GREATER_THAN,
                                      decmax);
        result = cpl_table_extract_selected(whole);

        /* second query */

        ramin1 = 0.000001;
        ramax1 = ramax;

        cpl_table_unselect_all(whole);
        cpl_table_or_selected_double(whole, "RA", CPL_NOT_LESS_THAN, ramin1);
        cpl_table_and_selected_double(whole, "RA", CPL_NOT_GREATER_THAN, 
                                      ramax1);
        cpl_table_and_selected_double(whole, "DEC", CPL_NOT_LESS_THAN, decmin);
        cpl_table_and_selected_double(whole, "DEC", CPL_NOT_GREATER_THAN,
                                      decmax);
        cpl_table * out2 = cpl_table_extract_selected(whole);

        /* merge query results */

        nrows = cpl_table_get_nrow(result);
        cpl_table_insert(result, out2, nrows+1);
        cpl_table_delete(out2);

    } else {

        /* no, only one query needed */

        cpl_table_unselect_all(whole);
        cpl_table_or_selected_double(whole, "RA", CPL_NOT_LESS_THAN, ramin);
        cpl_table_and_selected_double(whole, "RA", CPL_NOT_GREATER_THAN,
                                      ramax);
        cpl_table_and_selected_double(whole, "DEC", CPL_NOT_LESS_THAN, decmin);
        cpl_table_and_selected_double(whole, "DEC", CPL_NOT_GREATER_THAN,
                                      decmax);
        result = cpl_table_extract_selected(whole);

        for (cpl_size ir=0; ir<nrows; ir++) {
            int null =0;
            double ra = cpl_table_get(whole, "RA", ir, &null);
            double dec = cpl_table_get(whole, "DEC", ir, &null);

            cpl_msg_info(cpl_func, "%8.6f %8.6f %8.6f %8.6f %8.6f %8.6f", ra, ramin, ramax, dec, decmin, decmax);
        }
    }

    /* Sort the table by DEC */

    cpl_propertylist * p = cpl_propertylist_new();
    cpl_propertylist_append_bool(p, "DEC", 0);
    cpl_table_sort(result, p);
    cpl_propertylist_delete(p);

    /* cleanup */

    cpl_table_delete(whole);
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_table_delete(result);
        result = NULL;
    }
    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Extract standards from a FITS table tagged in the frameset
  @return   A table structure with the extracted catalogue objects

  The function searches for a tagged catalogue in the frameset. If one is
  present it loads the contents in the RA, Dec window into a cpl_table and 
  returns it.
 */
/*----------------------------------------------------------------------------*/

cpl_table * en_catalogue_load_from_frameset(cpl_frameset * frameset,
                                            const char * tag,
                                            const double ramin,
                                            const double ramax,
                                            const double decmin,
                                            const double decmax,
                                            cpl_frameset * used,
                                            const int required) {


    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_table * result = NULL;

    /* parameter check */

    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);
 
    /* Look for tagged file in frameset */

    const cpl_frame * target_frame = cpl_frameset_find_const(frameset, tag);
    if (target_frame != NULL) {
        const char * filename = cpl_frame_get_filename(target_frame);

        result = en_catalogue_load_from_file(filename, ramin, ramax, decmin,
                                             decmax);
        cpl_frameset_insert(used, cpl_frame_duplicate(target_frame));
    } else {
        enu_check(!required, CPL_ERROR_DATA_NOT_FOUND,
                  "SoF has no file tagged %s", tag);
    }

 cleanup:
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_table_delete(result);
        result = NULL;
    }
    return result;
}

/**@}*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Rename essential columns to standard names
  @return   An error code or CPL_ERROR_NONE if all goes well

  TBD
 */
/*----------------------------------------------------------------------------*/

cpl_error_code en_catalogue_name_conformance(cpl_table * catalogue) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_ensure_code(catalogue, CPL_ERROR_NULL_INPUT);

    /* Naming problems. HDRL expects columns RA and DEC. CASU expects
       RA and Dec. If not present in table look for ra, dec or Dec, if these are found then rename. */

    if (!cpl_table_has_column(catalogue, "RA")) {
        if (cpl_table_has_column(catalogue, "ra")) {
            cpl_table_name_column(catalogue, "ra", "RA");
        }
    }
    if (!cpl_table_has_column(catalogue, "DEC")) {
        if (cpl_table_has_column(catalogue, "dec")) {
            cpl_table_name_column(catalogue, "dec", "DEC");
        } else if (cpl_table_has_column(catalogue, "Dec")) {
            cpl_table_duplicate_column(catalogue, "DEC", catalogue, "Dec");
        }
    }

    return cpl_error_get_code();
}

/**@}*/

