/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef _OPENMP
#include <omp.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_detector.h"

#include "eris_nix_casu_utils.h"
#include "eris_nix_dfs.h"

#include <casu_mods.h>
#include <hdrl.h>
#include <libgen.h>
#include <math.h>
#include <string.h>
#include <time.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_detector     Detector Utilities
 */
/*----------------------------------------------------------------------------*/

inline static
void engl_lin_correct_(double *,
                       const cpl_size,
                       const double *,
                       const cpl_binary,
                       const double,
                       cpl_binary *, 
                       const int) CPL_ATTR_NONNULL;

inline static
double engl_lin_find_(const double,
                      double,
                      double,
                      const cpl_size,
                      const double *,
                      double *) CPL_ATTR_NONNULL;

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Estimate the raw samples from an 'up the ramp' result.
  @param    start_intensity  The current estimate of the actual intensity 
  @param    ramp_intensity   The intensity that fits the non-linearized 'ramp'
  @param    nr               The number of samples in the 'ramp'
  @param    dit              The detector integration time
  @param    f                Relating to the position of each pixel in the readout sequence
  @return   An eris_nix_samples struct.

  The function estimates for each pixel the samples that were taken
  then fit with a line to give the observed 'up the ramp' intensity. 
  The method followed is that of Vacca, Cushing and Rayner, 2003. 
  PASP, 115, 389, eqn 49.

           si = (i - f)*I*dt 

  where dit = (nr - 1) * dt, f is the fraction < 1, whose value for each pixel
  depends on its position in the readout sequence.

  The intensity is split between 'start' and 'ramp' values to allow for
  iterative linearization, see method 'eris_nix_linearize_image'.
 */
/*----------------------------------------------------------------------------*/

eris_nix_samples * end_calculate_samples(const cpl_image * start_intensity,
                                         const cpl_image * ramp_intensity,
                                         cpl_size nr,
                                         const double dit,
                                         const cpl_image * f) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(start_intensity, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(ramp_intensity, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(f, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(nr > 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(dit > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    eris_nix_samples * result = cpl_calloc(1, sizeof(eris_nix_samples));
    result->nr = nr;
    result->samples = cpl_calloc(nr, sizeof(cpl_image *));
    const double dt = dit / (float) (nr - 1);

    /* first sample calculated from start_intensity */

    result->samples[0] = cpl_image_multiply_scalar_create(f, -1.0);
    cpl_image_add_scalar(result->samples[0], 1.0);
    cpl_image_multiply(result->samples[0], start_intensity);
    cpl_image_multiply_scalar(result->samples[0], dt);

    /* subsequent (ramp) samples with slope from ramp_intensity, _relative_
       to Vacca sample 1 (=sample 0 in struct); f drops out. */

    for (cpl_size i = 2; i <= nr; i++) {
        result->samples[i-1] = cpl_image_duplicate(ramp_intensity);
        cpl_image_multiply_scalar(result->samples[i-1], (double) (i-1) * dt);
        /* add sample 0 to give absolute value */
        cpl_image_add(result->samples[i-1], result->samples[0]);
    }    

    /* tidy up on error */
 
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        end_samples_delete(result);
        result = NULL;
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete an eris_nix_samples struct.
  @param    samples    The samples struct to be deleted
  @return   void

  The function deletes the given eris_nix_samples structure.
 */
/*----------------------------------------------------------------------------*/

void end_samples_delete(eris_nix_samples * samples) {

    if (samples) {
        for (cpl_size i = 0; i < samples->nr; i++) {
            cpl_image_delete(samples->samples[i]);
        }
        cpl_free(samples->samples);
        cpl_free(samples);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Use DETMON-produced files to directly linearize an image.
  @param    data           The image to be linearized
  @param    gain_lin struct containing gain/linearity info
  @param    x_probe        x-coord of pixel producing debug info
  @param    y_probe        y-coord of pixel producing debug info
  @return   CPL_ERROR_NONE if all goes well, otherwise an error.

  This is not the linearization procedure described in Vacca et al. 
  where the pedestal and signal values are estimated and corrected
  in an iterative procedure.

  Rather, the linearization is the direct correction of an image of detector
  samples. DETMON has previously been used to fit a curve to the signal v DIT 
  data of form:

       s_obs = a + bt + ct^2 + dt^3

  'a' should be small because the curve should pass through the origin
  (and could be forced to do so but the current DETMON (Mar 2018)
  fitting routine does not do this)

  A linear detector response would be 
    
       s_lin = bt

       We want s_lin = s_obs * correction

                     = s_obs *          bt
                               --------------------
                               a + bt + ct^2 + dt^3

       for the t where s_obs = a + bt + ct^2 +dt^3

  This routine calculates for each pixel, from the DETMON curve, the 
  appropriate correction, and applies it.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code end_linearize_image(cpl_image * data,
                                   const gain_linearity * gain_lin,
                                   const cpl_size rot,
                                   const cpl_size strx,
                                   const cpl_size stry,
                                   const cpl_size x_probe,
                                   const cpl_size y_probe) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

#ifdef _OPENMP
    double time1, time2, time3;
    time1 = omp_get_wtime();
#else
    clock_t time1, time2, time3;
    time1 = clock();
#endif

    cpl_ensure_code(data, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(gain_lin, CPL_ERROR_NULL_INPUT);
    /* code assumes rot = 0 for now */
    cpl_ensure_code(rot == 0, CPL_ERROR_UNSUPPORTED_MODE);

    /* Get various parameters need for linearization */
            
    const cpl_size fit_order = cpl_imagelist_get_size(gain_lin->
                                                      lin_coeffs);
    const cpl_binary * fit_bpm = cpl_mask_get_data_const(gain_lin->bpm);

    enu_check(cpl_error_get_code()==CPL_ERROR_NONE && fit_order > 1,
              CPL_ERROR_INCOMPATIBLE_INPUT, 
              "linearization data in unexpected format");

    /* sizes of linearity data and image */

    cpl_size coeff_nx = cpl_image_get_size_x(cpl_imagelist_get_const(
                                             gain_lin->lin_coeffs, 0));
    cpl_size coeff_ny = cpl_image_get_size_y(cpl_imagelist_get_const(
                                             gain_lin->lin_coeffs, 0));
    const cpl_size nx = cpl_image_get_size_x(data);
    const cpl_size ny = cpl_image_get_size_y(data);

    /* check that the image data is double */
    enu_check(cpl_image_get_type(data)==CPL_TYPE_DOUBLE,
              CPL_ERROR_INCOMPATIBLE_INPUT, "image data is not double");

    /* Get pointers to image data and bpm */

    double * pdata = cpl_image_get_data_double(data);
    cpl_binary * pbpm = cpl_mask_get_data(cpl_image_get_bpm(data));
    const cpl_boolean do_info_msg = cpl_msg_get_level() <= CPL_MSG_INFO &&
        (x_probe >= 0) && (y_probe >= 0);

    enu_check(nx + strx - 1 <= coeff_nx, CPL_ERROR_UNSPECIFIED,
              "Illegal X-windowing");
    enu_check(ny + stry - 1 <= coeff_ny, CPL_ERROR_UNSPECIFIED,
              "Illegal Y-windowing");
             
    /* Loop through pixels */

#ifdef _OPENMP
    time2 = omp_get_wtime();
#else
    time2 = clock();
#endif

    if (strx == 1 && stry == 1 && coeff_nx == nx) {

#ifdef _OPENMP
#pragma omp parallel for
#endif
        for (cpl_size ipos= 0 ; ipos < nx * ny ; ipos++) { /* pixel pos in image */
            if (!pbpm[ipos]) { /* Skip rejected pixels */

                const int debug = do_info_msg &&
                    (ipos % nx == x_probe) && (ipos / nx == y_probe);
             
                /* Correct the good pixel */
                engl_lin_correct_(pdata + ipos, fit_order, gain_lin
                                  ->ordered_lin_coeffs + ipos * fit_order,
                                  fit_bpm[ipos],
                                  gain_lin->saturation_limit,
                                  pbpm + ipos,
                                  debug);
            }
        }
    } else {
        cpl_msg_info(cpl_func, "Windowing: %d %d %d %d", (int)strx, (int)stry,
                     (int)coeff_nx, (int)coeff_ny);

#ifdef _OPENMP
#pragma omp parallel for
#endif
        for (cpl_size iy=0 ; iy < ny ; iy++) {
            for (cpl_size ix=0 ; ix < nx ; ix++) {

                /* pixel pos in image */
                const cpl_size ipos = ix + iy * nx;

                if (!pbpm[ipos]) { /* Skip rejected pixels */

                    /* pixel pos in linearity arrays, strx and stry are 1-based */
                    const cpl_size coeff_x = ix + strx - 1;
                    const cpl_size coeff_y = iy + stry - 1;
                    const cpl_size coeff_pos = coeff_x + coeff_y * coeff_nx;
                    const int debug = do_info_msg && (ix == x_probe) && (iy == y_probe);

                    /* Correct the good pixel */

                    engl_lin_correct_(pdata + ipos, fit_order, gain_lin
                                      ->ordered_lin_coeffs + coeff_pos * fit_order,
                                      fit_bpm[coeff_pos],
                                      gain_lin->saturation_limit,
                                      pbpm + ipos,
                                      debug);
                }
            }
        }
    }

#ifdef _OPENMP
    time3 = omp_get_wtime();
#else
    time3 = clock();
#endif

#ifdef _OPENMP
    cpl_msg_info(cpl_func, "end_linearize_image OPM setup %5.2e lin %5.2e", 
                 time2-time1, time3-time2);
#else
    cpl_msg_info(cpl_func, "end_linearize_image setup %5.2e lin %5.2e", 
                 (double)(time2-time1) / CLOCKS_PER_SEC,
                 (double)(time3-time2) / CLOCKS_PER_SEC);
#endif

cleanup:

    return cpl_error_get_code();    
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate a linear fit to a series of 'up the ramp' samples
  @param    samples    A struct containing the estimated samples for an image  
  @param    dit        The detector integration  time for the image
  @return   An image with the fitted slope for each pixel

  The function calculates for each pixel in an image a linear fit to 
  a series of 'up the ramp' samples. The method followed is that of
  Vacca, Cushing and Rayner, 2003. PASP, 115, 389, eqns 48 and 49.

           I = sum[i=1 to nr](si * (i - (nr + 1)/2))
                            -------------------------
                             nr * (nr + 1) * dit / 12

 */
/*----------------------------------------------------------------------------*/

cpl_image * end_uptheramp_reduce(eris_nix_samples * samples, 
                                 const double dit) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(samples, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(samples->nr > 1, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(dit > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    /* initialize result to 0 */

    cpl_image * result = cpl_image_new(
                         cpl_image_get_size_x(samples->samples[0]),
                         cpl_image_get_size_y(samples->samples[0]),
                         cpl_image_get_type(samples->samples[0]));

    /* calculate result */

    const cpl_size nr = samples->nr;
    for (cpl_size i = 1; i <= nr && cpl_error_get_code()==CPL_ERROR_NONE;
         i++) {

        cpl_image * temp = cpl_image_multiply_scalar_create(
                           samples->samples[i-1],
                           (double)i - ((double)nr + 1.0) / 2.0);
        cpl_image_add(result, temp);
        cpl_image_delete(temp);
    }
    const double alpha = (double) (nr * (nr + 1)) * dit / 12.0;
    cpl_image_divide_scalar(result, alpha);

    /* tidy up if an error has occurred */

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_image_delete(result);
        result = NULL;
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Linearize and estimate the variance of an image
  @param    gain_lin  struct containing gain/linearity info
  @param    mdark     struct containing a master dark
  @param    himage          The hdrl_image to be processed
  @param    plist           The propertylist associated with the himage
  @param    x_probe         x-coord of pixel producing debug info
  @param    y_probe         y-coord of pixel producing debug info
  @return   The linearized image or NULL on error

  The function takes a dark-subtracted image, corrects its intensity for
  the non-linear behaviour of the detector, and estimates the error plane.
  The method followed is a variant of
  Vacca, Cushing and Rayner, 2003. PASP, 115, 389.

  The variation is that the linearization is done by direct correction of
  the intensity according to the detmon curves, rather than by the Vacca
  deconstruction into samples method.

  The original routine with the full Vacca is 'end_linearize_and_variance'.

  For each pixel in the image:

     1. Linearize the intensity using the detmon curves.
     2. Divide the intensity by DIT.
     3. Estimate the image error plane following Vacca et al.  
 */
/*----------------------------------------------------------------------------*/

hdrl_image *
end_linearize_and_variance_detmon(const gain_linearity * gain_lin,
                                  const master_dark * mdark,
                                  const hdrl_image * himage,
                                  const cpl_propertylist * plist,
                                  const cpl_size x_probe,
                                  const cpl_size y_probe) {

    hdrl_image * self = NULL;
    cpl_image * linear_data;
    cpl_image * variance;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(gain_lin, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(mdark, CPL_ERROR_NULL_INPUT, NULL);

    /* get the detector window used */

    cpl_size nx = 0;
    cpl_size ny = 0;
    int rot = 0;
    cpl_size strx = 0;
    cpl_size stry = 0;
    cpl_size nx_chip = 0;
    cpl_size ny_chip = 0;
    cpl_boolean windowed = 0;
    enu_get_window_info(&nx,
                        &ny,
                        &rot,
                        &strx,
                        &stry,
                        &nx_chip,
                        &ny_chip,
                        &windowed,
                        plist);
    enu_check_error_code("failed to read detector window information");

    int probing = CPL_FALSE;
    if (x_probe >= 1 && x_probe <= nx && y_probe >=1 && y_probe <= ny) {
        probing = CPL_TRUE;
    }

    const char * det_mode = enu_get_det_mode(plist);
    enu_check_error_code("failed to read detector mode information");
    cpl_msg_info(cpl_func, "curname=%s rot=%d strx=%d stry=%d nx=%d ny=%d",
                 det_mode, (int)rot, (int)strx, (int)stry, (int)nx, (int)ny );

    /* Create the new himage */
    self = hdrl_image_duplicate(himage);
    linear_data = hdrl_image_get_image(self);
    variance = hdrl_image_get_error(self);

    if (!strcmp(det_mode, "Double_RdRstRd")) {

        /* CDS. Correlated Double Sampling for NACO.
           In this mode the intensity is calculated from the difference
           between 2 measurements at the start and end of the
           integration. */

        /* get the detector parameters required. */

        double dit = enu_get_dit(plist);
        const int ndit = cpl_propertylist_get_int(plist,
                                                  "ESO DET NDIT");
        const int nr = 2; /* is this right? */

        const double g = gain_lin->gain.data;
        const double sigma_read = cpl_propertylist_get_double(
                                       mdark->plist,
                                       "ESO QC READ NOISE");

        /* correct for non-linearity first by direct application of the
           detmon curve correction. We cannot use the Vacca method for 
           this because detmon linerization info is not appropriate */

        end_linearize_image(linear_data, gain_lin, rot, strx, stry,
                            x_probe, y_probe);

        if (probing) {
            int ignore;
            double probe_v = cpl_image_get(linear_data, x_probe, y_probe,
                                    &ignore);
            cpl_msg_info(cpl_func, "linear_data %4.2e %d", probe_v,
                         ignore);
        }

        /* The Vacca formula for the variance works with intensity 
           measured in DN /sec. */ 

        cpl_image_divide_scalar(linear_data, dit);

        /* Calculate the variance in stages */

        /* gain is in electrons/ADU.
           read-noise is in electrons. */

        /* the photon noise component of Vacca eqn 44 */

        cpl_image_copy(variance, linear_data, 1, 1);

        /* dt is the time taken to read the entire array - need to find
           out this number */

        /* actually, first set -ve numbers to zero as these make no
           sense in terms of photon noise */

        cpl_image_threshold(variance, 0.0, DBL_MAX, 0.0, DBL_MAX);
        const double dt = 0.05;
        cpl_image_multiply_scalar(variance, 
                                  (1.0 - dt * (nr * nr - 1) / 
                                  (3.0 * dit * nr)) /
                                  (g * ndit * dit));
        if (probing) { 
            cpl_msg_info(cpl_func, "photnoise factor %5.3e %d %5.3e %5.3e "
                         "%d %5.3e", dt, nr, dit, g, ndit, 
                         (1.0 - dt * (nr * nr - 1) / (3.0 * dit * nr)) /
                         (g * ndit * dit));
        }

        /* add the read noise component of Vacca eqn 44 */

        if (probing) {
            cpl_msg_info(cpl_func, "readnoise comp %5.3e %5.3e %d %d"
                         "%5.3e% 5.3e", sigma_read, g, nr, ndit, dit, 
                         2.0 * sigma_read * sigma_read /
                         (g * g * nr * ndit * dit * dit));
        }
        cpl_image_add_scalar(variance, 
                             2.0 * sigma_read * sigma_read / 
                             (g * g * nr * ndit * dit * dit));

        /* make sure bad pixels are 0 */

        cpl_image_fill_rejected(variance, 0.0);

    } else if (!strcmp(det_mode, "SLOW_UP_THE_RAMP") ||
               !strcmp(det_mode, "SLOW_GR_UTR")) {

        cpl_msg_info(cpl_func, "slow gr utr %s", cpl_error_get_message());

        /* Vacca et al. call this the 'Continuous Sampling Technique'.
           In this mode the intensity is calculated by fitting a line
           to a sequence of non-destructive reads carried out through
           the DIT. */

        /* get the detector parameters required */

        double dit = enu_get_dit(plist);
        const int ndit = cpl_propertylist_get_int(plist,
                                                  "ESO DET NDIT");
        const int nr = cpl_propertylist_get_int(plist,
                                                "ESO DET NDSAMPLES");
        const double g = gain_lin->gain.data;
        const double sigma_read = cpl_propertylist_get_double(
                                       mdark->plist,
                                       "ESO QC READ NOISE");

        /* correct for non-linearity first by direct application of the
           detmon curve correction. We cannot use the Vacca method for 
           this because detmon linerization info is derived from results
           that have been processed by the controller and are not 
           raw counts */

        end_linearize_image(linear_data, gain_lin, rot, strx, stry,
                            x_probe, y_probe);

        /* The Vacca formula for the variance works with intensity 
           measured in DN /sec. */ 

        cpl_image_divide_scalar(linear_data, dit);

        /* Calculate the variance in stages */

        /* gain is in electrons/ADU.
           read-noise is in electrons. */

        /* the photon noise component of Vacca eqn 56 */

        cpl_image_copy(variance, linear_data, 1, 1);
        cpl_image_multiply_scalar(variance, 1.2 * (nr*nr + 1) / 
                                  ((nr + 1) * nr * g * dit));

        /* add the read noise component of Vacca eqn 56 */

        cpl_image_add_scalar(variance, 
                             12.0 * sigma_read * sigma_read *
                             (nr - 1) / 
                             ((nr + 1) * nr * g * g * dit * dit));

        /* Eqn 56 does not account for the NDIT repeats, so divide
           the variance by NDIT now */

        cpl_image_divide_scalar(variance, (double) ndit);

        cpl_image_power(variance, 0.5);

        /* ensure data and error masks are compatible to avoid potential 
           warning message from hdrl_create_image */

        cpl_image_reject_from_mask(linear_data, cpl_image_get_bpm(variance));
      
   } else if (!strcmp(det_mode, "FAST_UNCORR")) {

        /* correct for non-linearity first by direct application of the
           detmon curve correction */

        end_linearize_image(linear_data, gain_lin, rot, strx, stry,
                            x_probe, y_probe);
        if (probing) {
            int ignore = 0;
            double val = cpl_image_get(linear_data, x_probe, y_probe,
                                       &ignore);
            cpl_msg_info(cpl_func, "..eris_nix_detector probe (%d %d) "
                         "linearized_data v=%5.3e", (int)x_probe, 
                         (int)y_probe, val);
        }

        /* Now calculate the variance ..
           get the detector parameters required */

        double dit = enu_get_dit(plist);
        const double ndit = (double) cpl_propertylist_get_int(plist,
                                                              "ESO DET NDIT");
        const double g = gain_lin->gain.data;
        const double sigma_read = cpl_propertylist_get_double(
                                       mdark->plist,
                                       "ESO QC READ NOISE");

        /* The Vacca formula for the variance works with intensity 
           measured in DN /sec. */ 

        cpl_image_divide_scalar(linear_data, dit);

        /* Calculate the variance as a special case of Vacca CDS treatment.
           With nr = 1, equation 44 simplifies to:

           VI =     I         +   2 * sigma_read^2
               ------------       ----------------
               g * nc * DIT         g^2 * nc * DIT^2

           first part being photon noise, second readout noise.

           The 2 in the numerator of the readout noise component derives
           from the 2 readouts in CDS ('pedestal' and 'signal') where 
           FAST_UNCORR only has 'signal'. Consequently we replace 2 by 1.

           There presumably is some error associated with the reset at
           the start of each integration but I don't know how to 
           characterise that.*/

        /* Calculate the variance in stages */

        /* gain is in electrons/ADU.
           read-noise is in electrons. */

        /* the photon noise component, can't be negative */

        cpl_image_copy(variance, linear_data, 1, 1);
        cpl_image_threshold(variance, 0.0, 1e32, 0.0, 1e32);
        cpl_image_multiply_scalar(variance, 1.0 / (g * ndit * dit));

        /* add the read noise component of Vacca eqn 56 */

        cpl_image_add_scalar(variance,
                             sigma_read * sigma_read /
                             (g * g * ndit * dit * dit));

        cpl_image_power(variance, 0.5);
        if (probing) {
            int ignore = 0;
            double val = cpl_image_get(variance, x_probe, y_probe,
                                       &ignore);
            cpl_msg_info(cpl_func, "..eris_nix_detector probe %d %d "
                         "variance %5.3e", (int)x_probe, (int)y_probe, val);
            cpl_msg_info(cpl_func, "..probe information sigma=%5.3e "
                         "g=%5.3e ndit=%d dit=%5.3e", sigma_read, g, 
                         (int)ndit, dit);
        }

        /* ensure data and error masks are compatible to avoid potential 
           warning message from hdrl_create_image */

        cpl_image_reject_from_mask(linear_data, cpl_image_get_bpm(variance));

    } else {
        cpl_msg_warning(cpl_func, "linearization not done - unsupported "
                        "detector mode: %s", det_mode);
    }

    /* tidy up */
cleanup:

    return self;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Linearize and estimate the variance of each in a list of images
  @param    gain_lin  struct containing gain/linearity info
  @param    mdark     struct containing a master dark
  @param    limlist         The list of images to be processed
  @param    x_probe         x-coord of pixel producing debug info
  @param    y_probe         y-coord of pixel producing debug info
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code

  The function takes located_image_list containing dark-subtracted images,
  corrects their intensity for the non-linear behaviour of the detector, 
  and estimates their error planes. The method followed is that of
  Vacca, Cushing and Rayner, 2003. PASP, 115, 389.

  For each image in the list:
     If the detector mode is SLOW_UP_THE_RAMP:

     1. Divide the intensity by DIT.
     2. Call 'end_vacca_linearize_cds' or 'end_vacca_linearize_ramp' to 
        correct the intensity for detector non-linearity.
     3. Estimate the image error plane using eqn 55.  
 */
/*----------------------------------------------------------------------------*/

cpl_error_code end_linearize_and_variance(const gain_linearity * gain_lin,
                                          const master_dark * mdark,
                                          located_imagelist * limlist,
                                          const cpl_size x_probe,
                                          const cpl_size y_probe) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_msg_info(cpl_func, "%p %p %p", (const void*)gain_lin,
                 (const void*)mdark, (const void*)limlist);
    cpl_ensure_code(gain_lin, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mdark, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(limlist, CPL_ERROR_NULL_INPUT);
    cpl_msg_info(cpl_func, "after %p %p %p", (const void*)gain_lin,
                 (const void*)mdark, (const void*)limlist);
    
    if (limlist->size <= 0) return CPL_ERROR_NONE;

    /* The non-linearity correction depends on f, a measure of the time in
       in the read-out sequence that each pixel is read.

       I think the result depends weakly on f, so set it to 1.0 for all 
       pixels, implying all pixels are read instantly on start of the 
       readout sequence */

    cpl_size nx = hdrl_image_get_size_x(limlist->limages[0]->himage);
    cpl_size ny = hdrl_image_get_size_y(limlist->limages[0]->himage);
    cpl_image * f = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(f, 1, 1, nx, ny, 1.0);

    int probing = CPL_FALSE;
    if (x_probe >= 1 && x_probe <= nx && y_probe >=1 && y_probe <= ny) {
        probing = CPL_TRUE;
    }

    /* Loop through images */

    for (cpl_size i = 0; i < limlist->size; i++) {

        cpl_msg_info(cpl_func, "image %d", (int) i);

        const char * det_mode = cpl_propertylist_get_string(
                                       (limlist->limages[i])->plist,
                                       "ESO DET READ CURNAME");
        const int rot = cpl_propertylist_get_int(
                                       (limlist->limages[i])->plist,
                                       "ESO DET SEQ1 WIN ROT");
        const cpl_size strx = cpl_propertylist_get_int(
                                       (limlist->limages[i])->plist,
                                       "ESO DET SEQ1 WIN STRX");
        const cpl_size stry = cpl_propertylist_get_int(
                                       (limlist->limages[i])->plist,
                                       "ESO DET SEQ1 WIN STRY");
        enu_check_error_code("failed to read detector mode information");
        cpl_msg_info(cpl_func, "curname=%s rot=%d strx=%d stry=%d nx=%d ny=%d",
                     det_mode, (int)rot, (int)strx, (int)stry, (int)nx, (int)ny );

        if (!strcmp(det_mode, "SLOW_LR_CDS")) {

            /* CDS. Correlated Double Sampling.
               In this mode the intensity is calculated from the difference
               between 2 measurements at the start and end of the
               integration. */

            /* get the detector parameters required. */

            double dit = enu_get_dit(limlist->limages[i]->plist);
            const int ndit = cpl_propertylist_get_int(
                             (limlist->limages[i])->plist, "ESO DET NDIT");
            const int nr = cpl_propertylist_get_int(
                           (limlist->limages[i])->plist, "ESO DET NDSAMPLES");
            const double g = gain_lin->gain.data;
            const double sigma_read = cpl_propertylist_get_double(
                                      mdark->plist, "ESO QC READ NOISE");

            cpl_image * data = cpl_image_duplicate(hdrl_image_get_image(
                               (limlist->limages[i])->himage));

            /* The Vacca formula works with intensity measured in DN /sec. */ 

            cpl_image_divide_scalar(data, dit);

            /* correct for non-linearity first */

            int ignore = 0;
            if (probing) {
                double probe_v = cpl_image_get(data, x_probe, y_probe,
                                               &ignore);
                cpl_msg_info(cpl_func, "data %4.2e %d", probe_v, ignore);
            }

            cpl_image * linear_data = end_vacca_linearize_cds(data,
                                      gain_lin, dit, nr, f,
                                      rot, strx, stry, x_probe, y_probe);

            if (probing) {
                double probe_v = cpl_image_get(linear_data, x_probe, y_probe,
                                        &ignore);
                cpl_msg_info(cpl_func, "linear_data %4.2e %d", probe_v,
                             ignore);
            }

            /* Calculate the variance in stages */

            /* gain is in electrons/ADU.
               read-noise is in electrons. */

            /* the photon noise component of Vacca eqn 44 */

            cpl_image * variance = cpl_image_duplicate(linear_data);

            /* dt is the time taken to read the entire array - need to find
               out this number */

            /* actually, first set -ve numbers to zero as these make no
               sense in terms of photon noise */

            cpl_image_threshold(variance, 0.0, DBL_MAX, 0.0, DBL_MAX);
            const double dt = 0.05;
            cpl_image_multiply_scalar(variance, 
                                      (1.0 - dt * (nr * nr - 1) / 
                                      (3.0 * dit * nr)) /
                                      (g * ndit * dit));
            if (probing) { 
                cpl_msg_info(cpl_func, "photnoise factor %5.3e %d %5.3e %5.3e "
                             "%d %5.3e", dt, nr, dit, g, ndit, 
                             (1.0 - dt * (nr * nr - 1) / (3.0 * dit * nr)) /
                             (g * ndit * dit));
            }

            /* add the read noise component of Vacca eqn 44 */

            if (probing) {
                cpl_msg_info(cpl_func, "readnoise comp %5.3e %5.3e %d %d"
                             "%5.3e% 5.3e", sigma_read, g, nr, ndit, dit, 
                             2.0 * sigma_read * sigma_read /
                             (g * g * nr * ndit * dit * dit));
            }
            cpl_image_add_scalar(variance, 
                                 2.0 * sigma_read * sigma_read / 
                                 (g * g * nr * ndit * dit * dit));

            /* make sure bad pixels are 0 */

            cpl_image_fill_rejected(variance, 0.0);

            /* construct the hdrl_image and replace the limlist entry */

            cpl_image_power(variance, 0.5);
            hdrl_image_delete((limlist->limages[i])->himage);
            (limlist->limages[i])->himage = hdrl_image_create(linear_data,
                                                              variance);
            /* tidy up */

            cpl_image_delete(data);
            cpl_image_delete(linear_data);
            cpl_image_delete(variance);

        } else if (!strcmp(det_mode, "SLOW_UP_THE_RAMP") ||
                   !strcmp(det_mode, "SLOW_GR_UTR")) {

            cpl_msg_info(cpl_func, "slow gr utr %s", cpl_error_get_message());

            /* Vacca et al. call this the 'Continuous Sampling Technique'.
               In this mode the intensity is calculated by fitting a line
               to a sequence of non-destructive reads carried out through
               the DIT. */

            /* get the detector parameters required */

            double dit = enu_get_dit(limlist->limages[i]->plist);
            const int ndit = cpl_propertylist_get_int(
                             (limlist->limages[i])->plist, "ESO DET NDIT");
            const int nr = cpl_propertylist_get_int(
                           (limlist->limages[i])->plist, "ESO DET NDSAMPLES");
            const double g = gain_lin->gain.data;
            const double sigma_read = cpl_propertylist_get_double(
                                      mdark->plist, "ESO QC READ NOISE");

            /* The Vacca formula works with intensity measured in DN /sec. */ 

            cpl_image * data = cpl_image_duplicate(hdrl_image_get_image(
                               (limlist->limages[i])->himage));
            cpl_image_divide_scalar(data, dit);

            /* correct for non-linearity first. This requires reconstruction
               of the ramp samples from:

              si = I * (i-f) * dt where I is DN/s
                                        i is number of read
                                        f is fractional pos. of det. in read
                                        dt is duration of read cycle
            */

            cpl_image * linear_data = end_vacca_linearize_ramp(data, 
                                      gain_lin, dit, nr, f,
                                      rot, strx, stry, x_probe, y_probe);

            /* Calculate the variance in stages */

            /* gain is in electrons/ADU.
               read-noise is in electrons. */

            /* the photon noise component of Vacca eqn 56 */

            cpl_image * variance = cpl_image_duplicate(linear_data);
            cpl_image_multiply_scalar(variance, 1.2 * (nr*nr + 1) / 
                                      ((nr + 1) * nr * g * dit));

            /* add the read noise component of Vacca eqn 56 */

            cpl_image_add_scalar(variance, 
                                 12.0 * sigma_read * sigma_read *
                                 (nr - 1) / 
                                 ((nr + 1) * nr * g * g * dit * dit));

            /* Eqn 56 does not account for the NDIT repeats, so divide
               the variance by NDIT now */

            cpl_image_divide_scalar(variance, (double) ndit);

            /* construct the hdrl_image and replace the limlist entry */

            cpl_image_power(variance, 0.5);

            /* ensure data and error masks are compatible to avoid potential 
               warning message from hdrl_create_image */

            cpl_mask * vbpm = cpl_mask_duplicate(cpl_image_get_bpm(variance));
            cpl_mask * ignore = cpl_image_set_bpm(linear_data, vbpm);
            cpl_mask_delete(ignore); ignore = NULL;

            hdrl_image_delete((limlist->limages[i])->himage);
            (limlist->limages[i])->himage = hdrl_image_create(linear_data,
                                                              variance);
            /* tidy up */

            cpl_image_delete(data);
            cpl_image_delete(linear_data);
            cpl_image_delete(variance);

        } else {
            cpl_msg_warning(cpl_func, "linearization not done - unsupported "
                            "detector mode: %s", det_mode);
        }
    }


cleanup:

    cpl_image_delete(f);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Linearize each pixel in a CDS or MCDS image. 
  @param    intensity      Image with the measured intensity 
  @param    gain_lin struct containing gain/linearity info
  @param    dit            The detector integration time
  @param    nr             The number of samples in the pedestal/signal
  @param    f              Relating to the position of each pixel in the readout sequence
  @param    x_probe        x-coord of pixel producing debug info
  @param    y_probe        y-coord of pixel producing debug info
  @return   An image withe each pixel linearized.

  The function follows an iterative procedure described by
  Vacca, Cushing and Rayner, 2003. PASP, 115, 389. 

  1. The MCDS reduction averages 'pedestal' / 'signal' samples taken at the
     start / end of the integration, then subtracts pedestal from signal,
     to give the intensity integrated over dit. To linearize, the mean 
     pedestal and signal samples are calculated, themselves linearized  
     using method 'end_linearize_image', then the intensity recalculated. 
  2. The new value for the intensity is used to re-estimate the mean 
     'pedestal' sample, and the initial measured intensity used to
     calculate the mean 'signal'. The aim is simply to bump the samples 
     closer to the correct part of the linearization regime.

  The procedure is time-consuming so a small, fixed number of iterations 
  are performed. Output pixels are set 'bad' if the change in the last 
  iteration was more than 1 per cent.
 */
/*----------------------------------------------------------------------------*/

cpl_image * end_vacca_linearize_cds(cpl_image * intensity, 
                                    const gain_linearity * gain_lin,
                                    const double dit,
                                    const cpl_size nr,
                                    const cpl_image * f,
                                    cpl_size rot,
                                    cpl_size strx,
                                    cpl_size stry,
                                    cpl_size x_probe,
                                    cpl_size y_probe) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_ensure(intensity, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(gain_lin, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(nr > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_image * change = NULL;
    cpl_image * result = NULL;

    cpl_size nx = cpl_image_get_size_x(intensity);
    cpl_size ny = cpl_image_get_size_y(intensity);

    int probing = CPL_FALSE;
    if (x_probe >= 1 && x_probe <= nx && y_probe >=1 && y_probe <= ny) {
        probing = CPL_TRUE;
    }

    /* now do the iterative linearization */

    cpl_image ** iteration = cpl_calloc(LINEARIZE_MAXITER,
                                        sizeof(cpl_image *));
    cpl_size niter = 0;
    iteration[0] = cpl_image_duplicate(intensity);

    /* dt is the time taken to read the entire array - need to find
       out this number! */

    const double dt = 0.05;

    while (niter+1 < LINEARIZE_MAXITER && 
           cpl_error_get_code() == CPL_ERROR_NONE) {

        /* estimate the pedestal values following Vacca eqn 23. For NIX
           the equations look different as intensity = Stot / nr * nc * dit */

        cpl_image * pedestal = cpl_image_multiply_scalar_create(f, -1.0);
        cpl_image_add_scalar(pedestal, (double) (nr + 1) / 2);
        cpl_image_multiply(pedestal, iteration[niter]);
        cpl_image_multiply_scalar(pedestal, dt);
        cpl_image * signal = cpl_image_multiply_scalar_create(iteration[0],
                                                              dit);
        cpl_image_add(signal, pedestal);

        if (probing) {
            int ignore = 0;
            double probe_val = cpl_image_get(iteration[niter], x_probe,
                                             y_probe, &ignore);
            if (!ignore) {
                double probe_p = cpl_image_get(pedestal, x_probe, y_probe,
                                               &ignore);
                double probe_s = cpl_image_get(signal, x_probe, y_probe,
                                               &ignore);
                cpl_msg_info(cpl_func, "linearizing: (pix %d,%d) niter=%d "
                             "I=%4.2e pedestal=%4.2e signal=%4.2e",
                             (int)x_probe, (int)y_probe, (int)niter,
                             probe_val, probe_p, probe_s);
            } else {
                cpl_msg_info(cpl_func, "linearizing: (pix %d,%d) niter=%d "
                             "bad pixel", (int)x_probe, (int)y_probe, 
                             (int)niter);
            }
        } 

        /* linearize the samples in the current estimate */

        end_linearize_image(pedestal, gain_lin, rot, strx, stry,
                            x_probe, y_probe);
        end_linearize_image(signal, gain_lin, rot, strx, stry,
                            x_probe, y_probe);

        /* calculate new value for intensity */

        niter++;
        iteration[niter] = cpl_image_subtract_create(signal, pedestal);
        cpl_image_divide_scalar(iteration[niter], dit);
        cpl_image_delete(pedestal);
        cpl_image_delete(signal);
    }

    result = cpl_image_duplicate(iteration[niter]);
    if (niter > 0) {

        /* Set the image mask bad where the linearization did not
           converge; more than 1 per cent change in last iteration */

        change = cpl_image_divide_create(iteration[niter],
                                         iteration[niter-1]);
        cpl_mask_threshold_image(cpl_image_get_bpm(result), change, 0.99,
                                 1.01, CPL_BINARY_0);

        /* Set bad pixels to 0 */

        cpl_image_fill_rejected(result, 0.0);
    }

    /* tidy up memory */

    for (cpl_size i = 0; i < LINEARIZE_MAXITER; i++) {
        cpl_image_delete(iteration[i]);
    }
    cpl_free(iteration);
    cpl_image_delete(change);

    /* return NULL on error */

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_image_delete(result);
        result = NULL;   
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Linearize each pixel in an 'up the ramp' image. 
  @param    intensity      Image with the measured intensity 
  @param    gain_lin struct containing gain/linearity info
  @param    dit            The detector integration time
  @param    nr             The number of samples in the 'ramp'
  @param    f              Relating to the position of each pixel in the readout sequence
  @param    x_probe        x-coord of pixel producing debug info
  @param    y_probe        y-coord of pixel producing debug info
  @return   An image withe each pixel linearized.

  The function follows an iterative procedure hinted at but not described by
  Vacca, Cushing and Rayner, 2003. PASP, 115, 389. 

  1. The 'up the ramp' reduction has fitted a line to samples spaced through
     the DIT. These samples are estimated, linearized using a basic curve
     in method 'end_linearize_image', then the line re-fit to give an updated
     (higher) value for the intensity. 
  2. The new value for the intensity is used to re-estimate the _first_
     sample of the sequence, subsequent samples retain the slope matching 
     the initial intensity. The idea is simply to bump the samples closer 
     to the correct part of the linearization regime.

  The procedure is time-consuming so a small, fixed number of iterations 
  are performed. Output pixels are set 'bad' if the change in the last 
  iteration was more than 1 per cent.
 */
/*----------------------------------------------------------------------------*/

cpl_image * end_vacca_linearize_ramp(cpl_image * intensity, 
                                     const gain_linearity * gain_lin,
                                     const double dit,
                                     const cpl_size nr,
                                     const cpl_image * f,
                                     const cpl_size rot,
                                     const cpl_size strx,
                                     const cpl_size stry,                                     
                                     const cpl_size x_probe,
                                     const cpl_size y_probe) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

#ifdef _OPENMP
    double time1, time2, time3, time4, time11, time12, time13, time14;
    time1 = omp_get_wtime();
#else
    clock_t time1, time2, time3, time4, time11, time12, time13, time14;
    time1 = clock();
#endif

    cpl_ensure(intensity, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(gain_lin, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(nr > 1, CPL_ERROR_ILLEGAL_INPUT, NULL);

    cpl_image * result = NULL;
    cpl_image * change = NULL;

    cpl_size nx = cpl_image_get_size_x(intensity);
    cpl_size ny = cpl_image_get_size_y(intensity);
    int probing = CPL_FALSE;
    cpl_msg_info(cpl_func, "%d %d %d %d", (int)x_probe, (int)y_probe, (int)nx, (int)ny);
    if (x_probe >= 1 && x_probe <= nx && y_probe >=1 && y_probe <= ny) {
        probing = CPL_TRUE;
        cpl_msg_info(cpl_func, "messaging on");
    }
    double s = 0.0;
    int reject = 0;

    /* now do the iterative linearization */

    cpl_image ** iteration = cpl_calloc(LINEARIZE_MAXITER,
                                        sizeof(cpl_image *));
    cpl_size niter = 0;
    iteration[0] = cpl_image_duplicate(intensity);
    if (probing) {
        s = cpl_image_get(intensity, x_probe, y_probe, &reject);
        cpl_msg_info(cpl_func, "initial %s %5.2f", cpl_error_get_message(), s);
    }

#ifdef _OPENMP
    time2 = omp_get_wtime();
#else
    time2 = clock();
#endif

    while (niter+1 < LINEARIZE_MAXITER && 
           cpl_error_get_code() == CPL_ERROR_NONE) {

        /* estimate the sample values following Vacca eqn 49 */

#ifdef _OPENMP
        time11 = omp_get_wtime();
#else
        time11 = clock();
#endif
        eris_nix_samples * samples = end_calculate_samples(iteration[niter],
                                                           iteration[0],
                                                           nr, dit, f);

        if (probing) {
            cpl_msg_info(cpl_func, "samples before linearization: is, reject, s"); 
            for (cpl_size is=0; is<samples->nr; is++) {
                s = cpl_image_get(samples->samples[is], x_probe, y_probe, &reject);
                cpl_msg_info(cpl_func, "%d %d %5.2f", (int)is, reject, s);
            }
        }

        /* linearize the samples in the current estimate */

#ifdef _OPENMP
        time12 = omp_get_wtime();
#else
        time12 = clock();
#endif
        for (cpl_size i = 0; i < nr; i++) {
            end_linearize_image(samples->samples[i], gain_lin, rot,
                                strx, stry, x_probe, y_probe);
        }

        if (probing) {
            cpl_msg_info(cpl_func, "samples after: is, reject, s"); 
            for (cpl_size is=0; is<samples->nr; is++) {
                s = cpl_image_get(samples->samples[is], x_probe, y_probe, &reject);
                cpl_msg_info(cpl_func, "%d %d %5.2f", (int)is, reject, s);
            }
        }

        /* calculate new value for intensity */

#ifdef _OPENMP
        time13 = omp_get_wtime();
#else
        time13 = clock();
#endif
        niter++;
        iteration[niter] = end_uptheramp_reduce(samples, dit);

        if (probing) {
            s = cpl_image_get(iteration[niter], x_probe, y_probe, &reject);
            cpl_msg_info(cpl_func, "iteration %s %d %9.4e", cpl_error_get_message(),
                         (int)niter, s);

        }

#ifdef _OPENMP
        time14 = omp_get_wtime();
        cpl_msg_info(cpl_func, "%d ramp samp %5.2e lin %5.2e calc %5.2e", 
                     (int)niter, 
                     time12-time11,
                     time13-time12,
                     time14-time13);
#else
        time14 = clock();
        cpl_msg_info(cpl_func, "%d ramp samp %5.2e lin %5.2e calc %5.2e", 
                     (int)niter, 
                     (double)(time12-time11) / CLOCKS_PER_SEC,
                     (double)(time13-time12) / CLOCKS_PER_SEC,
                     (double)(time14-time13) / CLOCKS_PER_SEC);
#endif
        end_samples_delete(samples);
    }

#ifdef _OPENMP
    time3 = omp_get_wtime();
#else
    time3 = clock();
#endif

    result = cpl_image_duplicate(iteration[niter]);

    if (probing) {
        s = cpl_image_get(result, x_probe, y_probe, &reject);
        cpl_msg_info(cpl_func, "result %s %5.2f", cpl_error_get_message(), s);
    }

    if (niter > 0) {

        /* Set the image mask bad where the linearization did not
           converge; more than 1 per cent change in last iteration */

        change = cpl_image_divide_create(iteration[niter],
                                         iteration[niter-1]);
        cpl_mask_threshold_image(cpl_image_get_bpm(result), change, 0.99,
                                 1.01, CPL_BINARY_0); 
    }

    /* tidy up memory */

    for (cpl_size i = 0; i < LINEARIZE_MAXITER; i++) {
        cpl_image_delete(iteration[i]);
    }
    cpl_free(iteration);
    cpl_image_delete(change);

    /* return NULL on error */

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_image_delete(result);
        result = NULL;   
    }

#ifdef _OPENMP
    time4 = omp_get_wtime();
    cpl_msg_info(cpl_func, "ramp setup %5.2e lin %5.2e tidy %5.2e", 
                 time2-time1,
                 time3-time2,
                 time4-time3);
#else
    time4 = clock();
    cpl_msg_info(cpl_func, "ramp setup %5.2e lin %5.2e tidy %5.2e", 
                 (double)(time2-time1) / CLOCKS_PER_SEC,
                 (double)(time3-time2) / CLOCKS_PER_SEC,
                 (double)(time4-time3) / CLOCKS_PER_SEC);
#endif

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Apply the linearization correction to a pixel.
  @param    obs               The value to be linearized
  @param    fit_order         The order of the fit polynomial, at least 2
  @param    coeffs            The linearity polynomial coeffs, c0=0
  @param    fit_quality       The quality of the fit
  @param    saturation        Saturation level for array
  @param    rejected          Set to CPL_BINARY_0 on success, CPL_BINARY_1 otherwise
  @param    saturated         Set to CPL_BINARY_1 if saturated, CPL_BINARY_0 otherwise
  @param    debug             True for debug info to be printed
  @return   The corrected pixel. -1 if the linearization fails.

  The function applies the linearity correction from the gain_linearity
  structure to the given pixel.
 */
/*----------------------------------------------------------------------------*/

double engl_lin_correct(const double obs,
                        const cpl_size fit_order,
                        const double* coeffs,
                        const cpl_binary fit_quality, 
                        const double saturation,
                        cpl_binary * rejected,
                        cpl_binary * saturated,
                        const int debug) {

    double result = obs;
    if (cpl_error_get_code()) return 0.0;
    if (debug)
        cpl_msg_info(cpl_func, "Saturation ptr: %p", (const void*)saturated);

    engl_lin_correct_(&result,
                      fit_order,
                      coeffs,
                      fit_quality, 
                      saturation,
                      rejected,
                      debug);
    return result;

}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Apply the linearization correction to a non-bad pixel.
  @param    pobs              The value to be linearized, zero on failure
  @param    fit_order         The order of the fit polynomial, at least 2
  @param    coeffs            The linearity polynomial coeffs
  @param    fit_quality       The quality of the fit
  @param    saturation        Saturation level for array
  @param    rejected          Set to CPL_BINARY_1 on failure
  @param    debug             True for debug info to be printed
  @return   Nothing

  The function applies the linearity correction from the gain_linearity
  structure to the given pixel.
 */
/*----------------------------------------------------------------------------*/
inline static
void engl_lin_correct_(double * pobs,
                       const cpl_size fit_order,
                       const double* coeffs,
                       const cpl_binary fit_quality, 
                       const double saturation,
                       cpl_binary * rejected,
                       const int debug) {

    if (fit_quality == CPL_BINARY_1) {

        if (debug) cpl_msg_info(cpl_func, "....debug bad quality");
        *rejected = CPL_BINARY_1;

    } else if (coeffs[1] <= 0.0) {

        /* Negative sloping curve makes no sense */

        if (debug) cpl_msg_info(cpl_func, "....debug bad slope");
        *rejected = CPL_BINARY_1;

    } else if (*pobs > 0.0) {

        /* find the t value for where the obs value sits on the linearisation
           curve. Points outside curve range are set bad */

        /* In saturation correct the pixel as it it was _at_ saturation. The
           aim of this is to avoid introducing discontinuities to the image */

        const double useobs = *pobs > saturation ? saturation : *pobs;
        double curve = 0.0;
        /* On failure t is zero */
        const double t = engl_lin_find_(*pobs, 0.9 * useobs / coeffs[1],
                                        1.5 * useobs / coeffs[1],
                                        fit_order, coeffs, &curve); 
        /* Calculate the correction and apply it (zero on failure) */
        const double correction = (coeffs[1] * t) / curve;

        *pobs *= correction;
        *rejected = t > 0.0 ? CPL_BINARY_0 : CPL_BINARY_1;

        if (debug) {
            if (useobs == saturation)
                cpl_msg_info(cpl_func, "....debug saturated");

            for (int j = fit_order-1; j>-1; j--) {
                cpl_msg_info(cpl_func, "....debug curve coeffs %d %5.3e",
                             j, coeffs[j]);
            }
            cpl_msg_info(cpl_func, "....debug t %4.2f", t);
            cpl_msg_info(cpl_func, "....debug correction %6.4e %4.2e "
                         "%4.2e", correction, *pobs, curve);
        }

    } else if (debug) {

        /* Negative values are off the linearization curve, do nothing */

        cpl_msg_info(cpl_func, "....debug negative");
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Find 't' for where the linearisation curve has a specified value.
  @param    obs               The observed value
  @param    low_t             The lower bound expected for t
  @param    high_t            The upper bound expected for t
  @param    fit_order         The order of the curve polynomial
  @param    coeffs            The linearity curve polynomial coefficients
  @return   The corrected pixel, or -1 if the process fails. 

  The function is called recursively to locate the t at which the given 
  linearisation has the specified value.
 */
/*----------------------------------------------------------------------------*/

double engl_lin_find(const double obs,
                     double low_t,
                     double high_t,
                     const cpl_size fit_order, 
                     const double* coeffs) {

    double dummy = 0.0;
    return cpl_error_get_code() ? -1.0 :
        engl_lin_find_(obs, low_t, high_t, fit_order, coeffs, &dummy);
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Find 't' for where the linearisation curve has a specified value.
  @param    obs               The observed value
  @param    low_t             The lower bound expected for t
  @param    high_t            The upper bound expected for t
  @param    fit_order         The order of the curve polynomial, at least 2
  @param    coeffs            The linearity curve polynomial coefficients, c0=0
  @param    pcurve            The curve at the computed solution, or zero
  @return   The corrected pixel, or zero if the process fails. 
  @note     The polynomial constant term is assumed to be zero

  The function is called recursively to locate the t at which the given 
  linearisation has the specified value.
 */
/*----------------------------------------------------------------------------*/

inline static
double engl_lin_find_(const double obs,
                      double low_t,
                      double high_t,
                      const cpl_size fit_order, 
                      const double* coeffs,
                      double * pcurve) {

    double result = 0.0;

    cpl_size iter = 0;
    double low_curve = 0.0;
    double high_curve = 0.0;

    *pcurve = 0.0;
    do {

        /* calculate values of curve as required at low_t, high_t and
           mid_t */
 
        if (low_curve == 0.0) {
            int j = fit_order-2; /* Peel off iterations below 4 */
            low_curve = coeffs[fit_order-1];
            for (; j >= 3; j--) {
                low_curve = low_curve * low_t + coeffs[j];
            }
            switch (j) {
            case 2:
                low_curve = low_curve * low_t + coeffs[2];
                CPL_ATTR_FALLTRHU; /* fall through */
            case 1:
                low_curve = low_curve * low_t + coeffs[1];
                CPL_ATTR_FALLTRHU; /* fall through */
            case 0:
                low_curve *= low_t; /* Constant term is zero */
            }
        }
        if (high_curve == 0.0) {
            int j = fit_order-2; /* Peel off iterations below 4 */
            high_curve = coeffs[fit_order-1];
            for (; j >= 3; j--) {
                high_curve = high_curve * high_t + coeffs[j];
            }
            switch (j) {
            case 2:
                high_curve = high_curve * high_t + coeffs[2];
                CPL_ATTR_FALLTRHU; /* fall through */
            case 1:
                high_curve = high_curve * high_t + coeffs[1];
                CPL_ATTR_FALLTRHU; /* fall through */
            case 0:
                high_curve *= high_t; /* Constant term is zero */
            }
        }
        double mid_t = (low_t + high_t) / 2.0;
        double mid_curve = coeffs[fit_order-1];
        int j = fit_order-2; /* Peel off iterations below 4 */
        for (; j >= 3; j--) {
            mid_curve = mid_curve * mid_t + coeffs[j];
        }
        switch (j) {
        case 2:
            mid_curve = mid_curve * mid_t + coeffs[2];
            CPL_ATTR_FALLTRHU; /* fall through */
        case 1:
            mid_curve = mid_curve * mid_t + coeffs[1];
            CPL_ATTR_FALLTRHU; /* fall through */
        case 0:
            mid_curve *= mid_t; /* Constant term is zero */
        }

        /* decide what to do based on position of obs */

        if (fabs(obs - mid_curve) < MAX_LIN_DIFFERENCE) {
            *pcurve = mid_curve;
            result = mid_t;
            break;
        } else if ((obs > low_curve) && (obs < mid_curve)) {
            high_t = mid_t;
            high_curve = 0.0;
        } else if ((obs > mid_curve) && (obs < high_curve)) {
            low_t = mid_t;
            low_curve = 0.0;
        } else {

            /* point is outside range of curve */

            break;
        }

    } while (iter++ < MAX_LIN_DEPTH);

    /* trap failure to reach result inside max number of iterations */

    if (iter >= MAX_LIN_DEPTH) {
        cpl_msg_info(cpl_func, "max depth %4.2e %4.2e %4.2e %d %4.2e %4.2e "
                     "%4.2e", obs, low_t, high_t,
                     (int)fit_order, coeffs[0], coeffs[1],coeffs[3]);
    }

    return result;
}


/**@}*/
