/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <hdrl.h>
#include "eris_nix_dfs.h"
#include "eris_nix_utils.h"
#include "eris_pfits.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_dfs     DFS related functions
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Set the group as RAW or CALIB in a frameset
  @param    set     the input frameset
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_nix_dfs_set_groups(cpl_frameset * set) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_frame * frame = NULL;
    int i = 0;

    /* Loop on frames */

    cpl_frameset_iterator * fsiter = cpl_frameset_iterator_new(set);
    frame = cpl_frameset_iterator_get(fsiter);
    while (frame) {

        const char * tag = cpl_frame_get_tag(frame);
        int data_found = 0;

        if (tag == NULL) {
            cpl_msg_error(cpl_func, "Frame %d has no tag", i);
        } else if (!strcmp(tag, ERIS_NIX_RAW_DARK_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_PERSISTENCE_IMG_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PERSISTENCE_CUBE_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_FLAT_LAMP_ON_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_FLAT_LAMP_OFF_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_FLAT_SKY_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_FLAT_SKY_LSS_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_FLAT_TWILIGHT_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_LAMP_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_SKY_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_DARK_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_BKG_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_LAMP_OPEN_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_LAMP_MASK_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_SKY_OPEN_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_SKY_MASK_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_DARK_OPEN_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_PUPIL_DARK_MASK_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_OBJECT_APP_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_OBJECT_FPC_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_OBJECT_SAM_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_OBJECT_JITTER_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_SKY_FPC_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_SKY_APP_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_SKY_JITTER_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_STD_JITTER_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_OBJECT_LSS_JITTER_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_STD_APP_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_STD_FPC_DO_CATG) ||
				   !strcmp(tag, ERIS_NIX_RAW_STD_SAM_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_SKY_LSS_JITTER_DO_CATG) ||
                   !strcmp(tag, ERIS_NIX_RAW_STD_LSS_JITTER_DO_CATG)) {

            cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
            data_found = 1;

        } else if (!strcmp(tag, ERIS_NIX_COEFFS_CUBE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_DET_LIN_INFO_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_GAIN_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_BPM_LSS_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_BPM_SKY_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_DARK_IMG_PRO_CATG) ||
				   !strcmp(tag, ERIS_NIX_MASTER_DARK_IFU_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_LAMP_LOFREQ_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_LSS_HIFREQ_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_LSS_LOFREQ_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_SKY_LOFREQ_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_SKY_HIFREQ_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_MASTER_FLAT_TWILIGHT_LOFREQ_PRO_CATG)||
                   !strcmp(tag, ERIS_NIX_MASTER_STARTRACE_PRO_CATG)||
                   !strcmp(tag, ERIS_NIX_MASTER_WAVE_PRO_CATG)||
                   !strcmp(tag, ERIS_NIX_NL_BPM_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_PHOT_DATA_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_WCS_MATCHED_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_WCS_REFINE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_REFERENCE_DARK_PRO_CATG)) {

            cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
            data_found = 1;

        } else if (!strcmp(tag, ERIS_NIX_CALIB_OBJECT_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_DET_OBJECT_LSS_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_DET_SKY_LSS_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_DET_STD_LSS_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_PHOT_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_PHOT_MATCH_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_PHOT_REF_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_WCS_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_WCS_MATCH_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_WCS_REF_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_WCS_REPORT_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CAL_WCS_STD_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CORRECTED_OBJECT_LSS_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_CORRECTED_STD_LSS_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_IMG_CATALOGUE_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_IMG_OBS_COMBINED_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_IMG_STD_COMBINED_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_LSS_OBS_COMBINED_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_LSS_STD_COMBINED_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_SKYSUB_OBJECT_LSS_JITTER_PRO_CATG) ||
                   !strcmp(tag, ERIS_NIX_SKYSUB_STD_LSS_JITTER_PRO_CATG)) {

/*
            cpl_frame_set_group(frame, CPL_FRAME_GROUP_PRODUCT);
*/
            cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
            data_found = 1;
        } else {
        	cpl_frame_set_group(frame, CPL_FRAME_GROUP_NONE);
        	data_found = 1;
        }

        if (!data_found) {
		if(tag != NULL) {
                    cpl_error_set_message("eris_nix_dfs_set_groups",
                                  CPL_ERROR_UNSUPPORTED_MODE,
                                  "data tag not recognised: %s", tag);
		}
            break;
        }       

        cpl_frameset_iterator_advance(fsiter, 1);
        frame = cpl_frameset_iterator_get(fsiter);
        i++;
    }
    enu_check_error_code("Could not identify RAW and CALIB frames");
   
 cleanup:
    cpl_frameset_iterator_delete(fsiter);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**Routine to save a master_bpm to a DFS-compliant FITS file. A wrapper for 
   cpl_dfs_save_image.

  @brief  Save a master_bpm as a DFS-compliant pipeline product
  @param  procat     The product CATG
  @param  allframes  The list of input frames for the recipe
  @param  parlist    The list of input parameters
  @param  usedframes The list of raw/calibration frames used for this product
  @param  mbad_pix_map The master_bpm to be saved
  @param  recipe     The recipe name
  @param  applist    Propertylist to append to primary header
  @param  pipe_id    PACKAGE "/" PACKAGE_VERSION
  @param  filename   Filename of created product
  @return CPL_ERROR_NONE or the relevant CPL error code on error

 */
/*----------------------------------------------------------------------------*/
cpl_error_code enu_dfs_save_bpm(
      const char              * procat,
      cpl_frameset            * allframes,
      const cpl_parameterlist * parlist,
      const cpl_frameset      * usedframes,
      const master_bpm        * mbad_pix_map,
      const char              * recipe,
      const cpl_propertylist  * applist,
      const char              * pipe_id,
      const char              * filename) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_propertylist * plist = NULL;

    if (applist == NULL) {
        plist = cpl_propertylist_new();
    } else {
        plist = cpl_propertylist_duplicate(applist);
    }
    cpl_propertylist_append_string(plist, CPL_DFS_PRO_CATG, procat);
    cpl_dfs_save_image(allframes, NULL, parlist, usedframes, NULL,
      mbad_pix_map->bpm, CPL_TYPE_INT, recipe, plist, NULL, pipe_id, filename);
    cpl_ensure_code(!cpl_error_get_code(), cpl_error_get_code());

    /* Cleanup */

    cpl_propertylist_delete(plist);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**Routine to write an hdrl_image data product to a MEF (DFS-compliant)
  file. Adapted from cpl_dfs_product_save.

  @brief    Save a catalogue as DFS product
  @param    frameset   The list of input frames for the recipe
  @param    image      Frame to which the catalogue refers
  @param    parlist    The list of input parameters 
  @param    catalogue  The catalogue to be saved
  @param    tablelist  NULL or optional propertylist to use in table extension
  @param    cat_name   The name of the catalogue to be saved
  @param    recipe     The recipe name 
  @param    pipe_id    PACKAGE "/" PACKAGE_VERSION    
  @param    preface    The catalogue filename will be "<preface>.<raw filename>"
  @param    procat     The DFS category of the product
  @return   CPL_ERROR_NONE on success, a CPL error code otherwise

  The function saves the table to a DFS-compliant FITS file of name
  "<preface>.<inherit frame filename>", with keyword DPR.CATG set to the
  specified value. If the catalogue is NULL no action is taken.
 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_dfs_save_catalogue(cpl_frameset * frameset,
                                      cpl_frame * image,
                                      const cpl_parameterlist * parlist,
                                      const cpl_table * catalogue,
                                      const cpl_propertylist * tablelist,
                                      const char * cat_name,
                                      const char * recipe,
                                      const char * pipe_id,
                                      const char * preface,
                                      const char * procat) {

    /* immediate return for some conditions */

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    if (!catalogue) return CPL_ERROR_NONE; 

    /* otherwise, check inputs */

    cpl_ensure_code(image, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cat_name, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(preface, CPL_ERROR_NULL_INPUT);

    cpl_propertylist * applist = NULL;
    cpl_propertylist * image_plist = NULL;
    char * prodname = NULL;
    cpl_frame * product_frame = NULL;
    cpl_propertylist * tablelist_copy = NULL;

    /* ..create a product frameset containing the input image of the 
         catalogue */

    cpl_frameset * provenance = cpl_frameset_new();
    cpl_frameset_insert(provenance, image);

    /* ..create the product filename and frame */

    prodname = enu_repreface(cpl_frame_get_filename(image), preface);
    product_frame = cpl_frame_new();
    cpl_frame_set_filename(product_frame, prodname);
    cpl_frame_set_tag(product_frame, procat);
    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, CPL_FRAME_LEVEL_FINAL);

    /* ..set up the phase3/IDP keywords for the primary header */

    applist = cpl_propertylist_new();

    /* ..copy some items from the image header */

    image_plist = cpl_propertylist_load(cpl_frame_get_filename(image), 0);
    cpl_propertylist_copy_property_regexp(applist,
                                          image_plist,
                                          "ORIGIN|TELESCOP|INSTRUME|FILTER|"
                                          "OBJECT|RA|DEC|EQUINOX|RADESYS|"
                                          "TIMESYS|EXPTIME|TEXPTIME|MJD-OBS|"
                                          "MJD-END|PROG_ID|OBID[1-9]([0-9])*|"
                                          "NCOMBINE|OBSTECH|PROCSOFT|"
                                          "REFERENC",
                                          CPL_FALSE);
    enu_check_error_code("error copying image HDU info to applist");

    /* ..add other standard keywords */

    enu_dfs_setup_product_header(applist,
                                 product_frame,
                                 frameset,
                                 parlist,
                                 recipe,
                                 pipe_id,
                                 image,
                                 provenance,
                                 CPL_FALSE);         

    /* ..set new catg */

    cpl_propertylist_update_string(applist, "PRODCATG", "SCIENCE.SRCTBL");
    cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procat);

    /* ..Point the PROV1 keyword to the associated image */

    if (cpl_frame_get_filename(image)) {
        char * filename = cpl_strdup(cpl_frame_get_filename(image));
        cpl_propertylist_update_string(applist, "PROV1", basename(filename));
        cpl_free(filename);
    }

    /* ..and set the plist for the table extension */

    if (tablelist != NULL) {
        tablelist_copy = cpl_propertylist_duplicate(tablelist);
    } else {
        tablelist_copy = cpl_propertylist_new();
    }
    cpl_propertylist_update_string(tablelist_copy, "EXTNAME", "PHASE3CATALOG");

    /* OK, save the table */

    cpl_table_save(catalogue,
                   applist,
                   tablelist_copy,
                   prodname,
                   CPL_IO_CREATE); 	

    enu_check_error_code("error saving dfs table");

    /* save the product frame to the overall frameset so that DFS knows a new
       product has been created */

    cpl_frameset_insert(frameset, cpl_frame_duplicate(product_frame));

 cleanup:

    cpl_propertylist_delete(applist);
    cpl_propertylist_delete(image_plist);
    cpl_free(prodname);
    cpl_frame_delete(product_frame);
    cpl_propertylist_delete(tablelist_copy);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**Routine to write an hdrl_image or hdrl_imagelist data product to a MEF
  (DFS-compliant) file. Adapted from cpl_dfs_product_save.

  @brief  Save an hdrl_image/imagelist as a DFS-compliant MEF pipeline product

  @param  allframes   The list of input frames for the recipe
  @param  parlist     The list of input parameters
  @param  usedframes  The raw/calibration frames used for this product
  @param  provenance  The frames that contributed directly to this product
  @param  prov_raw    CPL_TRUE for raw data filenames, CPL_FALSE for actual names 
  @param  image       The image to be saved or...
  @param  imagelist   ...the imagelist to be saved
  @param  mefs        List of extension components associated with the image
  @param  recipe      The recipe name
  @param  inherit     See cpl_dfs_setup_product_header.
  @param  applist     Propertyies to add to primary header, w. PRO.CATG
  @param  wcs_list    Properties defining image wcs  
  @param  pipe_id     PACKAGE "/" PACKAGE_VERSION
  @param  filename    Filename of created product

  @return CPL_ERROR_NONE or the relevant CPL error code on error

 */
/*----------------------------------------------------------------------------*/
cpl_error_code enu_dfs_save_himage(cpl_frameset            * allframes,
                                   const cpl_parameterlist * parlist,
                                   const cpl_frameset      * provenance,
                                   const cpl_boolean         prov_raw,
                                   const hdrl_image        * image,
                                   const hdrl_imagelist    * imagelist,
                                   const mef_extension_list* mefs,
                                   const char              * recipe,
                                   const cpl_frame         * inherit_frame,
                                   const cpl_propertylist  * applist,
                                   const cpl_propertylist  * wcs_plist,
                                   const char              * pipe_id,
                                   const char              * filename) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_msg_debug(cpl_func, "enu_dfs_save_himage entry");

    cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(provenance, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pipe_id, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);

    const cpl_frame * inherit = NULL;
    cpl_propertylist * inherit_plist = NULL;
    cpl_frameset * product_frameset = NULL;
    cpl_propertylist * plist = NULL;

    int conf = CPL_FALSE;
    int bkgd = CPL_FALSE;
    int bkge = CPL_FALSE;
    int bkgc = CPL_FALSE;

    cpl_propertylist * dicd = NULL;

    /* find what optional extensions are present */

    if (mefs) {
        for (cpl_size i=0; i<mefs->size; i++) {
            cpl_msg_info(cpl_func, "%d", (int)i);
            cpl_msg_info(cpl_func, "%s", mefs->mef[i]->name);
            if (!strcmp(mefs->mef[i]->name, "CONFIDENCE")) conf = CPL_TRUE;
            if (!strcmp(mefs->mef[i]->name, "BKG_DATA")) bkgd = CPL_TRUE;
            if (!strcmp(mefs->mef[i]->name, "BKG_ERR")) bkge = CPL_TRUE;
            if (!strcmp(mefs->mef[i]->name, "BKG_CONF")) bkgc = CPL_TRUE;
        }
    }

    /* the code generally follows the procedure of cpl_dfs_product_save */

    /* ..report what is happening */

    const char * procat = cpl_propertylist_get_string(applist,
                                                      CPL_DFS_PRO_CATG);
    cpl_ensure_code(procat != NULL, cpl_error_get_code());
    cpl_msg_info(cpl_func, "Writing MEF product(%s): %s", procat, filename);

    /* ..construct a frameset for the product */

    product_frameset = cpl_frameset_duplicate(provenance);

    /* ..if inherit_frame is NULL, use the first frame in the frameset */

    if (inherit_frame) {
        inherit = inherit_frame;
    } else {
        cpl_frameset_iterator * frameset_iter = cpl_frameset_iterator_new(
                                                allframes);
        inherit = cpl_frameset_iterator_get_const(frameset_iter);
        cpl_frameset_iterator_delete(frameset_iter);
    }

    /* ..create product frame and insert it in the product frameset */

    cpl_frame * product_frame = cpl_frame_new();
    cpl_frame_set_filename(product_frame, filename);
    cpl_frame_set_tag(product_frame, procat);
    cpl_frame_set_type(product_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(product_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(product_frame, CPL_FRAME_LEVEL_FINAL);

    cpl_frameset_insert(product_frameset, product_frame);

    /* ..construct the propertylist for the primary header... */

    plist = cpl_propertylist_duplicate(applist);
    enu_dfs_setup_product_header(plist,
                                 product_frame,
                                 product_frameset,
                                 parlist,
                                 recipe,
                                 pipe_id,
                                 inherit,
                                 provenance,
                                 prov_raw);         

    /* ..save the propertylist to HDU 0 */
    cpl_propertylist_update_string(plist, "ESO PRO REC1 ID", recipe);

    cpl_propertylist_save(plist, filename, CPL_IO_CREATE);
    enu_check_error_code("error saving header 0");

    /* Now for the extensions 
       ..construct a list of the DICD keywords common to all extensions */ 

    dicd = cpl_propertylist_new();
    cpl_propertylist_append_string(dicd, "HDUCLASS", "ESO");
    cpl_propertylist_append_string(dicd, "HDUDOC", "SDP");
    cpl_propertylist_append_string(dicd, "HDUVERS", "SDP version 8");
    cpl_propertylist_append_string(dicd, "SCIDATA", "DATA");
    cpl_propertylist_append_string(dicd, "ERRDATA", "ERR");
    cpl_propertylist_append_string(dicd, "QUALDATA", "DQ");
    if (conf) {
        cpl_propertylist_append_string(dicd, "CONFDATA", "CONFIDENCE");   
    }
    if (bkgd) {
        cpl_propertylist_append_string(dicd, "BKGDATA", "BKG_DATA");   
    }
    if (bkge) {
        cpl_propertylist_append_string(dicd, "BKGERR", "BKG_ERR");   
    }
    if (bkgc) {
        cpl_propertylist_append_string(dicd, "BKGCONF", "BKG_CONF");   
    }
    cpl_size hlist_sz = 0;
    if (image || imagelist) {

        /* ..save image data to HDU 1, copy wcs header info only */

        cpl_propertylist * header = cpl_propertylist_duplicate(dicd);
        cpl_propertylist_update_string(header, "EXTNAME", "DATA");
        cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
        cpl_propertylist_update_string(header, "HDUCLAS2", "DATA");
        cpl_propertylist_erase(header, "SCIDATA");

        if (wcs_plist) cpl_propertylist_append(header, wcs_plist);

        /* ..copy mandatory keywords from the applist to the extension
           header */

        const char * data_mandatory[] =  {"BUNIT",
                                          "CUNIT1",
                                          "CUNIT2",
                                          "FLUXCAL"};
        const int count_data_mandatory = sizeof(data_mandatory) / 
                                         sizeof(char *);

        const char * data_mandatory2[] =  {"PHOTSYS",
				"ABMAGLIM",
				"ABMAGSAT",
				"PSF_FWHM",
				"PHOTZP",
				"PHOTZPER",
				"ZPMETHOD"};
        const int count_data_mandatory2 = sizeof(data_mandatory2) /
        		sizeof(char *);


        const char * ins_mode = "ins_mode";
        if(cpl_propertylist_has(plist,"ESO INS1 MODE")) {
            ins_mode = cpl_propertylist_get_string(plist, "ESO INS1 MODE");
        }
        //cpl_msg_info(cpl_func,"ins_mode: %s",ins_mode);

        for (int mandi=0; mandi < count_data_mandatory; mandi++) {
        	if (!cpl_propertylist_has(header, data_mandatory[mandi])) {
        		if (cpl_propertylist_has(applist, data_mandatory[mandi])) {
        			cpl_propertylist_copy_property(header, applist,
        					data_mandatory[mandi]);
        		}
        	}
        }

        if(strcmp(ins_mode,"nixLSS") != 0) {
        	//cpl_msg_info(cpl_func,"add mandatory 2");
        	for (int mandi2=0; mandi2 < count_data_mandatory2; mandi2++) {
        		if (!cpl_propertylist_has(header, data_mandatory2[mandi2])) {
        			if (cpl_propertylist_has(applist, data_mandatory2[mandi2])) {
        				cpl_propertylist_copy_property(header, applist,
        						data_mandatory2[mandi2]);
        			}
        		}
        	}
        }

        /* ..fill mandatory keywords that are still missing */
       	for (int mandi=0; mandi < count_data_mandatory; mandi++) {
       		if (!cpl_propertylist_has(header, data_mandatory[mandi])) {
       			if (!strcmp(data_mandatory[mandi], "BUNIT")) {
       				cpl_propertylist_update_string(header, "BUNIT",
       						"unknown");
       			} else if (!strcmp(data_mandatory[mandi], "CUNIT1")) {
       				cpl_propertylist_update_string(header, "CUNIT1", "deg");
       			} else if (!strcmp(data_mandatory[mandi], "CUNIT2")) {
       				cpl_propertylist_update_string(header, "CUNIT2", "deg");
       			} else if (!strcmp(data_mandatory[mandi], "FLUXCAL")) {
       				cpl_propertylist_update_string(header, "FLUXCAL",
       						"UNCALIBRATED");
       			} else {
       				cpl_msg_info(cpl_func,
       						"missing extension keyword %s not handled",
							data_mandatory[mandi]);
       			}
       		}
       	}
       	/* if not nixLSS..fill mandatory keywords that are still missing */
        if(strcmp(ins_mode,"nixLSS") != 0) {
        	//cpl_msg_info(cpl_func,"add2 mandatory 2");
        	for (int mandi2=0; mandi2 < count_data_mandatory2; mandi2++) {
        		if (!cpl_propertylist_has(header, data_mandatory2[mandi2])) {

        			if (!strcmp(data_mandatory2[mandi2], "PHOTSYS")) {
        				cpl_propertylist_update_string(header, "PHOTSYS", "VEGA");
        			} else if (!strcmp(data_mandatory2[mandi2], "ABMAGLIM")) {
        				cpl_msg_info(cpl_func, "dummy value for ABMAGLIM");
        				cpl_propertylist_update_double(header, "ABMAGLIM", 0.0);
        			} else if (!strcmp(data_mandatory2[mandi2], "ABMAGSAT")) {
        				cpl_msg_info(cpl_func, "dummy value for ABMAGSAT");
        				cpl_propertylist_update_double(header, "ABMAGSAT", 0.0);
        			} else if (!strcmp(data_mandatory2[mandi2], "PSF_FWHM")) {
        				cpl_msg_info(cpl_func, "dummy value for PSF_FWHM");
        				cpl_propertylist_update_double(header, "PSF_FWHM", -1.0);
        			} else if (!strcmp(data_mandatory2[mandi2], "PHOTZP")) {
        				cpl_msg_info(cpl_func, "PHOTZP not found, set to 0.0");
        				cpl_propertylist_update_double(header, "PHOTZP", 0.0);
        				cpl_propertylist_set_comment(header, "PHOTZP", "Photometric "
        						"zeropoint so that MAG=-2.5*"
        						"log(data)+PHOTZP+APCOR");
        			} else if (!strcmp(data_mandatory2[mandi2], "PHOTZPER")) {
        				cpl_msg_info(cpl_func, "PHOTZPER not found, set to 0.0");
        				cpl_propertylist_update_double(header, "PHOTZPER", 0.0);
        				cpl_propertylist_set_comment(header, "PHOTZPER", "Error on "
        						"PHOTZP");
        			} else if (!strcmp(data_mandatory2[mandi2], "ZPMETHOD")) {
        				cpl_msg_info(cpl_func, "ZPMETHOD not found");
        				cpl_propertylist_update_string(header, "ZPMETHOD",
        						"PHOTZP not calculated");
        				cpl_propertylist_set_comment(header, "ZPMETHOD",
        						"Method of ZP calculation");
        			} else {
        				cpl_msg_info(cpl_func,
        						"missing extension keyword %s not handled",
								data_mandatory[mandi2]);
        			}
        		}
        	}
        }

        if (image) {
            cpl_image_save(hdrl_image_get_image_const(image),
                           filename,
                           HDRL_TYPE_DATA,
                           header,
                           CPL_IO_EXTEND);
        }
        //exit(0);
        if (imagelist) {

            /* create a temporary cpl_imagelist of the image planes */

            cpl_imagelist * tlist = cpl_imagelist_new();
            hlist_sz = hdrl_imagelist_get_size(imagelist);
            if(hlist_sz > 1) {
            for (cpl_size i = 0; i < hlist_sz; i++) {
                const hdrl_image * timage = hdrl_imagelist_get(imagelist, i);

                //char* filename2 = cpl_sprintf("warp4_%s", filename);
                //cpl_image_save(hdrl_image_get_image((hdrl_image *) timage),
                //           filename2, CPL_TYPE_UNSPECIFIED, NULL,
                //           CPL_IO_CREATE);
                //cpl_free(filename2);

                cpl_imagelist_set(tlist,
                                  (cpl_image *) hdrl_image_get_image_const(timage), i);
            }

            cpl_imagelist_save(tlist,
                               filename,
                               CPL_TYPE_DOUBLE,
                               header,
                               CPL_IO_EXTEND);

            cpl_imagelist_unwrap(tlist);
            } else {
            	const hdrl_image * timage = hdrl_imagelist_get(imagelist, 0);
            	eris_ifu_plist_erase_wcs3D(header);
            	 cpl_image_save(hdrl_image_get_image_const(timage),
            	                           filename,
            	                           HDRL_TYPE_DATA,
            	                           header,
            	                           CPL_IO_EXTEND);
            }
        }

        cpl_propertylist_delete(header);
        enu_check_error_code("failed to save header 1");

        /* ..save image err to HDU 2 */
 
        header = cpl_propertylist_duplicate(dicd);
        cpl_propertylist_update_string(header, "EXTNAME", "ERR");
        cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
        cpl_propertylist_update_string(header, "HDUCLAS2", "ERROR");
        cpl_propertylist_update_string(header, "HDUCLAS3", "RMSE");
        cpl_propertylist_erase(header, "ERRDATA");
        if (cpl_propertylist_has(applist, "BUNIT")) {
            cpl_propertylist_copy_property(header, applist, "BUNIT");
        }

        if (wcs_plist) cpl_propertylist_append(header, wcs_plist);

        if (image) {
            cpl_image_save(hdrl_image_get_error_const(image), filename,
                           HDRL_TYPE_ERROR, header, CPL_IO_EXTEND);
        }
        if (imagelist) {
            cpl_imagelist * tlist = cpl_imagelist_new();
            hlist_sz = hdrl_imagelist_get_size(imagelist);
            if(hlist_sz > 1) {
            for (cpl_size i = 0; i < hlist_sz; i++) {
                const hdrl_image * timage = hdrl_imagelist_get(imagelist, i);
                cpl_imagelist_set(tlist,
                                  (cpl_image *) hdrl_image_get_error_const(timage),
                                  i);
            }
            cpl_imagelist_save(tlist,
                               filename,
                               CPL_TYPE_DOUBLE,
                               header,
                               CPL_IO_EXTEND);
            cpl_imagelist_unwrap(tlist);
            } else {
            	hdrl_image* timage = hdrl_imagelist_get(imagelist, 0);
            	eris_ifu_plist_erase_wcs3D(header);
            	cpl_image_save(hdrl_image_get_image_const(timage),
            	                               filename,
            	                               CPL_TYPE_DOUBLE,
            	                               header,
            	                               CPL_IO_EXTEND);
            }
        }

        cpl_propertylist_delete(header);
        enu_check_error_code("failed to save header 2");

        /* ..save image mask to HDU 3 */ 

        header = cpl_propertylist_duplicate(dicd);
        cpl_propertylist_update_string(header, "EXTNAME", "DQ");
        cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
        cpl_propertylist_update_string(header, "HDUCLAS2", "QUALITY");
        cpl_propertylist_update_string(header, "HDUCLAS3", "MASKZERO");
        cpl_propertylist_erase(header, "QUALDATA");

        if (wcs_plist) cpl_propertylist_append(header, wcs_plist);

        /* get the mask 'non const' as in this case one is created if it
           does not exist */
    
        if (image) {
            cpl_mask_save(hdrl_image_get_mask((hdrl_image *) image), filename,
                          header, CPL_IO_EXTEND);
        }
        if (imagelist) {
            const hdrl_image * timage = hdrl_imagelist_get(imagelist, 0);
            if(hlist_sz == 1) {
                       	eris_ifu_plist_erase_wcs3D(header);
                       }
            cpl_mask_save(hdrl_image_get_mask((hdrl_image *) timage), filename,
                          header, CPL_IO_EXTEND);
        }

        cpl_propertylist_delete(header);
        enu_check_error_code("failed to save header 3");
    }

    /* save the optional extensions */

    if (mefs) {
        cpl_msg_info(cpl_func, "..there are %d mefs", (int) (mefs->size));

        for (cpl_size i=0; i<mefs->size; i++) {
            cpl_propertylist * header = NULL;
            if (mefs->mef[i]->plist) {
                header = cpl_propertylist_duplicate(mefs->mef[i]->plist);
            } else {
                header = cpl_propertylist_new();
            }

            /* keywords for ESO DICD conformance */

            cpl_propertylist_copy_property_regexp(header, dicd, ".+",
                                                  CPL_FALSE);
            cpl_propertylist_update_string(header, "EXTNAME",
                                           mefs->mef[i]->name);

            cpl_msg_info(cpl_func, "..saving mef %s", mefs->mef[i]->name);
            if (!strcmp(mefs->mef[i]->name, "CONFIDENCE")) {
                cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
                cpl_propertylist_update_string(header, "HDUCLAS2", "CONF");
                cpl_propertylist_erase(header, "CONFDATA");
            } else if (!strcmp(mefs->mef[i]->name, "BKG_DATA")) {
                cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
                cpl_propertylist_update_string(header, "HDUCLAS2", "BKG_DATA");
                cpl_propertylist_erase(header, "BKGDATA");
                if (cpl_propertylist_has(applist, "BUNIT")) {
                    cpl_propertylist_copy_property(header, applist, "BUNIT");
                }
            } else if (!strcmp(mefs->mef[i]->name, "BKG_ERR")) {
                cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
                cpl_propertylist_update_string(header, "HDUCLAS2", "BKG_ERROR");
                cpl_propertylist_update_string(header, "HDUCLAS3", "RMSE");
                cpl_propertylist_erase(header, "BKGERR");
                if (cpl_propertylist_has(applist, "BUNIT")) {
                    cpl_propertylist_copy_property(header, applist, "BUNIT");
                }
            } else if (!strcmp(mefs->mef[i]->name, "BKG_CONF")) {
                cpl_propertylist_update_string(header, "HDUCLAS1", "IMAGE");
                cpl_propertylist_update_string(header, "HDUCLAS2", "BKG_CONF");
                cpl_propertylist_erase(header, "BKGCONF");
            } else if (!strcmp(mefs->mef[i]->data_type,
                       MEF_EXTENSION_CONTAINING_IMAGE)) {
                cpl_propertylist_update_string(header, "HDUCLAS1", 
                                               MEF_EXTENSION_CONTAINING_IMAGE);
            } else if (!strcmp(mefs->mef[i]->data_type,
                       MEF_EXTENSION_CONTAINING_MASK)) {
                cpl_propertylist_update_string(header, "HDUCLAS1",
                                               MEF_EXTENSION_CONTAINING_MASK);
            } else if (!strcmp(mefs->mef[i]->data_type,
                       MEF_EXTENSION_CONTAINING_TABLE)) {
                cpl_propertylist_update_string(header, "HDUCLAS1", 
                                               MEF_EXTENSION_CONTAINING_TABLE);
            }

            /* copy wcs if present */

            if (wcs_plist) cpl_propertylist_append(header, wcs_plist);

            /* save the header */
            if(hlist_sz == 1) {
            	eris_ifu_plist_erase_wcs3D(header);
            }
            enu_mef_extension_save(mefs->mef[i], filename, header,
                                   CPL_IO_EXTEND);
            cpl_propertylist_delete(header);
            enu_check_error_code("failed to save MEF extension %d", (int) i);
        }
    }

    /* save the new frame to the overall frameset so that DFS knows a new
       product has been created */

    cpl_frameset_insert(allframes, cpl_frame_duplicate(product_frame));

 cleanup:

    cpl_propertylist_delete(dicd);
    cpl_propertylist_delete(inherit_plist);
    cpl_propertylist_delete(plist);
    cpl_frameset_delete(product_frameset);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Save a located image structure to a MEF.
  @param allframes   The list of input frames for the recipe 
  @param parlist     The list of input parameters
  @param provenance  The frames that contribute directly to this product 
  @param prov_raw    CPL_TRUE for raw data filenames, CPL_FALSE for actual names 
  @param limage      The locate image to save
  @param recipe      The recipe name
  @param inherit     The frame to inherit header info from
  @param applist     The propertylist to be added to the primary header
  @param pipe_id     PACKAGE "/" PACKAGE_VERSION
  @param filename    The name of the pipeline
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_dfs_save_limage(cpl_frameset            * allframes,
                                   const cpl_parameterlist * parlist,
                                   const cpl_frameset      * provenance,
                                   const cpl_boolean         prov_raw, 
                                   const located_image     * limage,
                                   const char              * recipe,
                                   const cpl_frame         * inherit,
                                   cpl_propertylist        * applist,
                                   const char              * pipe_id,
                                   const char              * filename) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_msg_debug(cpl_func, "enu_dfs_save_limage entry");

    cpl_ensure_code(filename, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(limage, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(applist, CPL_ERROR_NULL_INPUT);

    cpl_propertylist * wcs_plist = NULL;

    /* how many mefs? */

    int nmefs = 0;
    if (limage->confidence) nmefs++;
    if (limage->bkg) nmefs = nmefs + 2;
    if (limage->bkg_confidence) nmefs++;

    mef_extension_list * mefs = NULL;
    if (nmefs > 0) mefs = enu_mef_extension_list_new(nmefs);
    cpl_msg_info(cpl_func, "nmefs %d", nmefs);

    int imef = 0;
    if (limage->confidence) {
        cpl_msg_info(cpl_func, "CONF");
        mefs->mef[imef] = enu_mef_new_image("CONFIDENCE", 
                                            limage->confidence, NULL);
        imef++;
    }
    if (limage->bkg) {
        cpl_msg_info(cpl_func, "BKG");
        mefs->mef[imef] = enu_mef_new_image("BKG_DATA",
                                            hdrl_image_get_image(
                                            limage->bkg), NULL);
        imef++;
        cpl_msg_info(cpl_func, "BKG_ERR");
        mefs->mef[imef] = enu_mef_new_image("BKG_ERR",
                                            hdrl_image_get_error(
                                            limage->bkg), NULL);
        imef++;
    }
    if (limage->bkg_confidence) {
        cpl_msg_info(cpl_func, "BKG_CONF");
        mefs->mef[imef] = enu_mef_new_image("BKG_CONF", 
                                            limage->bkg_confidence, NULL);
        imef++;
    }

    /* extract the wcs propertylist from the located image */

    wcs_plist = cpl_propertylist_duplicate(limage->plist);
    cpl_propertylist_erase_regexp(wcs_plist,
                                  "CRPIX1|CRPIX2|CRPIX3|"
                                  "CRVAL1|CRVAL2|CRVAL3|"
                                  "CDELT1|CDELT2|CDELT3|"
                                  "CD1_1|CD1_2|CD1_3|"
                                  "CD2_1|CD2_2|CD2_3|"
                                  "CD3_1|CD3_2|CD3_3|"
                                  "CTYPE1|CTYPE2|CTYPE3|"
                                  "CUNIT1|CUNIT2|CUNIT3",
                                  CPL_TRUE);

    /* copy other important keywords to applist unless an overriding value 
       is already there  */

    if (!cpl_propertylist_has(applist, "BUNIT")) {
        cpl_propertylist_copy_property(applist, limage->plist, "BUNIT");
    }
    if (!cpl_propertylist_has(applist, "PSF_FWHM")) {
    	if(cpl_propertylist_has(limage->plist, "PSF_FWHM")) {
    		cpl_propertylist_copy_property(applist, limage->plist, "PSF_FWHM");
    	}
    }
    if (!cpl_propertylist_has(applist, "ABMAGLIM")) {
        if (cpl_propertylist_has(limage->plist, "ABMAGLIM")) {
            cpl_propertylist_copy_property(applist, limage->plist, "ABMAGLIM");
        }
    }

    cpl_msg_info(cpl_func, ".. writing %s", filename);
    enu_dfs_save_himage(allframes,
                        parlist,
                        provenance,
                        prov_raw,
                        limage->himage,
                        limage->himagelist,
                        mefs,
                        recipe,
                        inherit,
                        applist,
                        wcs_plist,
                        pipe_id,
                        filename);

    enu_mef_extension_list_delete(mefs);
    cpl_propertylist_delete(wcs_plist);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Add phase3/IDP keywords to a pipeline product propertyist list.
  @param header        The required propertylist 
  @param product_frame Frame containing the productFilename of created product
  @param framelist     The list of input frames for the recipe 
  @param parlist       The list of input parameters
  @param recipe_id     The recipe name
  @param pipe_id       PACKAGE "/" PACKAGE_VERSION
  @param inherit       The frame to inherit header info from
  @param provenance    The frames that contribute directly to this product
  @param prov_raw      CPL_TRUE for raw data filenames, CPL_FALSE for actual names 
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code

 */
/*----------------------------------------------------------------------------*/

cpl_error_code enu_dfs_setup_product_header(cpl_propertylist * header,
                                            const cpl_frame * product_frame,
                                            const cpl_frameset * framelist,
                                            const cpl_parameterlist * parlist,
                                            const char * recipe_id,
                                            const char * pipe_id,
                                            const cpl_frame * inherit,
                                            const cpl_frameset * provenance,
                                            cpl_boolean prov_raw) {         

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_ensure_code(header, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(product_frame, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(framelist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(recipe_id, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pipe_id, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(inherit, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(provenance, CPL_ERROR_NULL_INPUT);

    cpl_propertylist * inherit_plist = NULL;
    cpl_propertylist * input_header = NULL;

    /* ..add old-style DFS keywords to header but do not overwrite RA,DEC 
         keywords if already present */

    input_header = cpl_propertylist_duplicate(header);
    cpl_dfs_setup_product_header(header,
                                 product_frame,
                                 framelist,
                                 parlist,
                                 recipe_id,
                                 pipe_id,
                                 CPL_DFS_PRO_DID,
                                 inherit);
    cpl_propertylist_copy_property_regexp(header,
                                          input_header,
                                          "RA|DEC",
                                          CPL_FALSE);
    if (cpl_propertylist_has(header, "RA")) {
        cpl_propertylist_set_comment(header, "RA",
                                     "[deg] RA (J2000) pointing");
    }
    if (cpl_propertylist_has(header, "DEC")) {
        cpl_propertylist_set_comment(header, "DEC",
                                     "[deg] DEC (J2000) pointing");
    }

    /* ..remove any obsolete DICD, data-specific, WCS or keywords that
         should just not be in the primary header */

    cpl_propertylist_erase_regexp(header, 
                                  "HDUCLASS|HDUDOC|HDUVERS|SCIDATA|ERRDATA|"
                                  "QUALDATA|CONFDATA|BKGDATA|BKGERR|BKGCONF|"
                                  "CRPIX1|CRPIX2|CRVAL1|CRVAL2|CDELT1|CDELT2|"
                                  "CTYPE1|CTYPE2|CUNIT1|CUNIT2|"
                                  "CD1_1|CD1_2|CD2_1|CD2_2|"
                                  "PHOTSYS|PHOTZP|PHOTZERR|ABMAGSAT|ABMAGLIM|"
                                  "MJD-OBS|PSF_FWHM|BUNIT",
                                  CPL_FALSE);

    /* ..add the PROVIi keywords from the provenance frames */

    int iprov = 0;
    cpl_frameset_iterator * it = cpl_frameset_iterator_new(provenance);
    const cpl_frame * frame = cpl_frameset_iterator_get_const(it);
    while (frame) {
        iprov++;
        /* get the name of the raw data file */
        char * filename = cpl_strdup(cpl_frame_get_filename(frame));
        char * pch = NULL;
        if (prov_raw) {
            /* want name of raw data file */
            pch = strstr(filename, "ERIS.20");
            if (!pch) {
                pch = filename;
            }
        } else {
            pch = filename;
        }
        char * pname = cpl_sprintf("PROV%.0i", iprov);
        cpl_propertylist_update_string(header, pname, basename(pch));
        cpl_free(filename);
        cpl_free(pname);
        cpl_frameset_iterator_advance(it, 1);
        frame = cpl_frameset_iterator_get(it);
    }
    cpl_frameset_iterator_delete(it); it = NULL;
    enu_check_error_code("error storing PROVenance keywords");

    /* ..copy any missing mandatory keywords from the propertylist
       associated with inherit */

    const char * inherit_fname = cpl_frame_get_filename(inherit);
    inherit_plist = cpl_propertylist_load(inherit_fname, 0);
    enu_check_error_code("error reading propertylist from inherit");

    const char * mandatory[] =  {"TELESCOP",
                                 "INSTRUME",
                                 "FILTER",
                                 "OBJECT",
                                 "RA",
                                 "DEC",
                                 "EQUINOX",
                                 "RADESYS",
                                 "TIMESYS",
                                 "EXPTIME",
                                 "TEXPTIME",
                                 "MJD-OBS",
                                 "MJD-END",
                                 "PROG_ID",
                                 "NCOMBINE",
                                 "FLUXCAL",
                                 "PROCSOFT",
                                 "REFERENC",
                                 "OBSTECH",
                                 "OBID1"};

    const int count_mandatory = sizeof(mandatory) / sizeof(char *);
    for (int mandi=0; mandi < count_mandatory; mandi++) {
        if (!cpl_propertylist_has(header, mandatory[mandi])) {
            if (cpl_propertylist_has(inherit_plist, mandatory[mandi])) {
                cpl_msg_info(cpl_func, "copying from inherit %s",
                             mandatory[mandi]);
                cpl_propertylist_copy_property(header, inherit_plist,
                                               mandatory[mandi]);
            }
        }
    }     

    /* ..try to fill mandatory keywords that are still missing */

    for (int mandi=0; mandi < count_mandatory; mandi++) {
        if (!cpl_propertylist_has(header, mandatory[mandi])) {
            if (!strcmp(mandatory[mandi], "FILTER")) {
                const char * filter = NULL;
                if (inherit_plist) {
                    filter = enu_get_filter(inherit_plist); 
                }
                if (filter) {
                    cpl_propertylist_update_string(header, "FILTER",
                                                   filter);
                } else {
                    cpl_propertylist_update_string(header, "FILTER",
                                                   "unknown");
                }
            } else if (!strcmp(mandatory[mandi], "RADESYS")) {
                const char * radesys = NULL;
                if (inherit_plist && cpl_propertylist_has(inherit_plist,
                                                          "RADECSYS")) {
                    radesys = cpl_propertylist_get_string(inherit_plist,
                                                          "RADECSYS");
                }
                if (radesys) {
                    cpl_propertylist_update_string(header, "RADESYS",
                                                   radesys);
                } else {
                    cpl_propertylist_update_string(header, "RADESYS",
                                                   "unknown");
                }
            } else if (!strcmp(mandatory[mandi], "TIMESYS")) {
                cpl_propertylist_update_string(header, "TIMESYS",
                                               "UTC");
            } else if (!strcmp(mandatory[mandi], "PROG_ID")) {
                const char * prog_id = NULL;
                if (inherit_plist && cpl_propertylist_has(inherit_plist,
                                                          "ESO OBS PROG ID")) {
                    prog_id = cpl_propertylist_get_string(inherit_plist,
                                                          "ESO OBS PROG ID");
                }
                if (prog_id) {
                    cpl_propertylist_update_string(header, "PROG_ID",
                                                   prog_id);
                } else {
                    cpl_propertylist_update_string(header, "PROG_ID",
                                                   "unknown");
                }
            } else if (!strcmp(mandatory[mandi], "OBSTECH")) {
                const char * obstech = NULL;
                if (inherit_plist) {
                    if (cpl_propertylist_has(inherit_plist, "ESO DPR TECH")) {
                        obstech = cpl_propertylist_get_string(
                                 inherit_plist, "ESO DPR TECH");
                    } else if (cpl_propertylist_has(inherit_plist, "ESO PRO TECH")) {
                        obstech = cpl_propertylist_get_string(
                                  inherit_plist, "ESO PRO TECH");
                    }
                }
                if (obstech) {
                    cpl_propertylist_update_string(header, "OBSTECH",
                                                   obstech);
                } else {
                    cpl_propertylist_update_string(header, "OBSTECH",
                                                   "unknown");
                }
            } else if (!strcmp(mandatory[mandi], "FLUXCAL")) {
                cpl_propertylist_update_string(header, "FLUXCAL",
                                               "UNCALIBRATED");
            } else if (!strcmp(mandatory[mandi], "PROCSOFT")) {
                cpl_propertylist_update_string(header, "PROCSOFT", pipe_id);
            } else if (!strcmp(mandatory[mandi], "REFERENC")) {
                cpl_propertylist_update_string(header, "REFERENC",
                                               " ");
            } else if (!strcmp(mandatory[mandi], "OBID1")) {
                int obid = -1;
                if (inherit_plist && cpl_propertylist_has(inherit_plist,
                                                          "ESO OBS ID")) {
                    obid = cpl_propertylist_get_int(inherit_plist, "ESO OBS ID");
                }
                cpl_propertylist_update_int(header, "OBID1", obid);
            } else { 
                cpl_msg_warning(cpl_func, "missing %s", mandatory[mandi]); 
            }
        }
    }

 cleanup:
    cpl_propertylist_delete(inherit_plist);
    cpl_propertylist_delete(input_header);

    return cpl_error_get_code();
}



/**@}*/
