/* $Id: eris_nix_scired.c,v 1.9 2013-03-26 17:00:44 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:00:44 $
 * $Revision: 1.9 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/
#include <casu_utils.h>
#include <casu_stats.h>
#include <casu_mods.h>
#include <eris_nix_scired.h>
#include <eris_nix_dfs.h>
#include <eris_nix_utils.h>
#include <eris_nix_catalogue.h>
#include <eris_nix_gain_linearity.h>
#include <eris_nix_master_dark.h>
#include <eris_nix_master_bpm.h>
#include <eris_nix_master_flat.h>
#include <eris_nix_casu_match.h>
#include <eris_nix_casu_utils.h>
#include <eris_nix_match.h>
#include <eris_utils.h>
#include <string.h>
#include <libgen.h>
#include <math.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_scired  ERIS NIX science data reduction related functions
 *
 * TBD
 */
/*----------------------------------------------------------------------------*/

/**@{*/


/* science parameter setting functions (to allow to share params between wrapper
 * recipe eris_nix_img_scired and the others eris_nix_cal_det, eris_nix_skysub
 * eris_nix_cal_wcs, eris_nix_cal_phot, eris_nix_img_hdrl_stack
 * */

cpl_error_code eris_nix_pixel_coord_diagnostic_param_set(
                                       const char* context,
                                       cpl_parameterlist* parlist) {
	/* coords of pixel to be used for diagnostics during reduction */

	cpl_parameter* p;
	char* param_name = NULL;

	param_name = cpl_sprintf("%s.x_probe", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_INT,
			"x coord of diagnostic pixel", context, -1);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "x-probe");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.y_probe", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_INT,
			"y coord of diagnostic pixel", context, -1);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "y-probe");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_free(param_name);

	cpl_parameterlist_append(parlist, p);
	return cpl_error_get_code();
}

cpl_error_code eris_nix_skysub_param_set(const char* context,
                                         cpl_parameterlist* parlist) {

	/* skysub recipe-specific parameters */
	cpl_parameter* p;
	char* param_name = NULL;

	param_name = cpl_sprintf("%s.sky-source", context);
	p = cpl_parameter_new_enum(param_name, CPL_TYPE_STRING,
			"data to be used for calculation of sky "
			"background", context,
			"auto", 3, "auto", "target", "offset");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-source");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.sky-selector", context);
	p = cpl_parameter_new_enum(param_name, CPL_TYPE_STRING,
			"method for selecting sky frames",
			context, "bracket", 1, "bracket");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-selector");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.sky-method", context);
	p = cpl_parameter_new_enum(param_name, CPL_TYPE_STRING,
			"method for combining sky frames",
			context, "median-median", 2,
			"collapse-median", "median-median");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-method");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.sky-bracket-time", context);
	p = cpl_parameter_new_range(param_name, CPL_TYPE_DOUBLE,
			"2 * max.time between target and sky measurement",
			context, 1800.0, 60.0, 18000.0);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-bracket-time");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.debug-data", context);
	p = cpl_parameter_new_value(param_name,
			CPL_TYPE_BOOL, "true to save interim results",
			context, CPL_FALSE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	return cpl_error_get_code();
}

/* defaults skysub:
 * 20, 3.0, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
 *			ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL
 * default photom:
 * 4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
            ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);

   default hdrl_stack:
   4, 2.5, CPL_TRUE, 5.0, CPL_TRUE, 64, 2.0, 3.0,
			ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);

            ==> obj_min_pixels, obj_threshold may have recipe dependent values
 */
cpl_error_code
eris_nix_catalogue_param_set(const char* context,
		cpl_parameterlist* parlist,
		int obj_min_pixels, double obj_threshold, cpl_boolean obj_deblending,
		double obj_core_radius, cpl_boolean bkg_estimate, int bkg_mesh_size,
		double bkg_smooth_fwhm, double det_eff_gain, double det_saturation,
		hdrl_catalogue_options resulttype)
{

	/* generate the general parameter list for the catalogue */
	cpl_parameter * p = NULL;
	hdrl_parameter    * catalogue_defaults = NULL;
	cpl_parameterlist * catalogue_parlist = NULL;

	catalogue_defaults = hdrl_catalogue_parameter_create(obj_min_pixels,
			obj_threshold, obj_deblending, obj_core_radius, bkg_estimate,
			bkg_mesh_size, bkg_smooth_fwhm, det_eff_gain, det_saturation,
			resulttype);
	catalogue_parlist = hdrl_catalogue_parameter_create_parlist(context,
			"catalogue", catalogue_defaults);

	/* add the subset of parameters to be used */
	for (p = cpl_parameterlist_get_first(catalogue_parlist);
			p != NULL;
			p = cpl_parameterlist_get_next(catalogue_parlist)) {

		const char * pname = cpl_parameter_get_name(p);
		if (strstr(pname, "min-pixels") ||
				strstr(pname, "threshold") ||
				strstr(pname, "mesh-size") ||
				strstr(pname, "threshold") ||
				strstr(pname, "smooth-gauss-fwhm")) {
			cpl_parameter * duplicate = cpl_parameter_duplicate(p);
			cpl_parameterlist_append(parlist, duplicate);
		}
	}

        /* ..add ao-params which specifies how to set params depending on AO */

	char * pname = cpl_sprintf("%s.catalogue.ao-params", context);
        p = cpl_parameter_new_enum(pname,
                                   CPL_TYPE_STRING,
                                   "Default catalogue.core-radius and "
                                   "catalogue.mesh-size depending on "
                                   "AOMODE, or not",
                                   context,
                                   "auto", 2, "auto", "user");
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "catalogue.ao-params");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(parlist, p);
        cpl_free(pname);

	hdrl_parameter_delete(catalogue_defaults);
	cpl_parameterlist_delete(catalogue_parlist);
	return cpl_error_get_code();
}

cpl_error_code
eris_nix_astrometric_param_set(const char* context,
		cpl_parameterlist* parlist)
{

	cpl_parameter* p;
	char* param_name = NULL;

	/* Flag to decide how we get the astrometric standard star information.
	   If "none", then use the local catalogues specified in the sof. If not,
	   then use one of the selection of catalogues available from CDS -
	   only gaia has sufficient accuracy */

	param_name = cpl_sprintf("%s.cdssearch_astrom", context);
	p = cpl_parameter_new_enum(param_name,
			CPL_TYPE_STRING,
			"CDS astrometric catalogue",
			context,
			"none", 3, "none", "2mass", "gaiadr3");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdssearch-astrom");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.debug-data", context);
	p = cpl_parameter_new_value(param_name,
			CPL_TYPE_BOOL, "true to save interim results",
			context, CPL_FALSE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	return cpl_error_get_code();

}

cpl_error_code
eris_nix_photom_param_set(const char* context,
		cpl_parameterlist* parlist)
{
	cpl_parameter* p;
	char* param_name = NULL;

	/* Flag to decide how we get the photometric standard star information. */
	param_name = cpl_sprintf("%s.cdssearch_photom", context);
	p = cpl_parameter_new_enum(param_name,
			CPL_TYPE_STRING,
			"CDS photometric catalogue",
			context,
			"2MASS", 1, "2MASS");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdssearch_photom");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.pixel_radius", context);
	p = cpl_parameter_new_value(param_name,
			CPL_TYPE_DOUBLE,
			"Max. distance between object and catalogue "
			"entry for association (pixels)",
			context, 5.0);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pixel-radius");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.minphotom", context);
	p = cpl_parameter_new_range(param_name,
			CPL_TYPE_INT,
			"Min number of matched stars for photometric "
			"calibration", context, 1, 1, 100000);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "minphotom");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.magerrcut", context);
	p = cpl_parameter_new_value(param_name,
			CPL_TYPE_DOUBLE,
			"Matched stars with magnitude error above "
			"this cutoff will not be used.",
			context, 0.5);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "magerrcut");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.debug-data", context);
	p = cpl_parameter_new_value(param_name,
			CPL_TYPE_BOOL, "true to save interim results",
			context, CPL_FALSE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	return cpl_error_get_code();

}


cpl_error_code eris_nix_hdrl_stack_param_set(const char* context,
                                             cpl_parameterlist* parlist) {
	cpl_parameter* p;
	char* param_name = NULL;

	/* Interpolation method */
	param_name = cpl_sprintf("%s.interpolation_method", context);
	p = cpl_parameter_new_enum(param_name,
			CPL_TYPE_STRING, "The interpolation method",
			context, "lanczos", 6, "nearest",
			"linear", "quadratic", "renka", "drizzle",
			"lanczos");
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
			"interpolation-method");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);


	/* Loop distance */
	param_name = cpl_sprintf("%s.loop_distance", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_INT,
			"maximum pixel offset taken into account",
			context, 1);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "loop-distance");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	/* Kernel size */
	param_name = cpl_sprintf("%s.kernel_size", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_INT,
			"(Lanczos method) size of kernel in pixels",
			context, 2);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kernel-size");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	/* Critical radius */
	param_name = cpl_sprintf("%s.critical_radius", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_DOUBLE,
			"(Renka method) distance beyond which weights "
			"set to 0",
			context, 5.0);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "critical-radius");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	/* Drizzle frac parameters */
	param_name = cpl_sprintf("%s.pix_frac_x", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_DOUBLE,
			"(Drizzle method) percentage of flux "
			"to drizzle from original to target pixel",
			context, 50.0);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pix-frac-x");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.pix_frac_y", context);
	p = cpl_parameter_new_value(param_name, CPL_TYPE_DOUBLE,
			"(Drizzle method) percentage of flux "
			"to drizzle from original to target pixel",
			context, 50.0);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pix-frac-y");
	cpl_parameterlist_append(parlist, p);
	cpl_free(param_name);

	/* generate the general parameter list for the catalogue : TO BE TAKEN
	 * FROM COMMON
	 *

	catalogue_defaults = hdrl_catalogue_parameter_create(4, 2.5, CPL_TRUE,
			5.0, CPL_TRUE, 64, 2.0, 3.0,
			ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
	catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
			"catalogue", catalogue_defaults);
    */
	// add the subset of parameters to be used ??
    /* TO BE ADDED HERE??
	for (p = cpl_parameterlist_get_first(catalogue_parlist);
			p != NULL;
			p = cpl_parameterlist_get_next(catalogue_parlist)) {

		const char * pname = cpl_parameter_get_name(p);
		if (strstr(pname, "minmax") == NULL) {
			cpl_parameter * duplicate = cpl_parameter_duplicate(p);
			cpl_parameterlist_append(self, duplicate);
		}
	}

	p = cpl_parameter_new_value(CONTEXT".debug-data",
			CPL_TYPE_BOOL, "true to save interim results",
			CONTEXT, CPL_FALSE);
	cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
	cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
	cpl_parameterlist_append(self, p);

    */

	return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save the calibrated data to FITS files
  @param    limage         the located image to save
  @param    tag            the PRO.CATG tag to set in the file propertylist
  @param    frameset       the frameset input to the reduction
  @param    parlist        the parameters list
  @param    calib_frames   the calibration frames used in the reduction
  @return   CPL_ERROR_NONE if everthing OK, CPL error code otherwise
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_cal_det_save(
                                       located_image * limage,
                                       const char * procatg,
                                       cpl_frameset * frameset,
                                       const cpl_parameterlist * parlist,
                                       const cpl_frameset * calib_frames,
                                       const char* recipe_name) {

    /* check input */

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();
    cpl_ensure_code(limage, CPL_ERROR_NONE);
    cpl_ensure_code(procatg, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(calib_frames, CPL_ERROR_NULL_INPUT);

    /* construct the provenance frameset for this frame - the frames
       actually used */

    cpl_frameset * provenance = cpl_frameset_new();
    cpl_frameset_insert(provenance, cpl_frame_duplicate(limage->frame));

    /* applist, keywords to add to headers */

    cpl_propertylist * applist = cpl_propertylist_new();
    cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, procatg);
    cpl_propertylist_copy_property(applist, limage->plist, "BUNIT");
    cpl_propertylist_update_int(applist, "NCOMBINE", 1);
    double dit = enu_get_dit(limage->plist);
    int ndit = cpl_propertylist_get_int(limage->plist, "ESO DET NDIT");
    cpl_propertylist_update_double(applist, "TEXPTIME", ndit * dit);
    /* update EXPTIME as well because it has a different defn in product
       compared to raw file - see jira PIPE-10669 for discussion. */
    cpl_propertylist_update_double(applist, "EXPTIME", ndit * dit);
    double mjd_obs = cpl_propertylist_get_double(limage->plist, "MJD-OBS");
    double mjd_end = mjd_obs + (ndit * dit) / (3600.0 * 24.0);
    cpl_propertylist_update_double(applist, "MJD-END", mjd_end);
    if(cpl_propertylist_has(limage->plist, "PSF_FWHM")) {
    	cpl_propertylist_copy_property(applist, limage->plist, "PSF_FWHM");
    }
    cpl_propertylist_copy_property(applist, limage->plist,
                                   "ESO DETMON SATURATION"); 	
    cpl_propertylist_copy_property(applist, limage->plist,
                                   "ESO DARK GAIN");
    cpl_propertylist_update_string(applist, "PRODCATG", "ANCILLARY.IMAGE");

    /* set RA, DEC from WCS rather than let dfs copy it from another
       frame. This only in case of input data in 'single' format, that means NAXIS=2 */
   
    int naxis = cpl_propertylist_get_int(limage->plist,"NAXIS");
    if(naxis == 2)
    {
        cpl_wcs * wcs = cpl_wcs_new_from_propertylist(limage->plist);
        double ra = 0.0;
        double dec = 0.0;
        enu_get_ra_dec(wcs, &ra, &dec);

        cpl_propertylist_update_double(applist, "RA", ra);
        cpl_propertylist_update_double(applist, "DEC", dec);

        cpl_wcs_delete(wcs);
    }

    /* generate name of output file, write file */

    char * out_fname = enu_repreface(cpl_frame_get_filename(limage->frame),
                                     "cal_det");    

    /* save the result */

    enu_dfs_save_limage(frameset,
                        parlist,
                        provenance,
                        CPL_TRUE,
                        limage,
                        recipe_name,
                        limage->frame,
                        applist,
                        PACKAGE "/" PACKAGE_VERSION,
                        out_fname);

    cpl_free(out_fname);
    cpl_frameset_delete(provenance);
    cpl_propertylist_delete(applist);
    return cpl_error_get_code();
}


static char*
eris_nix_update_pupil_pro_catg(const char* pro_catg, located_image * limage)
{
	char* new_pro_catg=NULL;
	char* suffix=NULL;
	const char* nxpw = NULL;
	if(cpl_propertylist_has(limage->plist, "ESO INS2 NXPW NAME")) {
		nxpw = cpl_propertylist_get_string(limage->plist, "ESO INS2 NXPW NAME");
	}

	if(strcmp(nxpw, "Open1") == 0){
		suffix = cpl_sprintf("OPEN");
	} else if (strcmp(nxpw, "JHK-pupil") == 0){
		suffix = cpl_sprintf("JHK");
	} else if (strcmp(nxpw, "Crosshairs") == 0){
		suffix = cpl_sprintf("CROSS");
	} else if (strcmp(nxpw, "APP") == 0){
		suffix = cpl_sprintf("APP");
	} else if (strcmp(nxpw, "LM-pupil") == 0){
		suffix = cpl_sprintf("LM");
	} else if (strcmp(nxpw, "Lyot") == 0){
		suffix = cpl_sprintf("LYOT");
	} else if (strcmp(nxpw, "Lyot-ND") == 0){
		suffix = cpl_sprintf("LYOTND");
	} else if (strstr(nxpw, "SAM")){
		suffix = cpl_sprintf("SAM");
	} else if (strstr(nxpw, "Spider")){
		suffix = cpl_sprintf("SPIDER");
	} else {
		cpl_msg_warning(cpl_func,"%s",nxpw);
		suffix = cpl_sprintf("NOT_SUPPORTED");
	}

	if(strcmp(nxpw, "Open1") == 0 || strcmp(nxpw, "ND") == 0) {
		new_pro_catg = cpl_sprintf("%s_%s", pro_catg, suffix);
	} else {
		new_pro_catg = cpl_sprintf("%s_%s", pro_catg, suffix);
	}

	cpl_free(suffix);
	return new_pro_catg;
}

/*----------------------------------------------------------------------------*/
/**
  @brief   This function removes the detector signature from science frames

  @param    frameset   the frames list
  @param    parlist    the parameters list
  @param    recipe_name name of recipe
  @param    context  recipe context (eris)

  This comprises dark subtraction, linearization, flat-fielding,
  and association with the master bad-pixel mask.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/


cpl_error_code eris_nix_scired_cal_det(cpl_frameset * frameset,
                                       const cpl_parameterlist * parlist,
                                       const char* recipe_name,
                                       const char* context) {

    cpl_frameset          * base_frames = NULL;
    gain_linearity        * gain_lin = NULL;
    located_image         * limage = NULL;
    master_bpm            * master_bpm_used = NULL;
    master_bpm            * master_bpm_lamp = NULL;
    master_bpm            * master_bpm_sky = NULL;
    master_dark           * master_drk = NULL;
    master_flat           * master_flat_hifreq = NULL;
    master_flat           * master_flat_lofreq = NULL;
    const cpl_parameter   * p = NULL;
    char                  * param_name = NULL;
    cpl_table             * refine_wcs = NULL;
    cpl_image             * copyconf = NULL;

    enu_check_error_code("%s():%d: An error is already set: %s",
                         cpl_func, __LINE__, cpl_error_get_where());

    /* Check input parameters */

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    /* Set the msg verbosity level from environment variable CPL_MSG_LEVEL */
    /* check for invalid input */
    if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
    	return CPL_ERROR_BAD_FILE_FORMAT;
    }

    cpl_msg_set_level_from_env();

    /* check required input tags are present */
    const int nfile_tags = 4;
    const char* required_tags[4] = {ERIS_NIX_NL_BPM_PRO_CATG,
                                    ERIS_NIX_COEFFS_CUBE_PRO_CATG,
                                    ERIS_NIX_GAIN_PRO_CATG,
// opt                              ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG,
                                    ERIS_NIX_MASTER_DARK_IMG_PRO_CATG
// opt                              ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG,
// opt                              ERIS_NIX_MASTER_FLAT_LAMP_LOFREQ_PRO_CATG,  // opt
// opt                              ERIS_NIX_MASTER_FLAT_TWILIGHT_LOFREQ_PRO_CATG, //opt
// opt                              ERIS_NIX_RAW_OBJECT_JITTER_DO_CATG,
// opt                              ERIS_NIX_WCS_REFINE_PRO_CATG
    };

    cpl_ensure_code(CPL_ERROR_NONE ==
                    eris_dfs_check_input_tags(frameset, required_tags, nfile_tags, 1),
                    CPL_ERROR_ILLEGAL_INPUT);

    /* Read parameters */

    param_name = cpl_sprintf("%s.fill-rejected", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const char * fill_rejected = cpl_parameter_get_string(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.fill-value", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const double fill_value = cpl_parameter_get_double(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.cd_matrix_modify", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const int cd_matrix_modify = cpl_parameter_get_bool(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.x_probe", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_size x_probe = (cpl_size) cpl_parameter_get_int(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.y_probe", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_size y_probe = (cpl_size) cpl_parameter_get_int(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.collapse_cube", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_size collapse_cube = (cpl_size) cpl_parameter_get_int(p);
    cpl_free(param_name);

    enu_check_error_code("Could not retrieve input parameters");

    /* Identify the RAW and CALIB frames in the input frameset */

    eris_nix_dfs_set_groups(frameset);
    enu_check_error_code("Could not identify RAW and CALIB frames");

    base_frames = cpl_frameset_new();

    /* read the gain and linearity information. This is required
       as the linearization process changes the data units to
       electrons/sec which is assumed later for the photometric
       calibration*/

    int required = CPL_TRUE;
    gain_lin = engl_gain_linearity_load_from_frameset(frameset,
                     ERIS_NIX_GAIN_PRO_CATG, ERIS_NIX_COEFFS_CUBE_PRO_CATG,
                     ERIS_NIX_NL_BPM_PRO_CATG, required, base_frames);
    enu_check_error_code("failed to read gain/linearity information from SoF");

    /* read the master_dark */

    master_drk = en_master_dark_load_from_frameset(frameset,
                  ERIS_NIX_MASTER_DARK_IMG_PRO_CATG, base_frames);
    enu_check_error_code("failed to read master dark from SoF");

    /* read the master_bpm
       ..there is a choice between sky bpm/flatfield or lamp bpm/flatfield
       ..detect that choice here by the bpm used */

    master_bpm_lamp = en_master_bpm_load_from_frameset(frameset,
                      ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG,
                      base_frames, CPL_FALSE);
    if (!master_bpm_lamp) {
        master_bpm_sky = en_master_bpm_load_from_frameset(frameset,
                         ERIS_NIX_MASTER_BPM_SKY_PRO_CATG, base_frames,
                         CPL_FALSE);

    }
    enu_check_error_code("failed to read master BPM from SoF");
    enu_check(master_bpm_lamp || master_bpm_sky, CPL_ERROR_ILLEGAL_INPUT,
              "SoF contains neither lamp nor sky BPM");
    enu_check(!(master_bpm_lamp && master_bpm_sky), CPL_ERROR_ILLEGAL_INPUT,
              "SoF contains both lamp and sky BPM");
    if (master_bpm_lamp) {
        cpl_msg_info(cpl_func, "lamp bpm read");
        master_bpm_used = master_bpm_lamp;
    }
    if (master_bpm_sky) {
        cpl_msg_info(cpl_func, "sky bpm read");
        master_bpm_used = master_bpm_sky;
    }

    /* read the high-freq flatfield (optional) */

    if (master_bpm_lamp) {
        master_flat_hifreq = en_master_flat_load_from_frameset(frameset,
                             ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG,
                             base_frames, CPL_FALSE);
        if (master_flat_hifreq) {
            cpl_msg_info(cpl_func, "lamp hifreq flat read");
        }
    } else {
        master_flat_hifreq = en_master_flat_load_from_frameset(frameset,
                             ERIS_NIX_MASTER_FLAT_SKY_HIFREQ_PRO_CATG,
                             base_frames, CPL_FALSE);
        if (master_flat_hifreq) {
            cpl_msg_info(cpl_func, "sky hifreq flat read");
        }
    }

    if (master_bpm_lamp) {
        /* read the low-freq flatfield (twilight preferred, optional) */

        master_flat_lofreq = en_master_flat_load_from_frameset(frameset,
                             ERIS_NIX_MASTER_FLAT_TWILIGHT_LOFREQ_PRO_CATG,
                             base_frames, CPL_FALSE);
        if(master_flat_lofreq) {
            cpl_msg_info(cpl_func, "twilight lofreq flat read");
        } else {
            master_flat_lofreq = en_master_flat_load_from_frameset(frameset,
                                 ERIS_NIX_MASTER_FLAT_LAMP_LOFREQ_PRO_CATG,
                                 base_frames, CPL_FALSE);
            if(master_flat_lofreq) {
                cpl_msg_info(cpl_func, "lamp lofreq flat read");
            }
        }
    }
    if (master_bpm_sky) {
        master_flat_lofreq = en_master_flat_load_from_frameset(frameset,
                             ERIS_NIX_MASTER_FLAT_SKY_LOFREQ_PRO_CATG,
                             base_frames, CPL_FALSE);
        if(master_flat_lofreq) {
            cpl_msg_info(cpl_func, "sky lofreq flat read");
        }
    }

    /* load wcs refinement table if required */

    if (cd_matrix_modify) {
        const cpl_frame * target_frame = cpl_frameset_find_const(
                                       frameset, 
                                       ERIS_NIX_WCS_REFINE_PRO_CATG);
        enu_check(target_frame != NULL, CPL_ERROR_DATA_NOT_FOUND,
                  "SoF has no file tagged %s", ERIS_NIX_WCS_REFINE_PRO_CATG);

        refine_wcs = cpl_table_load(cpl_frame_get_filename(target_frame),
                                    1, CPL_TRUE); 	    
        if (refine_wcs) {
            cpl_msg_info(cpl_func, "wcs refinement table read:");
            cpl_table_dump(refine_wcs, 0, 100, NULL);
        }
    }

    const char * datatags[] = {ERIS_NIX_RAW_OBJECT_JITTER_DO_CATG,
                               ERIS_NIX_RAW_SKY_JITTER_DO_CATG,
                               ERIS_NIX_RAW_STD_JITTER_DO_CATG,
                               ERIS_NIX_RAW_OBJECT_LSS_JITTER_DO_CATG,
                               ERIS_NIX_RAW_SKY_LSS_JITTER_DO_CATG,
                               ERIS_NIX_RAW_STD_LSS_JITTER_DO_CATG,
							   ERIS_NIX_RAW_STD_APP_DO_CATG,
                               ERIS_NIX_RAW_STD_FPC_DO_CATG,
							   ERIS_NIX_RAW_STD_SAM_DO_CATG,
							   ERIS_NIX_RAW_OBJECT_APP_DO_CATG,
                               ERIS_NIX_RAW_OBJECT_FPC_DO_CATG,
							   ERIS_NIX_RAW_OBJECT_SAM_DO_CATG,
                               ERIS_NIX_RAW_SKY_FPC_DO_CATG,
							   ERIS_NIX_RAW_SKY_APP_DO_CATG,
							   ERIS_NIX_RAW_PUPIL_LAMP_DO_CATG,
							//   ERIS_NIX_RAW_PUPIL_LAMP_OPEN_DO_CATG,
							//   ERIS_NIX_RAW_PUPIL_LAMP_MASK_DO_CATG,
							   ERIS_NIX_RAW_PUPIL_SKY_DO_CATG,
                            //   ERIS_NIX_RAW_PUPIL_SKY_OPEN_DO_CATG,
							//   ERIS_NIX_RAW_PUPIL_SKY_MASK_DO_CATG,
							   ERIS_NIX_RAW_PUPIL_BKG_DO_CATG,
							   //ERIS_NIX_RAW_PUPIL_BKG_MASK_DO_CATG,
							   ERIS_NIX_RAW_PUPIL_DARK_DO_CATG
							//   ERIS_NIX_RAW_PUPIL_DARK_OPEN_DO_CATG,
							//   ERIS_NIX_RAW_PUPIL_DARK_MASK_DO_CATG
							   };
    cpl_size ntags = sizeof(datatags) / sizeof(const char *);
    const char * protags[] = {ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG,
                              ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG,
                              ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG,
                              ERIS_NIX_CAL_DET_OBJECT_LSS_JITTER_PRO_CATG,
                              ERIS_NIX_CAL_DET_SKY_LSS_JITTER_PRO_CATG,
                              ERIS_NIX_CAL_DET_STD_LSS_JITTER_PRO_CATG,
							  ERIS_NIX_CAL_DET_STD_APP_PRO_CATG,
							  ERIS_NIX_CAL_DET_STD_FPC_PRO_CATG,
							  ERIS_NIX_CAL_DET_STD_SAM_PRO_CATG,
							  ERIS_NIX_CAL_DET_OBJECT_APP_PRO_CATG,
                              ERIS_NIX_CAL_DET_OBJECT_FPC_PRO_CATG,
							  ERIS_NIX_CAL_DET_OBJECT_SAM_PRO_CATG,
							  ERIS_NIX_CAL_DET_SKY_APP_PRO_CATG,
                              ERIS_NIX_CAL_DET_SKY_FPC_PRO_CATG,
							  ERIS_NIX_CAL_DET_PUPIL_LAMP_PRO_CATG,
							 // ERIS_NIX_CAL_DET_PUPIL_LAMP_OPEN_PRO_CATG,
							 // ERIS_NIX_CAL_DET_PUPIL_LAMP_MASK_PRO_CATG,
							  ERIS_NIX_CAL_DET_PUPIL_SKY_PRO_CATG,
							 // ERIS_NIX_CAL_DET_PUPIL_SKY_OPEN_PRO_CATG,
							 // ERIS_NIX_CAL_DET_PUPIL_SKY_MASK_PRO_CATG,
							  ERIS_NIX_CAL_DET_PUPIL_BKG_PRO_CATG,
							  //ERIS_NIX_CAL_DET_PUPIL_BKG_MASK_PRO_CATG,
							  ERIS_NIX_CAL_DET_PUPIL_DARK_PRO_CATG
                             // ERIS_NIX_CAL_DET_PUPIL_DARK_OPEN_PRO_CATG,
	                         // ERIS_NIX_CAL_DET_PUPIL_DARK_MASK_PRO_CATG
							  };
    cpl_size nprotags = sizeof(protags) / sizeof(const char *);
    cpl_msg_debug(cpl_func,"ntags: %lld nprotags: %lld",ntags,nprotags);
    enu_check(ntags == nprotags, CPL_ERROR_UNSPECIFIED, "programming error");

    /* Loop through the possible data tags */

    for (cpl_size itag = 0; itag < ntags; itag++) {

        /* handle each frame individually. Ones taken in cube mode can
           be large, and there is no need to read them together */

        cpl_frameset_iterator * frameset_iter =  cpl_frameset_iterator_new(frameset);
        for (cpl_size iframe = 0; iframe < cpl_frameset_get_size(frameset);
             iframe++) {

            cpl_frame * frame = cpl_frameset_iterator_get(frameset_iter);
            if (!strcmp(datatags[itag], cpl_frame_get_tag(frame))) {
                cpl_msg_info(cpl_func, "processing %s", cpl_frame_get_filename(frame));

                limage = enu_load_limage_from_frame(frame, &copyconf, collapse_cube);
                enu_check_error_code("failed to read data to calibrate");
                if (limage && limage->confidence) {
                    cpl_image_delete(copyconf);
                    copyconf = cpl_image_duplicate(limage->confidence);
                }

                /* check that the jitter has reasonable WCS keywords 
                   (i.e. non-zero) - early commissioning data suffered from this */
                
                //AMO: TODO is the following check really needed?
                if(strcmp(cpl_frame_get_tag(frame), "PUPIL_LAMP") ){
                	//enu_check_wcs(limage);
                }

                /* Do 'detector calibration': dark subtraction, linearize,
                   calculate error plane, flatfield, associate with bpm */

                int remove_read_offsets = CPL_TRUE;
                int flag_mask = 0;
                flag_mask = ~flag_mask;

                enu_basic_calibrate(limage,
                                    remove_read_offsets,
                                    refine_wcs,
                                    master_drk,
                                    gain_lin,
                                    master_flat_hifreq,
                                    master_flat_lofreq,
                                    master_bpm_used,
                                    flag_mask,
                                    fill_rejected,
                                    fill_value,
                                    x_probe,
                                    y_probe);
                enu_check_error_code("error performing detector calibration of frames");

                /* handle pupil data */
                char* pro_catg = NULL;

                if(strstr(protags[itag],"PUPIL")) {
                    pro_catg = eris_nix_update_pupil_pro_catg(protags[itag],limage);
                } else {
                    pro_catg = cpl_sprintf("%s",protags[itag]);
                }

                /* save the calibrated image to FITS */
                eris_nix_cal_det_save(limage,
                                      pro_catg,
                                      frameset,
                                      parlist,
                                      base_frames,
                                      recipe_name);

                enu_check_error_code("error saving frame");

                enu_located_image_delete(limage);
                limage = NULL;
                if(pro_catg) {
                	cpl_free(pro_catg);
                }
            }
            cpl_frameset_iterator_advance(frameset_iter, 1);
        }
        cpl_frameset_iterator_delete(frameset_iter);
    }

cleanup:
    cpl_image_delete(copyconf);
    cpl_frameset_delete(base_frames);
    engl_gain_linearity_delete(gain_lin);
    enu_located_image_delete(limage);
    en_master_bpm_delete(master_bpm_lamp);
    en_master_bpm_delete(master_bpm_sky);
    en_master_dark_delete(master_drk);
    en_master_flat_delete(master_flat_hifreq);
    en_master_flat_delete(master_flat_lofreq);
    cpl_table_delete(refine_wcs);

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Calculate and attach object masks to jitter images
  @param    jitters          The input list of jitters
  @param    sky_selector     How to select sky frames: ["bracket"]
  @param    sky_time_range   Width of time bracket for selecting skys
  @param    sky_method       Method for calculating sky background: ["collapse-median", "median-median"]
  @param    debug_data       TRUE to write files containing interim results
  @param    debug_preface    String to preface filenames of interim results
  @param    frameset         The frames list
  @param    parlist          The parameters list
  @param    used_frameset    The list of raw/calibration frames used for this product
  @param    x_probe          x index of debug pixel, ignored if outside range
  @param    y_probe          y index of debug pixel, ignored if outside range
  @param    objmask_jitters  List of jitters with object masks attached
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_sky_objmask(
                      const char* recipe_name, 
                      const char* context,
                      const located_imagelist * jitters,
                      const char * sky_selector,
                      const double sky_time_range,
                      const char * sky_method,
                      const int debug_data,
                      const char * debug_preface,
                      cpl_frameset * frameset,
                      const cpl_parameterlist * parlist,
                      const cpl_frameset * used_frameset,
                      const cpl_size x_probe,
                      const cpl_size y_probe,
                      located_imagelist ** objmask_jitters) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    located_imagelist     * jitters1 = NULL;
    located_imagelist     * jitters2 = NULL;
    const cpl_parameter   * p = NULL;
    char* param_name = NULL;

    /* We only need a subset of catalogue parameters but they are not defined
       outside of hdrl - so re-read them here */

    param_name = cpl_sprintf("%s.catalogue.obj.min-pixels", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    int obj_min_pixels = cpl_parameter_get_int(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.catalogue.obj.threshold", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double obj_threshold = cpl_parameter_get_double(p);
    cpl_free(param_name);

    /* use AOMODE to decide if set catalogue.core-radius and mesh-size to
       defaults or read them from parameters */

    double obj_core_radius = -1.0;
    int bkg_mesh_size = -1;
    enu_get_rcore_and_mesh_size(context,
                                parlist,
                                jitters->limages[0]->plist,
                                &obj_core_radius,
                                &bkg_mesh_size);

    param_name = cpl_sprintf("%s.catalogue.bkg.smooth-gauss-fwhm", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double bkg_smooth_fwhm = cpl_parameter_get_double(p);
    cpl_free(param_name);

    /* Estimate and subtract the sky background with no objmasks */

    jitters1 = enu_located_imagelist_duplicate(jitters);
    cpl_msg_info(cpl_func, "Estimating sky background with no source blanking ...");
    enu_sky_subtract_limlist(sky_method, sky_selector, sky_time_range,
                             jitters1, x_probe, y_probe, jitters1);
    enu_check_error_code("error in sky_objmask: sky estimation");

    /* derive object masks for each image */

    enu_opm_limlist(obj_min_pixels, obj_threshold, bkg_mesh_size,
                    bkg_smooth_fwhm, jitters1);
    enu_check_error_code("error in sky_objmask: deriving object mask");

    /* in debug mode, save the sky-subtracted images */

    char * preface = cpl_sprintf("%s_%s", debug_preface, "sky1");
    enu_debug_limlist_save(debug_data, jitters1, preface, recipe_name,
                           frameset, parlist, used_frameset);
    cpl_free(preface);
    enu_check_error_code("error in sky_objmask: saving debug data");

    /* Return a copy the raw jitter data with objmasks attached */

    jitters2 = enu_located_imagelist_duplicate(jitters);
    for (cpl_size j = 0; j < jitters2->size; j++) {
        jitters2->limages[j]->object_mask = cpl_mask_duplicate(jitters1->
                                            limages[j]->object_mask);
    }

 cleanup:

    enu_located_imagelist_delete(jitters1);
    if (cpl_error_get_code() == CPL_ERROR_NONE) {
        *objmask_jitters = jitters2;
    } else {
        enu_located_imagelist_delete(jitters2);
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief   This recipe estimates and subtracts the sky background from a set of
           science frames

  @param    frameset   the frames list
  @param    parlist    the parameters list
  @param    recipe_name name of recipe
  @param    context  recipe context (eris)

  This comprises dark subtraction, linearization, flat-fielding,
  and association with the master bad-pixel mask.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/


cpl_error_code eris_nix_scired_skysub(cpl_frameset * frameset,
                                      const cpl_parameterlist * parlist,
                                      const char* recipe_name,
                                      const char* context) {

	located_imagelist     * object_jitters = NULL;
	located_imagelist     * objmask_jitters = NULL;
	located_imagelist     * objmask_sky_jitters = NULL;
	const char            * out_tag = NULL;
	const cpl_parameter   * p = NULL;
	located_imagelist     * result_jitters = NULL;
	located_imagelist     * skysub_jitters = NULL;
	located_imagelist     * sky_jitters = NULL;
	located_imagelist     * std_jitters = NULL;
	located_imagelist     * target_jitters = NULL;
	cpl_frameset          * used_frameset = NULL;
	char                  * param_name = NULL;

	enu_check_error_code("%s():%d: An error is already set: %s",
			cpl_func, __LINE__, cpl_error_get_where());

	/* Check input parameters */

	cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

	/* Set the msg verbosity level from environment variable CPL_MSG_LEVEL
       - not working, so set manually for now */

	cpl_msg_set_level_from_env();
	cpl_msg_severity severity = cpl_msg_get_level();
	cpl_msg_info(cpl_func, "level %d", (int) severity);

	/* check for invalid input */
	if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
		return CPL_ERROR_BAD_FILE_FORMAT;
	}

	/* Retrieve input parameters */
	param_name = cpl_sprintf("%s.sky-source", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	const char * sky_source = cpl_parameter_get_string(p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.sky-selector", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	const char * sky_selector = cpl_parameter_get_string(p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.sky-method", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	const char * sky_method = cpl_parameter_get_string(p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.sky-bracket-time", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	const double sky_time_range = cpl_parameter_get_double(p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.debug-data", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	const int debug_data = cpl_parameter_get_bool(p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.x_probe", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	cpl_size x_probe = (cpl_size) cpl_parameter_get_int(p);
	cpl_free(param_name);

	param_name = cpl_sprintf("%s.y_probe", context);
	p = cpl_parameterlist_find_const(parlist, param_name);
	cpl_size y_probe = (cpl_size) cpl_parameter_get_int(p);
	cpl_free(param_name);

	enu_check_error_code("Could not retrieve input parameters");

	/* Identify the RAW and CALIB frames in the input frameset */

	eris_nix_dfs_set_groups(frameset);
	enu_check_error_code("Could not identify RAW and CALIB frames");

	used_frameset = cpl_frameset_new();

	/* Read in CAL_DET_OBJECT_JITTER data, CAL_DET_SKY_JITTER, or
           CAL_DET_STD_JITTER frames if present */

	object_jitters = enu_limlist_load_from_frameset(frameset,
			ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG, used_frameset);
	enu_check_error_code("Could not load "
			ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG" frames");
	sky_jitters = enu_limlist_load_from_frameset(frameset,
			ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG, used_frameset);
	enu_check_error_code("Could not load "
			ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG" frames");
	std_jitters = enu_limlist_load_from_frameset(frameset,
			ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG, used_frameset);
	enu_check_error_code("Could not load "
			ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG" frames");

	cpl_msg_info(cpl_func, "%d "
			ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG" frames read",
			(int) object_jitters->size);
	cpl_msg_info(cpl_func, "%d "
			ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG" frames read",
			(int) sky_jitters->size);
	cpl_msg_info(cpl_func, "%d "
			ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG" frames read",
			(int) std_jitters->size);

	/* check that the SOF collection makes sense */
	if (object_jitters->size > 0 || std_jitters->size) {
		if(!strcmp(sky_source,"auto")) {
			/* OK */
		} else {
			if (sky_jitters->size > 0 || !strcmp(sky_source,"target")) {
				/* OK */
			} else {
				cpl_msg_error(cpl_func,"sky_source: %s, but missing input sky frames. Exit!", sky_source);
				return CPL_ERROR_DATA_NOT_FOUND;
			}
		}
	} else {
		cpl_msg_error(cpl_func,"Missing input object frames. Exit!");
		return CPL_ERROR_DATA_NOT_FOUND;
	}

	enu_check(!(object_jitters->size == 0 && std_jitters->size == 0),
			CPL_ERROR_DATA_NOT_FOUND, "SoF contains no "
			ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG" or "
			ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG" frames");
	enu_check(!(object_jitters->size > 0 && std_jitters->size > 0),
			CPL_ERROR_ILLEGAL_INPUT, "SoF contains both "
			ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG" and "
			ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG" frames");
        enu_check(!(!strcmp(sky_source,"offset") && sky_jitters->size==0),
                        CPL_ERROR_INCOMPATIBLE_INPUT,
                        "offset sky position requested but no"
                        ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG" in sof");

	/* is target an object or std? */

	if (object_jitters->size > 0) {
		target_jitters = object_jitters;
		out_tag = ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG;
                enu_located_imagelist_delete(std_jitters);
                std_jitters = NULL;
	} else if (std_jitters->size > 0) {
		target_jitters = std_jitters;
		out_tag = ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG;
                enu_located_imagelist_delete(object_jitters);
                object_jitters = NULL;
	}

        for (cpl_size j=0; j<target_jitters->size; j++) {
            double temp = cpl_propertylist_get_double(target_jitters->limages[j]->plist, "RA");
            const char * nm = cpl_frame_get_filename(target_jitters->limages[j]->frame);
            cpl_msg_info(cpl_func, "%s %f %s", nm, temp, cpl_error_get_message());
        }

	/* do right thing when sky-source=="auto", namely use offset
           sky data if available */

	if (!strcmp(sky_source, "auto")) {
		if (sky_jitters->size > 0) {
			sky_source = "offset";
		} else {
			sky_source = "target";
		}
		cpl_msg_info(cpl_func, "sky-source=='auto', resetting to '%s'",
				sky_source);
	}

	if (!strcmp(sky_source, "offset")) {
                if (sky_jitters->size > 1) {

                    /* Calculate object masks for the 'offset' jitter images */

                    eris_nix_img_sky_objmask(recipe_name, context, sky_jitters,
                                             sky_selector, sky_time_range,
                                             sky_method,
                                             debug_data, "offset",
                                             frameset, parlist, used_frameset,
                                             x_probe, y_probe,
                                             &objmask_sky_jitters);

                    /* Use the masked images to estimate the sky background and
                       subtract it */

                    cpl_msg_info(cpl_func,
                                 "Estimating sky background with source blanking ...");
                    enu_sky_subtract_limlist(sky_method, sky_selector,
                                             sky_time_range, objmask_sky_jitters,
                                             x_probe, y_probe,
                                             target_jitters);
                    enu_check_error_code("error in sky background subtraction");

                    char * preface = cpl_sprintf("%s_%s", "offset", "sky2");
                    enu_debug_limlist_save(debug_data, target_jitters, preface,
                                           recipe_name, frameset, parlist,
                                           used_frameset);
                    cpl_free(preface);
                    result_jitters = target_jitters;

                } else {

                    /* There is only one sky image. Can't think of ways to improve
                       it so just use the direct image as the sky background and 
                       subtract it */

                    cpl_msg_info(cpl_func,
                                 "Using sky-offset image as background ...");
                    enu_sky_subtract_limlist(sky_method, sky_selector,
                                             sky_time_range, sky_jitters,
                                             x_probe, y_probe,
                                             target_jitters);
                    enu_check_error_code("error in sky background subtraction");

                    char * preface = cpl_sprintf("%s_%s", "offset", "sky2");
                    enu_debug_limlist_save(debug_data, target_jitters, preface,
                                           recipe_name, frameset, parlist,
                                           used_frameset);
                    cpl_free(preface);
                    result_jitters = target_jitters;
                }

	} else if (!strcmp(sky_source, "target")) {

		/* Calculate object masks for the target jitter images
                   and attach them the data */

		eris_nix_img_sky_objmask(recipe_name, context, target_jitters,
				sky_selector, sky_time_range,
				sky_method,
				debug_data, "target",
				frameset, parlist, used_frameset,
				x_probe, y_probe,
				&objmask_jitters);

		/* Use the masked images to estimate the sky background and
                   subtract it */

		cpl_msg_info(cpl_func, "Estimating sky background with source blanking ...");
		enu_sky_subtract_limlist(sky_method, sky_selector, sky_time_range,
				objmask_jitters, x_probe, y_probe,
				objmask_jitters);
		enu_check_error_code("error in sky background subtraction");

		char * preface = cpl_sprintf("%s_%s", "target", "sky2");
		enu_debug_limlist_save(debug_data, objmask_jitters, preface,
				recipe_name, frameset, parlist,
				used_frameset);
		cpl_free(preface);
		result_jitters = objmask_jitters;
	}

	enu_check_error_code("failed to remove sky background");

	/* save the sky-subtracted jitter images to FITS files */

	for (cpl_size i = 0; i < result_jitters->size; i++) {

                /* add necessary things to output header. RA and DEC are needed as
                   would otherwise be overwritten by DFS */
		cpl_propertylist * applist = cpl_propertylist_new();
		cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, out_tag);
                cpl_propertylist_update_string(applist, "PRODCATG",
                                               "ANCILLARY.IMAGE");
                cpl_propertylist_copy_property_regexp(applist, 
                                                      result_jitters->limages[i]->plist,
                                                      "RA|DEC",
                                                      CPL_FALSE);

                /* set RA, DEC from WCS rather than let dfs copy it from ano†her
                   frame */

                {
                    cpl_wcs * wcs = cpl_wcs_new_from_propertylist(
                                       result_jitters->limages[i]->plist);
                    double ra = 0.0;
                    double dec = 0.0;
                    enu_get_ra_dec(wcs, &ra, &dec);

                    cpl_propertylist_update_double(applist, "RA", ra);
                    cpl_propertylist_update_double(applist, "DEC", dec);

                    cpl_wcs_delete(wcs);
                }

		/* Generate output file name and write the file */

                char * out_fname = enu_repreface(cpl_frame_get_filename(
                                                 result_jitters->limages[i]->
                                                 frame),
                                                 "skysub");    

		enu_dfs_save_limage(frameset,
                                    parlist,
                                    used_frameset,
                                    CPL_TRUE,
                                    result_jitters->limages[i],
                                    recipe_name,
                                    result_jitters->limages[i]->frame,
                                    applist,
                                    PACKAGE "/" PACKAGE_VERSION,
                                    out_fname);

		cpl_free(out_fname);
		cpl_propertylist_delete(applist);
	}

	cleanup:
	enu_located_imagelist_delete(object_jitters);
	enu_located_imagelist_delete(objmask_jitters);
	enu_located_imagelist_delete(skysub_jitters);
	enu_located_imagelist_delete(sky_jitters);
	enu_located_imagelist_delete(std_jitters);
	cpl_frameset_delete(used_frameset);

	return (int) cpl_error_get_code();
}

static cpl_error_code
eris_nix_scired_get_split_param_int(const cpl_parameterlist * parlist,
		const char* context, const char* pname, const int min_x, const int min_y,
		const int max_x, const int max_y,
		cpl_size* txmin, cpl_size* tymin, cpl_size* txmax, cpl_size* tymax)
{

	const cpl_parameter* par = NULL;
	char* param_name;


	const char* pname_string_value;
	param_name = cpl_sprintf("%s.%s", context, pname);
	par = cpl_parameterlist_find_const(parlist, param_name);
	pname_string_value = cpl_parameter_get_string(par);
	cpl_free(param_name);

	int nFields;
	unsigned  int end;
	*txmin = 0;
	*tymin = 0;
	*txmax = 0;
	*tymax = 0;
	cpl_msg_warning(cpl_func,"param value: %s", pname_string_value);
	nFields = sscanf(pname_string_value, "%lld,%lld,%lld,%lld%n", txmin,  tymin, txmax, tymax, &end);
	cpl_msg_warning(cpl_func,"nFields:%d", nFields);
	cpl_msg_warning(cpl_func,"end:%d", end);
	if (nFields != 4 || end != strlen(pname_string_value)) {
		cpl_msg_error(cpl_func, "The %s parameter must be "
				"a list of four integers separated by a comma", pname);
		printf("Error reading sky-box-center string\n");
	}

	if (*txmin < min_x) {
		cpl_msg_warning(cpl_func,"%s_x: %lld set it to 1",pname, *txmin);
		*txmin = 1;
	}
	if (*tymin < min_y) {
		cpl_msg_warning(cpl_func,"%s_y: %lld set it to 1",pname, *tymin);
		*tymin = 1;
	}
	if (*txmax > max_x) {
		cpl_msg_warning(cpl_func,"%s_x: %lld set it to 1",pname, *txmax);
		*txmax = 1;
	}
	if (*tymax > max_y) {
		cpl_msg_warning(cpl_func,"%s_y: %lld set it to 1", pname, *tymax);
		*tymax = 1;
	}

	cpl_msg_debug(cpl_func,"%s_x: %lld, %s_y: %lld, %s_x: %lld, %s_y: %lld",
				pname, *txmin, pname, *tymin, pname, *txmax, pname, *tymax);

	eris_check_error_code("eris_nix_scired_get_split_param_int");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief   Calculate matrix with overlaps between jitters. 

  @param    jitters        the jitter sequence
  @param    overlaps       an existing matrix to be filled with overlap values

  Calculate the spatial overlaps between jitters in a sequence.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_calculate_overlaps(
                                       const located_imagelist * jitters,
                                       cpl_matrix ** overlaps) {

    /* Make table of jitter offsets relative to reference jitter, calculate
       the overlap of each jitter with the reference */

    double pixsize = fabs(cpl_propertylist_get_double(jitters->limages[0]->
                          plist, "CD1_1") * 3600.0);

    cpl_size nrow = cpl_matrix_get_nrow(*overlaps);

    for (cpl_size row=0; row < nrow; row++) {
        double ra_centre = cpl_propertylist_get_double(
                                       jitters->limages[row]->plist,
                                       "CRVAL1");
        double dec_centre = cpl_propertylist_get_double(
                                       jitters->limages[row]->plist,
                                       "CRVAL2");

        for (cpl_size col = 0; col <= row; col++) {
            double ra = cpl_propertylist_get_double(
                                       jitters->limages[col]->plist,
                                       "CRVAL1");
            double dec = cpl_propertylist_get_double(
                                       jitters->limages[col]->plist,
                                       "CRVAL2");
            /* size of window if used */
            cpl_size nx = 2048;
            if (cpl_propertylist_has(jitters->limages[col]->plist,
                                     "ESO DET SEQ1 WIN NX")) {
                nx = cpl_propertylist_get_int(
                                       jitters->limages[col]->plist,
                                       "ESO DET SEQ1 WIN NX");
            }
            cpl_size ny = 2048;
            if (cpl_propertylist_has(jitters->limages[col]->plist,
                                     "ESO DET SEQ1 WIN NY")) {
                ny = cpl_propertylist_get_int(
                                       jitters->limages[col]->plist,
                                       "ESO DET SEQ1 WIN NY");
            }
//            cpl_msg_info(cpl_func, "window %d %d %s", (int)nx, (int)ny,
//                         cpl_error_get_message());

            /* calculate overlap with jitter index, -1 for no overlap */

            double ra_size = nx * pixsize;
            double ra_shift = fabs(ra - ra_centre) * 3600.0;
            double ra_overlap = (ra_size - ra_shift) / ra_size;

            double dec_size = ny * pixsize;
            double dec_shift = fabs(dec - dec_centre) * 3600.0;
            double dec_overlap = (dec_size - dec_shift) / dec_size;

            double overlap = -1.0;
            if (ra_overlap > 0.0 && dec_overlap > 0.0) {
                overlap = ra_overlap * dec_overlap;
            }            

            cpl_matrix_set(*overlaps, row, col, overlap);
            cpl_matrix_set(*overlaps, col, row, overlap);
        }
    }

   return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief   Construct a skeleton report calibration report table. 

  @param    ref            index of the reference jitter
  @param    jitters        the jitter sequence
  @param    report         pointer to pre-existing table to be fleshed out

  Construct a table to hold the wcs calibration report. Construct columns and
  fill in those entries that can be.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_report_table(
                                       const cpl_size ref,
                                       const located_imagelist * jitters,
                                       cpl_table ** report) {

    /* Make table of jitter offsets relative to reference jitter, calculate
       the overlap of each jitter with the reference */

    cpl_table_new_column(*report, "Index", CPL_TYPE_INT);
    cpl_table_new_column(*report, "RA offset", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*report, "Dec offset", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*report, "Reference", CPL_TYPE_INT);
    cpl_table_new_column(*report, "Overlap", CPL_TYPE_DOUBLE);
    cpl_table_new_column(*report, "WCS_METHOD", CPL_TYPE_STRING);
    cpl_table_new_column(*report, "Catalogue/File", CPL_TYPE_STRING);
    cpl_table_new_column(*report, "# matches", CPL_TYPE_INT);

    double ra_centre = cpl_propertylist_get_double(
                                       jitters->limages[ref]->plist,
                                       "CRVAL1");
    double dec_centre = cpl_propertylist_get_double(
                                       jitters->limages[ref]->plist,
                                       "CRVAL2");

    for (cpl_size i = 0; i < jitters->size; i++) {
        double ra = cpl_propertylist_get_double(jitters->limages[i]->plist,
                                                "CRVAL1");
        double dec = cpl_propertylist_get_double(jitters->limages[i]->plist,
                                                 "CRVAL2");
        cpl_table_set_int(*report, "Index", i, (int) i);
        cpl_table_set_double(*report, "RA offset", i,
                             (ra - ra_centre) * 3600.0);
        cpl_table_set_double(*report, "Dec offset", i,
                             (dec - dec_centre) * 3600.0);
    }

   return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief   Match catalogues, correct wcs and save catalogues. 

  @param    limage         the jitter to be corrected
  @param    wcs_method     string describing the correction method
  @param    reference      catalogue of reference objects
  @param    ref_qclist     QC keyword/value list of ref catalogue
  @param    reference_name name of reference catalogue (e.g. gaia or image)
  @param    match_radius   size of 'match' circle
  @param    frameset       the frames list
  @param    parlist        the parameters list
  @param    recipe_name    name of recipe
  @param    cat_params     HDRL catalogue parameter structure

  Match reference wcs sources with those in an image, correct the image wcs
  appropriately.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_match_and_correct(
                                       located_image * limage,
                                       const char * wcs_method,
                                       cpl_table * reference,
                                       const cpl_propertylist * ref_qclist, 
                                       const char * reference_name,
                                       const double match_radius,
                                       cpl_size * nmatches,
                                       cpl_frameset * frameset,
                                       const cpl_parameterlist * parlist,
                                       const char * recipe_name,
                                       hdrl_parameter * cat_params) {

   cpl_table * matched_stds = NULL;

   if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code(); 

   /* beware situation where no sources were found in the image */

   if (limage->objects && limage->objects->catalogue) {
       cpl_matrix * ignore = NULL;
       enu_correct_wcs(reference,
                       wcs_method,
                       reference_name,
                       limage,
                       match_radius,
                       &matched_stds,
                       &ignore);
       cpl_matrix_delete(ignore);
       enu_check_error_code("error correcting jitter wcs");

       /* measure of how good was the match */

       if (matched_stds) {
           *nmatches = cpl_table_get_nrow(matched_stds);
       } else {
           *nmatches = 0;
       }

       /* save the reference catalogue */
  
       enu_calc_pixel_coords(reference, limage->plist);
       enu_dfs_save_catalogue(frameset,
                              limage->frame,
                              parlist,
                              reference,
                              ref_qclist,
                              "reference jitter",
                              recipe_name,
                              PACKAGE "/" PACKAGE_VERSION,
                              "refcat.cal_wcs",
                              ERIS_NIX_CAL_WCS_REF_CATALOGUE_PRO_CATG);

       /* save the matched standards catalogue */

       enu_dfs_save_catalogue(frameset,
                              limage->frame,
                              parlist,
                              matched_stds,
                              NULL,
                              "matched standards",
                              recipe_name,
                              PACKAGE "/" PACKAGE_VERSION,
                              "matchcat.cal_wcs",
                              ERIS_NIX_CAL_WCS_MATCH_CATALOGUE_PRO_CATG);

       /* save the image catalogue, recalculated for new wcs */

       cpl_wcs * iwcs = cpl_wcs_new_from_propertylist(limage->plist);
       hdrl_catalogue_result_delete(limage->objects);
       limage->objects = enu_catalogue_compute(limage->himage,
                                               limage->confidence,
                                               iwcs,
                                               cat_params);
       cpl_wcs_delete(iwcs);

       enu_dfs_save_catalogue(frameset,
                              limage->frame,
                              parlist,
                              limage->objects->catalogue,
                              limage->objects->qclist,
                              cpl_frame_get_filename(limage->frame),
                              recipe_name,
                              PACKAGE "/" PACKAGE_VERSION,
                              "cat.cal_wcs",
                              ERIS_NIX_CAL_WCS_CATALOGUE_PRO_CATG);

   } else {
       cpl_msg_info(cpl_func, "cannot match jitters %s", 
                    cpl_error_get_message());
   }

 cleanup:
   cpl_table_delete(matched_stds);   
   return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief   This recipe calibrates the astrometry of ERIS/NIX frames. 

  @param    frameset   the frames list
  @param    parlist    the parameters list
  @param    recipe_name name of recipe
  @param    context  recipe context (eris)

  This comprises dark subtraction, linearization, flat-fielding,
  and association with the master bad-pixel mask.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code eris_nix_scired_cal_wcs(cpl_frameset * frameset,
                                       const cpl_parameterlist * parlist,
                                       const char* recipe_name,
                                       const char* context) {

    cpl_table             * astrom_refcat = NULL;
    hdrl_parameter        * cat_params = NULL;
    const char            * cdssearch_astrom = NULL;
    cpl_table             * ref_jitter_match = NULL;
    located_imagelist     * jitters = NULL; 
    cpl_matrix            * jitter_overlaps = NULL;
    char                  * jitter_ref_name = NULL;
    cpl_table             * matched_stds = NULL;
    int                     minstds = 1;
    located_imagelist     * object_jitters = NULL; 
    const char            * out_tag = NULL;
    const cpl_parameter   * p = NULL;
    cpl_vector            * plan_ref = NULL;
    cpl_vector            * plan_cal = NULL;
    cpl_table             * report = NULL;
    cpl_matrix            * jitter_ref_shift = NULL;
    located_imagelist     * std_jitters = NULL; 
    cpl_frameset          * used = NULL;
    cpl_table             * wcs_match_list = NULL;
    char                  * param_name = NULL;

    enu_check_error_code("%s():%d: An error is already set: %s",
                         cpl_func, __LINE__, cpl_error_get_where());

    /* Check input parameters */

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    /* Set the msg verbosity level from environment variable CPL_MSG_LEVEL
       - not working, so set manually for now */

    cpl_msg_set_level_from_env();
    cpl_msg_severity severity = cpl_msg_get_level();
    cpl_msg_info(cpl_func, "level %d", (int) severity);

    /* check for invalid input */
    if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
    	return CPL_ERROR_BAD_FILE_FORMAT;
    }

    /* Retrieve general input parameters */

    param_name = cpl_sprintf("%s.cdssearch_astrom", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cdssearch_astrom = cpl_parameter_get_string(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.debug-data", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
//    const int debug = cpl_parameter_get_bool(p);
    cpl_free(param_name);

    enu_check_error_code("Error retrieving input parameters");

    /* check required input tags are present */
//    const int ntags = 1;
//    const char* required_tags[1] = {
//    		ERIS_NIX_WCS_MATCHED_CATALOGUE_PRO_CATG // opt
//    };

    // ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG,  or ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG
//    cpl_ensure_code(CPL_ERROR_NONE ==
//    		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
//			CPL_ERROR_ILLEGAL_INPUT);

    /* Identify the RAW and CALIB frames in the input frameset */

    eris_nix_dfs_set_groups(frameset);
    enu_check_error_code("Could not identify RAW and CALIB frames");    

    used = cpl_frameset_new();

    /* Read input data */

    object_jitters = enu_limlist_load_from_frameset(frameset, 
                     ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG, used);
    std_jitters = enu_limlist_load_from_frameset(frameset, 
                  ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG, used);
    enu_check_error_code("Could not load data frames");

    cpl_msg_info(cpl_func, "%d %s frames read", (int) object_jitters->size,
                 ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG);
    cpl_msg_info(cpl_func, "%d %s frames read", (int) std_jitters->size,
                 ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG);

    /* check that the data files in the SoF are as expected */

    enu_check(object_jitters->size > 0 || std_jitters->size > 0,
              CPL_ERROR_DATA_NOT_FOUND, "no input frames found");
    enu_check(!(object_jitters->size > 0 && std_jitters->size > 0),
              CPL_ERROR_ILLEGAL_INPUT, "SoF contains both "
              ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG" and "
              ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG" frames");

    /* is target an object or std? */

    if (object_jitters->size > 0) {
        jitters = object_jitters;
        out_tag = ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG;
        enu_located_imagelist_delete(std_jitters);
        std_jitters = NULL;
    } else if (std_jitters->size > 0) {
        jitters = std_jitters;
        out_tag = ERIS_NIX_CAL_WCS_STD_JITTER_PRO_CATG;
        enu_located_imagelist_delete(object_jitters);
        object_jitters = NULL;
    }

    /* Now that we have access to the data, setup the parameters for
       cataloguing - some elements can be overridden by contents
       of data files */

    param_name = cpl_sprintf("%s.catalogue.det.effective-gain", context);
    p = cpl_parameterlist_find_const(parlist, param_name); 
    double gain = cpl_parameter_get_double(p);
    cpl_free(param_name);

    if (cpl_propertylist_has(jitters->limages[0]->plist, "ESO DARK GAIN")) {
        gain = cpl_propertylist_get_double(jitters->limages[0]->plist,
                                           "ESO DARK GAIN");
        cpl_msg_info(cpl_func, "catalogue params - reading detector gain "
                     "from first jitter %4.1f", gain);
    }
    param_name = cpl_sprintf("%s.catalogue.det.saturation", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double saturation = cpl_parameter_get_double(p);
    cpl_free(param_name);

    if (cpl_propertylist_has(jitters->limages[0]->plist,
                             "ESO DETMON SATURATION")) {
        saturation = cpl_propertylist_get_double(jitters->limages[0]->plist,
                                                 "ESO DETMON SATURATION");
        cpl_msg_info(cpl_func, "catalogue params - reading detector "
                     "saturation level from first jitter %6.1f", saturation);
    }

    param_name = cpl_sprintf("%s.catalogue.obj.min-pixels", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    int minpixels = cpl_parameter_get_int(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.catalogue.obj.threshold", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double threshold = cpl_parameter_get_double(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.catalogue.obj.deblending", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_boolean deblending = cpl_parameter_get_bool(p);
    cpl_free(param_name);

    /* use AOMODE to decide if set catalogue.core-radius and mesh-size to
       defaults or read them from parameters */

    double core_radius = -1.0;
    int mesh_size = -1;
    enu_get_rcore_and_mesh_size(context,
                                parlist,
                                jitters->limages[0]->plist,
                                &core_radius,
                                &mesh_size);

    param_name = cpl_sprintf("%s.catalogue.bkg.estimate", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_boolean estimate = cpl_parameter_get_bool(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.catalogue.bkg.smooth-gauss-fwhm", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double smooth_gauss_fwhm = cpl_parameter_get_double(p);
    cpl_free(param_name);

    cat_params = hdrl_catalogue_parameter_create(minpixels,
                                                 threshold,
                                                 deblending,
                                                 core_radius,
                                                 estimate,
                                                 mesh_size,
                                                 smooth_gauss_fwhm,
                                                 gain,
                                                 saturation,
                                                 HDRL_CATALOGUE_ALL);

    enu_check_error_code("Error setting catalogue params");

    /* set initial description for wcs calibration: raw telescope pointing */

    for (cpl_size i=0; i < jitters->size; i++) {
        cpl_propertylist_update_string(jitters->limages[i]->plist,
                                       "ESO WCS_METHOD",
                                       ERIS_NIX_WCS_TEL_POINTING);
        cpl_propertylist_update_string(jitters->limages[i]->plist,
                                       "WCS_CAT", "none");
    }

    /* estimate the centre of the jitter pattern */

    float diff[1000];
    for (cpl_size i = 0; i < jitters->size; i++) {
        diff[i] = cpl_propertylist_get_double(jitters->limages[i]->plist,
                                              "CRVAL1");
    }
    float ra_centre = casu_med(diff, NULL, jitters->size);
    for (cpl_size i = 0; i < jitters->size; i++) {
        diff[i] = cpl_propertylist_get_double(jitters->limages[i]->plist,
                                              "CRVAL2");
    }
    float dec_centre = casu_med(diff, NULL, jitters->size);
    cpl_msg_info(cpl_func, "Jitter pattern centre at RA=%10.7f Dec=%10.7f",
                 ra_centre, dec_centre);

    /* find the 'reference' jitter, the one nearest the centre of the
       pattern */

    cpl_size jitter_ref = -1;
    double minr2 = 1.0e6;
    for (cpl_size i = 0; i < jitters->size; i++) {
        double ra = cpl_propertylist_get_double(jitters->limages[i]->plist,
                                                "CRVAL1");
        double dec = cpl_propertylist_get_double(jitters->limages[i]->plist,
                                                 "CRVAL2");
        double r2 = pow(ra - ra_centre, 2) + pow(dec - dec_centre, 2);
        if (r2 < minr2) {
            jitter_ref = i;
            minr2 = r2;
        }
    }
    cpl_msg_info(cpl_func, "Nearest jitter to centre is %d", (int) jitter_ref);
    char * refname = cpl_strdup(cpl_frame_get_filename(jitters->
                                limages[jitter_ref]->frame));
    cpl_msg_info(cpl_func, "..filename %s", basename(refname));
    cpl_free(refname);

    /* AMo: TODO creates a mask to trim pixels at image edge. Maybe we can also remove 
            entries from catalog without using a mask */

    cpl_size sx = hdrl_image_get_size_x(jitters->limages[jitter_ref]->himage);
    cpl_size sy = hdrl_image_get_size_y(jitters->limages[jitter_ref]->himage);
    cpl_size txmin = 0;
    cpl_size tymin = 0;
    cpl_size txmax = 0;
    cpl_size tymax = 0;
    eris_nix_scired_get_split_param_int(parlist, context, "edges-trim", 1, 1,
    		sx, sy, &txmin, &tymin, &txmax, &tymax);

    cpl_msg_info(cpl_func,"trim edge params: %lld %lld %lld %lld", txmin, tymin, txmax, tymax);
    cpl_image* image_edge = cpl_image_new(sx, sy, CPL_TYPE_INT);
    if(txmin > 1) {
    	cpl_image_fill_window(image_edge,1, 1, txmin, sy, 1);
    }
   
    if(txmax > 1) {
    	cpl_image_fill_window(image_edge,sx - txmax, 1, sx, sy, 1);
    }
   
    if(tymin > 1) {
    	cpl_image_fill_window(image_edge,1, 1, sx, tymin, 1);
    }
   
    if(tymax > 1) {
    	cpl_image_fill_window(image_edge,1, sy - tymax, sx, sy, 1);
    }
    cpl_mask* mask_edge = cpl_mask_threshold_image_create(image_edge,0.9, 1.1);
    cpl_image_delete(image_edge);
    cpl_mask_save(mask_edge, "mask_edge.fits", NULL, CPL_IO_DEFAULT);
    //cpl_mask* hmask = hdrl_image_get_mask(jitters->limages[jitter_ref]->himage);
    //cpl_mask_or(hmask, mask_edge);
    hdrl_image_reject_from_mask(jitters->limages[jitter_ref]->himage, mask_edge);
    cpl_mask_delete(mask_edge);
    /* AMo: END coded added to trim edges of any input image (to prevent outliers to affect WCS sol) */ 
    
    /* Make table of jitter offsets relative to reference jitter, calculate
       the overlap of each jitter with the reference */

    jitter_overlaps = cpl_matrix_new(jitters->size, jitters->size);
    eris_nix_calculate_overlaps(jitters, &jitter_overlaps);
    cpl_msg_info(cpl_func, " ");
    cpl_msg_info(cpl_func, "Jitter overlaps matrix:");
    cpl_matrix_dump(jitter_overlaps, NULL);

    /* Make calibration plan, that is the list of (reference, jitter) pairs */

    cpl_vector * matched = cpl_vector_new(jitters->size);
    plan_ref = cpl_vector_new(jitters->size);
    plan_cal = cpl_vector_new(jitters->size);
    for (cpl_size j=0; j<cpl_vector_get_size(matched); j++) {
        cpl_vector_set(matched, j, 0.0);
        cpl_vector_set(plan_ref, j, -1.0);
        cpl_vector_set(plan_cal, j, -1.0);
    }

    /* ..reference jitter agains Gaia first */
    cpl_vector_set(matched, jitter_ref, 1.0);
    cpl_msg_info(cpl_func, " ");
    cpl_msg_info(cpl_func, "WCS calibration plan:");
    cpl_msg_info(cpl_func, "ref Gaia calibrates %d", (int)jitter_ref); 

    /* ..then subsequent jitters, based on the size of the overlap
         between them */
 
    int matching = CPL_TRUE;
    cpl_size iplan = 0;
    while (matching) {
        cpl_size row_next = -1;
        cpl_size col_next = -1;
        cpl_matrix * overlaps_copy = cpl_matrix_duplicate(jitter_overlaps);

        /* highlight calibrated columns and uncalibrated rows */

        for (cpl_size col=0; col<jitters->size; col++) {
            if (cpl_vector_get(matched, col) < 1.0) {
                for (cpl_size row=0; row<jitters->size; row++) {
                    cpl_matrix_set(overlaps_copy, row, col, -1.0);
                }
            }
        }
        for (cpl_size row=0; row<jitters->size; row++) {
            if (cpl_vector_get(matched, row) > 0.0) {
                for (cpl_size col=0; col<jitters->size; col++) {
                    cpl_matrix_set(overlaps_copy, row, col, -1.0);
                }
            }
        }

        /* look for max overlap remaining - the col will be the 
           ref jitter and the row the jitter to calibrate */

        if (cpl_matrix_get_max(overlaps_copy) > 0.0) {
            cpl_matrix_get_maxpos(overlaps_copy, &row_next, &col_next);
            double overlap_next = cpl_matrix_get(overlaps_copy, row_next,
                                                 col_next);
            cpl_msg_info(cpl_func, "ref %d -> %d overlap %4.2f", (int)col_next,
                         (int)row_next, overlap_next); 
            cpl_vector_set(matched, row_next, 1.0);

            cpl_vector_set(plan_ref, iplan, col_next);
            cpl_vector_set(plan_cal, iplan, row_next);
            iplan++;
        } else {
            matching = CPL_FALSE;
        }
        cpl_matrix_delete(overlaps_copy);
    }
    cpl_vector_delete(matched);

    /* build the calibration 'report' table which will hold information
       on the calibration plan and results */

    report = cpl_table_new(jitters->size);
    eris_nix_report_table(jitter_ref, jitters, &report);

    /* Starting WCS calibration ... 

       First, is there a file with a list of positions to calibrate 
       the reference jitter? */

    cpl_frame * wcs_matched_frame = cpl_frameset_find (frameset,
                                    ERIS_NIX_WCS_MATCHED_CATALOGUE_PRO_CATG);
    const char * wcs_match_list_name = NULL;
    if (wcs_matched_frame) {
        wcs_match_list_name = cpl_frame_get_filename (wcs_matched_frame);

        /* There is, read it */

        cpl_msg_info(cpl_func, "found manual matching list %s",
                     wcs_match_list_name);
        wcs_match_list = cpl_table_load(wcs_match_list_name, 1, 0);
        enu_check_error_code("failed to read match list");

        /* Select the table row that matches the reference jitter */

        const char * ref_jitter_name = cpl_frame_get_filename(
                                       jitters->limages[jitter_ref]->frame);
        cpl_msg_info(cpl_func, "..reference jitter %s", ref_jitter_name);
        cpl_table_unselect_all(wcs_match_list);
        for (cpl_size row = 0; row < cpl_table_get_nrow(wcs_match_list);
             row++) {
            if (strstr(ref_jitter_name, cpl_table_get_string(wcs_match_list,
                       "Filename", row))) {
                cpl_table_select_row(wcs_match_list, row);
                ref_jitter_match = cpl_table_extract_selected(wcs_match_list);
                cpl_msg_info(cpl_func, "..found entry for reference jitter");
                break;
            }
        }

        if (!ref_jitter_match) {
            cpl_msg_info(cpl_func, "..reference jitter not found in manual "
                         "matching file");
        } else {
            cpl_msg_info(cpl_func, "..calculating WCS correction for "
                         "reference jitter");

            /* Using the existing wcs info, calculate the predicted x and y
               positions for the given celestial point in the reference 
               jitter */

            cpl_table_new_column(ref_jitter_match, "xpredict", CPL_TYPE_FLOAT);
            cpl_table_new_column(ref_jitter_match, "ypredict", CPL_TYPE_FLOAT);
            double * ra = cpl_table_get_data_double(ref_jitter_match, "RA");
            double * dec = cpl_table_get_data_double(ref_jitter_match, "DEC");
            cpl_matrix * celestial = cpl_matrix_new(1, 2);
            cpl_matrix_set(celestial, 0, 0, ra[0]);
            cpl_matrix_set(celestial, 0, 1, dec[0]);
            cpl_wcs * iwcs = cpl_wcs_new_from_propertylist(
                             jitters->limages[jitter_ref]->plist);
            cpl_matrix * predicted = NULL;
            cpl_array * status = NULL;
            cpl_wcs_convert(iwcs, celestial, &predicted, &status,
                            CPL_WCS_WORLD2PHYS);
            cpl_wcs_delete(iwcs);

            /* subtract the predicted positions from the measured in jitter 1,
               giving the observed coarse shift in x, y. This will be an
               estimate of the telescope pointing error for the first jitter;
               the relative error of subsequent jitters will be much smaller */

            double * x_coordinate = cpl_table_get_data_double(ref_jitter_match, 
                                                              "X_coordinate");
            double * y_coordinate = cpl_table_get_data_double(ref_jitter_match,
                                                              "Y_coordinate");
            cpl_matrix * xy = cpl_matrix_new(1, 2);
            cpl_matrix_set(xy, 0, 0, x_coordinate[0]);
            cpl_matrix_set(xy, 0, 1, y_coordinate[0]);
            cpl_matrix_subtract(xy, predicted);
            jitter_ref_shift = xy;
            cpl_msg_info(cpl_func, "..reference jitter shift is");
            cpl_matrix_dump(jitter_ref_shift, NULL);

            cpl_msg_info(cpl_func, "..applying WCS correction to reference "
                         "jitter");

            double crpix1 = cpl_propertylist_get_double(
                            jitters->limages[jitter_ref]->plist, "CRPIX1");
            cpl_propertylist_update_double(jitters->limages[jitter_ref]->plist,
                                       "CRPIX1", crpix1 +
                                       cpl_matrix_get(jitter_ref_shift, 0, 0));
            double crpix2 = cpl_propertylist_get_double(
                            jitters->limages[jitter_ref]->plist, "CRPIX2");
            cpl_propertylist_update_double(jitters->limages[jitter_ref]->plist,
                                       "CRPIX2", crpix2 +
                                       cpl_matrix_get(jitter_ref_shift, 0, 1));

            /* store correction method */

            cpl_propertylist_update_string(jitters->limages[jitter_ref]->plist,
                                           "ESO WCS_METHOD",
                                           ERIS_NIX_WCS_CATALOGUE_MATCH);
            cpl_propertylist_update_string(jitters->limages[jitter_ref]->plist,
                                           "WCS_CAT",
                                           cpl_table_get_string(ref_jitter_match, 
                                           "Catalogue", 0));
            enu_check_error_code("failure to set corrected wcs");

            /* save the reference catalogue */

            enu_calc_pixel_coords(ref_jitter_match,
                                  jitters->limages[jitter_ref]->plist);
            enu_dfs_save_catalogue(frameset,
                                   jitters->limages[jitter_ref]->frame,
                                   parlist,
                                   ref_jitter_match,
                                   NULL,
                                   cpl_table_get_string(ref_jitter_match, 
                                                        "Catalogue", 0),
                                   recipe_name,
                                   PACKAGE "/" PACKAGE_VERSION,
                                   "refcat.cal_wcs",
                                   ERIS_NIX_CAL_WCS_REF_CATALOGUE_PRO_CATG);

            /* save the matched standards catalogue - same as reference
               cat but done for consistency with other jitters */

            enu_dfs_save_catalogue(frameset,
                                   jitters->limages[jitter_ref]->frame,
                                   parlist,
                                   ref_jitter_match,
                                   NULL,
                                   cpl_table_get_string(ref_jitter_match, 
                                                        "Catalogue", 0),
                                   recipe_name,
                                   PACKAGE "/" PACKAGE_VERSION,
                                   "matchcat.cal_wcs",
                                   ERIS_NIX_CAL_WCS_MATCH_CATALOGUE_PRO_CATG);

            /* save the image catalogue, calculated for new wcs */

            cpl_wcs * wcs = cpl_wcs_new_from_propertylist(
                            jitters->limages[jitter_ref]->plist);
            cpl_msg_info(cpl_func, "cataloguing reference jitter");
            jitters->limages[jitter_ref]->objects = enu_catalogue_compute(
                                       jitters->limages[jitter_ref]->himage,
                                       jitters->limages[jitter_ref]->confidence,
                                       wcs,
                                       cat_params);
            cpl_wcs_delete(wcs);
            enu_dfs_save_catalogue(frameset,
                                   jitters->limages[jitter_ref]->frame,
                                   parlist,
                                   jitters->limages[jitter_ref]->objects->catalogue,
                                   jitters->limages[jitter_ref]->objects->qclist,
                                   cpl_frame_get_filename(jitters->
                                                          limages[jitter_ref]->
                                                          frame),
                                   recipe_name,
                                   PACKAGE "/" PACKAGE_VERSION,
                                   "cat.cal_wcs",
                                   ERIS_NIX_CAL_WCS_CATALOGUE_PRO_CATG);
        }
        cpl_msg_info(cpl_func, "..");
    }

    if (jitter_ref_shift == NULL) {

        /* If that didn't work, try matching sources in jitter_ref with 
           the astrometric catalogue specified */

        if (strcmp(cdssearch_astrom, "none")) {

            cpl_msg_info(cpl_func, " ");
            cpl_msg_info(cpl_func, "Try to match sources in reference jitter "
                         "with %s", cdssearch_astrom);

            /* Read the astrom catalogue for the image area of the reference
               jitter from CDS */

            /* use casu_getstds to do the read as it forms the table in the way
               expected by other casu routines */

            astrom_refcat = NULL;
            int casu_status = CASU_OK;
            char * catname = cpl_strdup(cdssearch_astrom);
            int cdschoice = casu_get_cdschoice(cdssearch_astrom);
            casu_getstds(jitters->limages[jitter_ref]->plist,
                         CPL_FALSE, NULL, catname, cdschoice, NULL,
                         &astrom_refcat, NULL, &casu_status);

            if (astrom_refcat) {

                /* Some standards were found, are there enough? */

                cpl_size nstds = cpl_table_get_nrow(astrom_refcat);
                enu_check(nstds >= minstds, CPL_ERROR_INCOMPATIBLE_INPUT,
                          "nstds = %d, need at least %d", (int)nstds, minstds);

                /* getstds returns a "Dec" column whereas the rest of the code
                   expects "DEC". Fix this. */

                if (cpl_table_has_column(astrom_refcat, "Dec") &&
                    !cpl_table_has_column(astrom_refcat, "DEC")) {
                    cpl_table_duplicate_column(astrom_refcat, "DEC", 
                                               astrom_refcat, "Dec");
                    cpl_table_erase_column(astrom_refcat, "Dec");
                }

                cpl_msg_info(cpl_func, "cataloguing reference jitter %d",
                             (int) jitter_ref);
                cpl_wcs * iwcs = cpl_wcs_new_from_propertylist
                    (jitters->limages[jitter_ref]->plist);
                jitters->limages[jitter_ref]->objects = enu_catalogue_compute(
                                        jitters->limages[jitter_ref]->himage,
                                        jitters->limages[jitter_ref]->confidence,
                                        iwcs,
                                        cat_params);
                cpl_wcs_delete(iwcs);

                /* catalogue standards can move so make match_rad fairly loose */
                double match_rad = 10.0;
                cpl_size nmatches = 0;
                eris_nix_match_and_correct(jitters->limages[jitter_ref],
                                           ERIS_NIX_WCS_CATALOGUE_MATCH,
                                           astrom_refcat,
                                           NULL,
                                           cdssearch_astrom,
                                           match_rad,
                                           &nmatches,
                                           frameset,
                                           parlist,
                                           recipe_name,
                                           cat_params);

                cpl_table_set_int(report, "Reference", jitter_ref, -1);
                cpl_table_set_double(report, "Overlap", jitter_ref, 
                                     cpl_matrix_get(jitter_overlaps,
                                     jitter_ref, jitter_ref));
                cpl_table_set_string(report, "WCS_METHOD", jitter_ref,
                                     ERIS_NIX_WCS_CATALOGUE_MATCH);
                cpl_table_set_string(report, "Catalogue/File", jitter_ref,
                                     catname);
                cpl_table_set_int(report, "# matches", jitter_ref, nmatches);
            }
            cpl_msg_info(cpl_func, "..");
            enu_check_error_code("error correcting wcs of reference jitter");
            cpl_free(catname);
        }
    }

    /* correction of other jitters - work through the calibration 'plan' calculated
       earlier and 'match and correct' the reference/jitter pairs.
       The first reference will be the reference jitter, which anchors the 
       whole thing. */

    enu_catalogue_limlist(jitters, cat_params);

    if (!(jitters->limages[jitter_ref]->objects)) {
        cpl_msg_warning(cpl_func, "Unable to absolute correct jitters "
                        "because no sources found in reference jitter");
    }

    for (cpl_size i = 0; i < jitters->size; i++) {
        cpl_size jref = cpl_vector_get(plan_ref, i);
        cpl_size jcal = cpl_vector_get(plan_cal, i);

        if (jref >= 0 && jitters->limages[jref]->objects != NULL) {
            cpl_msg_info(cpl_func, " ");
            cpl_msg_info(cpl_func, "using jitter %d to correct %d",
                         (int)jref, (int)jcal);

            /* basename does mysterious things sometimes (in this case 
               result is changed by subsequent calls in the loop), take 
               a copy of its result to freeze it */

            char * fname = cpl_strdup(cpl_frame_get_filename(
                                      jitters->limages[jref]->frame));
            char * jref_name = cpl_strdup(basename(fname));
            cpl_free(fname);

            /* objects do not move between jitters so match_rad can be 
               relatively tight */
            double match_rad = 2.0;
            cpl_size nmatches = 0;
            eris_nix_match_and_correct(jitters->limages[jcal],
                                       ERIS_NIX_WCS_JITTER_RELATIVE,
                                       jitters->limages[jref]->
                                         objects->catalogue,
                                       jitters->limages[jref]->
                                         objects->qclist,
                                       jref_name,
                                       match_rad,
                                       &nmatches,
                                       frameset,
                                       parlist,
                                       recipe_name,
                                       cat_params);


            cpl_table_set_int(report, "Reference", jcal, jref);
            cpl_table_set_double(report, "Overlap", jcal, 
                                 cpl_matrix_get(jitter_overlaps, jref, jcal));
            cpl_table_set_string(report, "WCS_METHOD", jcal,
                                 ERIS_NIX_WCS_JITTER_RELATIVE);
            cpl_table_set_string(report, "Catalogue/File", jcal, jref_name);
            cpl_table_set_int(report, "# matches", jcal, nmatches);
            cpl_free(jref_name);
        }
    }
    cpl_msg_info(cpl_func, "..");

    /* save the wcs-corrected jitter images to FITS files */

    for (cpl_size i = 0; i < jitters->size; i++) {

        /* Generate output file name, provenance frameset */

        char * out_fname = enu_repreface(cpl_frame_get_filename(jitters->
                                         limages[i]->frame),
                                         "cal_wcs");

        cpl_propertylist * applist = cpl_propertylist_new();
        cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, out_tag);
        cpl_propertylist_update_string(applist, "PRODCATG", "ANCILLARY.IMAGE");
        cpl_propertylist_copy_property(applist, jitters->limages[i]->plist, 
                                       "ESO WCS_METHOD");
        cpl_propertylist_copy_property(applist, jitters->limages[i]->plist, 
                                       "WCS_CAT");

       /* set RA, DEC from WCS rather than let dfs copy it from some random
          frame */

        {
            cpl_wcs * wcs = cpl_wcs_new_from_propertylist(
                                       jitters->limages[i]->plist);
            double ra = 0.0;
            double dec = 0.0;
            enu_get_ra_dec(wcs, &ra, &dec);

            cpl_propertylist_update_double(applist, "RA", ra);
            cpl_propertylist_update_double(applist, "DEC", dec);

            cpl_wcs_delete(wcs);
        }

        /* provenance is the frame itself plus the frame it's referenced to */
        cpl_frameset * provenance = cpl_frameset_new();
        cpl_frameset_insert(provenance, cpl_frame_duplicate(
                            jitters->limages[i]->frame));
        if (i != jitter_ref) {
            cpl_frameset_insert(provenance, cpl_frame_duplicate(
                                jitters->limages[jitter_ref]->frame));
        }
        enu_dfs_save_limage(frameset,
                            parlist,
                            provenance,
                            CPL_TRUE,
                            jitters->limages[i],
                            recipe_name,
                            jitters->limages[i]->frame,
                            applist,
                            PACKAGE "/" PACKAGE_VERSION,
                            out_fname);

        cpl_free(out_fname);
        cpl_frameset_delete(provenance);
        cpl_propertylist_delete(applist);
    }

    cpl_table_dump(report, 0, 100, NULL);

    char * in_fname = cpl_strdup(cpl_frame_get_filename(jitters->limages[0]->
                                 frame));
    char * out_fname = cpl_sprintf("cal_wcs_report.%s", basename(in_fname));

    cpl_propertylist * report_plist = cpl_propertylist_new(); 
    cpl_propertylist_update_string(report_plist, CPL_DFS_PRO_CATG,
                                   ERIS_NIX_CAL_WCS_REPORT_PRO_CATG);
    cpl_table_save (report, report_plist, NULL, out_fname, CPL_IO_CREATE);
    cpl_propertylist_delete(report_plist);
    cpl_free(in_fname);
    cpl_free(out_fname);        

cleanup:
    cpl_table_delete(astrom_refcat);
    hdrl_parameter_delete(cat_params);
    cpl_table_delete(ref_jitter_match);
    cpl_free(jitter_ref_name);
    cpl_matrix_delete(jitter_ref_shift);
    cpl_matrix_delete(jitter_overlaps);
    cpl_table_delete(matched_stds);
    cpl_vector_delete(plan_ref);
    cpl_vector_delete(plan_cal);
    enu_located_imagelist_delete(object_jitters);
    cpl_table_delete(report);
    enu_located_imagelist_delete(std_jitters);
    cpl_frameset_delete(used);
    cpl_table_delete(wcs_match_list);

    return (int) cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Cast any double columns in the given table to float.
  @return   A CPL error code.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code cal_phot_cast_columns_to_float(cpl_table * table) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* let NULL table drop through */

    if (table == NULL) return CPL_ERROR_NONE;

    cpl_array * colnames = cpl_table_get_column_names(table);
    for (cpl_size icol=0; icol < cpl_array_get_size(colnames); icol++) {
        const char * colname = cpl_array_get_string(colnames, icol);
        if (cpl_table_get_column_type(table, colname) == CPL_TYPE_DOUBLE ||
            cpl_table_get_column_type(table, colname) == CPL_TYPE_LONG_LONG) {
            cpl_table_cast_column(table, colname, colname, CPL_TYPE_FLOAT);
        }
    }
    cpl_array_delete(colnames);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Save the calibrated results to FITS
  @param limage      The located image to be saved
  @param tag         Value for ESO PRO CATG keyword
  @param frameset    The list of input frames for the recipe
  @param parlist     The list of input parameters
  @param used        The list of raw/calibration frames used for this product
  @param recipe_name The name of the recipe calling this routine
  @return   CPL_ERROR_NONE if all goes well, otherwise an error code
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_cal_phot_save(
                               located_image * limage,
                               const char * tag,
                               cpl_frameset * frameset,
                               const cpl_parameterlist * parlist,
                               const char * recipe_name) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    /* calculate magnitude limit using HDRL */
    /* ..set mode calculation following HDRL docs example */

    /* try to get actual fwhm from catalogue, failing that use the PSF
       currently in the header */

    double cd1_1 = cpl_propertylist_get_double(limage->plist, "CD1_1");
    double cd2_1 = cpl_propertylist_get_double(limage->plist, "CD2_1");
    double pixsize = sqrt(pow(cd1_1, 2) + pow(cd2_1, 2)) * 3600.0;

    double fwhm_pix = -1.0;
    if (limage->objects && 
        limage->objects->qclist &&
        cpl_propertylist_has(limage->objects->qclist, "ESO QC IMAGE_SIZE")) {

        fwhm_pix = cpl_propertylist_get_double(limage->objects->qclist,
                                               "ESO QC IMAGE_SIZE");
    }

    if (fwhm_pix != -1.0) {
        cpl_propertylist_update_double(limage->plist, "PSF_FWHM",
                                       fwhm_pix * pixsize);
        cpl_propertylist_set_comment(limage->plist, "PSF_FWHM",
                                    "Average FWHM of stellar objects[arcsec]");
    } else {
        double fwhm = cpl_propertylist_get_double(limage->plist, "PSF_FWHM");
        fwhm_pix = fwhm / pixsize;
    }

    double photzp = cpl_propertylist_get_double(limage->plist, "PHOTZP");
    double abmaglim = -1.0;
    enu_calc_maglim(limage, photzp, fwhm_pix, &abmaglim);

    /* update DFS keywords affected by this recipe */

    cpl_propertylist * applist = cpl_propertylist_new();

    cpl_propertylist_copy_property(applist, limage->plist, "FLUXCAL");
    cpl_propertylist_copy_property(applist, limage->plist, "ESO QC MAGZPT");
    cpl_propertylist_copy_property(applist, limage->plist, "ESO QC MAGZERR");
    cpl_propertylist_update_double(applist, "ABMAGLIM", abmaglim);
    cpl_propertylist_set_comment(applist, "ABMAGLIM", "5-sigma "
                                 "limiting AB magnitude");

    /* ABMAGSAT is calculated by casu_photcal_extinct assumimg satlev=65000,
       when it should be ESO.DETMON.SATURATION / ESO.DET.SEQ1.DIT.
       Modify the value accordingly */
    double saturation = cpl_propertylist_get_double(limage->plist,
                                                    "ESO DETMON SATURATION");
    double dit = enu_get_dit(limage->plist);
    double satlev = saturation / dit;
    double abmagsat = cpl_propertylist_get_double(limage->plist, "ABMAGSAT");
    abmagsat = abmagsat + 2.5 * log10(65000 / satlev);
    cpl_propertylist_copy_property(applist, limage->plist, "ABMAGSAT");
    cpl_propertylist_update_double(applist, "ABMAGSAT", abmagsat);
    cpl_propertylist_copy_property(applist, limage->plist, "PHOTZP");
    cpl_propertylist_copy_property(applist, limage->plist, "PHOTZPER");
    cpl_propertylist_copy_property(applist, limage->plist, "ZPMETHOD");
    if (cpl_propertylist_has(limage->plist, "ZP_CAT")) {
        cpl_propertylist_copy_property(applist, limage->plist, "ZP_CAT");
    }
    cpl_propertylist_copy_property(applist, limage->plist, "ESO DRS EXTCOEF");

    /* set PRO.CATG */

    cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, tag);
    cpl_propertylist_update_string(applist, "PRODCATG", "ANCILLARY.IMAGE");

    /* set RA, DEC from WCS rather than let dfs copy it from one of the
       frames */

    {
        cpl_wcs * wcs = cpl_wcs_new_from_propertylist(limage->plist);
        double ra = 0.0;
        double dec = 0.0;
        enu_get_ra_dec(wcs, &ra, &dec);

        cpl_propertylist_update_double(applist, "RA", ra);
        cpl_propertylist_update_double(applist, "DEC", dec);

        cpl_wcs_delete(wcs);
    }

    /* provenance frameset, files that contributed to this reduction */

    cpl_frameset * provenance = cpl_frameset_new();
    cpl_frameset_insert(provenance, cpl_frame_duplicate(limage->frame));

    /* Generate output file name, write the file. */

    char * out_fname = enu_repreface(cpl_frame_get_filename(limage->frame),
                                     "cal_phot");    

    enu_dfs_save_limage(frameset,
                        parlist,
                        provenance,
                        CPL_TRUE,
                        limage,
                        recipe_name,
                        limage->frame,
                        applist,
                        PACKAGE "/" PACKAGE_VERSION,
                        out_fname);

    cpl_free(out_fname);
    cpl_frameset_delete(provenance);
    cpl_propertylist_delete(applist);

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Set default photometric calibration from the photometric data file.
  @param    frameset   the frames list
  @param    jitter     the image to be calibrated
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_cal_phot_default(
                                           cpl_frameset * frameset,
                                           located_image * jitter) {

    cpl_table             * phot_data = NULL;
    cpl_table             * selected = NULL;

    /* Read in the data file with default photometric values */

    cpl_frame * phot_data_frame = cpl_frameset_find (frameset,
                                  ERIS_NIX_PHOT_DATA_PRO_CATG);
    const char * phot_data_file = cpl_frame_get_filename (phot_data_frame);
    enu_check_error_code("Failed to read NIX photometry file name");

    cpl_msg_info(cpl_func, "..transferring default photometric calibration "
                 "from %s", phot_data_file);

    /* Read in the instrument photometry table, with info on conversion
       between standard and instrument filters */

    const char * pcat = "2MASS";
    cpl_size phot_ext = cpl_fits_find_extension(phot_data_file, pcat);
    enu_check(phot_ext > 0, CPL_ERROR_INCOMPATIBLE_INPUT, 
              "photometry information for '%s' not found in %s",
              pcat, phot_data_file);

    phot_data = cpl_table_load(phot_data_file, phot_ext, 0);

    /* get the filter used for the observation and the relevant line of
       the phot_data table */

    const char * filter = enu_get_filter(jitter->plist);
    enu_check(filter != NULL, CPL_ERROR_INCOMPATIBLE_INPUT,
              "Failure to find filter name");

    /* comparison is treated as a regular expression */
    char * comparison = cpl_sprintf("^%s$", filter);
    cpl_size nsel = cpl_table_and_selected_string(phot_data, "filter_name",
                                                  CPL_EQUAL_TO, comparison);
    cpl_free(comparison);
    if (nsel == 1) {
        cpl_msg_info(cpl_func, "Filter %s located in %s", filter,
                     phot_data_file);
        selected = cpl_table_extract_selected(phot_data);
        cpl_table_dump(selected, 0, 1, NULL);
        cpl_table_delete(phot_data);
        phot_data = selected;
    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                              "No entry for filter %s in %s", filter,
                              phot_data_file);
    }
    enu_check_error_code("Failure to read photometry data for filter %s",
                          filter);

    int ignore = 0;
    double default_zp = cpl_table_get_double(phot_data, "default_zp", 0,
                                             &ignore);
    double default_zp_err = cpl_table_get_double(phot_data,
                                                 "default_zp_err", 0,
                                                 &ignore);
    double extcoef = cpl_table_get_double(phot_data, "atm_extcoef", 0,
                                          &ignore);

    /* Set default zeropoint in the jitter image */

    /* PHOTZP absorbs the exposure time and extinction for the jitter
       so this correction can be applied directly to the image data.
       Because data are in adu/s the exposure time for this calculation
       is 1.0 */

    double airmass = enu_get_airmass(jitter->plist);
    const double extinct = extcoef * (airmass - 1.0);
    const double exptime = 1.0;
    cpl_msg_info(cpl_func, "default %f %f %f", default_zp, exptime, extinct);
    const double photzp = default_zp + 2.5*log10(exptime) - extinct;

    /* Set ZPMETHOD to reflect the calibration method used */

    const char * zp_method = "DEFAULT";
    const char * zp_method_comment = "ZP taken from filter default value";
    const char * magzpt_comment = "[mag] photometric zeropoint";
    const char * magzerr_comment = "[mag] photometric zeropoint error";
    const char * photzp_comment = "[mag] photometric zeropoint";
    const char * photzper_comment = "[mag] uncertainty on PHOTZP";
    const char * extcoef_comment = "[mag] Assumed extinction coefficient";

    /* write the calibration keywords to the jitter header */

    cpl_propertylist_update_string(jitter->plist, "FLUXCAL", "ABSOLUTE");
    cpl_propertylist_set_comment(jitter->plist, "FLUXCAL", 
                                 "quality of flux calibration");
    cpl_propertylist_update_double(jitter->plist, "ESO QC MAGZPT",
                                   default_zp);
    cpl_propertylist_set_comment(jitter->plist, "ESO QC MAGZPT",
                                 magzpt_comment);
    cpl_propertylist_update_double(jitter->plist, "ESO QC MAGZERR",
                                   default_zp_err);
    cpl_propertylist_set_comment(jitter->plist, "ESO QC MAGZERR",
                                 magzerr_comment);
    cpl_propertylist_update_double(jitter->plist, "PHOTZP", photzp);
    cpl_propertylist_set_comment(jitter->plist, "PHOTZP", photzp_comment);
    cpl_propertylist_update_double(jitter->plist, "PHOTZPER", default_zp_err);
    cpl_propertylist_set_comment(jitter->plist, "PHOTZPER", photzper_comment);
    cpl_propertylist_update_string(jitter->plist, "ZPMETHOD", zp_method);
    cpl_propertylist_set_comment(jitter->plist, "ZPMETHOD",
                                 zp_method_comment);
    cpl_propertylist_update_double(jitter->plist, "ESO DRS EXTCOEF", extcoef);
    cpl_propertylist_set_comment(jitter->plist, "ESO DRS EXTCOEF",
                                 extcoef_comment);

    /* and to the catalogue header */

    if (jitter->objects && jitter->objects->qclist) {
        cpl_propertylist_update_double(jitter->objects->qclist,
                                       "ESO QC MAGZPT", default_zp);
        cpl_propertylist_set_comment(jitter->objects->qclist,
                                     "ESO QC MAGZPT", magzpt_comment);
        cpl_propertylist_update_double(jitter->objects->qclist,
                                       "ESO QC MAGZERR", default_zp_err);
        cpl_propertylist_set_comment(jitter->objects->qclist,
                                     "ESO QC MAGZERR", magzerr_comment);
        cpl_propertylist_update_double(jitter->objects->qclist, "PHOTZP",
                                       photzp);
        cpl_propertylist_set_comment(jitter->objects->qclist, "PHOTZP",
                                     photzp_comment);
        cpl_propertylist_update_string(jitter->objects->qclist,
                                       "ZPMETHOD", zp_method);
        cpl_propertylist_set_comment(jitter->objects->qclist,
                                     "ZPMETHOD", zp_method_comment);
        cpl_propertylist_update_double(jitter->plist, "ESO DRS EXTCOEF",
                                       extcoef);
        cpl_propertylist_set_comment(jitter->plist, "ESO DRS EXTCOEF",
                                     extcoef_comment);
    }

cleanup:

    cpl_table_delete(phot_data);
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Transfer photometric calibration from an external image to the list.
  @param    frameset    the frames list
  @param    parlist     the parameters list
  @param    jitters     the list of images to be calibrated
  @param    recipe_name the name of the recipe calling this routine
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_cal_phot_external(
                                           cpl_frameset * frameset,
                                           const cpl_parameterlist * parlist,
                                           located_imagelist * standards,
                                           located_imagelist * jitters,
                                           const char * out_catg,
                                           const char * recipe_name) {

    cpl_msg_info(cpl_func, "..transferring photometric calibration from %s",
                 cpl_frame_get_filename(standards->limages[0]->frame));

    /* get the filter used for the observation, check that it is the same
       as in the calibrated image */

    const char * filter = enu_get_filter(jitters->limages[0]->plist);
    const char * std_filter = cpl_propertylist_get_string(
                              standards->limages[0]->plist,
                              "FILTER");
    enu_check(!strcmp(filter, std_filter), CPL_ERROR_INCOMPATIBLE_INPUT,
                              "Standard field filter (%s) does not match "
                              "target (%s)", std_filter, filter);

    /* Read the photometric calibration of the standard image.
       MAGZPT is for the image normalised by EXPTIME and at zenith */

    const double magzpt = cpl_propertylist_get_double(
                          standards->limages[0]->plist, "ESO QC MAGZPT");
    const char * magzpt_comment = cpl_propertylist_get_comment(
                                  standards->limages[0]->plist,
                                  "ESO QC MAGZPT");
    const double magzerr = cpl_propertylist_get_double(
                           standards->limages[0]->plist, "ESO QC MAGZERR");
    const char * magzerr_comment = cpl_propertylist_get_comment(
                                   standards->limages[0]->plist,
                                   "ESO QC MAGZERR");
    const char * photzp_comment = cpl_propertylist_get_comment(
                                  standards->limages[0]->plist, "PHOTZP");

    /* read the atmospheric extinction assumed for the standard field
       and the zeropoint at zenith */

    const double extcoef = cpl_propertylist_get_double(
                           standards->limages[0]->plist,
                           "ESO DRS EXTCOEF");
    const char * extcoef_comment = "[mag] Assumed extinction coefficient";
/*
    const char * extcoef_comment = cpl_propertylist_get_comment(
                                   standards->limages[0]->plist, "EXTCOEF");
*/

    /* save the photometric calibrated jitter images to FITS files */

    for (cpl_size j = 0; j < jitters->size; j++) {

        /* PHOTZP absorbs the exposure time and extinction for the jitter
           so this correction can be applied directly to the image data.
           NIX data are in DN/sec so here exposure time = 1 */

        const double airmass = enu_get_airmass(jitters->limages[j]->plist);
        const double extinct = extcoef * (airmass - 1.0);
        const double exptime = 1.0;
        const double photzp = magzpt + 2.5*log10(exptime) - extinct;

        /* write the calibration keywords to the jitter header */

        cpl_propertylist_update_string(jitters->limages[j]->plist,
                                       "FLUXCAL", "ABSOLUTE");
        cpl_propertylist_set_comment(jitters->limages[j]->plist, "FLUXCAL", 
                                     "quality of flux calibration");
        cpl_propertylist_update_double(jitters->limages[j]->plist,
                                       "ESO QC MAGZPT", magzpt);
        cpl_propertylist_set_comment(jitters->limages[j]->plist,
                                     "ESO QC MAGZPT", magzpt_comment);
        cpl_propertylist_update_double(jitters->limages[j]->plist,
                                       "ESO QC MAGZERR", magzerr);
        cpl_propertylist_set_comment(jitters->limages[j]->plist,
                                     "ESO QC MAGZERR", magzerr_comment);
        cpl_propertylist_update_double(jitters->limages[j]->plist,
                                       "PHOTZP", photzp);
        cpl_propertylist_set_comment(jitters->limages[j]->plist,
                                     "PHOTZP", photzp_comment);
        cpl_propertylist_update_double(jitters->limages[j]->plist,
                                       "ESO DRS EXTCOEF", extcoef);
        cpl_propertylist_set_comment(jitters->limages[j]->plist,
                                     "ESO DRS EXTCOEF", extcoef_comment);

        /* and to the catalogue header, if present */

        if (jitters->limages[j]->objects) {
            cpl_propertylist_update_double(jitters->limages[j]->objects->qclist,
                                           "ESO QC MAGZPT", magzpt);
            cpl_propertylist_set_comment(jitters->limages[j]->objects->qclist,
                                         "ESO QC MAGZPT", magzpt_comment);
            cpl_propertylist_update_double(jitters->limages[j]->objects->qclist,
                                           "ESO QC MAGZERR", magzerr);
            cpl_propertylist_set_comment(jitters->limages[j]->objects->qclist,
                                         "ESO QC MAGZERR", magzerr_comment);
            cpl_propertylist_update_double(jitters->limages[j]->objects->qclist,
                                           "PHOTZP", photzp);
            cpl_propertylist_set_comment(jitters->limages[j]->objects->qclist,
                                         "PHOTZP", photzp_comment);
            cpl_propertylist_update_double(jitters->limages[j]->objects->qclist,
                                           "ESO DRS EXTCOEF", extcoef);
            cpl_propertylist_set_comment(jitters->limages[j]->objects->qclist,
                                         "ESO DRS EXTCOEF", extcoef_comment);
        }

        /* Set ZPMETHOD to reflect the calibration method used */

        const char * zp_method = "separate_STD_star";
        const char * zp_method_comment = 
                     "photometric cal method: "
                     "separate_STD_star = transfer from other image, "
                     "science_field_standards = using stds in field, "
                     "default = default value from phot data file";
        cpl_propertylist_update_string(jitters->limages[j]->plist,
                                       "ZPMETHOD", zp_method);
        cpl_propertylist_set_comment(jitters->limages[j]->plist,
                                     "ZPMETHOD", zp_method_comment);
        if (jitters->limages[j]->objects) {
            cpl_propertylist_update_string(jitters->limages[j]->objects->qclist,
                                           "ZPMETHOD", zp_method);
            cpl_propertylist_set_comment(jitters->limages[j]->objects->qclist,
                                         "ZPMETHOD", zp_method_comment);
        }
 
        /* save the result */

        eris_nix_img_cal_phot_save(jitters->limages[j],
                                   out_catg,
                                   frameset,
                                   parlist,
                                   recipe_name);
        enu_check_error_code("failure writing result");
    }

cleanup:

    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Calibrate the images using standards in the field.
  @param    frameset    the frames list
  @param    parlist     the parameters list
  @param    jitters     the list of images to be calibrated
  @param    recipe_name the name of the recipe calling this routine
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_cal_phot_internal(
                                            cpl_frameset * frameset,
                                            const cpl_parameterlist * parlist,
                                            located_imagelist * jitters,
                                            const char * out_catg,
                                            const char * recipe_name,
                                            const char* context) {

    int                     casu_status = CASU_OK;
    hdrl_parameter        * cat_params = NULL;
    casu_fits            ** in_image_data = NULL;
    casu_fits            ** in_image_conf = NULL;
    casu_fits            ** in_image_var = NULL;
    casu_tfits           ** in_cats = NULL;
    casu_tfits           ** in_mstds = NULL;
    const cpl_parameter   * p = NULL;
    cpl_table             * phot_data = NULL;
    cpl_table            ** phot_refcat = NULL;
    char                  * param_name = NULL;

    cpl_msg_info(cpl_func, "Calibrating using standard stars in field");

    /* Retrieve input parameters */

    param_name = cpl_sprintf("%s.cdssearch_photom", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const char * cdssearch_photom = cpl_parameter_get_string(p);
    cpl_msg_info(cpl_func,"cdssearch_photom: %s",cdssearch_photom);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.pixel_radius", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const double pixel_radius = cpl_parameter_get_double(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.minphotom", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const int minphotom = cpl_parameter_get_int(p);
    cpl_free(param_name);

    param_name = cpl_sprintf("%s.magerrcut", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const double magerrcut = cpl_parameter_get_double(p);
    cpl_free(param_name);

    /* some elements of catalogue parameters, gain and saturation,
       can be overridden by contents of data files. This is done in a
       roundabout way because you can't simply access fields in the
       catalogue parameter structure from outside HDRL */
   
    param_name = cpl_sprintf("%s.catalogue.det.effective-gain", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double gain = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    if (cpl_propertylist_has(jitters->limages[0]->plist, "ESO DARK GAIN")) {
        cpl_msg_info(cpl_func, "reading detector gain from first jitter");
        gain = cpl_propertylist_get_double(jitters->limages[0]->plist,
                                           "ESO DARK GAIN");
    }
   
    param_name = cpl_sprintf("%s.catalogue.det.saturation", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double saturation = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    /* saturation in adu/s is DETMON_saturation/dit */
    if (cpl_propertylist_has(jitters->limages[0]->plist,
                             "ESO DETMON SATURATION")) {
        cpl_msg_info(cpl_func,
                     "reading detector saturation level from first jitter");
        double saturation_adu = cpl_propertylist_get_double(
                                       jitters->limages[0]->plist,
                                       "ESO DETMON SATURATION");
        double dit = enu_get_dit(jitters->limages[0]->plist);
        saturation = saturation_adu / dit;
        cpl_msg_info(cpl_func, 
                     "saturation(adu)=%5.3e dit=%5.3e saturation(adu/s)=%5.3e",
                     saturation_adu, dit, saturation);
  
    }
    
    param_name = cpl_sprintf("%s.catalogue.obj.min-pixels", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    int minpixels = cpl_parameter_get_int(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.catalogue.obj.threshold", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double threshold = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.catalogue.obj.deblending", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_boolean deblending = cpl_parameter_get_bool(p);
    cpl_free(param_name);

    /* use AOMODE to decide if set catalogue.core-radius and mesh-size to
       defaults or read them from parameters */

    double core_radius = -1.0;
    int mesh_size = -1;
    enu_get_rcore_and_mesh_size(context,
                                parlist,
                                jitters->limages[0]->plist,
                                &core_radius,
                                &mesh_size);
        
    param_name = cpl_sprintf("%s.catalogue.bkg.estimate", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_boolean estimate = cpl_parameter_get_bool(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.catalogue.bkg.smooth-gauss-fwhm",
                             context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double smooth_gauss_fwhm = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    cat_params = hdrl_catalogue_parameter_create(minpixels,
                                                 threshold,
                                                 deblending,
                                                 core_radius,
                                                 estimate,
                                                 mesh_size,
                                                 smooth_gauss_fwhm,
                                                 gain,
                                                 saturation,
                                                 HDRL_CATALOGUE_ALL);

    cpl_msg_info(cpl_func, "gain=%4.2f saturation=%6.2f", gain, saturation);

    enu_check_error_code("Could not retrieve input parameters");
    enu_check(strcmp(cdssearch_photom, "none"), CPL_ERROR_ILLEGAL_INPUT,
              "photometric catalogue not specified");

    /* Read in the data file with default photometric values */

    cpl_frame * phot_data_frame = cpl_frameset_find (frameset,
                                  ERIS_NIX_PHOT_DATA_PRO_CATG);
    const char * phot_data_file = cpl_frame_get_filename (phot_data_frame);
    enu_check_error_code("Failed to read NIX photometry file name");

    cpl_msg_info(cpl_func, "..reading instrument photometric properties "
                 "from %s", phot_data_file);
    cpl_size phot_ext = cpl_fits_find_extension(phot_data_file,
                                                cdssearch_photom);
    enu_check(phot_ext > 0, CPL_ERROR_INCOMPATIBLE_INPUT, 
              "photometry information for '%s' not found in %s",
              cdssearch_photom, phot_data_file);
    phot_data = cpl_table_load(phot_data_file, phot_ext, 0);

    /* cast any double columns to float to suit casu */

    cal_phot_cast_columns_to_float(phot_data);

    /* get the filter used for the observation and the relevant line of
       the phot_data table */

    const char * filter = enu_get_filter(jitters->limages[0]->plist);
    enu_check_error_code("Failed to read filter name from file");

    /* comparison is treated as a regular expression */
    char * comparison = cpl_sprintf("^%s$", filter);
    cpl_size nsel = cpl_table_and_selected_string(phot_data, "filter_name",
                                                  CPL_EQUAL_TO, comparison);
    cpl_free(comparison);
    if (nsel == 1) {
        cpl_msg_info(cpl_func, "Filter %s located in %s", filter,
                     phot_data_file);
        cpl_table * selected = cpl_table_extract_selected(phot_data);
        cpl_table_dump(selected, 0, 10, NULL);
        cpl_table_delete(selected);
        cpl_table_select_all(phot_data);
    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                              "No entry for filter %s in %s", filter,
                              phot_data_file);
    }
    enu_check_error_code("Failure to read photometry data for filter %s",
                          filter);

    /* generate object catalogues for each image */

    enu_catalogue_limlist(jitters, cat_params);

    /* copy a few things to FITS places that casu_photcal_extinct expects */

    for (cpl_size j = 0; j < jitters->size; j++) {
        /* tell the CASU routine via CEXPTIME that the exposure time to
           use in magnitude calculations is 1 sec. This is because the
           linearization process outputs results in DM/sec */
        cpl_propertylist_update_double(jitters->limages[j]->plist, 
                                       "CEXPTIME", 1.0);

        if (jitters->limages[j]->objects) {
            cpl_propertylist * phu = jitters->limages[j]->plist;
            cpl_propertylist * ehu = jitters->limages[j]->objects->qclist;

            /* exptime for photcal_extinct is 1 sec, as the images are in
               DN/sec. Store this in temporary keyword that photcal_extinct
               can access */

            /* pixel scale, arcsec/pix */
            double cd1_1 = cpl_propertylist_get_double(phu, "CD1_1");
            double cd2_1 = cpl_propertylist_get_double(phu, "CD2_1");
            double pixsize = sqrt(cd1_1 * cd1_1 + cd2_1 * cd2_1) * 3600.0;
            cpl_propertylist_update_float(phu, "ESO QC WCS_SCALE", 
                                          (float) pixsize);

            /* sky noise */
            double val = cpl_propertylist_get_double(ehu, "ESO QC SKY_NOISE");
            cpl_propertylist_update_float(ehu, "ESO DRS SKYNOISE", (float) val);

            /* mean sky */
            val = cpl_propertylist_get_double(ehu, "ESO QC MEAN_SKY");
            /* set skylevel to a very small number. Should be 0 as sky has been
               subtracted but its log is taken in casu_photcal_extinct. */
            cpl_propertylist_update_double(ehu, "ESO DRS SKYLEVEL",
                                           max(val, 0.01));

            /* ESO QC IMAGE_SIZE claims to be the average fwhm
               of stellar objects in pixels */

            double fwhm_pix = cpl_propertylist_get_double(
                                       ehu, "ESO QC IMAGE_SIZE");

            /* casu_photcal_extinct expects PSF_FWHM in arcsec, if known */
            if (fwhm_pix > 0) {
                double fwhm = fwhm_pix * pixsize;
                cpl_propertylist_update_double(ehu, "PSF_FWHM", (float)fwhm);
                cpl_propertylist_set_comment(ehu, "PSF_FWHM", "[arcsec] Average "
                                             "FWHM of stellar objects");
            }

            /* also cast any double table columns to float to suit casu */
            if (jitters->limages[j]->objects->catalogue) {
                cal_phot_cast_columns_to_float(jitters->limages[j]->objects->
                                               catalogue);
            }
        }
    }

    /* Convert jitter data to casu_tfits */

    encu_limlist_to_casu_fits(jitters, &in_image_data, &in_image_conf,
                              &in_image_var);

    /* Initialise in_cats and in_mstds to values that will be safely
       deleted if something goes wrong */

    in_cats = cpl_malloc(jitters->size * sizeof(casu_tfits *));
    in_mstds = cpl_malloc(jitters->size * sizeof(casu_tfits *));
    phot_refcat = cpl_malloc(jitters->size * sizeof(cpl_table *));

    for (cpl_size j = 0; j < jitters->size; j++) {
        in_mstds[j] = cpl_malloc(sizeof(casu_tfits));
        in_mstds[j]->table = NULL; 
        in_mstds[j]->phu = NULL; 
        in_mstds[j]->ehu = NULL; 
        in_mstds[j]->fname = NULL; 
        in_mstds[j]->extname = NULL; 
        in_mstds[j]->fullname = NULL; 
        in_mstds[j]->nexten = -1;
        in_mstds[j]->status = CASU_OK; 

        in_cats[j] = cpl_malloc(sizeof(casu_tfits));
        in_cats[j]->table = NULL; 
        in_cats[j]->phu = NULL; 
        in_cats[j]->ehu = NULL; 
        in_cats[j]->fname = NULL; 
        in_cats[j]->extname = NULL; 
        in_cats[j]->fullname = NULL; 
        in_cats[j]->nexten = -1;
        in_cats[j]->status = CASU_OK;

        phot_refcat[j] = NULL; 
    }

    /* Now populate in_mstds and in_cats with values to be used by
       casu_photcal_extinct */

    for (cpl_size j = 0; j < jitters->size; j++) {
        cpl_msg_info(cpl_func, "processing jitter %d", (int)j);

        /* be wary of empty (NULL) catalogues */

        if (jitters->limages[j]->objects) {
            in_cats[j]->table = cpl_table_duplicate(jitters->limages[j]->
                                                    objects->catalogue);
            in_cats[j]->phu = cpl_propertylist_duplicate(jitters->limages[j]->
                                                         plist);
            in_cats[j]->ehu = cpl_propertylist_duplicate(jitters->limages[j]->
                                                         objects->qclist);
            in_cats[j]->fname = cpl_strdup(cpl_frame_get_filename(
                                       jitters->limages[j]->frame));

            /* use casu_getstds to read in photom reference catalogue entries 
               for the area covered by the jitter, the routine also adds
               some columns used by casu_photcal_extinct*/

            /* check not in error state before entry. If error set then
               casu_getstds will fail, ptentially with a confusing error 
               message */

            enu_check_error_code("error before calling casu_getstds");
            phot_refcat[j] = NULL;
            casu_status = CASU_OK;
            casu_getstds(jitters->limages[j]->plist, CPL_FALSE, 
                         NULL,
                         (char*) "2MASS",
                         1, NULL, &phot_refcat[j], NULL,
                         &casu_status);
            //cpl_msg_info(cpl_func, "casu %d", (int)casu_status);
            if (phot_refcat[j] == NULL) {
                cpl_msg_info(cpl_func, "...no phot standards found");
            } else {
                en_catalogue_name_conformance(phot_refcat[j]);
                cpl_size nstds = cpl_table_get_nrow(phot_refcat[j]);
                cpl_msg_info(cpl_func, "...%d phot standards read", (int)nstds);

                /* populate in_mstds by associating image catalogue objects with 
                   those in the reference catalog, again beware NULL catalogues
                   (from detecting no sources in the image) */

                const int strict = 0;
                enm_associate_std(jitters->limages[j]->objects->catalogue,
                                  phot_refcat[j], pixel_radius, strict, 
                                  &(in_mstds[j]->table));
                enu_check_error_code("Failure trying to match standards: %s",
                                     cpl_error_get_message());    

                /* cast any double columns to float to suit casu */
  
                cal_phot_cast_columns_to_float(in_mstds[j]->table);
                in_mstds[j]->phu = cpl_propertylist_new();
                in_mstds[j]->ehu = cpl_propertylist_new();
                enu_check(casu_status == CASU_OK, CPL_ERROR_ILLEGAL_INPUT,
                      "casu_matchstds failure status : %d", casu_status);
            }

        } else {

           /* No objects found hence no object catalogue. Add some dummy
              propertylists to keep casu_photcal_extinct happy - it will
              write to them */

            cpl_msg_info(cpl_func, ".no objects detected in field");
            in_cats[j]->phu = cpl_propertylist_new();
            in_cats[j]->ehu = cpl_propertylist_new();
        }

        /* If for some reason there are no matched standards then point
           to an empty table rather than NULL. casu_photcal_extinct can
           handle the former but not the latter

           Also, add some empty propertylists for casu_photcal_extinct 
           to write results into, otherwise it will fail */

        if (in_mstds[j]->table == NULL) {
            in_mstds[j]->table = cpl_table_new(0);
            in_mstds[j]->phu = cpl_propertylist_new();
            in_mstds[j]->ehu = cpl_propertylist_new();
        }
    }

    /* Now do the photometric calibration */

    cpl_msg_info(cpl_func, "performing photometric calibration");
    casu_status = CASU_OK;
    char * filter_copy = cpl_strdup(filter);
    casu_photcal_extinct(in_image_data, 
                         in_mstds,
                         in_cats,
                         jitters->size,
                         filter_copy,
                         phot_data,
                         minphotom,
                         NULL, NULL, 
                         "CEXPTIME",
                         "ESO TEL AIRM START",
                         magerrcut,
                         &casu_status);
    cpl_free(filter_copy);
    enu_check_error_code("Failure in casu_photcal_extinct: %s",
                         cpl_error_get_message());    

    /* assess the calibration and save the photometric calibrated jitter
       images to FITS files */

    for (cpl_size j = 0; j < jitters->size; j++) {

        const char * fluxcal = cpl_propertylist_get_string(
                                       in_image_data[j]->phu,
                                       "FLUXCAL");

        if (!strcmp(fluxcal, "UNCALIBRATED")) {

            /* did the casu flux calibration succeed? An outright failure
               will have casu_photcal_extinct set FLUXCAL to
               'UNCALIBRATED'. In this case set the default result */

            cpl_msg_warning(cpl_func, "casu_photcal_extinct has returned "
                            "a bad result, applying default");
            eris_nix_img_cal_phot_default(frameset, jitters->limages[j]);

        } else {

            /* otherwise set the result returned by casu_photcal_extinct */

            /* ..for in_image_data the phu and ehu are both copies of
               the original limage->plist. ehu is the one updated by 
               casu_photcal_extinct - now copy it back to get the updates. 
               *except for FLUXCAL which is set in phu*/

            cpl_propertylist_delete(jitters->limages[j]->plist);
            jitters->limages[j]->plist = cpl_propertylist_duplicate(
                                         in_image_data[j]->ehu);
            cpl_propertylist_set_string(jitters->limages[j]->plist, "FLUXCAL",
                                        fluxcal);
 
            /* add the photometric system */

            cpl_propertylist_update_string(jitters->limages[j]->plist,
                                           "PHOTSYS", "VEGA");

            /* Set ZPMETHOD to reflect the calibration method used
               If casu_photcal_extinct has left keyword ZPFUDGED==True
               then the calibration failed and the routine itself
               set zp/zp error to default values from file */

            const int zpfudged = cpl_propertylist_get_bool(
                                 jitters->limages[j]->plist, "ZPFUDGED");

            const char * zp_method = NULL;
            const char * zp_comment = NULL;
            if (zpfudged) {
                zp_method = "DEFAULT";
                zp_comment = "ZP taken from filter default value";
            } else {
                zp_method = "2MASS";
                zp_comment = "ZP computed from 2MASS sources in the field";
            }
            cpl_propertylist_update_string(jitters->limages[j]->plist,
                                           "ZPMETHOD", zp_method);
            cpl_propertylist_set_comment(jitters->limages[j]->plist,
                                         "ZPMETHOD", zp_comment);
            cpl_msg_info(cpl_func, "set ZPMETHOD %s", zp_method);
      
            if (!strcmp(zp_method, "2MASS")) {
                cpl_propertylist_update_string(jitters->limages[j]->plist,
                                               "ZP_CAT", cdssearch_photom);
            }

            /* copy updated plist to image catalogue */

            if (jitters->limages[j]->objects &&
                jitters->limages[j]->objects->qclist) {
                cpl_propertylist_delete(jitters->limages[j]->objects->qclist);
                jitters->limages[j]->objects->qclist =
                                        cpl_propertylist_duplicate(
                                        in_cats[j]->ehu);
                cpl_propertylist_update_string(
                                        jitters->limages[j]->objects->qclist,
                                        "ZPMETHOD", zp_method);
                cpl_propertylist_update_string(
                                        jitters->limages[j]->objects->qclist,
                                        "PHOTSYS", "VEGA");
            }

            /* if they exist and contain any rows, copy matched stds catalogues 
               and associated qclist to limage */

            if (in_mstds[j] && 
                in_mstds[j]->table && 
                cpl_table_get_nrow(in_mstds[j]->table) > 0) {
                jitters->limages[j]->matchstd_phot = cpl_calloc(
                                       sizeof(hdrl_catalogue_result), 1);
                jitters->limages[j]->matchstd_phot->catalogue = 
                                       cpl_table_duplicate(in_mstds[j]->table);
                jitters->limages[j]->matchstd_phot->qclist = 
                                       cpl_propertylist_duplicate(
                                       in_mstds[j]->ehu);
                cpl_propertylist_update_string(jitters->limages[j]->
                                       matchstd_phot->qclist,
                                       "ZPMETHOD",
                                       zp_method);
                if (!strcmp(zp_method, "science_field_standards")) {
                    cpl_propertylist_update_string(jitters->limages[j]->
                                       matchstd_phot->qclist, "ZP_CAT",
                                       cdssearch_photom);
                }
            }
        }

        /* Save the result */

        eris_nix_img_cal_phot_save(jitters->limages[j],
                                   out_catg,
                                   frameset,
                                   parlist,
                                   recipe_name);
        enu_check_error_code("failure writing result");

        /* save the reference catalogue */

        if (phot_refcat[j]) {
            enu_dfs_save_catalogue(frameset,
                                   jitters->limages[j]->frame,
                                   parlist,
                                   phot_refcat[j],
                                   NULL,
                                   cdssearch_photom,
                                   recipe_name,
                                   PACKAGE "/" PACKAGE_VERSION,
                                   "refcat.cal_phot",
                                   ERIS_NIX_CAL_PHOT_REF_CATALOGUE_PRO_CATG);
        }
    
        /* save the matched standards catalogue */

        if (jitters->limages[j]->matchstd_phot && 
            jitters->limages[j]->matchstd_phot->catalogue) {

            enu_dfs_save_catalogue(frameset,
                                   jitters->limages[j]->frame,
                                   parlist,
                                   jitters->limages[j]->matchstd_phot->catalogue,
                                   jitters->limages[j]->matchstd_phot->qclist,
                                   "2MASS",
                                   recipe_name,
                                   PACKAGE "/" PACKAGE_VERSION,
                                   "matchcat.cal_phot",
                                   ERIS_NIX_CAL_PHOT_MATCH_CATALOGUE_PRO_CATG);
        }

        /* save the catalogue of sources in the image itself */

        if (jitters->limages[j]->objects &&
            jitters->limages[j]->objects->catalogue) {

            enu_dfs_save_catalogue(frameset,
                                   jitters->limages[j]->frame,
                                   parlist,
                                   jitters->limages[j]->objects->catalogue,
                                   jitters->limages[j]->objects->qclist,
                                   cpl_frame_get_filename(jitters->
                                                          limages[j]->
                                                          frame),
                                   recipe_name,
                                   PACKAGE "/" PACKAGE_VERSION,
                                   "cat.cal_phot",
                                   ERIS_NIX_CAL_PHOT_CATALOGUE_PRO_CATG);
        }
    }

cleanup:
    hdrl_parameter_delete(cat_params);
    if (jitters) {
        if (in_image_data) {
            for (cpl_size j = 0; j < jitters->size; j++) {
                if (in_image_data[j]) casu_fits_unwrap(in_image_data[j]);
            }
            cpl_free(in_image_data);
        }
        if (in_image_var) {
            for (cpl_size j = 0; j < jitters->size; j++) {
                if (in_image_var[j]) casu_fits_unwrap(in_image_var[j]);
            }
            cpl_free(in_image_var);
        }
        if (in_image_conf) {
            for (cpl_size j = 0; j < jitters->size; j++) {
                if (in_image_conf[j]) casu_fits_unwrap(in_image_conf[j]);
            }
            cpl_free(in_image_conf);
        }
        casu_tfits_delete_list(in_cats, (int)jitters->size);
        casu_tfits_delete_list(in_mstds, jitters->size);
        if (phot_refcat != NULL) {
            for (cpl_size j = 0; j < jitters->size; j++) {
                cpl_table_delete(phot_refcat[j]);
            }
        }
    }
    cpl_table_delete(phot_data);
    cpl_free(phot_refcat);
    eris_check_error_code("eris_nix_img_cal_phot_internal");
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief   This recipe calibrates the photometry of ERIS/NIX frames. 

  @param    frameset   the frames list
  @param    parlist    the parameters list
  @param    recipe_name name of recipe
  @param    context  recipe context (eris)

            This recipe calibrates the photometry of ERIS/NIX frames.
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_nix_scired_cal_phot(cpl_frameset * frameset,
                                        const cpl_parameterlist * parlist,
                                        const char * recipe_name,
                                        const char* context) {

    located_imagelist     * jitters = NULL; 
    located_imagelist     * object_jitters = NULL; 
    located_imagelist     * standards = NULL;
    located_imagelist     * std_jitters = NULL;
    cpl_frameset          * used = NULL;

    enu_check_error_code("%s():%d: An error is already set: %s",
                         cpl_func, __LINE__, cpl_error_get_where());

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    /* Set the msg verbosity level from environment variable CPL_MSG_LEVEL
       - not working, so set manually for now */

    cpl_msg_set_level_from_env();
    cpl_msg_severity severity = cpl_msg_get_level();
    cpl_msg_info(cpl_func, "level %d", (int) severity);

    /* check for invalid input */
    if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
    	return CPL_ERROR_BAD_FILE_FORMAT;
    }
    /* check required input tags are present */
    const int ntags = 1;
    const char* required_tags[1] = {
    		ERIS_NIX_PHOT_DATA_PRO_CATG
    };

    // ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG,  or ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG
    cpl_ensure_code(CPL_ERROR_NONE ==
    		eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
			CPL_ERROR_ILLEGAL_INPUT);

    const int ntags_opt = 3;
    const char* optional_tags[3] = {
    		ERIS_NIX_CAL_WCS_MATCH_CATALOGUE_PRO_CATG, // opt
                ERIS_NIX_CAL_WCS_REF_CATALOGUE_PRO_CATG, // opt
                ERIS_NIX_CAL_WCS_CATALOGUE_PRO_CATG     //opt
    };

    // ERIS_NIX_SKYSUB_OBJECT_JITTER_PRO_CATG,  or ERIS_NIX_SKYSUB_STD_JITTER_PRO_CATG
    cpl_ensure_code(CPL_ERROR_NONE ==
    		eris_dfs_check_input_tags(frameset, optional_tags, ntags_opt, 0),
                                          CPL_ERROR_ILLEGAL_INPUT);

    /* Some initialization */

    used = cpl_frameset_new();

    /* Identify the RAW and CALIB frames in the input frameset */

    eris_nix_dfs_set_groups(frameset);
    enu_check_error_code("Could not identify RAW and CALIB frames");    

    /* Read in images to be calibrated */

    object_jitters = enu_limlist_load_from_frameset(frameset,
                     ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG, used);
    enu_check_error_code("Failed to read object jitter data");    
    std_jitters = enu_limlist_load_from_frameset(frameset,
                  ERIS_NIX_CAL_WCS_STD_JITTER_PRO_CATG, used);
    enu_check_error_code("Failed to read STD jitter data");    

    cpl_msg_info(cpl_func, "%d %s frames read", (int) object_jitters->size,
                 ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG);
    cpl_msg_info(cpl_func, "%d %s frames read", (int) std_jitters->size,
                 ERIS_NIX_CAL_WCS_STD_JITTER_PRO_CATG);

    /* check that the data files in the SoF are as expected */

    enu_check(object_jitters->size > 0 || std_jitters->size > 0,
              CPL_ERROR_DATA_NOT_FOUND, "no input frames found");
    enu_check(!(object_jitters->size > 0 && std_jitters->size > 0),
              CPL_ERROR_ILLEGAL_INPUT, "SoF contains both "
              ERIS_NIX_CAL_WCS_OBJECT_JITTER_PRO_CATG" and "
              ERIS_NIX_CAL_WCS_STD_JITTER_PRO_CATG" frames");

    /* set the catg of the output data */

    const char * out_catg = NULL;
    if (object_jitters->size > 0) {
        jitters = object_jitters;
        out_catg = ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG;
        enu_located_imagelist_delete(std_jitters);
        std_jitters = NULL;
    } else {
        jitters = std_jitters;
        out_catg = ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG;
        enu_located_imagelist_delete(object_jitters);
        object_jitters = NULL;
    }

    /* The calibration method depends on whether a photometrically calibrated
       frame is present in the SoF. If so, then transfer its calibration. 
       If not, try to calibrate the frames using their own measured 
       standards. */  

    standards = enu_limlist_load_from_frameset(frameset,
                ERIS_NIX_IMG_STD_COMBINED_PRO_CATG, used);
    enu_check_error_code("Failed to read external standard");    
    if (standards->size > 0) {
        cpl_msg_info(cpl_func, "%d "
                     ERIS_NIX_IMG_STD_COMBINED_PRO_CATG" frames read",
                     (int) standards->size);
    }

    if (standards->size == 0) {

        /* Not transferring calibration from another frame.
           Is the wcs astrometry calibration absolute? */

        int absolute_wcs = CPL_FALSE;

        for (cpl_size j = 0; j < jitters->size; j++) {

        	const char * wcs_method;
        	if (cpl_propertylist_has(jitters->limages[j]->plist,"ESO WCS_METHOD")) {
                wcs_method = cpl_propertylist_get_string(
                                      jitters->limages[j]->plist,
                                      "ESO WCS_METHOD");
           } else {
        	   /* old version may have stored the WCS method without prefix. Thus allow also this option */
        	   wcs_method = cpl_propertylist_get_string(
        	                                         jitters->limages[j]->plist,
        	                                         "WCS_METHOD");
           }

            cpl_msg_info(cpl_func,"jitter %d wcs_method: %s", (int)j, wcs_method);
            if (!strcmp(wcs_method, ERIS_NIX_WCS_CATALOGUE_MATCH)) {
                absolute_wcs = CPL_TRUE;
            }
        }
        enu_check_error_code("failed to determine type of wcs calibration");

        if (absolute_wcs) {

            /* Calibrate using standards measured in field */

            eris_nix_img_cal_phot_internal(frameset,
                                           parlist,
                                           jitters,
                                           out_catg,
                                           recipe_name, context);
            enu_check_error_code("error attempting internal photometric "
                                 "calibration");
        } else {

            for (cpl_size j = 0; j < jitters->size; j++) {

                /* Set zeropoint to default value for filter */
  
                cpl_msg_info(cpl_func, "..astrometry calibration is not absolute");
                eris_nix_img_cal_phot_default(frameset,
                                              jitters->limages[j]);
                enu_check_error_code("error setting default photometric "
                                     "calibration");

                /* and write the result */
   
                eris_nix_img_cal_phot_save(jitters->limages[j],
                                           out_catg,
                                           frameset,
                                           parlist,
                                           recipe_name);
                enu_check_error_code("failure writing result");
            }
        }

    } else if (standards->size == 1){

        /* Transfer calibration from the external image */

        eris_nix_img_cal_phot_external(frameset,
                                       parlist,
                                       standards,
                                       jitters,
                                       out_catg,
                                       recipe_name);
        enu_check_error_code("error attempting external photometric "
                             "calibration");
    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                              "only one "
                              ERIS_NIX_IMG_STD_COMBINED_PRO_CATG
                              " file is allowed, found %d", 
                              (int) standards->size);
    }

cleanup:
    enu_located_imagelist_delete(object_jitters);
    enu_located_imagelist_delete(std_jitters);
    enu_located_imagelist_delete(standards);
    cpl_frameset_delete(used);

    return (int) cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief   This rebin and stack a set of calibrated ERIS/NIX jitter frames. 

  @param    frameset   the frames list
  @param    parlist    the parameters list
  @param    recipe_name the name of the recipe

  The stacking uses hdrl_resample_compute to combine all jitter images
  in a single cloud of points. When creating the resampled output image all 
  information is used in a single interpolation step, i.e. no image-by-image 
  interpolation is done.
 
  @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_nix_scired_hdrl_stack(cpl_frameset * frameset,
                                          const cpl_parameterlist * parlist,
                                          const char* recipe_name,
                                          const char* context) {

    cpl_propertylist     * applist = NULL;
    cpl_property         * bunit = NULL;
    hdrl_parameter       * cat_params = NULL;
    cpl_property         * fluxcal_0 = NULL;
    const char           * funcid = "eris_nix_img_hdrl_stack";
    cpl_frame            * inherit = NULL;
    located_imagelist    * jitters = NULL; 
    cpl_property         * jitter_psf_fwhm = NULL;
    hdrl_parameter       * method = NULL;
    located_imagelist    * object_jitters = NULL; 
    cpl_vector           * obsid = NULL;
    hdrl_parameter       * outgrid = NULL;
    char                 * photsys_0 = NULL;
    char                 * proto_copy = NULL;
    cpl_frameset         * provenance = NULL;
    cpl_table            * restable = NULL;
    hdrl_resample_result * result = NULL;
    cpl_image            * stack_conf = NULL;
    cpl_propertylist     * stack_header = NULL;
    hdrl_image           * stack_himage = NULL;
    located_image        * stacked = NULL;
    located_imagelist    * std_jitters = NULL; 
    cpl_propertylist     * tablelist = NULL;
    cpl_wcs              * template_wcs = NULL;
    cpl_frameset         * used = NULL;
    cpl_property         * zp_method_0 = NULL;
    const cpl_parameter  * p = NULL;
    char                 * param_name = NULL;

    enu_check_error_code("%s():%d: An error is already set: %s",
      cpl_func, __LINE__, cpl_error_get_where());

    /* Check input parameters */

    cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);

    cpl_msg_set_level_from_env();

    /* check for invalid input */
    if(eris_files_dont_exist(frameset) != CPL_ERROR_NONE) {
    	return CPL_ERROR_BAD_FILE_FORMAT;
    }
    /* Retrieve input parameters */

    /* .. the resampling method and parameters */
    param_name = cpl_sprintf("%s.interpolation_method", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const char * interpolation_method = cpl_parameter_get_string(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.loop_distance", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const int loop_distance = cpl_parameter_get_int(p);    
    cpl_free(param_name);
    
    /* don't weight by image errors */
    const cpl_boolean use_errorweights = CPL_FALSE;
    
    param_name = cpl_sprintf("%s.kernel_size", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const int kernel_size = cpl_parameter_get_int(p);    
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.critical_radius", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const double critical_radius = cpl_parameter_get_double(p);    
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.pix_frac_x", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const double pix_frac_x = cpl_parameter_get_double(p);    
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.pix_frac_y", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    const double pix_frac_y = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    const double pix_frac_lambda = 100.0;
    
    enu_check_error_code("Could not retrieve input parameters");
    
   
    if (!strcmp(interpolation_method, "nearest")) {
        method = hdrl_resample_parameter_create_nearest();
    } else if (!strcmp(interpolation_method, "linear")) {
        method = hdrl_resample_parameter_create_linear(loop_distance,
                                                       use_errorweights);
    } else if (!strcmp(interpolation_method, "quadratic")) {
        method = hdrl_resample_parameter_create_quadratic(loop_distance,
                                                          use_errorweights);
    } else if (!strcmp(interpolation_method, "renka")) {
        method = hdrl_resample_parameter_create_renka(loop_distance,
                                                      use_errorweights,
                                                      critical_radius);
    } else if (!strcmp(interpolation_method, "drizzle")) {
        method = hdrl_resample_parameter_create_drizzle(loop_distance,
                                                        use_errorweights,
                                                        pix_frac_x,
                                                        pix_frac_y,
                                                        pix_frac_lambda);
    } else if (!strcmp(interpolation_method, "lanczos")) {
        method = hdrl_resample_parameter_create_lanczos(loop_distance,
                                                        use_errorweights, 
                                                        kernel_size);
    } else {
        cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                              "bad interpolation method: %s",
                              interpolation_method);
    }
    enu_check_error_code("Error constructing interpolation method");

    /* Identify the RAW and CALIB frames in the input frameset */

    eris_nix_dfs_set_groups(frameset);
    enu_check_error_code("Could not identify RAW and CALIB frames");    

    /* read the input frameset */

    used = cpl_frameset_new();
    object_jitters = enu_limlist_load_from_frameset(frameset,
                     ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG, used);
    enu_check_error_code("Could not read %s frames",
                         ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG);    
    cpl_msg_info(funcid, "%d %s frames read", (int) object_jitters->size,
                 ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG);
    std_jitters = enu_limlist_load_from_frameset(frameset,
                  ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG, used);
    enu_check_error_code("Could not read %s frames",
                         ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG);    
    cpl_msg_info(funcid, "%d %s frames read", (int) std_jitters->size,
                 ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG);

    /* is this a target object or standard? */

    enu_check(!(object_jitters->size > 0 && std_jitters->size > 0),
              CPL_ERROR_ILLEGAL_INPUT, 
              "SoF cannot contain data for both target and standard");
    enu_check(!(object_jitters->size == 0 && std_jitters->size == 0),
              CPL_ERROR_DATA_NOT_FOUND,
              "SoF contains no data");
    const char * out_catg = NULL;
    if(object_jitters->size > 0) {
        jitters = object_jitters;
        out_catg = ERIS_NIX_IMG_OBS_COMBINED_PRO_CATG;
        enu_located_imagelist_delete(std_jitters);
        std_jitters = NULL;
    } else {
        jitters = std_jitters;
        out_catg = ERIS_NIX_IMG_STD_COMBINED_PRO_CATG;
        enu_located_imagelist_delete(object_jitters);
        object_jitters = NULL;
    }

    /* get some info from jitters that we will need after they have been
       deleted */

    cpl_size njitters = jitters->size;
    const char * proto = cpl_frame_get_filename((jitters->limages[0])->
                                                frame);
    proto_copy = cpl_strdup(proto);

    inherit = cpl_frame_duplicate(jitters->limages[0]->frame);

    /* Loop through the jitter images and populate the resampling table */

    obsid = cpl_vector_new(jitters->size);
    cpl_vector_fill(obsid, -1.0);
    int ncombine = 0;
    double total_exptime = 0.0;
    double mjd_start = 1000000.0;
    double mjd_end = 0.0;
   
    /* take pixel scale and wcs from first jitter file */

    double cd1_1 = cpl_propertylist_get_double(jitters->limages[0]->plist,
                                               "CD1_1");
    double cd2_1 = cpl_propertylist_get_double(jitters->limages[0]->plist,
                                               "CD2_1");
    double pixsize = sqrt(cd1_1 * cd1_1 + cd2_1 * cd2_1) * 3600.0;
    cpl_msg_info(cpl_func, "pixel scale: %6.4f arcsec", pixsize);
    template_wcs = cpl_wcs_new_from_propertylist(jitters->limages[0]->plist);
    enu_check_error_code("error set after loading template wcs");

    /* some elements of catalogue parameters can be overridden by contents
       of data files. This is done in a roundabout way because you can't
       simply access fields in the catalogue parameter structure from
       outside HDRL */
   
    param_name = cpl_sprintf("%s.catalogue.det.effective-gain", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double gain = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    if (cpl_propertylist_has(jitters->limages[0]->plist, "ESO DARK GAIN")) {
        cpl_msg_info(cpl_func, "reading detector gain from first jitter");
        gain = cpl_propertylist_get_double(jitters->limages[0]->plist,
                                           "ESO DARK GAIN");
    }
   
    param_name = cpl_sprintf("%s.catalogue.det.saturation", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double saturation = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    /* saturation in adu/s is DETMON_saturation/dit */
    if (cpl_propertylist_has(jitters->limages[0]->plist,
                             "ESO DETMON SATURATION")) {
        cpl_msg_info(cpl_func,
                     "reading detector saturation level from first jitter");
        double saturation_adu = cpl_propertylist_get_double(
                                       jitters->limages[0]->plist,
                                       "ESO DETMON SATURATION");
        double dit = enu_get_dit(jitters->limages[0]->plist);
        saturation = saturation_adu / dit;
        cpl_msg_info(cpl_func, 
                     "saturation(adu)=%5.3e dit=%5.3e saturation(adu/s)=%5.3e",
                     saturation_adu, dit, saturation);
  
    }
    
    param_name = cpl_sprintf("%s.catalogue.obj.min-pixels", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    int minpixels = cpl_parameter_get_int(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.catalogue.obj.threshold", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double threshold = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.catalogue.obj.deblending", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_boolean deblending = cpl_parameter_get_bool(p);
    cpl_free(param_name);
    
    /* use AOMODE to decide if set catalogue.core-radius and mesh-size to
       defaults or read them from parameters */

    double core_radius = -1.0;
    int mesh_size = -1;
    enu_get_rcore_and_mesh_size(context,
                                parlist,
                                jitters->limages[0]->plist,
                                &core_radius,
                                &mesh_size);
    
    param_name = cpl_sprintf("%s.catalogue.bkg.estimate", context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    cpl_boolean estimate = cpl_parameter_get_bool(p);
    cpl_free(param_name);
    
    param_name = cpl_sprintf("%s.catalogue.bkg.smooth-gauss-fwhm",
                             context);
    p = cpl_parameterlist_find_const(parlist, param_name);
    double smooth_gauss_fwhm = cpl_parameter_get_double(p);
    cpl_free(param_name);
    
    cat_params = hdrl_catalogue_parameter_create(minpixels,
                                                 threshold,
                                                 deblending,
                                                 core_radius,
                                                 estimate,
                                                 mesh_size,
                                                 smooth_gauss_fwhm,
                                                 gain,
                                                 saturation,
                                                 HDRL_CATALOGUE_ALL);
    enu_check_error_code("error set after setting catalogue params");

    cpl_msg_info(cpl_func, "gain=%4.2f saturation=%6.2f", gain, saturation);

    provenance = cpl_frameset_new();

    cpl_msg_info(cpl_func, "assembling information on combined exposures");

    for (cpl_size j = 0; j < jitters->size; j++) {

        /* add the frame to the provenance of images contributing
           to the result */

        cpl_frameset_insert(provenance, cpl_frame_duplicate(
                            jitters->limages[j]->frame));

        /* calculate some other things needed by the DFS to describe
           what has gone into the measurement */

        if (!bunit) {
            bunit = cpl_property_duplicate(cpl_propertylist_get_property(
                                           jitters->limages[j]->plist,
                                           "BUNIT"));
        }

        if (!jitter_psf_fwhm) {
            jitter_psf_fwhm = cpl_property_duplicate(
                                       cpl_propertylist_get_property(
                                       jitters->limages[j]->plist,
                                       "PSF_FWHM"));
        }

        /* .. number of measurements combined */

        ncombine++;

        /* .. total integration time */

        double dit = enu_get_dit(jitters->limages[j]->plist);
        double exptime = dit * cpl_propertylist_get_int(
                         jitters->limages[j]->plist, "ESO DET NDIT");
        total_exptime += exptime;

        /* .. MJD of start and end of measurements used */

        double jitter_start = cpl_propertylist_get_double(
                              jitters->limages[j]->plist, "MJD-OBS");
        if (jitter_start < mjd_start) {
            mjd_start = jitter_start;
        }
        double jitter_end = cpl_propertylist_get_double(
                              jitters->limages[j]->plist, "MJD-END");
        if (jitter_end > mjd_end) {
            mjd_end = jitter_end;
        }
        cpl_msg_info(cpl_func, "..combined mjd start: %15.8f  end: %15.8f",
                     mjd_start, mjd_end);

        /* .. the OBS_IDs of the component measurements, no duplicates */

        int obs_id = cpl_propertylist_get_int(jitters->limages[j]->plist,
                                              "ESO OBS ID");
        for (cpl_size jj = 0; jj < cpl_vector_get_size(obsid); jj++) {
            if ((int) cpl_vector_get(obsid, jj) == obs_id) {
                break;
            } else if (cpl_vector_get(obsid, jj) < 0.0) {
                cpl_vector_set(obsid, jj, obs_id);
                break;                 
            }
        }
    }

    /* set the output grid */

    outgrid = hdrl_resample_parameter_create_outgrid2D(pixsize / 3600.0,
                                                       pixsize / 3600.0);

    double photzp_0 = 0.0;
    double photzper_0 = 0.0;
    double extcoef_0 = 0.0;

    cpl_msg_info(cpl_func, "stacking image data");
    {

        for (cpl_size j = 0; j < jitters->size; j++) {
            cpl_msg_info(cpl_func, "ingesting jitter %d", (int)j);

            /* get calibration info */

            const cpl_property * fluxcal_j = cpl_propertylist_get_property(
                                       jitters->limages[j]->plist, "FLUXCAL");
            const char * photsys_j = cpl_propertylist_get_string(
                                       jitters->limages[j]->plist, "PHOTSYS");
            const cpl_property * zp_method_j = cpl_propertylist_get_property(
                                       jitters->limages[j]->plist,
                                       "ZPMETHOD");
            double photzp_j = cpl_propertylist_get_double(
                                       jitters->limages[j]->plist, "PHOTZP");
            double photzper_j = cpl_propertylist_get_double(
                                       jitters->limages[j]->plist, "PHOTZPER");
            double extcoef_j = cpl_propertylist_get_double(
                                       jitters->limages[j]->plist,
                                       "ESO DRS EXTCOEF");

            cpl_msg_info(cpl_func, "..fluxcal: %s", 
                         cpl_property_get_string(fluxcal_j));
            cpl_msg_info(cpl_func, "..photsys: %s", photsys_j);
            cpl_msg_info(cpl_func, "..extcoef: %f", extcoef_j);
            cpl_msg_info(cpl_func, "..zp_method: %s",
                         cpl_property_get_string(zp_method_j));
            cpl_msg_info(cpl_func, "..photzp: %f photzper: %f", photzp_j,
                         photzper_j);

            if (j == 0) {
                photzp_0 = photzp_j;
                photzper_0 = photzper_j;
                fluxcal_0 = cpl_property_duplicate(fluxcal_j);
                photsys_0 = cpl_strdup(photsys_j);
                zp_method_0 = cpl_property_duplicate(zp_method_j);
                extcoef_0 = extcoef_j;
            } else {
 
                if (!strcmp(cpl_property_get_string(fluxcal_j), "CALIBRATED")) {
                    cpl_msg_warning(cpl_func, "jitter is not calbrated");
                }
                if (strcmp(photsys_j, "VEGA")) {
                    cpl_msg_warning(cpl_func, "PHOTSYS is not recognized");
                }
                if (extcoef_j != extcoef_0) {
                cpl_msg_warning(cpl_func,
                                "..inconsistent extcoef: %d %5.3e %5.3e",
                                (int) j, extcoef_0, extcoef_j);
                }

                /* other checks? */
            }

            /* factor to multiply jitter j to get on same intensity scale as
               jitter 0 */

            double factor = pow(10.0, (photzp_0 - photzp_j) / 2.5);
            cpl_msg_info(cpl_func, "..scaling jitter %d by %5.3e", (int)j,
                         factor);
            hdrl_image_mul_scalar(jitters->limages[j]->himage,
                                  (hdrl_value){factor, 0.0});

            /* The following commented-out code snippet would be useful if
               we wanted to weight according to the confidence array. In
               this case use_errorweights would be set TRUE in resampling
               'method' so that 'an additional weight defined as 1/variance'
               would be applied. To use confidence rather than pixel
               variance the 'resampling variance' should be 1/confidence,
               and the 'resampling error' the sqrt of that.

               Not being done now because I don't think it's currently
               possible to resample the image errors weighted by 
               the confidence. */
/*
            cpl_image * inverse_conf = cpl_image_duplicate(
                                       jitters->limages[j]->confidence);
            double * ic_data = cpl_image_get_data_double(inverse_conf);
            const cpl_size nx = cpl_image_get_size_x(inverse_conf);
            const cpl_size ny = cpl_image_get_size_y(inverse_conf);
            for (cpl_size i=0; i<nx*ny; i++) {
*/
                /* assume case of 0 confidence pixel is also 'bad' in image */
/*
                if (ic_data[i] > 0.0) {
                    ic_data[i] = sqrt(1.0 / ic_data[i]);
                }
            }
*/  

            /* add the jitter to the resampling table */

            cpl_wcs * wcs = cpl_wcs_new_from_propertylist(
                                       jitters->limages[j]->plist);
            cpl_table * jitter_table = hdrl_resample_image_to_table(
                                       jitters->limages[j]->himage,
                                       wcs);
            cpl_wcs_delete(wcs);

            if (restable == NULL) {
                restable = jitter_table;
            } else {
                cpl_size nrow = cpl_table_get_nrow(restable);
                cpl_table_insert(restable, jitter_table, nrow+1);
                cpl_table_delete(jitter_table);
            }
        }   

        enu_check_error_code("error set after loading jitters");

        /* delete the jitters to free up memory */

        enu_located_imagelist_delete(jitters); jitters = NULL;
        object_jitters = std_jitters = NULL;

        /* Stack the jitter image data */

        cpl_msg_info(cpl_func, "calling hdrl_resample_compute");
        result = hdrl_resample_compute(restable, method, outgrid,
                                       template_wcs); 

        /* release the resampling table as no longer needed */

        cpl_table_delete(restable); restable = NULL;
        enu_check_error_code("error set computing resample_compute");

        /* keep the resampled image and header for later but delete the
           rest of the result */

        stack_himage = hdrl_image_duplicate(
                                       hdrl_imagelist_get(
                                       result->himlist, 0));
        stack_header = cpl_propertylist_duplicate(result->header);
        hdrl_resample_result_delete(result); result = NULL;
    }

    cpl_msg_info(cpl_func, "stacking the confidence");

    {
        /* reload jitter data, with fewer checks this time*/

        cpl_frameset * ignore = cpl_frameset_new();
        object_jitters = enu_limlist_load_from_frameset(frameset,
                                       ERIS_NIX_CAL_PHOT_OBJECT_JITTER_PRO_CATG,
                                       ignore);
        std_jitters = enu_limlist_load_from_frameset(frameset,
                                       ERIS_NIX_CAL_PHOT_STD_JITTER_PRO_CATG,
                                       ignore);
        if(object_jitters->size > 0) {
            jitters = object_jitters;
            enu_located_imagelist_delete(std_jitters); std_jitters = NULL;
        } else {
            jitters = std_jitters;
            enu_located_imagelist_delete(object_jitters); object_jitters = NULL;
        }

        /* build the data table */

        for (cpl_size j = 0; j < jitters->size; j++) {
            cpl_msg_info(cpl_func, "ingesting jitter %d", (int)j);

            double photzp_j = cpl_propertylist_get_double(
                                       jitters->limages[j]->plist, "PHOTZP");

            double factor = pow(10.0, (photzp_0 - photzp_j) / 2.5);
            cpl_msg_info(cpl_func, "..scaling by %5.3e", factor);
            hdrl_image_mul_scalar(jitters->limages[j]->himage,
                                  (hdrl_value){factor, 0.0});

            /* To resample the confidence we replace the data error
               plane with numbers derived from the confidence. Confidence
               is 1 / variance so we want error = sqrt(1 / confidence) */

            cpl_image * inverse_conf = cpl_image_duplicate(
                                       jitters->limages[j]->confidence);
            double * ic_data = cpl_image_get_data_double(inverse_conf);
            const cpl_size nx = cpl_image_get_size_x(inverse_conf);
            const cpl_size ny = cpl_image_get_size_y(inverse_conf);
            for (cpl_size i=0; i<nx*ny; i++) {
                if (ic_data[i] > 0.0) {
                    ic_data[i] = sqrt(1.0 / ic_data[i]);
                }
            }

            hdrl_image * jitter_error = hdrl_image_create(hdrl_image_get_image(
                                       jitters->limages[j]->himage),
                                       inverse_conf);
            cpl_wcs * wcs = cpl_wcs_new_from_propertylist(
                                       jitters->limages[j]->plist);
            cpl_table * jitter_table = hdrl_resample_image_to_table(
                                       jitter_error,
                                       wcs);
            cpl_wcs_delete(wcs);
            cpl_image_delete(inverse_conf);
            hdrl_image_delete(jitter_error);

            if (restable == NULL) {
                restable = jitter_table;
            } else {
                cpl_size nrow = cpl_table_get_nrow(restable);
                cpl_table_insert(restable, jitter_table, nrow+1);
                cpl_table_delete(jitter_table);
            }
        }   

        enu_located_imagelist_delete(jitters); jitters = NULL;
        object_jitters = std_jitters = NULL;
        enu_check_error_code("error set after loading error image of jitters");

        cpl_msg_info(cpl_func, "calling hdrl_resample_compute");
        result = hdrl_resample_compute(restable, method, outgrid, template_wcs);
        cpl_table_delete(restable); restable = NULL;
        enu_check_error_code("error after resample_compute of image confidence");

        /* Calculate the confidence from the result errors. Raw confidence is
           1 / variance or result_error / result_error^3 */

        stack_conf = cpl_image_duplicate(
                                       hdrl_image_get_error(
                                       hdrl_imagelist_get(
                                       result->himlist, 0)));
        cpl_image * sc_copy = cpl_image_duplicate(stack_conf);
        cpl_image_divide(stack_conf, sc_copy);
        cpl_image_divide(stack_conf, sc_copy);
        cpl_image_divide(stack_conf, sc_copy);
        cpl_image_delete(sc_copy);

        enu_normalise_confidence(stack_conf);

        hdrl_resample_result_delete(result); result = NULL;
    }

    /* assemble result. hdrl_create makes copies, located_image_new takes
       ownership, hence the different memory management */

    stacked = enu_located_image_new(stack_himage,
                                    NULL,
                                    stack_conf,
                                    NULL,
                                    NULL,
                                    stack_header,
                                    NULL,
                                    NULL,
                                    NULL,
                                    NULL,
                                    NULL);
    enu_check_error_code("error set after image stacking");

    /* generate a source catalogue for the stacked result */

    cpl_wcs * result_wcs = cpl_wcs_new_from_propertylist(stacked->plist);
    stacked->objects = enu_catalogue_compute(stacked->himage,
                                             stacked->confidence,
                                             result_wcs,
                                             cat_params);
    cpl_wcs_delete(result_wcs);
    enu_check_error_code("error in target reduction: catalogue");

    /* add result properties in applist */

    applist = cpl_propertylist_new();

    /* update PRO.CATG and stack info */

    cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, out_catg);
    cpl_propertylist_update_string(applist, "PRODCATG", "SCIENCE.IMAGE");
    cpl_propertylist_update_string(applist, "BUNIT", cpl_property_get_string(bunit));
    cpl_propertylist_set_comment(applist, "BUNIT", cpl_property_get_comment(bunit));
    cpl_propertylist_update_int(applist, "NCOMBINE", ncombine);
    cpl_propertylist_update_double(applist, "TEXPTIME", total_exptime);
    cpl_propertylist_update_double(applist, "EXPTIME", total_exptime);
    cpl_propertylist_update_double(applist, "MJD-OBS", mjd_start);
    cpl_propertylist_update_double(applist, "MJD-END", mjd_end);
    for (cpl_size j = 0; j < njitters; ++j) {
        if (cpl_vector_get(obsid, j) > 0.0) {
            char * pname = cpl_sprintf("OBID%.0i", (int)(j+1));
            cpl_propertylist_update_int(applist, pname,
                                        (int)cpl_vector_get(obsid, j));
            cpl_free(pname);
        }
    }

    /* write PHOTZP info */

    cpl_propertylist_update_string(applist, "PHOTSYS", photsys_0);
    cpl_propertylist_update_double(applist, "PHOTZP", photzp_0);
    cpl_propertylist_set_comment(applist, "PHOTZP", 
                                 "MAG=-2.5*log(data)+PHOTZP+APCOR");
    cpl_propertylist_update_string(applist, "FLUXCAL", 
                                   cpl_property_get_string(fluxcal_0));
    cpl_propertylist_set_comment(applist, "FLUXCAL", 
                                 cpl_property_get_comment(fluxcal_0));
    cpl_propertylist_update_double(applist, "PHOTZPER", photzper_0);
    cpl_propertylist_update_string(applist, "ZPMETHOD", 
                                   cpl_property_get_string(zp_method_0));
    cpl_propertylist_set_comment(applist, "ZPMETHOD", 
                                 cpl_property_get_comment(zp_method_0));
    cpl_propertylist_update_double(applist, "ESO DRS EXTCOEF", extcoef_0);
    cpl_propertylist_set_comment(applist, "ESO DRS EXTCOEF",
                                 "[mag] Assumed extinction coefficient");

    /* try to get actual fwhm from catalogue, failing that use the PSF
       from the first stacked image */

    double fwhm_pix = -1.0;
    if (stacked->objects &&
        stacked->objects->qclist &&
        cpl_propertylist_has(stacked->objects->qclist, "ESO QC IMAGE_SIZE")) {

        fwhm_pix = cpl_propertylist_get_double(stacked->objects->qclist,
                                               "ESO QC IMAGE_SIZE");
    }

    if (fwhm_pix != -1.0) {
        cpl_propertylist_update_double(applist, "PSF_FWHM",
                                       fwhm_pix * pixsize);
        cpl_propertylist_set_comment(applist, "PSF_FWHM",
                                    "Average FWHM of stellar objects[arcsec]");
    } else {
        cpl_propertylist_update_double(applist, "PSF_FWHM",
                                       cpl_property_get_double(jitter_psf_fwhm));
        cpl_propertylist_set_comment(applist, "PSF_FWHM", 
                                     cpl_property_get_comment(jitter_psf_fwhm));
        fwhm_pix = cpl_property_get_double(jitter_psf_fwhm) / pixsize;
    }

    double abmaglim = -1.0;
    enu_calc_maglim(stacked, photzp_0, fwhm_pix, &abmaglim);
    cpl_propertylist_update_double(applist, "ABMAGLIM", abmaglim);
    cpl_propertylist_set_comment(applist, "ABMAGLIM", "5-sigma "
                                 "limiting AB magnitude");

    /* ABMAGSAT. Data calibrated as photons/s so exptime=1 for calculation.
       Formula copied from CASU casu_calculate_abmag_sat */

    double mean_sky = 0.0;
    if (cpl_propertylist_has(stacked->objects->qclist, "ESO QC MEAN_SKY")) {
        mean_sky = cpl_propertylist_get_double(stacked->objects->qclist,
                                               "ESO QC MEAN_SKY");
    } else {
        cpl_msg_warning(cpl_func, "ESO QC MEAN_SKY not found, mean "
                        "sky assumed 0");
    }
    double exptime = 1.0;
    double abmagsat = -1.0;
    if ((CPL_MATH_PI_4 * CPL_MATH_LN2 * (saturation - mean_sky) * 
        pow(fwhm_pix, 2.0) / exptime) > 0.0) {
        abmagsat = photzp_0 - 2.5 * log10(CPL_MATH_PI_4 * CPL_MATH_LN2 * 
                   (saturation - mean_sky) * pow(fwhm_pix, 2.0) / exptime);
    }
    cpl_msg_debug(cpl_func, "abmagsat stack %5.3e %5.3e %5.3e %5.3e %5.3e",
                  photzp_0, saturation, mean_sky, fwhm_pix, exptime);
    cpl_propertylist_update_double(applist, "ABMAGSAT", abmagsat);
    cpl_propertylist_set_comment(applist, "ABMAGSAT", "Saturation "
                                 "limit for point sources (AB mag)");

    /* set RA, DEC from WCS rather than let dfs copy it from one of the
       frames */

    {
        cpl_wcs * stack_wcs = cpl_wcs_new_from_propertylist(stacked->plist);
        double ra = 0.0;
        double dec = 0.0;
        enu_get_ra_dec(stack_wcs, &ra, &dec);

        cpl_propertylist_update_double(applist, "RA", ra);
        cpl_propertylist_update_double(applist, "DEC", dec);

        cpl_wcs_delete(stack_wcs);
    }
    enu_check_error_code("error set after setting header keywords");
 	
    /* generate name of file with stacked image */

    char * stack_fname = enu_repreface(proto_copy, "stack");    

    enu_dfs_save_limage(frameset,
                        parlist,
                        used,
                        CPL_FALSE,
                        stacked,
                        recipe_name,
                        inherit,
                        applist,
                        PACKAGE "/" PACKAGE_VERSION,
                        stack_fname);

    /* save the catalogue */

    cpl_frame * stack_frame = cpl_frame_new();
    cpl_frame_set_filename(stack_frame, stack_fname);
    cpl_frame_set_tag(stack_frame, ERIS_NIX_IMG_CATALOGUE_PRO_CATG);
    cpl_frame_set_type(stack_frame, CPL_FRAME_TYPE_ANY);
    cpl_frame_set_group(stack_frame, CPL_FRAME_GROUP_PRODUCT);
    cpl_frame_set_level(stack_frame, CPL_FRAME_LEVEL_FINAL);
    cpl_frameset_insert(used, stack_frame);
    cpl_free(stack_fname);

    //cpl_table_dump(stacked->objects->catalogue, 1, 10, NULL);

    /* ..the header for the table extension */
    tablelist = cpl_propertylist_duplicate(stacked->objects->qclist);
    cpl_propertylist_copy_property_regexp(tablelist,
                                          applist,
                                          "ABMAGLIM|ABMAGSAT|PSF_FWHM|PHOTSYS|"
                                          "RA|DEC",
                                          CPL_FALSE);
    enu_dfs_save_catalogue(frameset,
                           stack_frame,
                           parlist,
                           stacked->objects->catalogue,
                           tablelist,
                           "image source catalogue",
                           recipe_name,
                           PACKAGE "/" PACKAGE_VERSION,
                           "cat.stack",
                           ERIS_NIX_IMG_CATALOGUE_PRO_CATG);

    /* useful for outputting stack result with catalogue for debugging
       BUT CATALOGUE OUTPUT TBD IN enu_dfs_save_limage */
    if (CPL_FALSE) {
        cpl_propertylist_update_string(applist, "PRODCATG", "ANCILLARY.IMAGE");
        enu_dfs_save_limage(frameset,
                            parlist,
                            provenance,
                            CPL_FALSE,
                            stacked,
                            recipe_name,
                            inherit,
                            applist,
                            PACKAGE "/" PACKAGE_VERSION,
                            "stack_cat_debug.fits");
    }

cleanup:
    cpl_propertylist_delete(applist);
    cpl_property_delete(bunit);
    hdrl_parameter_delete(cat_params);
    cpl_property_delete(fluxcal_0);
    cpl_frame_delete(inherit);
    cpl_property_delete(jitter_psf_fwhm);
    hdrl_parameter_delete(method);
    enu_located_imagelist_delete(object_jitters);
    cpl_vector_delete(obsid);
    hdrl_parameter_delete(outgrid);
    cpl_free(photsys_0);
    cpl_free(proto_copy);
    cpl_frameset_delete(provenance);
    cpl_table_delete(restable);
    hdrl_resample_result_delete(result);
    enu_located_image_delete(stacked);
    enu_located_imagelist_delete(std_jitters);
    cpl_propertylist_delete(tablelist);
    cpl_wcs_delete(template_wcs);
    cpl_frameset_delete(used);
    cpl_property_delete(zp_method_0);

    return (int)cpl_error_get_code();
}



/**@}*/
