/* Unit tests for eris_ifu_dark_static.c */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include <hdrl.h>
#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_dark_static.h"
#include "eris_ifu_dfs.h"
#include "eris_utils.h"

/* Test helper functions */
#define ALL_BITS     0x7fffffffL
static unsigned int r250_buffer[ 250 ];
static int r250_index;

double eris_dr250(void)		/* returns a random double in range 0..1 */
{
	register int	j;
	register unsigned int new_rand;

	if ( r250_index >= 147 )
		j = r250_index - 147;	/* wrap pointer around */
	else
		j = r250_index + 103;

	new_rand = r250_buffer[ r250_index ] ^ r250_buffer[ j ];
	r250_buffer[ r250_index ] = new_rand;

	if ( r250_index >= 249 )	/* increment pointer for next time */
		r250_index = 0;
	else
		r250_index++;

	return (double)new_rand / ALL_BITS;

}
#define uniform(a,b)    ( a + (b - a) * eris_dr250() )

static hdrl_imagelist* create_test_dark_list(int nx, int ny, int nframes) {
    hdrl_imagelist* list = hdrl_imagelist_new();
    for (int i = 0; i < nframes; i++) {
        cpl_image* img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        cpl_image* err = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        
        // Fill with test data - simple pattern
        for (int x = 1; x <= nx; x++) {
            for (int y = 1; y <= ny; y++) {
                //cpl_image_set(img, x, y, 100.0 + i + x*0.1 + y*0.1);
                cpl_image_set(img, x, y, 100.0 + uniform(0, 1));
                cpl_image_set(err, x, y, 1.0);
            }
        }
        
        hdrl_image* hdrl_img = hdrl_image_create(img, err);
        hdrl_imagelist_set(list, hdrl_img, i);
        
        cpl_image_delete(img);
        cpl_image_delete(err);
    }
    return list;
}

static cpl_parameterlist* create_test_parlist(void) {
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    
    // Add required parameters for QC
    cpl_parameter* param;
    
    eris_ifu_add_std_params(parlist, "eris_ifu_dark");
    eris_parlist_config_add_bpm(parlist, "eris_ifu_dark");
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_xmin", 
            CPL_TYPE_INT, "FPN x min", "pixels", 100);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_xmax",
            CPL_TYPE_INT, "FPN x max", "pixels", 200);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_ymin",
            CPL_TYPE_INT, "FPN y min", "pixels", 100);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_ymax",
            CPL_TYPE_INT, "FPN y max", "pixels", 200);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_hsize",
            CPL_TYPE_INT, "FPN hsize", "pixels", 5);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_fpn_nsamp",
            CPL_TYPE_INT, "FPN nsamp", "samples", 10);
    cpl_parameterlist_append(parlist, param);
    
    // RON parameters
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_xmin",
            CPL_TYPE_INT, "RON x min", "pixels", 100);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_xmax",
            CPL_TYPE_INT, "RON x max", "pixels", 200);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_ymin",
            CPL_TYPE_INT, "RON y min", "pixels", 100);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_ymax",
            CPL_TYPE_INT, "RON y max", "pixels", 200);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_hsize",
            CPL_TYPE_INT, "RON hsize", "pixels", 5);
    cpl_parameterlist_append(parlist, param);
    
    param = cpl_parameter_new_value("eris.eris_ifu_dark.qc_ron_nsamp",
            CPL_TYPE_INT, "RON nsamp", "samples", 10);
    cpl_parameterlist_append(parlist, param);
    
    return parlist;
}

/* Test cases */

// Test eris_ifu_dark_noise function
static void test_eris_ifu_dark_noise(void) {
    //cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    // Create test data
    const int nx = 10;
    const int ny = 10;
    const int nframes = 5;
    hdrl_imagelist* dark_list = create_test_dark_list(nx, ny, nframes);
    
    // Test the function
    cpl_image* noise_img = eris_ifu_dark_noise(dark_list);
    
    // Verify results
    cpl_test_nonnull(noise_img);
    cpl_test_eq(cpl_image_get_size_x(noise_img), nx);
    cpl_test_eq(cpl_image_get_size_y(noise_img), ny);
    
    // Clean up
    cpl_image_delete(noise_img);
    hdrl_imagelist_delete(dark_list);
}

// Test eris_ifu_dark_get_dqi function
static void test_eris_ifu_dark_get_dqi(void) {
    //cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    // Create test masks
    cpl_mask* master_bpm = cpl_mask_new(ERIS_IFU_DETECTOR_SIZE, ERIS_IFU_DETECTOR_SIZE);
    cpl_mask* bpm2d = cpl_mask_new(ERIS_IFU_DETECTOR_SIZE, ERIS_IFU_DETECTOR_SIZE);
    cpl_mask* bpm3d = cpl_mask_new(ERIS_IFU_DETECTOR_SIZE, ERIS_IFU_DETECTOR_SIZE);
    
    // Set some test patterns
    cpl_mask_set(master_bpm, 10, 10, 1);
    cpl_mask_set(bpm2d, 20, 20, 1);
    cpl_mask_set(bpm3d, 30, 30, 1);
    
    // Test the function
    cpl_image* dqi = eris_ifu_dark_get_dqi(master_bpm, bpm2d, bpm3d);
    
    // Verify results
    cpl_test_nonnull(dqi);
    cpl_test_eq(cpl_image_get_size_x(dqi), ERIS_IFU_DETECTOR_SIZE);
    cpl_test_eq(cpl_image_get_size_y(dqi), ERIS_IFU_DETECTOR_SIZE);
    
    // Check specific pixel values
    int rejected;
    cpl_test_eq(cpl_image_get(dqi, 10, 10, &rejected), ERIS_DQI_SAT);
    cpl_test_eq(cpl_image_get(dqi, 20, 20, &rejected), ERIS_DQI_BP_BPM2D);
    cpl_test_eq(cpl_image_get(dqi, 30, 30, &rejected), ERIS_DQI_BP_BPM3D);
    
    // Clean up
    cpl_image_delete(dqi);
    cpl_mask_delete(master_bpm);
    cpl_mask_delete(bpm2d);
    cpl_mask_delete(bpm3d);
}

// Test eris_ifu_dark_qc function
static void test_eris_ifu_dark_qc(void) {
    //cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    // Create test data
    const int nx = ERIS_IFU_DETECTOR_SIZE;
    const int ny = ERIS_IFU_DETECTOR_SIZE;
    const int nframes = 5;
    
    hdrl_imagelist* dark_list = create_test_dark_list(nx, ny, nframes);
    cpl_parameterlist* parlist = create_test_parlist();
    cpl_propertylist* qc_params = cpl_propertylist_new();
    cpl_mask* master_bpm = cpl_mask_new(nx, ny);
    cpl_image* quality_image = cpl_image_new(nx, ny, CPL_TYPE_INT);
    
    // Create a test master dark image
    cpl_image* dark_img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image* dark_err = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    for (int x = 1; x <= nx; x++) {
        for (int y = 1; y <= ny; y++) {
            cpl_image_set(dark_img, x, y, 100.0);
            cpl_image_set(dark_err, x, y, 1.0);
        }
    }
    hdrl_image* master_dark = hdrl_image_create(dark_img, dark_err);
    
    // Test the function
    cpl_error_code error = eris_ifu_dark_qc(parlist, master_dark, dark_list,
            master_bpm, quality_image, qc_params);
    
    // Verify results
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_nonnull(qc_params);
    
    // Check some QC parameters exist
    cpl_test(cpl_propertylist_has(qc_params, "ESO QC DARK NBADPIX"));
    cpl_test(cpl_propertylist_has(qc_params, "ESO QC DARK BPIXFRAC"));
    cpl_test(cpl_propertylist_has(qc_params, "ESO QC MASTERDARK MEAN"));
    
    // Clean up
    cpl_propertylist_delete(qc_params);
    cpl_parameterlist_delete(parlist);
    hdrl_imagelist_delete(dark_list);
    hdrl_image_delete(master_dark);
    cpl_image_delete(dark_img);
    cpl_image_delete(dark_err);
    cpl_mask_delete(master_bpm);
    cpl_image_delete(quality_image);
}

// Test main eris_ifu_dark_static function
static void test_eris_ifu_dark_static(void) {
    //cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    // Create test data
    const int nx = ERIS_IFU_DETECTOR_SIZE;
    const int ny = ERIS_IFU_DETECTOR_SIZE;
    const int nframes = 5;
    cpl_error_code error = CPL_ERROR_NONE;
    hdrl_imagelist* dark_list = create_test_dark_list(nx, ny, nframes);
    cpl_parameterlist* parlist = create_test_parlist();
    
    //cpl_parameterlist_dump(parlist, stderr);
    hdrl_parameter* pdark_collapse = hdrl_collapse_parameter_parse_parlist(parlist,
        "eris.eris_ifu_dark.collapse");
        
    //hdrl_parameter* pdark_collapse = hdrl_parameter_new();
    cpl_propertylist* qc_params = cpl_propertylist_new();
    cpl_test_eq_error(error, CPL_ERROR_NONE);

    // Output variables
    hdrl_image* master_dark = NULL;
    cpl_image* quality_image = NULL;
    cpl_image* master_bpm = NULL;
    cpl_image* contrib_map = NULL;
    cpl_mask* bpm2d_mask = NULL;
    cpl_mask* bpm3d_mask = NULL;
    
    // Test the function
    error = eris_ifu_dark_static(parlist, dark_list,
            pdark_collapse, &master_dark, &quality_image, &master_bpm,
            &contrib_map, &bpm2d_mask, &bpm3d_mask, qc_params);
    
    // Verify results
    cpl_test_eq_error(error, CPL_ERROR_NONE);
    cpl_test_nonnull(master_dark);
    cpl_test_nonnull(quality_image);
    cpl_test_nonnull(master_bpm);
    cpl_test_nonnull(contrib_map);
    cpl_test_nonnull(bpm2d_mask);
    cpl_test_nonnull(bpm3d_mask);
    
    // Clean up
    hdrl_image_delete(master_dark);
    cpl_image_delete(quality_image);
    cpl_image_delete(master_bpm);
    cpl_image_delete(contrib_map);
    cpl_mask_delete(bpm2d_mask);
    cpl_mask_delete(bpm3d_mask);
    cpl_propertylist_delete(qc_params);
    hdrl_parameter_delete(pdark_collapse);
    cpl_parameterlist_delete(parlist);
    hdrl_imagelist_delete(dark_list);
}

int main(void) {
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    test_eris_ifu_dark_noise();
    test_eris_ifu_dark_get_dqi();
    test_eris_ifu_dark_qc();
    test_eris_ifu_dark_static();
    
    return cpl_test_end(0);
}