/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <eris_ifu_efficiency_response.h>
#include <eris_ifu_functions.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_efficiency_response_test  
 *           Unit test of eris_ifu_efficiency_response
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Prototypes
 -----------------------------------------------------------------------------*/

static void test_eris_can_extract(void);
static void test_eris_can_flux_calibrate(void);
static void test_eris_can_compute_efficiency(void);
static void test_eris_can_compute_response(void);
static void test_eris_efficiency_compute(void);
static void test_eris_response_compute(void);
static void test_eris_flux_calibrate_spectra(void);
static void test_eris_flux_calibrate_cube(void);
static void test_eris_flux_calibrate_cube2(void);

/* Helper functions */
static cpl_frame* create_test_frame(const char* tag, const char* filename);
static cpl_parameterlist* create_test_parlist(void);
static void cleanup_test_files(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test frame
  @param    tag     Frame tag
  @param    filename Frame filename
  @return   New frame or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_frame* create_test_frame(const char* tag, const char* filename)
{
    cpl_frame* frame = cpl_frame_new();
    if (frame == NULL) return NULL;
    
    cpl_frame_set_tag(frame, tag);
    cpl_frame_set_filename(frame, filename);
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    
    return frame;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create test parameter list
  @return   New parameter list or NULL
 */
/*----------------------------------------------------------------------------*/
static cpl_parameterlist* create_test_parlist(void)
{
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    if (parlist == NULL) return NULL;
    
    /* Add any required parameters here */
    cpl_parameter* param = cpl_parameter_new_value("test_param", 
        CPL_TYPE_STRING, "Test parameter", "test", "test_value");
    cpl_parameterlist_append(parlist, param);
    
    return parlist;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to cleanup test files
 */
/*----------------------------------------------------------------------------*/
static void cleanup_test_files(void)
{
    remove("test_sci.fits");
    remove("test_cat.fits");
    remove("test_atmext.fits");
    remove("test_response.fits");
    remove("test_efficiency.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_can_extract function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_can_extract(void)
{
    cpl_frameset* sof = NULL;
    
    /* Test with NULL input */
    cpl_test(!eris_can_extract(NULL));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Create empty frameset */
    sof = cpl_frameset_new();
    cpl_test(!eris_can_extract(sof));
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Add required frames */
    cpl_frame* frame = cpl_frame_new();
    cpl_frame_set_tag(frame, "OBJECT_CUBE_COADD");
    cpl_frameset_insert(sof, frame);
    
    cpl_test(eris_can_extract(sof));
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(sof);
    remove("test_sci.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_can_flux_calibrate function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_can_flux_calibrate(void)
{
    cpl_frameset* sof = NULL;
    
    /* Test with NULL input */
    cpl_test_eq(eris_can_flux_calibrate(NULL), 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Create empty frameset */
    sof = cpl_frameset_new();
    cpl_test_eq(eris_can_flux_calibrate(sof), 0);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Add required frames */
    cpl_frame* frame_sci = cpl_frame_new();
    cpl_frame_set_tag(frame_sci, "OBJECT_CUBE_COADD");
    cpl_frame_set_filename(frame_sci, "OBJECT_CUBE_COADD.fits");
    cpl_frameset_insert(sof, frame_sci);
    
    cpl_frame* frame_resp = cpl_frame_new();
    cpl_frame_set_tag(frame_resp, "RESPONSE");
    cpl_frame_set_filename(frame_resp, "RESPONSE.fits");
    cpl_frameset_insert(sof, frame_resp);

    cpl_test_eq(eris_can_flux_calibrate(sof), 0);
    cpl_test_error(CPL_ERROR_NONE);
    

    cpl_frame* frame_extcoeff = cpl_frame_new();
    cpl_frame_set_tag(frame_extcoeff, "EXTCOEFF_TABLE");
    cpl_frame_set_filename(frame_extcoeff, "EXTCOEFF_TABLE.fits");
    cpl_frameset_insert(sof, frame_extcoeff);

    cpl_test_eq(eris_can_flux_calibrate(sof), 1);
    cpl_test_error(CPL_ERROR_NONE);

    /* Cleanup */
    cpl_frameset_delete(sof);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_can_compute_efficiency function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_can_compute_efficiency(void)
{
    cpl_frameset* sof = NULL;
    
    /* Test with NULL input */
    cpl_test_eq(eris_can_compute_efficiency(NULL), 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Create empty frameset */
    sof = cpl_frameset_new();
    cpl_test_eq(eris_can_compute_efficiency(sof), 0);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Add required frames */
    cpl_frame* frame_sci = cpl_frame_new();
    cpl_frame_set_tag(frame_sci, "SPECTRUM_NOFLAT");
    cpl_frame_set_filename(frame_sci, "SPECTRUM_NOFLAT.fits");
    cpl_frameset_insert(sof, frame_sci);
    
    cpl_frame* frame_eff_w = cpl_frame_new();
    cpl_frame_set_tag(frame_eff_w, "EFFICIENCY_WINDOWS");
    cpl_frame_set_filename(frame_eff_w, "EFFICIENCY_WINDOWS.fits");
    cpl_frameset_insert(sof, frame_eff_w);
    
    cpl_frame* frame_extcoef = cpl_frame_new();
    cpl_frame_set_tag(frame_extcoef, "EXTCOEFF_TABLE");
    cpl_frame_set_filename(frame_extcoef, "EXTCOEFF_TABLE.fits");
    cpl_frameset_insert(sof, frame_extcoef);
    
    cpl_test_eq(eris_can_compute_efficiency(sof), 1);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(sof);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_can_compute_response function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_can_compute_response(void)
{
    cpl_frameset* sof = NULL;
    
    /* Test with NULL input */
    cpl_test_eq(eris_can_compute_response(NULL), 0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Create empty frameset */
    sof = cpl_frameset_new();
    cpl_test_eq(eris_can_compute_response(sof), 0);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Add required frames */
    cpl_frame* frame_sci = cpl_frame_new();
    cpl_frame_set_tag(frame_sci, "SPECTRUM");
    cpl_frame_set_filename(frame_sci, "SPECTRUM.fits");
    cpl_frameset_insert(sof, frame_sci);

    cpl_frame* frame_std_cat = cpl_frame_new();
    cpl_frame_set_tag(frame_std_cat, "FLUX_STD_CATALOG");
    cpl_frame_set_filename(frame_std_cat, "FLUX_STD_CATALOG.fits");
    cpl_frameset_insert(sof, frame_std_cat);

    cpl_frame* frame_tellmod_cat = cpl_frame_new();
    cpl_frame_set_tag(frame_tellmod_cat, "TELL_MOD_CATALOG");
    cpl_frame_set_filename(frame_tellmod_cat, "TELL_MOD_CATALOG.fits");
    cpl_frameset_insert(sof, frame_tellmod_cat);


    cpl_frame* frame_resp_fitp_cat = cpl_frame_new();
    cpl_frame_set_tag(frame_resp_fitp_cat, "RESP_FIT_POINTS_CATALOG");
    cpl_frame_set_filename(frame_resp_fitp_cat, "RESP_FIT_POINTS_CATALOG.fits");
    cpl_frameset_insert(sof, frame_resp_fitp_cat);

    cpl_frame* frame_ext_coeff = cpl_frame_new();
    cpl_frame_set_tag(frame_ext_coeff, "EXTCOEFF_TABLE");
    cpl_frame_set_filename(frame_ext_coeff, "EXTCOEFF_TABLE.fits");
    cpl_frameset_insert(sof, frame_ext_coeff);

    cpl_frame* frame_quality_areas = cpl_frame_new();
    cpl_frame_set_tag(frame_quality_areas, "QUALITY_AREAS");
    cpl_frame_set_filename(frame_quality_areas, "QUALITY_AREAS.fits");
    cpl_frameset_insert(sof, frame_quality_areas);


    cpl_frame* frame_resp_w = cpl_frame_new();
    cpl_frame_set_tag(frame_resp_w, "RESPONSE_WINDOWS");
    cpl_frame_set_filename(frame_resp_w, "RESPONSE_WINDOWS.fits");
    cpl_frameset_insert(sof, frame_resp_w);

    cpl_frame* frame_fit_areas = cpl_frame_new();
    cpl_frame_set_tag(frame_fit_areas, "FIT_AREAS");
    cpl_frame_set_filename(frame_fit_areas, "FIT_AREAS.fits");
    cpl_frameset_insert(sof, frame_fit_areas);

    cpl_test_eq(eris_can_compute_response(sof), 1);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(sof);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_efficiency_compute function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_efficiency_compute(void)
{
    cpl_frame* frm_sci = create_test_frame("SCIENCE", "test_sci.fits");
    cpl_frame* frm_cat = create_test_frame("CATALOG", "test_cat.fits");
    cpl_frame* frm_atmext = create_test_frame("ATMOSPHERIC_EXTINCTION", "test_atmext.fits");
    cpl_frameset* frameset = cpl_frameset_new();
    cpl_parameterlist* parlist = create_test_parlist();
    
    /* Test with NULL inputs */
    cpl_test_null(eris_efficiency_compute(NULL, frm_cat, frm_atmext, frameset, parlist, "test"));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_test_null(eris_efficiency_compute(frm_sci, NULL, frm_atmext, frameset, parlist, "test"));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    cpl_test_null(eris_efficiency_compute(frm_sci, frm_cat, NULL, frameset, parlist, "test"));
    cpl_test_error(CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frame_delete(frm_sci);
    cpl_frame_delete(frm_cat);
    cpl_frame_delete(frm_atmext);
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_response_compute function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_response_compute(void)
{
    cpl_frameset* ref_set = cpl_frameset_new();
    cpl_frameset* sof = cpl_frameset_new();
    cpl_parameterlist* config = create_test_parlist();
    
    /* Test with NULL inputs */
    cpl_test_eq(eris_response_compute(NULL, config, ref_set, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_response_compute("test", NULL, ref_set, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_response_compute("test", config, NULL, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_response_compute("test", config, ref_set, NULL), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Cleanup */
    cpl_frameset_delete(ref_set);
    cpl_frameset_delete(sof);
    cpl_parameterlist_delete(config);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_flux_calibrate_spectra function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_flux_calibrate_spectra(void)
{
    cpl_frameset* ref_set = cpl_frameset_new();
    cpl_frameset* sof = cpl_frameset_new();
    cpl_parameterlist* config = create_test_parlist();
    
    /* Test with NULL inputs */
    cpl_test_eq(eris_flux_calibrate_spectra(NULL, "test", config, ref_set, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_spectra("test", NULL, config, ref_set, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_spectra("test", "test", NULL, ref_set, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_spectra("test", "test", config, NULL, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_spectra("test", "test", config, ref_set, NULL), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Test with empty framesets */
    cpl_test_eq(eris_flux_calibrate_spectra("test", "test", config, ref_set, sof), CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* Add required frames */
    cpl_frame* frame_sci = create_test_frame("SPECTRUM", "test_spectrum.fits");
    cpl_frame_set_filename(frame_sci, "test_spectrum.fits");
    cpl_frame_set_group(frame_sci, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame_sci, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame_sci, "SPECTRUM");
    cpl_frame_set_type(frame_sci, CPL_FRAME_TYPE_IMAGE);

    cpl_frameset_insert(sof, frame_sci);
    cpl_test_error(CPL_ERROR_NONE);
    
    cpl_frame* frame_resp = create_test_frame("RESPONSE", "test_response.fits");
    cpl_frame_set_filename(frame_sci, "test_response.fits");
    cpl_frame_set_group(frame_sci, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame_sci, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame_sci, "RESPONSE");
    cpl_frame_set_type(frame_sci, CPL_FRAME_TYPE_TABLE);
    cpl_frameset_insert(ref_set, frame_resp);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with valid inputs */
    //cpl_test_eq(eris_flux_calibrate_spectra("test", "test", config, ref_set, sof), CPL_ERROR_NONE);
    //cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(ref_set);
    cpl_frameset_delete(sof);
    cpl_parameterlist_delete(config);
    remove("test_spectrum.fits");
    remove("test_response.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_flux_calibrate_cube function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_flux_calibrate_cube(void)
{
    cpl_frameset* sof = cpl_frameset_new();
    cpl_parameterlist* config = create_test_parlist();
    
    /* Test with NULL inputs */
    cpl_test_eq(eris_flux_calibrate_cube(NULL, "test", "test", config, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube("test", NULL, "test", config, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube("test", "test", NULL, config, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube("test", "test", "test", NULL, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube("test", "test", "test", config, NULL), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Test with empty frameset */
    cpl_test_eq(eris_flux_calibrate_cube("test", "test", "test", config, sof), CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* Add required frames */
    cpl_frame* frame_sci = create_test_frame("OBJECT_CUBE_COADD", "test_cube.fits");
    cpl_frame_set_filename(frame_sci, "test_cube.fits");
    cpl_frame_set_group(frame_sci, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame_sci, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame_sci, "OBJECT_CUBE_COADD");
    cpl_frame_set_type(frame_sci, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(sof, frame_sci);
    cpl_test_error(CPL_ERROR_NONE);
    
    cpl_frame* frame_resp = create_test_frame("RESPONSE", "test_response.fits");
    cpl_frame_set_filename(frame_sci, "test_response.fits");
    cpl_frame_set_group(frame_sci, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame_sci, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame_sci, "RESPONSE");
    cpl_frame_set_type(frame_sci, CPL_FRAME_TYPE_TABLE);
    cpl_frameset_insert(sof, frame_resp);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with valid inputs */
    //cpl_test_eq(eris_flux_calibrate_cube("test", "test", "test", config, sof), CPL_ERROR_NONE);
    //cpl_test_error(CPL_ERROR_NONE);

    /* Cleanup */
    cpl_frameset_delete(sof);
    cpl_parameterlist_delete(config);
    remove("test_cube.fits");
    remove("test_response.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_flux_calibrate_cube2 function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_flux_calibrate_cube2(void)
{
    cpl_frameset* sof = cpl_frameset_new();
    cpl_parameterlist* parlist = create_test_parlist();
    
    /* Test with NULL inputs */
    cpl_test_eq(eris_flux_calibrate_cube2(NULL, "test", "test", parlist, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube2("test", NULL, "test", parlist, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube2("test", "test", NULL, parlist, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube2("test", "test", "test", NULL, sof), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    cpl_test_eq(eris_flux_calibrate_cube2("test", "test", "test", parlist, NULL), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Test with empty frameset */
    cpl_test_eq(eris_flux_calibrate_cube2("test", "test", "test", parlist, sof), CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_error(CPL_ERROR_DATA_NOT_FOUND);

    /* Add required frames */
    cpl_frame* frame_sci = create_test_frame("OBJECT_CUBE_COADD", "test_cube.fits");
    cpl_frame_set_filename(frame_sci, "test_cube.fits");
    cpl_frame_set_group(frame_sci, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame_sci, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame_sci, "OBJECT_CUBE_COADD");
    cpl_frame_set_type(frame_sci, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(sof, frame_sci);
    cpl_test_error(CPL_ERROR_NONE);

    cpl_frame* frame_resp = create_test_frame("RESPONSE", "test_response.fits");
    cpl_frame_set_filename(frame_sci, "test_response.fits");
    cpl_frame_set_group(frame_sci, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_level(frame_sci, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_tag(frame_sci, "RESPONSE");
    cpl_frame_set_type(frame_sci, CPL_FRAME_TYPE_TABLE);
    cpl_frameset_insert(sof, frame_resp);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with valid inputs */
    //cpl_test_eq(eris_flux_calibrate_cube2("test", "test", "test", parlist, sof), CPL_ERROR_NONE);
    //cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_frameset_delete(sof);
    cpl_parameterlist_delete(parlist);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_efficiency_response module
 */
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_eris_can_extract();
    test_eris_can_flux_calibrate();
    test_eris_can_compute_efficiency();
    test_eris_can_compute_response();
    test_eris_efficiency_compute();
    test_eris_response_compute();
    test_eris_flux_calibrate_spectra(); //TODO: fix valid input test
    test_eris_flux_calibrate_cube();//TODO: fix valid input test
    test_eris_flux_calibrate_cube2();//TODO: fix valid input test
    
    cleanup_test_files();

    return cpl_test_end(0);
}

/**@}*/
