/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include <cpl.h>
#include <hdrl.h>
#include <eris_ifu_extract_spec_static.h>
#include <eris_ifu_functions.h>
#include <eris_ifu_vector.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_extract_spec_static_test  
 *           Unit test of eris_ifu_extract_spec_static
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Prototypes
 -----------------------------------------------------------------------------*/

static void test_eris_ifu_extract_spec_create_circle_mask(void);
static void test_eris_ifu_extract_spec_collapse(void);
static void test_eris_ifu_extract_spec_create_fit_mask(void);
static void test_eris_ifu_extract_spectrum(void);
static void test_eris_ifu_optimal_extraction(void);
static void test_eris_ifu_opt_extr_get_center_fwhm(void);
static void test_eris_ifu_opt_extr_create_mask(void);
static void test_eris_ifu_opt_extr_helper_functions(void);

/* Helper functions */
static hdrl_imagelist* create_test_cube(cpl_size nx, cpl_size ny, cpl_size nz);
static void cleanup_test_files(void);

/**@{*/


/**
  @brief    creates a normalised 2D Gauss image of given half size, sigma
  @param    hsize image half size
  @param    sigma image sigma
  @return   cpl_image* with 2D normalised Gaussian distribution
*/

static cpl_image*
eris_ifu_2DGauss_kernel(const cpl_size hsize, const double sigma)
{

	/* Check entries */
	cpl_ensure(hsize > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(sigma > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

	cpl_image* gauss2D = NULL;
	cpl_size size = 2 * hsize + 1;

	gauss2D = cpl_image_new(size, size, CPL_TYPE_DOUBLE);
	double* pdata = cpl_image_get_data(gauss2D);
	const double x0 = hsize;
	const double y0 = hsize;
	double dist = 0;
	double arg = 0;
	for(cpl_size j = 0; j < size; j++) {
		for(cpl_size i = 0; i < size; i++) {
			dist = ( (i - x0) * (i - x0) +
							    (j - y0) * (j - y0) );
			arg = 0.5 * ( dist / sigma / sigma);
			pdata[i+j*size] = exp( - arg);
		}
	}


	return gauss2D;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test cube
  @param    nx      X dimension
  @param    ny      Y dimension
  @param    nz      Z dimension (number of slices)
  @return   New cube or NULL
 */
/*----------------------------------------------------------------------------*/
static hdrl_imagelist* create_test_cube(cpl_size sx, cpl_size sy, cpl_size nz)
{
    hdrl_imagelist* cube = hdrl_imagelist_new();
    if (cube == NULL) return NULL;
    cpl_size hsize = 0.5 * sx;
    double sigma = 2.0;
    for (cpl_size i = 0; i < nz; i++) {
        
    	cpl_image* data = eris_ifu_2DGauss_kernel(hsize, sigma);
    	cpl_image* error = cpl_image_duplicate(data);
    	cpl_mask* mask = cpl_mask_new(sx, sy);
        cpl_image_set_bpm(data,mask);
    	cpl_image_power(error, 0.5);

        hdrl_image* slice = hdrl_image_create(data, error);
        hdrl_imagelist_set(cube, slice, i);
        cpl_image_delete(data);
        cpl_image_delete(error);
    }
    
    return cube;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to cleanup test files
 */
/*----------------------------------------------------------------------------*/
static void cleanup_test_files(void)
{
    //remove("test_cube.fits");
    remove("test_mask.fits");
    //remove("test_collapsed.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_extract_spec_create_circle_mask function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_extract_spec_create_circle_mask(void)
{
    cpl_size nx = 10;
    cpl_size ny = 10;
    cpl_size center_x = 5;
    cpl_size center_y = 5;
    double radius = 3.0;
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test normal case */
    cpl_image* mask = eris_ifu_extract_spec_create_circle_mask(center_x, center_y, 
                                                              radius, nx, ny);

    cpl_image_save(mask, "test_mask.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    cpl_test_nonnull(mask);
    cpl_test_eq(cpl_image_get_size_x(mask), nx);
    cpl_test_eq(cpl_image_get_size_y(mask), ny);
    
    /* Check some points inside and outside circle */
    int status;
    cpl_test_eq(cpl_image_get(mask, center_x, center_y, &status), 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_image_get(mask, 1, 1, &status), 0.0);
    cpl_test_error(CPL_ERROR_NONE);
    
    cpl_image_delete(mask);
    
    /* Test edge cases */
    mask = eris_ifu_extract_spec_create_circle_mask(1, 1, 0.5, nx, ny);
    cpl_test_nonnull(mask);
    cpl_image_delete(mask);
    
    /* Test invalid inputs */
    mask = eris_ifu_extract_spec_create_circle_mask(0, 0, radius, nx, ny);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);


    mask = eris_ifu_extract_spec_create_circle_mask(0, ny/2, radius, nx, ny);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    mask = eris_ifu_extract_spec_create_circle_mask(nx/2, 0, radius, nx, ny);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    mask = eris_ifu_extract_spec_create_circle_mask(nx/2, ny/2, 0, nx, ny);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    mask = eris_ifu_extract_spec_create_circle_mask(nx/2, ny/2, radius, 0, ny);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    mask = eris_ifu_extract_spec_create_circle_mask(nx/2, ny/2, radius, nx, 0);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_extract_spec_collapse function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_extract_spec_collapse(void)
{
    cpl_image* contribute = NULL;
    cpl_size sx = 11;
    cpl_size sy = 11;
    cpl_size sz = 5;
    hdrl_imagelist* cube = create_test_cube(sx, sy, sz);
    cpl_test_nonnull(cube);
    
    /* Test normal case */
    hdrl_image* collapsed = eris_ifu_extract_spec_collapse(cube, &contribute);
    cpl_test_nonnull(collapsed);
    cpl_test_nonnull(contribute);
    
    /* Check dimensions */
    cpl_test_eq(hdrl_image_get_size_x(collapsed), sx);
    cpl_test_eq(hdrl_image_get_size_y(collapsed), sy);
    
    /* Cleanup */
    hdrl_image_delete(collapsed);
    cpl_image_delete(contribute);
    hdrl_imagelist_delete(cube);
    
    /* Test NULL input */
    collapsed = eris_ifu_extract_spec_collapse(NULL, &contribute);
    cpl_test_null(collapsed);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    collapsed = eris_ifu_extract_spec_collapse(cube, NULL);
    cpl_test_null(collapsed);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_extract_spec_create_fit_mask function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_extract_spec_create_fit_mask(void)
{
    /* Create test image */
	cpl_size sx = 11;
	cpl_size sy = 11;


    cpl_image* data = eris_ifu_2DGauss_kernel(sx/2, 2.0);
    cpl_image* error = cpl_image_duplicate(data);
    cpl_image_power(error, 0.5);

    hdrl_image* himg = hdrl_image_create(data, error);
    cpl_image_delete(data);
    cpl_image_delete(error);
    cpl_test_nonnull(himg);

    /* Test normal case */
    cpl_image* mask = eris_ifu_extract_spec_create_fit_mask(himg);
    cpl_test_nonnull(mask);

    /* Check dimensions */
    cpl_test_eq(cpl_image_get_size_x(mask), sx);
    cpl_test_eq(cpl_image_get_size_y(mask), sy);
    cpl_image_save(mask, "test_mask.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    /* Check peak is included */

    int status;
    cpl_test_eq(cpl_image_get(mask, (sx+1)/2, (sy+1)/2, &status), 1.0);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Cleanup */
    cpl_image_delete(mask);
    hdrl_image_delete(himg);
    
    /* Test NULL input */
    mask = eris_ifu_extract_spec_create_fit_mask(NULL);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_extract_spectrum function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_extract_spectrum(void)
{
	cpl_size sx = 11;
	cpl_size sy = 11;
	cpl_size sz = 5;
    hdrl_imagelist* cube = create_test_cube(sx, sx, sz);
    cpl_test_nonnull(cube);
    
    cpl_image* mask = cpl_image_new(sx, sy, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(mask, 1, 1, sx, sy, 1.0);
    
    cpl_vector* error = NULL;
    cpl_vector* totalFlux = NULL;

    /* Test normal case */
    cpl_bivector* spectrum = eris_ifu_extract_spectrum(cube, mask, 1.0, 0.1, 
                                                      &error, &totalFlux);
    cpl_test_nonnull(spectrum);
    cpl_test_nonnull(error);
    cpl_test_nonnull(totalFlux);
    
    /* Check dimensions */
    cpl_test_eq(cpl_bivector_get_size(spectrum), sz);
    cpl_test_eq(cpl_vector_get_size(error), sz);
    cpl_test_eq(cpl_vector_get_size(totalFlux), sz);
    
    /* Cleanup */
    cpl_bivector_delete(spectrum);
    cpl_vector_delete(error);
    cpl_vector_delete(totalFlux);
    cpl_image_delete(mask);
    hdrl_imagelist_delete(cube);
    
    /* Test NULL inputs */
    spectrum = eris_ifu_extract_spectrum(NULL, mask, 1.0, 0.1, &error, &totalFlux);
    cpl_test_null(spectrum);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    spectrum = eris_ifu_extract_spectrum(cube, NULL, 1.0, 0.1, &error, &totalFlux);
    cpl_test_null(spectrum);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    spectrum = eris_ifu_extract_spectrum(cube, mask, 1.0, 0.1, NULL, &totalFlux);
    cpl_test_null(spectrum);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    spectrum = eris_ifu_extract_spectrum(cube, mask, 1.0, 0.1, &error, NULL);
    cpl_test_null(spectrum);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_optimal_extraction function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_optimal_extraction(void)
{
	cpl_size sx = 11;
	cpl_size sy = 11;
	cpl_size sz = 5;
    hdrl_imagelist* cube = create_test_cube(sx, sy, sz);
    cpl_test_nonnull(cube);

    cpl_imagelist* cube_dqi = cpl_imagelist_new();
    for (int i = 0; i < sz; i++) {
        cpl_image* dqi = cpl_image_new(sx, sy, CPL_TYPE_INT);
        cpl_image_fill_window(dqi, 1, 1, sx, sy, 1);
        cpl_imagelist_set(cube_dqi, dqi, i);
    }

    cpl_image* mask = cpl_image_new(sx, sy, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(mask, 1, 1, sx, sy, 1.0);

    cpl_vector* error_out = NULL;
    
    /* Test normal case */
    cpl_bivector* result = eris_ifu_optimal_extraction(cube, cube_dqi, mask, 
                                                      1.1, 0.1, 1, &error_out);
    cpl_test_nonnull(result);
    cpl_test_nonnull(error_out);
    
    /* Check dimensions */
    cpl_test_eq(cpl_bivector_get_size(result), sz);
    cpl_test_eq(cpl_vector_get_size(error_out), sz);
    
    /* Cleanup */
    cpl_bivector_delete(result);
    cpl_vector_delete(error_out);
    cpl_image_delete(mask);
    cpl_imagelist_delete(cube_dqi);
    hdrl_imagelist_delete(cube);
    
    /* Test NULL inputs */
    result = eris_ifu_optimal_extraction(NULL, cube_dqi, mask, 1.0, 0.1, 1, &error_out);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_opt_extr_get_center_fwhm function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_opt_extr_get_center_fwhm(void)
{
    /* Create test image with Gaussian peak */
	cpl_size sx = 21;
	cpl_size sy = 21;

    cpl_mask* mask = cpl_mask_new(sx, sy);
    double center_x = 10.0;
    double center_y = 10.0;
    double sigma = 2.0;
    cpl_size hsize = 10;
    cpl_image* data = eris_ifu_2DGauss_kernel(hsize, sigma);

    cpl_image_set_bpm(data, mask);
    cpl_image* error = cpl_image_duplicate(data);
    cpl_image_power(error, 0.5);
    
    hdrl_image* img = hdrl_image_create(data, error);

    cpl_test_nonnull(img);
    //cpl_image_save(data, "test_gauss.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(data);
    cpl_image_delete(error);

    cpl_size xcen, ycen;
    double fwhm;

    /* Test normal case */
    cpl_error_code err = eris_ifu_opt_extr_get_center_fwhm(img, 2, &xcen, &ycen, &fwhm);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    /* Check results are reasonable */
    cpl_test_abs(xcen, sx/2, 1);
    cpl_test_abs(ycen, sy/2, 1);
    cpl_test_abs(fwhm, CPL_MATH_FWHM_SIG * sigma, 0.5); /* FWHM ≈ CPL_MATH_FWHM_SIG*sigma for Gaussian */
    
    /* Cleanup */
    hdrl_image_delete(img);
    
    /* Test NULL input */
    err = eris_ifu_opt_extr_get_center_fwhm(NULL, 2, &xcen, &ycen, &fwhm);
    cpl_test_eq_error(err, CPL_ERROR_NULL_INPUT);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test eris_ifu_opt_extr_create_mask function
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_opt_extr_create_mask(void)
{
    int nx = 21;
    int ny = 21;
    int xcen = 10;
    int ycen = 10;
    double fwhm = 4.0;
    
    /* Test normal case */
    cpl_mask* mask = eris_ifu_opt_extr_create_mask(nx, ny, xcen, ycen, fwhm);
    cpl_test_nonnull(mask);
    //cpl_mask_save(mask, "mask.fits", NULL, CPL_IO_DEFAULT);
    /* Check dimensions */
    cpl_test_eq(cpl_mask_get_size_x(mask), nx);
    cpl_test_eq(cpl_mask_get_size_y(mask), ny);
    
    /* Check center is included */
    cpl_test_eq(cpl_mask_get(mask, xcen, ycen),0);
    
    /* Cleanup */
    cpl_mask_delete(mask);
    
    /* Test invalid inputs */
    mask = eris_ifu_opt_extr_create_mask(0, ny, xcen, ycen, fwhm);
    cpl_test_null(mask);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test helper functions
 */
/*----------------------------------------------------------------------------*/
static void test_eris_ifu_opt_extr_helper_functions(void)
{
    /* Test vector operations */
    cpl_vector* vec = cpl_vector_new(5);
    cpl_vector_fill(vec, 4.0);
    
    cpl_error_code err = eris_ifu_opt_extr_vector_sqrt(vec);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    cpl_test_abs(cpl_vector_get(vec, 0), 2.0, DBL_EPSILON);
    
    cpl_vector_delete(vec);
    
    /* Test get/set row/col functions */
    cpl_image* img = cpl_image_new(5, 5, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(img, 1, 1, 5, 5, 1.0);
    
    cpl_vector* row = eris_ifu_opt_extr_get_row(img, 1);
    cpl_test_nonnull(row);
    cpl_test_eq(cpl_vector_get_size(row), 5);
    
    cpl_vector* col = eris_ifu_opt_extr_get_col(img, 1);
    cpl_test_nonnull(col);
    cpl_test_eq(cpl_vector_get_size(col), 5);
    
    err = eris_ifu_opt_extr_set_row(img, 1, row);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    err = eris_ifu_opt_extr_set_col(img, 1, col);
    cpl_test_eq_error(err, CPL_ERROR_NONE);
    
    cpl_vector_delete(row);
    cpl_vector_delete(col);
    cpl_image_delete(img);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_ifu_extract_spec_static module
 */
/*----------------------------------------------------------------------------*/
int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_eris_ifu_extract_spec_create_circle_mask();
    test_eris_ifu_extract_spec_collapse();
    test_eris_ifu_extract_spec_create_fit_mask();
    test_eris_ifu_extract_spectrum();
    test_eris_ifu_optimal_extraction();
    test_eris_ifu_opt_extr_get_center_fwhm();
    test_eris_ifu_opt_extr_create_mask();
    test_eris_ifu_opt_extr_helper_functions();
    
    cleanup_test_files();

    return cpl_test_end(0);
}

/**@}*/
