/*
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cpl.h>
#include "eris_ifu_sky_tweak.h"
#include "eris_ifu_error.h"
//#include "eris_ifu_test_utils.h"
#include <math.h>
/* Helper functions */
static cpl_imagelist* create_test_cube(int nx, int ny, int nz);
static cpl_propertylist* create_test_header(void);
static void cleanup_test_files(void);

/* Test functions */
static void test_sky_tweak_null_input(void);
static void test_sky_tweak_illegal_input(void);
static void test_sky_tweak_basic(void);
static void test_sky_tweak_with_stretch(void);
static void test_sky_tweak_thermal_bgd(void);
static void test_plot_cube_background(void);

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test cube
  @param    nx      Size in X
  @param    ny      Size in Y
  @param    nz      Size in Z (wavelength)
  @return   The created cube or NULL on error
 */
/*----------------------------------------------------------------------------*/
static cpl_imagelist* create_test_cube(int nx, int ny, int nz) {
    cpl_imagelist* cube;
    cpl_image* img;
    int i, j, k;
    double* data;
    const double sigma = 2.0;
    const double amplitude = 100.0;
    const int center_x = nx/2;
    const int center_y = ny/2;
    double sky_ampl = 1000.0;
    double sky_ampl1 = 400.0;
    double sky_ampl2 = 100.0;

    /* Create the cube */
    cube = cpl_imagelist_new();
    for (k = 0; k < nz; k++) {
        img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        //data = cpl_image_get_data_double(img);
        
        /* Add a Gaussian PSF */
        for(int y = 1; y <= ny; y++) {
        	for(int x = 1; x <= nx; x++) {
        		double dx = x - center_x;
        		double dy = y - center_y;
        		double r2 = dx*dx + dy*dy;
        		double value = amplitude * exp(-r2/(2*sigma*sigma));
        		cpl_image_set(img, x, y, value);
        	}
        }
        /* Add sky lines at specific wavelengths */
        if (k == 10 || k == 20 || k == 30) {
        	cpl_image_add_scalar(img,sky_ampl);
        }
        if (k-1 == 10 || k-1 == 20 || k-1 == 30) {
        	cpl_image_add_scalar(img,sky_ampl1);
        }
        if (k-2 == 10 || k-2 == 20 || k-2 == 30) {
        	cpl_image_add_scalar(img,sky_ampl2);
        }
        if (k+1 == 10 || k+1 == 20 || k+1 == 30) {
        	cpl_image_add_scalar(img,sky_ampl1);
        }
        if (k+2 == 10 || k+2 == 20 || k+2 == 30) {
        	cpl_image_add_scalar(img,sky_ampl2);
        }
        cpl_imagelist_set(cube, img, k);
    }
    cpl_imagelist_save(cube, "test_cube.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    return cube;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to create a test header
  @return   The created header or NULL on error
 */
/*----------------------------------------------------------------------------*/
static cpl_propertylist* create_test_header(void) {
    cpl_propertylist* header;
    
    header = cpl_propertylist_new();
    cpl_propertylist_append_int(header, "NAXIS3", 50);
    cpl_propertylist_append_double(header, "CRVAL3", 1.0);
    cpl_propertylist_append_double(header, "CDELT3", 0.1);
    cpl_propertylist_append_double(header, "CRPIX3", 1.0);
    
    return header;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Helper function to clean up test files
 */
/*----------------------------------------------------------------------------*/
static void cleanup_test_files(void) {
    /* Remove any temporary files created during tests */
    cpl_msg_info(cpl_func, "Cleaning up test files");
    remove("test_sky_tweak.fits");
    remove("test_cube_background.fits");
    remove("test_cube.fits");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test sky tweak with NULL inputs
 */
/*----------------------------------------------------------------------------*/
static void test_sky_tweak_null_input(void) {
	cpl_size sx = 21;
	cpl_size sy = 21;
	cpl_size sz = 50;
    cpl_imagelist* obj = create_test_cube(sx, sy, sz);
    cpl_imagelist* sky = create_test_cube(sx, sy, sz);
    cpl_propertylist* header = create_test_header();
    cpl_imagelist* new_sky = NULL;
    cpl_imagelist* result;
    cpl_test_error(CPL_ERROR_NONE);
    /* Test NULL object cube */
    result = eris_ifu_sky_tweak(NULL, sky, header, 0.3, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Test NULL sky cube */
    result = eris_ifu_sky_tweak(obj, NULL, header, 0.3, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Test NULL header */
    result = eris_ifu_sky_tweak(obj, sky, NULL, 0.3, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Cleanup */
    cpl_imagelist_delete(obj);
    cpl_imagelist_delete(sky);
    cpl_propertylist_delete(header);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test sky tweak with illegal inputs
 */
/*----------------------------------------------------------------------------*/
static void test_sky_tweak_illegal_input(void) {
	cpl_size sx = 21;
    cpl_size sy = 21;
	cpl_size sz = 50;
	cpl_imagelist* obj = create_test_cube(sx, sy, sz);
    cpl_imagelist* sky = create_test_cube(2*sx, 2*sy, sz); /* Different size */
    cpl_propertylist* header = create_test_header();
    cpl_imagelist* new_sky = NULL;
    cpl_imagelist* result;

    /* Test mismatched cube sizes */
    result = eris_ifu_sky_tweak(obj, sky, header, 0.3, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    /* Test invalid min_frac */
    result = eris_ifu_sky_tweak(obj, obj, header, -1.0, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    /* Test invalid stretch_resampling */
    result = eris_ifu_sky_tweak(obj, obj, header, 0.3, 1, 0, 1, 2, 3, 0, &new_sky);
    cpl_test_null(result);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    /* Cleanup */
    cpl_imagelist_delete(obj);
    cpl_imagelist_delete(sky);
    cpl_propertylist_delete(header);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test basic sky tweak functionality
 */
/*----------------------------------------------------------------------------*/
static void test_sky_tweak_basic(void) {
	cpl_size sx = 21;
	cpl_size sy = 21;
	cpl_size sz = 50;

	cpl_imagelist* obj = create_test_cube(sx, sy, sz);
	cpl_imagelist* sky = create_test_cube(sx, sy, sz);
    cpl_propertylist* header = create_test_header();
    cpl_imagelist* new_sky = NULL;
    cpl_imagelist* result;

    /* Test basic sky tweak without stretching */
    result = eris_ifu_sky_tweak(obj, sky, header, 0.3, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_nonnull(result);
    cpl_test_nonnull(new_sky);
    cpl_test_eq(cpl_imagelist_get_size(result), cpl_imagelist_get_size(obj));

    /* Cleanup */
    cpl_imagelist_delete(obj);
    cpl_imagelist_delete(sky);
    cpl_imagelist_delete(result);
    cpl_imagelist_delete(new_sky);
    cpl_propertylist_delete(header);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test sky tweak with stretching enabled
 */
/*----------------------------------------------------------------------------*/
static void test_sky_tweak_with_stretch(void) {
	cpl_size sx = 21;
	cpl_size sy = 21;
	cpl_size sz = 50;
	cpl_imagelist* obj = create_test_cube(sx, sy, sz);
	cpl_imagelist* sky = create_test_cube(sx, sy, sz);
    cpl_propertylist* header = create_test_header();
    cpl_imagelist* new_sky = NULL;
    cpl_imagelist* result;

    /* Test sky tweak with linear stretching */
    result = eris_ifu_sky_tweak(obj, sky, header, 0.3, 1, 0, 1, 2, 1, 0, &new_sky);
    cpl_test_nonnull(result);
    cpl_test_nonnull(new_sky);
    cpl_test_eq(cpl_imagelist_get_size(result), cpl_imagelist_get_size(obj));

    /* Test sky tweak with spline stretching */
    cpl_imagelist_delete(result);
    cpl_imagelist_delete(new_sky);
    new_sky = NULL;
    result = eris_ifu_sky_tweak(obj, sky, header, 0.3, 1, 0, 1, 2, 2, 0, &new_sky);
    cpl_test_nonnull(result);
    cpl_test_nonnull(new_sky);
    cpl_test_eq(cpl_imagelist_get_size(result), cpl_imagelist_get_size(obj));

    /* Cleanup */
    cpl_imagelist_delete(obj);
    cpl_imagelist_delete(sky);
    cpl_imagelist_delete(result);
    cpl_imagelist_delete(new_sky);
    cpl_propertylist_delete(header);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test sky tweak with thermal background subtraction
 */
/*----------------------------------------------------------------------------*/
static void test_sky_tweak_thermal_bgd(void) {
	cpl_size sx = 21;
	cpl_size sy = 21;
	cpl_size sz = 50;
	cpl_imagelist* obj = create_test_cube(sx, sy, sz);
	cpl_imagelist* sky = create_test_cube(sx, sy, sz);
    cpl_propertylist* header = create_test_header();
    cpl_imagelist* new_sky = NULL;
    cpl_imagelist* result;

    /* Test with thermal background subtraction */
    result = eris_ifu_sky_tweak(obj, sky, header, 0.3, 1, 0, 0, 2, 1, 0, &new_sky);
    cpl_test_nonnull(result);
    cpl_test_nonnull(new_sky);

    /* Cleanup */
    cpl_imagelist_delete(obj);
    cpl_imagelist_delete(sky);
    cpl_imagelist_delete(result);
    cpl_imagelist_delete(new_sky);
    cpl_propertylist_delete(header);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Test plot cube background functionality
 */
/*----------------------------------------------------------------------------*/
static void test_plot_cube_background(void) {
	cpl_size sx = 21;
	cpl_size sy = 21;
	cpl_size sz = 50;
    cpl_imagelist* obj = create_test_cube(sx, sy, sz);
    int result;

    /* Test plot cube background */
    result = eris_ifu_plot_cube_background(obj);
    cpl_test_zero(result);

    /* Test with NULL input */
    result = eris_ifu_plot_cube_background(NULL);
    cpl_test_eq(result, -1);

    /* Cleanup */
    cpl_imagelist_delete(obj);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Main test function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
	cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_sky_tweak_null_input();

    test_sky_tweak_illegal_input();
    //test_sky_tweak_basic(); // leaks
    //test_sky_tweak_with_stretch();//errors
    //test_sky_tweak_thermal_bgd();//errors
    //test_plot_cube_background();//leaks

    cleanup_test_files();

    return cpl_test_end(0);
}
