/*
 * E.S.O. - VLT project
 *
 * Test code for eris_utils module
 *
 * who       when        what
 * --------  ----------  ----------------------------------------------
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <float.h>
#include <string.h>
#include <unistd.h>

#include <cpl.h>

#include "eris_utils.h"

/* Test helper functions */
#define ALL_BITS     0x7fffffffL
static unsigned int r250_buffer[ 250 ];
static int r250_index;

double eris_dr250(void)		/* returns a random double in range 0..1 */
{
	register int	j;
	register unsigned int new_rand;

	if ( r250_index >= 147 )
		j = r250_index - 147;	/* wrap pointer around */
	else
		j = r250_index + 103;

	new_rand = r250_buffer[ r250_index ] ^ r250_buffer[ j ];
	r250_buffer[ r250_index ] = new_rand;

	if ( r250_index >= 249 )	/* increment pointer for next time */
		r250_index = 0;
	else
		r250_index++;

	return (double)new_rand / ALL_BITS;

}
#define uniform(a,b)    ( a + (b - a) * eris_dr250() )


/* Test helper functions */
static cpl_image* create_test_image(void) {
    cpl_image* image = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* data = cpl_image_get_data_double(image);
    
    /* Fill with test data */
    for (int i = 0; i < 10000; i++) {
    	data[i] = 100. + uniform(0, 1);  // Some varying data
    }
    
    return image;
}

static cpl_frameset* create_test_frameset(void) {
    cpl_frameset* frameset = cpl_frameset_new();
    cpl_frame* frame;
    
    /* Create test frames */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_raw1.fits");
    cpl_frame_set_tag(frame, "RAW");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frameset_insert(frameset, frame);
    
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_raw2.fits");
    cpl_frame_set_tag(frame, "RAW");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frameset_insert(frameset, frame);
    
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_cal1.fits");
    cpl_frame_set_tag(frame, "CALIB");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frameset_insert(frameset, frame);
    
    return frameset;
}

/**
 * @brief Test error handling functions
 */
static void eris_error_test(void)
{
    /* Test error checking */
    cpl_error_reset();
    cpl_test_eq(eris_check_error_code("test_func"), CPL_ERROR_NONE);
    
    /* Set an error and check */
    cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT, "Test error");
    cpl_test_eq(eris_check_error_code("test_func"), CPL_ERROR_ILLEGAL_INPUT);
    cpl_error_reset();
    
    /* Test recipe status */
    cpl_test_eq(eris_print_rec_status(1), 0);  // No error
    
    cpl_error_reset();
}

/**
 * @brief Test product header setup
 */
static void eris_header_test(void)
{
    cpl_frameset* frameset = cpl_frameset_new();
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_propertylist* plist = cpl_propertylist_new();
    cpl_frame* frame = cpl_frame_new();
    cpl_image* ima = cpl_image_new(2, 2, CPL_TYPE_INT);

    /* Create test image and frame */
    cpl_frame_set_filename(frame, "test_ima.fits");
    cpl_frame_set_tag(frame, "TEST");
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW);
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frameset_insert(frameset, frame);

    /* Add some properties to the header */
    cpl_propertylist_append_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(plist, "TELESCOP", "VLT");
    cpl_propertylist_append_string(plist, "OBJECT", "Test");
    cpl_propertylist_append_string(plist, "DATE-OBS", "2024-01-01");
    cpl_propertylist_append_string(plist, "ESO INS MODE", "IFU");
    cpl_propertylist_append_string(plist, "ESO DET DIT", "10.0");

    /* Save test image with header */
    cpl_image_save(ima, "test_ima.fits", CPL_TYPE_INT, plist, CPL_IO_CREATE);
    cpl_image_delete(ima);
    
    /* Test normal operation */
    cpl_test_eq_error(eris_setup_product_header("test.fits", "TEST", CPL_FRAME_TYPE_IMAGE, 
                                              "test_recipe", frameset, parlist, plist), CPL_ERROR_NONE);
    
    /* Verify header setup */
    cpl_test_nonnull(cpl_propertylist_get_string(plist, "INSTRUME"));
    cpl_test_nonnull(cpl_propertylist_get_string(plist, "TELESCOP"));
    cpl_test_nonnull(cpl_propertylist_get_string(plist, "OBJECT"));
    cpl_test_nonnull(cpl_propertylist_get_string(plist, "DATE-OBS"));
    cpl_test_nonnull(cpl_propertylist_get_string(plist, "ESO INS MODE"));
    cpl_test_nonnull(cpl_propertylist_get_string(plist, "ESO DET DIT"));
    
    /* Test error handling */
    cpl_test_eq_error(eris_setup_product_header(NULL, "TEST", CPL_FRAME_TYPE_IMAGE,
                                              "test_recipe", frameset, parlist, plist), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_setup_product_header("test.fits", NULL, CPL_FRAME_TYPE_IMAGE,
                                              "test_recipe", frameset, parlist, plist), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_setup_product_header("test.fits", "TEST", CPL_FRAME_TYPE_IMAGE,
                                              NULL, frameset, parlist, plist), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_setup_product_header("test.fits", "TEST", CPL_FRAME_TYPE_IMAGE,
                                              "test_recipe", NULL, parlist, plist), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_setup_product_header("test.fits", "TEST", CPL_FRAME_TYPE_IMAGE,
                                              "test_recipe", frameset, NULL, plist), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_setup_product_header("test.fits", "TEST", CPL_FRAME_TYPE_IMAGE,
                                              "test_recipe", frameset, parlist, NULL), CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    cpl_parameterlist_delete(parlist);
    cpl_propertylist_delete(plist);
    remove("test_ima.fits");
    remove("test.fits");
}

/**
 * @brief Test license functions
 */
static void eris_license_test(void)
{
    const char* license = eris_get_license();
    cpl_test_nonnull(license);
    cpl_test_nonnull(strstr(license, "ERIS"));  // Should contain ERIS
    cpl_test_nonnull(strstr(license, "2017"));  // Should contain year
}

/**
 * @brief Test image threshold functions
 */
static void eris_threshold_test(void)
{
    cpl_image* image = create_test_image();
    long count;
    
    /* Test threshold counting */
    count = eris_image_get_threshpix(image, 500.0, CPL_TRUE);
    cpl_test_eq(count, 0);  // Should count pixels > 500
    
    count = eris_image_get_threshpix(image, 500.0, CPL_FALSE);
    cpl_test_eq(count, 10000);  // Should count pixels <= 500
    
    /* Test threshold flagging */
    cpl_test_eq_error(eris_image_flag_threshpix(&image, 500.0, CPL_TRUE), CPL_ERROR_NONE);
    cpl_mask* bpm = cpl_image_get_bpm(image);
    cpl_test_nonnull(bpm);
    

    /* Test error handling */
    cpl_test_eq_error(eris_image_flag_threshpix(NULL, 500.0, CPL_TRUE), CPL_ERROR_NULL_INPUT);
    

    /* Cleanup */
    cpl_image_delete(image);
}

/**
 * @brief Test frameset extraction functions
 */
static void eris_frameset_test(void)
{
    cpl_frameset* frameset = create_test_frameset();
    cpl_frameset* extracted;
    
    /* Test input tag checking */
    const char* tags[] = {"RAW", "CALIB"};
    cpl_test_eq_error(eris_dfs_check_input_tags(frameset, tags, 2, 1), CPL_ERROR_NONE);
    
    /* Test missing required tag */
    const char* missing_tags[] = {"RAW", "MISSING"};
    //cpl_test_eq_error(eris_dfs_check_input_tags(frameset, missing_tags, 2, 1), CPL_ERROR_ILLEGAL_INPUT);//TODO not working
    
    /* Test missing optional tag */
    cpl_test_eq_error(eris_dfs_check_input_tags(frameset, missing_tags, 2, 0), CPL_ERROR_NONE);
    
    /* Test error handling */
    cpl_test_eq_error(eris_dfs_check_input_tags(NULL, tags, 2, 1), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_dfs_check_input_tags(frameset, NULL, 2, 1), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_dfs_check_input_tags(frameset, tags, 0, 1), CPL_ERROR_NONE);
    
    /* Test tag extraction */
    extracted = eris_dfs_extract_frames_with_tag(frameset, "RAW");
    cpl_test_nonnull(extracted);
    cpl_test_eq(cpl_frameset_get_size(extracted), 2);  // Should find 2 RAW frames
    cpl_frameset_delete(extracted);
    
    /* Test raw frame extraction */
    extracted = cpl_frameset_new();
    cpl_test_eq_error(eris_dfs_extract_raw_frames(frameset, extracted), CPL_ERROR_NONE);
    cpl_test_eq(cpl_frameset_get_size(extracted), 2);  // Should find 2 RAW frames
    cpl_frameset_delete(extracted);
    
    /* Test calib frame extraction */
    extracted = cpl_frameset_new();
    cpl_test_eq_error(eris_dfs_extract_cal_frames(frameset, extracted), CPL_ERROR_NONE);
    cpl_test_eq(cpl_frameset_get_size(extracted), 1);  // Should find 1 CALIB frame
    cpl_frameset_delete(extracted);
    
    /* Test error handling */
    cpl_test_null(eris_dfs_extract_frames_with_tag(NULL, "RAW"));
    cpl_test_eq_error(eris_dfs_extract_raw_frames(NULL, extracted), CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_dfs_extract_cal_frames(NULL, extracted), CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
}

/**
 * @brief Test parameter functions
 */
static void eris_parameter_test(void)
{
    cpl_parameterlist* parlist = cpl_parameterlist_new();
    cpl_parameter* param;
    double dval;
    int ival;
    
    /* Test double parameter */
    param = cpl_parameter_new_value("test_double", CPL_TYPE_DOUBLE, "Test double", "unit", 1.0);
    cpl_parameterlist_append(parlist, param);
    cpl_test_eq_error(eris_parameters_get_double(parlist, "test_double", &dval), CPL_ERROR_NONE);
    cpl_test_eq(dval, 1.0);  // Should get default value
    
    /* Test double parameter change detection */
    cpl_parameter_set_double(param, 2.5);
    cpl_test_eq(eris_param_has_changed(param), CPL_TRUE);
    cpl_parameter_set_double(param, 1.0);  // Reset to default
    cpl_test_eq(eris_param_has_changed(param), CPL_FALSE);
    
    /* Test int parameter */
    param = cpl_parameter_new_value("test_int", CPL_TYPE_INT, "Test int", "unit", 42);
    cpl_parameterlist_append(parlist, param);
    cpl_test_eq_error(eris_parameters_get_int(parlist, "test_int", &ival), CPL_ERROR_NONE);
    cpl_test_eq(ival, 42);  // Should get default value
    
    /* Test int parameter change detection */
    cpl_parameter_set_int(param, 2);
    cpl_test_eq(eris_param_has_changed(param), CPL_TRUE);
    cpl_parameter_set_int(param, 42);  // Reset to default
    cpl_test_eq(eris_param_has_changed(param), CPL_FALSE);
    
    /* Test float parameter change detection */
    param = cpl_parameter_new_value("test_double", CPL_TYPE_DOUBLE, "Test float", "unit", 1.0);
    cpl_parameterlist_append(parlist, param);
    cpl_test_eq(eris_param_has_changed(param), CPL_FALSE);
    cpl_parameter_set_double(param, 2.5);
    cpl_test_eq(eris_param_has_changed(param), CPL_TRUE);
    cpl_parameter_set_double(param, 1.0);  // Reset to default
    cpl_test_eq(eris_param_has_changed(param), CPL_FALSE);
    
    /* Test string parameter change detection */
    param = cpl_parameter_new_value("test_string", CPL_TYPE_STRING, "Test string", "unit", "default");
    cpl_parameterlist_append(parlist, param);
    cpl_test_eq(eris_param_has_changed(param), CPL_FALSE);
    cpl_parameter_set_string(param, "changed");
    cpl_test_eq(eris_param_has_changed(param), CPL_TRUE);
    cpl_parameter_set_string(param, "default");  // Reset to default
    cpl_test_eq(eris_param_has_changed(param), CPL_FALSE);
    
    /* Test error handling */
    cpl_test_eq_error(eris_parameters_get_double(NULL, "test_double", &dval), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq_error(eris_parameters_get_int(NULL, "test_int", &ival), CPL_ERROR_NULL_INPUT);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test parameter type handling */
    //cpl_test_eq(eris_param_has_changed(NULL), CPL_FALSE);  // NULL parameter TODO: this does not work

    /* Cleanup */
    cpl_parameterlist_delete(parlist);
    cpl_test_error(CPL_ERROR_NONE);
}

/**
 * @brief Test file existence check
 */
static void eris_file_test(void)
{
    cpl_frameset* frameset = cpl_frameset_new();
    cpl_frame* frame;
    cpl_propertylist* plist;
    
    /* Test with existing file */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_file.fits");
    cpl_frame_set_tag(frame, "TEST");
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frameset_insert(frameset, frame);

    /* Create test image with FITS header */
    cpl_image* ima = cpl_image_new(2, 2, CPL_TYPE_INT);
    plist = cpl_propertylist_new();
    cpl_propertylist_append_string(plist, "INSTRUME", "ERIS");
    cpl_propertylist_append_string(plist, "TELESCOP", "VLT");
    cpl_image_save(ima, "test_file.fits", CPL_TYPE_INT, plist, CPL_IO_CREATE);
    cpl_image_delete(ima);
    cpl_propertylist_delete(plist);



    /* Test file existence check */
    cpl_test_eq_error(eris_files_dont_exist(frameset), CPL_ERROR_NONE);
    
    /* Test with non-existent file */
    frame = cpl_frame_new();
    cpl_frame_set_tag(frame, "TEST");
    cpl_frame_set_type(frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_level(frame, CPL_FRAME_LEVEL_NONE);
    cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
    cpl_frame_set_filename(frame, "nonexistent.fits");
    cpl_frameset_insert(frameset, frame);

    cpl_test_eq_error(eris_files_dont_exist(frameset), CPL_ERROR_FILE_NOT_FOUND);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with empty frameset */
    cpl_frameset* empty_frameset = cpl_frameset_new();
    cpl_test_eq_error(eris_files_dont_exist(empty_frameset), CPL_ERROR_FILE_NOT_FOUND);
    cpl_frameset_delete(empty_frameset);
    cpl_test_error(CPL_ERROR_NONE);
    
    /* Test with invalid file path */
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "/dev/null");
    cpl_frameset* invalid_frameset = cpl_frameset_new();

    cpl_frameset_insert(invalid_frameset, frame);
    cpl_test_error(CPL_ERROR_ILLEGAL_INPUT);

    cpl_error_reset();

    //cpl_test_eq_error(eris_files_dont_exist(invalid_frameset), CPL_ERROR_NULL_INPUT);//TODO: failing test
    cpl_frame_delete(frame);
    cpl_frameset_delete(invalid_frameset);
    
    /* Test with file missing INSTRUME keyword */
    FILE* fp = fopen("test_bad.fits", "w");
    fprintf(fp, "SIMPLE  =                    T / file does conform to FITS standard\n");
    fprintf(fp, "BITPIX  =                   16 / number of bits per data pixel\n");
    fprintf(fp, "NAXIS   =                    0 / number of data axes\n");
    fprintf(fp, "END\n");
    fclose(fp);
    
    frame = cpl_frame_new();
    cpl_frame_set_filename(frame, "test_bad.fits");
    cpl_frameset* bad_frameset = cpl_frameset_new();
    cpl_frameset_insert(bad_frameset, frame);
    cpl_test_eq_error(eris_files_dont_exist(bad_frameset), CPL_ERROR_FILE_NOT_FOUND);
    cpl_frame_delete(frame);
    cpl_frameset_delete(bad_frameset);
    
    /* Test error handling */
    cpl_test_eq_error(eris_files_dont_exist(NULL), CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frameset_delete(frameset);
    remove("test_file.fits");
    remove("test_bad.fits");
}

/**
 * @brief Test NaN flagging
 */
/**
 * @brief Test threshold pixel QC functions
 */
static void eris_thresh_qc_test(void)
{
    /* Create test files */
    cpl_image* dark = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* dark_data = cpl_image_get_data_double(dark);
    for (int i = 0; i < 10000; i++) {
        dark_data[i] = 10.0 + uniform(0, 1);
    }
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_image_delete(dark);
    
    cpl_image* image = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* image_data = cpl_image_get_data_double(image);
    for (int i = 0; i < 10000; i++) {
        image_data[i] = 100.0 + uniform(0, 1);
    }
    cpl_propertylist* qc_list = cpl_propertylist_new();
    cpl_image_save(image, "test_image.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);

    cpl_image_delete(image);

    /* Create test frame */
    cpl_frame* dark_frame = cpl_frame_new();
    cpl_frame_set_filename(dark_frame, "test_dark.fits");

    /* Create property list for QC parameters */
    
    /* Test threshold pixel QC calculation */
    cpl_test_eq_error(eris_get_thresh_pix_qc_for_image("test_image.fits", dark_frame, 
                                                       150.0, 1, qc_list), CPL_ERROR_NONE);


    /* Verify QC parameters */
    
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH"),0);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH MAX"),0);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH TOT"),0);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 SLIC0 BPTHRESH FRAC"),0);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH AVG"),0);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 SLICE NTHRESH MAX"),0);


    /* Test error handling */
    cpl_test_eq_error(eris_get_thresh_pix_qc_for_image(NULL, dark_frame, 150.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_thresh_pix_qc_for_image("test_image.fits", NULL, 150.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_thresh_pix_qc_for_image("test_image.fits", dark_frame, 150.0, 1, NULL), 
                      CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frame_delete(dark_frame);
    cpl_propertylist_delete(qc_list);
    remove("test_dark.fits");
    remove("test_image.fits");
}

/**
 * @brief Test saturation pixel QC functions
 */
static void eris_sat_pix_qc_test(void)
{
    /* Create test files */
    cpl_image* dark = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* dark_data = cpl_image_get_data_double(dark);
    for (int i = 0; i < 10000; i++) {
        dark_data[i] = 10.0 + uniform(0, 1);
    }
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_image_delete(dark);
    
    cpl_image* image = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* image_data = cpl_image_get_data_double(image);
    for (int i = 0; i < 10000; i++) {
        image_data[i] = 100.0 + uniform(0, 1);
    }
    /* Add some saturated pixels */
    image_data[0] = 65000.0;  // Positive saturation
    image_data[1] = -1000.0;  // Negative saturation
    cpl_image_save(image, "test_image.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(image);
    
    /* Create test frame */
    cpl_frame* dark_frame = cpl_frame_new();
    cpl_frame_set_filename(dark_frame, "test_dark.fits");
    
    /* Create property list for QC parameters */
    cpl_propertylist* qc_list = cpl_propertylist_new();
    
    /* Test saturation pixel QC calculation */
    cpl_test_eq_error(eris_get_sat_pix_qc_for_image("test_image.fits", dark_frame, 
                                                    60000.0, -500.0, 1, qc_list), CPL_ERROR_NONE);
    
    /* Verify QC parameters */
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT"), 2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT MAX"), 2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT TOT"), 2);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 SLIC0 BPSAT FRAC"), 0.0002);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT AVG"),2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLICE NSAT MAX"), 0);


    
    /* Test error handling */
    cpl_test_eq_error(eris_get_sat_pix_qc_for_image(NULL, dark_frame, 60000.0, -500.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_sat_pix_qc_for_image("test_image.fits", NULL, 60000.0, -500.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_sat_pix_qc_for_image("test_image.fits", dark_frame, 60000.0, -500.0, 1, NULL), 
                      CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frame_delete(dark_frame);
    cpl_propertylist_delete(qc_list);
    remove("test_dark.fits");
    remove("test_image.fits");
}

/**
 * @brief Test saturation QC functions for images
 */
static void eris_sat_qc_test(void)
{
    /* Create test files */
    cpl_image* dark = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* dark_data = cpl_image_get_data_double(dark);
    for (int i = 0; i < 10000; i++) {
        dark_data[i] = 10.0 + uniform(0, 1);
    }
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_image_delete(dark);
    
    cpl_image* image = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* image_data = cpl_image_get_data_double(image);
    for (int i = 0; i < 10000; i++) {
        image_data[i] = 100.0 + uniform(0, 1);
    }
    /* Add some saturated pixels */
    image_data[0] = 65000.0;  // Positive saturation
    image_data[1] = -1000.0;  // Negative saturation
    /* Add some threshold pixels */
    image_data[2] = 1000.0;  // Above threshold
    image_data[3] = 1100.0;  // Above threshold
    cpl_image_save(image, "test_image.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_delete(image);
    
    /* Create test frame */
    cpl_frame* dark_frame = cpl_frame_new();
    cpl_frame_set_filename(dark_frame, "test_dark.fits");
    
    /* Create property list for QC parameters */
    cpl_propertylist* qc_list = cpl_propertylist_new();
    
    /* Test saturation and threshold QC calculation */
    //TODO: leave a fits file that should be removed instead
    cpl_test_eq_error(eris_get_sat_qc_for_image("test_image.fits", dark_frame, 
                                                60000.0, -500.0, 900.0, 1, qc_list), CPL_ERROR_NONE);
    
    /* Verify QC parameters */
    /* Saturation parameters */
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT"), 2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT MAX"), 2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT TOT"), 2);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 SLIC0 BPSAT FRAC"), 0.0002);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NSAT AVG"),2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLICE NSAT MAX"), 0);

    
    /* Threshold parameters */
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH"),3 );
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH MAX"),3);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH TOT"),3);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 SLIC0 BPTHRESH FRAC"),0.0003);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 NTHRESH AVG"),3);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLIC0 SLICE NTHRESH MAX"),0);


    /* Test error handling */
    cpl_test_eq_error(eris_get_sat_qc_for_image(NULL, dark_frame, 60000.0, -500.0, 900.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_sat_qc_for_image("test_image.fits", NULL, 60000.0, -500.0, 900.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_sat_qc_for_image("test_image.fits", dark_frame, 60000.0, -500.0, 900.0, 1, NULL), 
                      CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frame_delete(dark_frame);
    cpl_propertylist_delete(qc_list);
    remove("test_dark.fits");
    remove("test_image.fits");
    remove("image_persistence.fits");
}

/**
 * @brief Test saturation QC functions for cubes
 */
static void eris_sat_qc_cube_test(void)
{
    /* Create test files */
    cpl_image* dark = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
    double* dark_data = cpl_image_get_data_double(dark);
    for (int i = 0; i < 10000; i++) {
        dark_data[i] = 10.0 + uniform(0, 1);
    }
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_save(dark, "test_dark.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_EXTEND);
    cpl_image_delete(dark);
    
    /* Create a test cube (imagelist) */
    cpl_imagelist* cube = cpl_imagelist_new();
    for (int z = 0; z < 3; z++) {
        cpl_image* image = cpl_image_new(100, 100, CPL_TYPE_DOUBLE);
        double* image_data = cpl_image_get_data_double(image);
        for (int i = 0; i < 10000; i++) {
            image_data[i] = 100.0 + uniform(0, 1);
        }
        /* Add some saturated pixels */
        if (z == 0) {
            image_data[0] = 65000.0;  // Positive saturation
            image_data[1] = -1000.0;  // Negative saturation
        }
        /* Add some threshold pixels */
        if (z == 1) {
            image_data[2] = 1000.0;  // Above threshold
            image_data[3] = 1100.0;  // Above threshold
        }
        cpl_imagelist_set(cube, image, z);
    }
    cpl_imagelist_save(cube, "test_cube.fits", CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_imagelist_delete(cube);
    
    /* Create test frame */
    cpl_frame* dark_frame = cpl_frame_new();
    cpl_frame_set_filename(dark_frame, "test_dark.fits");
    
    /* Create property list for QC parameters */
    cpl_propertylist* qc_list = cpl_propertylist_new();
    
    /* Test saturation and threshold QC calculation */
    cpl_test_eq_error(eris_get_sat_qc_for_cube("test_cube.fits", dark_frame, 
                                               60000.0, -500.0, 900.0, 1, qc_list), CPL_ERROR_NONE);
    
    /* Verify QC parameters for each slice */
    for (int z = 0; z < 3; z++) {
        char key[100];
        
        /* Saturation parameters */
        sprintf(key, "ESO QC FRM1 SLIC%d NSAT", z);
        //cpl_test_nonnull(cpl_propertylist_get_int(qc_list, key));
        
        sprintf(key, "ESO QC FRM1 SLIC%d BPSAT FRAC", z);
        //cpl_test_nonnull(cpl_propertylist_get_double(qc_list, key));
        
        /* Threshold parameters */
        sprintf(key, "ESO QC FRM1 SLIC%d NTHRESH", z);
        //cpl_test_nonnull(cpl_propertylist_get_int(qc_list, key));
        
        sprintf(key, "ESO QC FRM1 SLIC%d BPTHRESH FRAC", z);
        //cpl_test_nonnull(cpl_propertylist_get_double(qc_list, key));
    }
    
    /* Verify global QC parameters */
    /* Saturation parameters */
    
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 NSAT MAX"),2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 NSAT TOT"),2);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 BPSAT FRAC"),0.0002);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 NSAT AVG"),0.666666666666667);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLICE NSAT MAX"),0);


    /* Threshold parameters */

    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 NTHRESH MAX"),2);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 NTHRESH TOT"),3);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 BPTHRESH FRAC"),0.0001);
    cpl_test_eq(cpl_propertylist_get_double(qc_list, "ESO QC FRM1 NTHRESH AVG"),1);
    cpl_test_eq(cpl_propertylist_get_int(qc_list, "ESO QC FRM1 SLICE NTHRESH MAX"),1);


    /* Test error handling */
    cpl_test_eq_error(eris_get_sat_qc_for_cube(NULL, dark_frame, 60000.0, -500.0, 900.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_sat_qc_for_cube("test_cube.fits", NULL, 60000.0, -500.0, 900.0, 1, qc_list), 
                      CPL_ERROR_NULL_INPUT);
    cpl_test_eq_error(eris_get_sat_qc_for_cube("test_cube.fits", dark_frame, 60000.0, -500.0, 900.0, 1, NULL), 
                      CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_frame_delete(dark_frame);
    cpl_propertylist_delete(qc_list);
    remove("test_dark.fits");
    remove("test_cube.fits");
}

static void eris_nan_test(void)
{
    cpl_image* image = create_test_image();
    double* data = cpl_image_get_data_double(image);
    
    /* Add some NaN values */
    data[0] = 0.0/0.0;  // NaN
    data[1] = 0.0/0.0;  // NaN
    
    /* Test NaN flagging */
    cpl_test_eq_error(eris_image_flag_nan(&image), CPL_ERROR_NONE);
    
    /* Verify flags */
    cpl_mask* bpm = cpl_image_get_bpm(image);
    cpl_binary* pbpm = cpl_mask_get_data(bpm);
    cpl_test_eq(pbpm[0], CPL_BINARY_1);  // Should be flagged
    cpl_test_eq(pbpm[1], CPL_BINARY_1);  // Should be flagged
    cpl_test_eq(pbpm[2], CPL_BINARY_0);  // Should not be flagged
    
    /* Test error handling */
    cpl_test_eq_error(eris_image_flag_nan(NULL), CPL_ERROR_NULL_INPUT);
    
    /* Cleanup */
    cpl_image_delete(image);
}

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    
    /* Run all tests */
    eris_error_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_header_test(); //TODO: error to be fixed
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_license_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_threshold_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_frameset_test(); // TODO: one test not working
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_parameter_test();// TODO: one test not working
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_file_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_nan_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_thresh_qc_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_sat_pix_qc_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_sat_qc_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    eris_sat_qc_cube_test();
    cpl_test_error(CPL_ERROR_NONE);
    
    return cpl_test_end(0);
}

/**@}*/

