/* $Id: eris_ifu_recipe.c,v 1.33 2013-03-26 17:00:45 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: erw $
 * $Date: 2013-03-26 17:00:45 $
 * $Revision: 1.33 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <string.h>
#include <strings.h>
#include <stdio.h>
#include "eris_ifu_dfs.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_error.h"
#include "eris_ifu_extract_spec_static.h"

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
#define CONTEXT "eris.eris_ifu_extract_spec"

static const char eris_ifu_extract_spec_description[] = "\
This recipe extract a single spectrum from a masked spatial area of an ERIS/SPIFFIER cube.\n\
\n\
The area mask can be specified in the following ways selected by the \"mask_method\" recipe parameter:\n\
- mask: a user supplied mask in the SOF (tag is MASK)\n\
- position: a user specified circle (\"center\" and \"radius\")\n\
- max: a circle (with \"radius\") around the position of the maximum of the median collapsed cube\n\
- fit: a two dimensional gaussian fit of the median collapsed cube (NOT YET IMPLEMENTED!)\n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
  DO CATG           Explanation                            Required #Frames\n\
  -------           -----------                            -------- -------\n\
  CUBE              ERIS/SPIFFIER cube                        Y         1  \n\
  MASK              spatial mask file                         N       [0,1]\n\
\n\
Output files:\n\
  DO CATG           Explanation                                  Product Depth \n\
  -------           -------------------------------------------  --------------\n\
  SCIENCE.SPECTRUM  extracted spectrum (table)                   PD_SCIENCE (0)\n\
  SPECTRUM          extracted spectrum (vector)                  PD_AUX     (1)\n\
  MASK              generated and used spatial mask              PD_ALL     (2)\n\
  COLLAPSED_CUBE    generated and used collapsed cube image      PD_ALL     (2)\n\
\n\
QC parameters:\n\
None\n\
";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_ifu_extract_spec, ERIS_BINARY_VERSION, "Erich Wiezorrek",
                  "erw@mpe.mpg.de", "2022",
                  "This recipe extracts a spectrum for a data cube",
                  eris_ifu_extract_spec_description);

cpl_error_code eris_ifu_extract_spec_fetch_params(
		const cpl_parameterlist * parlist,
//		struct stdParamStruct *stdParams,
        struct esParamStruct *params);

cpl_error_code eris_ifu_extract_spec_processSof(cpl_frameset* frames,
        struct esSofStruct *sof);

cpl_error_code eris_ifu_extract_spec_save_products(
        cpl_frameset* frames,
        const cpl_parameterlist * parlist,
        struct esSofStruct sof,
        cpl_bivector *spectrum,
        cpl_vector *error,
        cpl_vector *totalFlux,
        productDepthType productDepth);
/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the nonn-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_extract_spec_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_error_code  err     = CPL_ERROR_NONE;
    cpl_parameter   *p      = NULL;
    char            *pName  = NULL;
    char            *context = NULL;

    TRY
    {
        BRK_IF_NULL(
            context = cpl_sprintf("eris.%s", REC_NAME_EXTRACT_SPEC));

/*
        BRK_IF_ERROR(
            eris_ifu_add_std_params(pl, REC_NAME_EXTRACT_SPEC));
*/
        /* Fill the parameters list */
        /* --mask_method */
        BRK_IF_NULL(
            pName = cpl_sprintf("%s.%s", context, "mask_method"));
        BRK_IF_NULL(
            p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
                "Method to specify extraction mask : mask, position, max, fit or optimal",
                context, "optimal"));
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "mask_method");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(pl, p);
        eris_ifu_free_string(&pName);

        /* --centre */
        BRK_IF_NULL(
            pName = cpl_sprintf("%s.%s", context, "center"));
        BRK_IF_NULL(
            p = cpl_parameter_new_value(pName, CPL_TYPE_STRING,
                "The centre of the circular mask (pixel)",
                context, "32,32"));
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "center");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(pl, p);
        eris_ifu_free_string(&pName);

        /* --radius */
        BRK_IF_NULL(
            pName = cpl_sprintf("%s.%s", context, "radius"));
        BRK_IF_NULL(
            p = cpl_parameter_new_value(pName, CPL_TYPE_DOUBLE,
                "The radius of the circular mask (pixel)",
                context, 4.0));
        cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "radius");
        cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
        cpl_parameterlist_append(pl, p);
        eris_ifu_free_string(&pName);

        //        productDepthType pdMin = PD_SCIENCE;
        //        productDepthType pdMax = PD_DEBUG;
        BRK_IF_NULL(
                pName = cpl_sprintf("%s.%s", context, "product_depth"));
        BRK_IF_NULL(
                p = cpl_parameter_new_value(pName, CPL_TYPE_INT,
                    "Specifies the product output depth "
                    "(>0 for auxiliary products)",
                    context, 0));
        BRK_IF_ERROR(
                cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                    "product_depth"));
        BRK_IF_ERROR(
                cpl_parameterlist_append(pl, p));
        eris_ifu_free_string(&pName);



    } CATCH {
        err = cpl_error_get_code();
    }
    eris_ifu_free_string(&context);

    return err;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_extract_spec(cpl_frameset *frameset,
                                 const cpl_parameterlist * parlist)
{
    struct esParamStruct    params;
    struct esSofStruct      sof;
    cpl_propertylist        *pl             = NULL;
    cpl_image               *mask           = NULL,
                            *contribMap     = NULL;
    cpl_bivector            *spectrum       = NULL;
    cpl_vector              *error          = NULL,
                            *totalFlux      = NULL;
    hdrl_image              *collapsedCube  = NULL;
    double                  startLambda     = 0.,
                            deltaLambda     = 0.,
                            refPixel        = 0.;
    cpl_propertylist        *h              = NULL,
                            *h_orig         = NULL;

    TRY
    {
        cpl_msg_info(cpl_func, "Reading recipe parameters");
        BRK_IF_ERROR(
        	eris_ifu_extract_spec_fetch_params(parlist, &params));

        cpl_msg_info(cpl_func, "Reading SOF");
        BRK_IF_ERROR(
            eris_ifu_extract_spec_processSof(frameset, &sof));

        if (params.mask_method == MASK && sof.mask == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                               "missing MASK input in the SOF which is required "
                               "if mask_method is MASK");
        }
        if (params.mask_method != MASK && sof.mask != NULL) {
            cpl_msg_warning(cpl_func,
                            "MASK input in the SOF is ignored because mask_method is not MASK");
        }

        h      = sof.header,
        h_orig = sof.header;
        refPixel = cpl_propertylist_get_double(h, "CRPIX3");
        if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND) {
            RECOVER();
            cpl_frame *frame = cpl_frameset_find(frameset, ERIS_IFU_UTIL_CUBE);
            h = cpl_propertylist_load(cpl_frame_get_filename(frame), 1);
            refPixel = cpl_propertylist_get_double(h, "CRPIX3");
            CHECK_ERROR_STATE();
        }
        startLambda = cpl_propertylist_get_double(h, "CRVAL3");
        if(cpl_propertylist_has(sof.header,"CDELT3")) {
            deltaLambda = cpl_propertylist_get_double(h, "CDELT3");
        } else {
            deltaLambda = cpl_propertylist_get_double(h, "CD3_3");
        }
        startLambda = startLambda - (refPixel - 1.) * deltaLambda;

        if (h_orig != h) {
            eris_ifu_free_propertylist(&h);
        }
        CHECK_ERROR_STATE();


        BRK_IF_NULL(
            collapsedCube = eris_ifu_extract_spec_collapse(sof.cube, &contribMap));
        eris_ifu_free_image(&contribMap);

//            const cpl_parameter *par = cpl_parameterlist_find_const(parlist, "edge-trim");
//            if (par != NULL) {
//                edge_trim = cpl_parameter_get_int(par);
//            }
// agudo: eventually take edge_trim in respect when creating mask?
        BRK_IF_NULL(
            mask = eris_ifu_extract_spec_create_mask(params, sof,
                                                     collapsedCube,
                                                     params.productDepth));

        if (params.productDepth >= PD_ALL) {
            BRK_IF_NULL(
                pl = cpl_propertylist_duplicate(sof.header));

            cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG, "COLLAPSED_IMAGE");
            cpl_image_save(hdrl_image_get_image(collapsedCube),
                           "collapsedCube.fits", CPL_TYPE_FLOAT, pl, CPL_IO_CREATE);

            cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG, "MASK");
            cpl_image_save(mask, "mask.fits", CPL_TYPE_FLOAT, pl, CPL_IO_CREATE);
            eris_ifu_free_propertylist(&pl);
        }

        if (params.mask_method == OPTIMAL) {
            BRK_IF_NULL(
                spectrum = eris_ifu_optimal_extraction(sof.cube,
                                                       sof.qualImagelist,
                                                       mask,
                                                       startLambda, deltaLambda,
                                                       params.productDepth,
                                                       &error));
        } else {

            BRK_IF_NULL(
                spectrum = eris_ifu_extract_spectrum(sof.cube, mask,
                                                     startLambda, deltaLambda,
                                                     &error, &totalFlux));

            eris_ifu_free_image(&mask);
        }

        BRK_IF_ERROR(
            eris_ifu_extract_spec_save_products(frameset, parlist,
                                                sof,
                                                spectrum, error, totalFlux,
                                                params.productDepth));        
    } CATCH {
        CATCH_MSGS();
    }

    eris_ifu_free_vector(&error);
    eris_ifu_free_vector(&totalFlux);
    eris_ifu_free_image(&mask);
    eris_ifu_free_bivector(&spectrum);
    eris_ifu_free_hdrl_image(&collapsedCube);

    return (int)cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    extract some parameters

  @param    parlist    the parameters list
  @param    params     structure holding extracted parameters
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/


cpl_error_code eris_ifu_extract_spec_fetch_params(
		const cpl_parameterlist * parlist,
        struct esParamStruct *params)
{
    TRY
    {
        const char* methodString;
        methodString = cpl_parameter_get_string(
            cpl_parameterlist_find_const(parlist,
                CONTEXT".mask_method"));
        if (strncasecmp(methodString, "mask", strlen("mask")) == 0) {
            params->mask_method = MASK;
        } else if (strncasecmp(methodString, "position", strlen("position")) == 0) {
            params->mask_method = POSITION;
        } else if (strncasecmp(methodString, "max", strlen("max")) == 0) {
            params->mask_method = MAX;
        } else if (strncasecmp(methodString, "fit", strlen("fit")) == 0) {
            params->mask_method = FIT;
        } else if (strncasecmp(methodString, "optimal", strlen("optimal")) == 0) {
            params->mask_method = OPTIMAL;
        } else {
            cpl_msg_error(cpl_func, "The mask_method parameter must be one "
                "of the list: mask, position, max, fit, optimal");
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                "Error reading recipe parameter, unknown mask method %s",
                methodString);

        }

        const char* center;
        center = cpl_parameter_get_string(
            cpl_parameterlist_find_const(parlist,
                CONTEXT".center"));
        int nFields;
        unsigned  int end;
        nFields = sscanf(center, "%d,%d%n",
            &params->center_x,  &params->center_y, &end);
        if (nFields != 2 || end != strlen(center)) {
            cpl_msg_error(cpl_func, "The center parameter must be "
                "a list of two integers separated by a comma");
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                "Error reading recipe parameter, cannot properly read center spec %s",
                center);
            printf("Error reading center string\n");
        }

       params->radius = cpl_parameter_get_double(
           cpl_parameterlist_find_const(parlist, CONTEXT".radius"));

       params->productDepth = cpl_parameter_get_int(
           cpl_parameterlist_find_const(parlist, CONTEXT".product_depth"));

        CHECK_ERROR_STATE();
    } CATCH
	{
	}

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    routine to process frames

  @param    frames    frames to be processed
  @param    params     structure holding extracted parameters
  @param    sof        structure containing input frames
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

cpl_error_code eris_ifu_extract_spec_processSof(
        cpl_frameset* frames,
//        struct stdParamStruct stdParams,
//        struct esParamStruct params,
        struct esSofStruct *sof)
{
	cpl_frame           *frame = NULL;

    TRY
    {
        if (frames == NULL) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "missing frameset");
        }
        if (cpl_frameset_is_empty(frames)) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "SOF file is empty or missing");
        }
        CHECK_ERROR_STATE();

        BRK_IF_ERROR(
            eris_ifu_dfs_set_groups(frames));

        // get data cube
        frame = cpl_frameset_find(frames, ERIS_IFU_UTIL_CUBE);
        CHECK_ERROR_STATE();
        if (frame != NULL) {
            BRK_IF_NULL(
                sof->cube = eris_ifu_load_deq_hdrl_imagelist(
                    cpl_frame_get_filename(frame),
                    &sof->header, &sof->qualImagelist, &sof->qualityType));
            sof->nx = hdrl_imagelist_get_size_x(sof->cube);
            sof->ny = hdrl_imagelist_get_size_y(sof->cube);
            sof->nz = hdrl_imagelist_get_size(sof->cube);
        } else {
            BRK_WITH_ERROR_MSG(CPL_ERROR_NULL_INPUT,
                "missing \"%s\" tag in the SOF, input data cube",
                ERIS_IFU_UTIL_CUBE);

        }

        // get mask (optional
        sof->mask = NULL;
        frame = cpl_frameset_find(frames, ERIS_IFU_UTIL_MASK);
        CHECK_ERROR_STATE();
        if (frame != NULL) {
            BRK_IF_NULL(
                sof->mask = cpl_image_load(
                    cpl_frame_get_filename(frame),
                    CPL_TYPE_DOUBLE, 0, 0));
        }

    } CATCH
	{
//    	CATCH_MSGS();
	}

    return cpl_error_get_code();

}


/*----------------------------------------------------------------------------*/
/**
  @brief    routine to save products from extraction

  @param    frames    frames to be processed
  @param    parlist   parameter list
  @param    sof        structure containing input frames
  @param    spectrum   spectrum (average)
  @param    error      error associated to spectrum
  @param    totalFlux  spectrum (total flux)
  @param    productDepth product depth parameter


  @return   CPL_ERROR_NONE if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_extract_spec_save_products(
        cpl_frameset* frames,
        const cpl_parameterlist * parlist,
        struct esSofStruct sof,
        cpl_bivector *spectrum,
        cpl_vector *error,
        cpl_vector *totalFlux,
        productDepthType productDepth)
{
    cpl_table           *productTable = NULL;
    cpl_propertylist    *pl;
    cpl_vector          *lambda = NULL;
    cpl_vector          *data = NULL;
    const cpl_array           *waveArray = NULL;
    const cpl_array           *fluxArray = NULL;
    const cpl_array           *errArray = NULL;
    const cpl_array           *totalFluxArray = NULL;
    double              firstLambda;
    double              deltaLambda;
    cpl_size            spectrumSize;

    TRY
    {
        spectrumSize = cpl_bivector_get_size(spectrum);
        BRK_IF_NULL(lambda = cpl_bivector_get_x(spectrum));
        BRK_IF_NULL(data = cpl_bivector_get_y(spectrum));
        firstLambda = cpl_vector_get(lambda,0);
        deltaLambda = cpl_vector_get(lambda,1) - firstLambda;

        if (productDepth >= PD_AUXILLIARY) {
            BRK_IF_NULL(pl = cpl_propertylist_duplicate(sof.header));
            cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG, "SPECTRUM");
            cpl_propertylist_update_string(pl, "CTYPE1", "WAVE");
            cpl_propertylist_update_double(pl, "CRPIX1", 1.);
            cpl_propertylist_update_double(pl, "CRVAL1", firstLambda);
            cpl_propertylist_update_double(pl, "CDELT1", deltaLambda);
            cpl_propertylist_erase(pl, "CTYPE2");
            cpl_propertylist_erase(pl, "CRPIX2");
            cpl_propertylist_erase(pl, "CRVAL2");
            cpl_propertylist_erase(pl, "CDELT2");
            cpl_propertylist_erase(pl, "CTYPE3");
            cpl_propertylist_erase(pl, "CRPIX3");
            cpl_propertylist_erase(pl, "CRVAL3");
            cpl_propertylist_erase(pl, "CDELT3");
            cpl_propertylist_update_double(pl, "CD1_1", 0.);
            cpl_propertylist_update_double(pl, "CD1_2", deltaLambda);
            cpl_propertylist_erase(pl, "CD1_3");
            cpl_propertylist_erase(pl, "CD2_1");
            cpl_propertylist_erase(pl, "CD2_2");
            cpl_propertylist_erase(pl, "CD2_3");
            cpl_propertylist_erase(pl, "CD3_1");
            cpl_propertylist_erase(pl, "CD3_2");
            cpl_propertylist_erase(pl, "CD3_3");

            CHECK_ERROR_STATE();
            BRK_IF_ERROR(
                cpl_vector_save(data, "spectrum_vector.fits",
                    CPL_TYPE_DOUBLE, pl, CPL_IO_CREATE));
            cpl_propertylist_delete(pl);
        }

        waveArray = cpl_array_wrap_double(cpl_vector_get_data(lambda), spectrumSize);
        fluxArray = cpl_array_wrap_double(cpl_vector_get_data(data), spectrumSize);
        errArray = cpl_array_wrap_double(cpl_vector_get_data(error), spectrumSize);
        if (totalFlux != NULL) {
            totalFluxArray = cpl_array_wrap_double(cpl_vector_get_data(totalFlux),
                                                   spectrumSize);
        }
        BRK_IF_NULL(pl = cpl_propertylist_duplicate(sof.header));
        cpl_propertylist_update_string(pl, CPL_DFS_PRO_CATG, "SCIENCE.SPECTRUM");
        BRK_IF_NULL(productTable = cpl_table_new(1));
        cpl_table_new_column_array(productTable, "WAVE",
                CPL_TYPE_DOUBLE, spectrumSize);
        if (totalFlux != NULL) {
            cpl_table_new_column_array(productTable, "FLUX",
                    CPL_TYPE_DOUBLE, spectrumSize);
        } else {
            cpl_table_new_column_array(productTable, "TOT_FLUX",
                                       CPL_TYPE_DOUBLE, spectrumSize);
        }
        cpl_table_new_column_array(productTable, "ERR",
                CPL_TYPE_DOUBLE, spectrumSize);
        if (totalFlux != NULL) {
            cpl_table_new_column_array(productTable, "TOT_FLUX",
                                        CPL_TYPE_DOUBLE, spectrumSize);
        }
        cpl_table_set_array(productTable, "WAVE", 0, waveArray);
        if (totalFlux != NULL) {
            cpl_table_set_array(productTable, "FLUX", 0, fluxArray);
        } else {
            cpl_table_set_array(productTable, "TOT_FLUX", 0, fluxArray);
        }
        cpl_table_set_array(productTable, "ERR",  0, errArray);
        if (totalFlux != NULL) {
            cpl_table_set_array(productTable, "TOT_FLUX",  0, totalFluxArray);
        }
        //cpl_table_save(productTable, pl, NULL, "spectrum.fits", CPL_IO_CREATE);

        cpl_dfs_save_table(frames, NULL, parlist, frames, NULL,
                productTable, NULL, REC_NAME_EXTRACT_SPEC, pl,
                NULL, PACKAGE "/" PACKAGE_VERSION, "spectrum.fits");

        cpl_propertylist_delete(pl);
        CHECK_ERROR_STATE();

    } CATCH{
    }

    return cpl_error_get_code();
}


