/* $Id: eris_ifu_recipe.c,v 1.33 2013-03-26 17:00:45 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:00:45 $
 * $Revision: 1.33 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include "eris_ifu_error.h"
#include <eris_ifu_jitter_interface.h>
#include <eris_ifu_jitter_static.h>
#include <eris_ifu_lambda_corr.h>
#include <eris_ifu_resample.h>
#include <eris_ifu_extract_spec_static.h>
#include <eris_ifu_efficiency_response.h>
#include <eris_utils.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
#define CONTEXT "eris.eris_ifu_jitter"

static const char eris_ifu_jitter_description[] = "\
This recipe performs ERIS/SPIFFIER science data reduction.\n\
\n\
When an OH_SPEC file is provided in the SOF the recipe will use the OH lines\n\
to correct the lambda calibration.\n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
   DO CATG              Explanation                          Required #Frames\n\
   -------              -----------                          -------- -------\n\
 science exposures with corresponding sky exposures:                         \n\
   OBJ                  Science object exposure                 Y         ?  \n\
   SKY_OBJ              Sky exposure                            Y         ?  \n\
 or                                                                          \n\
    STD                 Standard star exposure                  Y         ?  \n\
    SKY_STD             Standard star sky exposure              Y         ?  \n\
 or                                                                          \n\
    PSF_CALIBRATOR      PSF calibration source                  Y         ?  \n\
    SKY_PSF_CALIBRATOR  PSF calibration sky exposure            Y         ?  \n\
                                                                             \n\
 and calibration frames:                                                     \n\
    DISTORTION          Table with distortion correction pars.  Y         1  \n\
    WAVE_MAP            Wavelength calibration map              Y         1  \n\
    OH_SPEC             Vector holding OH lines (note 4)        Y         1  \n\
    MASTER_DARK         Optional master dark image              N       [0,1]\n\
    MASTER_FLAT         Optional master flat image              N       [0,1]\n\
    BPM_DARK            Optional bad pixel mask (note 1 & 3)    N       [0,1]\n\
    BPM_FLAT            Optional bad pixel mask (note 2 & 3)    N       [0,1]\n\
    BPM_LINEARITY       Optional bad pixel mask (note 3)        N       [0,1]\n\
	EXTCOEFF_TABLE      Table with atmospheric extinction       N       [0,1]\n\
    FLUX_STD_CATALOG    Table with ref flux std star spectra    N       [0,1]\n\
    EFFICIENCY_WINDOWS  Table with ranges to compute efficiency N       [0,1]\n\
	FIT_AREAS           Table with ranges to perform fit        N       [0,1]\n\
	HIGH_ABS_REGIONS    Table with listed High Abs. Regions     N       [0,1]\n\
	QUALITY_AREAS       Table with quality regions area         N       [0,1]\n\
	RESP_FIT_POINTS_CATALOG Table with points where to fit resp N       [0,1]\n\
	RESPONSE_WINDOWS    Table with regions for response computation N   [0,1]\n\
	TELL_MOD_CATALOG    Catalog with tellurics models           N       [0,1]\n\
\n\
 and only in case of old SINFONIE way of distortion correction:\n\
    DISTANCES           Table with slitlet distances                         \n\
                           with cube.slitlet-detection=\"DIST\" Y         1  \n\
    SLITLET_POS         Table with slitlet edge positions                    \n\
                           with cube.slitlet-detection=\"EDGE\" Y         1  \n\
\n\
Output files:\n\
   DO CATG          Explanation                                  Product Depth\n\
   -------          -------------------------------------------  -------------\n\
   OBJECT_CUBE      reconstructed object expoure (note 5)              PD_AUX\n\
   SKY_CUBE         reconstructed sky exposure (note 6)                PD_AUX\n\
\n\
Note 1) When BPM_DARK is provided this will be used as hot bad pixel mask.\n\
   Otherwise the qualiiy extension of MASTER_DARK will be used.\n\
Note 2 When BPM_FLAT is provided this will be used as cold bad pixel mask.\n\
   Otherwise the qualiiy extension of MASTER_FLAT will be used.\n\
Note 3) All BPMs will be ORed.\n\
Note 4) OH_SPEC can be a FITS file holding a single OH spectrum in the primary\n\
   HDU or a FITS file with several extensions each of them holding an OH spectrum\n\
   for one of the bands J,H,K,HK.\n\
   The pipeline package provides the file eris_oh_spec.fits holding OH spectra\n\
   for all bands.\n\
Note 5) If SKY_TWEAK is selected no image will be subtracted, else if there is\n\
   a matching sky exposure this exposure will be subtracted. Else if a masterDark\n\
   image is available this image will be subtracted\n\
Note 6) If SKY_TWEAK is selected no image will be subtracted. Else if a masterDark\n\
   image is available this image will be subtracted\n\
\n\
Information on relevant parameters may be found with\n\
  esorex --params "REC_NAME_JITTER"\n\
  esorex --help   "REC_NAME_JITTER"\n\
\n\
QC parameters:\n\
ESO QC LAMBDA SHIFT UM      OH based shift of the central wavelength in [um]\n\
ESO QC LAMBDA SHIFT PIXEL   OH based shift of the central wavelength in [pixel]\n\
";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_ifu_jitter, ERIS_BINARY_VERSION, "Erich Wiezorrek",
                  PACKAGE_BUGREPORT, "2019",
                  "This recipe reconstruct data cubes from object exposures",
                  eris_ifu_jitter_description);

#define RECIPE_NAME "eris_ifu_jitter"


/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the nonn-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_jitter_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_error_code  err     = CPL_ERROR_NONE;

    TRY
	{
            eris_ifu_jitter_fill_common_parameterlist(
                REC_NAME_JITTER, M_SCIENCE, pl);
    } CATCH {
        err = cpl_error_get_code();
    }

    return err;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    generate a (dummy) sky frame from an object raw frame
  @param    obj_set frameset with input object frames
  @param    method paramter to select which method to use to create the sky
  @return   cpl_frame* frame describing the sky model
 */
/*----------------------------------------------------------------------------*/
static cpl_frame*
eris_crea_dummy_sky_frame(cpl_frameset* obj_set, const int method)
{

    cpl_imagelist* obj_list=NULL;
    cpl_image* fake_sky=NULL;
    cpl_frame* frame=NULL;
    cpl_frame* sky_frame=NULL;

    cpl_propertylist* plist=NULL;
    const char* filename;

    obj_list = cpl_imagelist_load_frameset(obj_set,CPL_TYPE_DOUBLE,0,0);
    if(method == 2) {
    	fake_sky = cpl_imagelist_collapse_median_create(obj_list);
    } else if(method == 3) {
    	fake_sky = cpl_imagelist_collapse_create(obj_list);
    }
    frame = cpl_frameset_get_position(obj_set,0);
    filename = cpl_frame_get_filename(frame);

    plist = cpl_propertylist_load(filename, 0);

    if (cpl_propertylist_has(plist, FHDR_DPR_TYPE)) {
        cpl_propertylist_set_string(plist, FHDR_DPR_TYPE, "SKY");
    } else {
        cpl_propertylist_append_string(plist, FHDR_DPR_TYPE,"SKY") ;
    }

    cpl_image_save(fake_sky, "out_fake_sky.fits", CPL_BPP_IEEE_FLOAT,
                    plist,CPL_IO_DEFAULT);

    cpl_propertylist_delete(plist);

    sky_frame = cpl_frame_new();
    cpl_frame_set_filename(sky_frame, "out_fake_sky.fits") ;
    cpl_frame_set_tag(sky_frame, "SKY") ;
    cpl_frame_set_type(sky_frame, CPL_FRAME_TYPE_IMAGE);
    cpl_frame_set_group(sky_frame, CPL_FRAME_GROUP_RAW);

    cpl_frame_set_level(sky_frame, CPL_FRAME_LEVEL_FINAL);
    cpl_image_delete(fake_sky);
    cpl_imagelist_delete(obj_list);

    eris_check_error_code("eris_crea_dummy_sky_frame");
    return sky_frame;

}
/*----------------------------------------------------------------------------*/
/**
  @brief    generate a sky frame from a set of object raw frames
  @param    raw frameset containing input raw data
  @param    i index of raw frame in input frameset
  @param    mjd_obj MJD-OBS of input object frame
  @param    obj_cumx value of input OBJECT COMOFFSETX
  @param    obj_cumy value of input OBJECT COMOFFSETY
  @return   cpl_frame* frame describing the sky model
 */
/*----------------------------------------------------------------------------*/

static cpl_frame*
eris_get_sky_frame(cpl_frameset* raws, const cpl_size i, const double mjd_obj,
		const double obj_cumx, const double obj_cumy)
{


	cpl_frame* sky_frm_inf = NULL;
	cpl_frame* sky_frm_sup = NULL;
	cpl_frame* sky_frm = NULL;
	double mjd_sky_inf = 0;
	double mjd_sky_sup = 0;
	double sky_inf_cumx = 0;
	double sky_inf_cumy = 0;
	double sky_sup_cumx = 0;
	double sky_sup_cumy = 0;

	double cum_thres = 0.001;
	char* sky_name;
	char* fake_sky_name;
	cpl_image* sky_ima;
	cpl_propertylist* plist;

    cpl_size nobj = cpl_frameset_get_size(raws);

	if(i > 0) {
		sky_frm_inf = cpl_frameset_get_position(raws, i-1);
		mjd_sky_inf = eris_get_mjd_obs(sky_frm_inf);
		sky_inf_cumx = eris_get_cumoffs_ra(sky_frm_inf);
		sky_inf_cumy = eris_get_cumoffs_dec(sky_frm_inf);
	}

	if(i < nobj-1) {
		sky_frm_sup = cpl_frameset_get_position(raws, i+1);
		mjd_sky_sup = eris_get_mjd_obs(sky_frm_sup);
		sky_sup_cumx = eris_get_cumoffs_ra(sky_frm_sup);
		sky_sup_cumy = eris_get_cumoffs_dec(sky_frm_sup);
	}

	if(i == 0) {
		sky_frm = sky_frm_sup;
	}
	else if(i == (nobj-1)) {
		sky_frm = sky_frm_inf;
	} else {
		if( fabs( mjd_sky_inf - mjd_obj ) <
			fabs( mjd_sky_sup - mjd_obj ) ) {
			if((fabs(sky_inf_cumx - obj_cumx) > cum_thres) ||
			   (fabs(sky_inf_cumy - obj_cumy) > cum_thres)) {
				sky_frm = sky_frm_inf;
			} else {
				sky_frm = sky_frm_sup;
			}
		} else {
			if((fabs(sky_sup_cumx - obj_cumx) > cum_thres) ||
			   (fabs(sky_sup_cumy - obj_cumy) > cum_thres)) {
				sky_frm = sky_frm_sup;
			} else {
				sky_frm = sky_frm_inf;
			}
		}
	}

	sky_name = cpl_sprintf("%s", cpl_frame_get_filename(sky_frm));

	//cpl_msg_info(cpl_func, "obj: %s", cpl_frame_get_filename(obj_frm));
	//cpl_msg_info(cpl_func, "sky: %s", sky_name);

	if (strstr(sky_name, "." ) != NULL ) {
		fake_sky_name = cpl_sprintf("%s%lld%s", "out_fake_sky", i, ".fits");
	} else {
		fake_sky_name = cpl_sprintf("%s", sky_name) ;
	}

	sky_ima = cpl_image_load(sky_name, CPL_TYPE_DOUBLE,0,0);
	plist = cpl_propertylist_load(sky_name, 0);
	if( NULL == plist){
		cpl_msg_error(cpl_func,
				"getting header from reference ima frame %s",sky_name);
	}


	if (cpl_propertylist_has(plist, FHDR_DPR_TYPE)) {
		cpl_propertylist_set_string(plist, FHDR_DPR_TYPE, "SKY");
	} else {
		cpl_propertylist_append_string(plist, FHDR_DPR_TYPE,"SKY") ;
	}
	double mjd_obs_eps = 0.0001;
	if (cpl_propertylist_has(plist, FHDR_MJD_OBS)) {
		cpl_propertylist_set_double(plist, FHDR_MJD_OBS, (mjd_obj+mjd_obs_eps));
	} else {
		cpl_propertylist_append_double(plist, FHDR_MJD_OBS, (mjd_obj+mjd_obs_eps)) ;
	}

	cpl_image_save(sky_ima, fake_sky_name, CPL_BPP_IEEE_FLOAT, plist,
			CPL_IO_DEFAULT);

	cpl_propertylist_delete(plist);
	cpl_image_delete(sky_ima);

	cpl_frame_set_filename(sky_frm, fake_sky_name);
	cpl_frame_set_tag(sky_frm, ERIS_IFU_RAW_SKY) ;
	cpl_frame_set_type(sky_frm, CPL_FRAME_TYPE_IMAGE);
	cpl_frame_set_group(sky_frm, CPL_FRAME_GROUP_RAW);
	cpl_frame_set_level(sky_frm, CPL_FRAME_LEVEL_FINAL);

	cpl_free(fake_sky_name);
	cpl_free(sky_name);
	eris_check_error_code("eris_get_sky_frame");
	return sky_frm;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    generate a object and sky frame pairs
  @param    frameset frameset containing input raw data
  @param    method  set the algorithm to create the synthetic sky frame
  @param    crea_fake_sky create the fake sky frame
  @return   cpl_error_code error code value
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code
eris_create_obj_sky_pairs(cpl_frameset* frameset, const int method,
		cpl_boolean * crea_fake_sky){

	/* generate artificial obj-sky pair */

	cpl_frame* obj_frm = NULL;

	cpl_frame* sky_frm = NULL;

	double mjd_obj = 0;
    double obj_cumx = 0;
	double obj_cumy = 0;
	cpl_frame* dup_frm;

	cpl_frameset* raws = cpl_frameset_new();
	eris_dfs_extract_raw_frames (frameset, raws);
	cpl_size nobj = cpl_frameset_get_size(raws);
	if (nobj > 1) {
		if(method == 1) {
			/* get the sky frame from the next in mjd-obs */
			cpl_frameset* sky_set = cpl_frameset_new();
			for( cpl_size i = 0; i < nobj; i++) {

				obj_frm = cpl_frameset_get_position(raws, i);
				mjd_obj = eris_get_mjd_obs(obj_frm);
				obj_cumx = eris_get_cumoffs_ra(obj_frm);
				obj_cumy = eris_get_cumoffs_dec(obj_frm);

				sky_frm = eris_get_sky_frame(raws, i, mjd_obj, obj_cumx, obj_cumy);

				cpl_msg_info(cpl_func, "obj: %s %s", cpl_frame_get_filename(obj_frm), cpl_frame_get_tag(obj_frm));
				cpl_msg_info(cpl_func, "sky: %s %s", cpl_frame_get_filename(sky_frm), cpl_frame_get_tag(sky_frm));

				dup_frm = cpl_frame_duplicate(sky_frm);
				cpl_frameset_insert(sky_set, dup_frm);

			}
			//cpl_frameset_dump(sky_set,stdout);
			//cpl_frameset_dump(frameset,stdout);
			cpl_size nsky = cpl_frameset_get_size(sky_set);
			for(cpl_size i = 0; i < nsky; i++) {
				sky_frm = cpl_frameset_get_position(sky_set, i);
				cpl_frameset_insert(frameset, cpl_frame_duplicate(sky_frm));
			}
			cpl_frameset_delete(sky_set);
			*crea_fake_sky = CPL_TRUE;
		} else if(method > 1 && method < 4) {
			/* get the sky frame from the mean of all sky frames */
			sky_frm = eris_crea_dummy_sky_frame(raws, method);
			cpl_frameset_insert (frameset, cpl_frame_duplicate(sky_frm));
			*crea_fake_sky = CPL_TRUE;
		} else if(method == 4) {
			/* take sky from a single user specified rectangular region */
		} else if(method == 5) {
			/* take sky from the median of 4 rectangular regions at FOV edges */
		} else if(method == 6) {
			/* take sky from the median of 9 rectangular regions at FOV edges */
		} else if(method == 7) {
			/* take sky from each cube slice based on a mask computed on cube
			 * collapsed mean */
		} else if(method == 8) {
				/* take sky from each cube slice based on a %25 min flux from
				 * cube collapsed mean */

		} else {
			cpl_msg_error(cpl_func,"aj-method: %d not supported. Supported are: [0,3]", method);
		}
	}

	//cpl_frameset_dump(frameset,stdout);
	cpl_frameset_delete(raws);
    eris_check_error_code("eris_create_obj_sky_pairs");
	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_jitter(cpl_frameset *frameset,
        const cpl_parameterlist * parlist)
{
    struct stdParamStruct stdParams = stdParamStructInit;
    struct paramStruct params;
    struct sofStruct sof;
    (void)memset(&sof, 0, sizeof(sof)); /* Rough init so free works on error */
    cpl_boolean crea_fake_sky = CPL_FALSE;

    /* check required input tags are present */
     const int ntags = 1;
     const char* required_tags[1] = {
     		ERIS_IFU_CALIB_OH_SPEC
     };

     cpl_ensure_code(CPL_ERROR_NONE ==
            eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
                CPL_ERROR_ILLEGAL_INPUT);

    TRY
    {
        cpl_msg_info(cpl_func, "Reading recipe parameters");
        	eris_ifu_jitter_fetch_params(CONTEXT, RECIPE_NAME, parlist,
        			&stdParams, &params);

        cpl_msg_info(cpl_func,
                "Instrument is %d, requested product level is %d",
                stdParams.instrument, stdParams.productDepth);
        eris_ifu_dfs_set_groups(frameset);
        char* param_name = cpl_sprintf("%s.aj-method", CONTEXT);
        const int aj_method = cpl_parameter_get_int(
                     cpl_parameterlist_find_const(parlist, param_name));
        cpl_free(param_name);
        
        if(
        		(NULL == cpl_frameset_find(frameset, ERIS_IFU_RAW_SKY)) &&
				(NULL == cpl_frameset_find(frameset, ERIS_IFU_RAW_OBJ_SKY))
        )
        {
        	if(aj_method != 0) {
             	eris_create_obj_sky_pairs(frameset, aj_method, &crea_fake_sky);
        	}
        }
        //cpl_frameset_dump(frameset,stdout);
        BRK_IF_ERROR(
            eris_ifu_jitter_processSof(frameset, stdParams, params, &sof));

        cpl_msg_info(cpl_func,"Instrument %s, band %s, scale %s",
                eris_ifu_get_instrumentString(sof.instrument),
                eris_ifu_get_bandString(sof.band),
                eris_ifu_get_preopticsScaleString(sof.scale));

        if (sof.mode != SCIENCE) {
            BRK_WITH_ERROR_MSG(CPL_ERROR_ILLEGAL_INPUT,
                "Please use the eris_ifu_stdstar recipe for PSF, STD STAR ...");
        }

        BRK_IF_ERROR(
            eris_ifu_jitter_process_exposures(&sof, stdParams, params,
                                              frameset, parlist, RECIPE_NAME, CONTEXT));

        cubeType obj_type = OBJECT_CUBE;
        cubeType resampled_obj_type = OBJECT_CUBE_COADD;
        const char *proCatg;
        cpl_boolean is_pupil_tracking = CPL_FALSE;
        BRK_IF_ERROR(
            eris_ifu_jitter_process_cubes(&sof, stdParams, params, frameset,
                    parlist, RECIPE_NAME, &obj_type));



        if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_JITTER_DAR_CUBE)
                > 0) {
            obj_type = DAR_CUBE;
            resampled_obj_type = DAR_CUBE_COADD;
            proCatg = ERIS_IFU_PRO_JITTER_DAR_CUBE;
        } else if (cpl_frameset_count_tags(frameset, ERIS_IFU_PRO_JITTER_TWK_CUBE)
                > 0 ) {
            obj_type = TWEAKED_CUBE;
            resampled_obj_type = TWEAKED_CUBE_COADD;
            proCatg = ERIS_IFU_PRO_JITTER_TWK_CUBE;
        } else {
            obj_type = OBJECT_CUBE;
            resampled_obj_type = OBJECT_CUBE_COADD;
            proCatg = ERIS_IFU_PRO_JITTER_OBJ_CUBE;
       }

        /* check if Pupil Tracking */
        cpl_frame* frm_obj = cpl_frameset_find(frameset,ERIS_IFU_RAW_OBJ);
        const char* fname = cpl_frame_get_filename(frm_obj);
        cpl_propertylist* plist = cpl_propertylist_load(fname,0);
        const char* dpr_tech = cpl_propertylist_get_string(plist,FHDR_DPR_TECH);
        if(strstr(dpr_tech,"IFU,NODDING,PT") != NULL) {
        	is_pupil_tracking = CPL_TRUE;
        }
        cpl_propertylist_delete(plist);


        char *combDoCatg =  NULL;
        char   *filenameSpec = NULL;
        eris_ifu_jitter_get_procatg_and_filename(obj_type, &combDoCatg, &filenameSpec);

        /* combine cubes with HDRL functionality, create mean and median cube */
        if (params.combine && sof.objectCnt > 1) {
        	if(!is_pupil_tracking){
        		cpl_msg_warning(cpl_func,"Combines individual cubes with proCatg: %s",proCatg);
        		if( CPL_ERROR_NONE == eris_ifu_combine_pbp(frameset, parlist,
        				combDoCatg, filenameSpec,
						NULL, NULL, NULL,
						RECIPE_NAME, RECIPE_NAME)) {

        			/* extract spectra and for STD star computes response and efficiency */
        			if (params.extractSource && eris_can_extract(frameset)) {
        				eris_ifu_jitter_extract(frameset, parlist, resampled_obj_type,
        						proCatg, stdParams, RECIPE_NAME, CONTEXT);
        			}
        		} else {
        			cpl_error_set(cpl_func, CPL_ERROR_NONE);
        		}
        	}  else {
        		cpl_msg_warning(cpl_func,"pupil tracking: no cube combination (no extraction)");
        	}
        } else { // no resampling
        	cpl_msg_warning(cpl_func,"Duplicate cube with proCatg: %s",proCatg);
            eris_frameset_duplicate_cube_tag(frameset, proCatg, CPL_TRUE);

            eris_ifu_cube_collapse_mean_and_save(proCatg, frameset, parlist,
                    RECIPE_NAME, /*filenameSpec,*/ CPL_TRUE, CPL_FALSE);
            /* extract spectra and for STD star computes response and efficiency */
             if (params.extractSource && eris_can_extract(frameset)) {
                 eris_ifu_jitter_extract(frameset, parlist, obj_type,
                		 proCatg, stdParams, RECIPE_NAME, CONTEXT);
             }
		}
        eris_ifu_free_string(&combDoCatg);
        eris_ifu_free_string(&filenameSpec);

        param_name = cpl_sprintf("%s.flux-calibrate", CONTEXT);
        cpl_boolean flux_calibrate = cpl_parameter_get_bool(
        				cpl_parameterlist_find_const(parlist, param_name));
        cpl_free(param_name);

        param_name = cpl_sprintf("%s.extract-source", CONTEXT);
        cpl_boolean extract_source = cpl_parameter_get_bool(
        		cpl_parameterlist_find_const(parlist, param_name));
        cpl_free(param_name);


        if(flux_calibrate && eris_can_flux_calibrate(frameset)) {

        	if(extract_source) {
        		cpl_msg_info(cpl_func,"Flux calibrate extracted spectrum");
        		eris_flux_calibrate_spectra(RECIPE_NAME, RECIPE_NAME, parlist,
        				frameset, frameset);
        	}
        	char* cube_pro_catg = NULL;
        	if(obj_type == DAR_CUBE) {
        		cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_OBJ_DAR_CUBE_COADD);
        	} else if(obj_type == TWEAKED_CUBE) {
        		cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_TWK_CUBE_COADD);
        	} else {
        		cube_pro_catg = cpl_sprintf("%s",ERIS_IFU_PRO_JITTER_OBJ_CUBE_COADD);
        	}


        	cpl_msg_info(cpl_func,"proCatg: %s",cube_pro_catg);
        	//cpl_frameset_dump(frameset, stdout);
        	cpl_msg_info(cpl_func,"Flux calibrate combined data cube");
        	eris_flux_calibrate_cube2(cube_pro_catg, RECIPE_NAME, RECIPE_NAME, parlist,
                    /*frameset, */frameset);
        	cpl_free(cube_pro_catg);
        	
        }
    }
    CATCH
 	{
        CATCH_MSGS();
 	}

    eris_ifu_jitter_free_sofStruct(&sof);
    eris_ifu_free_std_param(&stdParams);
    if(crea_fake_sky == CPL_TRUE) {
    	char* cmd;
    	cmd = cpl_sprintf("rm out_fake_sky*.fits");
    	int status = system(cmd);
	if(status == -1) {
		cpl_msg_warning(cpl_func,"call to system() failed");
	}
    	cpl_free(cmd);

    }
    eris_check_error_code("eris_ifu_jitter");
    return (int)cpl_error_get_code();
}



