/* $Id: eris_ifu_recipe.c,v 1.33 2013-03-26 17:00:45 jtaylor Exp $
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: jtaylor $
 * $Date: 2013-03-26 17:00:45 $
 * $Revision: 1.33 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <string.h>
#include <strings.h>
#include "eris_ifu_error.h"
#include <eris_ifu_jitter_interface.h>
#include <eris_ifu_jitter_static.h>
#include <eris_ifu_stdstar_static.h>
#include <eris_ifu_lambda_corr.h>
#include <eris_ifu_resample.h>
#include <eris_ifu_strehl.h>
#include <eris_ifu_extract_spec_static.h>
#include <eris_ifu_efficiency_response.h>
#include <eris_utils.h>
#define M1_RADIUS  8.15
#define M2_RADIUS  0.9

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
#define CONTEXT "eris.eris_ifu_stdstar"

static const char eris_ifu_stdstar_description[] = "\
This recipe performs ERIS/SPIFFIER science data reduction.\n\
\n\
When an OH_SPEC file is provided in the SOF the recipe will use the OH lines\n\
to correct the lambda calibration.\n\
\n\
-----------------------------------------------------------------------------\n\
Input files:\n\
   DO CATG              Explanation                          Required #Frames\n\
   -------              -----------                          -------- -------\n\
 science exposures with corresponding sky exposures:                         \n\
   OBJ                  Science object exposure                 Y         ?  \n\
   SKY_OBJ              Sky exposure                            Y         ?  \n\
 or                                                                          \n\
    STD                 Standard star exposure                  Y         ?  \n\
    SKY_STD             Standard star sky exposure              Y         ?  \n\
 or                                                                          \n\
    PSF_CALIBRATOR      PSF calibration source                  Y         ?  \n\
    SKY_PSF_CALIBRATOR  PSF calibration sky exposure            Y         ?  \n\
                                                                             \n\
 and calibration frames:                                                     \n\
    DISTORTION          Table with distortion correction pars.  Y         1  \n\
    WAVE_MAP            Wavelength calibration map              Y         1  \n\
    OH_SPEC             Vector holding OH lines (note 4)        Y         1  \n\
    MASTER_DARK         Optional master dark image              N       [0,1]\n\
    MASTER_FLAT         Optional master flat image              N       [0,1]\n\
    BPM_DARK            Optional bad pixel mask (note 1 & 3)    N       [0,1]\n\
    BPM_FLAT            Optional bad pixel mask (note 2 & 3)    N       [0,1]\n\
    BPM_LINEARITY       Optional bad pixel mask (note 3)        N       [0,1]\n\
    EXTCOEFF_TABLE      Table with atmospheric extinction       N       [0,1]\n\
    FLUX_STD_CATALOG    Table with ref flux std star spectra    N       [0,1]\n\
    EFFICIENCY_WINDOWS  Table with ranges to compute efficiency N       [0,1]\n\
    FIT_AREAS           Table with ranges to perform fit        N       [0,1]\n\
    QUALITY_AREAS       Table with quality regions area         N       [0,1]\n\
    RESP_FIT_POINTS_CATALOG Table with points where to fit resp N       [0,1]\n\
    RESPONSE_WINDOWS    Table with regions for response computation N   [0,1]\n\
    TELL_MOD_CATALOG    Catalog with tellurics models           N       [0,1]\n\
\n\
 and only in case of old SINFONIE way of distortion correction:\n\
    DISTANCES           Table with slitlet distances                         \n\
                           with cube.slitlet-detection=\"DIST\" Y         1  \n\
    SLITLET_POS         Table with slitlet edge positions                    \n\
                           with cube.slitlet-detection=\"EDGE\" Y         1  \n\
\n\
Output files:\n\
   DO CATG          Explanation                                  Product Depth\n\
   -------          -------------------------------------------  -------------\n\
   OBJECT_CUBE      reconstructed object expoure (note 5)              PD_AUX\n\
   SKY_CUBE         reconstructed sky exposure (note 6)                PD_AUX\n\
\n\
Note 1) When BPM_DARK is provided this will be used as hot bad pixel mask.\n\
   Otherwise the qualiiy extension of MASTER_DARK will be used.\n\
Note 2 When BPM_FLAT is provided this will be used as cold bad pixel mask.\n\
   Otherwise the qualiiy extension of MASTER_FLAT will be used.\n\
Note 3) All BPMs will be ORed.\n\
Note 4) OH_SPEC can be a FITS file holding a single OH spectrum in the primary\n\
   HDU or a FITS file with several extensions each of them holding an OH spectrum\n\
   for one of the bands J,H,K,HK.\n\
   The pipeline package provides the file eris_oh_spec.fits holding OH spectra\n\
   for all bands.\n\
Note 5) If SKY_TWEAK is selected no image will be subtracted, else if there is\n\
   a matching sky exposure this exposure will be subtracted. Else if a masterDark\n\
   image is available this image will be subtracted\n\
Note 6) If SKY_TWEAK is selected no image will be subtracted. Else if a masterDark\n\
   image is available this image will be subtracted\n\
\n\
Information on relevant parameters may be found with\n\
  esorex --params "REC_NAME_STDSTAR"\n\
  esorex --help   "REC_NAME_STDSTAR"\n\
\n\
QC parameters:\n\
ESO QC LAMBDA SHIFT UM      OH based shift of the central wavelength in [um]\n\
ESO QC LAMBDA SHIFT PIXEL   OH based shift of the central wavelength in [pixel]\n\
";

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_ifu_stdstar, ERIS_BINARY_VERSION, "Erich Wiezorrek",
                  PACKAGE_BUGREPORT, "2019",
                  "This recipe reconstruct data cubes from std stars exposures",
                  eris_ifu_stdstar_description);

#define RECIPE_NAME "eris_ifu_stdstar"


/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    pl  the nonn-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code eris_ifu_stdstar_fill_parameterlist(cpl_parameterlist *pl)
{
    cpl_error_code  err     = CPL_ERROR_NONE;

    TRY
    {
        BRK_IF_ERROR(
            eris_ifu_jitter_fill_common_parameterlist(
                REC_NAME_STDSTAR, M_STDSTAR, pl));
        BRK_IF_ERROR(
                    eris_ifu_stdstar_fill_common_parameterlist(
                        REC_NAME_STDSTAR, M_STDSTAR, pl));
    } CATCH {
        err = cpl_error_get_code();
    }

    return err;
}

static cpl_error_code
eris_ifu_remove_noflat_products(const cpl_frameset *frameset_noflat){

	cpl_size nframes = cpl_frameset_get_size(frameset_noflat);
	const cpl_frame* frame = NULL;
	const char* fname = NULL;
	char* cmd;
	for(cpl_size i = 0; i < nframes; i++) {
		frame = cpl_frameset_get_position_const(frameset_noflat, i);
		if (cpl_frame_get_group(frame) == CPL_FRAME_GROUP_PRODUCT) {
			fname = cpl_frame_get_filename(frame);
			cmd = cpl_sprintf("rm -f %s",fname);
			system(cmd);
			cpl_free(cmd);
		}
	}
	eris_check_error_code("eris_ifu_remove_noflat_products");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int eris_ifu_stdstar(cpl_frameset *frameset,
        const cpl_parameterlist * parlist)
{
	/* check required input tags are present */
	const int ntags = 1;
	const char* required_tags[1] = {
			ERIS_IFU_CALIB_OH_SPEC
	};

	cpl_ensure_code(CPL_ERROR_NONE ==
			eris_dfs_check_input_tags(frameset, required_tags, ntags, 1),
			CPL_ERROR_ILLEGAL_INPUT);

    /* duplicate raw frameset BEFORE processing data */
    cpl_frame* flat_frame = NULL;
    cpl_frameset* frameset_noflat = NULL;
    cpl_boolean is_std_flux = CPL_FALSE;
    if(cpl_frameset_find(frameset,"STD") != NULL  ||
       cpl_frameset_find(frameset,"STD_FLUX") != NULL) {
        is_std_flux = CPL_TRUE;
    }
    flat_frame = cpl_frameset_find(frameset, ERIS_IFU_CALIB_FLATFIELD);
    if (flat_frame != NULL) {

        frameset_noflat = cpl_frameset_duplicate(frameset);
        cpl_frameset_erase(frameset_noflat,ERIS_IFU_CALIB_FLATFIELD);

    }
    
    eris_stdstar_reduction_common(frameset, parlist, CPL_TRUE, CONTEXT, RECIPE_NAME);

    if (is_std_flux && flat_frame != NULL) {
        /* repeat initial reduction with initial raw frameset without the flat
         * field to compute efficiency properly */
        eris_stdstar_reduction_common(frameset_noflat, parlist, CPL_FALSE, CONTEXT, RECIPE_NAME);
        eris_print_rec_status(3);
        cpl_frame* eff_frame = cpl_frameset_find(frameset_noflat,
                ERIS_IFU_PRO_JITTER_EFFICIENCY);
        if(eff_frame != NULL) {
            cpl_frameset_insert(frameset, cpl_frame_duplicate(eff_frame));
            cpl_frameset_erase_frame(frameset_noflat,eff_frame);
        }
        eris_ifu_remove_noflat_products(frameset_noflat);

    }
    if (flat_frame != NULL) {
    	cpl_frameset_delete(frameset_noflat);
    }
    eris_check_error_code("eris_ifu_stdstar");
    return (int)cpl_error_get_code();
}



