from .eris_utils import ErisSetupInfo
from adari_core.data_libs.master_detmon import MasterDetmonReport
from adari_core.plots.text import TextPlot
from adari_core.plots.panel import Panel
from adari_core.utils.utils import fetch_kw_or_default
from adari_core.plots.images import ImagePlot

import os


class ErisDetmonReport(MasterDetmonReport):
    files_needed = {
        "on": "ON_RAW",
        "off": "OFF_RAW",
        "gain_info": "GAIN_INFO",
        "detlin_info": "DET_LIN_INFO",
        "coeffs_cube": "COEFFS_CUBE",
        "bad_pixel_detlin": "BPM_DETLIN",
        "bad_pixel_nl": "BP_MAP_NL",
    }
    center_size = 200
    data_info = {}
    data_info["exptime_key"] = "EXPTIME"
    data_info["gain_nominal_key"] = "HIERARCH ESO DET CHIP GAIN"
    data_info["lintime_key"] = "DIT"

    def __init__(self):
        super().__init__("eris_detmon", self.files_needed)
        self.ext = "PRIMARY"
        self.port = 1

    def access_port(self, im_hdu, port):
        return im_hdu

    def generate_first_panel(self):
        panels = {}
        # Setup for master report

        new_panels = super().generate_panels(
            data_info=self.data_info,
            ext=self.ext,
            port=self.port,
        )

        for panel, panel_descr in new_panels.items():
            exptime_key = self.data_info["exptime_key"]
            lin_exptime_key = self.data_info["lintime_key"]

            longest_on_frame = sorted(
                [v for k, v in self.hdus[0].items() if "on" in k],
                key=lambda x: x[0].header.get(exptime_key, -999.0),
                reverse=True,
            )[0]

            print(longest_on_frame[0])
            # Metadata
            metadata = ErisSetupInfo.detmon(self.hdus[0]["bad_pixel_nl"])
            text2 = TextPlot(v_space=self.METADATA_V_SPACING)
            text2.add_data(metadata)
            panel.assign_plot(text2, 1, 0)

            curname = str(
                fetch_kw_or_default(
                    self.hdus[0]["bad_pixel_nl"]["PRIMARY"],
                    "HIERARCH ESO DET READ CURNAME",
                    default="N/A",
                )
            )
            if curname == "SLOW_GR_UTR":
                port_raw_data = self.access_port(
                    longest_on_frame[self.ext], self.port
                ).data
                raw_hist = panel.retrieve(1, 1)
                raw_hist.set_xlim(0, 10000)
                raw_hist.set_bins(200)

        panels = {**panels, **new_panels}

        return panels

    def generate_second_panel(self, keyword):
        panels = {}

        p = Panel(2, 2, height_ratios=[1, 4], x_stretch=1.4)
        exptime_key = self.data_info["exptime_key"]
        lin_exptime_key = self.data_info["lintime_key"]

        gain = self.hdus[0]["gain_info"][1]
        gain_hdr = self.hdus[0]["gain_info"][0]

        extname = self.ext
        procatg = fetch_kw_or_default(
            self.hdus[0][keyword][0], "HIERARCH ESO PRO CATG", "UNKNOWN"
        )
        filepath = os.path.basename(self.hdus[0][keyword].filename())

        if keyword == "bad_pixel_nl":
            lin = self.hdus[0]["detlin_info"][1]
            raw1_name = fetch_kw_or_default(
                lin, "HIERARCH ESO PRO REC1 RAW1 NAME", "UNKNOWN"
            )
        else:
            raw1_name = fetch_kw_or_default(
                gain_hdr, "HIERARCH ESO PRO REC1 RAW1 NAME", "UNKNOWN"
            )

        # Metadata
        text1 = TextPlot(v_space=self.METADATA_V_SPACING)
        text1.add_data(
            [
                f"INSTRUME: {fetch_kw_or_default(gain_hdr, 'INSTRUME', 'UNKNOWN')}",
                f"EXTNAME: {extname}",
                f"PRO.CATG: {procatg}",
                f"Filename: {filepath}",
                f"PRO.REC1.RAW1.NAME: {raw1_name}",
            ]
        )
        p.assign_plot(text1, 0, 0)

        bpm_im = ImagePlot(
            self.hdus[0][keyword][0].data,
            v_clip="val",
            v_clip_kwargs={"low": 0, "high": 1},
            interpolation="nearest",
        )
        p.assign_plot(bpm_im, 0, 1, xext=1)

        metadata = ErisSetupInfo.detmon(self.hdus[0][keyword])
        text2 = TextPlot(v_space=self.METADATA_V_SPACING)
        text2.add_data(metadata)
        p.assign_plot(text2, 1, 0)
        if keyword == "bad_pixel_nl":
            longest_on_frame = sorted(
                [v for k, v in self.hdus[0].items() if "on" in k],
                key=lambda x: x[0].header.get(exptime_key, -999.0),
                reverse=True,
            )[0]

            panels[p] = {
                "longest_on_frame": longest_on_frame,
                "gain": gain,
                "ext": self.ext,
                "port": self.port,
                "report_name": f"{self.hdus[0]['detlin_info'][0].header.get('INSTRUME', 'INSTRUMENT')}"
                f"_{self.hdus[0]['detlin_info'][0].header.get('HIERARCH ESO TPL START', 'TPL_START_TIME')}"
                f"_ext{self.ext}_port{self.port}_2",
                "report_description": f"Detector monitoring - "
                f"{self.hdus[0]['detlin_info'][0].header.get('INSTRUME', 'INSTRUMENT')}"
                f", template "
                f"{self.hdus[0]['detlin_info'][0].header.get('HIERARCH ESO TPL START', 'TPL_START_TIME')}",
                "report_tags": {},
            }
        else:
            panels[p] = {
                "gain": gain,
                "ext": self.ext,
                "port": self.port,
                "report_name": f"{self.hdus[0]['gain_info'][0].header.get('INSTRUME', 'INSTRUMENT')}"
                f"_{self.hdus[0]['gain_info'][0].header.get('HIERARCH ESO TPL START', 'TPL_START_TIME')}"
                f"_ext{self.ext}_port{self.port}",
                "report_description": f"Detector monitoring - "
                f"{self.hdus[0]['gain_info'][0].header.get('INSTRUME', 'INSTRUMENT')}"
                f", template "
                f"{self.hdus[0]['gain_info'][0].header.get('HIERARCH ESO TPL START', 'TPL_START_TIME')}",
                "report_tags": {},
            }

        return panels

    def generate_panels(self, **kwargs):

        if "bad_pixel_nl" in self.hdus[0].keys():

            panels = {
                **self.generate_first_panel(),
                **self.generate_second_panel("bad_pixel_nl"),
            }
        else:
            panels = {
                **self.generate_second_panel("bad_pixel_detlin"),
            }

        return panels


rep = ErisDetmonReport()
